package cds.healpix;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
// import java.nio.file.Files;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import org.junit.Test;

import cds.healpix.HealpixNestedBMOC.CurrentValueAccessor;

public class HealpixNestedUltraCompactMOCTest {

  public HealpixNestedBMOC coneTest(double coneCenterLonDeg, double coneCenterLatDeg, double radiusDeg, int depth) {
    System.out.println("Cone test. Center "
        + "lon: "  + coneCenterLonDeg + "deg, "
        + "lat: "  + coneCenterLatDeg + "deg, "
        + "radius: " + radiusDeg + " deg, "
        + "depth: " + depth
        );
    final double coneCenterLonRad = Math.toRadians(coneCenterLonDeg);
    final double coneCenterLatRad = Math.toRadians(coneCenterLatDeg);
    final double radiusRad = Math.toRadians(radiusDeg);
    final HealpixNestedFixedRadiusConeComputer cc = Healpix.getNested(depth).newConeComputer(radiusRad);
    System.out.println(cc.getClass().getName());
    
    return cc.overlappingCells(coneCenterLonRad, coneCenterLatRad);
  }
  
  double lonFromSexa(final String lon) {
    final String[] elems = lon.trim().split(" +");
    final double hour = Integer.parseInt(elems[0]);
    final double minh = Integer.parseInt(elems[1]) / 60.0;
    final double sech = Double.parseDouble(elems[2]) / 3600.0;
    return 15 * (hour + minh + sech);
  }

  double latFromSexa(final String lat) {
    final String[] elems = lat.trim().split(" +");
    final double deg = Integer.parseInt(elems[0]);
    final double min = Integer.parseInt(elems[1]) / 60.0;
    final double sec = Double.parseDouble(elems[2]) / 3600.0;
    return deg > 0 ? deg + min + sec : deg - min - sec; 
  }
  
  @Test
  public void test1() {
    double lon = lonFromSexa("04 04 17.99700");
    double lat = latFromSexa("+89 54 19.6682");
    double rad = 5.2403223034772175;
    int order = 3;
    HealpixNestedBMOC bmoc = coneTest(lon, lat, rad, order); //, new long[]{63, 127, 191, 255});

    System.out.println("T1 bmoc: ");
    for(final Iterator<CurrentValueAccessor> it = bmoc.iterator(); it.hasNext(); ) {
      final CurrentValueAccessor c = it.next();
      System.out.println(" - d: " + c.getDepth() + "; h: " + c.getHash() + "; bits: " + Long.toBinaryString(c.getHash()));
    }
    
    byte[] bytes = HealpixNestedUltraCompactMOC.compress(bmoc);
    System.out.println("T1 nBytes: " + bytes.length);
    // System.out.println(HealpixNestedUltraCompactMOC.compressB64(bytes));
    
    bmoc = HealpixNestedUltraCompactMOC.decompress(3, bytes);
    System.out.println("T1 bmoc: ");
    for(final Iterator<CurrentValueAccessor> it = bmoc.iterator(); it.hasNext(); ) {
      final CurrentValueAccessor c = it.next();
      System.out.println(" - d: " + c.getDepth() + "; h: " + c.getHash());
    }
    
    
    bytes = HealpixNestedUltraCompactMOC.compress(bmoc);
    // System.out.println(HealpixNestedUltraCompactMOC.compressB64(bytes));
  }
  
  @Test
  public void test3() {
    /*double lon = 205.0;
    double lat = 30.5;
    double rad = 0.05;
    int order = 12;
    
    HealpixNestedBMOC bmoc = coneTest(lon, lat, rad, order);*/
    int order = 3;
    long[] mocCells = new long[]{
      HealpixNestedBMOC.buildValue(1, 1L << 2 | 1L, false, order),
      HealpixNestedBMOC.buildValue(3, 4L << 6 | 1L, false, order),  
      HealpixNestedBMOC.buildValue(1, 6L << 2 | 1L, false, order),  
      HealpixNestedBMOC.buildValue(0, 8L, false, order),  
    };
    HealpixNestedBMOC bmoc = HealpixNestedBMOC.createUnsafe(order, mocCells);
    
    System.out.println("T3 bmoc: ");
    for(final Iterator<CurrentValueAccessor> it = bmoc.iterator(); it.hasNext(); ) {
      final CurrentValueAccessor c = it.next();
      System.out.println(" - d: " + c.getDepth() + "; h: " + c.getHash() + "; bits: " + Long.toBinaryString(c.getHash()));
    }
    
    System.out.println("T3 bmoc size: " + bmoc.size());
    
    byte[] bytes = HealpixNestedUltraCompactMOC.compress(bmoc);
    // System.out.println(HealpixNestedUltraCompactMOC.compressB64(bytes));
    
    bmoc = HealpixNestedUltraCompactMOC.decompress(order, bytes);
    System.out.println("T3 bmoc size: " + bmoc.size());
    
    bytes = HealpixNestedUltraCompactMOC.compress(bmoc);
    // System.out.println(HealpixNestedUltraCompactMOC.compressB64(bytes));
    
  }
  
  @Test
  public void test2() {
    double lon = 205.0;
    double lat = 30.5;
    double rad = 0.05;
    int order = 18;
    
    HealpixNestedBMOC bmoc = coneTest(lon, lat, rad, order);
    System.out.println("T2 bmoc size: " + bmoc.size());
    
    byte[] bytes = HealpixNestedUltraCompactMOC.compress(bmoc);
    // System.out.println(HealpixNestedUltraCompactMOC.compressB64(bytes));
    
    bmoc = HealpixNestedUltraCompactMOC.decompress(order, bytes);
    System.out.println("T2 bmoc size: " + bmoc.size());
    
    bytes = HealpixNestedUltraCompactMOC.compress(bmoc);
    // System.out.println(HealpixNestedUltraCompactMOC.compressB64(bytes));
    
  }
  
  
  private static final Map<Integer, int[]> mocGlimpse = new LinkedHashMap<Integer, int[]>();
  static {
  mocGlimpse.put(4, new int[]{1801});
  mocGlimpse.put(5, new int[]{
      7201,7203,7209,7212,7213,7214,7218,7219,7222,7224,7225,7228,7230,7554,7564,
      10552,10554,10597
  });
  mocGlimpse.put(6, new int[]{
    12970,12971,12974,14340,14341,14352,14353,14355,14356,14358,14359,14402,14403,14406,14409,
    14412,28734,28735,28778,28808,28809,28811,28833,28835,28860,28861,28862,28864,28866,28867,
    28870,28892,28894,28904,28905,28908,28909,28916,28918,28919,29067,29089,29090,29091,29096,
    29097,29098,29100,29101,29114,29115,29118,29119,29163,29164,29166,29440,29461,29504,29505,
    29507,29508,29510,29511,29513,29515,29516,29522,29528,29529,29532,29533,29535,30211,30238,
    30240,30241,30242,30244,30245,30260,30262,30267,30268,30269,30270,30312,30313,30315,30318,
    30319,30416,30417,30420,30422,30423,30594,30601,30604,30605,30618,30619,30622,30644,30645,
    30688,30689,30691,30694,30695,30706,30712,30713,30716,30717,30719,37904,37905,41599,41655,
    41659,41660,41675,41676,41677,41678,41681,41682,41683,41684,41696,41767,41769,41770,41772,
    41777,41778,41780,41781,41802,41803,41804,41805,41811,41814,41815,41816,42153,42155,42156,
    42158,42159,42170,42171,42172,42173,42214,42215,42225,42228,42362,42374,42377,42378,42379,
    42380,42381,42382,42392,42393,42395,42396,42397,42398,42401,42403,42404,42417,42420,42434,
    42435,42438,42439,42440,42441,42444,42445,42447,42448,42450,42451,42454,42456,42457,42460,
    42461,42468,42469,42496,42497,42500,42501,42502,43015,43019,43020,43024,43040
  });
  mocGlimpse.put(7, new int[]{
      51874,51875,51878,51879,51900,51902,51903,51946,51947,51950,57344,57345,57348,57349,57373,
      57416,57417,57428,57430,57431,57460,57461,57602,57603,57606,57628,57630,57632,57633,57652,
      114874,114875,114878,114879,114919,114922,114925,114927,114930,114931,114934,115116,115118,115207,115210,
      115211,115213,115214,115215,115240,115241,115243,115329,115331,115337,115339,115381,115383,115388,115389,
      115452,115454,115460,115462,115484,115486,115487,115530,115531,115574,115580,115628,115629,115640,115641,
      115644,115645,115670,115671,115696,115697,115698,115700,115701,115703,115709,115711,116265,116267,116280,
      116282,116353,116354,116355,116368,116370,116376,116378,116379,116382,116396,116397,116398,116408,116409,
      116412,116413,116415,116448,116450,116451,116454,116455,116466,116647,116650,116651,116662,116663,116668,
      116670,117077,117078,117079,117764,117766,117828,117829,117840,117841,117843,118024,118025,118027,118036,
      118038,118039,118049,118051,118056,118057,118068,118069,118072,118074,118094,118095,118106,118125,118136,
      118137,118139,118229,118231,118237,118238,118239,118261,120838,120841,120842,120843,120856,120858,120880,
      120888,120890,120891,120894,120941,120943,120956,120958,120959,120972,120974,120985,120988,120989,120991,
      120992,120993,120996,121044,121046,121052,121054,121055,121060,121061,121063,121065,121067,121085,121226,
      121227,121256,121257,121266,121267,121270,121320,121322,121323,121326,121327,121412,121413,121424,121425,
      121426,121616,121617,121620,121621,121623,121673,121676,121677,121684,121686,121687,121717,122370,122380,
      122382,122383,122394,122395,122400,122401,122424,122425,122428,122429,122431,122464,122466,122467,122470,
      122471,122492,122494,122495,122564,122565,122589,122666,122760,122761,122763,122770,122771,122774,122800,
      122801,122804,122805,122807,122830,122831,122842,122860,122861,122872,122873,122875,149374,149375,151581,
      151624,151625,151628,151629,151632,166391,166394,166395,166613,166614,166615,166617,166618,166619,166629,
      166630,166631,166633,166634,166635,166644,166648,166685,166686,166687,166695,166697,166698,166699,166716,
      166722,166723,166740,166741,166742,166744,166752,166753,166754,166788,166789,166790,166792,166793,166794,
      166800,167034,167035,167037,167038,167039,167055,167063,167065,167066,167067,167073,167074,167075,167084,
      167085,167086,167092,167093,167094,167096,167106,167107,167116,167117,167118,167128,167136,167198,167199,
      167203,167206,167207,167224,167225,167226,167228,167241,167242,167243,167251,167254,167255,167268,167269,
      167280,167296,167297,167424,168602,168603,168606,168628,168630,168661,168663,168669,168671,168675,168678,
      168679,168696,168697,168698,168700,168701,168703,168747,168750,168751,168762,168826,168827,168830,168831,
      168848,168850,168854,168855,168880,168881,168882,168888,168890,168897,168898,168899,168904,168905,168906,
      168908,168909,168916,168917,168918,169211,169214,169215,169384,169386,169407,169452,169453,169454,169483,
      169485,169486,169487,169494,169500,169502,169505,169507,169532,169533,169541,169543,169549,169550,169551,
      169576,169577,169600,169601,169603,169609,169611,169624,169625,169626,169667,169673,169676,169677,169688,
      169728,169749,169751,169785,169787,169796,169798,169820,169822,169823,169832,169834,169837,169848,169849,
      169852,169853,169880,169881,169884,169885,169920,169992,169993,169996,169997,169999,170012,170014,170015,
      170021,170023,170032,170033,170034,170036,170048,170071,170240,170241,170242,170244,170245,172053,172054,
      172055,172057,172058,172059,172069,172071,172075,172084,172086,172088,172090,172100,172104,172164,172165,
      172166,172168,172169,172170
  });
  mocGlimpse.put(8, new int[]{
      207490,207491,207494,207495,207562,207563,207566,207604,207606,207607,207778,207806,207807,207850,207851,
      229389,229400,229401,229404,229405,229407,229473,229476,229477,229488,229489,229491,229672,229673,229676,
      229677,229718,229719,229824,229825,229828,229829,230402,230403,230430,230431,230474,230475,230516,230518,
      230526,230536,230537,230540,230541,230616,230617,230618,230625,230628,230629,230640,459455,459491,459494,
      459495,459506,459671,459692,459694,459695,459697,459699,459705,459706,459707,459714,459742,459743,460426,
      460427,460470,460471,460476,460477,460478,460821,460823,460834,460835,460838,460839,460849,460850,460851,
      460969,460971,461312,461313,461321,461323,461345,461347,461353,461508,461509,461520,461521,461523,461529,
      461530,461531,461560,461561,461564,461565,461812,461813,461844,461846,461852,461854,461855,461898,461899,
      461902,461942,461943,462114,462115,462136,462138,462139,462142,462143,462290,462291,462300,462328,462330,
      462496,462497,462500,462501,462569,462572,462573,462584,462585,462588,462589,462672,462674,462675,462796,
      462797,462808,462809,462816,462817,462818,462824,462826,462833,462835,462841,462843,465037,465038,465039,
      465050,465051,465054,465065,465126,465127,465132,465133,465134,465408,465409,465411,465476,465478,465510,
      465534,465535,465596,465598,465644,465645,465656,465657,465659,465796,465798,465799,465810,465811,465870,
      465871,465882,466538,466579,466581,466582,466583,466585,466587,466594,466595,466598,466642,466643,466676,
      466678,467220,467221,467264,467265,467268,467269,467271,467280,467282,468240,468241,468244,468245,468305,
      468306,468307,468341,471062,471072,471073,471076,471077,471078,471088,471089,471090,471296,471297,471300,
      471301,471303,471324,471325,471368,471369,471408,471409,471412,471413,471415,472193,472201,472203,472232,
      472233,472236,472237,472239,472281,472284,472285,472292,472294,472330,472331,472334,472368,472370,472371,
      472374,472375,472428,472430,472431,472442,472484,472485,472496,472497,472499,472552,472553,472593,472596,
      472597,472640,472641,472642,472913,472921,472923,472943,472945,472946,472947,473040,473041,483339,483341,
      483342,483343,483358,483359,483360,483361,483363,483402,483428,483430,483431,483436,483438,483528,483529,
      483530,483558,483559,483570,483580,483582,483583,483743,483754,483755,483770,483771,483785,483786,483787,
      483790,483791,483828,483830,483831,483892,483893,483894,483900,483902,483936,483937,483939,483960,483961,
      483977,483980,483981,483983,483988,483992,483993,483994,484180,484181,484182,484188,484190,484212,484214,
      484215,484225,484228,484229,484249,484250,484251,484259,484266,484267,484336,484337,484338,484344,484514,
      484520,484898,484922,484923,485032,485033,485036,485037,485058,485059,485062,485086,485087,485258,485284,
      485286,485287,485298,485299,485354,485355,485358,485633,485636,485637,485639,485660,485661,485708,485710,
      485729,486417,486420,486421,486476,486477,486488,486489,486491,486688,486689,486712,486713,486716,486717,
      486719,486742,486743,486852,486853,486864,486865,489486,489487,489498,489499,489524,489526,489527,489570,
      489571,489592,489594,489595,489598,489599,489608,489609,489612,489613,489615,489649,489652,489653,489705,
      489709,489720,489721,489723,489860,489862,489863,489874,489930,489931,489934,489974,489975,490240,490241,
      490244,490245,490247,490265,490268,490269,490340,490341,490352,490353,490355,490365,490658,490668,490670,
      490671,490682,491048,491049,491074,491075,491078,491079,491102,491103,491153,491156,491157,491159,491213,
      491224,491225,491274,491275,491312,491314,491315,491318,491374,491375,491386,491425,491428,491429,491453,
      491496,491497,513360,513361,513364,513365,597487,606295,606301,606302,606303,606321,606323,606328,606329,
      606330,606332,606333,606504,606505,606508,606509,606520,606532,606536,606537,606538,608422,608423,608424,
      608425,608426,608428,608429,608434,608440,665531,665533,665534,665535,665559,665562,665563,665571,665573,
      665574,665575,666111,666365,666366,666367,666429,666430,666431,666445,666446,666447,666449,666450,666451,
      666465,666466,666467,666509,666510,666511,666513,666514,666515,666529,666530,666531,666580,666581,666582,
      666584,666585,666586,666596,666597,666598,666600,666601,666602,666719,666731,666733,666734,666735,666737,
      666738,666739,666767,666773,666774,666775,666777,666778,666779,666787,666868,666869,666870,666872,666873,
      666874,666885,666886,666887,666972,666973,666974,666980,666981,666982,666984,666985,666992,667020,667021,
      667024,667025,667026,667028,667040,667164,667165,667166,667180,667184,667185,667204,667264,668091,668094,
      668095,668127,668134,668135,668145,668146,668147,668215,668218,668219,668250,668251,668257,668258,668259,
      668289,668290,668291,668348,668349,668380,668381,668388,668389,668390,668418,668419,668421,668422,668423,
      668476,668477,668478,668516,668517,668518,668520,668521,668528,668529,668548,668549,668550,668552,668560,
      668782,668783,668810,668811,668818,668819,668821,668822,668823,668908,668909,668910,668916,668917,668918,
      668920,668958,668959,668962,668963,668999,669001,669002,669003,669010,669011,669013,669014,669015,669080,
      669081,669082,669084,669085,669088,669089,669092,669124,669125,669126,669136,669137,669192,669193,669196,
      669200,669201,669202,669204,669700,669701,669702,669704,669712,674362,674363,674366,674367,674418,674419,
      674420,674422,674423,674428,674430,674431,674475,674477,674478,674479,674516,674518,674519,674524,674526,
      674527,674671,674681,674682,674683,674697,674698,674699,674706,674707,674710,674711,674796,674797,674986,
      674987,674998,674999,675041,675042,675043,675259,675262,675263,675314,675315,675317,675318,675319,675406,
      675407,675415,675532,675536,675537,675538,675540,675541,675586,675587,675628,675629,675640,675641,675676,
      675677,675678,675680,675681,675682,675684,675685,675696,676799,676841,676842,676843,676851,676854,676855,
      677540,677541,677542,677599,677621,677622,677623,677770,677771,677774,677775,677786,677787,677790,677791,
      677820,677821,677822,677834,677927,677937,677938,677939,677962,677963,677966,677967,677980,677982,678004,
      678012,678025,678136,678137,678138,678140,678186,678187,678189,678190,678191,678193,678319,678384,678385,
      678386,678388,678389,678392,678394,678408,678409,678411,678433,678435,678441,678443,678480,678481,678482,
      678488,678489,678490,678508,678509,678510,678512,678514,678529,678531,678532,678533,678534,678544,678545,
      678660,678661,678663,678691,678697,678700,678701,678716,678736,678756,678920,678922,678926,678938,678943,
      678991,678993,678995,679001,679002,679003,679072,679073,679076,679077,679088,679089,679092,679093,679136,
      679137,679139,679145,679146,679147,679188,679190,679286,679287,679332,679333,679334,679340,679344,679345,
      679347,679477,679479,679688,679690,679691,679712,679976,679977,679978,679980,679981,679992,679993,679995,
      680052,680054,680055,680080,680081,680083,680088,680089,680091,680112,680113,680116,680140,680141,680148,
      680150,680152,680196,680197,680198,680208,680277,680278,680279,680308,680972,680973,680974,680984,680985,
      680988,681024,681025,681026,688191,688205,688207,688209,688210,688211,688227,688281,688282,688283,688293,
      688294,688295,688299,688340,688341,688342,688356,688357,688358,688364,688404,688406,688408,688409,688410,
      688420,688421,688422,688424,688425,688426,688668,688684,688685,688686,688688,688690,688704,688768,688769,
      688770,688776
  });
  mocGlimpse.put(9, new int[]{
      829952,829954,829955,829956,829958,829959,829970,830024,830026,830027,830030,830031,830042,830242,830243,
      830246,830270,830271,830314,830315,830422,830423,831106,831116,831118,831119,831128,831130,831131,831134,
      831135,831178,831218,831219,831222,831223,831418,831419,831422,831423,917536,917540,917541,917552,917553,
      917555,917608,917609,917612,917613,917615,917624,917625,917627,917713,917888,917889,917913,917916,917917,
      917960,917961,917962,917989,918001,918004,918005,918701,918713,918716,918717,918718,918866,918867,918870,
      919109,919120,919121,919124,919125,919309,919320,919321,919324,919325,919326,919397,919408,919409,919412,
      919413,919414,921626,921627,921630,921631,921674,921712,921714,921715,921718,921719,921890,921891,921912,
      921914,921915,921916,921918,921919,921962,921963,922068,922070,922071,922078,922096,922097,922098,922100,
      922157,922168,922169,922172,922173,922174,922309,922320,922324,922325,922326,922448,922450,922456,922476,
      922478,922496,922497,922499,922520,922524,922525,922564,922568,922569,922571,1837813,1837814,1837815,1837817,
      1837819,1837958,1837961,1837962,1837963,1837971,1838030,1838031,1838042,1838043,1838046,1838047,1838678,1838681,1838682,
      1838683,1838693,1838703,1838730,1838772,1838774,1838775,1838817,1838818,1838819,1838860,1838862,1838863,1838872,1838874,
      1838875,1838878,1838879,1838960,1838962,1838963,1838966,1838967,1841698,1841702,1841721,1841722,1841723,1841726,1841727,
      1841770,1841771,1841775,1841872,1841874,1841875,1841876,1841878,1841879,1841916,1843281,1843289,1843290,1843291,1843301,
      1843311,1843330,1843331,1843334,1843346,1843347,1843351,1843872,1843873,1843875,1843883,1845261,1845263,1845283,1845377,
      1845378,1845379,1845384,1845385,1845387,1845409,1845411,1845417,1845420,1845421,1845829,1845841,1845844,1846016,1846017,
      1846020,1846021,1846023,1846089,1846091,1846113,1846115,1846167,1846173,1846175,1846196,1846197,1846199,1846252,1846253,
      1846264,1846265,1846267,1846268,1846269,1846271,1847256,1847257,1847258,1847280,1847282,1847288,1847380,1847414,1847415,
      1847587,1847614,1847615,1847658,1847761,1847762,1847763,1847766,1847767,1848450,1848451,1848472,1848474,1848475,1848478,
      1848550,1848551,1848562,1848567,1848744,1848746,1848747,1848748,1848750,1848751,1848763,1849152,1849154,1849155,1849158,
      1849159,1849176,1849178,1849179,1849206,1849208,1849209,1849210,1849298,1849304,1849316,1849318,1849324,1849326,1849327,
      1849997,1850009,1850012,1850013,1850096,1850100,1850101,1850103,1850272,1850273,1850275,1850297,1850348,1850349,1850360,
      1850361,1850364,1850365,1850366,1850692,1850693,1850694,1850714,1850715,1850718,1850719,1851192,1851244,1851245,1851247,
      1851276,1851277,1851278,1851282,1851308,1851310,1851361,1851362,1851363,1851369,1860143,1860147,1860194,1860195,1860198,
      1860222,1860223,1860229,1860237,1860238,1860239,1860257,1860269,1860270,1860271,1860394,1860498,1860499,1860502,1860540,
      1860541,1860542,1860578,1861641,1861643,1861910,1861916,1861936,1861938,1861944,1861946,1861947,1862032,1862046,1862047,
      1862090,1862130,1862131,1862134,1862135,1862390,1862396,1862397,1862398,1862560,1862561,1862564,1862565,1862567,1862588,
      1862589,1862632,1862633,1862824,1862826,1862827,1862828,1862830,1862831,1862843,1863188,1863190,1863191,1863232,1863234,
      1863235,1863239,1863256,1863258,1863259,1863262,1863263,1863474,1863475,1863478,1863534,1863535,1863546,1865981,1865982,
      1865983,1866147,1866156,1866158,1866159,1866170,1866311,1866315,1866321,1866322,1866323,1866337,1866339,1866346,1866347,
      1866370,1866371,1866374,1866389,1866391,1866398,1866399,1866562,1866563,1866566,1866585,1866586,1866587,1866590,1866710,
      1866716,1866717,1866718,1866736,1866737,1866738,1866744,1866746,1866747,1867298,1867304,1868817,1868820,1868821,1868864,
      1868865,1868867,1868868,1868869,1868871,1868893,1869065,1869068,1869069,1869080,1869081,1869083,1869124,1869125,1869126,
      1869132,1869152,1872897,1872901,1872912,1872913,1872914,1872916,1872917,1872919,1872972,1872973,1872984,1872985,1872987,
      1872988,1872989,1872990,1873152,1873153,1873154,1873181,1873213,1873215,1873217,1873218,1873219,1873349,1873360,1873361,
      1884240,1884241,1884242,1884272,1884273,1884274,1884282,1884297,1884300,1884301,1884316,1884317,1884319,1884364,1884368,
      1884384,1884497,1884500,1884501,1885197,1885208,1885209,1885210,1885237,1885281,1885284,1885285,1885286,1885305,1885308,
      1885480,1885481,1885483,1885484,1885485,1885487,1885584,1885585,1885588,1885589,1885591,1885644,1885645,1885656,1885657,
      1885659,1886167,1886173,1886197,1886199,1888416,1888417,1888420,1888421,1888423,1888429,1888430,1888431,1888592,1888593,
      1888602,1888603,1888779,1888781,1888782,1888783,1888800,1888801,1888803,1888809,1888811,1888941,1888943,1888952,1888953,
      1888954,1889120,1889121,1889133,1889145,1889148,1889149,1889150,1889172,1889180,1889182,1889200,1889201,1889202,1889204,
      1889208,1889210,1889312,1889314,1889315,1889318,1889319,1889340,1889342,1889343,1889386,1889387,1889476,1889478,1889479,
      1889490,1889491,1889672,1889674,1889675,1889678,1889679,1889716,1889718,1889719,1889762,1889763,1889772,1889774,1889775,
      1889781,1889782,1889783,1889786,1889787,1889789,1889791,1889921,1889924,1889925,1889949,1889992,1889993,1890021,1890033,
      1890036,1890037,1890038,1890217,1890220,1890221,1890324,1890325,1890368,1890369,1890371,1890392,1890393,1890396,1890397,
      1890399,1890572,1890573,1890576,1890577,1890578,1890592,1891605,1891648,1891649,1891661,1891683,1891688,1891689,1891691,
      1891741,1891743,1891767,1891777,1891778,1891779,1892116,1892117,1892168,1892169,1892172,1892173,1892175,1892212,1892213,
      1933341,1933342,1933343,1933351,1933354,1933355,1933361,1933362,1933363,1933384,1933386,1933387,1933426,1933427,1933430,
      1933449,1933450,1933451,1933614,1933626,1933718,1933719,1933748,1933758,1933770,1934096,1934097,1934098,1934100,1934106,
      1934124,1934125,1934126,1934128,1934130,1934226,1934227,1934230,1934286,1934287,1934298,1934324,1934326,1934327,1934971,
      1935010,1935011,1935032,1935034,1935035,1935039,1935045,1935046,1935047,1935052,1935053,1935055,1935075,1935077,1935078,
      1935079,1935119,1935138,1935139,1935152,1935154,1935155,1935158,1935159,1935209,1935210,1935211,1935212,1935214,1935227,
      1935318,1935319,1935580,1935581,1935582,1935604,1935606,1935607,1935752,1935753,1935793,1935796,1935797,1935849,1935852,
      1935853,1935904,1935905,1935907,1935916,1935928,1935929,1935931,1935958,1935960,1935961,1935980,1935981,1935982,1935984,
      1935986,1935992,1936021,1936064,1936065,1936213,1936732,1936733,1936734,1936756,1936758,1936759,1936764,1936765,1936854,
      1936855,1936896,1936897,1936899,1936921,1936924,1936925,1936926,1936957,1936958,1936959,1936993,1936995,1937031,1937061,
      1937062,1937063,1937356,1937357,1937359,1937382,1937384,1937385,1937386,1937393,1937397,1938050,1938084,1938086,1938088,
      1938089,1938090,1939456,1939458,1939459,1939464,1939586,1939598,1939599,1939610,1939611,1939680,1939682,1939683,1939687,
      1939704,1939706,1939707,1939710,1939711,1940140,1940141,1940152,1940153,1940155,1940156,1940157,1940159,1940224,1940226,
      1940227,1940228,1940230,1940231,1940242,1940243,1940253,1940254,1940255,1940298,1940337,1940338,1940339,1940342,1941038,
      1941050,1941142,1941143,1941186,1941208,1941210,1941408,1941410,1941411,1941414,1941415,1941438,1941439,1941482,1942529,
      1942541,1942552,1942553,1942580,1942624,1942625,1942627,1942628,1942629,1942648,1942649,1942652,1942653,1942655,1942836,
      1942838,1942839,1942844,1942845,1942846,1942848,1942850,1942856,1942928,1945621,1945664,1945665,1945689,1945692,1945693,
      1945892,1945893,1945917,1945960,1945961,1945963,1946064,1946065,1946068,1946069,1946071,1946764,1946765,1946805,1946861,
      1946872,1946873,1946962,1947392,1947393,1947396,1947397,1947399,1947420,1947464,1947465,1947467,1947468,1947469,1947471,
      1947504,1947505,1947508,1947509,1947511,1957898,1957899,1957936,1957938,1957939,1957942,1957943,1957986,1958010,1958011,
      1958100,1958102,1958103,1958272,1958274,1958275,1958278,1958296,1958298,1958299,1958302,1958303,1958346,1958372,1958374,
      1958375,1958387,1958444,1958445,1958446,1958456,1958457,1958459,1958545,1958548,1958592,1958593,1958595,1958616,1958617,
      1958620,1958621,1958623,1958645,1958832,1958833,1958835,1958844,1958845,1958888,1958889,1958891,1959082,1959083,1959446,
      1959447,1959490,1959500,1959502,1959503,1959514,1959515,1959714,1959715,1959718,1959742,1959743,1959786,1959890,1959891,
      1959894,1960272,1960273,1960276,1960277,1960279,1960968,1960972,1960973,1960975,1960984,1960985,1960987,1961012,1961013,
      1961056,1961057,1961059,1961080,1961081,1961084,1961085,1961087,1961344,1961348,1961349,1961351,1961372,1961373,1961416,
      1961417,1961418,1961445,1961456,1961457,1962638,1962676,1962678,1962679,1962723,1962732,1962734,1962735,1962744,1962746,
      1962747,1964201,1964204,1964205,1964288,1964290,1964291,1964294,1964295,1964362,1964363,1964366,1964400,1964402,1964403,
      1964406,1964407,1964565,1964608,1964609,1964621,1964633,1964836,1964848,1964849,1964851,1964904,1964905,1964908,1964909,
      1964911,1965012,1965013,1965090,1965114,1965115,1965118,1965254,1965266,1965277,1965278,1965279,1965450,1965451,1965488,
      1965490,1965491,1965494,1965495,1965548,1965550,1965551,1965560,1965562,1965563,1965567,1965696,1965697,1965699,1965720,
      1965721,1965724,1965725,1965727,1965792,1965796,1965797,1965799,1965808,1965809,1965811,1965820,1965992,1965993,1965996,
      1965997,2053397,2053399,2053453,2053465,2053467,2053468,2053469,2053471,2389942,2389943,2389945,2389947,2389978,2389979,
      2389982,2389983,2391313,2391316,2391317,2391360,2391361,2425173,2425175,2425179,2425199,2425201,2425202,2425203,2425271,
      2425277,2425279,2425290,2425291,2425324,2425325,2425326,2425336,2425337,2425340,2425341,2426024,2426028,2426084,2426085,
      2426086,2426096,2426097,2426100,2426132,2426134,2426136,2426137,2426138,2426156,2426158,2426160,2426240,2426241,2426242,
      2426248,2426250,2433674,2433675,2433677,2433678,2433679,2433708,2433709,2433720,2433721,2433724,2433725,2433740,2433742,
      2433764,2433768,2654207,2662111,2662121,2662123,2662129,2662130,2662131,2662203,2662206,2662207,2662243,2662245,2662246,
      2662247,2662278,2662279,2662281,2662282,2662283,2662289,2662290,2662291,2664437,2664438,2664439,2664441,2664442,2664443,
      2665439,2665455,2665459,2665693,2665694,2665695,2665710,2665711,2665713,2665714,2665715,2665759,2665775,2665779,2665795,
      2665859,2666013,2666014,2666015,2666029,2666031,2666034,2666035,2666049,2666050,2666051,2666113,2666114,2666115,2666332,
      2666333,2666334,2666348,2666349,2666350,2666352,2666353,2666354,2666356,2666360,2666396,2666398,2666412,2666416,2666417,
      2666432,2666433,2666863,2666870,2666871,2666873,2666874,2666875,2666907,2666910,2666911,2666918,2666919,2666922,2666923,
      2666929,2666930,2666931,2666946,2666947,2667086,2667087,2667091,2667105,2667106,2667107,2667143,2667146,2667147,2667484,
      2667485,2667486,2667504,2667533,2667534,2667535,2667900,2667901,2667902,2667932,2667933,2667934,2667944,2667945,2667946,
      2667948,2667952,2667953,2667972,2667973,2667976,2667977,2667984,2668088,2668089,2668090,2668108,2668110,2668116,2668117,
      2668164,2668668,2668669,2668670,2668724,2668725,2668726,2668728,2668730,2668744,2668745,2668746,2668748,2668752,2668820,
      2668822,2668832,2668833,2668834,2669060,2669064,2672361,2672362,2672363,2672370,2672371,2672373,2672374,2672375,2672494,
      2672495,2672505,2672506,2672507,2672522,2672523,2672525,2672526,2672527,2672534,2672535,2672577,2672578,2672579,2672827,
      2672830,2672831,2672866,2672867,2672869,2672870,2672871,2672955,2672958,2672959,2672990,2672991,2672995,2672997,2672998,
      2672999,2673027,2673155,2673400,2673401,2673402,2673404,2673405,2673528,2673529,2673530,2673532,2673533,2673568,2673569,
      2673570,2673572,2673600,2673601,2673602,2673604,2673605,2673670,2673671,2673681,2673682,2673683,2673916,2673917,2673918,
      2674076,2674077,2674078,2674088,2674096,2674097,2674120,2674121,2674124,2674128,2674129,2674130,2674132,2674204,2674205,
      2674212,2674244,2674245,2674256,2675114,2675115,2675118,2675119,2675162,2675163,2675166,2675167,2675226,2675227,2675230,
      2675231,2675235,2675238,2675239,2675271,2675282,2675283,2675644,2675676,2675677,2675678,2675684,2675685,2675686,2675696,
      2675697,2675818,2675819,2675822,2675823,2675827,2675830,2675831,2675846,2675847,2675983,2675993,2675994,2675995,2676001,
      2676002,2676003,2676038,2676039,2676050,2676051,2676332,2676333,2676334,2676344,2676345,2676360,2676361,2676364,2676372,
      2676508,2676509,2676510,2676512,2676513,2676516,2676560,2676561,2676564,2676565,2676776,2676777,2676778,2676780,2676781,
      2676788,2676789,2676792,2676812,2676813,2676820,2676821,2676822,2676824,2676825,2676992,2676993,2676996,2678812,2678813,
      2678820,2678822,2678852,2678853,2697199,2697210,2697211,2697213,2697214,2697215,2697405,2697406,2697407,2697447,2697458,
      2697459,2697461,2697462,2697463,2697613,2697615,2697625,2697626,2697627,2697629,2697630,2697631,2697666,2697667,2697670,
      2697671,2697684,2697716,2697745,2697748,2697749,2697751,2697757,2697758,2697759,2697777,2697780,2697781,2697783,2697789,
      2697790,2697791,2697876,2697877,2697885,2697887,2697898,2697899,2698068,2698100,2698102,2698564,2698565,2698567,2698575,
      2698596,2698597,2698599,2698607,2698681,2698682,2698683,2698692,2698693,2698703,2698723,2698781,2698782,2698783,2698786,
      2698787,2698823,2698834,2698835,2698837,2698838,2698839,2699192,2699232,2699233,2699234,2699236,2699237,2699239,2699245,
      2699938,2699939,2699942,2699943,2699978,2699979,2699982,2699983,2699987,2699990,2699991,2700161,2700163,2700184,2700185,
      2700186,2700216,2700217,2700218,2700991,2701034,2701035,2701046,2701047,2701210,2701211,2701214,2701215,2701266,2701267,
      2701592,2701593,2701594,2701618,2701619,2701622,2701623,2701643,2701646,2701647,2701657,2701658,2701659,2702132,2702136,
      2702137,2702138,2702156,2702157,2702224,2702226,2702232,2702233,2702234,2702256,2702264,2702265,2702266,2702339,2702341,
      2702342,2702343,2702520,2702521,2702522,2702524,2702525,2702568,2702576,2702577,2702578,2702716,2702717,2702718,2702732,
      2702733,2702744,2702745,2702788,2702848,2707190,2707191,2707193,2707194,2707195,2707357,2707358,2707359,2707362,2707363,
      2707399,2707400,2707402,2707403,2707410,2707411,2707413,2707414,2707415,2710063,2710074,2710075,2710078,2710172,2710192,
      2710194,2710195,2710200,2710201,2710202,2710390,2710391,2710393,2710394,2710395,2710479,2710481,2710483,2710500,2710501,
      2710503,2710509,2710511,2711074,2711075,2711078,2711079,2711090,2711091,2711094,2711138,2711143,2711292,2711293,2711294,
      2711424,2711425,2711426,2711432,2711456,2711677,2711678,2711679,2711702,2711703,2711705,2711706,2711707,2711717,2711719,
      2711746,2711747,2711847,2711858,2711859,2711861,2711862,2711863,2711887,2711898,2711899,2711924,2711925,2711926,2712020,
      2712021,2712022,2712024,2712025,2712026,2712052,2712056,2712058,2712059,2712062,2712063,2712069,2712070,2712071,2712079,
      2712107,2712110,2712111,2712556,2712557,2712558,2712564,2712565,2712566,2712568,2712569,2712645,2712647,2712677,2712678,
      2712679,2712687,2712754,2712755,2712771,2712781,2712782,2712783,2713248,2713249,2713250,2713252,2713265,2713268,2713269,
      2713271,2713275,2713548,2713549,2713550,2713560,2713561,2713572,2713573,2713574,2713580,2713640,2713641,2713729,2713730,
      2713731,2713761,2713762,2713763,2713771,2713932,2713934,2713964,2714044,2714054,2714080,2714113,2714114,2714115,2714123,
      2714140,2714141,2714142,2714144,2714145,2714148,2714185,2714192,2714193,2714196,2714197,2714369,2714649,2714651,2714663,
      2714669,2714759,2714763,2714848,2714849,2714851,2714852,2714853,2714868,2714869,2714952,2714953,2714954,2714976,2714984,
      2714986,2715028,2715029,2715030,2715032,2715033,2715034,2715040,2715041,2715044,2715045,2715056,2715684,2715692,2715694,
      2715695,2715709,2715710,2715711,2715757,2715758,2715759,2715768,2715770,2715771,2715945,2715946,2715947,2715948,2715950,
      2715951,2715961,2715962,2715963,2715971,2715977,2715979,2716001,2716002,2716003,2716296,2716301,2716312,2716317,2716364,
      2716380,2716413,2716415,2716553,2716577,2716578,2716579,2716764,2716784,2716785,2716786,2716794,2716795,2716798,2716799,
      2716970,2716971,2717138,2717139,2717140,2717142,2717143,2717340,2717341,2717342,2717364,2717368,2717384,2717385,2717387,
      2717620,2717664,2717665,2717668,2717669,2717680,2717681,2717684,2717685,2717789,2717790,2717791,2717812,2717813,2717815,
      2717821,2717822,2717823,2717905,2717907,2717913,2717915,2717940,2718113,2718128,2718129,2718133,2718176,2718181,2718193,
      2718196,2718736,2718738,2718744,2718756,2718758,2718852,2718856,2719916,2719917,2719918,2719977,2719978,2719979,2720212,
      2720213,2720214,2720277,2720329,2720331,2720363,2720468,2720469,2720472,2720568,2720569,2720570,2720596,2720597,2720598,
      2720604,2720612,2720613,2720624,2720640,2720796,2720797,2720798,2720800,2720801,2720808,2720836,2720837,2720838,2720840,
      2720848,2720849,2720852,2720853,2720904,2720906,2720936,2720938,2721101,2721103,2721105,2721106,2721107,2721133,2721135,
      2721236,2721237,2721238,2721240,2723900,2723901,2723944,2723945,2723956,2723957,2723968,2723969,2723972,2724108,2724112,
      2724128,2724129,2752759,2752763,2752799,2752825,2752826,2752827,2752835,2752901,2752902,2752903,2752905,2752907,2753087,
      2753101,2753102,2753103,2753121,2753122,2753123,2753171,2753189,2753190,2753191,2753193,2753194,2753195,2753372,2753373,
      2753392,2753393,2753394,2753436,2753437,2753438,2753460,2753464,2753472,2753473,2753474,2753480,2753620,2753621,2753622,
      2753628,2753644,2753692,2753693,2753694,2753708,2753709,2753710,2753712,2753713,2753714,2753728,2753729,2753730,2753792,
      2753793,2754048,2754049,2754050,2754676,2754677,2754678,2754680,2754681,2754682,2754748,2754749,2754750,2754756,2754757,
      2754758,2754764,2754768,2754784,2754820,2754824,2754826,2755084,2755085,2755086,2755088,2755089,2755090,2755112
  });
  mocGlimpse.put(10, getD10());
  }
  private static final int[] getD10 () {
    return new int[]{
        3318442,3318443,3319812,3319814,3319815,3319830,3319831,3319874,3319885,3319886,3319887,3319896,3319898,3319899,3319902,
        3320100,3320102,3320103,3320114,3320115,3320174,3320960,3320962,3320963,3320966,3320967,3320988,3320990,3320991,3321034,
        3321035,3321038,3321073,3321074,3321075,3321078,3321079,3321250,3321251,3321272,3321274,3321275,3321278,3321322,3321683,
        3321686,3321687,3324430,3324431,3324442,3324443,3324468,3324470,3324471,3324516,3324518,3324519,3324530,3324718,3324719,
        3324730,3324731,3324865,3324866,3324867,3324870,3324871,3324882,3325576,3325578,3325579,3325582,3325583,3325594,3325666,
        3325667,3325670,3325671,3325682,3325864,3325866,3325867,3325870,3325882,3670148,3670149,3670169,3670172,3670173,3670217,
        3670244,3670245,3670256,3670260,3670261,3670263,3670441,3670456,3670457,3670459,3670504,3670505,3670507,3670864,3670865,
        3670866,3670868,3670869,3670871,3671560,3671561,3671563,3671564,3671565,3671601,3671604,3671605,3671672,3671677,3671852,
        3671853,3671854,3671953,3672000,3672001,3672013,3672024,3672025,3672029,3674784,3674785,3674788,3674801,3674803,3674813,
        3674848,3674849,3674850,3674861,3674876,3674877,3674879,3675456,3675458,3675459,3675462,3675463,3675474,3675485,3675486,
        3675487,3676245,3676416,3676417,3676419,3676420,3676421,3676432,3676433,3676435,3676444,3676492,3676493,3676495,3676504,
        3676505,3676506,3676508,3676509,3676511,3677216,3677217,3677220,3677232,3677233,3677235,3677244,3677245,3677292,3677293,
        3677295,3677308,3677309,3677311,3677568,3677569,3677571,3677572,3677573,3677574,3677584,3677585,3677587,3677596,3677597,
        3677640,3677644,3677645,3677647,3677660,3677661,3677662,3686408,3686410,3686411,3686414,3686415,3686498,3686499,3686502,
        3686503,3686514,3686515,3686702,3686703,3686715,3686718,3686854,3686855,3686866,3686867,3687578,3687579,3687582,3687583,
        3687626,3687653,3687654,3687655,3687670,3687671,3687843,3687846,3687864,3687866,3687867,3687869,3687870,3687871,3687914,
        3687915,3687918,3688276,3688278,3688279,3688304,3688305,3688306,3688396,3688397,3688398,3688408,3688609,3688612,3688613,
        3688614,3688624,3688625,3688636,3688637,3688680,3688685,3688700,3688701,3688702,3689217,3689220,3689221,3689222,3689232,
        3689233,3689284,3689285,3689287,3689288,3689293,3689308,3689309,3689310,3689796,3689828,3689832,3689908,3689909,3689916,
        3689918,3689920,3689922,3689928,3689930,3689992,3689993,3689995,3690020,3690021,3690036,3690037,3690039,3690084,3690085,
        3690104,3690105,3690108,3690109,3690260,3690262,3690264,3690265,3690281,3690288,3690290,3690296,3690298,3690373,3691016,
        7351167,7351247,7351251,7351267,7351273,7351275,7351821,7351822,7351823,7351826,7351827,7351836,7351838,7351839,7351841,
        7351842,7351843,7351883,7351898,7351903,7352074,7352113,7352114,7352115,7352116,7352118,7352119,7352160,7352162,7352163,
        7352167,7352178,7354687,7354703,7354716,7354718,7354719,7354723,7354780,7354781,7354783,7354805,7354807,7354811,7354924,
        7354926,7354927,7354938,7354939,7354942,7354943,7355094,7355095,7355143,7355148,7355149,7355151,7355172,7355173,7355175,
        7355181,7355267,7355403,7355406,7355407,7355445,7355446,7355447,7355494,7355507,7355510,7355688,7355690,7355691,7355693,
        7355694,7355695,7355704,7355706,7355707,7355844,7355846,7355847,7355858,7355862,7355863,7366786,7366798,7366799,7366812,
        7366814,7366815,7366880,7366882,7366883,7366896,7366898,7366899,7366902,7367074,7367075,7367078,7367098,7367099,7367146,
        7367493,7367494,7367495,7367508,7367510,7367511,7367668,7367669,7367670,7367672,7367673,7367674,7368200,7368202,7368224,
        7368226,7368232,7368330,7373117,7373118,7373119,7373133,7373134,7373135,7373153,7373154,7373155,7373201,7373213,7373215,
        7373239,7373243,7373315,7373316,7373318,7373319,7373341,7373342,7373343,7373378,7373379,7373382,7373402,7373403,7373575,
        7373576,7373578,7373579,7373581,7373583,7375496,7375497,7375527,7375531,7381024,7381025,7381027,7381028,7381029,7381040,
        7381041,7381049,7381050,7381051,7381125,7381126,7381127,7381157,7381167,7381507,7381544,7381545,7381547,7381635,7381641,
        7381643,7381665,7381667,7381676,7381677,7381688,7381689,7381690,7381692,7381693,7381694,7383108,7383124,7383125,7383296,
        7383297,7383300,7383301,7383312,7383313,7383315,7383360,7383361,7383373,7383380,7383381,7383383,7383384,7383385,7384072,
        7384073,7384075,7384076,7384077,7384078,7384089,7384117,7384165,7384176,7384177,7384179,7384180,7384181,7384309,7384311,
        7384316,7384317,7384319,7384352,7384353,7384355,7384361,7384363,7384448,7384449,7384451,7384457,7384458,7384459,7384661,
        7384662,7384663,7384699,7384792,7384793,7384821,7384992,7384993,7384996,7384997,7384999,7385016,7385017,7385064,7385065,
        7385080,7385081,7389036,7389037,7389038,7389040,7389041,7389042,7389045,7389126,7389160,7389162,7389528,7389529,7389530,
        7389552,7389554,7389560,7389562,7389649,7389650,7389651,7389652,7389654,7389655,7390343,7390344,7390346,7390347,7390360,
        7390362,7390363,7390410,7390411,7390414,7390450,7390453,7390454,7390455,7390626,7390638,7390639,7390649,7390650,7390651,
        7390652,7390654,7390655,7390703,7391040,7391042,7391043,7391056,7391058,7391059,7393795,7393798,7393818,7393819,7393822,
        7393823,7393866,7393894,7393895,7393906,7393918,7394089,7394090,7394091,7394094,7394194,7394197,7394198,7394199,7394242,
        7394252,7394254,7394255,7394265,7394266,7394267,7394981,7394982,7394983,7394998,7395046,7395049,7395050,7395051,7395066,
        7395067,7395242,7396612,7396614,7396615,7396630,7396631,7396680,7396682,7396683,7396708,7396710,7396816,7396818,7396819,
        7396844,7396848,7397184,7397186,7397198,7397220,7397222,7397224,7397225,7397226,7397228,7397268,7397269,7397270,7397300,
        7397302,7397354,7399968,7399969,7399973,7399984,7399985,7399987,7399996,7400032,7400033,7400035,7400044,7400045,7400047,
        7400056,7400057,7400058,7400061,7400321,7400324,7400325,7400337,7400340,7400341,7400343,7400388,7400389,7400391,7400392,
        7400408,7400409,7401096,7401097,7401125,7401136,7401137,7401138,7401184,7401185,7401187,7401196,7401197,7401200,7401201,
        7401202,7401204,7401205,7401376,7401381,7401401,7401404,7401452,7401453,7401455,7401468,7401469,7401471,7402780,7402816,
        7402839,7402848,7402850,7402851,7402854,7402855,7402866,7404772,7404776,7404788,7404789,7404960,7404964,7404965,7404967,
        7404985,7404986,7404987,7405116,7405122,7405123,7405152,7405200,7405208,7405210,7405211,7405236,7405238,7405244,7405245,
        7405246,7405317,7405318,7405319,7405324,7405325,7405327,7405359,7405443,7405472,7405473,7405475,7405484,7405485,7405487,
        7440506,7440507,7440510,7440565,7440566,7440567,7440569,7440571,7440581,7440582,7440583,7440585,7440586,7440587,7440770,
        7440774,7440775,7440786,7440798,7440842,7440843,7440846,7440882,7440886,7440887,7440913,7440925,7440927,7440943,7440947,
        7441037,7441069,7441070,7441071,7441073,7441074,7441075,7441570,7441582,7441594,7441595,7441598,7441599,7441986,7441990,
        7441991,7442002,7442003,7442012,7442014,7442172,7442304,7442306,7442336,7446561,7446563,7446570,7446571,7447632,7447634,
        7447646,7447668,7447670,7447672,7447673,7447674,7447748,7447749,7447750,7447758,7447780,7447782,7447783,7448132,7448136,
        7448142,7448176,7448178,7448179,7448354,7448364,7448366,7448367,7448378,7448512,7448514,7448515,7448518,7449552,7449553,
        7449554,7449556,7449557,7449566,7449596,7449597,7449598,7450249,7450252,7450253,7450254,7450265,7450293,7450336,7450337,
        7450341,7450361,7450364,7450365,7450367,7450536,7450537,7450540,7450541,7450543,7451274,7451301,7451302,7451303,7451318,
        7451319,7451369,7451370,7451371,7451386,7451387,7452756,7452758,7452759,7452932,7452934,7452935,7452952,7452954,7452955,
        7453000,7453002,7453003,7453006,7453028,7453030,7453031,7453041,7453042,7453043,7453044,7453046,7453047,7453736,7453738,
        7453739,7453741,7453742,7453743,7453752,7453754,7453755,7453890,7453891,7453894,7453895,7453918,7454090,7454091,7454094,
        7454130,7454134,7454135,7454178,7454188,7454190,7454203,7454206,7463903,7463917,7463918,7463919,7463923,7464585,7464586,
        7464587,7464602,7464631,7464674,7464675,7464684,7464686,7464687,7464702,7464703,7465215,7465239,7465243,7465253,7465254,
        7465255,7465257,7465258,7465259,7465281,7465282,7465283,7465344,7465345,7465347,7465379,7465381,7465382,7465383,7465472,
        7465474,7465475,7465478,7465500,7465502,7465503,7465553,7465555,7465561,7465584,7465586,7465587,7465589,7465590,7466240,
        7466242,7466243,7466258,7466268,7466270,7466271,7466337,7466338,7466339,7466352,7466354,7466366,7466832,7466833,7466834,
        7466836,7466846,7466876,7466877,7466878,7466956,7466960,7466961,7466980,7466982,7467002,7469184,7469186,7469198,7469220,
        7469222,7469224,7475201,7475216,7475217,7475220,7475221,7475223,7475264,7475265,7475289,7475464,7475465,7475480,7475481,
        7475509,7475552,7475553,7475568,7475569,7475571,7476256,7476257,7476269,7476280,7476281,7476328,7476329,7476331,7476421,
        7476437,7476508,7476509,7476510,7476532,7476536,7476544,7476546,7476612,7491584,7491585,7491587,7491600,7491601,7491602,
        7491613,7491660,7491661,7491662,7491672,7491673,7491675,7491700,7491701,7491872,7491873,7491874,7491876,7491877,7491879,
        7491897,7491900,7491901,7491945,7491964,7491965,7491966,7492037,7492048,7492049,7492620,7492621,7492622,7492624,7492625,
        7492626,7492640,7492641,7492642,7492701,7492702,7492703,7492721,7492732,7492733,7492735,7492829,7492830,7492831,7492849,
        7492851,7492867,7493205,7493376,7493377,7493380,7493392,7493393,7493405,7493448,7493453,7493489,7493492,7536972,7536973,
        7536974,7536976,7536977,7536978,7537016,7537018,7537100,7537104,7537120,7537122,7537134,7537185,7537197,7537208,7537213,
        7537272,7537273,7537275,7537348,7537364,7537365,7537367,7537460,7537461,7537462,7537464,7537465,7537466,7537476,7537478,
        7537480,7537481,7537482,7537540,7537936,7537937,7537939,7537940,7537984,7537985,7537987,7537996,7537997,7538008,7538009,
        7538010,7538012,7538013,7540769,7540772,7540773,7540774,7540785,7540797,7540799,7540844,7540845,7540945,7541120,7541121,
        7541133,7541148,7541149,7541150,7541236,7541237,7541239,7541928,7541929,7541931,7541944,7541945,7541947,7542292,7542293,
        7542295,7542345,7542361,7542389,7542564,7542565,7542567,7542584,7542585,7542588,7542633,7542724,7542725,7542736,7542737,
        7542741,7544277,7544279,7544285,7544659,7544661,7544662,7544663,7544665,7544667,7544689,7544700,7544701,7544703,7544787,
        7553673,7553676,7553677,7553688,7553689,7553709,7553710,7553711,7553713,7553714,7553715,7554376,7554377,7554378,7554384,
        7554400,7554402,7554403,7554406,7554424,7554426,7554427,7554430,7554431,7555076,7555077,7555079,7555111,7555115,7555121,
        7555122,7555123,7555208,7555209,7555211,7555233,7555241,7555242,7555243,7555744,7555745,7555747,7555748,7555749,7555750,
        7555760,7555761,7555763,7555820,7555821,7555823,7556488,7556489,7556490,7556492,7556493,7556512,7556514,7556515,7556520,
        7556521,7556522,7556528,7556529,7556531,7556576,7556577,7556579,7556588,7556589,7556591,7556604,7556605,7556607,7556692,
        7556693,7556694,7556696,7556732,7556734,7556812,7556813,7556814,7556820,7556824,7556844,7556846,7557162,7557254,7557266,
        7557267,7557322,7557323,7557326,7557327,7557366,7557562,7557563,7557909,7557910,7557911,7557952,7557954,7557955,7557958,
        7557959,7557976,7557978,7557979,7557982,7557983,7558655,7558666,7558692,7558694,7558695,7558706,7558707,7558710,7558760,
        7558762,7558763,7558870,7558871,7558997,7558999,7559038,7559039,7559066,7559067,7559092,7559094,7559095,7559121,7559122,
        7559123,7559136,7559138,7559139,7559142,7559153,7559155,7559160,7559162,7559163,7559680,7559681,7559693,7559708,7559709,
        7559711,7559776,7559777,7559780,7559793,7559795,7559804,7559805,7559976,7559977,7559981,7560081,7560083,7560128,7560129,
        7560141,7560143,7560156,7560157,7560158,7560865,7560877,7560888,7560893,7561285,7561305,7561481,7561509,7561520,7561525,
        7561527,7561577,7561580,7561581,7561582,7561593,7561685,7562296,7562297,7562298,7562300,7562316,7562317,7562318,7562336,
        7562372,7562373,7562374,7566401,7566404,7566405,7566417,7566429,7566600,7566601,7566602,7566604,7566605,7566607,7566629,
        7566640,7566641,7566643,7566653,7566654,7566655,7566727,7566730,7566731,7566761,7566763,7566942,7566943,7566961,7566963,
        7566969,7567060,7567061,7567063,7567067,7567077,7567078,7567079,7567084,7567085,7567087,7567107,7568453,7568473,7568476,
        7568477,7568479,7568501,7568681,7568684,7568685,7568697,7568789,7568832,7568833,7568837,7568857,7568860,7568861,7568862,
        7733343,7733357,7733358,7733359,7733363,7733399,7733402,7733403,7733413,7733414,7733415,7733441,7733442,7733443,7733514,
        7733542,7733562,7733563,7733564,7733566,7733567,7733696,7733698,7733699,7733714,7733715,7733718,7733724,7733726,7733727,
        7733793,7733795,7734415,7734448,7734450,7734451,7734460,7734462,7734463,7734508,7734510,7734864,7734866,7734996,7734997,
        7734998,7735000,7735026,7735072,7735074,7735075,7736396,7736397,7736398,7736404,7736408,7736416,7736417,7736418,7736508,
        7736510,7736511,7736516,7736518,7736524,7736525,7736526,7736544,7736554,7736896,7736898,7736899,7736901,7736902,7736903,
        7736912,7736914,7736915,7736924,7736926,7736927,7737102,7737103,7737136,7737138,7737139,7737143,7737186,7737198,7737211,
        7737300,7737302,7737303,7739870,7739871,7739879,7739881,7739882,7739883,7740038,7740056,7740058,7740063,7740106,7740132,
        7740134,7740135,7740152,7740154,7740155,7740173,7740175,7740179,7740197,7740216,7740217,7740219,7740295,7740298,7740299,
        7740305,7740306,7740307,7740474,7740475,7740520,7740522,7740547,7740549,7740550,7740551,7740614,7740626,7740627,7740630,
        7740810,7740832,7740834,7740835,7740860,7740862,7740863,7740905,7740906,7740907,7741264,7741266,7741267,7741271,7742334,
        7742420,7742422,7742423,7742448,7742449,7742450,7742452,7743020,7743021,7743023,7743112,7743121,7743124,7743125,7743169,
        7743181,7743192,7743193,7743195,7743196,7743197,7743198,7743392,7743393,7743421,7743624,7743625,7743627,7743653,7743655,
        7743668,7743669,7743671,7743720,7743721,7743723,7743824,7743826,7743827,7743848,7743849,7743850,7743856,7743932,7743933,
        7743934,7743948,7743950,7743972,7743976,7744068,7744081,7744093,7744264,7744265,7744266,7744268,7744272,7744273,7744836,
        7744837,7744848,7744849,7744861,7745527,7746940,7746941,7746942,7747028,7747030,7747031,7747064,7747065,7747066,7747068,
        7747408,7747410,7747411,7747592,7747593,7747636,7747637,7747639,7747680,7747681,7747682,7747693,7747708,7747709,7747711,
        7747807,7747819,7747821,7747822,7747823,7747826,7747827,7747968,7747969,7747971,7747977,7747978,7747979,7748111,7748118,
        7748119,7748121,7748122,7748123,7748133,7748135,7748235,7748237,7748238,7748239,7748241,7748243,7749432,7749434,7749520,
        7749522,7749523,7749525,7749532,7749534,7749548,7749549,7749552,7749553,7749554,7749556,7749568,7749569,7749584,7749585,
        7749587,7749596,7749597,7749599,7752194,7752195,7752198,7752204,7752206,7752207,7752240,7752242,7752248,7752250,7752342,
        7752348,7752350,7752368,7757828,7757830,7757860,7757962,7757984,7758348,7758350,7758351,7758362,7758384,7758386,7758387,
        7758389,7758390,7758391,7758432,7758434,7758435,7758458,7758462,7758463,7758726,7758745,7758746,7758747,7758794,7758822,
        7758834,7758835,7758838,7759530,7760549,7760616,7760617,7760619,7760632,7760633,7760635,7760901,7760902,7760903,7760918,
        7761008,7761010,7761011,7761184,7761186,7761198,7761199,7761208,7761210,7761211,7761214,7761345,7761346,7761347,7761362,
        7761374,7764106,7764107,7764110,7764144,7764146,7764147,7764157,7764158,7764159,7764204,7764206,7764207,7764560,7764562,
        7764563,7764566,7764567,7764738,7764739,7764748,7764750,7764751,7764767,7764838,7764844,7764846,7764847,7764858,7764859,
        7764860,7764862,7764863,7765546,7765638,7765651,7765710,7765711,7765744,7765746,7765747,7765934,7770113,7770124,7770125,
        7770127,7770160,7770161,7770163,7770169,7770172,7770173,7770174,7770216,7770220,7770221,7770223,7770309,7770324,7770325,
        7770520,7770521,7770522,7770524,7770525,7770527,7770549,7770601,7770616,7770617,7770619,7770964,7770965,7770967,7771348,
        7771350,7771388,7771389,7771390,7771398,7771404,7771406,7771407,7771428,7771429,7771430,7771432,7771433,7771530,7771552,
        7771554,7771648,7771649,7771650,7771652,7771653,7771655,7771661,7771697,7771700,7771716,7771717,7771718,7771720,7771721,
        7771728,7782669,7782671,7782708,7782709,7782752,7782753,7782765,7782776,7782777,7782779,7782780,7782781,7782782,7783553,
        7783556,7783557,7783559,7783577,7783581,7783652,7783653,7783664,7783665,7783848,7783849,7784212,7784213,7784215,7784265,
        7784268,7784269,7784280,7784281,7784308,7784309,7787040,7787041,7787044,7787045,7787047,7787064,7787065,7787068,7787069,
        7787201,7787204,7787205,7787216,7787217,7787219,7787228,7787229,7787441,7787500,7787501,7787503,7787840,7787842,7787843,
        7787852,7787854,7787855,7787866,7787867,7787870,7788885,7789592,7789593,7789595,7789665,7789668,7789669,7789671,7789684,
        7789685,7789686,7789688,7789689,7789865,7789880,7789881,7789883,7789972,7789973,7789975,7789981,7790024,7790025,7790040,
        7790041,7790043,7790068,7790069,7831586,7831587,7831590,7831591,7831608,7831610,7831611,7831614,7831615,7831658,7831750,
        7831762,7831763,7831766,7831767,7831950,7831951,7831962,7832034,7832038,7832056,7832058,7832059,7832404,7832406,7832407,
        7833092,7833094,7833095,7833106,7833117,7833118,7833119,7833162,7833163,7833188,7833190,7833191,7833203,7833206,7833207,
        7833390,7833402,7833492,7833494,7833495,7833544,7833546,7833547,7833560,7833562,7833563,7833566,7833760,7833764,7833765,
        7833767,7833788,7833789,7833790,7833832,7833833,7833835,7834196,7834197,7834199,7834376,7834377,7834379,7834401,7834404,
        7834405,7834420,7834421,7834423,7834472,7834473,7834476,7834488,7834489,7834491,7835264,7835265,7835266,7835268,7835269,
        7835271,7835336,7835337,7835339,7835364,7835365,7835367,7835373,7835384,7835385,7835560,7835561,7835563,7836321,7836322,
        7836323,7836326,7836327,7836344,7836346,7836347,7836350,7836351,7836394,7836395,7837778,7837779,7837782,7837783,7837966,
        7838004,7838006,7838007,7838050,7838072,7838074,7838075,7838078,7838849,7838850,7838851,7838852,7838854,7838855,7838866,
        7838878,7838879,7838922,7838923,7838926,7838962,7838966,7838967,7839150,7839162,7839163,7839554,7839582,7841025,7841028,
        7841029,7841040,7841044,7841045,7841047,7841096,7841097,7841099,7841100,7841101,7841102,7841112,7841113,7841115,7841140,
        7841141,7841143,7843876,7843877,7843879,7843896,7843897,7843899,7843944,7843945,7843947,7843988,7844036,7844037,7844039,
        7844056,7844057,7844232,7844233,7844235,7844260,7844261,7844272,7844276,7844277,7844279,7844328,7844329,7844331,7844332,
        7844333,7844334,7844344,7844345,7844347,7844692,7844693,7844695,7845380,7845381,7845383,7845400,7845401,7845403,7845429,
        7845477,7845479,7845496,7845497,7845500,7845501,7845676,7845677,7845678,7845777,7845789,7845837,7845873,7845876,7850546,
        7850547,7850556,7850558,7850559,7850600,7850602,7850603,7850606,7850710,7850711,7850888,7850890,7850891,7850906,7850907,
        7850934,7850935,7850982,7851000,7851002,7851003,7851005,7851006,7851007,7851690,7851691,7856801,7856813,7856824,7856825,
        7856829,7857156,7857158,7857159,7857170,7857171,7857174,7857175,7857224,7857226,7857227,7857231,7857242,7857243,7857446,
        7857447,7857514,7857515,7857604,7857606,7857607,7857618,7857623,7858241,7858244,7858245,7858247,7858256,7858257,7858269,
        7858440,7858441,7858443,7858444,7858445,7858481,7858528,7858529,7858530,7859332,7859333,7859335,7859348,7859349,7859351,
        7859400,7859401,7859403,7859428,7859429,7859444,7859445,7859447,7859624,7859625,7859640,7859641,7859643,7859988,7859989,
        7860032,7860036,7860037,7860039,7860056,7860057,7860060,7860366,7860450,7860451,7860478,7860479,7861010,7861011,7861020,
        7861022,7861023,7861068,7861070,7861071,7861104,7861106,7861107,7861792,7861794,7861795,7861798,7861817,7861818,7861819,
        7861820,7861822,7861823,7861866,7861957,7861958,7861959,7861968,7861970,7861971,7861974,7862153,7862154,7862155,7862156,
        7862158,7862159,7862170,7862171,7862196,7862198,7862199,7862246,7862247,7862264,7862266,7862267,7862792,7862793,7862795,
        7862820,7862821,7862832,7862833,7862837,7862889,7862892,7862893,7862894,7862905,7862997,7862999,7863172,7863173,7863193,
        7863240,7863241,7863269,7863284,7863285,7863286,7863980,7863981,7863983,7863992,7863993,7863994,7863996,7863997,7863999,
        8213568,8213569,8213571,8213572,8213573,8213574,8213584,8213585,8213587,8213792,8213808,8213809,8213811,8213856,8213857,
        8213859,8213880,8213881,8213882,8213973,9559751,9559757,9559759,9559762,9559763,9559766,9559767,9559851,9559854,9559855,
        9559866,9559867,9559869,9559870,9559871,9559927,9565249,9565251,9565257,9565260,9565261,9565272,9565273,9565456,9565457,
        9565458,9565460,9565461,9565504,9565505,9565508,9565509,9700699,9700711,9700789,9700790,9700791,9700793,9700794,9700795,
        9700801,9700802,9700803,9700991,9701077,9701078,9701079,9701083,9701105,9701107,9701113,9701115,9701127,9701133,9701135,
        9701157,9701158,9701159,9701308,9701309,9701352,9701353,9701356,9701357,9701368,9702481,9704100,9704101,9704102,9704116,
        9704117,9704118,9704160,9704161,9704164,9704165,9704176,9704180,9704181,9704348,9704349,9704350,9704352,9704356,9704392,
        9704393,9704396,9704404,9704405,9704406,9704532,9704534,9704540,9704542,9704556,9704558,9704560,9704561,9704628,9704630,
        9704636,9704638,9704644,9704646,9704648,9704649,9704976,9704977,9704978,9705088,9734690,9734691,9734694,9734695,9734706,
        9734707,9734747,9734750,9734751,9734840,9734841,9734844,9734845,9734888,9734889,9734892,9734922,9734923,9734926,9734927,
        9734964,9734966,9734972,9735064,9735065,9735066,9735076,9735077,9735078,9735088,10616815,10616825,10616826,10616827,10648315,
        10648318,10648319,10648437,10648438,10648439,10648443,10648477,10648479,10648489,10648490,10648491,10648513,10648514,10648515,10648823,
        10648895,10648938,10648939,10648942,10648943,10648969,10648970,10648971,10649103,10649107,10649110,10649111,10649121,10649122,10649123,
        10649153,10649154,10649155,10657663,10657723,10657725,10657726,10657727,10657741,10657742,10657743,10657747,10657761,10657762,10657763,
        10661749,10661750,10661751,10661753,10661754,10661755,10661813,10661814,10661815,10661817,10661818,10661819,10661829,10661830,10661831,
        10661833,10661834,10661835,10662751,10662771,10662815,10662831,10662837,10662838,10662839,10662851,10663031,10663034,10663035,10663094,
        10663095,10663097,10663098,10663099,10663109,10663110,10663111,10663113,10663114,10663115,10663175,10663179,10663429,10663430,10663431,
        10663433,10663434,10663435,10664047,10664051,10664095,10664115,10664121,10664122,10664123,10664133,10664134,10664135,10664195,10664451,
        10665340,10665341,10665342,10665404,10665405,10665406,10665420,10665421,10665422,10665432,10665448,10665588,10665589,10665590,10665596,
        10665652,10665654,10665656,10665657,10665658,10665672,10665673,10665676,10665677,10665678,10665680,10665682,10665736,10665737,10665738,
        10665740,10665741,10665742,10665744,10665745,10665746,10665752,10667003,10667005,10667006,10667007,10667451,10667469,10667470,10667471,
        10667615,10667627,10667637,10667638,10667639,10667663,10667685,10667687,10667713,10667715,10668254,10668255,10668263,10668269,10668270,
        10668271,10668335,10668341,10668342,10668343,10668361,10668362,10668363,10668419,10668565,10668566,10668567,10668569,10668570,10668571,
        10668579,10668581,10668582,10668583,10669948,10669949,10669950,10670000,10670001,10670002,10670004,10670020,10670021,10670024,10670025,
        10670026,10670032,10670111,10670122,10670123,10670126,10670127,10670154,10670155,10670157,10670158,10670159,10671612,10671613,10671740,
        10671741,10671742,10671788,10671790,10671796,10671797,10671816,10671818,10671824,10671825,10671826,10671828,10671896,10671897,10671898,
        10671900,10671901,10671912,10671913,10671920,10671940,10671942,10671944,10672364,10672366,10672368,10672369,10672436,10672437,10672438,
        10672444,10672446,10672472,10672474,10672480,10672481,10672484,10672512,10672513,10672514,10672516,10672517,10672520,10672522,10672660,
        10672661,10672664,10672665,10672666,10672672,10672673,10672676,10673152,10673153,10674908,10674910,10674916,10674917,10674918,10674928,
        10674988,10674989,10674990,10674996,10674997,10674998,10675012,10675013,10675014,10675016,10675024,10675072,10675073,10675074,10675076,
        10675284,10675285,10675286,10675292,10675296,10675297,10675298,10675300,10675301,10675304,10675306,10675340,10675341,10675342,10675344,
        10675345,10675348,10675349,10675360,10675361,10675362,10675456,10676244,10676245,10676248,10676249,10676264,10676265,10676266,10676288,
        10689262,10689263,10689274,10689275,10689278,10689279,10689406,10689407,10689437,10689439,10689441,10689443,10689474,10689475,10689478,
        10689479,10689490,10689491,10689965,10689967,10690010,10690011,10690014,10690015,10690079,10690097,10690099,10690122,10690123,10690126,
        10690127,10690130,10690131,10690134,10690135,10690306,10690307,10691295,10691302,10691303,10691305,10691306,10691307,10691314,10691315,
        10691317,10691318,10691319,10691390,10691391,10691434,10691435,10691437,10691438,10691439,10691473,10691475,10691807,10691815,10691817,
        10691819,10691826,10691827,10691829,10691830,10691831,10691903,10691946,10691947,10691950,10691951,10691955,10691957,10691958,10691959,
        10691977,10691978,10691979,10692102,10692103,10692615,10693612,10693613,10693624,10694124,10694125,10694136,10694256,10694257,10694258,
        10694260,10694261,10694264,10694284,10694285,10694286,10694292,10694294,10694296,10694297,10694412,10694413,10694414,10694432,10694433,
        10694464,10694465,10694669,10694671,10694675,10694678,10694679,10694722,10694723,10695676,10695677,10696316,10696317,10696318,10696356,
        10696357,10696360,10696368,10696369,10696400,10696401,10696402,10696404,10696488,10696489,10696492,10696493,10696524,10696525,10696526,
        10696532,10696533,10696535,10696536,10696544,10696824,10696825,10696826,10696852,10696853,10696854,10696856,10696857,10696860,10696864,
        10696896,10696897,10696984,10696985,10696988,10696992,10696993,10696994,10696996,10696997,10697028,10697030,10699775,10700469,10700471,
        10700490,10700491,10700494,10700495,10700503,10700506,10700507,10700509,10700510,10700511,10700605,10700607,10700643,10700646,10700647,
        10700658,10700659,10700661,10700662,10700663,10700917,10700919,10700931,10700933,10700934,10700935,10700937,10700938,10700939,10700946,
        10700947,10700949,10700950,10700951,10701066,10701067,10701070,10701071,10701078,10701079,10701082,10701083,10701120,10701122,10701123,
        10701126,10701127,10702580,10702581,10702584,10702585,10702716,10702717,10702752,10702753,10702756,10702757,10702800,10702801,10703102,
        10703103,10703270,10703271,10703282,10703283,10703286,10703287,10703302,10703303,10703305,10703307,10703314,10703315,10703318,10703319,
        10703370,10703371,10703373,10703374,10703375,10703918,10703919,10703929,10703930,10703931,10704001,10704002,10704003,10704138,10704139,
        10704141,10704142,10704143,10704146,10704147,10704150,10704151,10704194,10704195,10704197,10704198,10704199,10705340,10705392,10705393,
        10705394,10705396,10705448,10705449,10705452,10705453,10705492,10705493,10705496,10705497,10705664,10705665,10705668,10705669,10705680,
        10706044,10706046,10706056,10706057,10706060,10706061,10706068,10706069,10706070,10706072,10706112,10706208,10706209,10706210,10706212,
        10706213,10706214,10706224,10706225,10706228,10706248,10706249,10706252,10706253,10707116,10707117,10707118,10707128,10707129,10707132,
        10707160,10707256,10707257,10707258,10707292,10707293,10707312,10707313,10707328,10707329,10707456,10707457,10707460,10707976,10707977,
        10707980,10707981,10715256,10715257,10715258,10715284,10715285,10715286,10715292,10715296,10715298,10715328,10715329,10715424,10715425,
        10715428,10715456,10715457,10788790,10788791,10788793,10788795,10788829,10788831,10788833,10788834,10788835,10788836,10788838,10788839,
        10788850,10788851,10789615,10789617,10789618,10789619,10789758,10789759,10789782,10789785,10789786,10789787,10789827,10789841,10789842,
        10789843,10790378,10790379,10790382,10790383,10790442,10790443,10790446,10790447,10790457,10790458,10790459,10790493,10790494,10790495,
        10790496,10790498,10790499,10790513,10790514,10790515,10790659,10790677,10790678,10790679,10790744,10790746,10790868,10790872,10790874,
        10790875,10790988,10790989,10791000,10791001,10791023,10791025,10791027,10791117,10791128,10791129,10791155,10791493,10791495,10791501,
        10791512,10791513,10791516,10791517,10791519,10791539,10791545,10791546,10791547,10791582,10791583,10791621,10791623,10791626,10791627,
        10791630,10791631,10792276,10792280,10792282,10792283,10792286,10792404,10792414,10792415,10793506,10793512,10793632,10793634,10793640,
        10794247,10794253,10794264,10794265,10794293,10794294,10794295,10794297,10794298,10794299,10794375,10794381,10794392,10794393,10794421,
        10794422,10794423,10794425,10794426,10794427,10794666,10794667,10794670,10794671,10794679,10794682,10794683,10794685,10794686,10794687,
        10794721,10794722,10794723,10794759,10794765,10794776,10794777,10794780,10794781,10794783,10794805,10794806,10794807,10794809,10794810,
        10794811,10794886,10794887,10794889,10794890,10794891,10795040,10795042,10795048,10795066,10795067,10795070,10795071,10795114,10795115,
        10795118,10795119,10795122,10795123,10795141,10795142,10795143,10795289,10795291,10795335,10795346,10795347,10796772,10796773,10796774,
        10796776,10796784,10796785,10796788,10796789,10796940,10796941,10796952,10796953,10796960,10796977,10796979,10796985,10796988,10796989,
        10796991,10799722,10799723,10799726,10799738,10799743,10799767,10799907,10799910,10799911,10799922,10799923,10799926,10799927,10799947,
        10800643,10800649,10800650,10800651,10800704,10800706,10800712,10800714,10800715,10800718,10800748,10800749,10800750,10800752,10800754,
        10800832,10800833,10800834,10800842,10800843,10800846,10800847,10800876,10803961,10803962,10803963,10804126,10804127,10804130,10804131,
        10804134,10804135,10804169,10804170,10804171,10804172,10804174,10804175,10804179,10804182,10804183,10804797,10804798,10804799,10804838,
        10804839,10804850,10804851,10804853,10804854,10804855,10805003,10805006,10805007,10805016,10805018,10805019,10805021,10805022,10805023,
        10805058,10805059,10805061,10805062,10805063,10806336,10806337,10806338,10806346,10806347,10806350,10806380,10806399,10806464,10806466,
        10806467,10806470,10806471,10806482,10806483,10806485,10806486,10806487,10806569,10806570,10806571,10806607,10806618,10806619,10806622,
        10806623,10806625,10806627,10808536,10808538,10808539,10808556,10808557,10808558,10808560,10808562,10808632,10808633,10808636,10808672,
        10808673,10808674,10808676,10808677,10808688,10808692,10808908,10808910,10808911,10808922,10808923,10808940,10808941,10808942,10808944,
        10808945,10808946,10809032,10809034,10809035,10809038,10809039,10809068,10809069,10809070,10809072,10809074,10809351,10809354,10809355,
        10809362,10809363,10810092,10810093,10810094,10810104,10810105,10810109,10810276,10810277,10810278,10810280,10810316,10810317,10810318,
        10810320,10810321,10810322,10810324,10810325,10810876,10810877,10810878,10810936,10810937,10810938,10810988,10810992,10810993,10810994,
        10811008,10811009,10811012,10811013,10811156,10811157,10811158,10811160,10811161,10811162,10811200,10811201,10811204,10811205,10811216,
        10811217,10811357,10811359,10811396,10811397,10811398,10811400,10811401,10811402,10811408,10811409,10811412,10811413,10811456,10811457,
        10812765,10812767,10828735,10828751,10828755,10828758,10828759,10828771,10829311,10829371,10829374,10829375,10829418,10829419,10829422,
        10829423,10829427,10829442,10829443,10829446,10829447,10829595,10829606,10829607,10829651,10840234,10840235,10840238,10840239,10840247,
        10840250,10840251,10840290,10840291,10840294,10840295,10840306,10840307,10840310,10840318,10840319,10840692,10840696,10840772,10840774,
        10840775,10840792,10840794,10840795,10840812,10840813,10840814,10840816,10840818,10840832,10840834,10840835,10840840,10841549,10841550,
        10841551,10841554,10841555,10841558,10841569,10841570,10841571,10841911,10841915,10841920,10841921,10841923,10841929,10841930,10841931,
        10842008,10842009,10842011,10842041,10842042,10842043,10844290,10844291,10844294,10844295,10844310,10844358,10844381,10844382,10844383,
        10844546,10844556,10844558,10844559,10844571,10844618,10844619,10844623,10844634,10844635,10844638,10844639,10845180,10845181,10845182,
        10845326,10845327,10845360,10845361,10845362,10845368,10845369,10845708,10845710,10845712,10845732,10845736,10845738,10845739,10845742,
        10845830,10845832,10845833,10845835,10846650,10846651,10846652,10846654,10846655,10846696,10846698,10846699,10846701,10846702,10846703,
        10846705,10846706,10846707,10846783,10846797,10846798,10846799,10846802,10846803,10846806,10846807,10846817,10846819,10846867,10846872,
        10846873,10846875,10846909,10846910,10846911,10846978,10846979,10846981,10846982,10846983,10847358,10847359,10847383,10847385,10847386,
        10847387,10847426,10847441,10847442,10847443,10847530,10847531,10847534,10847535,10847542,10847543,10847545,10847546,10847547,10847584,
        10847586,10847587,10847590,10847591,10847602,10847608,10847609,10847610,10847708,10847709,10847710,10847728,10847729,10847730,10847736,
        10847738,10847739,10848092,10848108,10848110,10848111,10848112,10848122,10848212,10848214,10848216,10848217,10848218,10848228,10848229,
        10848230,10848245,10848246,10848247,10848273,10848274,10848275,10848311,10848315,10848391,10848397,10848426,10848427,10848437,10848438,
        10848439,10850236,10850268,10850269,10850271,10850280,10850281,10850288,10850434,10850585,10850586,10850587,10850608,10850609,10850612,
        10850613,10850621,10850622,10850623,10850701,10850703,10850705,10850706,10850707,10850725,10850736,10850737,10850740,10850741,10850743,
        10850747,10850958,10850959,10850970,10850971,10850973,10850974,10850975,10851011,10851014,10851015,10851079,10851108,10851109,10851114,
        10851115,10851117,10851118,10851119,10851120,10851121,10853004,10853005,10853006,10853012,10853013,10853016,10853056,10853057,10853059,
        10853080,10853081,10853083,10853095,10853099,10854204,10854206,10854207,10854249,10854252,10854253,10854256,10854257,10854258,10854260,
        10854261,10854263,10854300,10854301,10854302,10854328,10854336,10854338,10854573,10854574,10854575,10854913,10854915,10854948,10854949,
        10854951,10854957,10854958,10854959,10855041,10855042,10855043,10855076,10855077,10855079,10855083,10855732,10855733,10855734,10855744,
        10855746,10855747,10855752,10855753,10855754,10855860,10855864,10855866,10855872,10855874,10855880,10856180,10856182,10856184,10856185,
        10856186,10856240,10856241,10856242,10856324,10856328,10856449,10856450,10856451,10856480,10856481,10856484,10856485,10856487,10856491,
        10856572,10856573,10856574,10856588,10856596,10856597,10856598,10856600,10856736,10856737,10856738,10856752,10856753,10856754,10856756,
        10856776,10856777,10856780,10856781,10856792,10856793,10856796,10857472,10857473,10857475,10857488,10857489,10857490,10857537,10857540,
        10857541,10858593,10858601,10858602,10858603,10858645,10858647,10858651,10858673,10858675,10858684,10858685,10858687,10859029,10859030,
        10859031,10859033,10859034,10859035,10859045,10859046,10859047,10859141,10859173,10859184,10859185,10859187,10859188,10859189,10859190,
        10859232,10859234,10859237,10859248,10859249,10859250,10859252,10859400,10859401,10859403,10859416,10859417,10859418,10859420,10859421,
        10859428,10859429,10859441,10859480,10859484,10859485,10859800,10859802,10859803,10859820,10859821,10859822,10859824,10859826,10859912,
        10859914,10859915,10859942,10859948,10860124,10860126,10860140,10860141,10860142,10860144,10860145,10860146,10860168,10860169,10860170,
        10860172,10860184,10860185,10860188,10860189,10860190,10860228,10860232,10860288,10862656,10862657,10862658,10862688,10862690,10862691,
        10862696,10862697,10862698,10862740,10862741,10862742,10862744,10862745,10862746,10862774,10862775,10862832,10862834,10862835,10863025,
        10863026,10863027,10863078,10863079,10863776,10863778,10863779,10863796,10863798,10863799,10863839,10863841,10863842,10863843,10863877,
        10863878,10863879,10863881,10863882,10863883,10863905,10863907,10864003,10865188,10865189,10865190,10865192,10865193,10865200,10865201,
        10865213,10865252,10865253,10865256,10865264,10865265,10865267,10865440,10865442,10865444,10865445,10865447,10865460,10865461,10865462,
        10865505,10865508,10865509,10865511,10865524,10865631,10866208,10866209,10866220,10866221,10866223,10866305,10866307,10867024,10867025,
        10867026,10867032,10867034,10867035,10867038,10867062,10867064,10867066,10867067,10867148,10867150,10867152,10867168,10867169,10867170,
        10867896,10867898,10867899,10867901,10867902,10867903,10867946,10867947,10867950,10867951,10867966,10867967,10868138,10868139,10868142,
        10868155,10868158,10868544,10868546,10868547,10868549,10868550,10868551,10868566,10869372,10869373,10869464,10869465,10869466,10869476,
        10869478,10869479,10869480,10869481,10869482,10869484,10869544,10869545,10869547,10869700,10869701,10869717,10870401,10870404,10870405,
        10870416,10870417,10870420,10870421,10870423,10870464,10870465,10870466,10870468,10870469,10870471,10870484,10870485,10870486,10870669,
        10870681,10870685,10870728,10870729,10870731,10870732,10870733,10870745,10870748,10870749,10871125,10871133,10871135,10871151,10871155,
        10871237,10871239,10871256,10871257,10871259,10871277,10871278,10871279,10871281,10871282,10871283,10871617,10871618,10871619,10871624,
        10871625,10871627,10871649,10871748,10871749,10871751,10871764,10871765,10871766,10872448,10872449,10872451,10872464,10872465,10872466,
        10872468,10872469,10872471,10872528,10872529,10872541,10872708,10872720,10872721,10872723,10872732,10872768,10872769,10872770,10872788,
        10872789,10872792,10874958,10874980,10874982,10874984,10874985,10875028,10875036,10875038,10875106,10875412,10875416,10875428,10875429,
        10875430,10875432,10875433,10875434,10876767,10876789,10876791,10876797,10879676,10879677,10879712,10879713,10879714,10879716,10879717,
        10879728,10879732,10879743,10879905,10879907,10880860,10880862,10880863,10881113,10881116,10881117,10881119,10881237,10881239,10881244,
        10881245,10881247,10881313,10881315,10881321,10881322,10881323,10881445,10881446,10881447,10881449,10881451,10881623,10881629,10881829,
        10881840,10881841,10881844,10881845,10881881,10881884,10881885,10881892,10882284,10882286,10882288,10882289,10882292,10882293,10882396,
        10882397,10882398,10882420,10882424,10882426,10882461,10882464,10882465,10882466,10882468,10882500,10882501,10882502,10882504,10882505,
        10882564,10882565,10882566,10882568,10882569,10882572,10883208,10883209,10883210,10883212,10883213,10883216,10883217,10883218,10883220,
        10883221,10883224,10883238,10883240,10883241,10883242,10883264,10883265,10883269,10883356,10883357,10883358,10883364,10883365,10883366,
        10883376,10883377,10883378,10883380,10883400,10883401,10883404,10883405,10883416,10883417,10883584,10883620,10883622,10883628,10883712,
        10883748,10883750,10883756,10884381,10884383,10884403,10884408,10884409,10884411,10884417,10884418,10884419,10884501,10884509,10884511,
        10884531,10884537,10884538,10884539,10884881,10884885,10884895,10884917,10884919,10884956,10884957,10884958,10884964,10884965,10884967,
        10884976,10884977,10884978,10885120,10885121,10885122,10895608,10895609,10895610,10895613,10895785,10895792,10895832,10895841,10895880,
        10895881,10895884,10895885,10895886,10895892,10895893,10895896,10895936,10895937,10895940,10895941,10896436,10896438,10896440,10896441,
        10896442,10896456,11011029,11011030,11011031,11011033,11011034,11011035,11011045,11011046,11011047,11011051,11011191,11011195,11011262,
        11011263,11011271,11011277,11011278,11011279,11011299,11011335,11011339,11011599,11011601,11011603,11011625,11011626,11011627,11012319,
        11012341,11012342,11012343,11012345,11012346,11012347,11012381,11012382,11012383,11012397,11012398,11012399,11012401,11012402,11012403,
        11012481,11012483,11012663,11012669,11012670,11012671,11012677,11012678,11012679,11012681,11012683,11012749,11012750,11012751,11012754,
        11012755,11012771,11013496,11013498,11013501,11013580,11013600,11013601,11013602,11013608,11013756,11013757,11013758,11013848,11013850,
        11013860,11013861,11013864,11013900,11013901,11013902,11013904,11013908,11013924,11014492,11014493,11014494,11014516,11014520,11014521,
        11014522,11014580,11014581,11014582,11014584,11014586,11014592,11014593,11014594,11014597,11014600,11014780,11014781,11014782,11014844,
        11014845,11014846,11014860,11014861,11014862,11014864,11014865,11014866,11014880,11014881,11014882,11014924,11014926,11014928,11014944,
        11014945,11015176,11015177,11015179,11015180,11015181,11015184,11016204,11016208,11016209,11016210,11016224,11016225,11016226,11018716,
        11018717,11018732,11018733,11018734,11018736,11018738,11019004,11019006,11019036,11019037,11019038,11019060,11019064,11019065,11019066,
        11019076,11019141,11019144,11019146,11019284,11019285,11019286,11019288,11019289,11019291,11019292,11019300,11019301,11019302,11019392,
        11019394,11020348,11020349,11020351,11020364,11020366,11020384,11020385,11020386,11020392,11020432,11020433,11020434,11020440,11020452,
        11020453,11020454,11020456,11020457,11020460
    };
  }
  
  
  @Test
  public void test4() {
    int order = 10;
    int nCells = 0;
    for (final int[] values : mocGlimpse.values()) {
      nCells += values.length;
    }
    long[] mocVals = new long[nCells];
    int i = 0;
    for (final Map.Entry<Integer, int[]> e : mocGlimpse.entrySet()) {
      int o = e.getKey();
      for (final int v : e.getValue()) {
        mocVals[i++] = HealpixNestedBMOC.buildValue(o, v, false, order);
      }
    }
    Arrays.sort(mocVals);
    
    HealpixNestedBMOC bmoc = HealpixNestedBMOC.createUnsafe(order, mocVals);
    System.out.println("bmoc size: " + bmoc.size());
    
    long l1 = System.nanoTime();
    byte[] bytes = HealpixNestedUltraCompactMOC.compress(bmoc);
    long l2 = System.nanoTime();
    // final String s = HealpixNestedUltraCompactMOC.compressB64(bytes);
    long l3 = System.nanoTime();
    bmoc = HealpixNestedUltraCompactMOC.decompress(order, bytes);
    long l4 = System.nanoTime();
    
    // System.out.println(s);
    System.out.println("bmoc size: " + bmoc.size());
    
    bytes = HealpixNestedUltraCompactMOC.compress(bmoc);
    // System.out.println(HealpixNestedUltraCompactMOC.compressB64(bytes));
    
    System.out.println("Compress: " + (l2 - l1) / 1.0e6 + " ms. nBytes: " + bytes.length);
    System.out.println("Base64: " + (l3 - l2) / 1.0e6 + " ms");
    System.out.println("DeCompress: " + (l4 - l3) / 1.0e6 + " ms");
  }
  
  
  /* TEST REMOVED TO STAY COMPATIBLE WITH JAVA 6
  @Test
  public void test5() throws IOException {
    final String file = "src/test/resources/moc.tyc2.11.ascii.v2.txt";
    // final String file = "src/test/resources/moc.sdss.11.ascii.v2.txt";
    final int orderMax = 11;
    long[] mocCells = new long[0];
    int i = 0;
    long la = System.nanoTime();
    for (final String line : Files.readAllLines(new File(file).toPath())) {
      final String[] orderAndCells = line.split("/");
      int order = Integer.parseInt(orderAndCells[0]);
      final String[] cells = orderAndCells[1].split(",");
      mocCells = Arrays.copyOf(mocCells, mocCells.length + cells.length);
      for (final String cell : cells) {
        mocCells[i++] =  HealpixNestedBMOC.buildValue(order, Long.parseLong(cell), false, orderMax);
      }
      assert i == mocCells.length;
    }
    long lb = System.nanoTime();
    
    Arrays.sort(mocCells);
    
    HealpixNestedBMOC bmoc = HealpixNestedBMOC.createUnsafe(orderMax, mocCells);
    System.out.println("bmoc size: " + bmoc.size());
    
    long l1 = System.nanoTime();
    byte[] bytes = HealpixNestedUltraCompactMOC.compress(bmoc);
    long l2 = System.nanoTime();
    // final String s = HealpixNestedUltraCompactMOC.compressB64(bytes);
    long l3 = System.nanoTime();
    bmoc = HealpixNestedUltraCompactMOC.decompress(orderMax, bytes);
    long l4 = System.nanoTime();
    
    // System.out.println(s);
    System.out.println("bmoc size: " + bmoc.size());
    
    byte[] bytes2 = HealpixNestedUltraCompactMOC.compress(bmoc);
    // System.out.println(HealpixNestedUltraCompactMOC.compressB64(bytes));
    
    System.out.println("Parse: " + (lb - la) / 1.0e6 + " ms.");
    System.out.println("Compress: " + (l2 - l1) / 1.0e6 + " ms. nBytes: " + bytes.length + " / " + bytes2.length);
    System.out.println("Base64: " + (l3 - l2) / 1.0e6 + " ms");
    System.out.println("DeCompress: " + (l4 - l3) / 1.0e6 + " ms. nCells: " + bmoc.size());
    
    long l5 =  System.nanoTime();
    final byte[] cbytes = gzipCompress(bytes);
    long l6 =  System.nanoTime();
    
    System.out.println("Gzip compress: " + (l6 - l5) / 1.0e6 + " ms. nBytes: " + cbytes.length);
    
    // decompress: GZIPInputStream gis = new GZIPInputStream(new ByteArrayInputStream(bytes));
    
  }*/
  
  
  public static byte[] gzipCompress(final String str) throws IOException {
    if ((str == null) || (str.length() == 0)) {
      return null;
    }
    ByteArrayOutputStream obj = new ByteArrayOutputStream();
    GZIPOutputStream gzip = new GZIPOutputStream(obj);
    gzip.write(str.getBytes("UTF-8"));
    gzip.flush();
    gzip.close();
    return obj.toByteArray();
  }
  public static byte[] gzipCompress(final byte[] bytes) throws IOException {
    ByteArrayOutputStream obj = new ByteArrayOutputStream();
    GZIPOutputStream gzip = new GZIPOutputStream(obj);
    gzip.write(bytes);
    gzip.flush();
    gzip.close();
    return obj.toByteArray();
  }

  
}
