;; DFA-based pipeline descriptions for MIPS32 4K processor family
;; Contributed by Nigel Stephens (nigel@mips.com)
;;   and David Ung (davidu@mips.com)
;;
;; References:
;;   "MIPS32 4K Processor Core Family Software User's Manual,
;;     Doc no: MD00016, Rev 1.18, Nov 15, 2004."
;;
;; 4Kc - pipelined multiplier and translation lookaside buffer (TLB)
;; 4km - pipelined multiplier and block address translator (BAT)
;; 4kp - non-pipelined multiplier and block address translator (BAT)
;;
;; Copyright (C) 2005-2021 Free Software Foundation, Inc.
;;
;; This file is part of GCC.
;;
;; GCC is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published
;; by the Free Software Foundation; either version 3, or (at your
;; option) any later version.

;; GCC is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
;; or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
;; License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GCC; see the file COPYING3.  If not see
;; <http://www.gnu.org/licenses/>.

(define_automaton "r4k_cpu, r4k_mdu")

;; Integer execution unit.
(define_cpu_unit "r4k_ixu_arith"       "r4k_cpu")
(define_cpu_unit "r4k_ixu_mpydiv"      "r4k_mdu")

(define_insn_reservation "r4k_int_load" 2
  (and (eq_attr "cpu" "4kc,4kp")
       (eq_attr "type" "load"))
  "r4k_ixu_arith")

(define_insn_reservation "r4k_int_prefetch" 1
  (and (eq_attr "cpu" "4kc,4kp")
       (eq_attr "type" "prefetch"))
  "r4k_ixu_arith")

(define_insn_reservation "r4k_int_store" 1
  (and (eq_attr "cpu" "4kc,4kp")
       (eq_attr "type" "store"))
  "r4k_ixu_arith")

;; 4Kc/4Km 
;; unsigned divide - 8/16/24/32-bit operand have latencies  9/17/25/33
;;   signed divide - 8/16/24/32-bit operand have latencies 10/18/26/34
(define_insn_reservation "r4k_idiv_4kc" 34
  (and (eq_attr "cpu" "4kc")
       (and (eq_attr "type" "idiv")
	    (eq_attr "mode" "!DI")))
  "r4k_ixu_arith+(r4k_ixu_mpydiv*34)")

;; 4Kp
;; unsigned divide - 33
;;   signed divide - 33-35
(define_insn_reservation "r4k_idiv_4kp" 35
  (and (eq_attr "cpu" "4kp")
       (and (eq_attr "type" "idiv")
	    (eq_attr "mode" "!DI")))
  "r4k_ixu_arith+(r4k_ixu_mpydiv*35)")

;; 4Kc/4Km fast 32x32 multiply
;; 16x32 is faster, but there's no way to detect this
(define_insn_reservation "r4k_mult_4kc" 2
  (and (eq_attr "cpu" "4kc")
       (and (eq_attr "type" "imul,imadd")
	    (eq_attr "mode" "SI")))
  "r4k_ixu_arith+(r4k_ixu_mpydiv*2)")

;; 4Kc/4Km MUL has 2 cycle latency, but has the special property that it will
;; stall the integer unit pipeline. MUL 16x16 or 32x16 forces 1 cycle stall,
;; while MUL 32x32 forces 2 cycle stall.  If next insn use the result, an
;; additional stall is forced.
(define_insn_reservation "r4k_mul_4kc" 4
  (and (eq_attr "cpu" "4kc")
       (and (eq_attr "type" "imul3")
	    (eq_attr "mode" "SI")))
  "(r4k_ixu_arith+r4k_ixu_mpydiv)*3")

;; 4Kp slow iterative 2-op MULT
;; Latency of 32 if next insn is MADD/MSUB,MFHI/MFLO.
;; Repeat rate of 33 cycles.
(define_insn_reservation "r4k_mult_4kp" 32
  (and (eq_attr "cpu" "4kp")
       (and (eq_attr "type" "imul")
	    (eq_attr "mode" "SI")))
  "r4k_ixu_arith+(r4k_ixu_mpydiv*32)")

;; 4Kp slow iterative 3-op MUL
;; Latency of 32 cycles, but stalls the whole pipeline until complete.
(define_insn_reservation "r4k_mul_4kp" 32
  (and (eq_attr "cpu" "4kp")
       (and (eq_attr "type" "imul3")
	    (eq_attr "mode" "SI")))
  "(r4k_ixu_arith+r4k_ixu_mpydiv)*32")

;; 4Kp slow iterative MADD
;; Latency of 34 if next use insn is MADD/MSUB,MFHI/MFLO.
;; Repeat rate of 35 cycles.
(define_insn_reservation "r4k_madd_4kp" 34
  (and (eq_attr "cpu" "4kp")
       (and (eq_attr "type" "imadd")
	    (eq_attr "mode" "SI")))
  "r4k_ixu_arith+(r4k_ixu_mpydiv*34)")

;; Move to HI/LO -> MADD/MSUB,MFHI/MFLO has a 1 cycle latency.
(define_insn_reservation "r4k_int_mthilo" 1
  (and (eq_attr "cpu" "4kc,4kp")
       (eq_attr "type" "mthi,mtlo"))
  "r4k_ixu_arith+r4k_ixu_mpydiv")

;; Move from HI/LO -> integer operation has a 2 cycle latency.
(define_insn_reservation "r4k_int_mfhilo" 2
  (and (eq_attr "cpu" "4kc,4kp")
       (eq_attr "type" "mfhi,mflo"))
  "r4k_ixu_arith+r4k_ixu_mpydiv")

;; All other integer insns.
(define_insn_reservation "r4k_int_alu" 1
  (and (eq_attr "cpu" "4kc,4kp")
       (eq_attr "type" "arith,condmove,const,logical,move,nop,shift,signext,slt"))
  "r4k_ixu_arith")

(define_insn_reservation "r4k_int_branch" 1
  (and (eq_attr "cpu" "4kc,4kp")
       (eq_attr "type" "branch"))
  "r4k_ixu_arith")

(define_insn_reservation "r4k_int_jump_4k" 1
  (and (eq_attr "cpu" "4kc,4kp")
       (eq_attr "type" "jump,call"))
  "r4k_ixu_arith")

;; mfcx/mtcx - non FPU
;; (Disabled until we add cop0 support)
;; (define_insn_reservation "r4k_int_cop" 2
;;   (and (eq_attr "cpu" "4kc,4kp")
;;      (eq_attr "type" "cop0"))
;;  "r4k_ixu_arith")

;; Unknown or multi - single issue
(define_insn_reservation "r4k_unknown" 1
  (and (eq_attr "cpu" "4kc,4kp")
       (eq_attr "type" "unknown,multi,atomic,syncloop"))
  "r4k_ixu_arith+r4k_ixu_mpydiv")
