# Copyright 2013 Sean Burford <sburford@google.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging

from chirp import chirp_common

LOG = logging.getLogger(__name__)


class Band(object):
    def __init__(self, limits, name, mode=None, step_khz=None,
                 input_offset=None, output_offset=None, tones=None):
        # Apply semantic and chirp limitations to settings.
        # memedit applies radio limitations when settings are applied.
        try:
            assert limits[0] <= limits[1], "Lower freq > upper freq"
            if mode is not None:
                assert mode in chirp_common.MODES, "Mode %s not one of %s" % (
                    mode, chirp_common.MODES)
            if step_khz is not None:
                assert step_khz in chirp_common.TUNING_STEPS, (
                    "step_khz %s not one of %s" %
                    (step_khz, chirp_common.TUNING_STEPS))
            if tones:
                for tone in tones:
                    assert tone in chirp_common.TONES, (
                        "tone %s not one of %s" % (tone, chirp_common.TONES))
        except AssertionError as e:
            raise ValueError("%s %s: %s" % (name, limits, e))

        self.name = name
        self.mode = mode
        self.step_khz = step_khz
        self.tones = tones
        self.limits = limits
        self.offset = None
        self.duplex = "simplex"
        if input_offset is not None:
            self.offset = input_offset
            self.duplex = "rpt TX"
        elif output_offset is not None:
            self.offset = output_offset
            self.duplex = "rpt RX"

    def __eq__(self, other):
        return (other.limits[0] == self.limits[0] and
                other.limits[1] == self.limits[1])

    def contains(self, other):
        return (other.limits[0] >= self.limits[0] and
                other.limits[1] <= self.limits[1])

    def width(self):
        return self.limits[1] - self.limits[0]

    def inverse(self):
        """Create an RX/TX shadow of this band using the offset."""
        if not self.offset:
            return self
        limits = (self.limits[0] + self.offset, self.limits[1] + self.offset)
        offset = -1 * self.offset
        if self.duplex == "rpt RX":
            return Band(limits, self.name, self.mode, self.step_khz,
                        input_offset=offset, tones=self.tones)
        return Band(limits, self.name, self.mode, self.step_khz,
                    output_offset=offset, tones=self.tones)

    def __repr__(self):
        desc = '%s%s%s%s' % (
            self.mode and 'mode: %s ' % (self.mode,) or '',
            self.step_khz and 'step_khz: %s ' % (self.step_khz,) or '',
            self.offset and 'offset: %s ' % (self.offset,) or '',
            self.tones and 'tones: %s ' % (self.tones,) or '')

        return "%s-%s %s %s %s" % (
            self.limits[0], self.limits[1], self.name, self.duplex, desc)


class BandPlans(object):
    def __init__(self, config):
        self._config = config
        self.plans = {}

        # Migrate old "automatic repeater offset" setting to
        # "North American Amateur Band Plan"
        ro = self._config.get("autorpt", "memedit")
        if ro is not None:
            self._config.set_bool("north_america", ro, "bandplan")
            self._config.remove_option("autorpt", "memedit")
        # And default new users to North America.
        if not self._config.is_defined("north_america", "bandplan"):
            self._config.set_bool("north_america", True, "bandplan")

        from chirp import bandplan_na, bandplan_au
        from chirp import bandplan_iaru_r1, bandplan_iaru_r2, bandplan_iaru_r3

        for plan in (bandplan_na, bandplan_au, bandplan_iaru_r1,
                     bandplan_iaru_r2, bandplan_iaru_r3):
            name = plan.DESC.get("name", plan.SHORTNAME)
            self.plans[plan.SHORTNAME] = (name, plan)

            rpt_inputs = []
            for band in plan.BANDS:
                # Check for duplicates.
                duplicates = [x for x in plan.BANDS if x == band]
                if len(duplicates) > 1:
                    LOG.info("Bandplan %s has duplicates %s" %
                             (name, duplicates))
                # Add repeater inputs.
                rpt_input = band.inverse()
                if rpt_input not in plan.BANDS:
                    rpt_inputs.append(band.inverse())
            plan.bands = list(plan.BANDS)
            plan.bands.extend(rpt_inputs)

    def get_defaults_for_frequency(self, freq):
        freq = int(freq)
        result = Band((freq, freq), repr(freq))

        for shortname, details in self.plans.items():
            if self._config.get_bool(shortname, "bandplan"):
                matches = [x for x in details[1].bands if x.contains(result)]
                # Add matches to defaults, favoring more specific matches.
                matches = sorted(matches, key=lambda x: x.width(),
                                 reverse=True)
                for match in matches:
                    result.mode = match.mode or result.mode
                    result.step_khz = match.step_khz or result.step_khz
                    result.offset = match.offset or result.offset
                    result.duplex = match.duplex or result.duplex
                    result.tones = match.tones or result.tones
                    if match.name:
                        result.name = '/'.join((result.name or '', match.name))
                # Limit ourselves to one band plan match for simplicity.
                # Note that if the user selects multiple band plans by editing
                # the config file it will work as expected (except where plans
                # conflict).
                if matches:
                    break

        return result
