/**
 *  Copyright Notice:
 *  Copyright 2023 DMTF. All rights reserved.
 *  License: BSD 3-Clause License. For full text see link: https://github.com/DMTF/libspdm/blob/main/LICENSE.md
 **/

#include "internal/libspdm_crypt_lib.h"
#include "internal/libspdm_common_lib.h"
#include "internal/libspdm_fips_lib.h"
#include "library/spdm_crypt_lib.h"

#if LIBSPDM_FIPS_MODE

/**
 * RSA_PSS self_test
 **/
bool libspdm_fips_selftest_rsa_pss(void *fips_selftest_context)
{
    bool result = true;

#if (LIBSPDM_RSA_PSS_SUPPORT) && (LIBSPDM_SHA256_SUPPORT)
    libspdm_fips_selftest_context *context = fips_selftest_context;
    LIBSPDM_ASSERT(fips_selftest_context != NULL);

    /* any test fail cause the FIPS fail*/
    if (context->tested_algo != context->self_test_result) {
        return false;
    }

    /* check if run before.*/
    if ((context->tested_algo & LIBSPDM_FIPS_SELF_TEST_RSA_PSS) != 0) {
        return true;
    }

    uint8_t signature[LIBSPDM_MAX_ASYM_KEY_SIZE];
    uint8_t digest[LIBSPDM_MAX_HASH_SIZE];
    size_t sig_size;
    void *rsa;

    sig_size = sizeof(signature);
    libspdm_zero_mem(signature, sig_size);
    libspdm_zero_mem(digest, LIBSPDM_MAX_HASH_SIZE);

    /*test vectors from: https://github.com/openssl/openssl/blob/OpenSSL-fips-2_0-stable/fips/rsa/fips_rsa_selftest.c*/
    const uint8_t public_n[] = {
        0xDB, 0x10, 0x1A, 0xC2, 0xA3, 0xF1, 0xDC, 0xFF, 0x13, 0x6B, 0xED, 0x44,
        0xDF, 0xF0, 0x02, 0x6D, 0x13, 0xC7, 0x88, 0xDA, 0x70, 0x6B, 0x54, 0xF1,
        0xE8, 0x27, 0xDC, 0xC3, 0x0F, 0x99, 0x6A, 0xFA, 0xC6, 0x67, 0xFF, 0x1D,
        0x1E, 0x3C, 0x1D, 0xC1, 0xB5, 0x5F, 0x6C, 0xC0, 0xB2, 0x07, 0x3A, 0x6D,
        0x41, 0xE4, 0x25, 0x99, 0xAC, 0xFC, 0xD2, 0x0F, 0x02, 0xD3, 0xD1, 0x54,
        0x06, 0x1A, 0x51, 0x77, 0xBD, 0xB6, 0xBF, 0xEA, 0xA7, 0x5C, 0x06, 0xA9,
        0x5D, 0x69, 0x84, 0x45, 0xD7, 0xF5, 0x05, 0xBA, 0x47, 0xF0, 0x1B, 0xD7,
        0x2B, 0x24, 0xEC, 0xCB, 0x9B, 0x1B, 0x10, 0x8D, 0x81, 0xA0, 0xBE, 0xB1,
        0x8C, 0x33, 0xE4, 0x36, 0xB8, 0x43, 0xEB, 0x19, 0x2A, 0x81, 0x8D, 0xDE,
        0x81, 0x0A, 0x99, 0x48, 0xB6, 0xF6, 0xBC, 0xCD, 0x49, 0x34, 0x3A, 0x8F,
        0x26, 0x94, 0xE3, 0x28, 0x82, 0x1A, 0x7C, 0x8F, 0x59, 0x9F, 0x45, 0xE8,
        0x5D, 0x1A, 0x45, 0x76, 0x04, 0x56, 0x05, 0xA1, 0xD0, 0x1B, 0x8C, 0x77,
        0x6D, 0xAF, 0x53, 0xFA, 0x71, 0xE2, 0x67, 0xE0, 0x9A, 0xFE, 0x03, 0xA9,
        0x85, 0xD2, 0xC9, 0xAA, 0xBA, 0x2A, 0xBC, 0xF4, 0xA0, 0x08, 0xF5, 0x13,
        0x98, 0x13, 0x5D, 0xF0, 0xD9, 0x33, 0x34, 0x2A, 0x61, 0xC3, 0x89, 0x55,
        0xF0, 0xAE, 0x1A, 0x9C, 0x22, 0xEE, 0x19, 0x05, 0x8D, 0x32, 0xFE, 0xEC,
        0x9C, 0x84, 0xBA, 0xB7, 0xF9, 0x6C, 0x3A, 0x4F, 0x07, 0xFC, 0x45, 0xEB,
        0x12, 0xE5, 0x7B, 0xFD, 0x55, 0xE6, 0x29, 0x69, 0xD1, 0xC2, 0xE8, 0xB9,
        0x78, 0x59, 0xF6, 0x79, 0x10, 0xC6, 0x4E, 0xEB, 0x6A, 0x5E, 0xB9, 0x9A,
        0xC7, 0xC4, 0x5B, 0x63, 0xDA, 0xA3, 0x3F, 0x5E, 0x92, 0x7A, 0x81, 0x5E,
        0xD6, 0xB0, 0xE2, 0x62, 0x8F, 0x74, 0x26, 0xC2, 0x0C, 0xD3, 0x9A, 0x17,
        0x47, 0xE6, 0x8E, 0xAB
    };

    const uint8_t e[] = {
        0x00, 0x01, 0x00, 0x01,
    };

    const uint8_t private_d[] = {
        0x52, 0x41, 0xF4, 0xDA, 0x7B, 0xB7, 0x59, 0x55, 0xCA, 0xD4, 0x2F, 0x0F,
        0x3A, 0xCB, 0xA4, 0x0D, 0x93, 0x6C, 0xCC, 0x9D, 0xC1, 0xB2, 0xFB, 0xFD,
        0xAE, 0x40, 0x31, 0xAC, 0x69, 0x52, 0x21, 0x92, 0xB3, 0x27, 0xDF, 0xEA,
        0xEE, 0x2C, 0x82, 0xBB, 0xF7, 0x40, 0x32, 0xD5, 0x14, 0xC4, 0x94, 0x12,
        0xEC, 0xB8, 0x1F, 0xCA, 0x59, 0xE3, 0xC1, 0x78, 0xF3, 0x85, 0xD8, 0x47,
        0xA5, 0xD7, 0x02, 0x1A, 0x65, 0x79, 0x97, 0x0D, 0x24, 0xF4, 0xF0, 0x67,
        0x6E, 0x75, 0x2D, 0xBF, 0x10, 0x3D, 0xA8, 0x7D, 0xEF, 0x7F, 0x60, 0xE4,
        0xE6, 0x05, 0x82, 0x89, 0x5D, 0xDF, 0xC6, 0xD2, 0x6C, 0x07, 0x91, 0x33,
        0x98, 0x42, 0xF0, 0x02, 0x00, 0x25, 0x38, 0xC5, 0x85, 0x69, 0x8A, 0x7D,
        0x2F, 0x95, 0x6C, 0x43, 0x9A, 0xB8, 0x81, 0xE2, 0xD0, 0x07, 0x35, 0xAA,
        0x05, 0x41, 0xC9, 0x1E, 0xAF, 0xE4, 0x04, 0x3B, 0x19, 0xB8, 0x73, 0xA2,
        0xAC, 0x4B, 0x1E, 0x66, 0x48, 0xD8, 0x72, 0x1F, 0xAC, 0xF6, 0xCB, 0xBC,
        0x90, 0x09, 0xCA, 0xEC, 0x0C, 0xDC, 0xF9, 0x2C, 0xD7, 0xEB, 0xAE, 0xA3,
        0xA4, 0x47, 0xD7, 0x33, 0x2F, 0x8A, 0xCA, 0xBC, 0x5E, 0xF0, 0x77, 0xE4,
        0x97, 0x98, 0x97, 0xC7, 0x10, 0x91, 0x7D, 0x2A, 0xA6, 0xFF, 0x46, 0x83,
        0x97, 0xDE, 0xE9, 0xE2, 0x17, 0x03, 0x06, 0x14, 0xE2, 0xD7, 0xB1, 0x1D,
        0x77, 0xAF, 0x51, 0x27, 0x5B, 0x5E, 0x69, 0xB8, 0x81, 0xE6, 0x11, 0xC5,
        0x43, 0x23, 0x81, 0x04, 0x62, 0xFF, 0xE9, 0x46, 0xB8, 0xD8, 0x44, 0xDB,
        0xA5, 0xCC, 0x31, 0x54, 0x34, 0xCE, 0x3E, 0x82, 0xD6, 0xBF, 0x7A, 0x0B,
        0x64, 0x21, 0x6D, 0x88, 0x7E, 0x5B, 0x45, 0x12, 0x1E, 0x63, 0x8D, 0x49,
        0xA7, 0x1D, 0xD9, 0x1E, 0x06, 0xCD, 0xE8, 0xBA, 0x2C, 0x8C, 0x69, 0x32,
        0xEA, 0xBE, 0x60, 0x71
    };

    const char kat_tbs[] = "OpenSSL FIPS 140-2 Public Key RSA KAT";
    const uint8_t kat_RSA_PSS_SHA256[] = {
        0x51, 0x4F, 0x0B, 0x8A, 0xE6, 0x7F, 0xF0, 0x39, 0x9B, 0x5F, 0x21, 0x26,
        0x99, 0x71, 0xDC, 0x64, 0x5A, 0x60, 0x9B, 0xFB, 0xB9, 0x48, 0xE3, 0xD4,
        0xDE, 0xAE, 0x2D, 0x87, 0x72, 0x14, 0x2A, 0x01, 0x4C, 0xC7, 0x5C, 0x00,
        0x15, 0xAE, 0x23, 0x09, 0x5F, 0xD5, 0xB5, 0x6F, 0x94, 0x59, 0x76, 0xEA,
        0x8C, 0x28, 0x90, 0xC4, 0xCD, 0xB9, 0xD7, 0xB5, 0x82, 0x9D, 0x4E, 0x37,
        0xE0, 0x79, 0x03, 0x88, 0x9F, 0x31, 0x71, 0x70, 0x80, 0xD6, 0x99, 0x49,
        0x20, 0x09, 0x30, 0x64, 0x52, 0xCD, 0x5C, 0x10, 0x72, 0x29, 0x4D, 0x33,
        0x10, 0xF6, 0xEB, 0xB7, 0x27, 0x95, 0x55, 0x48, 0xBE, 0x5A, 0xBE, 0xA7,
        0x74, 0xCF, 0x0E, 0x95, 0x28, 0xF4, 0x05, 0x98, 0x86, 0xBA, 0x36, 0x5C,
        0x91, 0x8F, 0x5D, 0xC1, 0xBD, 0xBE, 0x74, 0xD9, 0x74, 0x7B, 0xEC, 0xD1,
        0x4E, 0x79, 0x66, 0x14, 0x24, 0x21, 0x27, 0x2D, 0xCF, 0x93, 0xC0, 0xE1,
        0x92, 0x79, 0xE0, 0x4D, 0x65, 0xE2, 0xA4, 0xBA, 0x1F, 0x96, 0x16, 0xB1,
        0x90, 0xE1, 0x50, 0xDE, 0x2F, 0xB0, 0x07, 0xAF, 0x84, 0xCC, 0x81, 0x1C,
        0x42, 0xA9, 0x1B, 0xF6, 0x25, 0xDF, 0x6F, 0xEB, 0x9A, 0xF0, 0x93, 0xF3,
        0x66, 0xE7, 0x88, 0x7B, 0xF9, 0xA5, 0x92, 0xA9, 0xD5, 0x35, 0xAA, 0x33,
        0x8F, 0xEA, 0x68, 0xDB, 0x82, 0xE5, 0x69, 0xBB, 0x95, 0x31, 0xC3, 0x4F,
        0x0A, 0x68, 0x15, 0x8A, 0x93, 0x91, 0xE2, 0x3B, 0xDC, 0xA4, 0x94, 0x2F,
        0x34, 0xBC, 0xCE, 0xC9, 0xA6, 0xF0, 0x49, 0x33, 0x5B, 0x8F, 0xDB, 0x60,
        0xE8, 0x37, 0x8C, 0x97, 0xF9, 0xFB, 0x28, 0x0A, 0xF1, 0xAD, 0x4D, 0x9D,
        0xC0, 0x57, 0x4D, 0xD0, 0x26, 0x76, 0x36, 0x21, 0x67, 0x97, 0x73, 0xCD,
        0x26, 0x3A, 0xCD, 0x44, 0x77, 0xF2, 0x59, 0xBB, 0xDF, 0x77, 0x3B, 0x98,
        0xC5, 0x57, 0xB7, 0x5D
    };

    result = libspdm_sha256_hash_all(kat_tbs, sizeof(kat_tbs) - 1, digest);
    if (!result) {
        LIBSPDM_DEBUG((LIBSPDM_DEBUG_INFO, "Hash sha256 failed \n"));
        return false;
    }

    rsa = libspdm_rsa_new();
    if (rsa == NULL) {
        LIBSPDM_DEBUG((LIBSPDM_DEBUG_INFO, "RSA_PSS rsa new failed \n"));
        result = false;
        goto update;
    }

    result = libspdm_rsa_set_key(rsa, LIBSPDM_RSA_KEY_N, public_n, sizeof(public_n));
    if (!result) {
        LIBSPDM_DEBUG((LIBSPDM_DEBUG_INFO, "RSA_PSS rsa set key failed \n"));
        libspdm_rsa_free(rsa);
        result = false;
        goto update;
    }

    result = libspdm_rsa_set_key(rsa, LIBSPDM_RSA_KEY_E, e, sizeof(e));
    if (!result) {
        LIBSPDM_DEBUG((LIBSPDM_DEBUG_INFO, "RSA_PSS rsa set key failed \n"));
        libspdm_rsa_free(rsa);
        result = false;
        goto update;
    }

    result = libspdm_rsa_set_key(rsa, LIBSPDM_RSA_KEY_D, private_d, sizeof(private_d));
    if (!result) {
        LIBSPDM_DEBUG((LIBSPDM_DEBUG_INFO, "RSA_PSS rsa set key failed \n"));
        libspdm_rsa_free(rsa);
        result = false;
        goto update;
    }

    result = libspdm_rsa_pss_sign_fips(rsa, LIBSPDM_CRYPTO_NID_SHA256,
                                       digest, LIBSPDM_SHA256_DIGEST_SIZE,
                                       signature, &sig_size);
    if (!result) {
        LIBSPDM_DEBUG((LIBSPDM_DEBUG_INFO, "RSA_PSS rsa sign failed \n"));
        libspdm_rsa_free(rsa);
        result = false;
        goto update;
    }

    /*KAT test*/
    if (!libspdm_consttime_is_mem_equal(signature, kat_RSA_PSS_SHA256,
                                        sizeof(kat_RSA_PSS_SHA256))) {
        LIBSPDM_DEBUG((LIBSPDM_DEBUG_INFO, "RSA_PSS KAT failed \n"));
        libspdm_rsa_free(rsa);
        result = false;
        goto update;
    }

    result = libspdm_rsa_pss_verify_fips(rsa, LIBSPDM_CRYPTO_NID_SHA256,
                                         digest, LIBSPDM_SHA256_DIGEST_SIZE,
                                         signature, sig_size);
    if (!result) {
        LIBSPDM_DEBUG((LIBSPDM_DEBUG_INFO, "RSA_PSS selftest failed \n"));
        libspdm_rsa_free(rsa);
        result = false;
        goto update;
    }

    libspdm_rsa_free(rsa);

update:
    /* mark it as tested*/
    context->tested_algo |= LIBSPDM_FIPS_SELF_TEST_RSA_PSS;

    /* record test result*/
    if (result) {
        context->self_test_result |= LIBSPDM_FIPS_SELF_TEST_RSA_PSS;
    } else {
        context->self_test_result &= ~LIBSPDM_FIPS_SELF_TEST_RSA_PSS;
    }

#endif/*(LIBSPDM_RSA_PSS_SUPPORT) && (LIBSPDM_SHA256_SUPPORT)*/

    return result;
}

#endif
