/*
 * Copyright (c) 2023, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * @test
 * @bug 8311546
 * @summary Adopt de-facto standards on x509 Name Constraints with leading dot. Certs
 * can be generated by running generate-certs.sh
 * @run main LeadingPeriod
 */

import java.io.*;
import java.nio.file.*;
import java.util.*;
import java.security.Security;
import java.security.cert.*;

public class LeadingPeriod {

    public static void main(String[] args) throws Exception {
        String certs = System.getProperty("test.src", "./") + "/certs/";
        validate(certs + "withoutLeadingPeriod");
        validate(certs + "withLeadingPeriod");
    }

    public static void validate(String certPath) throws Exception {
        byte[] targetCertBytes = Files.readAllBytes(Paths.get(certPath + "/leaf.pem"));
        byte[] caCertBytes = Files.readAllBytes(Paths.get(certPath + "/ca.pem"));

        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        Certificate caCert = cf.generateCertificate(new ByteArrayInputStream(caCertBytes));
        Certificate targetCert = cf.generateCertificate(new ByteArrayInputStream(targetCertBytes));

        TrustAnchor anchor = new TrustAnchor((X509Certificate) caCert, null);

        PKIXParameters params = new PKIXParameters(Collections.singleton(anchor));

        // Disable certificate revocation checking
        params.setRevocationEnabled(false);

        // Set validity date, so that validation won't fail when cert expires
        params.setDate(((X509Certificate)targetCert).getNotBefore());

        CertPath path = cf.generateCertPath(List.of(targetCert, caCert));

        CertPathValidator validator = CertPathValidator.getInstance("PKIX");
        validator.validate(path, params);
    }
}
