use util:serialize;
use core:io;
use crypto;
use ui;
use layout;
use core:geometry;

// Base url where we store our temporary files.
private Url configBaseUrl on Ui = userConfigUrl("progvis");

private Url configBase() on Ui {
	configBaseUrl;
}

/**
 * Settings for Progvis.
 */
class Settings : serializable {
	// Create a new instance.
	init() {
		init {
			zoom = 1.0;
			onlineId = generateId();
			downloadDir = Url(["problems"], UrlFlags:isDir);
		}
	}

	// Zoom level.
	Float zoom;

	// Identity of this instance (for online usage).
	Str onlineId;

	// Program to use when opening code-files (if specified).
	Url? customProgram;

	// Directory to use when storing downloaded files. May be a relative path.
	Url downloadDir;

	// Model-checker settings.
	check:Checks checks = check:Checks();

	// Load from a file, or create default settings if the file is missing.
	Settings load(Url from) : static {
		try {
			ObjIStream input(from.read());
			Settings s = Settings:read(input);
			input.close();
			return s;
		} catch (EndOfStream e) {
			Settings s;
			// Make sure to persist the random ID.
			s.save(from);
			return s;
		} catch (Exception e) {
			print("Error loading settings: ${e}");
			Settings s;
			// Make sure to persist the random ID.
			s.save(from);
			return s;
		}
	}

	// Load from the default location.
	Settings load() : static {
		load(configBase / "settings");
	}

	// Save settings to a file.
	void save(Url to) {
		ObjOStream output(to.write());
		write(output);
		output.close();
	}

	// Save to the default location.
	void save() {
		save(configBase / "settings");
	}

	// Open a file given the current configuration.
	void open(Url file) {
		if (customProgram) {
			// Use the program...
			core:sys:execute(customProgram, [file.toS]);
		} else {
			core:sys:open(file);
		}
	}

	// Get the download directory as an absolute directory.
	Url absDownloadDir() {
		if (!downloadDir.absolute)
			configBase / downloadDir;
		else
			downloadDir;
	}

	// Get an Url for a file in the download directory.
	Url downloadFile(Str name) {
		Url abs = absDownloadDir();
		if (!abs.exists())
			abs.createDir();
		abs / name;
	}
}


private Str generateId() {
	RandomStream stream;
	var data = stream.fill(100);
	stream.close();

	StrBuf out;
	for (Nat i = 0; i < data.filled; i++) {
		out << toChar(data[i]);
	}

	out.toS;
}

private Char toChar(Byte b) {
	Nat scaled = b.nat % (26n * 2n + 10n);
	if (scaled < 10) {
		// 0-9
		Char(0x30 + scaled);
	} else if (scaled < 36) {
		// A-Z
		Char(0x41 + scaled - 10);
	} else {
		// a-z
		Char(0x61 + scaled - 36);
	}
}

/**
 * Settings dialog.
 */
dialog SettingsDlg {
	layout Grid {
		expandCol: 0;
		wrapCols: 2;
		Group("Interface") {
			colspan: 2;
			content: container Grid {
				expandCol: 0;
				Label("Zoom level:") {}
				RadioButton small("Small", 1) {}
				RadioButton medium("Medium", 1) {}
				RadioButton large("Large", 1) {}
				RadioButton huge("Huge", 1) {}
				RadioButton gigantic("Gigantic", 1) {}
			}
		}

		nextLine;

		Group("System") {
			colspan: 2;
			content: container Grid {
				expandCol: 0;
				Label("Editor to use to open code from Progvis (leave blank for default):") { colspan: 2; }
				nextLine;
				Edit program("") {}
				Button pickProgram("Pick a program") {}
				nextLine;

				Label("Directory to use for downloaded problems:") { colspan: 2; }
				nextLine;
				Edit downloadDir("") {}
				Button pickDir("Pick a directory") {}
			}
		}

		nextLine;

		Group("Errors reported by \"Look for errors\"") {
			colspan: 2;
			content: container Grid {
				wrapCols: 1;
				CheckButton checkLeaks("Report memory leaks") {}
				CheckButton checkTermination("Report livelocks") {}
			}
		}

		nextLine;
		Button ok("Save") { col: 1; }
	}

	private Settings toEdit;
	private fn()->void callback;

	init(Settings settings, fn()->void callback) {
		init("Settings", Size(300, 50)) {
			toEdit = settings;
			callback = callback;
		}

		if (prog = toEdit.customProgram) {
			program.text = prog.toS;
		}

		downloadDir.text = toEdit.absDownloadDir.toS;

		if (settings.zoom < 0.8)
			small.checked = true;
		else if (settings.zoom < 1.1)
			medium.checked = true;
		else if (settings.zoom < 1.3)
			large.checked = true;
		else if (settings.zoom < 1.6)
			huge.checked = true;
		else
			gigantic.checked = true;

		checkTermination.checked = settings.checks.termination;
		checkLeaks.checked = settings.checks.leaks;

		defaultChoice = ok;
		ok.onClick = &this.onSave();
		pickProgram.onClick = &this.onPickProg();
		pickDir.onClick = &this.onPickDir();
	}

	private void onSave() {
		if (small.checked)
			toEdit.zoom = 0.75;
		else if (medium.checked)
			toEdit.zoom = 1.0;
		else if (large.checked)
			toEdit.zoom = 1.25;
		else if (huge.checked)
			toEdit.zoom = 1.5;
		else
			toEdit.zoom = 1.75;

		{
			Str p = program.text;
			if (p.empty())
				toEdit.customProgram = null;
			else
				toEdit.customProgram = parsePath(p);
		}

		if (downloadDir.text.any) {
			Url p = parsePath(downloadDir.text);
			if (p.absolute)
				toEdit.downloadDir = p.relativeIfBelow(configBase);
			else
				toEdit.downloadDir = p;
		}

		toEdit.checks.termination = checkTermination.checked;
		toEdit.checks.leaks = checkLeaks.checked;

		toEdit.save();
		close();
		callback.call();
	}

	private void onPickProg() {
		FileTypes ft("Programs");
		// TODO: Switch on the OS here.
		ft.add("Executable files", ["*.exe", "*"]);

		FilePicker p = FilePicker:open(ft).okLabel("Select");
		if (!p.show(this))
			return;

		program.text = p.results[0].toS;
	}

	private void onPickDir() {
		try {
			FilePicker p = FilePicker:folder();
			p.defaultFolder = parsePath(downloadDir.text);
			if (!p.show(this))
				return;

			downloadDir.text = p.results[0].toS;
		} catch (Exception e) {
			showMessage(this, "Error", "Unknown error: ${e}");
		}
	}
}
