﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/ConnectionTrackingSpecificationResponse.h>
#include <aws/ec2/model/GroupIdentifier.h>
#include <aws/ec2/model/InstanceIpv4Prefix.h>
#include <aws/ec2/model/InstanceIpv6Address.h>
#include <aws/ec2/model/InstanceIpv6Prefix.h>
#include <aws/ec2/model/InstanceNetworkInterfaceAssociation.h>
#include <aws/ec2/model/InstanceNetworkInterfaceAttachment.h>
#include <aws/ec2/model/InstancePrivateIpAddress.h>
#include <aws/ec2/model/NetworkInterfaceStatus.h>
#include <aws/ec2/model/OperatorResponse.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>Describes a network interface.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/InstanceNetworkInterface">AWS
 * API Reference</a></p>
 */
class InstanceNetworkInterface {
 public:
  AWS_EC2_API InstanceNetworkInterface() = default;
  AWS_EC2_API InstanceNetworkInterface(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API InstanceNetworkInterface& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The association information for an Elastic IPv4 associated with the network
   * interface.</p>
   */
  inline const InstanceNetworkInterfaceAssociation& GetAssociation() const { return m_association; }
  inline bool AssociationHasBeenSet() const { return m_associationHasBeenSet; }
  template <typename AssociationT = InstanceNetworkInterfaceAssociation>
  void SetAssociation(AssociationT&& value) {
    m_associationHasBeenSet = true;
    m_association = std::forward<AssociationT>(value);
  }
  template <typename AssociationT = InstanceNetworkInterfaceAssociation>
  InstanceNetworkInterface& WithAssociation(AssociationT&& value) {
    SetAssociation(std::forward<AssociationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The network interface attachment.</p>
   */
  inline const InstanceNetworkInterfaceAttachment& GetAttachment() const { return m_attachment; }
  inline bool AttachmentHasBeenSet() const { return m_attachmentHasBeenSet; }
  template <typename AttachmentT = InstanceNetworkInterfaceAttachment>
  void SetAttachment(AttachmentT&& value) {
    m_attachmentHasBeenSet = true;
    m_attachment = std::forward<AttachmentT>(value);
  }
  template <typename AttachmentT = InstanceNetworkInterfaceAttachment>
  InstanceNetworkInterface& WithAttachment(AttachmentT&& value) {
    SetAttachment(std::forward<AttachmentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  InstanceNetworkInterface& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The security groups.</p>
   */
  inline const Aws::Vector<GroupIdentifier>& GetGroups() const { return m_groups; }
  inline bool GroupsHasBeenSet() const { return m_groupsHasBeenSet; }
  template <typename GroupsT = Aws::Vector<GroupIdentifier>>
  void SetGroups(GroupsT&& value) {
    m_groupsHasBeenSet = true;
    m_groups = std::forward<GroupsT>(value);
  }
  template <typename GroupsT = Aws::Vector<GroupIdentifier>>
  InstanceNetworkInterface& WithGroups(GroupsT&& value) {
    SetGroups(std::forward<GroupsT>(value));
    return *this;
  }
  template <typename GroupsT = GroupIdentifier>
  InstanceNetworkInterface& AddGroups(GroupsT&& value) {
    m_groupsHasBeenSet = true;
    m_groups.emplace_back(std::forward<GroupsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IPv6 addresses associated with the network interface.</p>
   */
  inline const Aws::Vector<InstanceIpv6Address>& GetIpv6Addresses() const { return m_ipv6Addresses; }
  inline bool Ipv6AddressesHasBeenSet() const { return m_ipv6AddressesHasBeenSet; }
  template <typename Ipv6AddressesT = Aws::Vector<InstanceIpv6Address>>
  void SetIpv6Addresses(Ipv6AddressesT&& value) {
    m_ipv6AddressesHasBeenSet = true;
    m_ipv6Addresses = std::forward<Ipv6AddressesT>(value);
  }
  template <typename Ipv6AddressesT = Aws::Vector<InstanceIpv6Address>>
  InstanceNetworkInterface& WithIpv6Addresses(Ipv6AddressesT&& value) {
    SetIpv6Addresses(std::forward<Ipv6AddressesT>(value));
    return *this;
  }
  template <typename Ipv6AddressesT = InstanceIpv6Address>
  InstanceNetworkInterface& AddIpv6Addresses(Ipv6AddressesT&& value) {
    m_ipv6AddressesHasBeenSet = true;
    m_ipv6Addresses.emplace_back(std::forward<Ipv6AddressesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The MAC address.</p>
   */
  inline const Aws::String& GetMacAddress() const { return m_macAddress; }
  inline bool MacAddressHasBeenSet() const { return m_macAddressHasBeenSet; }
  template <typename MacAddressT = Aws::String>
  void SetMacAddress(MacAddressT&& value) {
    m_macAddressHasBeenSet = true;
    m_macAddress = std::forward<MacAddressT>(value);
  }
  template <typename MacAddressT = Aws::String>
  InstanceNetworkInterface& WithMacAddress(MacAddressT&& value) {
    SetMacAddress(std::forward<MacAddressT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the network interface.</p>
   */
  inline const Aws::String& GetNetworkInterfaceId() const { return m_networkInterfaceId; }
  inline bool NetworkInterfaceIdHasBeenSet() const { return m_networkInterfaceIdHasBeenSet; }
  template <typename NetworkInterfaceIdT = Aws::String>
  void SetNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    m_networkInterfaceIdHasBeenSet = true;
    m_networkInterfaceId = std::forward<NetworkInterfaceIdT>(value);
  }
  template <typename NetworkInterfaceIdT = Aws::String>
  InstanceNetworkInterface& WithNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    SetNetworkInterfaceId(std::forward<NetworkInterfaceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Amazon Web Services account that created the network
   * interface.</p>
   */
  inline const Aws::String& GetOwnerId() const { return m_ownerId; }
  inline bool OwnerIdHasBeenSet() const { return m_ownerIdHasBeenSet; }
  template <typename OwnerIdT = Aws::String>
  void SetOwnerId(OwnerIdT&& value) {
    m_ownerIdHasBeenSet = true;
    m_ownerId = std::forward<OwnerIdT>(value);
  }
  template <typename OwnerIdT = Aws::String>
  InstanceNetworkInterface& WithOwnerId(OwnerIdT&& value) {
    SetOwnerId(std::forward<OwnerIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The private DNS name.</p>
   */
  inline const Aws::String& GetPrivateDnsName() const { return m_privateDnsName; }
  inline bool PrivateDnsNameHasBeenSet() const { return m_privateDnsNameHasBeenSet; }
  template <typename PrivateDnsNameT = Aws::String>
  void SetPrivateDnsName(PrivateDnsNameT&& value) {
    m_privateDnsNameHasBeenSet = true;
    m_privateDnsName = std::forward<PrivateDnsNameT>(value);
  }
  template <typename PrivateDnsNameT = Aws::String>
  InstanceNetworkInterface& WithPrivateDnsName(PrivateDnsNameT&& value) {
    SetPrivateDnsName(std::forward<PrivateDnsNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IPv4 address of the network interface within the subnet.</p>
   */
  inline const Aws::String& GetPrivateIpAddress() const { return m_privateIpAddress; }
  inline bool PrivateIpAddressHasBeenSet() const { return m_privateIpAddressHasBeenSet; }
  template <typename PrivateIpAddressT = Aws::String>
  void SetPrivateIpAddress(PrivateIpAddressT&& value) {
    m_privateIpAddressHasBeenSet = true;
    m_privateIpAddress = std::forward<PrivateIpAddressT>(value);
  }
  template <typename PrivateIpAddressT = Aws::String>
  InstanceNetworkInterface& WithPrivateIpAddress(PrivateIpAddressT&& value) {
    SetPrivateIpAddress(std::forward<PrivateIpAddressT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The private IPv4 addresses associated with the network interface.</p>
   */
  inline const Aws::Vector<InstancePrivateIpAddress>& GetPrivateIpAddresses() const { return m_privateIpAddresses; }
  inline bool PrivateIpAddressesHasBeenSet() const { return m_privateIpAddressesHasBeenSet; }
  template <typename PrivateIpAddressesT = Aws::Vector<InstancePrivateIpAddress>>
  void SetPrivateIpAddresses(PrivateIpAddressesT&& value) {
    m_privateIpAddressesHasBeenSet = true;
    m_privateIpAddresses = std::forward<PrivateIpAddressesT>(value);
  }
  template <typename PrivateIpAddressesT = Aws::Vector<InstancePrivateIpAddress>>
  InstanceNetworkInterface& WithPrivateIpAddresses(PrivateIpAddressesT&& value) {
    SetPrivateIpAddresses(std::forward<PrivateIpAddressesT>(value));
    return *this;
  }
  template <typename PrivateIpAddressesT = InstancePrivateIpAddress>
  InstanceNetworkInterface& AddPrivateIpAddresses(PrivateIpAddressesT&& value) {
    m_privateIpAddressesHasBeenSet = true;
    m_privateIpAddresses.emplace_back(std::forward<PrivateIpAddressesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether source/destination checking is enabled.</p>
   */
  inline bool GetSourceDestCheck() const { return m_sourceDestCheck; }
  inline bool SourceDestCheckHasBeenSet() const { return m_sourceDestCheckHasBeenSet; }
  inline void SetSourceDestCheck(bool value) {
    m_sourceDestCheckHasBeenSet = true;
    m_sourceDestCheck = value;
  }
  inline InstanceNetworkInterface& WithSourceDestCheck(bool value) {
    SetSourceDestCheck(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the network interface.</p>
   */
  inline NetworkInterfaceStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(NetworkInterfaceStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline InstanceNetworkInterface& WithStatus(NetworkInterfaceStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the subnet.</p>
   */
  inline const Aws::String& GetSubnetId() const { return m_subnetId; }
  inline bool SubnetIdHasBeenSet() const { return m_subnetIdHasBeenSet; }
  template <typename SubnetIdT = Aws::String>
  void SetSubnetId(SubnetIdT&& value) {
    m_subnetIdHasBeenSet = true;
    m_subnetId = std::forward<SubnetIdT>(value);
  }
  template <typename SubnetIdT = Aws::String>
  InstanceNetworkInterface& WithSubnetId(SubnetIdT&& value) {
    SetSubnetId(std::forward<SubnetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the VPC.</p>
   */
  inline const Aws::String& GetVpcId() const { return m_vpcId; }
  inline bool VpcIdHasBeenSet() const { return m_vpcIdHasBeenSet; }
  template <typename VpcIdT = Aws::String>
  void SetVpcId(VpcIdT&& value) {
    m_vpcIdHasBeenSet = true;
    m_vpcId = std::forward<VpcIdT>(value);
  }
  template <typename VpcIdT = Aws::String>
  InstanceNetworkInterface& WithVpcId(VpcIdT&& value) {
    SetVpcId(std::forward<VpcIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of network interface.</p> <p>Valid values: <code>interface</code> |
   * <code>efa</code> | <code>efa-only</code> | <code>evs</code> | <code>trunk</code>
   * </p>
   */
  inline const Aws::String& GetInterfaceType() const { return m_interfaceType; }
  inline bool InterfaceTypeHasBeenSet() const { return m_interfaceTypeHasBeenSet; }
  template <typename InterfaceTypeT = Aws::String>
  void SetInterfaceType(InterfaceTypeT&& value) {
    m_interfaceTypeHasBeenSet = true;
    m_interfaceType = std::forward<InterfaceTypeT>(value);
  }
  template <typename InterfaceTypeT = Aws::String>
  InstanceNetworkInterface& WithInterfaceType(InterfaceTypeT&& value) {
    SetInterfaceType(std::forward<InterfaceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IPv4 delegated prefixes that are assigned to the network interface.</p>
   */
  inline const Aws::Vector<InstanceIpv4Prefix>& GetIpv4Prefixes() const { return m_ipv4Prefixes; }
  inline bool Ipv4PrefixesHasBeenSet() const { return m_ipv4PrefixesHasBeenSet; }
  template <typename Ipv4PrefixesT = Aws::Vector<InstanceIpv4Prefix>>
  void SetIpv4Prefixes(Ipv4PrefixesT&& value) {
    m_ipv4PrefixesHasBeenSet = true;
    m_ipv4Prefixes = std::forward<Ipv4PrefixesT>(value);
  }
  template <typename Ipv4PrefixesT = Aws::Vector<InstanceIpv4Prefix>>
  InstanceNetworkInterface& WithIpv4Prefixes(Ipv4PrefixesT&& value) {
    SetIpv4Prefixes(std::forward<Ipv4PrefixesT>(value));
    return *this;
  }
  template <typename Ipv4PrefixesT = InstanceIpv4Prefix>
  InstanceNetworkInterface& AddIpv4Prefixes(Ipv4PrefixesT&& value) {
    m_ipv4PrefixesHasBeenSet = true;
    m_ipv4Prefixes.emplace_back(std::forward<Ipv4PrefixesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IPv6 delegated prefixes that are assigned to the network interface.</p>
   */
  inline const Aws::Vector<InstanceIpv6Prefix>& GetIpv6Prefixes() const { return m_ipv6Prefixes; }
  inline bool Ipv6PrefixesHasBeenSet() const { return m_ipv6PrefixesHasBeenSet; }
  template <typename Ipv6PrefixesT = Aws::Vector<InstanceIpv6Prefix>>
  void SetIpv6Prefixes(Ipv6PrefixesT&& value) {
    m_ipv6PrefixesHasBeenSet = true;
    m_ipv6Prefixes = std::forward<Ipv6PrefixesT>(value);
  }
  template <typename Ipv6PrefixesT = Aws::Vector<InstanceIpv6Prefix>>
  InstanceNetworkInterface& WithIpv6Prefixes(Ipv6PrefixesT&& value) {
    SetIpv6Prefixes(std::forward<Ipv6PrefixesT>(value));
    return *this;
  }
  template <typename Ipv6PrefixesT = InstanceIpv6Prefix>
  InstanceNetworkInterface& AddIpv6Prefixes(Ipv6PrefixesT&& value) {
    m_ipv6PrefixesHasBeenSet = true;
    m_ipv6Prefixes.emplace_back(std::forward<Ipv6PrefixesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A security group connection tracking configuration that enables you to set
   * the timeout for connection tracking on an Elastic network interface. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/security-group-connection-tracking.html#connection-tracking-timeouts">Connection
   * tracking timeouts</a> in the <i>Amazon EC2 User Guide</i>.</p>
   */
  inline const ConnectionTrackingSpecificationResponse& GetConnectionTrackingConfiguration() const {
    return m_connectionTrackingConfiguration;
  }
  inline bool ConnectionTrackingConfigurationHasBeenSet() const { return m_connectionTrackingConfigurationHasBeenSet; }
  template <typename ConnectionTrackingConfigurationT = ConnectionTrackingSpecificationResponse>
  void SetConnectionTrackingConfiguration(ConnectionTrackingConfigurationT&& value) {
    m_connectionTrackingConfigurationHasBeenSet = true;
    m_connectionTrackingConfiguration = std::forward<ConnectionTrackingConfigurationT>(value);
  }
  template <typename ConnectionTrackingConfigurationT = ConnectionTrackingSpecificationResponse>
  InstanceNetworkInterface& WithConnectionTrackingConfiguration(ConnectionTrackingConfigurationT&& value) {
    SetConnectionTrackingConfiguration(std::forward<ConnectionTrackingConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The service provider that manages the network interface.</p>
   */
  inline const OperatorResponse& GetOperator() const { return m_operator; }
  inline bool OperatorHasBeenSet() const { return m_operatorHasBeenSet; }
  template <typename OperatorT = OperatorResponse>
  void SetOperator(OperatorT&& value) {
    m_operatorHasBeenSet = true;
    m_operator = std::forward<OperatorT>(value);
  }
  template <typename OperatorT = OperatorResponse>
  InstanceNetworkInterface& WithOperator(OperatorT&& value) {
    SetOperator(std::forward<OperatorT>(value));
    return *this;
  }
  ///@}
 private:
  InstanceNetworkInterfaceAssociation m_association;

  InstanceNetworkInterfaceAttachment m_attachment;

  Aws::String m_description;

  Aws::Vector<GroupIdentifier> m_groups;

  Aws::Vector<InstanceIpv6Address> m_ipv6Addresses;

  Aws::String m_macAddress;

  Aws::String m_networkInterfaceId;

  Aws::String m_ownerId;

  Aws::String m_privateDnsName;

  Aws::String m_privateIpAddress;

  Aws::Vector<InstancePrivateIpAddress> m_privateIpAddresses;

  bool m_sourceDestCheck{false};

  NetworkInterfaceStatus m_status{NetworkInterfaceStatus::NOT_SET};

  Aws::String m_subnetId;

  Aws::String m_vpcId;

  Aws::String m_interfaceType;

  Aws::Vector<InstanceIpv4Prefix> m_ipv4Prefixes;

  Aws::Vector<InstanceIpv6Prefix> m_ipv6Prefixes;

  ConnectionTrackingSpecificationResponse m_connectionTrackingConfiguration;

  OperatorResponse m_operator;
  bool m_associationHasBeenSet = false;
  bool m_attachmentHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_groupsHasBeenSet = false;
  bool m_ipv6AddressesHasBeenSet = false;
  bool m_macAddressHasBeenSet = false;
  bool m_networkInterfaceIdHasBeenSet = false;
  bool m_ownerIdHasBeenSet = false;
  bool m_privateDnsNameHasBeenSet = false;
  bool m_privateIpAddressHasBeenSet = false;
  bool m_privateIpAddressesHasBeenSet = false;
  bool m_sourceDestCheckHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_subnetIdHasBeenSet = false;
  bool m_vpcIdHasBeenSet = false;
  bool m_interfaceTypeHasBeenSet = false;
  bool m_ipv4PrefixesHasBeenSet = false;
  bool m_ipv6PrefixesHasBeenSet = false;
  bool m_connectionTrackingConfigurationHasBeenSet = false;
  bool m_operatorHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
