#!/usr/bin/python

# ----------------------------------------------------------------------------
#
#  Copyright (C) 2013-2022 Fons Adriaensen <fons@linuxaudio.org>
#    
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http:#www.gnu.org/licenses/>.
#
# ----------------------------------------------------------------------------


import sys
import matplotlib.pyplot as plt
import numpy as np
from cmath import exp
from zita_jacktools.jacksignal import JackSignal
sys.path.append ('..')
from utils.sinewave import *


# Harmonic distortion + noise measurement, 'analog style'.

# Run jack_nonlin 0.01 0.01 0.01
# Run jaaa and jnoisemeter

# This program
#
# 1. Creates a 1 kHz test signal, sends it to jack_nonlin
#    to add some distortion, and captures the result.
# 2. Removes the fundamental from the captured signal.
# 3. Plays the result to jaaa and jnoisemeter for
#    measurement and analysis. In jnoisemeter, use at
#    least the DC filter (even harmonics include DC).

# Test level (dB) 
#
Level = 0.0

# Test frequency (Hz) 
#
Freq = 1000


# Create a Jacksignal object.
#
J = JackSignal("THDtest")
if J.get_state() < 0:
    print ("Failed to create JackSignal -- is the server running ?")
    exit(1)

# Get Jack info.
#
name, Fsamp, period = J.get_jack_info()

# Create ports and connect.
#
J.create_output (0, "out_1")
J.create_output (1, "out_2")
J.create_input (0, "in_1")
J.silence()
J.connect_output (1, "jaaa:in_1")
J.connect_output (1, "jnoisemeter:in_1")
J.connect_output (0, "jack_nonlin:in")
J.connect_input (0, "jack_nonlin:out")

# Length of test signals.
#
siglen = int (Fsamp + 0.5) # One second
margin = siglen
buflen = siglen + margin

# Generate test signal, will be scaled for output.
#
Freq = adjust_freq (Freq, Fsamp, siglen)
Aout = gen_sinewave (1.00, Freq, Fsamp, buflen).astype (np.float32)

# Create input buffer.
#
Ainp = np.zeros ((siglen,), dtype = np.float32)

# Generate reference signal for selective measurement.
#
M1 = gen_complex (Freq, Fsamp, siglen, True)

# Define signal buffers and run test. 
# We skip the first half second.
#
amp = pow (10.0, Level / 20.0) 
J.set_output_data (0, amp * Aout)
J.set_input_data (0, Ainp, nskip = margin // 2) 
J.process()
J.wait()

# Subtract detected fundamental.    
#
M1 = gen_complex (Freq, Fsamp, siglen, False)
r, p = sigdetect (Ainp, M1)
a = r * exp ((0 -1j) * p)
Ainp -= (a * M1).real

# Output residual.
#
J.set_output_data (0, None)
J.set_output_data (1, Ainp, nloop = 1000)
J.set_input_data (0, None)
J.process()
J.wait()
