/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.utils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.TimeZone;
import org.apache.flink.annotation.Internal;
import org.apache.flink.table.api.TableException;
import org.apache.flink.table.data.DecimalData;
import org.apache.flink.table.data.TimestampData;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.LogicalTypeRoot;
import org.apache.flink.table.types.logical.TimestampType;
import org.apache.flink.table.utils.ThreadLocalCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class DateTimeUtils {
    private static final Logger LOG = LoggerFactory.getLogger(DateTimeUtils.class);
    public static final int EPOCH_JULIAN = 2440588;
    private static final long MILLIS_PER_SECOND = 1000L;
    private static final long MILLIS_PER_MINUTE = 60000L;
    private static final long MILLIS_PER_HOUR = 3600000L;
    public static final long MILLIS_PER_DAY = 86400000L;
    private static final String DATE_FORMAT_STRING = "yyyy-MM-dd";
    private static final String TIME_FORMAT_STRING = "HH:mm:ss";
    private static final String TIMESTAMP_FORMAT_STRING = "yyyy-MM-dd HH:mm:ss";
    public static final TimeZone UTC_ZONE = TimeZone.getTimeZone("UTC");
    public static final TimeZone LOCAL_TZ = TimeZone.getDefault();
    private static final long MIN_EPOCH_MILLS = -62167219200000L;
    private static final long MIN_EPOCH_SECONDS = -62167219200L;
    private static final long MAX_EPOCH_MILLS = 253402300799999L;
    private static final long MAX_EPOCH_SECONDS = 253402300799L;
    private static final DateTimeFormatter DEFAULT_TIMESTAMP_FORMATTER = new DateTimeFormatterBuilder().appendPattern("yyyy-[MM][M]-[dd][d]").optionalStart().appendPattern(" [HH][H]:[mm][m]:[ss][s]").appendFraction(ChronoField.NANO_OF_SECOND, 0, 9, true).optionalEnd().toFormatter();
    private static final ThreadLocalCache<String, SimpleDateFormat> FORMATTER_CACHE = ThreadLocalCache.of(SimpleDateFormat::new);
    private static final ThreadLocalCache<String, DateTimeFormatter> DATETIME_FORMATTER_CACHE = ThreadLocalCache.of(DateTimeFormatter::ofPattern);
    private static final ThreadLocalCache<String, TimeZone> TIMEZONE_CACHE = ThreadLocalCache.of(TimeZone::getTimeZone);
    private static final TimestampType REUSE_TIMESTAMP_TYPE = new TimestampType(9);

    public static Date toSQLDate(int v) {
        long t = (long)v * 86400000L;
        return new Date(t - (long)LOCAL_TZ.getOffset(t));
    }

    public static Time toSQLTime(int v) {
        return new Time(v - LOCAL_TZ.getOffset(v));
    }

    public static Timestamp toSQLTimestamp(long v) {
        return new Timestamp(v - (long)LOCAL_TZ.getOffset(v));
    }

    public static int toInternal(Date date) {
        long ts = date.getTime() + (long)LOCAL_TZ.getOffset(date.getTime());
        return (int)(ts / 86400000L);
    }

    public static int toInternal(Time time) {
        long ts = time.getTime() + (long)LOCAL_TZ.getOffset(time.getTime());
        return (int)(ts % 86400000L);
    }

    public static long toInternal(Timestamp ts) {
        long time = ts.getTime();
        return time + (long)LOCAL_TZ.getOffset(time);
    }

    public static LocalDate toLocalDate(int date) {
        return DateTimeUtils.julianToLocalDate(date + 2440588);
    }

    private static LocalDate julianToLocalDate(int julian) {
        int j = julian + 32044;
        int g = j / 146097;
        int dg = j % 146097;
        int c = (dg / 36524 + 1) * 3 / 4;
        int dc = dg - c * 36524;
        int b = dc / 1461;
        int db = dc % 1461;
        int a = (db / 365 + 1) * 3 / 4;
        int da = db - a * 365;
        int y = g * 400 + c * 100 + b * 4 + a;
        int m = (da * 5 + 308) / 153 - 2;
        int d = da - (m + 4) * 153 / 5 + 122;
        int year = y - 4800 + (m + 2) / 12;
        int month = (m + 2) % 12 + 1;
        int day = d + 1;
        return LocalDate.of(year, month, day);
    }

    public static int toInternal(LocalDate date) {
        return DateTimeUtils.ymdToUnixDate(date.getYear(), date.getMonthValue(), date.getDayOfMonth());
    }

    private static int ymdToUnixDate(int year, int month, int day) {
        int julian = DateTimeUtils.ymdToJulian(year, month, day);
        return julian - 2440588;
    }

    private static int ymdToJulian(int year, int month, int day) {
        int a = (14 - month) / 12;
        int y = year + 4800 - a;
        int m = month + 12 * a - 3;
        return day + (153 * m + 2) / 5 + 365 * y + y / 4 - y / 100 + y / 400 - 32045;
    }

    public static LocalTime toLocalTime(int time) {
        int h = time / 3600000;
        int time2 = time % 3600000;
        int m = time2 / 60000;
        int time3 = time2 % 60000;
        int s = time3 / 1000;
        int ms = time3 % 1000;
        return LocalTime.of(h, m, s, ms * 1000000);
    }

    public static int toInternal(LocalTime time) {
        return time.getHour() * 3600000 + time.getMinute() * 60000 + time.getSecond() * 1000 + time.getNano() / 1000000;
    }

    public static LocalDateTime toLocalDateTime(long timestamp) {
        int date = (int)(timestamp / 86400000L);
        int time = (int)(timestamp % 86400000L);
        if (time < 0) {
            --date;
            time = (int)((long)time + 86400000L);
        }
        LocalDate localDate = DateTimeUtils.toLocalDate(date);
        LocalTime localTime = DateTimeUtils.toLocalTime(time);
        return LocalDateTime.of(localDate, localTime);
    }

    public static long toTimestampMillis(LocalDateTime dateTime) {
        return DateTimeUtils.unixTimestamp(dateTime.getYear(), dateTime.getMonthValue(), dateTime.getDayOfMonth(), dateTime.getHour(), dateTime.getMinute(), dateTime.getSecond(), dateTime.getNano() / 1000000);
    }

    private static long unixTimestamp(int year, int month, int day, int hour, int minute, int second, int mills) {
        int date = DateTimeUtils.ymdToUnixDate(year, month, day);
        return (long)date * 86400000L + (long)hour * 3600000L + (long)minute * 60000L + (long)second * 1000L + (long)mills;
    }

    public static TimestampData toTimestampData(long v, int precision) {
        switch (precision) {
            case 0: {
                if (-62167219200L <= v && v <= 253402300799L) {
                    return DateTimeUtils.timestampDataFromEpochMills(v * 1000L);
                }
                return null;
            }
            case 3: {
                return DateTimeUtils.timestampDataFromEpochMills(v);
            }
        }
        throw new TableException("The precision value '" + precision + "' for function TO_TIMESTAMP_LTZ(numeric, precision) is unsupported, the supported value is '0' for second or '3' for millisecond.");
    }

    public static TimestampData toTimestampData(double v, int precision) {
        switch (precision) {
            case 0: {
                if (-6.21672192E10 <= v && v <= 2.53402300799E11) {
                    return DateTimeUtils.timestampDataFromEpochMills((long)(v * 1000.0));
                }
                return null;
            }
            case 3: {
                return DateTimeUtils.timestampDataFromEpochMills((long)v);
            }
        }
        throw new TableException("The precision value '" + precision + "' for function TO_TIMESTAMP_LTZ(numeric, precision) is unsupported, the supported value is '0' for second or '3' for millisecond.");
    }

    public static TimestampData toTimestampData(DecimalData v, int precision) {
        switch (precision) {
            case 0: {
                long epochMills = v.toBigDecimal().setScale(0, RoundingMode.DOWN).longValue() * 1000L;
                return DateTimeUtils.timestampDataFromEpochMills(epochMills);
            }
            case 3: {
                long epochMills = DateTimeUtils.toMillis(v);
                return DateTimeUtils.timestampDataFromEpochMills(epochMills);
            }
        }
        throw new TableException("The precision value '" + precision + "' for function TO_TIMESTAMP_LTZ(numeric, precision) is unsupported, the supported value is '0' for second or '3' for millisecond.");
    }

    private static TimestampData timestampDataFromEpochMills(long epochMills) {
        if (-62167219200000L <= epochMills && epochMills <= 253402300799999L) {
            return TimestampData.fromEpochMillis(epochMills);
        }
        return null;
    }

    private static long toMillis(DecimalData v) {
        return v.toBigDecimal().setScale(0, RoundingMode.DOWN).longValue();
    }

    public static TimestampData parseTimestampData(String dateStr) throws DateTimeException {
        return DateTimeUtils.parseTimestampData(dateStr, 3);
    }

    public static TimestampData parseTimestampData(String dateStr, int precision) throws DateTimeException {
        return TimestampData.fromLocalDateTime(DateTimeUtils.fromTemporalAccessor(DEFAULT_TIMESTAMP_FORMATTER.parse(dateStr), precision));
    }

    public static TimestampData parseTimestampData(String dateStr, int precision, TimeZone timeZone) throws DateTimeException {
        return TimestampData.fromInstant(DateTimeUtils.fromTemporalAccessor(DEFAULT_TIMESTAMP_FORMATTER.parse(dateStr), precision).atZone(timeZone.toZoneId()).toInstant());
    }

    public static TimestampData parseTimestampData(String dateStr, String format) {
        DateTimeFormatter formatter = DATETIME_FORMATTER_CACHE.get(format);
        try {
            TemporalAccessor accessor = formatter.parse(dateStr);
            LocalDateTime ldt = DateTimeUtils.fromTemporalAccessor(accessor, 3);
            return TimestampData.fromLocalDateTime(ldt);
        }
        catch (DateTimeParseException e) {
            try {
                dateStr = dateStr.trim();
                int space = dateStr.indexOf(32);
                if (space >= 0) {
                    Timestamp ts = Timestamp.valueOf(dateStr);
                    return TimestampData.fromTimestamp(ts);
                }
                Date dt = Date.valueOf(dateStr);
                return TimestampData.fromLocalDateTime(LocalDateTime.of(dt.toLocalDate(), LocalTime.MIDNIGHT));
            }
            catch (IllegalArgumentException ie) {
                return null;
            }
        }
    }

    private static LocalDateTime fromTemporalAccessor(TemporalAccessor accessor, int precision) {
        int nanoOfSecond;
        int year = accessor.isSupported(ChronoField.YEAR) ? accessor.get(ChronoField.YEAR) : 1970;
        int month = accessor.isSupported(ChronoField.MONTH_OF_YEAR) ? accessor.get(ChronoField.MONTH_OF_YEAR) : 1;
        int day = accessor.isSupported(ChronoField.DAY_OF_MONTH) ? accessor.get(ChronoField.DAY_OF_MONTH) : 1;
        int hour = accessor.isSupported(ChronoField.HOUR_OF_DAY) ? accessor.get(ChronoField.HOUR_OF_DAY) : 0;
        int minute = accessor.isSupported(ChronoField.MINUTE_OF_HOUR) ? accessor.get(ChronoField.MINUTE_OF_HOUR) : 0;
        int second = accessor.isSupported(ChronoField.SECOND_OF_MINUTE) ? accessor.get(ChronoField.SECOND_OF_MINUTE) : 0;
        int n = nanoOfSecond = accessor.isSupported(ChronoField.NANO_OF_SECOND) ? accessor.get(ChronoField.NANO_OF_SECOND) : 0;
        if (precision == 0) {
            nanoOfSecond = 0;
        } else if (precision != 9) {
            nanoOfSecond = (int)DateTimeUtils.floor(nanoOfSecond, DateTimeUtils.powerX(10L, 9 - precision));
        }
        return LocalDateTime.of(year, month, day, hour, minute, second, nanoOfSecond);
    }

    private static long parseTimestampMillis(String dateStr, String format, TimeZone tz) throws ParseException {
        SimpleDateFormat formatter = FORMATTER_CACHE.get(format);
        formatter.setTimeZone(tz);
        return formatter.parse(dateStr).getTime();
    }

    private static long parseTimestampTz(String dateStr, String tzStr) throws ParseException {
        TimeZone tz = TIMEZONE_CACHE.get(tzStr);
        return DateTimeUtils.parseTimestampMillis(dateStr, TIMESTAMP_FORMAT_STRING, tz);
    }

    public static int parseDate(String dateStr, String fromFormat) {
        long ts = DateTimeUtils.internalParseTimestampMillis(dateStr, fromFormat, TimeZone.getTimeZone("UTC"));
        ZoneId zoneId = ZoneId.of("UTC");
        Instant instant = Instant.ofEpochMilli(ts);
        ZonedDateTime zdt = ZonedDateTime.ofInstant(instant, zoneId);
        return DateTimeUtils.ymdToUnixDate(zdt.getYear(), zdt.getMonthValue(), zdt.getDayOfMonth());
    }

    public static Integer parseDate(String s) {
        int d;
        int m;
        int y;
        int hyphen1;
        int ws1 = s.indexOf(" ");
        if (ws1 > 0) {
            s = s.substring(0, ws1);
        }
        if ((hyphen1 = s.indexOf(45)) < 0) {
            if (!DateTimeUtils.isInteger(s.trim())) {
                return null;
            }
            y = Integer.parseInt(s.trim());
            m = 1;
            d = 1;
        } else {
            if (!DateTimeUtils.isInteger(s.substring(0, hyphen1).trim())) {
                return null;
            }
            y = Integer.parseInt(s.substring(0, hyphen1).trim());
            int hyphen2 = s.indexOf(45, hyphen1 + 1);
            if (hyphen2 < 0) {
                if (!DateTimeUtils.isInteger(s.substring(hyphen1 + 1).trim())) {
                    return null;
                }
                m = Integer.parseInt(s.substring(hyphen1 + 1).trim());
                d = 1;
            } else {
                if (!DateTimeUtils.isInteger(s.substring(hyphen1 + 1, hyphen2).trim())) {
                    return null;
                }
                m = Integer.parseInt(s.substring(hyphen1 + 1, hyphen2).trim());
                if (!DateTimeUtils.isInteger(s.substring(hyphen2 + 1).trim())) {
                    return null;
                }
                d = Integer.parseInt(s.substring(hyphen2 + 1).trim());
            }
        }
        if (!DateTimeUtils.isIllegalDate(y, m, d)) {
            return null;
        }
        return DateTimeUtils.ymdToUnixDate(y, m, d);
    }

    public static Integer parseTime(String v) {
        int milli;
        int second;
        int minute;
        int hour;
        int timezoneMinute;
        int timezoneHour;
        boolean start = false;
        int colon1 = v.indexOf(58, 0);
        int operator = -1;
        int end = v.length();
        int timezone = v.indexOf(45, 0);
        if (timezone < 0) {
            timezone = v.indexOf(43, 0);
            operator = 1;
        }
        if (timezone < 0) {
            timezoneHour = 0;
            timezoneMinute = 0;
        } else {
            end = timezone;
            int colon3 = v.indexOf(58, timezone);
            if (colon3 < 0) {
                if (!DateTimeUtils.isInteger(v.substring(timezone + 1).trim())) {
                    return null;
                }
                timezoneHour = Integer.parseInt(v.substring(timezone + 1).trim());
                timezoneMinute = 0;
            } else {
                if (!DateTimeUtils.isInteger(v.substring(timezone + 1, colon3).trim())) {
                    return null;
                }
                timezoneHour = Integer.parseInt(v.substring(timezone + 1, colon3).trim());
                if (!DateTimeUtils.isInteger(v.substring(colon3 + 1).trim())) {
                    return null;
                }
                timezoneMinute = Integer.parseInt(v.substring(colon3 + 1).trim());
            }
        }
        if (colon1 < 0) {
            if (!DateTimeUtils.isInteger(v.substring(0, end).trim())) {
                return null;
            }
            hour = Integer.parseInt(v.substring(0, end).trim());
            minute = 0;
            second = 0;
            milli = 0;
        } else {
            if (!DateTimeUtils.isInteger(v.substring(0, colon1).trim())) {
                return null;
            }
            hour = Integer.parseInt(v.substring(0, colon1).trim());
            int colon2 = v.indexOf(58, colon1 + 1);
            if (colon2 < 0) {
                if (!DateTimeUtils.isInteger(v.substring(colon1 + 1, end).trim())) {
                    return null;
                }
                minute = Integer.parseInt(v.substring(colon1 + 1, end).trim());
                second = 0;
                milli = 0;
            } else {
                if (!DateTimeUtils.isInteger(v.substring(colon1 + 1, colon2).trim())) {
                    return null;
                }
                minute = Integer.parseInt(v.substring(colon1 + 1, colon2).trim());
                int dot = v.indexOf(46, colon2);
                if (dot < 0) {
                    if (!DateTimeUtils.isInteger(v.substring(colon2 + 1, end).trim())) {
                        return null;
                    }
                    second = Integer.parseInt(v.substring(colon2 + 1, end).trim());
                    milli = 0;
                } else {
                    if (!DateTimeUtils.isInteger(v.substring(colon2 + 1, dot).trim())) {
                        return null;
                    }
                    second = Integer.parseInt(v.substring(colon2 + 1, dot).trim());
                    milli = DateTimeUtils.parseFraction(v.substring(dot + 1, end).trim());
                }
            }
        }
        return (hour += operator * timezoneHour) * 3600000 + (minute += operator * timezoneMinute) * 60000 + second * 1000 + milli;
    }

    private static int parseFraction(String v) {
        int multiplier = 100;
        int r = 0;
        for (int i = 0; i < v.length(); ++i) {
            char c = v.charAt(i);
            int x = c < '0' || c > '9' ? 0 : c - 48;
            r += multiplier * x;
            if (multiplier < 10) {
                if (i + 1 >= v.length() || v.charAt(i + 1) < '5') break;
                ++r;
                break;
            }
            multiplier /= 10;
        }
        return r;
    }

    public static String formatTimestamp(TimestampData ts, String format) {
        return DateTimeUtils.formatTimestamp(ts, format, ZoneId.of("UTC"));
    }

    public static String formatTimestamp(TimestampData ts, String format, TimeZone zone) {
        return DateTimeUtils.formatTimestamp(ts, format, zone.toZoneId());
    }

    private static String formatTimestamp(TimestampData ts, int precision) {
        LocalDateTime ldt = ts.toLocalDateTime();
        String fraction = DateTimeUtils.pad(9, ldt.getNano());
        if (fraction.length() > precision) {
            fraction = fraction.substring(0, precision);
        }
        StringBuilder ymdhms = DateTimeUtils.ymdhms(new StringBuilder(), ldt.getYear(), ldt.getMonthValue(), ldt.getDayOfMonth(), ldt.getHour(), ldt.getMinute(), ldt.getSecond());
        if (fraction.length() > 0) {
            ymdhms.append(".").append(fraction);
        }
        return ymdhms.toString();
    }

    public static String formatTimestamp(TimestampData ts, TimeZone tz, int precision) {
        return DateTimeUtils.formatTimestamp(DateTimeUtils.timestampWithLocalZoneToTimestamp(ts, tz), precision);
    }

    private static String formatTimestamp(TimestampData ts, String format, ZoneId zoneId) {
        DateTimeFormatter formatter = DATETIME_FORMATTER_CACHE.get(format);
        Instant instant = ts.toInstant();
        return LocalDateTime.ofInstant(instant, zoneId).format(formatter);
    }

    public static String formatTimestampMillis(long ts, String format, TimeZone tz) {
        SimpleDateFormat formatter = FORMATTER_CACHE.get(format);
        formatter.setTimeZone(tz);
        java.util.Date dateTime = new java.util.Date(ts);
        return formatter.format(dateTime);
    }

    public static String formatTimestampString(String dateStr, String fromFormat, String toFormat, TimeZone tz) {
        return DateTimeUtils.formatTimestampStringWithOffset(dateStr, fromFormat, toFormat, tz, 0L);
    }

    public static String formatTimestampStringWithOffset(String dateStr, String fromFormat, String toFormat, TimeZone tz, long offsetMills) {
        SimpleDateFormat fromFormatter = FORMATTER_CACHE.get(fromFormat);
        fromFormatter.setTimeZone(tz);
        SimpleDateFormat toFormatter = FORMATTER_CACHE.get(toFormat);
        toFormatter.setTimeZone(tz);
        try {
            java.util.Date date = fromFormatter.parse(dateStr);
            if (offsetMills != 0L) {
                date = new java.util.Date(date.getTime() + offsetMills);
            }
            return toFormatter.format(date);
        }
        catch (ParseException e) {
            LOG.error("Exception when formatting: '" + dateStr + "' from: '" + fromFormat + "' to: '" + toFormat + "' with offsetMills: '" + offsetMills + "'", (Throwable)e);
            return null;
        }
    }

    public static String formatTimestampString(String dateStr, String toFormat, TimeZone tz) {
        return DateTimeUtils.formatTimestampString(dateStr, TIMESTAMP_FORMAT_STRING, toFormat, tz);
    }

    public static String formatTimestampString(String dateStr, String toFormat) {
        return DateTimeUtils.formatTimestampString(dateStr, toFormat, UTC_ZONE);
    }

    public static String formatTimestampMillis(int time, int precision) {
        StringBuilder buf = new StringBuilder(8 + (precision > 0 ? precision + 1 : 0));
        DateTimeUtils.formatTimestampMillis(buf, time, precision);
        return buf.toString();
    }

    private static void formatTimestampMillis(StringBuilder buf, int time, int precision) {
        while (time < 0) {
            time = (int)((long)time + 86400000L);
        }
        int h = time / 3600000;
        int time2 = time % 3600000;
        int m = time2 / 60000;
        int time3 = time2 % 60000;
        int s = time3 / 1000;
        int ms = time3 % 1000;
        DateTimeUtils.int2(buf, h);
        buf.append(':');
        DateTimeUtils.int2(buf, m);
        buf.append(':');
        DateTimeUtils.int2(buf, s);
        if (precision > 0) {
            buf.append('.');
            while (precision > 0) {
                buf.append((char)(48 + ms / 100));
                ms %= 100;
                if ((ms *= 10) == 0) break;
                --precision;
            }
        }
    }

    private static void int2(StringBuilder buf, int i) {
        buf.append((char)(48 + i / 10 % 10));
        buf.append((char)(48 + i % 10));
    }

    public static String formatDate(int date) {
        StringBuilder buf = new StringBuilder(10);
        DateTimeUtils.formatDate(buf, date);
        return buf.toString();
    }

    private static void formatDate(StringBuilder buf, int date) {
        DateTimeUtils.julianToString(buf, date + 2440588);
    }

    private static void julianToString(StringBuilder buf, int julian) {
        int j = julian + 32044;
        int g = j / 146097;
        int dg = j % 146097;
        int c = (dg / 36524 + 1) * 3 / 4;
        int dc = dg - c * 36524;
        int b = dc / 1461;
        int db = dc % 1461;
        int a = (db / 365 + 1) * 3 / 4;
        int da = db - a * 365;
        int y = g * 400 + c * 100 + b * 4 + a;
        int m = (da * 5 + 308) / 153 - 2;
        int d = da - (m + 4) * 153 / 5 + 122;
        int year = y - 4800 + (m + 2) / 12;
        int month = (m + 2) % 12 + 1;
        int day = d + 1;
        DateTimeUtils.int4(buf, year);
        buf.append('-');
        DateTimeUtils.int2(buf, month);
        buf.append('-');
        DateTimeUtils.int2(buf, day);
    }

    public static String formatIntervalYearMonth(int v) {
        StringBuilder buf = new StringBuilder();
        if (v >= 0) {
            buf.append('+');
        } else {
            buf.append('-');
            v = -v;
        }
        int y = v / 12;
        int m = v % 12;
        buf.append(y);
        buf.append('-');
        DateTimeUtils.number(buf, m, 2);
        return buf.toString();
    }

    public static StringBuilder number(StringBuilder buf, int v, int n) {
        for (int k = DateTimeUtils.digitCount(v); k < n; ++k) {
            buf.append('0');
        }
        return buf.append(v);
    }

    private static int digitCount(int v) {
        int n = 1;
        while ((v /= 10) != 0) {
            ++n;
        }
        return n;
    }

    private static long roundUp(long dividend, long divisor) {
        long remainder = dividend % divisor;
        dividend -= remainder;
        if (remainder * 2L > divisor) {
            dividend += divisor;
        }
        return dividend;
    }

    private static void fraction(StringBuilder buf, int scale, long ms) {
        if (scale > 0) {
            buf.append('.');
            long v1 = scale == 3 ? ms : (scale == 2 ? ms / 10L : (scale == 1 ? ms / 100L : 0L));
            DateTimeUtils.number(buf, (int)v1, scale);
        }
    }

    private static long powerX(long a, long b) {
        long x = 1L;
        while (b > 0L) {
            x *= a;
            --b;
        }
        return x;
    }

    public static String formatIntervalDayTime(long v) {
        int scale = 3;
        StringBuilder buf = new StringBuilder();
        if (v >= 0L) {
            buf.append('+');
        } else {
            buf.append('-');
            v = -v;
        }
        v = DateTimeUtils.roundUp(v, DateTimeUtils.powerX(10L, 0L));
        long ms = v % 1000L;
        long s = (v /= 1000L) % 60L;
        long m = (v /= 60L) % 60L;
        long h = (v /= 60L) % 24L;
        long d = v /= 24L;
        buf.append((int)d);
        buf.append(' ');
        DateTimeUtils.number(buf, (int)h, 2);
        buf.append(':');
        DateTimeUtils.number(buf, (int)m, 2);
        buf.append(':');
        DateTimeUtils.number(buf, (int)s, 2);
        DateTimeUtils.fraction(buf, 3, ms);
        return buf.toString();
    }

    private static long internalParseTimestampMillis(String dateStr, String format, TimeZone tz) {
        SimpleDateFormat formatter = FORMATTER_CACHE.get(format);
        formatter.setTimeZone(tz);
        try {
            java.util.Date date = formatter.parse(dateStr);
            return date.getTime();
        }
        catch (ParseException e) {
            LOG.error(String.format("Exception when parsing datetime string '%s' in format '%s'", dateStr, format), (Throwable)e);
            return Long.MIN_VALUE;
        }
    }

    public static long extractFromTimestamp(TimeUnitRange range, TimestampData ts, TimeZone tz) {
        return DateTimeUtils.convertExtract(range, ts, REUSE_TIMESTAMP_TYPE, tz);
    }

    public static long extractFromDate(TimeUnitRange range, long date) {
        return DateTimeUtils.extractFromDate(range, (int)date);
    }

    public static long extractFromDate(TimeUnitRange range, int date) {
        switch (range) {
            case EPOCH: {
                return (long)date * 86400L;
            }
        }
        return DateTimeUtils.julianExtract(range, date + 2440588);
    }

    private static int julianExtract(TimeUnitRange range, int julian) {
        int j = julian + 32044;
        int g = j / 146097;
        int dg = j % 146097;
        int c = (dg / 36524 + 1) * 3 / 4;
        int dc = dg - c * 36524;
        int b = dc / 1461;
        int db = dc % 1461;
        int a = (db / 365 + 1) * 3 / 4;
        int da = db - a * 365;
        int y = g * 400 + c * 100 + b * 4 + a;
        int m = (da * 5 + 308) / 153 - 2;
        int d = da - (m + 4) * 153 / 5 + 122;
        int year = y - 4800 + (m + 2) / 12;
        int month = (m + 2) % 12 + 1;
        int day = d + 1;
        switch (range) {
            case YEAR: {
                return year;
            }
            default: {
                throw new AssertionError((Object)range);
            }
            case MONTH: {
                return month;
            }
            case DAY: {
                return day;
            }
            case ISOYEAR: {
                int weekNumber = DateTimeUtils.getIso8601WeekNumber(julian, year, month, day);
                if (weekNumber == 1 && month == 12) {
                    return year + 1;
                }
                if (month == 1 && weekNumber > 50) {
                    return year - 1;
                }
                return year;
            }
            case QUARTER: {
                return (month + 2) / 3;
            }
            case DOW: {
                return (int)DateTimeUtils.floorMod(julian + 1, 7L) + 1;
            }
            case ISODOW: {
                return (int)DateTimeUtils.floorMod(julian, 7L) + 1;
            }
            case WEEK: {
                return DateTimeUtils.getIso8601WeekNumber(julian, year, month, day);
            }
            case DOY: {
                long janFirst = DateTimeUtils.ymdToJulian(year, 1, 1);
                return (int)((long)julian - janFirst) + 1;
            }
            case DECADE: {
                return year / 10;
            }
            case CENTURY: {
                return year > 0 ? (year + 99) / 100 : (year - 99) / 100;
            }
            case MILLENNIUM: 
        }
        return year > 0 ? (year + 999) / 1000 : (year - 999) / 1000;
    }

    private static long firstMondayOfFirstWeek(int year) {
        long janFirst = DateTimeUtils.ymdToJulian(year, 1, 1);
        long janFirstDow = DateTimeUtils.floorMod(janFirst + 1L, 7L);
        return janFirst + (11L - janFirstDow) % 7L - 3L;
    }

    private static int getIso8601WeekNumber(int julian, int year, int month, int day) {
        long fmofw = DateTimeUtils.firstMondayOfFirstWeek(year);
        if (month == 12 && day > 28) {
            return 31 - day + 4 > 7 - ((int)DateTimeUtils.floorMod(julian, 7L) + 1) && 31 - day + (int)(DateTimeUtils.floorMod(julian, 7L) + 1L) >= 4 ? (int)((long)julian - fmofw) / 7 + 1 : 1;
        }
        if (month == 1 && day < 5) {
            return 4 - day <= 7 - ((int)DateTimeUtils.floorMod(julian, 7L) + 1) && day - (int)(DateTimeUtils.floorMod(julian, 7L) + 1L) >= -3 ? 1 : (int)((long)julian - DateTimeUtils.firstMondayOfFirstWeek(year - 1)) / 7 + 1;
        }
        return (int)((long)julian - fmofw) / 7 + 1;
    }

    private static long floorDiv(long x, long y) {
        long r = x / y;
        if ((x ^ y) < 0L && r * y != x) {
            --r;
        }
        return r;
    }

    private static long floorMod(long x, long y) {
        return x - DateTimeUtils.floorDiv(x, y) * y;
    }

    private static long convertExtract(TimeUnitRange range, TimestampData ts, LogicalType type, TimeZone tz) {
        TimeUnit startUnit = range.startUnit;
        long millisecond = ts.getMillisecond();
        int nanoOfMillisecond = ts.getNanoOfMillisecond();
        long offset = tz.getOffset(millisecond);
        long utcTs = millisecond + offset;
        switch (startUnit) {
            case MILLENNIUM: 
            case CENTURY: 
            case DECADE: 
            case YEAR: 
            case QUARTER: 
            case MONTH: 
            case DAY: 
            case DOW: 
            case DOY: 
            case ISODOW: 
            case ISOYEAR: 
            case WEEK: {
                if (type.is(LogicalTypeRoot.TIMESTAMP_WITHOUT_TIME_ZONE)) {
                    long d = DateTimeUtils.divide(utcTs, TimeUnit.DAY.multiplier);
                    return DateTimeUtils.extractFromDate(range, d);
                }
                if (type.is(LogicalTypeRoot.DATE)) {
                    return DateTimeUtils.divide(utcTs, TimeUnit.DAY.multiplier);
                }
                throw new TableException(startUnit + " for " + type + " is unsupported now.");
            }
            case EPOCH: {
                if (type.isAnyOf(LogicalTypeRoot.TIMESTAMP_WITHOUT_TIME_ZONE, LogicalTypeRoot.DATE)) {
                    return utcTs / 1000L;
                }
                throw new TableException(startUnit + " for " + type + " is unsupported now.");
            }
            case MICROSECOND: {
                if (type.is(LogicalTypeRoot.TIMESTAMP_WITHOUT_TIME_ZONE)) {
                    long millis = DateTimeUtils.divide(DateTimeUtils.mod(utcTs, DateTimeUtils.getFactor(startUnit)), startUnit.multiplier);
                    int micros = nanoOfMillisecond / 1000;
                    return millis + (long)micros;
                }
                throw new TableException(startUnit + " for " + type + " is unsupported now.");
            }
            case NANOSECOND: {
                if (type.is(LogicalTypeRoot.TIMESTAMP_WITHOUT_TIME_ZONE)) {
                    long millis = DateTimeUtils.divide(DateTimeUtils.mod(utcTs, DateTimeUtils.getFactor(startUnit)), startUnit.multiplier);
                    return millis + (long)nanoOfMillisecond;
                }
                throw new TableException(startUnit + " for " + type + " is unsupported now.");
            }
        }
        long res = DateTimeUtils.mod(utcTs, DateTimeUtils.getFactor(startUnit));
        res = DateTimeUtils.divide(res, startUnit.multiplier);
        return res;
    }

    private static long divide(long res, BigDecimal value) {
        if (value.equals(BigDecimal.ONE)) {
            return res;
        }
        if (value.compareTo(BigDecimal.ONE) < 0 && value.signum() == 1) {
            BigDecimal reciprocal = BigDecimal.ONE.divide(value, RoundingMode.UNNECESSARY);
            return reciprocal.multiply(BigDecimal.valueOf(res)).longValue();
        }
        return res / value.longValue();
    }

    private static long mod(long res, BigDecimal value) {
        if (value.equals(BigDecimal.ONE)) {
            return res;
        }
        return res % value.longValue();
    }

    private static BigDecimal getFactor(TimeUnit unit) {
        switch (unit) {
            case DAY: {
                return BigDecimal.ONE;
            }
            case HOUR: {
                return TimeUnit.DAY.multiplier;
            }
            case MINUTE: {
                return TimeUnit.HOUR.multiplier;
            }
            case SECOND: {
                return TimeUnit.MINUTE.multiplier;
            }
            case MICROSECOND: 
            case NANOSECOND: 
            case MILLISECOND: {
                return TimeUnit.SECOND.multiplier;
            }
            case YEAR: {
                return BigDecimal.ONE;
            }
            case MONTH: {
                return TimeUnit.YEAR.multiplier;
            }
            case QUARTER: {
                return TimeUnit.YEAR.multiplier;
            }
            case MILLENNIUM: 
            case CENTURY: 
            case DECADE: {
                return BigDecimal.ONE;
            }
        }
        throw new IllegalArgumentException("Invalid start unit.");
    }

    public static long timestampFloor(TimeUnitRange range, long ts, TimeZone tz) {
        long offset = tz.getOffset(ts);
        long utcTs = ts + offset;
        switch (range) {
            case HOUR: {
                return DateTimeUtils.floor(utcTs, 3600000L) - offset;
            }
            case DAY: {
                return DateTimeUtils.floor(utcTs, 86400000L) - offset;
            }
            case YEAR: 
            case MONTH: 
            case QUARTER: 
            case WEEK: 
            case DECADE: 
            case CENTURY: 
            case MILLENNIUM: {
                int days = (int)(utcTs / 86400000L + 2440588L);
                return DateTimeUtils.julianDateFloor(range, days, true) * 86400000L - offset;
            }
        }
        throw new AssertionError((Object)range);
    }

    public static long timestampCeil(TimeUnitRange range, long ts, TimeZone tz) {
        long offset = tz.getOffset(ts);
        long utcTs = ts + offset;
        switch (range) {
            case HOUR: {
                return DateTimeUtils.ceil(utcTs, 3600000L) - offset;
            }
            case DAY: {
                return DateTimeUtils.ceil(utcTs, 86400000L) - offset;
            }
            case YEAR: 
            case MONTH: 
            case QUARTER: 
            case WEEK: 
            case DECADE: 
            case CENTURY: 
            case MILLENNIUM: {
                int days = (int)(utcTs / 86400000L + 2440588L);
                return DateTimeUtils.julianDateFloor(range, days, false) * 86400000L - offset;
            }
        }
        throw new AssertionError((Object)range);
    }

    private static long floor(long a, long b) {
        long r = a % b;
        if (r < 0L) {
            return a - r - b;
        }
        return a - r;
    }

    private static long ceil(long a, long b) {
        long r = a % b;
        if (r > 0L) {
            return a - r + b;
        }
        return a - r;
    }

    private static long julianDateFloor(TimeUnitRange range, int julian, boolean floor) {
        int b = 0;
        int c = 0;
        if (julian > 2299160) {
            int a = julian + 32044;
            b = (4 * a + 3) / 146097;
            c = a - b * 146097 / 4;
        } else {
            b = 0;
            c = julian + 32082;
        }
        int d = (4 * c + 3) / 1461;
        int e = c - 1461 * d / 4;
        int m = (5 * e + 2) / 153;
        int day = e - (153 * m + 2) / 5 + 1;
        int month = m + 3 - 12 * (m / 10);
        int quarter = (month + 2) / 3;
        int year = b * 100 + d - 4800 + m / 10;
        switch (range) {
            case MILLENNIUM: {
                return floor ? (long)DateTimeUtils.ymdToUnixDate(1000 * ((year + 999) / 1000) - 999, 1, 1) : (long)DateTimeUtils.ymdToUnixDate(1000 * ((year + 999) / 1000) + 1, 1, 1);
            }
            case CENTURY: {
                return floor ? (long)DateTimeUtils.ymdToUnixDate(100 * ((year + 99) / 100) - 99, 1, 1) : (long)DateTimeUtils.ymdToUnixDate(100 * ((year + 99) / 100) + 1, 1, 1);
            }
            case DECADE: {
                return floor ? (long)DateTimeUtils.ymdToUnixDate(10 * (year / 10), 1, 1) : (long)DateTimeUtils.ymdToUnixDate(10 * (1 + year / 10), 1, 1);
            }
            case YEAR: {
                if (!(floor || month <= 1 && day <= 1)) {
                    ++year;
                }
                return DateTimeUtils.ymdToUnixDate(year, 1, 1);
            }
            case MONTH: {
                if (!floor && day > 1) {
                    ++month;
                }
                return DateTimeUtils.ymdToUnixDate(year, month, 1);
            }
            case QUARTER: {
                if (!(floor || month <= 1 && day <= 1)) {
                    ++quarter;
                }
                return DateTimeUtils.ymdToUnixDate(year, quarter * 3 - 2, 1);
            }
            case WEEK: {
                int dow;
                int offset = dow = (int)DateTimeUtils.floorMod(julian + 1, 7L);
                if (!floor && offset > 0) {
                    offset -= 7;
                }
                return DateTimeUtils.ymdToUnixDate(year, month, day) - offset;
            }
            case DAY: {
                int res = DateTimeUtils.ymdToUnixDate(year, month, day);
                return floor ? (long)res : (long)(res + 1);
            }
        }
        throw new AssertionError((Object)range);
    }

    public static String convertTz(String dateStr, String tzFrom, String tzTo) {
        try {
            return DateTimeUtils.formatTimestampTz(DateTimeUtils.parseTimestampTz(dateStr, tzFrom), tzTo);
        }
        catch (ParseException e) {
            return null;
        }
    }

    private static String formatTimestampTz(long ts, String tzStr) {
        TimeZone tz = TIMEZONE_CACHE.get(tzStr);
        return DateTimeUtils.formatTimestampMillis(ts, TIMESTAMP_FORMAT_STRING, tz);
    }

    public static int timestampMillisToDate(long ts) {
        int days = (int)(ts / 86400000L);
        if (days < 0) {
            --days;
        }
        return days;
    }

    public static int timestampMillisToTime(long ts) {
        return (int)(ts % 86400000L);
    }

    public static long fromTimestamp(long ts) {
        return ts;
    }

    public static String formatUnixTimestamp(long unixtime, TimeZone tz) {
        return DateTimeUtils.formatUnixTimestamp(unixtime, TIMESTAMP_FORMAT_STRING, tz);
    }

    public static String formatUnixTimestamp(long unixtime, String format, TimeZone tz) {
        SimpleDateFormat formatter = FORMATTER_CACHE.get(format);
        formatter.setTimeZone(tz);
        java.util.Date date = new java.util.Date(unixtime * 1000L);
        try {
            return formatter.format(date);
        }
        catch (Exception e) {
            LOG.error("Exception when formatting.", (Throwable)e);
            return null;
        }
    }

    public static long unixTimestamp() {
        return System.currentTimeMillis() / 1000L;
    }

    public static long unixTimestamp(long ts) {
        return ts / 1000L;
    }

    public static long unixTimestamp(String dateStr, TimeZone tz) {
        return DateTimeUtils.unixTimestamp(dateStr, TIMESTAMP_FORMAT_STRING, tz);
    }

    public static long unixTimestamp(String dateStr, String format, TimeZone tz) {
        long ts = DateTimeUtils.internalParseTimestampMillis(dateStr, format, tz);
        if (ts == Long.MIN_VALUE) {
            return Long.MIN_VALUE;
        }
        return ts / 1000L;
    }

    public static TimestampData timestampToTimestampWithLocalZone(TimestampData ts, TimeZone tz) {
        return TimestampData.fromInstant(ts.toLocalDateTime().atZone(tz.toZoneId()).toInstant());
    }

    public static TimestampData timestampWithLocalZoneToTimestamp(TimestampData ts, TimeZone tz) {
        return TimestampData.fromLocalDateTime(LocalDateTime.ofInstant(ts.toInstant(), tz.toZoneId()));
    }

    public static int timestampWithLocalZoneToDate(TimestampData ts, TimeZone tz) {
        return DateTimeUtils.toInternal(LocalDateTime.ofInstant(Instant.ofEpochMilli(ts.getMillisecond()), tz.toZoneId()).toLocalDate());
    }

    public static int timestampWithLocalZoneToTime(TimestampData ts, TimeZone tz) {
        return DateTimeUtils.toInternal(LocalDateTime.ofInstant(Instant.ofEpochMilli(ts.getMillisecond()), tz.toZoneId()).toLocalTime());
    }

    public static TimestampData dateToTimestampWithLocalZone(int date, TimeZone tz) {
        return TimestampData.fromInstant(LocalDateTime.of(DateTimeUtils.toLocalDate(date), LocalTime.MIDNIGHT).atZone(tz.toZoneId()).toInstant());
    }

    public static TimestampData timeToTimestampWithLocalZone(int time, TimeZone tz) {
        return TimestampData.fromInstant(DateTimeUtils.toLocalDateTime(time).atZone(tz.toZoneId()).toInstant());
    }

    private static boolean isInteger(String s) {
        boolean isInt = s.length() > 0;
        for (int i = 0; i < s.length(); ++i) {
            if (s.charAt(i) >= '0' && s.charAt(i) <= '9') continue;
            isInt = false;
            break;
        }
        return isInt;
    }

    private static boolean isLeapYear(int s) {
        return s % 400 == 0 || s % 4 == 0 && s % 100 != 0;
    }

    private static boolean isIllegalDate(int y, int m, int d) {
        int[] monthOf31Days = new int[]{1, 3, 5, 7, 8, 10, 12};
        if (y < 0 || y > 9999 || m < 1 || m > 12 || d < 1 || d > 31) {
            return false;
        }
        if (!(m != 2 || d <= 28 || DateTimeUtils.isLeapYear(y) && d == 29)) {
            return false;
        }
        if (d == 31) {
            for (int i : monthOf31Days) {
                if (i != m) continue;
                return true;
            }
            return false;
        }
        return true;
    }

    private static String pad(int length, long v) {
        StringBuilder s = new StringBuilder(Long.toString(v));
        while (s.length() < length) {
            s.insert(0, "0");
        }
        return s.toString();
    }

    private static StringBuilder hms(StringBuilder b, int h, int m, int s) {
        DateTimeUtils.int2(b, h);
        b.append(':');
        DateTimeUtils.int2(b, m);
        b.append(':');
        DateTimeUtils.int2(b, s);
        return b;
    }

    private static StringBuilder ymdhms(StringBuilder b, int year, int month, int day, int h, int m, int s) {
        DateTimeUtils.ymd(b, year, month, day);
        b.append(' ');
        DateTimeUtils.hms(b, h, m, s);
        return b;
    }

    private static StringBuilder ymd(StringBuilder b, int year, int month, int day) {
        DateTimeUtils.int4(b, year);
        b.append('-');
        DateTimeUtils.int2(b, month);
        b.append('-');
        DateTimeUtils.int2(b, day);
        return b;
    }

    private static void int4(StringBuilder buf, int i) {
        buf.append((char)(48 + i / 1000 % 10));
        buf.append((char)(48 + i / 100 % 10));
        buf.append((char)(48 + i / 10 % 10));
        buf.append((char)(48 + i % 10));
    }

    public static TimestampData truncate(TimestampData ts, int precision) {
        String fraction = Integer.toString(ts.toLocalDateTime().getNano());
        if (fraction.length() <= precision) {
            return ts;
        }
        if (precision <= 3) {
            return TimestampData.fromEpochMillis(DateTimeUtils.zeroLastDigits(ts.getMillisecond(), 3 - precision));
        }
        return TimestampData.fromEpochMillis(ts.getMillisecond(), (int)DateTimeUtils.zeroLastDigits(ts.getNanoOfMillisecond(), 9 - precision));
    }

    private static long zeroLastDigits(long l, int n) {
        long tenToTheN = (long)Math.pow(10.0, n);
        return l / tenToTheN * tenToTheN;
    }

    public static long unixDateCeil(TimeUnitRange range, long date) {
        return DateTimeUtils.julianDateFloor(range, (int)date + 2440588, false);
    }

    public static long unixDateFloor(TimeUnitRange range, long date) {
        return DateTimeUtils.julianDateFloor(range, (int)date + 2440588, true);
    }

    public static long unixTimestampFloor(TimeUnitRange range, long timestamp) {
        int date = (int)(timestamp / 86400000L);
        long f = DateTimeUtils.julianDateFloor(range, date + 2440588, true);
        return f * 86400000L;
    }

    public static long unixTimestampCeil(TimeUnitRange range, long timestamp) {
        int date = (int)(timestamp / 86400000L);
        long f = DateTimeUtils.julianDateFloor(range, date + 2440588, false);
        return f * 86400000L;
    }

    public static long addMonths(long timestamp, int m) {
        long millis = DateTimeUtils.floorMod(timestamp, 86400000L);
        long x = DateTimeUtils.addMonths((int)((timestamp -= millis) / 86400000L), m);
        return x * 86400000L + millis;
    }

    public static int addMonths(int date, int m) {
        int last;
        int y0 = (int)DateTimeUtils.extractFromDate(TimeUnitRange.YEAR, date);
        int m0 = (int)DateTimeUtils.extractFromDate(TimeUnitRange.MONTH, date);
        int d0 = (int)DateTimeUtils.extractFromDate(TimeUnitRange.DAY, date);
        int deltaYear = (int)DateTimeUtils.floorDiv(m0 += m, 12L);
        y0 += deltaYear;
        if ((m0 = (int)DateTimeUtils.floorMod(m0, 12L)) == 0) {
            --y0;
            m0 += 12;
        }
        if (d0 > (last = DateTimeUtils.lastDay(y0, m0))) {
            d0 = last;
        }
        return DateTimeUtils.ymdToUnixDate(y0, m0, d0);
    }

    private static int lastDay(int y, int m) {
        switch (m) {
            case 2: {
                return y % 4 == 0 && (y % 100 != 0 || y % 400 == 0) ? 29 : 28;
            }
            case 4: 
            case 6: 
            case 9: 
            case 11: {
                return 30;
            }
        }
        return 31;
    }

    public static int subtractMonths(int date0, int date1) {
        if (date0 < date1) {
            return -DateTimeUtils.subtractMonths(date1, date0);
        }
        int m = (date0 - date1) / 31;
        int date2;
        while ((date2 = DateTimeUtils.addMonths(date1, m)) < date0) {
            int date3 = DateTimeUtils.addMonths(date1, m + 1);
            if (date3 > date0) {
                return m;
            }
            ++m;
        }
        return m;
    }

    public static int subtractMonths(long t0, long t1) {
        int x;
        long millis0 = DateTimeUtils.floorMod(t0, 86400000L);
        int d0 = (int)DateTimeUtils.floorDiv(t0 - millis0, 86400000L);
        long millis1 = DateTimeUtils.floorMod(t1, 86400000L);
        int d1 = (int)DateTimeUtils.floorDiv(t1 - millis1, 86400000L);
        long d2 = DateTimeUtils.addMonths(d1, x = DateTimeUtils.subtractMonths(d0, d1));
        if (d2 == (long)d0 && millis0 < millis1) {
            --x;
        }
        return x;
    }

    @Internal
    public static final class TimeUnitRange
    extends Enum<TimeUnitRange> {
        public static final /* enum */ TimeUnitRange YEAR = new TimeUnitRange(TimeUnit.YEAR, null);
        public static final /* enum */ TimeUnitRange YEAR_TO_MONTH = new TimeUnitRange(TimeUnit.YEAR, TimeUnit.MONTH);
        public static final /* enum */ TimeUnitRange MONTH = new TimeUnitRange(TimeUnit.MONTH, null);
        public static final /* enum */ TimeUnitRange DAY = new TimeUnitRange(TimeUnit.DAY, null);
        public static final /* enum */ TimeUnitRange DAY_TO_HOUR = new TimeUnitRange(TimeUnit.DAY, TimeUnit.HOUR);
        public static final /* enum */ TimeUnitRange DAY_TO_MINUTE = new TimeUnitRange(TimeUnit.DAY, TimeUnit.MINUTE);
        public static final /* enum */ TimeUnitRange DAY_TO_SECOND = new TimeUnitRange(TimeUnit.DAY, TimeUnit.SECOND);
        public static final /* enum */ TimeUnitRange HOUR = new TimeUnitRange(TimeUnit.HOUR, null);
        public static final /* enum */ TimeUnitRange HOUR_TO_MINUTE = new TimeUnitRange(TimeUnit.HOUR, TimeUnit.MINUTE);
        public static final /* enum */ TimeUnitRange HOUR_TO_SECOND = new TimeUnitRange(TimeUnit.HOUR, TimeUnit.SECOND);
        public static final /* enum */ TimeUnitRange MINUTE = new TimeUnitRange(TimeUnit.MINUTE, null);
        public static final /* enum */ TimeUnitRange MINUTE_TO_SECOND = new TimeUnitRange(TimeUnit.MINUTE, TimeUnit.SECOND);
        public static final /* enum */ TimeUnitRange SECOND = new TimeUnitRange(TimeUnit.SECOND, null);
        public static final /* enum */ TimeUnitRange ISOYEAR = new TimeUnitRange(TimeUnit.ISOYEAR, null);
        public static final /* enum */ TimeUnitRange QUARTER = new TimeUnitRange(TimeUnit.QUARTER, null);
        public static final /* enum */ TimeUnitRange WEEK = new TimeUnitRange(TimeUnit.WEEK, null);
        public static final /* enum */ TimeUnitRange MILLISECOND = new TimeUnitRange(TimeUnit.MILLISECOND, null);
        public static final /* enum */ TimeUnitRange MICROSECOND = new TimeUnitRange(TimeUnit.MICROSECOND, null);
        public static final /* enum */ TimeUnitRange NANOSECOND = new TimeUnitRange(TimeUnit.NANOSECOND, null);
        public static final /* enum */ TimeUnitRange DOW = new TimeUnitRange(TimeUnit.DOW, null);
        public static final /* enum */ TimeUnitRange ISODOW = new TimeUnitRange(TimeUnit.ISODOW, null);
        public static final /* enum */ TimeUnitRange DOY = new TimeUnitRange(TimeUnit.DOY, null);
        public static final /* enum */ TimeUnitRange EPOCH = new TimeUnitRange(TimeUnit.EPOCH, null);
        public static final /* enum */ TimeUnitRange DECADE = new TimeUnitRange(TimeUnit.DECADE, null);
        public static final /* enum */ TimeUnitRange CENTURY = new TimeUnitRange(TimeUnit.CENTURY, null);
        public static final /* enum */ TimeUnitRange MILLENNIUM = new TimeUnitRange(TimeUnit.MILLENNIUM, null);
        public final TimeUnit startUnit;
        public final TimeUnit endUnit;
        private static final Map<Pair<TimeUnit>, TimeUnitRange> MAP;
        private static final /* synthetic */ TimeUnitRange[] $VALUES;

        public static TimeUnitRange[] values() {
            return (TimeUnitRange[])$VALUES.clone();
        }

        public static TimeUnitRange valueOf(String name) {
            return Enum.valueOf(TimeUnitRange.class, name);
        }

        private TimeUnitRange(TimeUnit startUnit, TimeUnit endUnit) {
            assert (startUnit != null);
            this.startUnit = startUnit;
            this.endUnit = endUnit;
        }

        public static TimeUnitRange of(TimeUnit startUnit, TimeUnit endUnit) {
            return MAP.get(new Pair<TimeUnit>(startUnit, endUnit));
        }

        private static Map<Pair<TimeUnit>, TimeUnitRange> createMap() {
            HashMap<Pair<TimeUnit>, TimeUnitRange> map = new HashMap<Pair<TimeUnit>, TimeUnitRange>();
            for (TimeUnitRange value : TimeUnitRange.values()) {
                map.put(new Pair<TimeUnit>(value.startUnit, value.endUnit), value);
            }
            return Collections.unmodifiableMap(map);
        }

        public boolean monthly() {
            return this.ordinal() <= MONTH.ordinal();
        }

        static {
            $VALUES = new TimeUnitRange[]{YEAR, YEAR_TO_MONTH, MONTH, DAY, DAY_TO_HOUR, DAY_TO_MINUTE, DAY_TO_SECOND, HOUR, HOUR_TO_MINUTE, HOUR_TO_SECOND, MINUTE, MINUTE_TO_SECOND, SECOND, ISOYEAR, QUARTER, WEEK, MILLISECOND, MICROSECOND, NANOSECOND, DOW, ISODOW, DOY, EPOCH, DECADE, CENTURY, MILLENNIUM};
            MAP = TimeUnitRange.createMap();
        }

        private static class Pair<E> {
            final E left;
            final E right;

            private Pair(E left, E right) {
                this.left = left;
                this.right = right;
            }

            public int hashCode() {
                int k = this.left == null ? 0 : this.left.hashCode();
                int k1 = this.right == null ? 0 : this.right.hashCode();
                return (k << 4 | k) ^ k1;
            }

            public boolean equals(Object obj) {
                return obj == this || obj instanceof Pair && Objects.equals(this.left, ((Pair)obj).left) && Objects.equals(this.right, ((Pair)obj).right);
            }
        }
    }

    @Internal
    public static enum TimeUnit {
        YEAR(true, ' ', BigDecimal.valueOf(12L), null),
        MONTH(true, '-', BigDecimal.ONE, BigDecimal.valueOf(12L)),
        DAY(false, '-', BigDecimal.valueOf(86400000L), null),
        HOUR(false, ' ', BigDecimal.valueOf(3600000L), BigDecimal.valueOf(24L)),
        MINUTE(false, ':', BigDecimal.valueOf(60000L), BigDecimal.valueOf(60L)),
        SECOND(false, ':', BigDecimal.valueOf(1000L), BigDecimal.valueOf(60L)),
        QUARTER(true, '*', BigDecimal.valueOf(3L), BigDecimal.valueOf(4L)),
        ISOYEAR(true, ' ', BigDecimal.valueOf(12L), null),
        WEEK(false, '*', BigDecimal.valueOf(604800000L), BigDecimal.valueOf(53L)),
        MILLISECOND(false, '.', BigDecimal.ONE, BigDecimal.valueOf(1000L)),
        MICROSECOND(false, '.', BigDecimal.ONE.scaleByPowerOfTen(-3), BigDecimal.valueOf(1000000L)),
        NANOSECOND(false, '.', BigDecimal.ONE.scaleByPowerOfTen(-6), BigDecimal.valueOf(1000000000L)),
        DOW(false, '-', null, null),
        ISODOW(false, '-', null, null),
        DOY(false, '-', null, null),
        EPOCH(false, '*', null, null),
        DECADE(true, '*', BigDecimal.valueOf(120L), null),
        CENTURY(true, '*', BigDecimal.valueOf(1200L), null),
        MILLENNIUM(true, '*', BigDecimal.valueOf(12000L), null);

        public final boolean yearMonth;
        public final char separator;
        public final BigDecimal multiplier;
        private final BigDecimal limit;
        private static final TimeUnit[] CACHED_VALUES;

        private TimeUnit(boolean yearMonth, char separator, BigDecimal multiplier, BigDecimal limit) {
            this.yearMonth = yearMonth;
            this.separator = separator;
            this.multiplier = multiplier;
            this.limit = limit;
        }

        public static TimeUnit getValue(int ordinal) {
            return ordinal < 0 || ordinal >= CACHED_VALUES.length ? null : CACHED_VALUES[ordinal];
        }

        public boolean isValidValue(BigDecimal field) {
            return field.compareTo(BigDecimal.ZERO) >= 0 && (this.limit == null || field.compareTo(this.limit) < 0);
        }

        static {
            CACHED_VALUES = TimeUnit.values();
        }
    }
}

