/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.checkpoint;

import java.io.StringWriter;
import java.io.Writer;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.Nullable;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.JobID;
import org.apache.flink.metrics.Gauge;
import org.apache.flink.metrics.MetricGroup;
import org.apache.flink.runtime.checkpoint.AbstractCheckpointStats;
import org.apache.flink.runtime.checkpoint.CheckpointMetrics;
import org.apache.flink.runtime.checkpoint.CheckpointProperties;
import org.apache.flink.runtime.checkpoint.CheckpointStatsCounts;
import org.apache.flink.runtime.checkpoint.CheckpointStatsHistory;
import org.apache.flink.runtime.checkpoint.CheckpointStatsSnapshot;
import org.apache.flink.runtime.checkpoint.CompletedCheckpointStats;
import org.apache.flink.runtime.checkpoint.CompletedCheckpointStatsSummary;
import org.apache.flink.runtime.checkpoint.FailedCheckpointStats;
import org.apache.flink.runtime.checkpoint.JobInitializationMetrics;
import org.apache.flink.runtime.checkpoint.JobInitializationMetricsBuilder;
import org.apache.flink.runtime.checkpoint.PendingCheckpointStats;
import org.apache.flink.runtime.checkpoint.RestoredCheckpointStats;
import org.apache.flink.runtime.checkpoint.SubTaskInitializationMetrics;
import org.apache.flink.runtime.checkpoint.SubtaskStateStats;
import org.apache.flink.runtime.executiongraph.ExecutionAttemptID;
import org.apache.flink.runtime.jobgraph.JobVertexID;
import org.apache.flink.runtime.rest.messages.checkpoints.CheckpointStatistics;
import org.apache.flink.runtime.rest.util.RestMapperUtils;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.flink.traces.Span;
import org.apache.flink.traces.SpanBuilder;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CheckpointStatsTracker {
    private static final Logger LOG = LoggerFactory.getLogger(CheckpointStatsTracker.class);
    private static final ObjectMapper MAPPER = RestMapperUtils.getStrictObjectMapper();
    private final ReentrantLock statsReadWriteLock = new ReentrantLock();
    private final CheckpointStatsCounts counts = new CheckpointStatsCounts();
    private final CompletedCheckpointStatsSummary summary = new CompletedCheckpointStatsSummary();
    private final CheckpointStatsHistory history;
    private final JobID jobID;
    private final MetricGroup metricGroup;
    private int totalNumberOfSubTasks;
    private Optional<JobInitializationMetricsBuilder> jobInitializationMetricsBuilder = Optional.empty();
    private volatile CheckpointStatsSnapshot latestSnapshot;
    private volatile boolean dirty;
    @Nullable
    private volatile CompletedCheckpointStats latestCompletedCheckpoint;
    @VisibleForTesting
    static final String NUMBER_OF_CHECKPOINTS_METRIC = "totalNumberOfCheckpoints";
    @VisibleForTesting
    static final String NUMBER_OF_IN_PROGRESS_CHECKPOINTS_METRIC = "numberOfInProgressCheckpoints";
    @VisibleForTesting
    static final String NUMBER_OF_COMPLETED_CHECKPOINTS_METRIC = "numberOfCompletedCheckpoints";
    @VisibleForTesting
    static final String NUMBER_OF_FAILED_CHECKPOINTS_METRIC = "numberOfFailedCheckpoints";
    @VisibleForTesting
    static final String LATEST_RESTORED_CHECKPOINT_TIMESTAMP_METRIC = "lastCheckpointRestoreTimestamp";
    @VisibleForTesting
    static final String LATEST_COMPLETED_CHECKPOINT_SIZE_METRIC = "lastCheckpointSize";
    @VisibleForTesting
    static final String LATEST_COMPLETED_CHECKPOINT_FULL_SIZE_METRIC = "lastCheckpointFullSize";
    @VisibleForTesting
    static final String LATEST_COMPLETED_CHECKPOINT_DURATION_METRIC = "lastCheckpointDuration";
    @VisibleForTesting
    static final String LATEST_COMPLETED_CHECKPOINT_PROCESSED_DATA_METRIC = "lastCheckpointProcessedData";
    @VisibleForTesting
    static final String LATEST_COMPLETED_CHECKPOINT_PERSISTED_DATA_METRIC = "lastCheckpointPersistedData";
    @VisibleForTesting
    static final String LATEST_COMPLETED_CHECKPOINT_EXTERNAL_PATH_METRIC = "lastCheckpointExternalPath";
    @VisibleForTesting
    static final String LATEST_COMPLETED_CHECKPOINT_ID_METRIC = "lastCompletedCheckpointId";

    public CheckpointStatsTracker(int numRememberedCheckpoints, MetricGroup metricGroup, JobID jobID) {
        this(numRememberedCheckpoints, metricGroup, jobID, Integer.MAX_VALUE);
    }

    CheckpointStatsTracker(int numRememberedCheckpoints, MetricGroup metricGroup, JobID jobID, int totalNumberOfSubTasks) {
        Preconditions.checkArgument((numRememberedCheckpoints >= 0 ? 1 : 0) != 0, (Object)"Negative number of remembered checkpoints");
        this.history = new CheckpointStatsHistory(numRememberedCheckpoints);
        this.jobID = jobID;
        this.metricGroup = metricGroup;
        this.totalNumberOfSubTasks = totalNumberOfSubTasks;
        this.latestSnapshot = new CheckpointStatsSnapshot(this.counts.createSnapshot(), this.summary.createSnapshot(), this.history.createSnapshot(), null);
        this.registerMetrics(metricGroup);
    }

    public CheckpointStatsTracker updateTotalNumberOfSubtasks(int totalNumberOfSubTasks) {
        this.totalNumberOfSubTasks = totalNumberOfSubTasks;
        return this;
    }

    @VisibleForTesting
    Optional<JobInitializationMetricsBuilder> getJobInitializationMetricsBuilder() {
        return this.jobInitializationMetricsBuilder;
    }

    public CheckpointStatsSnapshot createSnapshot() {
        CheckpointStatsSnapshot snapshot = this.latestSnapshot;
        if (this.dirty && this.statsReadWriteLock.tryLock()) {
            try {
                this.latestSnapshot = snapshot = new CheckpointStatsSnapshot(this.counts.createSnapshot(), this.summary.createSnapshot(), this.history.createSnapshot(), this.jobInitializationMetricsBuilder.flatMap(JobInitializationMetricsBuilder::buildRestoredCheckpointStats).orElse(null));
                this.dirty = false;
            }
            finally {
                this.statsReadWriteLock.unlock();
            }
        }
        return snapshot;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    PendingCheckpointStats reportPendingCheckpoint(long checkpointId, long triggerTimestamp, CheckpointProperties props, Map<JobVertexID, Integer> vertexToDop) {
        PendingCheckpointStats pending = new PendingCheckpointStats(checkpointId, triggerTimestamp, props, vertexToDop);
        this.statsReadWriteLock.lock();
        try {
            this.counts.incrementInProgressCheckpoints();
            this.history.addInProgressCheckpoint(pending);
            this.dirty = true;
        }
        finally {
            this.statsReadWriteLock.unlock();
        }
        return pending;
    }

    @Deprecated
    void reportRestoredCheckpoint(RestoredCheckpointStats restored) {
        Preconditions.checkNotNull((Object)restored, (String)"Restored checkpoint");
        this.reportRestoredCheckpoint(restored.getCheckpointId(), restored.getProperties(), restored.getExternalPath(), restored.getStateSize());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void reportRestoredCheckpoint(long checkpointID, CheckpointProperties properties, String externalPath, long stateSize) {
        this.statsReadWriteLock.lock();
        try {
            this.counts.incrementRestoredCheckpoints();
            Preconditions.checkState((boolean)this.jobInitializationMetricsBuilder.isPresent(), (Object)"JobInitializationMetrics should have been set first, before RestoredCheckpointStats");
            this.jobInitializationMetricsBuilder.get().setRestoredCheckpointStats(checkpointID, stateSize, properties, externalPath);
            this.dirty = true;
        }
        finally {
            this.statsReadWriteLock.unlock();
        }
    }

    void reportCompletedCheckpoint(CompletedCheckpointStats completed) {
        this.statsReadWriteLock.lock();
        try {
            this.latestCompletedCheckpoint = completed;
            this.counts.incrementCompletedCheckpoints();
            this.history.replacePendingCheckpointById(completed);
            this.summary.updateSummary(completed);
            this.dirty = true;
            this.logCheckpointStatistics(completed);
        }
        finally {
            this.statsReadWriteLock.unlock();
        }
    }

    void reportFailedCheckpoint(FailedCheckpointStats failed) {
        this.statsReadWriteLock.lock();
        try {
            this.counts.incrementFailedCheckpoints();
            this.history.replacePendingCheckpointById(failed);
            this.dirty = true;
            this.logCheckpointStatistics(failed);
        }
        finally {
            this.statsReadWriteLock.unlock();
        }
    }

    private void logCheckpointStatistics(AbstractCheckpointStats checkpointStats) {
        try {
            this.metricGroup.addSpan(Span.builder(CheckpointStatsTracker.class, (String)"Checkpoint").setStartTsMillis(checkpointStats.getTriggerTimestamp()).setEndTsMillis(checkpointStats.getLatestAckTimestamp()).setAttribute("checkpointId", checkpointStats.getCheckpointId()).setAttribute("fullSize", checkpointStats.getStateSize()).setAttribute("checkpointedSize", checkpointStats.getCheckpointedSize()).setAttribute("checkpointStatus", checkpointStats.getStatus().name()));
            if (LOG.isDebugEnabled()) {
                StringWriter sw = new StringWriter();
                MAPPER.writeValue((Writer)sw, (Object)CheckpointStatistics.generateCheckpointStatistics(checkpointStats, true));
                String jsonDump = sw.toString();
                LOG.debug("CheckpointStatistics (for jobID={}, checkpointId={}) dump = {} ", new Object[]{this.jobID, checkpointStats.checkpointId, jsonDump});
            }
        }
        catch (Exception ex) {
            LOG.warn("Fail to log CheckpointStatistics", (Throwable)ex);
        }
    }

    public void reportFailedCheckpointsWithoutInProgress() {
        this.statsReadWriteLock.lock();
        try {
            this.counts.incrementFailedCheckpointsWithoutInProgress();
            this.dirty = true;
        }
        finally {
            this.statsReadWriteLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PendingCheckpointStats getPendingCheckpointStats(long checkpointId) {
        this.statsReadWriteLock.lock();
        try {
            AbstractCheckpointStats stats = this.history.getCheckpointById(checkpointId);
            PendingCheckpointStats pendingCheckpointStats = stats instanceof PendingCheckpointStats ? (PendingCheckpointStats)stats : null;
            return pendingCheckpointStats;
        }
        finally {
            this.statsReadWriteLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void reportIncompleteStats(long checkpointId, ExecutionAttemptID attemptId, CheckpointMetrics metrics) {
        this.statsReadWriteLock.lock();
        try {
            AbstractCheckpointStats stats = this.history.getCheckpointById(checkpointId);
            if (stats instanceof PendingCheckpointStats) {
                ((PendingCheckpointStats)stats).reportSubtaskStats(attemptId.getJobVertexId(), new SubtaskStateStats(attemptId.getSubtaskIndex(), System.currentTimeMillis(), metrics.getBytesPersistedOfThisCheckpoint(), metrics.getTotalBytesPersisted(), metrics.getSyncDurationMillis(), metrics.getAsyncDurationMillis(), metrics.getBytesProcessedDuringAlignment(), metrics.getBytesPersistedDuringAlignment(), metrics.getAlignmentDurationNanos() / 1000000L, metrics.getCheckpointStartDelayNanos() / 1000000L, metrics.getUnalignedCheckpoint(), false));
                this.dirty = true;
            }
        }
        finally {
            this.statsReadWriteLock.unlock();
        }
    }

    public void reportInitializationStartTs(long initializationStartTs) {
        this.jobInitializationMetricsBuilder = Optional.of(new JobInitializationMetricsBuilder(this.totalNumberOfSubTasks, initializationStartTs));
    }

    public void reportInitializationMetrics(SubTaskInitializationMetrics initializationMetrics) {
        this.statsReadWriteLock.lock();
        try {
            if (!this.jobInitializationMetricsBuilder.isPresent()) {
                LOG.warn("Attempted to report SubTaskInitializationMetrics [{}] without jobInitializationMetricsBuilder present", (Object)initializationMetrics);
                return;
            }
            JobInitializationMetricsBuilder builder = this.jobInitializationMetricsBuilder.get();
            builder.reportInitializationMetrics(initializationMetrics);
            if (builder.isComplete()) {
                this.traceInitializationMetrics(builder.build());
            }
        }
        catch (Exception ex) {
            LOG.warn("Failed to log SubTaskInitializationMetrics[{}]", (Object)ex, (Object)initializationMetrics);
        }
        finally {
            this.statsReadWriteLock.unlock();
        }
    }

    private void traceInitializationMetrics(JobInitializationMetrics jobInitializationMetrics) {
        SpanBuilder span = Span.builder(CheckpointStatsTracker.class, (String)"JobInitialization").setStartTsMillis(jobInitializationMetrics.getStartTs()).setEndTsMillis(jobInitializationMetrics.getEndTs()).setAttribute("initializationStatus", jobInitializationMetrics.getStatus().name());
        for (JobInitializationMetrics.SumMaxDuration duration : jobInitializationMetrics.getDurationMetrics().values()) {
            this.setDurationSpanAttribute(span, duration);
        }
        if (jobInitializationMetrics.getCheckpointId() != -1L) {
            span.setAttribute("checkpointId", jobInitializationMetrics.getCheckpointId());
        }
        if (jobInitializationMetrics.getStateSize() != -1L) {
            span.setAttribute("fullSize", jobInitializationMetrics.getStateSize());
        }
        this.metricGroup.addSpan(span);
    }

    private void setDurationSpanAttribute(SpanBuilder span, JobInitializationMetrics.SumMaxDuration duration) {
        span.setAttribute("max" + duration.getName(), duration.getMax());
        span.setAttribute("sum" + duration.getName(), duration.getSum());
    }

    private void registerMetrics(MetricGroup metricGroup) {
        metricGroup.gauge(NUMBER_OF_CHECKPOINTS_METRIC, (Gauge)new CheckpointsCounter());
        metricGroup.gauge(NUMBER_OF_IN_PROGRESS_CHECKPOINTS_METRIC, (Gauge)new InProgressCheckpointsCounter());
        metricGroup.gauge(NUMBER_OF_COMPLETED_CHECKPOINTS_METRIC, (Gauge)new CompletedCheckpointsCounter());
        metricGroup.gauge(NUMBER_OF_FAILED_CHECKPOINTS_METRIC, (Gauge)new FailedCheckpointsCounter());
        metricGroup.gauge(LATEST_RESTORED_CHECKPOINT_TIMESTAMP_METRIC, (Gauge)new LatestRestoredCheckpointTimestampGauge());
        metricGroup.gauge(LATEST_COMPLETED_CHECKPOINT_SIZE_METRIC, (Gauge)new LatestCompletedCheckpointSizeGauge());
        metricGroup.gauge(LATEST_COMPLETED_CHECKPOINT_FULL_SIZE_METRIC, (Gauge)new LatestCompletedCheckpointFullSizeGauge());
        metricGroup.gauge(LATEST_COMPLETED_CHECKPOINT_DURATION_METRIC, (Gauge)new LatestCompletedCheckpointDurationGauge());
        metricGroup.gauge(LATEST_COMPLETED_CHECKPOINT_PROCESSED_DATA_METRIC, (Gauge)new LatestCompletedCheckpointProcessedDataGauge());
        metricGroup.gauge(LATEST_COMPLETED_CHECKPOINT_PERSISTED_DATA_METRIC, (Gauge)new LatestCompletedCheckpointPersistedDataGauge());
        metricGroup.gauge(LATEST_COMPLETED_CHECKPOINT_EXTERNAL_PATH_METRIC, (Gauge)new LatestCompletedCheckpointExternalPathGauge());
        metricGroup.gauge(LATEST_COMPLETED_CHECKPOINT_ID_METRIC, (Gauge)new LatestCompletedCheckpointIdGauge());
    }

    private class LatestCompletedCheckpointIdGauge
    implements Gauge<Long> {
        private LatestCompletedCheckpointIdGauge() {
        }

        public Long getValue() {
            CompletedCheckpointStats completed = CheckpointStatsTracker.this.latestCompletedCheckpoint;
            if (completed != null) {
                return completed.getCheckpointId();
            }
            return -1L;
        }
    }

    private class LatestCompletedCheckpointExternalPathGauge
    implements Gauge<String> {
        private LatestCompletedCheckpointExternalPathGauge() {
        }

        public String getValue() {
            CompletedCheckpointStats completed = CheckpointStatsTracker.this.latestCompletedCheckpoint;
            if (completed != null && completed.getExternalPath() != null) {
                return completed.getExternalPath();
            }
            return "n/a";
        }
    }

    private class LatestCompletedCheckpointPersistedDataGauge
    implements Gauge<Long> {
        private LatestCompletedCheckpointPersistedDataGauge() {
        }

        public Long getValue() {
            CompletedCheckpointStats completed = CheckpointStatsTracker.this.latestCompletedCheckpoint;
            if (completed != null) {
                return completed.getPersistedData();
            }
            return -1L;
        }
    }

    private class LatestCompletedCheckpointProcessedDataGauge
    implements Gauge<Long> {
        private LatestCompletedCheckpointProcessedDataGauge() {
        }

        public Long getValue() {
            CompletedCheckpointStats completed = CheckpointStatsTracker.this.latestCompletedCheckpoint;
            if (completed != null) {
                return completed.getProcessedData();
            }
            return -1L;
        }
    }

    private class LatestCompletedCheckpointDurationGauge
    implements Gauge<Long> {
        private LatestCompletedCheckpointDurationGauge() {
        }

        public Long getValue() {
            CompletedCheckpointStats completed = CheckpointStatsTracker.this.latestCompletedCheckpoint;
            if (completed != null) {
                return completed.getEndToEndDuration();
            }
            return -1L;
        }
    }

    private class LatestCompletedCheckpointFullSizeGauge
    implements Gauge<Long> {
        private LatestCompletedCheckpointFullSizeGauge() {
        }

        public Long getValue() {
            CompletedCheckpointStats completed = CheckpointStatsTracker.this.latestCompletedCheckpoint;
            if (completed != null) {
                return completed.getStateSize();
            }
            return -1L;
        }
    }

    private class LatestCompletedCheckpointSizeGauge
    implements Gauge<Long> {
        private LatestCompletedCheckpointSizeGauge() {
        }

        public Long getValue() {
            CompletedCheckpointStats completed = CheckpointStatsTracker.this.latestCompletedCheckpoint;
            if (completed != null) {
                return completed.getCheckpointedSize();
            }
            return -1L;
        }
    }

    private class LatestRestoredCheckpointTimestampGauge
    implements Gauge<Long> {
        private LatestRestoredCheckpointTimestampGauge() {
        }

        public Long getValue() {
            return CheckpointStatsTracker.this.jobInitializationMetricsBuilder.map(JobInitializationMetricsBuilder::getStartTs).orElse(-1L);
        }
    }

    private class FailedCheckpointsCounter
    implements Gauge<Long> {
        private FailedCheckpointsCounter() {
        }

        public Long getValue() {
            return CheckpointStatsTracker.this.counts.getNumberOfFailedCheckpoints();
        }
    }

    private class CompletedCheckpointsCounter
    implements Gauge<Long> {
        private CompletedCheckpointsCounter() {
        }

        public Long getValue() {
            return CheckpointStatsTracker.this.counts.getNumberOfCompletedCheckpoints();
        }
    }

    private class InProgressCheckpointsCounter
    implements Gauge<Integer> {
        private InProgressCheckpointsCounter() {
        }

        public Integer getValue() {
            return CheckpointStatsTracker.this.counts.getNumberOfInProgressCheckpoints();
        }
    }

    private class CheckpointsCounter
    implements Gauge<Long> {
        private CheckpointsCounter() {
        }

        public Long getValue() {
            return CheckpointStatsTracker.this.counts.getTotalNumberOfCheckpoints();
        }
    }
}

