/*
 * Decompiled with CFR 0.152.
 */
package org.apache.amoro.server.optimizing;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.amoro.AmoroTable;
import org.apache.amoro.api.OptimizerProperties;
import org.apache.amoro.api.OptimizingTaskId;
import org.apache.amoro.api.ServerTableIdentifier;
import org.apache.amoro.api.resource.ResourceGroup;
import org.apache.amoro.optimizing.RewriteFilesInput;
import org.apache.amoro.server.exception.OptimizingClosedException;
import org.apache.amoro.server.manager.MetricManager;
import org.apache.amoro.server.optimizing.KeyedTableCommit;
import org.apache.amoro.server.optimizing.MetricsSummary;
import org.apache.amoro.server.optimizing.OptimizerGroupMetrics;
import org.apache.amoro.server.optimizing.OptimizingProcess;
import org.apache.amoro.server.optimizing.OptimizingStatus;
import org.apache.amoro.server.optimizing.OptimizingType;
import org.apache.amoro.server.optimizing.SchedulingPolicy;
import org.apache.amoro.server.optimizing.TaskRuntime;
import org.apache.amoro.server.optimizing.UnKeyedTableCommit;
import org.apache.amoro.server.optimizing.plan.OptimizingPlanner;
import org.apache.amoro.server.optimizing.plan.TaskDescriptor;
import org.apache.amoro.server.persistence.PersistentBase;
import org.apache.amoro.server.persistence.TaskFilesPersistence;
import org.apache.amoro.server.persistence.mapper.OptimizingMapper;
import org.apache.amoro.server.resource.OptimizerInstance;
import org.apache.amoro.server.resource.QuotaProvider;
import org.apache.amoro.server.table.TableManager;
import org.apache.amoro.server.table.TableRuntime;
import org.apache.amoro.server.table.TableRuntimeMeta;
import org.apache.amoro.shade.guava32.com.google.common.annotations.VisibleForTesting;
import org.apache.amoro.shade.guava32.com.google.common.base.Preconditions;
import org.apache.amoro.shade.guava32.com.google.common.collect.Lists;
import org.apache.amoro.shade.guava32.com.google.common.collect.Maps;
import org.apache.amoro.table.MixedTable;
import org.apache.amoro.utils.CompatiblePropertyUtil;
import org.apache.amoro.utils.ExceptionUtil;
import org.apache.amoro.utils.MixedDataFiles;
import org.apache.amoro.utils.TablePropertyUtil;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.data.GenericRecord;
import org.apache.iceberg.types.Types;
import org.apache.iceberg.util.StructLikeMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OptimizingQueue
extends PersistentBase {
    private static final Logger LOG = LoggerFactory.getLogger(OptimizingQueue.class);
    private final QuotaProvider quotaProvider;
    private final Queue<TableOptimizingProcess> tableQueue = new LinkedTransferQueue<TableOptimizingProcess>();
    private final Queue<TaskRuntime> retryTaskQueue = new LinkedTransferQueue<TaskRuntime>();
    private final SchedulingPolicy scheduler;
    private final TableManager tableManager;
    private final Executor planExecutor;
    private final Set<ServerTableIdentifier> planningTables = new HashSet<ServerTableIdentifier>();
    private final Lock scheduleLock = new ReentrantLock();
    private final Condition planningCompleted = this.scheduleLock.newCondition();
    private final int maxPlanningParallelism;
    private final OptimizerGroupMetrics metrics;
    private ResourceGroup optimizerGroup;

    public OptimizingQueue(TableManager tableManager, ResourceGroup optimizerGroup, QuotaProvider quotaProvider, Executor planExecutor, List<TableRuntimeMeta> tableRuntimeMetaList, int maxPlanningParallelism) {
        Preconditions.checkNotNull((Object)optimizerGroup, (Object)"Optimizer group can not be null");
        this.planExecutor = planExecutor;
        this.optimizerGroup = optimizerGroup;
        this.quotaProvider = quotaProvider;
        this.scheduler = new SchedulingPolicy(optimizerGroup);
        this.tableManager = tableManager;
        this.maxPlanningParallelism = maxPlanningParallelism;
        this.metrics = new OptimizerGroupMetrics(optimizerGroup.getName(), MetricManager.getInstance().getGlobalRegistry(), this);
        this.metrics.register();
        tableRuntimeMetaList.forEach(this::initTableRuntime);
    }

    private void initTableRuntime(TableRuntimeMeta tableRuntimeMeta) {
        TableRuntime tableRuntime = tableRuntimeMeta.getTableRuntime();
        if (tableRuntime.getOptimizingStatus().isProcessing() && tableRuntimeMeta.getOptimizingProcessId() != 0L) {
            tableRuntime.recover(new TableOptimizingProcess(tableRuntimeMeta));
        }
        if (tableRuntime.isOptimizingEnabled()) {
            tableRuntime.resetTaskQuotas(System.currentTimeMillis() - 3600000L);
            if (!tableRuntime.getOptimizingStatus().isProcessing()) {
                this.scheduler.addTable(tableRuntime);
            } else if (tableRuntime.getOptimizingStatus() != OptimizingStatus.COMMITTING) {
                this.tableQueue.offer(new TableOptimizingProcess(tableRuntimeMeta));
            }
        } else {
            OptimizingProcess process = tableRuntime.getOptimizingProcess();
            if (process != null) {
                process.close();
            }
        }
    }

    public String getContainerName() {
        return this.optimizerGroup.getContainer();
    }

    public void refreshTable(TableRuntime tableRuntime) {
        if (tableRuntime.isOptimizingEnabled() && !tableRuntime.getOptimizingStatus().isProcessing()) {
            LOG.info("Bind queue {} success with table {}", (Object)this.optimizerGroup.getName(), (Object)tableRuntime.getTableIdentifier());
            tableRuntime.resetTaskQuotas(System.currentTimeMillis() - 3600000L);
            this.scheduler.addTable(tableRuntime);
        }
    }

    public void releaseTable(TableRuntime tableRuntime) {
        this.scheduler.removeTable(tableRuntime);
        LOG.info("Release queue {} with table {}", (Object)this.optimizerGroup.getName(), (Object)tableRuntime.getTableIdentifier());
    }

    public boolean containsTable(ServerTableIdentifier identifier) {
        return this.scheduler.getTableRuntime(identifier) != null;
    }

    private void clearProcess(TableOptimizingProcess optimizingProcess) {
        this.tableQueue.removeIf(process -> process.getProcessId() == optimizingProcess.getProcessId());
        this.retryTaskQueue.removeIf(taskRuntime -> taskRuntime.getTaskId().getProcessId() == optimizingProcess.getProcessId());
    }

    public TaskRuntime pollTask(long maxWaitTime) {
        long deadline = this.calculateDeadline(maxWaitTime);
        TaskRuntime task = this.fetchTask();
        while (task == null && this.waitTask(deadline)) {
            task = this.fetchTask();
        }
        return task;
    }

    private long calculateDeadline(long maxWaitTime) {
        long deadline = System.currentTimeMillis() + maxWaitTime;
        return deadline <= 0L ? Long.MAX_VALUE : deadline;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean waitTask(long waitDeadline) {
        this.scheduleLock.lock();
        try {
            long currentTime = System.currentTimeMillis();
            this.scheduleTableIfNecessary(currentTime);
            boolean bl = waitDeadline > currentTime && this.planningCompleted.await(waitDeadline - currentTime, TimeUnit.MILLISECONDS);
            return bl;
        }
        catch (InterruptedException e) {
            LOG.error("Schedule table interrupted", (Throwable)e);
            boolean bl = false;
            return bl;
        }
        finally {
            this.scheduleLock.unlock();
        }
    }

    private TaskRuntime fetchTask() {
        return Optional.ofNullable(this.retryTaskQueue.poll()).orElseGet(this::fetchScheduledTask);
    }

    private TaskRuntime fetchScheduledTask() {
        return this.tableQueue.stream().map(TableOptimizingProcess::poll).filter(Objects::nonNull).findFirst().orElse(null);
    }

    private void scheduleTableIfNecessary(long startTime) {
        if (this.planningTables.size() < this.maxPlanningParallelism) {
            HashSet<ServerTableIdentifier> skipTables = new HashSet<ServerTableIdentifier>(this.planningTables);
            Optional.ofNullable(this.scheduler.scheduleTable(skipTables)).ifPresent(tableRuntime -> this.triggerAsyncPlanning((TableRuntime)tableRuntime, (Set<ServerTableIdentifier>)skipTables, startTime));
        }
    }

    private void triggerAsyncPlanning(TableRuntime tableRuntime, Set<ServerTableIdentifier> skipTables, long startTime) {
        LOG.info("Trigger planning table {} by policy {}", (Object)tableRuntime.getTableIdentifier(), (Object)this.scheduler.name());
        this.planningTables.add(tableRuntime.getTableIdentifier());
        CompletableFuture.supplyAsync(() -> this.planInternal(tableRuntime), this.planExecutor).whenComplete((process, throwable) -> {
            long currentTime = System.currentTimeMillis();
            this.scheduleLock.lock();
            try {
                tableRuntime.setLastPlanTime(currentTime);
                this.planningTables.remove(tableRuntime.getTableIdentifier());
                if (process != null) {
                    this.tableQueue.offer((TableOptimizingProcess)process);
                    LOG.info("Completed planning on table {} with {} tasks with a total cost of {} ms, skipping tables {}", new Object[]{tableRuntime.getTableIdentifier(), ((TableOptimizingProcess)process).getTaskMap().size(), currentTime - startTime, skipTables});
                } else if (throwable == null) {
                    LOG.info("Skip planning table {} with a total cost of {} ms.", (Object)tableRuntime.getTableIdentifier(), (Object)(currentTime - startTime));
                }
                this.planningCompleted.signalAll();
            }
            finally {
                this.scheduleLock.unlock();
            }
        });
    }

    private TableOptimizingProcess planInternal(TableRuntime tableRuntime) {
        tableRuntime.beginPlanning();
        try {
            AmoroTable<?> table = this.tableManager.loadTable(tableRuntime.getTableIdentifier());
            OptimizingPlanner planner = new OptimizingPlanner(tableRuntime.refresh(table), (MixedTable)table.originalTable(), this.getAvailableCore(), this.maxInputSizePerThread());
            if (planner.isNecessary()) {
                return new TableOptimizingProcess(planner);
            }
            tableRuntime.completeEmptyProcess();
            return null;
        }
        catch (Throwable throwable) {
            tableRuntime.planFailed();
            LOG.error("Planning table {} failed", (Object)tableRuntime.getTableIdentifier(), (Object)throwable);
            throw throwable;
        }
    }

    public TaskRuntime getTask(OptimizingTaskId taskId) {
        return this.tableQueue.stream().filter(p -> p.getProcessId() == taskId.getProcessId()).findFirst().map(p -> (TaskRuntime)((TableOptimizingProcess)p).getTaskMap().get(taskId)).orElse(null);
    }

    public List<TaskRuntime> collectTasks() {
        return this.tableQueue.stream().flatMap(p -> ((TableOptimizingProcess)p).getTaskMap().values().stream()).collect(Collectors.toList());
    }

    public List<TaskRuntime> collectTasks(Predicate<TaskRuntime> predicate) {
        return this.tableQueue.stream().flatMap(p -> ((TableOptimizingProcess)p).getTaskMap().values().stream()).filter(predicate).collect(Collectors.toList());
    }

    public void retryTask(TaskRuntime taskRuntime) {
        taskRuntime.reset();
        this.retryTaskQueue.offer(taskRuntime);
    }

    public void updateOptimizerGroup(ResourceGroup optimizerGroup) {
        Preconditions.checkArgument((boolean)this.optimizerGroup.getName().equals(optimizerGroup.getName()), (Object)"optimizer group name mismatch");
        this.optimizerGroup = optimizerGroup;
        this.scheduler.setTableSorterIfNeeded(optimizerGroup);
    }

    public void addOptimizer(OptimizerInstance optimizerInstance) {
        this.metrics.addOptimizer(optimizerInstance);
    }

    public void removeOptimizer(OptimizerInstance optimizerInstance) {
        this.metrics.removeOptimizer(optimizerInstance);
    }

    public void dispose() {
        this.metrics.unregister();
    }

    private double getAvailableCore() {
        return Math.max(this.quotaProvider.getTotalQuota(this.optimizerGroup.getName()), 1);
    }

    private long maxInputSizePerThread() {
        return CompatiblePropertyUtil.propertyAsLong((Map)this.optimizerGroup.getProperties(), (String)"max-input-file-size-per-thread", (long)OptimizerProperties.MAX_INPUT_FILE_SIZE_PER_THREAD_DEFAULT);
    }

    @VisibleForTesting
    SchedulingPolicy getSchedulingPolicy() {
        return this.scheduler;
    }

    private class TableOptimizingProcess
    implements OptimizingProcess,
    TaskRuntime.TaskOwner {
        private final long processId;
        private final OptimizingType optimizingType;
        private final TableRuntime tableRuntime;
        private final long planTime;
        private final long targetSnapshotId;
        private final long targetChangeSnapshotId;
        private final Map<OptimizingTaskId, TaskRuntime> taskMap = Maps.newHashMap();
        private final Queue<TaskRuntime> taskQueue = new LinkedList<TaskRuntime>();
        private final Lock lock = new ReentrantLock();
        private volatile OptimizingProcess.Status status = OptimizingProcess.Status.RUNNING;
        private volatile String failedReason;
        private long endTime = 0L;
        private Map<String, Long> fromSequence = Maps.newHashMap();
        private Map<String, Long> toSequence = Maps.newHashMap();
        private boolean hasCommitted = false;

        public TaskRuntime poll() {
            this.lock.lock();
            try {
                TaskRuntime taskRuntime = this.taskQueue.poll();
                return taskRuntime;
            }
            finally {
                this.lock.unlock();
            }
        }

        public TableOptimizingProcess(OptimizingPlanner planner) {
            this.processId = planner.getProcessId();
            this.tableRuntime = planner.getTableRuntime();
            this.optimizingType = planner.getOptimizingType();
            this.planTime = planner.getPlanTime();
            this.targetSnapshotId = planner.getTargetSnapshotId();
            this.targetChangeSnapshotId = planner.getTargetChangeSnapshotId();
            this.loadTaskRuntimes(planner.planTasks());
            this.fromSequence = planner.getFromSequence();
            this.toSequence = planner.getToSequence();
            this.beginAndPersistProcess();
        }

        public TableOptimizingProcess(TableRuntimeMeta tableRuntimeMeta) {
            this.processId = tableRuntimeMeta.getOptimizingProcessId();
            this.tableRuntime = tableRuntimeMeta.getTableRuntime();
            this.optimizingType = tableRuntimeMeta.getOptimizingType();
            this.targetSnapshotId = tableRuntimeMeta.getTargetSnapshotId();
            this.targetChangeSnapshotId = tableRuntimeMeta.getTargetChangeSnapshotId();
            this.planTime = tableRuntimeMeta.getPlanTime();
            if (tableRuntimeMeta.getFromSequence() != null) {
                this.fromSequence = tableRuntimeMeta.getFromSequence();
            }
            if (tableRuntimeMeta.getToSequence() != null) {
                this.toSequence = tableRuntimeMeta.getToSequence();
            }
            if (this.status != OptimizingProcess.Status.CLOSED) {
                tableRuntimeMeta.getTableRuntime().recover(this);
            }
            this.loadTaskRuntimes(this);
        }

        @Override
        public long getProcessId() {
            return this.processId;
        }

        @Override
        public OptimizingType getOptimizingType() {
            return this.optimizingType;
        }

        @Override
        public OptimizingProcess.Status getStatus() {
            return this.status;
        }

        @Override
        public void close() {
            this.lock.lock();
            try {
                if (this.status != OptimizingProcess.Status.RUNNING) {
                    return;
                }
                this.status = OptimizingProcess.Status.CLOSED;
                this.endTime = System.currentTimeMillis();
                this.persistProcessCompleted(false);
                OptimizingQueue.this.clearProcess(this);
            }
            finally {
                this.lock.unlock();
            }
            this.releaseResourcesIfNecessary();
        }

        @Override
        public void acceptResult(TaskRuntime taskRuntime) {
            this.lock.lock();
            try {
                try {
                    this.tableRuntime.addTaskQuota(taskRuntime.getCurrentQuota());
                }
                catch (Throwable throwable) {
                    LOG.warn("{} failed to add task quota {}, ignore it", new Object[]{this.tableRuntime.getTableIdentifier(), taskRuntime.getTaskId(), throwable});
                }
                if (this.isClosed()) {
                    throw new OptimizingClosedException(this.processId);
                }
                if (taskRuntime.getStatus() == TaskRuntime.Status.SUCCESS) {
                    if (this.allTasksPrepared() && this.tableRuntime.getOptimizingStatus().isProcessing() && this.tableRuntime.getOptimizingStatus() != OptimizingStatus.COMMITTING) {
                        this.tableRuntime.beginCommitting();
                        OptimizingQueue.this.clearProcess(this);
                    }
                } else if (taskRuntime.getStatus() == TaskRuntime.Status.FAILED) {
                    if (taskRuntime.getRetry() < this.tableRuntime.getMaxExecuteRetryCount()) {
                        LOG.info("Put task {} to retry queue, because {}", (Object)taskRuntime.getTaskId(), (Object)taskRuntime.getFailReason());
                        OptimizingQueue.this.retryTask(taskRuntime);
                    } else {
                        OptimizingQueue.this.clearProcess(this);
                        this.failedReason = taskRuntime.getFailReason();
                        this.status = OptimizingProcess.Status.FAILED;
                        this.endTime = taskRuntime.getEndTime();
                        this.persistProcessCompleted(false);
                    }
                }
            }
            catch (Exception e) {
                LOG.error("accept result error:", (Throwable)e);
            }
            finally {
                this.lock.unlock();
            }
        }

        @Override
        public void releaseResourcesIfNecessary() {
            if (this.status == OptimizingProcess.Status.FAILED || this.status == OptimizingProcess.Status.CLOSED) {
                this.cancelTasks();
            }
        }

        @Override
        public boolean isClosed() {
            return this.status == OptimizingProcess.Status.CLOSED;
        }

        @Override
        public long getPlanTime() {
            return this.planTime;
        }

        @Override
        public long getDuration() {
            long dur = this.endTime == 0L ? System.currentTimeMillis() - this.planTime : this.endTime - this.planTime;
            return Math.max(0L, dur);
        }

        @Override
        public long getTargetSnapshotId() {
            return this.targetSnapshotId;
        }

        @Override
        public long getTargetChangeSnapshotId() {
            return this.targetChangeSnapshotId;
        }

        public String getFailedReason() {
            return this.failedReason;
        }

        private Map<OptimizingTaskId, TaskRuntime> getTaskMap() {
            return this.taskMap;
        }

        private boolean allTasksPrepared() {
            if (!this.taskMap.isEmpty()) {
                return this.taskMap.values().stream().allMatch(t -> t.getStatus() == TaskRuntime.Status.SUCCESS);
            }
            return false;
        }

        @Override
        public long getRunningQuotaTime(long calculatingStartTime, long calculatingEndTime) {
            return this.taskMap.values().stream().filter(t -> !t.finished()).mapToLong(task -> task.getQuotaTime(calculatingStartTime, calculatingEndTime)).sum();
        }

        @Override
        public void commit() {
            LOG.debug("{} get {} tasks of {} partitions to commit", new Object[]{this.tableRuntime.getTableIdentifier(), this.taskMap.size(), this.taskMap.values()});
            this.lock.lock();
            try {
                if (this.hasCommitted) {
                    LOG.warn("{} has already committed, give up", (Object)this.tableRuntime.getTableIdentifier());
                    throw new IllegalStateException("repeat commit, and last error " + this.failedReason);
                }
                try {
                    this.hasCommitted = true;
                    this.buildCommit().commit();
                    this.status = OptimizingProcess.Status.SUCCESS;
                    this.endTime = System.currentTimeMillis();
                    this.persistProcessCompleted(true);
                }
                catch (Exception e) {
                    LOG.error("{} Commit optimizing failed ", (Object)this.tableRuntime.getTableIdentifier(), (Object)e);
                    this.status = OptimizingProcess.Status.FAILED;
                    this.failedReason = ExceptionUtil.getErrorMessage((Throwable)e, (int)4000);
                    this.endTime = System.currentTimeMillis();
                    this.persistProcessCompleted(false);
                }
                finally {
                    OptimizingQueue.this.clearProcess(this);
                }
            }
            finally {
                this.lock.unlock();
            }
        }

        @Override
        public MetricsSummary getSummary() {
            return new MetricsSummary(this.taskMap.values());
        }

        private UnKeyedTableCommit buildCommit() {
            MixedTable table = (MixedTable)OptimizingQueue.this.tableManager.loadTable(this.tableRuntime.getTableIdentifier()).originalTable();
            if (table.isUnkeyedTable()) {
                return new UnKeyedTableCommit(this.targetSnapshotId, table, this.taskMap.values());
            }
            return new KeyedTableCommit(table, this.taskMap.values(), this.targetSnapshotId, this.convertPartitionSequence(table, this.fromSequence), this.convertPartitionSequence(table, this.toSequence));
        }

        private StructLikeMap<Long> convertPartitionSequence(MixedTable table, Map<String, Long> partitionSequence) {
            PartitionSpec spec = table.spec();
            StructLikeMap results = StructLikeMap.create((Types.StructType)spec.partitionType());
            partitionSequence.forEach((partition, sequence) -> {
                if (spec.isUnpartitioned()) {
                    results.put(TablePropertyUtil.EMPTY_STRUCT, sequence);
                } else {
                    GenericRecord partitionData = MixedDataFiles.data((PartitionSpec)spec, (String)partition);
                    results.put((StructLike)partitionData, sequence);
                }
            });
            return results;
        }

        private void beginAndPersistProcess() {
            OptimizingQueue.this.doAsTransaction(new Runnable[]{() -> OptimizingQueue.this.doAs(OptimizingMapper.class, mapper -> mapper.insertOptimizingProcess(this.tableRuntime.getTableIdentifier(), this.processId, this.targetSnapshotId, this.targetChangeSnapshotId, this.status, this.optimizingType, this.planTime, this.getSummary(), this.fromSequence, this.toSequence)), () -> OptimizingQueue.this.doAs(OptimizingMapper.class, mapper -> mapper.insertTaskRuntimes(Lists.newArrayList(this.taskMap.values()))), () -> TaskFilesPersistence.persistTaskInputs(this.processId, this.taskMap.values()), () -> this.tableRuntime.beginProcess(this)});
        }

        private void persistProcessCompleted(boolean success) {
            OptimizingQueue.this.doAsTransaction(new Runnable[]{() -> OptimizingQueue.this.doAs(OptimizingMapper.class, mapper -> mapper.updateOptimizingProcess(this.tableRuntime.getTableIdentifier().getId(), this.processId, this.status, this.endTime, this.getSummary(), this.getFailedReason())), () -> this.tableRuntime.completeProcess(success)});
        }

        private void cancelTasks() {
            this.taskMap.values().forEach(TaskRuntime::tryCanceling);
        }

        private void loadTaskRuntimes(OptimizingProcess optimizingProcess) {
            List taskRuntimes = (List)OptimizingQueue.this.getAs(OptimizingMapper.class, mapper -> mapper.selectTaskRuntimes(this.tableRuntime.getTableIdentifier().getId(), this.processId));
            try {
                Map<Integer, RewriteFilesInput> inputs = TaskFilesPersistence.loadTaskInputs(this.processId);
                taskRuntimes.forEach(taskRuntime -> {
                    taskRuntime.claimOwnership(this);
                    taskRuntime.setInput((RewriteFilesInput)inputs.get(taskRuntime.getTaskId().getTaskId()));
                    this.taskMap.put(taskRuntime.getTaskId(), (TaskRuntime)taskRuntime);
                    if (taskRuntime.getStatus() == TaskRuntime.Status.PLANNED) {
                        this.taskQueue.offer((TaskRuntime)taskRuntime);
                    } else if (taskRuntime.getStatus() == TaskRuntime.Status.FAILED) {
                        OptimizingQueue.this.retryTask((TaskRuntime)taskRuntime);
                    }
                });
            }
            catch (IllegalArgumentException e) {
                LOG.warn("Load task inputs failed, close the optimizing process : {}", (Object)optimizingProcess.getProcessId(), (Object)e);
                optimizingProcess.close();
            }
        }

        private void loadTaskRuntimes(List<TaskDescriptor> taskDescriptors) {
            int taskId = 1;
            for (TaskDescriptor taskDescriptor : taskDescriptors) {
                TaskRuntime taskRuntime = new TaskRuntime(new OptimizingTaskId(this.processId, taskId++), taskDescriptor, taskDescriptor.properties());
                LOG.info("{} plan new task {}, summary {}", new Object[]{this.tableRuntime.getTableIdentifier(), taskRuntime.getTaskId(), taskRuntime.getSummary()});
                this.taskMap.put(taskRuntime.getTaskId(), taskRuntime.claimOwnership(this));
                this.taskQueue.offer(taskRuntime);
            }
        }
    }
}

