/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.x509.certificate.authority;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.KeyPair;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.cert.CertPath;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Date;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Future;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.metadata.SCMMetadataStore;
import org.apache.hadoop.hdds.security.SecurityConfig;
import org.apache.hadoop.hdds.security.exception.SCMSecurityException;
import org.apache.hadoop.hdds.security.x509.certificate.authority.CAType;
import org.apache.hadoop.hdds.security.x509.certificate.authority.CertificateApprover;
import org.apache.hadoop.hdds.security.x509.certificate.authority.CertificateServer;
import org.apache.hadoop.hdds.security.x509.certificate.authority.CertificateStore;
import org.apache.hadoop.hdds.security.x509.certificate.authority.DefaultApprover;
import org.apache.hadoop.hdds.security.x509.certificate.authority.profile.PKIProfile;
import org.apache.hadoop.hdds.security.x509.certificate.utils.CertificateCodec;
import org.apache.hadoop.hdds.security.x509.certificate.utils.SelfSignedCertificate;
import org.apache.hadoop.hdds.security.x509.keys.HDDSKeyGenerator;
import org.apache.hadoop.hdds.security.x509.keys.KeyStorage;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultCAServer
implements CertificateServer {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultCAServer.class);
    private final String subject;
    private final String clusterID;
    private final String scmID;
    private String componentName;
    private Path caRootX509Path;
    private SecurityConfig config;
    private PKIProfile profile;
    private CertificateApprover approver;
    private CertificateStore store;
    private Lock lock;
    private BigInteger rootCertificateId;

    public DefaultCAServer(String subject, String clusterID, String scmID, CertificateStore certificateStore, BigInteger rootCertId, PKIProfile pkiProfile, String componentName) {
        this.subject = subject;
        this.clusterID = clusterID;
        this.scmID = scmID;
        this.store = certificateStore;
        this.rootCertificateId = rootCertId;
        this.profile = pkiProfile;
        this.componentName = componentName;
        this.lock = new ReentrantLock();
    }

    public DefaultCAServer(String subject, String clusterID, String scmID, CertificateStore certificateStore, PKIProfile pkiProfile, String componentName) {
        this(subject, clusterID, scmID, certificateStore, BigInteger.ONE, pkiProfile, componentName);
    }

    @Override
    public void init(SecurityConfig securityConfig, CAType type) throws IOException {
        this.caRootX509Path = securityConfig.getCertificateLocation(this.componentName);
        this.config = securityConfig;
        this.approver = new DefaultApprover(this.profile, this.config);
        VerificationStatus status = this.verifySelfSignedCA(securityConfig);
        Consumer<SecurityConfig> caInitializer = this.processVerificationStatus(status, type);
        caInitializer.accept(securityConfig);
    }

    @Override
    public X509Certificate getCACertificate() throws IOException {
        CertificateCodec certificateCodec = new CertificateCodec(this.config, this.componentName);
        try {
            return certificateCodec.getTargetCert();
        }
        catch (CertificateException e) {
            throw new IOException(e);
        }
    }

    @Override
    public CertPath getCaCertPath() throws CertificateException, IOException {
        CertificateCodec codec = new CertificateCodec(this.config, this.componentName);
        return codec.getCertPath();
    }

    @Override
    public X509Certificate getCertificate(String certSerialId) throws IOException {
        return this.store.getCertificateByID(new BigInteger(certSerialId));
    }

    private PrivateKey getPrivateKey() throws IOException {
        KeyStorage keyStorage = new KeyStorage(this.config, this.componentName);
        return keyStorage.readPrivateKey();
    }

    @Override
    public Future<CertPath> requestCertificate(PKCS10CertificationRequest csr, CertificateApprover.ApprovalType approverType, HddsProtos.NodeType role, String certSerialId) {
        LocalDateTime beginDate = LocalDateTime.now();
        LocalDateTime endDate = this.expiryFor(beginDate, role);
        CompletableFuture<Void> csrInspection = this.approver.inspectCSR(csr);
        CompletableFuture<CertPath> certPathPromise = new CompletableFuture<CertPath>();
        if (csrInspection.isCompletedExceptionally()) {
            try {
                csrInspection.get();
            }
            catch (Exception e) {
                certPathPromise.completeExceptionally((Throwable)new SCMSecurityException("Failed to verify the CSR.", (Throwable)e));
            }
        }
        try {
            switch (approverType) {
                case MANUAL: {
                    certPathPromise.completeExceptionally((Throwable)new SCMSecurityException("Manual approval is not yet implemented."));
                    break;
                }
                case KERBEROS_TRUSTED: 
                case TESTING_AUTOMATIC: {
                    X509Certificate signedCertificate = this.signAndStoreCertificate(beginDate, endDate, csr, role, certSerialId);
                    CertificateCodec codec = new CertificateCodec(this.config, this.componentName);
                    CertPath certPath = codec.getCertPath();
                    CertPath updatedCertPath = codec.prependCertToCertPath(signedCertificate, certPath);
                    certPathPromise.complete(updatedCertPath);
                    break;
                }
                default: {
                    return null;
                }
            }
        }
        catch (IOException | CertificateException | OperatorCreationException e) {
            LOG.error("Unable to issue a certificate.", e);
            certPathPromise.completeExceptionally((Throwable)new SCMSecurityException((Exception)e, SCMSecurityException.ErrorCode.UNABLE_TO_ISSUE_CERTIFICATE));
        }
        return certPathPromise;
    }

    private LocalDateTime expiryFor(LocalDateTime beginDate, HddsProtos.NodeType role) {
        if (role == HddsProtos.NodeType.SCM) {
            return beginDate.plus(this.config.getMaxCertificateDuration());
        }
        return beginDate.plus(this.config.getDefaultCertDuration());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private X509Certificate signAndStoreCertificate(LocalDateTime beginDate, LocalDateTime endDate, PKCS10CertificationRequest csr, HddsProtos.NodeType role, String certSerialId) throws IOException, OperatorCreationException, CertificateException {
        X509Certificate xcert;
        this.lock.lock();
        try {
            Preconditions.checkState((!Strings.isNullOrEmpty((String)certSerialId) ? 1 : 0) != 0);
            xcert = this.approver.sign(this.config, this.getPrivateKey(), this.getCACertificate(), Date.from(beginDate.atZone(ZoneId.systemDefault()).toInstant()), Date.from(endDate.atZone(ZoneId.systemDefault()).toInstant()), csr, this.scmID, this.clusterID, certSerialId);
            if (this.store != null) {
                this.store.checkValidCertID(xcert.getSerialNumber());
                this.store.storeValidCertificate(xcert.getSerialNumber(), xcert, role);
            }
        }
        finally {
            this.lock.unlock();
        }
        return xcert;
    }

    @Override
    public List<X509Certificate> listCertificate(HddsProtos.NodeType role, long startSerialId, int count) throws IOException {
        return this.store.listCertificate(role, BigInteger.valueOf(startSerialId), count);
    }

    @Override
    public void reinitialize(SCMMetadataStore scmMetadataStore) {
        this.store.reinitialize(scmMetadataStore);
    }

    private void generateSelfSignedCA(SecurityConfig securityConfig) throws NoSuchAlgorithmException, NoSuchProviderException, IOException {
        KeyPair keyPair = this.generateKeys(securityConfig);
        this.generateRootCertificate(securityConfig, keyPair);
    }

    private VerificationStatus verifySelfSignedCA(SecurityConfig securityConfig) {
        boolean keyStatus = this.checkIfKeysExist();
        boolean certStatus = this.checkIfCertificatesExist();
        if (certStatus == keyStatus && certStatus) {
            return VerificationStatus.SUCCESS;
        }
        if (certStatus == keyStatus) {
            return VerificationStatus.INITIALIZE;
        }
        if (certStatus) {
            return VerificationStatus.MISSING_KEYS;
        }
        return VerificationStatus.MISSING_CERTIFICATE;
    }

    private boolean checkIfKeysExist() {
        KeyStorage storage = null;
        try {
            storage = new KeyStorage(this.config, this.componentName);
            storage.readKeyPair();
        }
        catch (IOException e) {
            if (storage != null && this.config.useExternalCACertificate(this.componentName)) {
                try {
                    storage.readPrivateKey();
                    return true;
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            return false;
        }
        return true;
    }

    private boolean checkIfCertificatesExist() {
        if (!Files.exists(this.caRootX509Path, new LinkOption[0])) {
            return false;
        }
        return Files.exists(Paths.get(this.caRootX509Path.toString(), this.config.getCertificateFileName()), new LinkOption[0]);
    }

    @VisibleForTesting
    Consumer<SecurityConfig> processVerificationStatus(VerificationStatus status, CAType type) {
        Consumer<SecurityConfig> consumer = null;
        switch (status) {
            case SUCCESS: {
                consumer = arg -> LOG.info("CertificateServer validation is successful");
                break;
            }
            case MISSING_KEYS: {
                consumer = arg -> {
                    LOG.error("We have found the Certificate for this CertificateServer, but keys used by this CertificateServer is missing. This is a non-recoverable error. Please restart the system after locating the Keys used by the CertificateServer.");
                    LOG.error("Exiting due to unrecoverable CertificateServer error.");
                    throw new IllegalStateException("Missing Keys, cannot continue.");
                };
                break;
            }
            case MISSING_CERTIFICATE: {
                if (this.config.useExternalCACertificate(this.componentName) && type == CAType.ROOT) {
                    consumer = this::initRootCa;
                    break;
                }
                consumer = arg -> {
                    LOG.error("We found the keys, but the root certificate for this CertificateServer is missing. Please restart SCM after locating the Certificates.");
                    LOG.error("Exiting due to unrecoverable CertificateServer error.");
                    throw new IllegalStateException("Missing Root Certs, cannot continue.");
                };
                break;
            }
            case INITIALIZE: {
                if (type == CAType.ROOT) {
                    consumer = this::initRootCa;
                    break;
                }
                if (type != CAType.SUBORDINATE) break;
                consumer = arg -> {
                    LOG.error("Sub SCM CA Server is missing keys/certs. SCM is started with out init/bootstrap");
                    throw new IllegalStateException("INTERMEDIARY_CA Should not be in Initialize State during startup.");
                };
                break;
            }
        }
        return consumer;
    }

    private void initRootCa(SecurityConfig securityConfig) {
        if (securityConfig.useExternalCACertificate(this.componentName)) {
            this.initWithExternalRootCa(securityConfig);
        } else {
            try {
                this.generateSelfSignedCA(securityConfig);
            }
            catch (IOException | NoSuchAlgorithmException | NoSuchProviderException e) {
                LOG.error("Unable to initialize CertificateServer.", (Throwable)e);
            }
        }
        VerificationStatus newStatus = this.verifySelfSignedCA(securityConfig);
        if (newStatus != VerificationStatus.SUCCESS) {
            LOG.error("Unable to initialize CertificateServer, failed in verification.");
        }
    }

    private KeyPair generateKeys(SecurityConfig securityConfig) throws NoSuchProviderException, NoSuchAlgorithmException, IOException {
        HDDSKeyGenerator keyGenerator = new HDDSKeyGenerator(securityConfig);
        KeyPair keys = keyGenerator.generateKey();
        KeyStorage keyStorage = new KeyStorage(securityConfig, this.componentName);
        keyStorage.storeKeyPair(keys);
        return keys;
    }

    private void generateRootCertificate(SecurityConfig securityConfig, KeyPair key) throws IOException, SCMSecurityException {
        Preconditions.checkNotNull((Object)this.config);
        LocalDateTime beginDate = LocalDateTime.now();
        LocalDateTime endDate = beginDate.plus(securityConfig.getMaxCertificateDuration());
        SelfSignedCertificate.Builder builder = SelfSignedCertificate.newBuilder().setSubject(this.subject).setScmID(this.scmID).setClusterID(this.clusterID).setBeginDate(beginDate).setEndDate(endDate).makeCA(this.rootCertificateId).setConfiguration(securityConfig).setKey(key);
        builder.addInetAddresses();
        X509Certificate selfSignedCertificate = builder.build();
        CertificateCodec certCodec = new CertificateCodec(this.config, this.componentName);
        certCodec.writeCertificate(selfSignedCertificate);
    }

    private void initWithExternalRootCa(SecurityConfig conf) {
        Path extCertPath = Paths.get(conf.getExternalRootCaCert(), new String[0]);
        try {
            CertificateCodec certificateCodec = new CertificateCodec(this.config, this.componentName);
            Path extCertParent = extCertPath.getParent();
            Path extCertName = extCertPath.getFileName();
            if (extCertParent == null || extCertName == null) {
                throw new IOException("External cert path is not correct: " + extCertPath);
            }
            X509Certificate certificate = certificateCodec.getTargetCert(extCertParent, extCertName.toString());
            certificateCodec.writeCertificate(certificate);
        }
        catch (IOException | CertificateException e) {
            LOG.error("External root CA certificate initialization failed", (Throwable)e);
        }
    }

    @VisibleForTesting
    static enum VerificationStatus {
        SUCCESS,
        MISSING_KEYS,
        MISSING_CERTIFICATE,
        INITIALIZE;

    }
}

