/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.examples.filestore;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.ratis.examples.filestore.FileStore;
import org.apache.ratis.examples.filestore.FileStoreCommon;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.util.CollectionUtils;
import org.apache.ratis.util.JavaUtils;
import org.apache.ratis.util.LogUtils;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.TaskQueue;
import org.apache.ratis.util.function.CheckedFunction;
import org.apache.ratis.util.function.CheckedSupplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

abstract class FileInfo {
    public static final Logger LOG = LoggerFactory.getLogger(FileInfo.class);
    private final Path relativePath;

    FileInfo(Path relativePath) {
        this.relativePath = relativePath;
    }

    Path getRelativePath() {
        return this.relativePath;
    }

    long getWriteSize() {
        throw new UnsupportedOperationException("File " + this.getRelativePath() + " size is unknown.");
    }

    long getCommittedSize() {
        throw new UnsupportedOperationException("File " + this.getRelativePath() + " size is unknown.");
    }

    ByteString read(CheckedFunction<Path, Path, IOException> resolver, long offset, long length, boolean readCommitted) throws IOException {
        if (readCommitted && offset + length > this.getCommittedSize()) {
            throw new IOException("Failed to read Committed: offset (=" + offset + " + length (=" + length + ") > size = " + this.getCommittedSize() + ", path=" + this.getRelativePath());
        }
        if (offset + length > this.getWriteSize()) {
            throw new IOException("Failed to read Wrote: offset (=" + offset + " + length (=" + length + ") > size = " + this.getWriteSize() + ", path=" + this.getRelativePath());
        }
        try (SeekableByteChannel in = Files.newByteChannel(resolver.apply(this.getRelativePath()), StandardOpenOption.READ);){
            ByteBuffer buffer = ByteBuffer.allocateDirect(FileStoreCommon.getChunkSize(length));
            in.position(offset).read(buffer);
            buffer.flip();
            ByteString byteString = ByteString.copyFrom(buffer);
            return byteString;
        }
    }

    UnderConstruction asUnderConstruction() {
        throw new UnsupportedOperationException("File " + this.getRelativePath() + " is not under construction.");
    }

    static class UnderConstruction
    extends FileInfo {
        private FileStore.FileStoreDataChannel out;
        private volatile long writeSize;
        private volatile long committedSize;
        private final TaskQueue writeQueue = new TaskQueue("writeQueue");
        private final Map<Long, WriteInfo> writeInfos = new ConcurrentHashMap<Long, WriteInfo>();
        private final AtomicLong lastWriteIndex = new AtomicLong(-1L);

        UnderConstruction(Path relativePath) {
            super(relativePath);
        }

        @Override
        UnderConstruction asUnderConstruction() {
            return this;
        }

        @Override
        long getCommittedSize() {
            return this.committedSize;
        }

        @Override
        long getWriteSize() {
            return this.writeSize;
        }

        CompletableFuture<Integer> submitCreate(CheckedFunction<Path, Path, IOException> resolver, ByteString data, boolean close, boolean sync, ExecutorService executor, RaftPeerId id, long index) {
            Supplier<String> name = () -> "create(" + this.getRelativePath() + ", " + close + ") @" + id + ":" + index;
            CheckedSupplier<Integer, IOException> task = LogUtils.newCheckedSupplier(LOG, () -> {
                if (this.out == null) {
                    this.out = new FileStore.FileStoreDataChannel((Path)resolver.apply(this.getRelativePath()));
                }
                return this.write(0L, data, close, sync);
            }, name);
            return this.submitWrite(task, executor, id, index);
        }

        CompletableFuture<Integer> submitWrite(long offset, ByteString data, boolean close, boolean sync, ExecutorService executor, RaftPeerId id, long index) {
            Supplier<String> name = () -> "write(" + this.getRelativePath() + ", " + offset + ", " + close + ") @" + id + ":" + index;
            CheckedSupplier<Integer, IOException> task = LogUtils.newCheckedSupplier(LOG, () -> this.write(offset, data, close, sync), name);
            return this.submitWrite(task, executor, id, index);
        }

        private CompletableFuture<Integer> submitWrite(CheckedSupplier<Integer, IOException> task, ExecutorService executor, RaftPeerId id, long index) {
            CompletableFuture<Integer> f = this.writeQueue.submit(task, executor, e -> new IOException("Failed " + task, (Throwable)e));
            WriteInfo info = new WriteInfo(f, this.lastWriteIndex.getAndSet(index));
            CollectionUtils.putNew(Long.valueOf(index), info, this.writeInfos, () -> id + ":writeInfos");
            return f;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private int write(long offset, ByteString data, boolean close, boolean sync) throws IOException {
            if (offset < this.writeSize) {
                return data.size();
            }
            if (offset != this.writeSize) {
                throw new IOException("Offset/size mismatched: offset = " + offset + " != writeSize = " + this.writeSize + ", path=" + this.getRelativePath());
            }
            if (this.out == null) {
                throw new IOException("File output is not initialized, path=" + this.getRelativePath());
            }
            FileStore.FileStoreDataChannel fileStoreDataChannel = this.out;
            synchronized (fileStoreDataChannel) {
                int n = 0;
                if (data != null) {
                    ByteBuffer buffer = data.asReadOnlyByteBuffer();
                    try {
                        while (buffer.remaining() > 0) {
                            n += this.out.write(buffer);
                        }
                    }
                    finally {
                        this.writeSize += (long)n;
                    }
                }
                if (sync) {
                    this.out.force(false);
                }
                if (close) {
                    this.out.close();
                }
                return n;
            }
        }

        CompletableFuture<Integer> submitCommit(long offset, int size, Function<UnderConstruction, ReadOnly> closeFunction, ExecutorService executor, RaftPeerId id, long index) {
            boolean close = closeFunction != null;
            Supplier<String> name = () -> "commit(" + this.getRelativePath() + ", " + offset + ", " + size + ", close? " + close + ") @" + id + ":" + index;
            WriteInfo info = this.writeInfos.get(index);
            if (info == null) {
                return JavaUtils.completeExceptionally(new IOException(name.get() + " is already committed."));
            }
            CheckedSupplier task = LogUtils.newCheckedSupplier(LOG, () -> {
                if (offset != this.committedSize) {
                    throw new IOException("Offset/size mismatched: offset = " + offset + " != committedSize = " + this.committedSize + ", path=" + this.getRelativePath());
                }
                if (this.committedSize + (long)size > this.writeSize) {
                    throw new IOException("Offset/size mismatched: committed (=" + this.committedSize + ") + size (=" + size + ") > writeSize = " + this.writeSize);
                }
                this.committedSize += (long)size;
                if (close) {
                    ReadOnly ignored = (ReadOnly)closeFunction.apply(this);
                    this.writeInfos.remove(index);
                }
                info.getCommitFuture().complete(size);
                return size;
            }, name);
            WriteInfo previous = this.writeInfos.remove(info.getPreviousIndex());
            CompletableFuture<Integer> previousCommit = previous != null ? previous.getCommitFuture() : CompletableFuture.completedFuture(0);
            return info.getWriteFuture().thenCombineAsync(previousCommit, (wSize, previousCommitSize) -> {
                Preconditions.assertTrue(size == wSize);
                try {
                    return (Integer)task.get();
                }
                catch (IOException e) {
                    throw new CompletionException("Failed " + task, e);
                }
            }, (Executor)executor);
        }
    }

    static class WriteInfo {
        private final CompletableFuture<Integer> writeFuture;
        private final CompletableFuture<Integer> commitFuture;
        private final long previousIndex;

        WriteInfo(CompletableFuture<Integer> writeFuture, long previousIndex) {
            this.writeFuture = writeFuture;
            this.commitFuture = new CompletableFuture();
            this.previousIndex = previousIndex;
        }

        CompletableFuture<Integer> getCommitFuture() {
            return this.commitFuture;
        }

        CompletableFuture<Integer> getWriteFuture() {
            return this.writeFuture;
        }

        long getPreviousIndex() {
            return this.previousIndex;
        }
    }

    static class ReadOnly
    extends FileInfo {
        private final long committedSize;
        private final long writeSize;

        ReadOnly(UnderConstruction f) {
            super(f.getRelativePath());
            this.committedSize = f.getCommittedSize();
            this.writeSize = f.getWriteSize();
        }

        @Override
        long getCommittedSize() {
            return this.committedSize;
        }

        @Override
        long getWriteSize() {
            return this.writeSize;
        }
    }

    static class Watch
    extends FileInfo {
        private final CompletableFuture<UnderConstruction> future = new CompletableFuture();

        Watch(Path relativePath) {
            super(relativePath);
        }

        CompletableFuture<UnderConstruction> getFuture() {
            return this.future;
        }

        CompletableFuture<UnderConstruction> complete(UnderConstruction uc) {
            Preconditions.assertTrue(this.getRelativePath().equals(uc.getRelativePath()));
            this.future.complete(uc);
            return this.future;
        }
    }
}

