/*
 * Copyright 1997, 1998 Phil Schwan <pschwan@cmu.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#else
#  error You did not run configure, did you?
#endif

#ifdef ENABLE_ARENAS
#  ifndef SMFS
#    include <signal.h>
#    ifdef HAVE_SIGNUM_H
#      include <signum.h>
#    else
#      ifdef HAVE_SYS_SIGNUM_H
#        include <sys/signal.h> /* BSD */
#      endif
#    endif
#  endif
#endif

#include <stdio.h>
#include <stdlib.h>
#include "xferstats.h"

__ptr_t
amalloc(arena_t ** arena, size_t size)
{
  __ptr_t mem; /* the bit to return to the user */
#ifdef ENABLE_ARENAS
  arena_t * current_arena_ptr, * new_arena;

  if (size > ARENA_SIZE)
    {
      fprintf(stderr, "xferstats: fatal: arena size too small.  recompile with a larger ARENA_SIZE (at least %ld bytes, probably more)\n", (long) size);
      exit(1);
    }
  else if (*arena == NULL)
    {
      /* create the first arena */
#  ifdef DEBUG_MALLOC
      fprintf(stderr, "amalloc: creating first arena\n");
#  endif
      MY_MALLOC(new_arena, sizeof(arena_t));
      MY_MALLOC(new_arena->memory, ARENA_SIZE);
      new_arena->prev_ptr = new_arena->next_ptr = NULL;
#  ifndef SMFS
      new_arena->allocs = 1;
#  endif
      new_arena->free = new_arena->memory + size;
      *arena = new_arena;
      return new_arena->memory;
    }

  for (current_arena_ptr = *arena; ;
       current_arena_ptr = current_arena_ptr->next_ptr)
    {
      if (size <= (ARENA_SIZE -
		   (current_arena_ptr->free - current_arena_ptr->memory)))
	{
	  /* it'll fit! */
#  ifdef DEBUG_MALLOC
	  fprintf(stderr, "amalloc: found an arena that will hold the %ld "
		  "byte request (%p)\n", (long) size, current_arena_ptr);
#  endif
#  ifndef SMFS
	  current_arena_ptr->allocs++;
#  endif
	  mem = current_arena_ptr->free;
	  current_arena_ptr->free += size;
	  return mem;
	}
#  ifdef SMFS
      else
#  else
      else if (current_arena_ptr->next_ptr == NULL)
#  endif /* ifdef SMFS */
	{
	  /* we've checked them all and it won't fit anywhere. make a new
	   * one */
#  ifdef DEBUG_MALLOC
	  fprintf(stderr, "amalloc: checked all arenas, need a new one\n");
#  endif
	  MY_MALLOC(new_arena, sizeof(arena_t));
	  MY_MALLOC(new_arena->memory, ARENA_SIZE);
	  new_arena->prev_ptr = NULL;
	  (*arena)->prev_ptr = new_arena;
	  new_arena->next_ptr = *arena;
	  *arena = new_arena;
#  ifndef SMFS
	  new_arena->allocs = 1;
#  endif
	  new_arena->free = new_arena->memory + size;
	  return new_arena->memory;
	}
    }
#else /* ENABLE_ARENAS */
  if (!(mem = (__ptr_t)malloc(size)))
    {
      perror("amalloc: malloc");
      exit(1);
    }

  *arena = NULL; /* it's still not clear why I have to do this, but no
		  * matter */

  return mem;
#endif /* ENABLE_ARENAS */
} /* amalloc */


#ifndef SMFS
void
afree(arena_t * arena, __ptr_t memory)
{
#ifdef ENABLE_ARENAS
  arena_t * current_arena_ptr;

#  ifdef DEBUG_MALLOC
  fprintf(stderr, "afree: looking for arena containing block starting at %p\n",
	  memory);
#  endif
  for (current_arena_ptr = arena;
       current_arena_ptr != NULL;
       current_arena_ptr = current_arena_ptr->next_ptr)
    {
      if (memory - current_arena_ptr->memory >= 0 &&
	  memory - current_arena_ptr->memory <= ARENA_SIZE)
	{
#  ifdef DEBUG_MALLOC
	  fprintf(stderr, "afree: found it, this block starts at %p and goes "
		  "through %p\n", current_arena_ptr->memory,
		  current_arena_ptr->memory + ARENA_SIZE);
#  endif

	  current_arena_ptr->allocs--;
	  return;
	}
    }

  /* the block wasn't found.  helpfully segfault */
  raise(SIGSEGV);
#else /* ENABLE_ARENAS */
  free(memory);
#endif /* ENABLE_ARENAS */
} /* afree */


void
aclean(arena_t ** arena)
{
#ifndef ENABLE_ARENAS
  return;
#else /* ENABLE_ARENAS */
  arena_t * current_arena_ptr = *arena, * tmp_arena_ptr;

  while (current_arena_ptr != NULL)
    {
      if (!current_arena_ptr->allocs)
	{
	  /* found one that's unused, freeing */
#ifdef DEBUG_MALLOC
	  fprintf(stderr, "aclean: found an unused block (%p).  freeing\n",
		  current_arena_ptr->memory);
#endif
	  free(current_arena_ptr->memory);
	  if (current_arena_ptr->prev_ptr == NULL)
	    {
	      /* this is the first block */
#ifdef DEBUG_MALLOC
	      fprintf(stderr, "aclean:    it's the first block, moving *arena "
		      "down a notch\n");
#endif
	      if (current_arena_ptr->next_ptr != NULL)
		current_arena_ptr->next_ptr->prev_ptr = NULL;
	      *arena = current_arena_ptr->next_ptr;
	      free(current_arena_ptr);
	      current_arena_ptr = *arena;
	    }
	  else
	    {
#ifdef DEBUG_MALLOC
	      fprintf(stderr, "aclean:    it's not the first block, removing "
		      "from the list\n");
	      fprintf(stderr, "cap        : %p\n", current_arena_ptr);
	      fprintf(stderr, "cap->pp    : %p\n",
		      current_arena_ptr->prev_ptr);
	      fprintf(stderr, "cap->pp->np: %p\n",
		      current_arena_ptr->prev_ptr->next_ptr);
#endif
	      current_arena_ptr->prev_ptr->next_ptr = tmp_arena_ptr =
		current_arena_ptr->next_ptr;
	      if (current_arena_ptr->next_ptr != NULL)
		current_arena_ptr->next_ptr->prev_ptr =
		  current_arena_ptr->prev_ptr;
	      free(current_arena_ptr);
	      current_arena_ptr = tmp_arena_ptr;
	    }
	}
      else
	current_arena_ptr = current_arena_ptr->next_ptr;
      if (current_arena_ptr == NULL)
	break;
    }
#endif /* ENABLE_ARENAS */
} /* aclean */
#endif /* ifndef SMFS */
