/*
 * Copyright 1997, 1998 Phil Schwan <pschwan@cmu.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#else
#  error You did not run configure, did you?
#endif

#ifdef HAVE_STRING_H
#  include <string.h> /* needed everywhere */
#else
#  error The <string.h> header file is required to compile xferstats
#endif

#ifdef HAVE_CTYPE_H
#  include <ctype.h> /* needed for isdigit, tolower, isalpha */
#else
#  error The <ctype.h> header file is required to compile xferstats
#endif

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>

#include "xferstats.h"

typedef struct
{
  char *command;
  void (*function)(pointers_t *, char *, char *, char *);
  char args;
} config_function;

/* prototypes for config functions */
void config_logfile(pointers_t *, char *);
void config_anon(pointers_t *);
void config_guest(pointers_t *);
void config_real(pointers_t *);
void config_inbound(pointers_t *);
void config_outbound(pointers_t *);
void config_hourly(pointers_t *);
void config_dow(pointers_t *);
void config_dom(pointers_t *);
void config_tld(pointers_t *);
void config_domain(pointers_t *);
void config_host(pointers_t *);
void config_dir(pointers_t *);
void config_file(pointers_t *);
void config_monthly(pointers_t *);
void config_html(pointers_t *);
void config_stdin(pointers_t *);
void config_logtype(pointers_t *, char *);
void config_dir_depth(pointers_t *, char *);
void config_dir_filter(pointers_t *, char *);
/* void config_tld_filter(pointers_t *, char *); */
/* This will have to wait for another version */
void config_number_file_stats(pointers_t *, char *);
void config_number_dir_stats(pointers_t *, char *);
void config_number_daily_stats(pointers_t *, char *);
void config_number_tld_stats(pointers_t *, char *);
void config_number_domain_stats(pointers_t *, char *);
void config_number_host_stats(pointers_t *, char *);
void config_max_report_size(pointers_t *, char *);
void config_graph_path(pointers_t *, char *);
void config_no_html_headers(pointers_t *);
void config_single_page(pointers_t *);

static config_function config_commands[] =
{
  {"LOGFILE", (void *)config_logfile, 1},
  {"ANON_TRAFFIC", (void *)config_anon, 0},
  {"GUEST_TRAFFIC", (void *)config_guest, 0},
  {"REAL_TRAFFIC", (void *)config_real, 0},
  {"INBOUND", (void *)config_inbound, 0},
  {"OUTBOUND", (void *)config_outbound, 0},
  {"HOURLY_REPORT", (void *)config_hourly, 0},
  {"DOW_REPORT", (void *)config_dow, 0},
  {"DOM_REPORT", (void *)config_dom, 0},
  {"TLD_REPORT", (void *)config_tld, 0},
  {"DOMAIN_REPORT", (void *)config_domain, 0},
  {"HOST_REPORT", (void *)config_host, 0},
  {"DIR_REPORT", (void *)config_dir, 0},
  {"FILE_REPORT", (void *)config_file, 0},
  {"MONTHLY_REPORT", (void *)config_monthly, 0},
  {"HTML_OUTPUT", (void *)config_html, 0},
  {"USE_STDIN", (void *)config_stdin, 0},
  {"LOG_TYPE", (void *)config_logtype, 1},
  {"DIR_DEPTH", (void *)config_dir_depth, 1},
  {"DIR_FILTER", (void *)config_dir_filter, 1},
  /* {"TLD_FILTER", (void *)config_tld_filter, 1}, */
  /* This will have to wait for another version */
  {"NUMBER_FILE_STATS", (void *)config_number_file_stats, 1},
  {"NUMBER_DIR_STATS", (void *)config_number_dir_stats, 1},
  {"NUMBER_DAILY_STATS", (void *)config_number_daily_stats, 1},
  {"NUMBER_TLD_STATS", (void *)config_number_tld_stats, 1},
  {"NUMBER_DOMAIN_STATS", (void *)config_number_domain_stats, 1},
  {"NUMBER_HOST_STATS", (void *)config_number_host_stats, 1},
  {"MAX_REPORT_SIZE", (void *)config_max_report_size, 1},
  {"GRAPH_PATH", (void *)config_graph_path, 1},
  {"NO_HTML_HEADERS", (void *)config_no_html_headers, 0},
  {"SINGLE_PAGE", (void *)config_single_page, 0},
  {"", NULL, 0}
};


void
config_logfile(pointers_t *pointers, char *logfile)
{
  if (pointers->config->file_name != NULL)
    free(pointers->config->file_name);

  pointers->config->file_name = (char *)malloc(strlen(logfile) + 1);
  strcpy(pointers->config->file_name, logfile);
} /* config_logfile */


void
config_anon(pointers_t *pointers)
{
  pointers->config->anon_traffic = 1;
} /* config_anon */


void
config_guest(pointers_t *pointers)
{
  pointers->config->guest_traffic = 1;
} /* config_guest */


void
config_real(pointers_t *pointers)
{
  pointers->config->real_traffic = 1;
} /* config_real */


void
config_inbound(pointers_t *pointers)
{
  pointers->config->inbound = 1;
} /* config_inbound */


void
config_outbound(pointers_t *pointers)
{
  pointers->config->outbound = 1;
} /* config_outbound */


void
config_hourly(pointers_t *pointers)
{
  pointers->config->hourly_traffic = 1;
} /* config_hourly */


void
config_dow(pointers_t *pointers)
{
  pointers->config->dow_traffic = 1;
} /* config_dow */


void
config_dom(pointers_t *pointers)
{
  pointers->config->dom_traffic = 1;
} /* config_dom */


void
config_tld(pointers_t *pointers)
{
  pointers->config->tld_traffic = 1;
} /* config_tld */


void
config_domain(pointers_t *pointers)
{
  pointers->config->domain_traffic = 1;
} /* config_domain */


void
config_host(pointers_t *pointers)
{
  pointers->config->host_traffic = 1;
} /* config_host */


void
config_dir(pointers_t *pointers)
{
  pointers->config->dir_traffic = 1;
} /* config_dir */


void
config_file(pointers_t *pointers)
{
  pointers->config->file_traffic = 1;
} /* config_file */


void
config_monthly(pointers_t *pointers)
{
  pointers->config->monthly_traffic = 1;
} /* config_monthly */


void
config_html(pointers_t *pointers)
{
  pointers->config->html_output = 1;
} /* config_html */


void
config_stdin(pointers_t *pointers)
{
  pointers->config->use_stdin = 1;
} /* config_stdin */


void
config_logtype(pointers_t *pointers, char *typestr)
{
  int logtype;

  /* check text cases */
  if (!strcasecmp(typestr, "wu-ftpd") || !strcasecmp(typestr, "wuftpd")
      || !strcasecmp(typestr, "wu-ftp") || !strcasecmp(typestr, "wuftp"))
    {
      pointers->config->log_type = 0;
      return;
    }
  else if (!strcasecmp(typestr, "ncftpd") || !strcasecmp(typestr, "ncftp"))
    {
      pointers->config->log_type = 1;
      return;
    }
  else if (!strcasecmp(typestr, "apache"))
    {
      pointers->config->log_type = 2;
      return;
    }

  /* check numerical cases and whine if nothing valid was found */
  switch (logtype = atoi(typestr))
    {
    case 1:
    case 2:
    case 3:
      pointers->config->log_type = logtype - 1;
      break;
    default:
      fprintf(stderr, "xferstats: config_logtype: invalid LOG_TYPE (%s).\n",
	      typestr);
      exit(1);
    }
} /* config_logtype */


void
config_dir_depth(pointers_t *pointers, char *depthstr)
{
  int depth = atoi(depthstr);

  if (depth > UCHAR_MAX)
    {
      pointers->config->depth = UCHAR_MAX;
      fprintf(stderr, "xferstats: config_dir_depth: warning: DIR_DEPTH reduced"
	      " to %d.\n", UCHAR_MAX);
    }
  else if ((pointers->config->depth = depth) < 1)
    {
      fprintf(stderr, "xferstats: config_dir_depth: DIR_DEPTH is too low or "
	      "invalid (%s).  DIR_DEPTH must be >= 1.\n", depthstr);
      exit (1);
    }
} /* config_dir_depth */


void
config_dir_filter(pointers_t * pointers, char * path)
{
  only_dir_t * item;

  MY_MALLOC(item, sizeof(only_dir_t));

  MY_MALLOC(item->dir, (item->len = strlen(path)) + 1);
  strcpy(item->dir, path);

  list_add_item(&pointers->config->only_dir, item);
} /* config_dir_filter */


/* This will have to wait for another version */
/*void
config_tld_filter(pointers_t *pointers, char *tld)
{
  if (pointers->config->only_tld != NULL)
    free(pointers->config->only_tld);
  pointers->config->only_tld = (char *)malloc((pointers->config->only_tld_length = strlen(tld)) + 1);
  strcpy(pointers->config->only_tld, tld);
} */ /* config_tld_filter */


void
config_number_file_stats(pointers_t *pointers, char *numstr)
{
  long number = atoi(numstr);

  if (number > INT_MAX)
    {
      pointers->config->number_file_stats = INT_MAX;
      fprintf(stderr,"xferstats: config_number_file_stats: warning: "
	      "NUMBER_FILE_STATS reduced to %d.\n", INT_MAX);
    }
  else if ((pointers->config->number_file_stats = number) < 0)
    {
      fprintf(stderr, "xferstats: config_number_file_stats: NUMBER_FILE_STATS "
	      "is too low or invalid.  It must be >= 0.\n");
      exit(1);
    }
} /* config_number_file_stats */


void
config_number_dir_stats(pointers_t *pointers, char *numstr)
{
  long number = atoi(numstr);

  if (number > INT_MAX)
    {
      pointers->config->number_dir_stats = INT_MAX;
      fprintf(stderr,"xferstats: config_number_dir_stats: warning: "
	      "NUMBER_DIR_STATS reduced to %d.\n", INT_MAX);
    }
  else if ((pointers->config->number_dir_stats = number) < 0)
    {
      fprintf(stderr, "xferstats: config_number_dir_stats: NUMBER_DIR_STATS "
	      "is too low or invalid.  It must be >= 0.\n");
      exit(1);
    }
} /* config_number_dir_stats */


void
config_number_daily_stats(pointers_t *pointers, char *numstr)
{
  long number = atoi(numstr);

  if (number > INT_MAX)
    {
      pointers->config->number_daily_stats = INT_MAX;
      fprintf(stderr,"xferstats: config_number_daily_stats: warning: "
	      "NUMBER_DAILY_STATS reduced to %d.\n", INT_MAX);
    }
  else if ((pointers->config->number_daily_stats = number) < 0)
    {
      fprintf(stderr, "xferstats: config_number_daily_stats: "
	      "NUMBER_DAILY_STATS is too low or invalid.  It must be >= 0.\n");
      exit(1);
    }
} /* config_number_daily_stats */


void
config_number_tld_stats(pointers_t *pointers, char *numstr)
{
  long number = atoi(numstr);

  if (number > INT_MAX)
    {
      pointers->config->number_tld_stats = INT_MAX;
      fprintf(stderr,"xferstats: config_number_tld_stats: warning: "
	      "NUMBER_TLD_STATS reduced to %d.\n", INT_MAX);
    }
  else if ((pointers->config->number_tld_stats = number) < 0)
    {
      fprintf(stderr, "xferstats: config_number_tld_stats: NUMBER_TLD_STATS "
	      "is too low or invalid.  It must be >= 0.\n");
      exit(1);
    }
} /* config_number_tld_stats */


void
config_number_domain_stats(pointers_t *pointers, char *numstr)
{
  long number = atoi(numstr);

  if (number > INT_MAX)
    {
      pointers->config->number_domain_stats = INT_MAX;
      fprintf(stderr,"xferstats: config_number_domain_stats: warning: "
	      "NUMBER_DOMAIN_STATS reduced to %d.\n", INT_MAX);
    }
  else if ((pointers->config->number_domain_stats = number) < 0)
    {
      fprintf(stderr, "xferstats: config_number_domain_stats: "
	      "NUMBER_DOMAIN_STATS is too low or invalid.  It must be >= "
	      "0.\n");
      exit(1);
    }
} /* config_number_domain_stats */


void
config_number_host_stats(pointers_t *pointers, char *numstr)
{
  long number = atoi(numstr);

  if (number > INT_MAX)
    {
      pointers->config->number_host_stats = INT_MAX;
      fprintf(stderr,"xferstats: config_number_host_stats: warning: "
	      "NUMBER_HOST_STATS reduced to %d.\n", INT_MAX);
    }
  else if ((pointers->config->number_host_stats = number) < 0)
    {
      fprintf(stderr, "xferstats: config_number_host_stats: NUMBER_HOST_STATS"
	      " is too low or invalid.  It must be >= 0.\n");
      exit(1);
    }
} /* config_number_host_stats */


void
config_max_report_size(pointers_t *pointers, char *numstr)
{
  long number = atoi(numstr);

  if (number > INT_MAX)
    {
      pointers->config->max_report_size = INT_MAX;
      fprintf(stderr,"xferstats: config_max_report_size: warning: "
	      "MAX_REPORT_SIZE reduced to %d.\n", INT_MAX);
    }
  else if ((pointers->config->max_report_size = number) < 0)
    {
      fprintf(stderr, "xferstats: config_max_report_size: MAX_REPORT_SIZE is "
	      "too low or invalid.  It must be >= 0.\n");
      exit(1);
    }
} /* config_max_report_size */


void
config_graph_path(pointers_t *pointers, char *path)
{
  if (pointers->config->graph_path)
    free(pointers->config->graph_path);

  pointers->config->graph_path = (char *)malloc(strlen(path) + 1);
  strcpy(pointers->config->graph_path, path);
} /* config_graph_path */


void
config_no_html_headers(pointers_t *pointers)
{
  pointers->config->no_html_headers = 1;
} /* config_no_html_headers */


void
config_single_page(pointers_t *pointers)
{
  pointers->config->single_page = 1;
} /* config_single_page */


/* parse_config_line returns 0 if there is no valid data, 1 if there is only a
 * valid "command", 2 if there is a valid "command" and a valid "arg1", 3 if
 * there is also a valid "arg2", and 4 if the command and all 3 args are
 * valid */
int
parse_config_line(char *line, char **command, char *args[3])
{
  int arg_no = 0;

  *command = NULL; args[0] = NULL; args[1] = NULL; args[2] = NULL;
  /* zip past whitespace */
  while (*line == ' ') line++;
  
  /* see if there's anything left to parse */
  if (*line == '#' || *line == '\0' || *line == '\n')
    return 0;

  /* set the beginning of "command" */
  *command = line;
  /* don't care about #s unless they're prepended by a space */
  while (*line != ' ' && *line != '\0' && *line != '\n') line++;
  /* it's possible that there was no terminating carriage return.  if we find
   * a null, terminate "command" and return */
  if (*line == '\0')
    return 1;
  else if (*line == '\n')
    {
      /* terminate "command" */
      *line = '\0';
      return 1;
    }
  /* terminate "command" */
  *line = '\0';

  /* loop until we reach the end of the line or we have 3 arguments
   * (our maximum) */
  for (; arg_no <= 2; arg_no++)
    {
      /* set the beginning of arg */
      args[arg_no] = ++line;
      /* don't care about #s unless they're prepended by a space */
      while (*line != ' ' && *line != '\0' && *line != '\n') line++;
      /* catch nulls again... */
      if (*line == '\0')
	return (arg_no + 2);
      else if (*line == '\n')
	{
	  /* terminate arg */
	  *line = '\0';
	  return (arg_no + 2);
	}
      /* terminate arg */
      *line = '\0';
    }

  fprintf(stderr, "\"The obvious mathematical breakthrough would be "
	  "development of an easy way to factor large prime numbers.\" -- "
	  "Bill Gates, The Road Ahead, 1995");
  fprintf(stderr, "WARNING: something is whacked in parse_config_line, you "
	  "should never see this.\n");
  return 0;
} /* parse_config_line */


int
init_config(pointers_t *pointers)
{
  char tmp_str[1024], *command, *args[3];
  FILE *file_ptr = NULL;
  int index, valid;
  char *upcase_char;
  
  if (pointers->config->config_file == NULL)
    {
      if ((file_ptr = fopen(DEFAULT_CONFIG, "r")) == NULL)
	{
	  fprintf(stderr, "xferstats: init_config: %s: ", DEFAULT_CONFIG);
	  perror("fopen");
	  return 0;
	  exit(1);
	}
    }
  else
    if ((file_ptr = fopen(pointers->config->config_file, "r")) == NULL)
      {
	fprintf(stderr, "xferstats: init_config: %s: ",
		pointers->config->config_file);
	perror("fopen");
	return 0;
	exit(1);
      }

  while (!feof(file_ptr))
    {
      tmp_str[0] = '\0';
      fgets(tmp_str, sizeof(tmp_str) - 1, file_ptr);
#ifdef DEBUG
      fprintf(stderr, "init_config: RAW: %s", tmp_str);
#endif

      /* parse the config line into 'command' and 'args' */
      if ((valid = parse_config_line(tmp_str, &command, args) - 1) < 0)
	/* parse_config_line couldn't find a 'command' let alone args...move
	 * along... */
	continue;

#ifdef DEBUG
      fprintf(stderr, "command: %s arg1: %s arg2: %s arg3: %s\n",
	     command ? command : "(nil)",
	     args[0] ? args[0] : "(nil)",
	     args[1] ? args[1] : "(nil)",
	     args[2] ? args[2] : "(nil)");
#endif

      /* convert all to upper case */
      for (upcase_char = command;
	   (*upcase_char = toupper((int)*upcase_char)); upcase_char++);

      /* scan the table to see if this matches any of the commands */
      for (index = 0; config_commands[index].function != NULL; index++)
	if (!strcmp(config_commands[index].command, command))
	  {
	    /* correct # of args? */
	    if (valid == config_commands[index].args)
	      switch(valid)
		{
		case 0:
		  config_commands[index].function(pointers, NULL, NULL, NULL);
		  break;
		case 1:
		  config_commands[index].function(pointers, args[0], NULL,
						  NULL);
		  break;
		case 2:
		  config_commands[index].function(pointers, args[0], args[1],
						  NULL);
		  break;
		case 3:
		  config_commands[index].function(pointers, args[0], args[1],
						  args[2]);
		  break;
		}
	    else
	      /* incorrect # of args, whine about it */
	      fprintf(stderr, "xferstats: warning: bad config line \"%s\" "
		      "(wrong # of args)\n", tmp_str);
	    break;
	  }

      /* command wasn't found, whine about it */
      if (config_commands[index].function == NULL)
	fprintf(stderr, "xferstats: warning: bad config line \"%s\" (unknown "
		"config command)\n", tmp_str);
    }

  fclose(file_ptr);
  return 0;
} /* init_config */
