/*
 * Copyright 1997, 1998 Phil Schwan <pschwan@cmu.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#else
#  error You did not run configure, did you?
#endif

#ifdef HAVE_STRING_H
#  include <string.h> /* needed everywhere */
#else
#  error The <string.h> header file is required to compile xferstats
#endif

#include <stdlib.h>
#include <stdio.h>
#include "xferstats.h"

const char LONG_DAYS[7][10] = {"Sunday", "Monday", "Tuesday", "Wednesday",
			       "Thursday", "Friday", "Saturday"};
const char LONG_MONTHS[12][10] = {"January", "February", "March", "April",
				  "May", "June", "July", "August", "September",
				  "October", "November", "December"};
extern const char T_NAMES[5][7];

char *
big_number(unsigned LONGLONG bignum, char **buffer, int *size)
{
  char tmpstr[1024];
  int a, b, len;

#ifdef BROKEN_LL
  len = sprintf(tmpstr, "%lu", bignum);
#else
#  ifdef IS_DEC
  len = sprintf(tmpstr, "%Lu", bignum);
#  else
  len = sprintf(tmpstr, "%llu", bignum);
#  endif /* IS_DEC */
#endif /* BROKEN_LL */

  /* make a = the final length of the string (numbers plus commas) */
  a = len + (b = (len / 3));
  /* allocate ram for the string */
  if (*size < a + 1)
    {
      *buffer = realloc(*buffer, a + 1);
      *size = a + 1;
    }

  /* copy the digits before the first comma (if any) */
  strncpy(*buffer, tmpstr, (a = len % 3));
  (*buffer)[a] = '\0';

  /* loop until there are no more blocks of 3 numbers left */
  for (; b > 0; b--, a += 3)
    {
      /* only put a comma in if there were digits before */
      if (a)
	strcat(*buffer, ",");
      /* copy the next 3 numbers into out string */
      strncat(*buffer, tmpstr + a, 3);
    }

  return *buffer;
} /* big_number */


void
html_graph(float percent, FILE *html, char *path)
{
  float foo;
  char printed = 0;

  if (percent >= 1)
    printed = 1;  

  for (foo = 64; foo >= 1; foo /= 2)
    if (percent > foo)
      {
	if (path)
	  fprintf(html, "<IMG SRC=\"%sg%d.jpg\" ALT=\"g%d\">", path, (int) foo,
		  (int) foo);
	else
	  fprintf(html, "<IMG SRC=\"g%d.jpg\" ALT=\"g%d\">", (int) foo,
		  (int) foo);
	percent -= foo;
      }

  if (percent >= .1)
    printed = 1;

  for (foo = .8; foo >= .1; foo /= 2.0)
    if (percent > foo)
      {
	printed = 1;
	if (path)
	  fprintf(html, "<IMG SRC=\"%sg_%d.jpg\" ALT=\"g%1.1f\">",
		  path, (int) (foo * 10), foo);
	else
	  fprintf(html, "<IMG SRC=\"g_%d.jpg\" ALT=\"g%1.1f\">",
		  (int) (foo * 10), foo);
	percent -= foo;
      }

  if (!printed)
    fprintf(html, "&nbsp");

} /* html_graph */


void
display_daily_totals(pointers_t * pointers)
{
  xfdays_t * current_day_ptr = pointers->first_day_ptr;
  char * buffer = NULL;
  int displayed_count = 0, count = 0, size = 0;
  unsigned long num_days = 0;
  double percent;
  unsigned LONGLONG highest = 0;
  FILE * html =  NULL;

#ifdef DEBUGS
  fprintf(stderr, "Beginning display_daily_totals\n");
#endif

  strncpy(pointers->low_date + 10, pointers->low_date + 19, 5);
  strncpy(pointers->high_date + 10, pointers->high_date + 19, 5);
  pointers->low_date[15] = pointers->high_date[15] = '\0';

  for (; current_day_ptr != NULL; current_day_ptr = current_day_ptr->next_ptr)
    {
      pointers->file_count += current_day_ptr->file_count;
      pointers->data += current_day_ptr->data;
      if (current_day_ptr->data > highest)
	highest = current_day_ptr->data;
      num_days++;
    }

  /* dump our totals information--here is as good a place as any */
  if (pointers->config->html_output)
    {
      if (pointers->config->single_page)
        {
	  if (!(html = fopen("xferstats.html", "w")))
	    {
	      perror("fopen");
	      exit(1);
	    }
        }
      else
	if (!(html = fopen("totals.html", "w")))
	  {
	    perror("fopen");
	    exit(1);
	  }
      /* first for HTML */
      if (!pointers->config->no_html_headers && !pointers->config->single_page)
	fprintf(html, "<HTML>"); 
      if (!pointers->config->no_html_headers)
	fprintf(html, "<TITLE>xferstats: %s to %s</TITLE>\n",
		pointers->low_date, pointers->high_date);
      if (!pointers->config->no_html_headers && !pointers->config->single_page)
	fprintf(html, "<BODY>");

      fprintf(html, "<CENTER><H1>TOTALS FOR SUMMARY PERIOD <I>%s</I> TO "
	      "<I>%s</I> (%s days)</H1><HR>\n",
	      pointers->low_date, pointers->high_date,
	      big_number(num_days, &buffer, &size));
      fprintf(html, "<TABLE BORDER = 0 WIDTH = 70%%>\n");
      fprintf(html, "<TR><TH ALIGN=LEFT>Files Transmitted During Summary "
	      "Period</TH><TH ALIGN=RIGHT>%s</TH></TR>\n",
	      big_number(pointers->file_count, &buffer, &size));
      fprintf(html, "<TR><TH ALIGN=LEFT>Average Files Transmitted Daily</TH>"
	      "<TH ALIGN=RIGHT>%s</TH><TR>\n",
	      big_number(pointers->file_count / num_days, &buffer, &size));
      fprintf(html, "<TR><TH ALIGN=LEFT>Bytes Transmitted During Summary "
	      "Period</TH><TH ALIGN=RIGHT>%s</TH></TR>\n",
	      big_number(pointers->data, &buffer, &size));
      fprintf(html, "<TR><TH ALIGN=LEFT>Average Bytes Transmitted Per File"
	      "</TH><TH ALIGN=RIGHT>%s</TH></TR>\n",
	      big_number(pointers->data/pointers->file_count, &buffer, &size));
      fprintf(html, "<TR><TH ALIGN=LEFT>Average Bytes Transmitted Daily</TH>"
	      "<TH ALIGN=RIGHT>%s</TH></TR></TABLE><BR>\n",
	      big_number(pointers->data / num_days, &buffer, &size));
      if (!pointers->config->single_page)
	{
	  fprintf(html, "<HR><A HREF=\"daily.html\">Daily Transmission "
		  "Statistics </A><BR>");
	  fprintf(html, "<A HREF=\"monthly.html\">Monthly Transmission "
		  "Statistics </A><BR>");
	  fprintf(html, "<A HREF=\"hour.html\">Hourly Transmission "
		  "Statistics</A> <BR>");
	  fprintf(html, "<A HREF=\"dow.html\">Day-of-the-Week Transmission "
		  "Statistics</A><BR>");
	  fprintf(html, "<A HREF=\"dom.html\">Day-of-the-Month Transmission "
		  "Statistics</A><BR><BR>");
	  fprintf(html, "<A HREF=\"host.html\">Transfer Statistics By Host</A>"
		  "<BR>");
	  fprintf(html, "<A HREF=\"domain.html\">Transfer Statistics By Domain"
		  "</A><BR>");
	  fprintf(html, "<A HREF=\"tld.html\">Transfer Statistics By Top Level"
		  " Domain</A><BR><BR>");
	  fprintf(html, "<A HREF=\"file.html\">Transfer Statistics By File</A>"
		  "<BR>");
	  fprintf(html, "<A HREF=\"dir.html\">Transfer Statistics By Directory"
		  "</A> </CENTER>\n");
	}
      fprintf(html, "<BR><BR><font size=\"-1\"><I><CENTER>Statistics generated"
	      " by <A HREF=\"http://xferstats.off.net:8080\">xferstats</A> "
	      "%s</CENTER></I></FONT><BR>\n", VERSION);
      if (!pointers->config->single_page)
	{
	  if (!pointers->config->no_html_headers)
	    fprintf(html, "</BODY></HTML>\n");
	  fclose(html);
	  if (!(html = fopen("daily.html", "w")))
	    {
	      perror("fopen");
	      exit(1);
	    }
	}
      else
	fprintf(html, "<HR>");
      if (!pointers->config->no_html_headers)
	{
	  fprintf(html, "<TITLE>xferstats: %s to %s</TITLE><BODY>\n",
		  pointers->low_date, pointers->high_date);
	}
    }
  else
    {
      /* and now for text */
      printf("TOTALS FOR SUMMARY PERIOD %s TO %s\n\n", pointers->low_date,
	     pointers->high_date);
      printf("Files Transmitted During Summary Period     %16s\n",
	     big_number(pointers->file_count, &buffer, &size));
      printf("Average Files Transmitted Daily             %16s\n",
	     big_number(pointers->file_count / num_days, &buffer, &size));
      printf("Bytes Transmitted During Summary Period     %16s\n\n",
	     big_number(pointers->data, &buffer, &size));
      printf("Average Bytes Transmitted Per File          %16s\n",
	     big_number(pointers->data/pointers->file_count, &buffer, &size));
      printf("Average Bytes Transmitted Daily             %16s\n\n",
	     big_number(pointers->data / num_days, &buffer, &size));
    }

  for (current_day_ptr = pointers->first_day_ptr;
       current_day_ptr != NULL;
       current_day_ptr = current_day_ptr->next_ptr)
    {
      /* These 4 lines skip all of the entries before the last 'n' dictated by
       * number_daily_stats */
      count++;
      if (pointers->config->number_daily_stats &&
	  count <= num_days - pointers->config->number_daily_stats)
	continue;

      displayed_count++;
      if (displayed_count == 1 || !pointers->config->number_daily_stats ||
	  pointers->config->number_daily_stats > num_days ||
	  displayed_count > num_days - pointers->config->number_daily_stats)
	{
	  if (displayed_count == 1 ||
	      (pointers->config->max_report_size &&
	       !(displayed_count % (pointers->config->max_report_size + 1))))
	    {
	      /* print our headers */
	      if (pointers->config->html_output)
		{
		  /* first for HTML */

		  /* this isn't the first set of headers, so close the previous
		   * table */
		  if (displayed_count != 1)
		    {
		      if (pointers->config->graph_path)
		        fprintf(html, "</TABLE><BR><IMG SRC=\"%sg1.jpg\" "
			        "ALT=\"g1\"> = %s bytes<BR><HR>\n",
			        pointers->config->graph_path,
			        big_number(highest / 50, &buffer, &size));
		      else
		        fprintf(html, "</TABLE><BR><IMG SRC=\"g1.jpg\" "
			        "ALT=\"g1\"> = %s bytes<BR><HR>\n",
			        big_number(highest / 50, &buffer, &size));
		    }
		  else
		    {
		      fprintf(html, "<CENTER>");

		      if (!pointers->config->number_daily_stats ||
			  pointers->config->number_daily_stats >= num_days)
			fprintf(html, "<U><H1>Daily Transmission Statistics "
				"for <I>%s</I> to <I>%s</I></H1></U><BR>\n",
				pointers->low_date, pointers->high_date);
		      else
			fprintf(html, "<U><H1>Daily Transmission Statistics "
				"(last %d days)</H1></U><BR>\n",
				pointers->config->number_daily_stats);
		    }
		  fprintf(html, "<TABLE BORDER=2><TR><TH></TH><TH>Number Of"
			  "</TH><TH>Number of</TH><TH>Average</TH><TH>Percent "
			  "Of</TH><TH>Percent Of</TH></TR>\n");
		  fprintf(html, "<TR><TH>Date</TH><TH>Files Sent</TH><TH>"
			  "Bytes Sent</TH><TH>Xmit Rate</TH><TH>Files Sent"
			  "</TH><TH>Bytes Sent</TH></TR>\n");
		}
	      else
		{
		  if (displayed_count == 1)
		    {
		      /* then for text */
		      if (!pointers->config->number_daily_stats ||
			  pointers->config->number_daily_stats >= num_days)
			printf("Daily Transmission Statistics\n\n");
		      else
			printf("Daily Transmission Statistics (last %d days)"
			       "\n\n", pointers->config->number_daily_stats);
		    }

		  printf("                 Number Of    Number of    Average   Percent Of  Percent Of\n");
		  printf("     Date        Files Sent  Bytes  Sent  Xmit Rate  Files Sent  Bytes Sent\n");
		  printf("---------------  ----------  -----------  ---------  ----------  ----------\n");
		}
	    }
	}

      if (pointers->config->html_output)
	{
	  fprintf(html, "<TR><TD NOWRAP>%s</TD><TD ALIGN=RIGHT>%u</TD>"
		  "<TD ALIGN=RIGHT>%s</TD><TD NOWRAP ALIGN=RIGHT>%5.1f KB/s"
		  "</TD><TD ALIGN=RIGHT>%10.2f</TD><TD ALIGN=RIGHT>%10.2f"
		  "</TD>", current_day_ptr->name, current_day_ptr->file_count,
		  big_number(current_day_ptr->data, &buffer, &size),
		  (double) current_day_ptr->data /
		  current_day_ptr->seconds / 1024,
		  (double) current_day_ptr->file_count /
		  pointers->file_count * 100,
		 (double) current_day_ptr->data / pointers->data * 100);
	  percent = (double) current_day_ptr->data / highest * 50;
	  fprintf(html, "<TD NOWRAP ALIGN=LEFT>");
	  html_graph(percent, html, pointers->config->graph_path);
	  fprintf(html, "</TD></TR>\n");
	}
      else
#ifdef BROKEN_LL
	printf("%s  %10u  %11lu  %5.1f K/s  %10.2f  %10.2f\n",
#else
#  ifdef IS_DEC
	printf("%s  %10u  %11Lu  %5.1f K/s  %10.2f  %10.2f\n",
#  else
	printf("%s  %10u  %11llu  %5.1f K/s  %10.2f  %10.2f\n",
#  endif /* IS_DEC */
#endif /* BROKEN_LL */
	       current_day_ptr->name, current_day_ptr->file_count,
	       current_day_ptr->data,
	       (double) current_day_ptr->data /
	       current_day_ptr->seconds / 1024,
	       (double) current_day_ptr->file_count /
	       pointers->file_count * 100,
	       (double) current_day_ptr->data / pointers->data * 100);
    }

  if (pointers->config->html_output)
    {
      if (pointers->config->graph_path)
	fprintf(html, "</TABLE><BR><IMG SRC=\"%sg1.jpg\" ALT=\"g1\"> = %s "
		"bytes<BR>\n", pointers->config->graph_path,
		big_number(highest / 50, &buffer, &size));
	else
	  fprintf(html, "</TABLE><BR><IMG SRC=\"g1.jpg\" ALT=\"g1\"> = %s "
		  "bytes<BR>\n", big_number(highest / 50, &buffer, &size));
      if (!pointers->config->single_page)
	{
	  fprintf(html, "<HR><A HREF=\"totals.html\">Return to Totals and "
		  "Index </A></CENTER>\n");
	  if (!pointers->config->no_html_headers)
	    fprintf(html, "</BODY></HTML>\n");
	}
      fclose(html);
    }

#ifdef DEBUGS
  fprintf(stderr, "display_daily_totals complete\n");
#endif
} /* display_daily_totals */


void
display_misc_totals(pointers_t * pointers, int type)
{
  arena_t ** arena = NULL;
  xfmisc_t * current_path_ptr, ** pointers_ptr = NULL;
#ifndef SMFS
#ifndef PTHREADS
#ifdef ENABLE_ARENAS
  arena_t * current_arena_ptr = NULL, * next_arena_ptr;
#endif
#endif
#endif
  unsigned int count = 1, number = 0, a;
  char *buffer = NULL, temp_name[55] = {0};
  double percent = 0;
  int size = 0;
  unsigned LONGLONG highest_count = 0;
  FILE *html = NULL;

#ifdef DEBUGS
  fprintf(stderr, "Beginning display_%s_totals\n", T_NAMES[type]);
#endif

  switch(type)
    {
    case T_DIR:
      arena = &pointers->dir_arena;
      number = pointers->config->number_dir_stats;
      pointers_ptr = &pointers->first_dir_ptr;
      highest_count = (*pointers_ptr)->data;
      break;
    case T_FILE:
      arena = &pointers->file_arena;
      number = pointers->config->number_file_stats;
      pointers_ptr = &pointers->first_file_ptr;
      highest_count = (*pointers_ptr)->file_count;
      break;
    case T_TLD:
      arena = &pointers->tld_arena;
      number = pointers->config->number_tld_stats;
      pointers_ptr = &pointers->first_tld_ptr;
      highest_count = (*pointers_ptr)->data;
      break;
    case T_HOST:
      arena = &pointers->host_arena;
      number = pointers->config->number_host_stats;
      pointers_ptr = &pointers->first_host_ptr;
      highest_count = (*pointers_ptr)->data;
      break;
    case T_DOMAIN:
      arena = &pointers->domain_arena;
      number = pointers->config->number_domain_stats;
      pointers_ptr = &pointers->first_domain_ptr;
      highest_count = (*pointers_ptr)->data;
      break;
    }

  if (*pointers_ptr == NULL)
    {
#ifdef DEBUGS
      fprintf(stderr, "No data to display.\n");
#endif
      return;
    }

  for (current_path_ptr = *pointers_ptr;
       current_path_ptr != NULL;
       current_path_ptr = current_path_ptr->next_ptr, count++)
    {
      
      if (number && count > number)
	break;

      if (count == 1 ||
	  (pointers->config->max_report_size &&
	   !(count % (pointers->config->max_report_size + 1))))
	if (pointers->config->html_output)
	  {
	    /* this isn't the first set of headers, so close the previous
	     * table */
	    if (count != 1)
	      {
		if (pointers->config->graph_path)
		  fprintf(html, "</TABLE><BR><IMG SRC=\"%sg1.jpg\" ALT=\"g1\">"
			  " = %s %s<BR><HR>\n", pointers->config->graph_path,
			  big_number(highest_count / 50, &buffer, &size), 
			  (type == T_FILE) ? "transfers" : "bytes");
		else
		  fprintf(html, "</TABLE><BR><IMG SRC=\"g1.jpg\" ALT=\"g1\"> ="
			  " %s %s<BR><HR>\n",
			  big_number(highest_count / 50, &buffer, &size), 
			  (type == T_FILE) ? "transfers" : "bytes");
	      }
	    else
	      {
		if (pointers->config->single_page)
		  {
		    if (!(html = fopen("xferstats.html", "a")))
		     {
	 	       perror("fopen");
		       exit(1);
		     }
		    fprintf(html, "<HR>");
		  }
		else
		  {
		    switch(type)
		      {
		      case T_DIR:
			if (!(html = fopen("dir.html", "w")))
			  {
			    perror("fopen");
			    exit(1);
			  }
			break;
		      case T_FILE:
			if (!(html = fopen("file.html", "w")))
			  {
			    perror("fopen");
			    exit(1);
			  }
			break;
		      case T_TLD:
			if (!(html = fopen("tld.html", "w")))
			  {
			    perror("fopen");
			    exit(1);
			  }
			break;
		      case T_HOST:
			if (!(html = fopen("host.html", "w")))
			  {
			    perror("fopen");
			    exit(1);
			  }
			break;
		      case T_DOMAIN:
			if (!(html = fopen("domain.html", "w")))
			  {
			    perror("fopen");
			    exit(1);
			  }
			break;
		      }
		    if (!pointers->config->no_html_headers)
		      fprintf(html, "<HTML><TITLE>xferstats: %s to %s</TITLE>"
			      "<BODY>", pointers->low_date,
			      pointers->high_date);
		  }
	      }

	    switch(type)
	      {
	      case T_DIR:
		if (count == 1)
		  {
		    if (number)
		      fprintf(html, "<CENTER><U><H1>Top %d Total Transfers "
			      "from each Archive Section for <I>%s</I> to "
			      "<I>%s</I> (By bytes)</H1></U><BR>\n",
			      number, pointers->low_date, pointers->high_date);
		    else
		      fprintf(html, "<CENTER><U><H1>Total Transfers from each "
			      "Archive Section for <I>%s</I> to <I>%s</I> (By "
			      "bytes)</H1></U><BR>\n",
			      pointers->low_date, pointers->high_date);
		  }
		fprintf(html, "<TABLE BORDER=2><TR><TH></TH><TH>Number Of</TH>"
			"<TH>Number of</TH><TH>Average</TH><TH>Percent Of</TH>"
			"<TH>Percent Of</TH></TR>\n");
		fprintf(html, "<TR><TH>Archive Section</TH><TH>Files Sent</TH>"
			"<TH>Bytes Sent</TH><TH>Xmit Rate</TH><TH>Files Sent"
			"</TH><TH>Bytes Sent</TH></TR>\n");
		break;
	      case T_FILE:
		if (count == 1)
		  {
		    if (number)
		      fprintf(html, "<CENTER><U><H1>Top %d Most Transferred "
			      "Files for <I>%s</I> to <I>%s</I> (by transfer "
			      "count)</H1></U><BR>\n", 
			      number, pointers->low_date, pointers->high_date);
		    else
		      fprintf(html, "<CENTER><U><H1>All Transferred Files for "
			      "<I>%s</I> to <I>%s</I> (by transfer count)</H1>"
			      "</U><BR>\n",
			      pointers->low_date, pointers->high_date);
		  }
		fprintf(html, "<TABLE BORDER=2><TR><TH></TH><TH>Number Of</TH>"
			"<TH>Number of</TH><TH>Average</TH><TH>Percent Of</TH>"
			"<TH>Percent Of</TH></TR>\n");
		fprintf(html, "<TR><TH>Filename</TH><TH>Files Sent</TH><TH>"
			"Bytes Sent</TH><TH>Xmit Rate</TH><TH>Files Sent</TH>"
			"<TH>Bytes Sent</TH></TR>\n");
		break;
	      case T_TLD:
		if (count == 1)
		  {
		    if (number)
		      fprintf(html, "<CENTER><U><H1>Top %d Top Level Domains "
			      "for <I>%s</I> to <I>%s</I> (By bytes)</H1></U>"
			      "<BR>\n", number, pointers->low_date,
			      pointers->high_date);
		    else
		      fprintf(html, "<CENTER><U><H1>Top Level Domain "
			      "Statistics for <I>%s</I> to <I>%s</I> (By "
			      "bytes)</H1></U><BR>\n", pointers->low_date,
			      pointers->high_date);
		  }
		fprintf(html, "<TABLE BORDER=2><TR><TH></TH><TH>Number Of</TH>"
			"<TH>Number of</TH><TH>Average</TH><TH>Percent Of</TH>"
			"<TH>Percent Of</TH></TR>\n");
		fprintf(html, "<TR><TH>Top Level Domain</TH><TH>Files Sent"
			"</TH><TH>Bytes Sent</TH><TH>Xmit Rate</TH><TH>Files "
			"Sent</TH><TH>Bytes Sent</TH></TR>\n");
		break;
	      case T_HOST:
		if (count == 1)
		  {
		    if (number)
		      fprintf(html, "<CENTER><U><H1>Top %d Hosts for <I>%s"
			      "</I> to <I>%s</I> (By bytes)</H1></U><BR>\n",
			      number, pointers->low_date, pointers->high_date);
		    else
		      fprintf(html, "<CENTER><U><H1>Host Statistics for <I>%s"
			      "</I> to <I>%s</I> (By bytes)</H1></U><BR>\n",
			      pointers->low_date, pointers->high_date);
		  }
		fprintf(html, "<TABLE BORDER=2><TR><TH></TH><TH>Number Of</TH>"
			"<TH>Number of</TH><TH>Average</TH><TH>Percent Of</TH>"
			"<TH>Percent Of</TH></TR>\n");
		fprintf(html, "<TR><TH>Host</TH><TH>Files Sent</TH><TH>Bytes "
			"Sent</TH><TH>Xmit Rate</TH><TH>Files Sent</TH><TH>"
			"Bytes Sent</TH></TR>\n");
		break;
	      case T_DOMAIN:
		if (count == 1)
		  {
		    if (number)
		      fprintf(html, "<CENTER><U><H1>Top %d Domains for <I>%s"
			      "</I> to <I>%s</I> (By bytes)</H1></U><BR>\n",
			      number, pointers->low_date, pointers->high_date);
		    else
		      fprintf(html, "<CENTER><U><H1>Domain Statistics for <I>"
			      "%s</I> to <I>%s</I> (By bytes)</H1></U><BR>\n",
			      pointers->low_date, pointers->high_date);
		  }
		fprintf(html, "<TABLE BORDER=2><TR><TH></TH><TH>Number Of</TH>"
			"<TH>Number of</TH><TH>Average</TH><TH>Percent Of</TH>"
			"<TH>Percent Of</TH></TR>\n");
		fprintf(html, "<TR><TH>Domain</TH><TH>Files Sent</TH><TH>"
			"Bytes Sent</TH><TH>Xmit Rate</TH><TH>Files Sent</TH>"
			"<TH>Bytes Sent</TH></TR>\n");
		break;
	      }	      
	  }
	else
	  {
	    switch(type)
	      {
	      case T_DIR:
		if (count == 1)
		  {
		    if (number)
		      printf("%c\nTop %d Total Transfers from each Archive "
			     "Section (By bytes)\n\n", 12, number);
		    else
		      printf("%c\nTotal Transfers from each Archive Section "
			     "(By bytes)\n\n", 12);
		  }
		printf("                                                                ---- %% Of ----\n");
		printf("                       Archive Section                    Files  Files   Bytes\n");
		break;
	      case T_FILE:
		if (count == 1)
		  {
		    if (number)
		      printf("%c\nTop %d Most Transferred Files (By transfer "
			     "count)\n\n", 12, number);
		    else
		      printf("%c\nAll Transferred Files (By transfer count)\n"
			     "\n", 12);
		  }
		printf("                                                                ---- %% Of ----\n");
		printf("                           Filename                       Files  Files   Bytes\n");
		printf(" -------------------------------------------------------- ----- ------- -------\n");
		break;
	      case T_TLD:
		if (count == 1)
		  {
		    if (number)
		      printf("%c\nTop %d Top Level Domains (By bytes)\n\n",
			     12, number);
		    else
		      printf("%c\nTop Level Domain Statistics (By bytes)\n\n",
			     12);
		  }
		printf("                 Number Of    Number of    Average   Percent Of  Percent Of\n");
		printf("    Domain       Files Sent  Bytes  Sent  Xmit Rate  Files Sent  Bytes Sent\n");
		printf("---------------  ----------  -----------  ---------  ----------  ----------\n");
		break;
	      case T_HOST:
		if (count == 1)
		  {
		    if (number)
		      printf("%c\nTop %d Hosts (By bytes)\n\n", 12, number);
		    else
		      printf("%c\nHost Statistics (By bytes)\n\n", 12);
		  }
		printf("                                   Number Of   Number of  Percent Of Percent Of\n");
		printf("               Host                Files Sent Bytes  Sent Files Sent Bytes Sent\n");
		printf("---------------------------------- ---------- ----------- ---------- ----------\n");
		break;
	      case T_DOMAIN:
		if (count == 1)
		  {
		    if (number)
		      printf("%c\nTop %d Domains (By bytes)\n\n", 12, number);
		    else
		      printf("%c\nDomain Statistics (By bytes)\n\n", 12);
		  }
		printf("                                   Number Of   Number of  Percent Of Percent Of\n");
		printf("              Domain               Files Sent Bytes  Sent Files Sent Bytes Sent\n");
		printf("---------------------------------- ---------- ----------- ---------- ----------\n");
		break;
	      }
	  }
      
      if (pointers->config->html_output)
	{
	  fprintf(html, "<TR><TD>%s</TD><TD ALIGN=RIGHT>%u</TD>"
		  "<TD ALIGN=RIGHT>%s</TD><TD NOWRAP ALIGN=RIGHT>%5.1f KB/s"
		  "</TD><TD ALIGN=RIGHT>%10.2f</TD><TD ALIGN=RIGHT>%10.2f"
		  "</TD>", current_path_ptr->name,
		  current_path_ptr->file_count,
		  big_number(current_path_ptr->data, &buffer, &size),
		  (double) current_path_ptr->data / current_path_ptr->seconds
		  / 1024, (double) current_path_ptr->file_count /
		  pointers->file_count * 100, (double) current_path_ptr->data /
		  pointers->data * 100);
	  switch(type)
	    {
	    case T_DIR:
	    case T_TLD:
	    case T_HOST:
	    case T_DOMAIN:
	      percent = (double) current_path_ptr->data / highest_count * 50;
	      break;
	    case T_FILE:
	      percent =
		(double) current_path_ptr->file_count / highest_count * 50;
	      break;
	    }
	  fprintf(html, "<TD NOWRAP ALIGN=LEFT>");
	  html_graph(percent, html, pointers->config->graph_path);
	  fprintf(html, "</TD></TR>\n");
	}
      else
	{
	  switch(type)
	    {
	    case T_DIR:
	    case T_FILE:
	      if ((a = strlen(current_path_ptr->name)) > 54)
		{
		  strcpy(temp_name, current_path_ptr->name + a - 54);
		  printf("...%-53s %5u %7.2f %7.2f\n", temp_name,
			 current_path_ptr->file_count,
			 (double) current_path_ptr->file_count /
			 pointers->file_count * 100,
			 (double) current_path_ptr->data /
			 pointers->data * 100);
		}
	      else
		printf(" %-56s %5u %7.2f %7.2f\n", current_path_ptr->name,
		       current_path_ptr->file_count,
		       (double) current_path_ptr->file_count /
		       pointers->file_count * 100,
		       (double) current_path_ptr->data / pointers->data * 100);
	      break;
	    case T_TLD:
#ifdef BROKEN_LL
	      printf("%-15s  %10u  %11lu  %5.1f K/s  %10.2f  %10.2f\n",
#else
#  ifdef IS_DEC
	      printf("%-15s  %10u  %11Lu  %5.1f K/s  %10.2f  %10.2f\n",
#  else
	      printf("%-15s  %10u  %11llu  %5.1f K/s  %10.2f  %10.2f\n",
#  endif /* IS_DEC */
#endif /* BROKEN_LL */
		     current_path_ptr->name, current_path_ptr->file_count,
		     current_path_ptr->data, (double) current_path_ptr->data /
		     current_path_ptr->seconds / 1024,
		     (double) current_path_ptr->file_count /
		     pointers->file_count * 100,
		     (double) current_path_ptr->data / pointers->data * 100);
	      break;
	    case T_HOST:
	    case T_DOMAIN:
	      strncpy(temp_name, current_path_ptr->name, 31);
	      temp_name[31] = '\0';
#ifdef BROKEN_LL
	      printf("%-34s %10u %11lu %10.2f %10.2f\n", temp_name,
#else
#  ifdef IS_DEC
	      printf("%-34s %10u %11Lu %10.2f %10.2f\n", temp_name,
#  else
	      printf("%-34s %10u %11llu %10.2f %10.2f\n", temp_name,
#  endif /* IS_DEC */
#endif /* BROKEN_LL */
		     current_path_ptr->file_count, current_path_ptr->data,
		     (double) current_path_ptr->file_count /
		     pointers->file_count * 100,
		     (double) current_path_ptr->data / pointers->data * 100);
	    }
	}
    }
  
  if (pointers->config->html_output)
    {
      if (pointers->config->graph_path)
	fprintf(html, "</TABLE><BR><IMG SRC=\"%sg1.jpg\" ALT=\"g1\"> = %s %s"
		"<BR>\n", pointers->config->graph_path,
		big_number((long) ((double) highest_count / 50), &buffer,
			   &size), (type == T_FILE) ? "transfers" : "bytes");
      else
	fprintf(html, "</TABLE><BR><IMG SRC=\"g1.jpg\" ALT=\"g1\"> = %s %s<BR>"
		"\n", big_number((long) ((double) highest_count / 50),
				     &buffer, &size),
		(type == T_FILE) ? "transfers" : "bytes");
      if (!pointers->config->single_page)
	{
	  fprintf(html, "<HR><A HREF=\"totals.html\">Return to Totals and "
		  "Index </A></CENTER>\n");
	  if (!pointers->config->no_html_headers)
	    fprintf(html, "</BODY></HTML>\n");
	}
      fclose(html);
    }
  
#ifndef SMFS
#ifndef PTHREADS
#ifdef ENABLE_ARENAS
  if (type != T_FILE || !pointers->config->dir_traffic)
    {
      for (current_arena_ptr = *arena;
	   current_arena_ptr != NULL;)
	{
	  next_arena_ptr = current_arena_ptr->next_ptr;
	  free(current_arena_ptr);
	  current_arena_ptr = next_arena_ptr;
	}
      *pointers_ptr = NULL;
      *arena = NULL;
    }

  if (type == T_DIR && pointers->file_arena != NULL)
    {
      for (current_arena_ptr = pointers->file_arena;
	   current_arena_ptr != NULL;)
	{
	  next_arena_ptr = current_arena_ptr->next_ptr;
	  free(current_arena_ptr);
	  current_arena_ptr = next_arena_ptr;
	}
      pointers->first_file_ptr = NULL;
      pointers->file_arena = NULL;
    }
  if (type == T_DOMAIN && pointers->host_arena != NULL)
    {
      for (current_arena_ptr = pointers->host_arena;
	   current_arena_ptr != NULL;)
	{
	  next_arena_ptr = current_arena_ptr->next_ptr;
	  free(current_arena_ptr);
	  current_arena_ptr = next_arena_ptr;
	}
      pointers->first_host_ptr = NULL;
      pointers->host_arena = NULL;
    }
#endif /* ENABLE_ARENAS */
#endif /* ifndef PTHREADS */
#endif /* ifndef SMFS */

#ifdef DEBUGS
  fprintf(stderr, "display_%s_totals complete\n", T_NAMES[type]);
#endif
#ifdef PTHREADS
  pthread_mutex_lock(pointers->active_threads_mutex);
  if (--pointers->active_threads < MAX_ACTIVE_THREADS)
    pthread_cond_signal(pointers->active_threshold_cond);
  pthread_mutex_unlock(pointers->active_threads_mutex);
  pthread_mutex_unlock(pointers->active_display_thread_mutex);
  pthread_cond_signal(pointers->data_gen_finished_cond);
#endif
} /* display_misc_totals */


void
display_file_totals(pointers_t *pointers)
{
  display_misc_totals(pointers, T_FILE);
} /* display_file_totals */


void
display_dir_totals(pointers_t *pointers)
{
  display_misc_totals(pointers, T_DIR);
} /* display_dir_totals */


void
display_tld_totals(pointers_t *pointers)
{
  display_misc_totals(pointers, T_TLD);
} /* display_tld_totals */


void
display_host_totals(pointers_t *pointers)
{
  display_misc_totals(pointers, T_HOST);
} /* display_host_totals */


void
display_domain_totals(pointers_t *pointers)
{
  display_misc_totals(pointers, T_DOMAIN);
} /* display_domain_totals */


void
display_hourly_totals(pointers_t * pointers)
{
  int i, size = 0;
  double percent;
  char *buffer = NULL;
  FILE *html = NULL;

#ifdef DEBUGS
  fprintf(stderr, "Beginning display_hourly_totals\n");
#endif

  if (pointers->config->html_output)
    {
      if (!pointers->config->single_page)
	{
	  if (!(html = fopen("hour.html", "w")))
	    {
	      perror("fopen");
	      exit(1);
	    }
	  if (!pointers->config->no_html_headers)
	    fprintf(html, "<HTML><TITLE>xferstats: %s to %s</TITLE><BODY>\n",
		    pointers->low_date, pointers->high_date);
	}
      else
	{
	  if (!(html = fopen("xferstats.html", "a")))
	    {
	      perror("fopen");
	      exit(1);
	    }
	  fprintf(html, "<HR>");
	}

      fprintf(html, "<CENTER><H1><U>Hourly Transmission Statistics for <I>%s"
	      "</I> to <I>%s</I></U></H1><BR><TABLE BORDER=2>\n",
	      pointers->low_date, pointers->high_date);
      fprintf(html, "<TR><TH></TH><TH>Number Of</TH><TH>Number of</TH><TH>"
	      "Average</TH><TH>Percent Of</TH><TH>Percent Of</TH></TR>\n");
      fprintf(html, "<TR><TH>Time</TH><TH>Files Sent</TH><TH>Bytes Sent</TH>"
	      "<TH>Xmit Rate</TH><TH>Files Sent</TH><TH>Bytes Sent</TH></TR>"
	      "\n");
      
      for (i = 0; i <= 23; i++)
	if (pointers->hour_data[i].file_count)
	  {
	    fprintf(html, "<TR><TD>%d</TD><TD ALIGN=RIGHT>%u</TD>"
		    "<TD ALIGN=RIGHT>%s</TD><TD NOWRAP ALIGN=RIGHT>%6.2f KB/s"
		    "</TD><TD ALIGN=RIGHT>%10.2f</TD><TD ALIGN=RIGHT>%10.2f"
		    "</TD>", i, pointers->hour_data[i].file_count,
		    big_number(pointers->hour_data[i].data, &buffer, &size),
		    (double) pointers->hour_data[i].data /
		    pointers->hour_data[i].seconds / 1024,
		    (double) pointers->hour_data[i].file_count /
		    pointers->file_count * 100,
		    (percent = (double) pointers->hour_data[i].data /
		     pointers->data * 100));
	    fprintf(html, "<TD NOWRAP ALIGN=LEFT>");
	    html_graph(percent, html, pointers->config->graph_path);
	    fprintf(html, "</TD></TR>\n");
	  }
      if (pointers->config->graph_path)
	fprintf(html, "</TABLE><BR>\n<IMG SRC=\"%sg1.jpg\" ALT=\"g1\"> = %s "
		"bytes<BR>\n", pointers->config->graph_path,
		big_number((double) pointers->data / 50, &buffer, &size));
      else
	fprintf(html, "</TABLE><BR>\n<IMG SRC=\"g1.jpg\" ALT=\"g1\"> = %s "
		"bytes<BR>\n",
		big_number((double) pointers->data / 50, &buffer, &size));
      if (!pointers->config->single_page)
	{
	  fprintf(html, "<HR><A HREF=\"totals.html\">Return to Totals and "
		  "Index </A></CENTER>\n");
	  if (!pointers->config->no_html_headers)
	    fprintf(html, "</BODY></HTML>\n");
	}
      fclose(html);
    }
  else
    {
      printf("%c\nHourly Transmission Statistics\n\n", 12);
      printf("      Number Of    Number of     Average    Percent Of  Percent Of\n");
      printf("Time  Files Sent  Bytes  Sent  Xmit   Rate  Files Sent  Bytes Sent\n");
      printf("----  ----------  -----------  -----------  ----------  ----------\n");
      for (i = 0; i <= 23; i++)
	if (pointers->hour_data[i].file_count)
#ifdef BROKEN_LL
	  printf("%2d    %10u  %11lu  %6.2f KB/s  %10.2f  %10.2f\n",
#else
#  ifdef IS_DEC
	  printf("%2d    %10u  %11Lu  %6.2f KB/s  %10.2f  %10.2f\n",
#  else
	  printf("%2d    %10u  %11llu  %6.2f KB/s  %10.2f  %10.2f\n",
#  endif /* IS_DEC */
#endif /* BROKEN_LL */
		 i, pointers->hour_data[i].file_count,
		 pointers->hour_data[i].data,
		 (double) pointers->hour_data[i].data /
		 pointers->hour_data[i].seconds / 1024,
		 (double) pointers->hour_data[i].file_count /
		 pointers->file_count * 100,
		 (double) pointers->hour_data[i].data / pointers->data * 100);
    }
#ifdef DEBUGS
  fprintf(stderr, "display_hourly_totals complete\n");
#endif
#ifdef PTHREADS
  pthread_mutex_lock(pointers->active_threads_mutex);
  if (--pointers->active_threads < MAX_ACTIVE_THREADS)
    pthread_cond_signal(pointers->active_threshold_cond);
  pthread_mutex_unlock(pointers->active_threads_mutex);
  pthread_mutex_unlock(pointers->active_display_thread_mutex);
  pthread_cond_signal(pointers->data_gen_finished_cond);
#endif
} /* display_hourly_totals */


void
display_monthly_totals(pointers_t * pointers)
{
  int i, size = 0;
  double percent;
  char *buffer = NULL;
  FILE *html = NULL;

#ifdef DEBUGS
  fprintf(stderr, "Beginning display_monthly_totals\n");
#endif

  if (pointers->config->html_output)
    {
      if (!pointers->config->single_page)
	{
	  if (!(html = fopen("monthly.html", "w")))
	    {
	      perror("fopen");
	      exit(1);
	    }
	  if (!pointers->config->no_html_headers)
	    fprintf(html, "<HTML><TITLE>xferstats: %s to %s</TITLE><BODY>\n",
		    pointers->low_date, pointers->high_date);
	}
      else
	{
	  if (!(html = fopen("xferstats.html", "a")))
	    {
	      perror("fopen");
	      exit(1);
	    }
	  fprintf(html, "<HR>");
	}

      fprintf(html, "<CENTER><H1><U>Monthly Transmission Statistics for <I>%s"
	      "</I> to <I>%s</I></U></H1><BR><TABLE BORDER=2>\n",
	      pointers->low_date, pointers->high_date);
      fprintf(html, "<TR><TH></TH><TH>Number Of</TH><TH>Number of</TH><TH>"
	      "Average</TH><TH>Percent Of</TH><TH>Percent Of</TH></TR>\n");
      fprintf(html, "<TR><TH>Month</TH><TH>Files Sent</TH><TH>Bytes Sent</TH>"
	      "<TH>Xmit Rate</TH><TH>Files Sent</TH><TH>Bytes Sent</TH></TR>"
	      "\n");
      
      for (i = 0; i <= 11; i++)
	if (pointers->month_data[i].file_count)
	  {
	    fprintf(html, "<TR><TD>%s</TD><TD ALIGN=RIGHT>%u</TD>"
		    "<TD ALIGN=RIGHT>%s</TD><TD NOWRAP ALIGN=RIGHT>%6.2f KB/s"
		    "</TD><TD ALIGN=RIGHT>%10.2f</TD><TD ALIGN=RIGHT>%10.2f"
		    "</TD>", LONG_MONTHS[i],
		    pointers->month_data[i].file_count,
		    big_number(pointers->month_data[i].data, &buffer, &size),
		    (double) pointers->month_data[i].data /
		    pointers->month_data[i].seconds / 1024,
		    (double) pointers->month_data[i].file_count /
		    pointers->file_count * 100,
		    (percent = (double) pointers->month_data[i].data /
		     pointers->data * 100));
	    fprintf(html, "<TD NOWRAP ALIGN=LEFT>");
	    html_graph(percent, html, pointers->config->graph_path);
	    fprintf(html, "</TD></TR>\n");
	  }
      if (pointers->config->graph_path)
	fprintf(html, "</TABLE><BR>\n<IMG SRC=\"%sg1.jpg\" ALT=\"g1\"> = %s "
		"bytes<BR>\n", pointers->config->graph_path,
		big_number((double) pointers->data / 50, &buffer, &size));
      else
	fprintf(html, "</TABLE><BR>\n<IMG SRC=\"g1.jpg\" ALT=\"g1\"> = %s "
		"bytes<BR>\n",
		big_number((double) pointers->data / 50, &buffer, &size));
      if (!pointers->config->single_page)
	{
	  fprintf(html, "<HR><A HREF=\"totals.html\">Return to Totals and "
		  "Index </A></CENTER>\n");
	  if (!pointers->config->no_html_headers)
	    fprintf(html, "</BODY></HTML>\n");
	}
      fclose(html);
    }
  else
    {
      printf("%c\nMonthly Transmission Statistics\n\n", 12);
      printf("           Number Of    Number of     Average    Percent Of  Percent Of\n");
      printf("  Month    Files Sent  Bytes  Sent  Xmit   Rate  Files Sent  Bytes Sent\n");
      printf("---------  ----------  -----------  -----------  ----------  ----------\n");
      for (i = 0; i <= 11; i++)
	if (pointers->month_data[i].file_count)
#ifdef BROKEN_LL
	  printf("%-9s  %10u  %11lu  %6.2f KB/s  %10.2f  %10.2f\n",
#else
#  ifdef IS_DEC
	  printf("%-9s  %10u  %11Lu  %6.2f KB/s  %10.2f  %10.2f\n",
#  else
	  printf("%-9s  %10u  %11llu  %6.2f KB/s  %10.2f  %10.2f\n",
#  endif /* IS_DEC */
#endif /* BROKEN_LL */
		 LONG_MONTHS[i], pointers->month_data[i].file_count,
		 pointers->month_data[i].data,
		 (double) pointers->month_data[i].data /
		 pointers->month_data[i].seconds / 1024,
		 (double) pointers->month_data[i].file_count /
		 pointers->file_count * 100,
		 (double) pointers->month_data[i].data / pointers->data * 100);
    }
#ifdef DEBUGS
  fprintf(stderr, "display_monthly_totals complete\n");
#endif
#ifdef PTHREADS
  pthread_mutex_lock(pointers->active_threads_mutex);
  if (--pointers->active_threads < MAX_ACTIVE_THREADS)
    pthread_cond_signal(pointers->active_threshold_cond);
  pthread_mutex_unlock(pointers->active_threads_mutex);
  pthread_mutex_unlock(pointers->active_display_thread_mutex);
  pthread_cond_signal(pointers->data_gen_finished_cond);
#endif
} /* display_monthly_totals */


void
display_dow_totals(pointers_t * pointers)
{
  int i, size = 0;
  double percent;
  char *buffer = NULL;
  FILE *html = NULL;

#ifdef DEBUGS
  fprintf(stderr, "display_dow_totals beginning\n");
#endif

  if (pointers->config->html_output)
    {
      if (!pointers->config->single_page)
	{
	  if (!(html = fopen("dow.html", "w")))
	    {
	      perror("fopen");
	      exit(1);
	    }
	  if (!pointers->config->no_html_headers)
	    fprintf(html, "<HTML><TITLE>xferstats: %s to %s</TITLE><BODY>\n",
		    pointers->low_date, pointers->high_date);
	}
      else
	{
	  if (!(html = fopen("xferstats.html", "a")))
	    {
	      perror("fopen");
	      exit(1);
	    }
	  fprintf(html, "<HR>");
	}

      fprintf(html, "<H1><U>Days of the Week Transmission Statistics for <I>%s"
	      "</I> to <I>%s</I></U></H1><BR><TABLE BORDER=2>\n",
	      pointers->low_date, pointers->high_date);
      fprintf(html, "<TR><TH></TH><TH>Number Of</TH><TH>Number of</TH><TH>"
	      "Average</TH><TH>Percent Of</TH><TH>Percent Of</TH></TR>\n");
      fprintf(html, "<TR><TH>Day</TH><TH>Files Sent</TH><TH>Bytes Sent</TH>"
	      "<TH>Xmit Rate</TH><TH>Files Sent</TH><TH>Bytes Sent</TH></TR>"
	      "\n");
      
      for (i = 0; i <= 6; i++)
	if (pointers->weekday_data[i].file_count)
	  {
	    fprintf(html, "<TR><TD>%s</TD><TD ALIGN=RIGHT>%u</TD>"
		    "<TD ALIGN=RIGHT>%s</TD><TD NOWRAP ALIGN=RIGHT>%6.2f KB/s"
		    "</TD><TD ALIGN=RIGHT>%10.2f</TD><TD ALIGN=RIGHT>%10.2f"
		    "</TD>", LONG_DAYS[i],
		    pointers->weekday_data[i].file_count,
		    big_number(pointers->weekday_data[i].data, &buffer, &size),
		    (double) pointers->weekday_data[i].data /
		    pointers->weekday_data[i].seconds / 1024,
		    (double) pointers->weekday_data[i].file_count /
		    pointers->file_count * 100,
		    percent = ((double) pointers->weekday_data[i].data /
			       pointers->data * 100));
	    fprintf(html, "<TD NOWRAP ALIGN=LEFT>");
	    html_graph(percent, html, pointers->config->graph_path);
	    fprintf(html, "</TD></TR>\n");
	  }
      if (pointers->config->graph_path)
	fprintf(html, "</TABLE><BR>\n<IMG SRC=\"%sg1.jpg\" ALT=\"g1\"> = %s "
		"bytes<BR>\n", pointers->config->graph_path,
		big_number((double) pointers->data / 50, &buffer, &size));
      else
	fprintf(html, "</TABLE><BR>\n<IMG SRC=\"g1.jpg\" ALT=\"g1\"> = %s "
		"bytes<BR>\n",
		big_number((double) pointers->data / 50, &buffer, &size));
      if (!pointers->config->single_page)
	{
	  fprintf(html, "<HR><A HREF=\"totals.html\">Return to Totals and "
		  "Index </A></CENTER>\n");
	  if (!pointers->config->no_html_headers)
	    fprintf(html, "</BODY></HTML>\n");
	}
      fclose(html);
    }
  else
    {
      printf("%c\nDays of the Week Transmission Statistics\n\n", 12);
      printf("           Number Of    Number of     Average    Percent Of  Percent Of\n");
      printf("   Day     Files Sent  Bytes  Sent  Xmit   Rate  Files Sent  Bytes Sent\n");
      printf("---------  ----------  -----------  -----------  ----------  ----------\n");
      for (i = 0; i <= 6; i++)
	if (pointers->weekday_data[i].file_count)
#ifdef BROKEN_LL
	  printf("%-9s  %10u  %11lu  %6.2f KB/s  %10.2f  %10.2f\n",
#else
#  ifdef IS_DEC
	  printf("%-9s  %10u  %11Lu  %6.2f KB/s  %10.2f  %10.2f\n",
#  else
	  printf("%-9s  %10u  %11llu  %6.2f KB/s  %10.2f  %10.2f\n",
#  endif /* IS_DEC */
#endif /* BROKEN_LL */
		 LONG_DAYS[i], pointers->weekday_data[i].file_count,
		 pointers->weekday_data[i].data,
		 (double) pointers->weekday_data[i].data /
		 pointers->weekday_data[i].seconds / 1024,
		 (double) pointers->weekday_data[i].file_count /
		 pointers->file_count * 100,
		 (double) pointers->weekday_data[i].data /
		 pointers->data * 100);
    }
#ifdef DEBUGS
  fprintf(stderr, "display_dow_totals complete\n");
#endif
#ifdef PTHREADS
  pthread_mutex_lock(pointers->active_threads_mutex);
  if (--pointers->active_threads < MAX_ACTIVE_THREADS)
    pthread_cond_signal(pointers->active_threshold_cond);
  pthread_mutex_unlock(pointers->active_threads_mutex);
  pthread_mutex_unlock(pointers->active_display_thread_mutex);
  pthread_cond_signal(pointers->data_gen_finished_cond);
#endif
} /* display_dow_totals */


void
display_dom_totals(pointers_t * pointers)
{
  int i, size = 0;
  double percent;
  char *buffer = NULL;
  FILE *html = NULL;

#ifdef DEBUGS
  fprintf(stderr, "display_dom_totals beginning\n");
#endif

  if (pointers->config->html_output)
    {
      if (!pointers->config->single_page)
	{
	  if (!(html = fopen("dom.html", "w")))
	    {
	      perror("fopen");
	      exit(1);
	    }
	  if (!pointers->config->no_html_headers)
	    fprintf(html, "<HTML><TITLE>xferstats: %s to %s</TITLE><BODY>\n",
		    pointers->low_date, pointers->high_date);
	}
      else
	{
	  if (!(html = fopen("xferstats.html", "a")))
	    {
	      perror("fopen");
	      exit(1);
	    }
	  fprintf(html, "<HR>");
	}

      fprintf(html, "<H1><U>Days of the Month Transmission Statistics for <I>"
	      "%s</I> to <I>%s</I></U></H1><BR><TABLE BORDER=2>\n",
	      pointers->low_date, pointers->high_date);
      fprintf(html, "<TR><TH></TH><TH>Number Of</TH><TH>Number of</TH><TH>"
	      "Average</TH><TH>Percent Of</TH><TH>Percent Of</TH></TR>\n");
      fprintf(html, "<TR><TH>Day</TH><TH>Files Sent</TH><TH>Bytes Sent</TH>"
	      "<TH>Xmit Rate</TH><TH>Files Sent</TH><TH>Bytes Sent</TH>"
	      "</TR>\n");
      
      for (i = 0; i <= 30; i++)
	if (pointers->dom_data[i].file_count)
	  {
	    fprintf(html, "<TR><TD>%d</TD><TD ALIGN=RIGHT>%u</TD>"
		    "<TD ALIGN=RIGHT>%s</TD><TD NOWRAP ALIGN=RIGHT>%6.2f KB/s"
		    "</TD><TD ALIGN=RIGHT>%10.2f</TD><TD ALIGN=RIGHT>%10.2f"
		    "</TD>", i + 1, pointers->dom_data[i].file_count,
		    big_number(pointers->dom_data[i].data, &buffer, &size),
		    (double) pointers->dom_data[i].data /
		    pointers->dom_data[i].seconds / 1024,
		    (double) pointers->dom_data[i].file_count /
		    pointers->file_count * 100,
		    (percent = (double) pointers->dom_data[i].data /
		     pointers->data * 100));
	    fprintf(html, "<TD NOWRAP ALIGN=LEFT>");
	    html_graph(percent, html, pointers->config->graph_path);
	    fprintf(html, "</TD></TR>\n");
	  }
      if (pointers->config->graph_path)
	fprintf(html, "</TABLE><BR>\n<IMG SRC=\"%sg1.jpg\" ALT=\"g1\"> = %s "
		"bytes<BR>\n", pointers->config->graph_path,
		big_number((double) pointers->data / 50, &buffer, &size));
      else
	fprintf(html, "</TABLE><BR>\n<IMG SRC=\"g1.jpg\" ALT=\"g1\"> = %s "
		"bytes<BR>\n",
		big_number((double) pointers->data / 50, &buffer, &size));
      if (!pointers->config->single_page)
	{
	  fprintf(html, "<HR><A HREF=\"totals.html\">Return to Totals and "
		  "Index </A></CENTER>\n");
	  if (!pointers->config->no_html_headers)
	    fprintf(html, "</BODY></HTML>\n");
	}
      fclose(html);
    }
  else
    {
      printf("%c\nDays of the Month Transmission Statistics\n\n", 12);
      printf("     Number Of    Number of     Average    Percent Of  Percent Of\n");
      printf("Day  Files Sent  Bytes  Sent  Xmit   Rate  Files Sent  Bytes Sent\n");
      printf("---  ----------  -----------  -----------  ----------  ----------\n");
      for (i = 0; i <= 30; i++)
	if (pointers->dom_data[i].file_count)
#ifdef BROKEN_LL
	    printf("%2d   %10u  %11lu  %6.2f KB/s  %10.2f  %10.2f\n",
#else
#  ifdef IS_DEC
	    printf("%2d   %10u  %11Lu  %6.2f KB/s  %10.2f  %10.2f\n",
#  else
	    printf("%2d   %10u  %11llu  %6.2f KB/s  %10.2f  %10.2f\n",
#  endif /* IS_DEC */
#endif /* BROKEN_LL */
		   i + 1, pointers->dom_data[i].file_count,
		   pointers->dom_data[i].data,
		   (double) pointers->dom_data[i].data /
		   pointers->dom_data[i].seconds / 1024,
		   (double) pointers->dom_data[i].file_count /
		   pointers->file_count * 100,
		   (double) pointers->dom_data[i].data / pointers->data * 100);
    }
#ifdef DEBUGS
  fprintf(stderr, "display_dom_totals complete\n");
#endif
#ifdef PTHREADS
  pthread_mutex_lock(pointers->active_threads_mutex);
  if (--pointers->active_threads < MAX_ACTIVE_THREADS)
    pthread_cond_signal(pointers->active_threshold_cond);
  pthread_mutex_unlock(pointers->active_threads_mutex);
  pthread_mutex_unlock(pointers->active_display_thread_mutex);
  pthread_cond_signal(pointers->data_gen_finished_cond);
#endif
} /* display_dom_totals */
