/*
 * Copyright 1998 Nat Friedman, Massachusetts Institute of Technology
 * <ndf@mit.edu>
 *
 * Small optimizations, modifications, and additions to adapt hashstr.c to
 * xferstats source are Copyright 1998 Phil Schwan <pschwan@cmu.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#else
#  error You did not run configure, did you?
#endif

#ifdef HAVE_STRING_H
#  include <string.h> /* needed everywhere */
#else
#  error The <string.h> header file is required to compile xferstats
#endif

#include <stdio.h>
#include <stdlib.h>
#include "xferstats.h"
#include "hashstr.h"

/* A set of hash table routines to store and retrieve strings with
   associated values.  ndf@mit.edu */

htable *
create_hash_table(size_t size)
{
  htable * ht;

  if ((ht = (htable *) malloc(sizeof(htable))) == NULL)
    {
      fprintf(stderr, "Could not allocate space for hash table: malloc(%ld)\n",
	      (long) sizeof(htable));
      perror("malloc");
      exit(1);
    }

  if ((ht->bucket = (hbucket **) malloc(sizeof(hbucket *)*size)) == NULL)
    {
      fprintf(stderr, "Could not allocate space for hash table buckets: "
	      "malloc(%ld)\n", (long) sizeof(hbucket) * size);
      perror("malloc");
      exit(1);
    }

  ht->size = size;

  for (size = 0; size < ht->size ; size++)
    ht->bucket[size] = (hbucket *)NULL;

  return ht;
} /* create_hash_table */


/* The algorithm used in this function was learned from  _Compilers:
   Principles, Techniques, and Tools_, by Aho, Sethi, and Ullman. */
unsigned int
hash_str(char * str)
{
  char * p;
  unsigned int g, h=0;

  p = str;
  while (*p != '\0')
    {
      h = (h<<4) + *p++;
      if ((g = h&0xf0000000))
   	h = (h ^ (g>>24)) ^ g;
    }
  return h;
} /* hash_str */


void
add_hash_string(htable * ht, char * str, void * value)
{
  unsigned int bindex;
  hbucket * curr;

  bindex = hash_str(str) % ht->size;

  curr = ht->bucket[bindex];
  while (curr != NULL)
    {
      if (!strcmp(str, curr->str))
	{
	  curr->value = value;
	  return;
	}
      curr = curr->next;
    }

  if ((curr = (hbucket *) malloc(sizeof(hbucket))) == NULL)
    {
      fprintf(stderr, "Could not allocate space for new bucket: malloc(%ld)\n",
	      (long) sizeof(hbucket));
      perror("malloc");
      exit(1);
    }

  if ((curr->str = (char *)malloc(strlen(str)+1)) == NULL)
    {
      fprintf(stderr, "Could not allocate space for key in bucket: "
	      "malloc(%ld)\n", (long) strlen(str) + 1);
      perror("malloc");
      exit(1);
    }
  strcpy(curr->str, str);

  curr->value = value;

  if (ht->bucket[bindex] == NULL)
    {
      ht->bucket[bindex] = curr;
      curr->next = NULL;
    }
  else
    {
      curr->next = ht->bucket[bindex];
      ht->bucket[bindex] = curr;
    }
} /* add_hash_string */


unsigned char
get_value(htable * ht, char * str, void ** value)
{
  unsigned int bindex;
  hbucket * curr;

  bindex = hash_str(str) % ht->size;

  curr = ht->bucket[bindex];
  while (curr != NULL)
    {
      if (!strcmp(str, curr->str))
	{
	  *value = curr->value;
	  return 1;
	}
      curr = curr->next; 
    }
  return 0;
} /* get_value */


size_t
num_entries(htable * ht)
{
  hbucket * curr;
  size_t num = 0, i;

  for (i=0;i<ht->size;i++)
    {
      curr = ht->bucket[i];
      while (curr != NULL)
	{
	  curr = curr->next;
	  num++;
	}
    }

  return num;
} /* num_entries */


/* This is slow but convenient */
int
element_at_index(htable * ht, size_t idx, char ** str, void ** v)
{
  hbucket * curr;
  size_t i, bnum;

  
  bnum = 0;
  i = 0;
  curr = ht->bucket[bnum];
  while (1)
    {
      if (curr != NULL)
	{
	  if (i==idx)
	    {
	      *str = curr->str;
	      *v = curr->value;
	      return 1;
	    }
	  i++;
	  curr = curr->next;
	}
      else
	{
	  bnum++;
	  if (bnum >= ht->size)
	    return 0;
	  curr = ht->bucket[bnum];
	}
    }
} /* element_at_index */


xfmisc_t *
make_linked_list(htable * ht)
{
  xfmisc_t * first_ptr = NULL, * current_ptr = NULL;
  hbucket * curr;
  size_t i;

  for (i = 0; i < ht->size; i++)
    {
      curr = ht->bucket[i];
      while (curr != NULL)
	{
	  if (current_ptr == NULL)
	    current_ptr = first_ptr = curr->value;
	  else
	    current_ptr = current_ptr->next_ptr = curr->value;
	  curr = curr->next;
	}
    }
  current_ptr->next_ptr = NULL;

  return first_ptr;
} /* make_linked_list */


void
clean_table(htable * ht)
{
  size_t i;

  for (i = 0; i < ht->size; i++)
    if (ht->bucket[i] != NULL)
      {
	free(ht->bucket[i]->str);
	free(ht->bucket[i]);
      }
} /* clean_table */
