/*
 * Copyright 1997, 1998 Phil Schwan <pschwan@cmu.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#else
#  error You did not run configure, did you?
#endif

#ifdef HAVE_SYS_TYPES_H /* required for FreeBSD to work */
#  include <sys/types.h>
#endif

#ifdef HAVE_STRING_H
#  include <string.h> /* needed everywhere */
#else
#  error The <string.h> header file is required to compile xferstats
#endif

#ifdef HAVE_MMAP
#  include <sys/mman.h>
#else
#  error The mmap/munmap functions (and therefore the <sys/mman.h> include file) is required to compile xferstats
#endif

#ifdef HAVE_FCNTL_H
#  include <fcntl.h>
#else
#  error The <fcntl.h> header file is required to compile xferstats
#endif

#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#else
#  error The <unistd.h> header file is required to compile xferstats
#endif

#include <sys/stat.h>
#include <time.h>

#include <stdio.h>
#include <stdlib.h>
#include "xferstats.h"

const char MONTHS[12][3] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul",
			    "Aug", "Sep", "Oct", "Nov", "Dec"};

int
dir_in_list(list_t * list, char * path)
{
  list_t * clist;

  if (!list)
    return 0;

  for (clist = list; clist; clist = clist->next)
    if (!strncmp(((only_dir_t *)clist->item)->dir, path,
		 ((only_dir_t *)clist->item)->len))
      return 1;

  return 0;
} /* dir_in_list */


void
parse_wuftpd_logfile(pointers_t * pointers)
{
  struct stat stat_buf;
  ftp_entry_t * ftp_line, * current_ftp_ptr = NULL;
  char foo[2048], tmp_host[2048], tmp_path[2048], * map = NULL,
    * map_curr = NULL, * parse_str, user_type, in_out;
  int tmp_size, file_fd = -1;

#ifdef DEBUGS
  fprintf(stderr, "Beginning parse_wuftpd_logfile...(%ld)\n", time(NULL));
#endif

  if (!pointers->config->use_stdin)
    {
      if ((file_fd = open(pointers->config->file_name, O_RDONLY)) < 0)
	{
	  printf("xferstats: fatal: file \"%s\" not found.\n",
		 pointers->config->file_name);
	  exit(1);
	}
      stat(pointers->config->file_name, &stat_buf);
#ifdef _AIX
      /* Only AIX, it seems, wants this mmap typecast to int... */
      if ((int)(map_curr = map =
		(char *)mmap(NULL, stat_buf.st_size, PROT_READ, MAP_SHARED,
			     file_fd, 0)) < 0)
	{
	  perror("mmap");
	  exit(1);
	}
#else
      /* Linux doesn't care, and Solaris segfaults if it's an int.  God how I
       * love Linux... */
      if ((map_curr = map =
	   (char *)mmap(NULL, stat_buf.st_size, PROT_READ, MAP_SHARED,
			file_fd, 0)) < 0)
	{
	  perror("mmap");
	  exit(1);
	}
#endif /* _AIX */
    }

  while (1)
    {
      if (pointers->config->use_stdin)
	{
	  if (feof(stdin))
	    break;
	  /* there's probably a better way to do this :) */
	  fgets(foo, 2047, stdin);
	}
      else
	{
	  if (map_curr - map >= stat_buf.st_size)
	    {
	      munmap(map, stat_buf.st_size);
	      close(file_fd);
	      break;
	    }
	  parse_str = map_curr;
	  if ((map_curr = strchr(map_curr, '\n')) != NULL)
	    {
	      map_curr++;
	      strncpy(foo, parse_str,
		      (map_curr - parse_str) >= 2047 ? (tmp_size = 2047) :
		      (tmp_size = map_curr - parse_str));
	      foo[tmp_size] = '\0';
	    }
	  else
	    {
	      /* there may be some corruption, so we'll check to see if this \0
	       * is really the end of the file */
	      map_curr = strchr(parse_str, '\0');
	      foo[map_curr - parse_str] = '\0';
	      if (map_curr - map < stat_buf.st_size)
		{
		  /* ah-ha, it's not, so we'll keep going */
		  map_curr++;
#ifdef DEBUG
		  fprintf(stderr, "strchr returned null, but it wasn't EOF..."
			  "continuing.\n");
#endif
		}
	      strncpy(foo, parse_str, 2047); /* don't drop that last line! */
	    }
	}

      if (strlen(foo) < 42) continue; /* data is too small to be valid */
#ifdef DEBUG3
      fprintf(stderr, "RAW READ: %s", foo);
#endif
      ftp_line = (ftp_entry_t *)amalloc(&pointers->ftp_line_arena,
					sizeof(ftp_entry_t));

      ftp_line->date[24] = '\0';
      memcpy(ftp_line->date, foo, 24);
      ftp_line->seconds = -1;
      ftp_line->next_ptr = NULL;
      sscanf(foo + 25, "%ld %s %lu %s %*c %*c %c %c", &ftp_line->seconds,
	     tmp_host, &ftp_line->data, tmp_path, &in_out, &user_type);

      if (ftp_line->date[0] == '\0' || tmp_host[0] == '\0' ||
	  tmp_path[0] == '\0' || ftp_line->data == 0 ||
	  ftp_line->seconds < 0 ||
	  (user_type == 'a' && !pointers->config->anon_traffic) ||
	  (user_type == 'g' && !pointers->config->guest_traffic) ||
	  (user_type == 'r' && !pointers->config->real_traffic) ||
	  (in_out == 'i' && !pointers->config->inbound) ||
	  (in_out == 'o' && !pointers->config->outbound) ||
	  (pointers->config->only_dir &&
	   !dir_in_list(pointers->config->only_dir, tmp_path)))
	{
#ifndef SMFS
	  afree(pointers->ftp_line_arena, (__ptr_t) ftp_line);
#endif
	  continue; /* the line is invalid */
	}

      if (!ftp_line->seconds)
	ftp_line->seconds = 1;

      ftp_line->host = amalloc(&pointers->ftp_line_arena,
			       (tmp_size = strlen(tmp_host)) > MAXHOSTNAMELEN ?
			       (tmp_size = MAXHOSTNAMELEN + 1) : ++tmp_size);
      memcpy(ftp_line->host, tmp_host, tmp_size - 1);
      ftp_line->host[tmp_size - 1] = '\0';
      
      ftp_line->path = amalloc(&pointers->ftp_line_arena,
			       (tmp_size = strlen(tmp_path)) > MAXPATHLEN ?
			       (tmp_size = MAXPATHLEN + 1) : ++tmp_size);
      memcpy(ftp_line->path, tmp_path, tmp_size - 1);
      ftp_line->path[tmp_size - 1] = '\0';
      
      if (pointers->first_ftp_line == NULL)
	{
	  pointers->first_ftp_line = current_ftp_ptr = ftp_line;
#ifdef DEBUG3
	  fprintf(stderr, "*** First ftp_line structure created (%p)\n",
		  ftp_line);
#endif
	}
      else
	{
	  current_ftp_ptr->next_ptr = ftp_line;
	  current_ftp_ptr = current_ftp_ptr->next_ptr;
#ifdef DEBUG3
	  fprintf(stderr, "*** New ftp_line structure added (%p)\n", ftp_line);
#endif
	}
#ifdef DEBUG3
      fprintf(stderr, "%s %d %s %lu %s %c %c\n", ftp_line->date,
	      ftp_line->seconds, ftp_line->host, ftp_line->data,
	      ftp_line->path, in_out, user_type);
#endif
    }

  if (!pointers->config->use_stdin)
    munmap(map_curr, stat_buf.st_size);

#ifdef DEBUGS
  fprintf(stderr, "parse_wuftpd_logfile finished. (%ld)\n", time(NULL));
#endif
} /* parse_wuftpd_logfile */


void
parse_ncftpd_logfile(pointers_t * pointers)
{
  struct stat stat_buf;
  ftp_entry_t * ftp_line, * current_ftp_ptr = NULL;
  char foo[2048], * tmp_char, tmp_host[2048], tmp_path[2048], * map = NULL,
    * map_curr = NULL, tmp_date[4], user_name[2048], tmp_str[2048],
    * parse_str;
  int tmp_size, temp, file_fd = -1;

#ifdef DEBUGS
  fprintf(stderr, "Beginning parse_ncftpd_logfile...\n");
#endif

  if (!pointers->config->use_stdin)
    {
      if ((file_fd = open(pointers->config->file_name, O_RDONLY)) < 0)
	{
	  printf("xferstats: fatal: file \"%s\" not found.\n",
		 pointers->config->file_name);
	  exit(1);
	}
      stat(pointers->config->file_name, &stat_buf);
#ifdef _AIX
      /* Only AIX, it seems, wants this mmap typecast to int... */
      if ((int)(map_curr = map =
		(char *)mmap(NULL, stat_buf.st_size, PROT_READ, MAP_SHARED,
			      file_fd, 0)) < 0)
	{
	  perror("mmap");
	  exit(1);
	}
#else
      if ((map_curr = map =
	   (char *)mmap(NULL, stat_buf.st_size, PROT_READ, MAP_SHARED,
			file_fd, 0)) < 0)
	{
	  perror("mmap");
	  exit(1);
	}
#endif /* _AIX */
    }

  while (1)
    {
      if (pointers->config->use_stdin)
	{
	  if (feof(stdin))
	    break;
	  /* there's probably a better way to do this :) */
	  fgets(foo, 2047, stdin);
	}
      else
	{
	  if (map_curr - map >= stat_buf.st_size)
	    {
	      munmap(map, stat_buf.st_size);
	      close(file_fd);
	      break;
	    }
	  parse_str = map_curr;
	  if ((map_curr = strchr(map_curr, '\n')) != NULL)
	    {
	      map_curr++;
	      strncpy(foo, parse_str, (map_curr - parse_str) >= 2047 ?
		      (tmp_size = 2047) : (tmp_size = map_curr - parse_str));
	      foo[tmp_size] = '\0';
	    }
	  else
	    {
	      /* there may be some corruption, so we'll check to see if this \0
	       * is really the end of the file */
	      map_curr = strchr(parse_str, '\0');
	      foo[map_curr - parse_str] = '\0';
	      if (map_curr - map < stat_buf.st_size)
		{
		  /* ah-ha, it's not, so we'll keep going */
		  map_curr++;
#ifdef DEBUG
		  fprintf(stderr, "strchr returned null, but it wasn't EOF..."
			  "continuing.\n");
#endif
		}
	      strncpy(foo, parse_str, 2047); /* don't drop that last line! */
	    }
	}

      if (strlen(foo) < 54) continue; /* data is too small to be valid */
#ifdef DEBUG2
      fprintf(stderr, "RAW READ: %s", foo);
#endif
      /* this is a fairly ugly quick hack, but it seems to work.  I
	 don't plan on replacing this parsing code util a) ncftpd
	 logfiles change or b) someone sends me better code :) */

      if ((foo[30] == 'R' || foo[30] == 'S') && foo[31] == ',')
	{
	  ftp_line = (ftp_entry_t *)amalloc(&pointers->ftp_line_arena,
					    sizeof(ftp_entry_t));
	
	  ftp_line->seconds = -1;
	  ftp_line->next_ptr = NULL;
	  
	  /* This is the new (maybe better, maybe not) way of doing it,
	     with individual byte setting */
	  strcpy(ftp_line->date, "    ");
	  memcpy(tmp_date, foo + 5, 2);
	  tmp_date[2] = '\0';
	  temp = atoi(tmp_date);
	  memcpy(ftp_line->date + 4, MONTHS[temp - 1], 3);
	  ftp_line->date[7] = ' ';
	  memcpy(ftp_line->date + 8, foo + 8, 2);
	  ftp_line->date[10] = ' ';
	  memcpy(ftp_line->date + 11, foo + 11, 8);
	  ftp_line->date[19] = ' ';
	  ftp_line->date[20] = *foo;
	  memcpy(ftp_line->date + 21, foo + 1, 3);
	  continue;

	  tmp_char = foo + 32;
	  while ((tmp_char = strchr(tmp_char, ',')) != NULL)
	      *tmp_char = ' ';

	  /* there's an extra %s in the next line to catch the decimal
	   * portion of the 'seconds' */
	  sscanf(foo + 32, "%s %lu %ld %s %s %s %s %s", tmp_path,
		 &ftp_line->data, &ftp_line->seconds, tmp_str, tmp_str,
		 user_name, tmp_str, tmp_host);

	  if ((foo[30] == 'R' && !pointers->config->outbound) ||
	      (foo[30] != 'R' && !pointers->config->inbound) ||
	      !strcmp(ftp_line->date, "") || ftp_line->seconds < 0 ||
	      (pointers->config->only_dir &&
	       !dir_in_list(pointers->config->only_dir, tmp_path)))
	    {
#ifndef SMFS
	      afree(pointers->ftp_line_arena, (__ptr_t) ftp_line);
#endif
	      continue; /* we don't want this log line */
	    }
	  if (!strcmp(user_name, "anonymous"))
	    {
	      if (!pointers->config->anon_traffic)
		{
#ifndef SMFS
		  afree(pointers->ftp_line_arena, (__ptr_t) ftp_line);
#endif
		  continue; /* we don't want anon traffic */
		}
	    }
	  else
	    {
	      if (!pointers->config->real_traffic)
		{
#ifndef SMFS
		  afree(pointers->ftp_line_arena, (__ptr_t) ftp_line);
#endif
		  continue; /* we don't want anon traffic */
		}
	      /* for a real user, the log line (after comma parsing) looks
	       * like:
	       * /foo 12345 5.123 30.069 pschwan  127.0.0.1  OK
	       * which means that the hostname is in tmp_str instead of
	       * tmp_host so we must use it instead */
	      memcpy(tmp_host, tmp_str, MAXHOSTNAMELEN);
	    }

	  if (!ftp_line->seconds)
	    ftp_line->seconds = 1;

	  ftp_line->host =
	    amalloc(&pointers->ftp_line_arena,
		    (tmp_size = strlen(tmp_host)) > MAXHOSTNAMELEN ?
		    (tmp_size = MAXHOSTNAMELEN + 1) : ++tmp_size);
	  memcpy(ftp_line->host, tmp_host, tmp_size - 1);
	  ftp_line->host[tmp_size - 1] = '\0';

	  ftp_line->path =
	    amalloc(&pointers->ftp_line_arena,
		    (tmp_size = strlen(tmp_path)) > MAXPATHLEN ?
		    (tmp_size = MAXPATHLEN + 1) : ++tmp_size);
	  memcpy(ftp_line->path, tmp_path, tmp_size - 1);
	  ftp_line->path[tmp_size - 1] = '\0';

	  if (pointers->first_ftp_line == NULL)
	    {
	      pointers->first_ftp_line = current_ftp_ptr = ftp_line;
#ifdef DEBUG
	      fprintf(stderr, "*** First ftp_line structure created (%p)\n",
		      ftp_line);
#endif
	    }
	  else
	    {
	      while (current_ftp_ptr->next_ptr != NULL)		
		current_ftp_ptr = current_ftp_ptr->next_ptr;
	      current_ftp_ptr->next_ptr = ftp_line;
#ifdef DEBUG
	      fprintf(stderr, "*** New ftp_line structure added (%p)\n",
		      ftp_line);
#endif
	    }
#ifdef DEBUG2
	  fprintf(stderr, "%s %ld %s %lu %s\n", ftp_line->date,
		  ftp_line->seconds, ftp_line->host, ftp_line->data,
		  ftp_line->path);
#endif
	}
    }

  if (!pointers->config->use_stdin)
    munmap(map_curr, stat_buf.st_size);

#ifdef DEBUGS
  fprintf(stderr, "parse_ncftpd_logfile finished.\n");
#endif
} /* parse_ncftpd_logfile */


void
parse_apache_logfile(pointers_t * pointers)
{
  struct stat stat_buf;
  ftp_entry_t * ftp_line, * current_ftp_ptr = NULL;
  char foo[2048], tmp_host[2048], tmp_path[2048], * map = NULL, * parse_str,
    * map_curr = NULL, tmp_user[2048];
  int tmp_size, file_fd = -1;

#ifdef DEBUGS
  fprintf(stderr, "Beginning parse_apache_logfile...(%ld)\n", time(NULL));
#endif

  if (!pointers->config->use_stdin)
    {
      if ((file_fd = open(pointers->config->file_name, O_RDONLY)) < 0)
	{
	  printf("xferstats: fatal: file \"%s\" not found.\n",
		 pointers->config->file_name);
	  exit(1);
	}
      stat(pointers->config->file_name, &stat_buf);
#ifdef _AIX
      /* Only AIX, it seems, wants this mmap typecast to int... */
      if ((int)(map_curr = map =
		(char *)mmap(NULL, stat_buf.st_size, PROT_READ, MAP_SHARED,
			     file_fd, 0)) < 0)
	{
	  perror("mmap");
	  exit(1);
	}
#else
      /* Linux doesn't care, and Solaris segfaults if it's an int.  God how I
       * love Linux... */
      if ((map_curr = map =
	   (char *)mmap(NULL, stat_buf.st_size, PROT_READ, MAP_SHARED,
			file_fd, 0)) < 0)
	{
	  perror("mmap");
	  exit(1);
	}
#endif /* _AIX */
    }

  while (1)
    {
      if (pointers->config->use_stdin)
	{
	  if (feof(stdin))
	    break;
	  /* there's probably a better way to do this :) */
	  fgets(foo, 2047, stdin);
	}
      else
	{
	  if (map_curr - map >= stat_buf.st_size)
	    {
	      munmap(map, stat_buf.st_size);
	      close(file_fd);
	      break;
	    }
	  parse_str = map_curr;
	  if ((map_curr = strchr(map_curr, '\n')) != NULL)
	    {
	      map_curr++;
	      strncpy(foo, parse_str,
		      (map_curr - parse_str) >= 2047 ? (tmp_size = 2047) :
		      (tmp_size = map_curr - parse_str));
	      foo[tmp_size] = '\0';
	    }
	  else
	    {
	      /* there may be some corruption, so we'll check to see if this \0
	       * is really the end of the file */
	      map_curr = strchr(parse_str, '\0');
	      foo[map_curr - parse_str] = '\0';
	      if (map_curr - map < stat_buf.st_size)
		{
		  /* ah-ha, it's not, so we'll keep going */
		  map_curr++;
#ifdef DEBUG
		  fprintf(stderr, "strchr returned null, but it wasn't EOF..."
			  "continuing.\n");
#endif
		}
	      strncpy(foo, parse_str, 2047); /* don't drop that last line! */
	    }
	}

      if (strlen(foo) < 34) continue; /* data is too small to be valid */
#ifdef DEBUG3
      fprintf(stderr, "RAW READ: %s", foo);
#endif
      ftp_line = (ftp_entry_t *)amalloc(&pointers->ftp_line_arena,
					sizeof(ftp_entry_t));

      ftp_line->date[24] = '\0';
      strncpy(ftp_line->date, foo, 24);

      ftp_line->seconds = -1;
      ftp_line->next_ptr = NULL;

      sscanf(foo + 25, "%ld %s %lu %s %s", &ftp_line->seconds,
	     tmp_host, &ftp_line->data, tmp_path, tmp_user);

      if (ftp_line->date[0] == '\0' || tmp_host[0] == '\0' ||
	  tmp_path[0] == '\0' || ftp_line->data == 0 ||
	  ftp_line->seconds < 0 ||
	  (!strcmp("-", tmp_user) && !pointers->config->anon_traffic) ||
	  (strcmp("-", tmp_user) && !pointers->config->real_traffic) ||
	  !pointers->config->outbound ||
	  (pointers->config->only_dir &&
	   !dir_in_list(pointers->config->only_dir, tmp_path)))
	{
#ifndef SMFS
	  afree(pointers->ftp_line_arena, (__ptr_t) ftp_line);
#endif
	  continue; /* the line is invalid */
	}

      if (!ftp_line->seconds)
	ftp_line->seconds = 1;

      ftp_line->host = amalloc(&pointers->ftp_line_arena,
			       (tmp_size = strlen(tmp_host)) > MAXHOSTNAMELEN ?
			       (tmp_size = MAXHOSTNAMELEN + 1) : ++tmp_size);
      strncpy(ftp_line->host, tmp_host, tmp_size - 1);
      ftp_line->host[tmp_size - 1] = '\0';

      ftp_line->path = amalloc(&pointers->ftp_line_arena,
			       (tmp_size = strlen(tmp_path)) > MAXPATHLEN ?
			       (tmp_size = MAXPATHLEN + 1) : ++tmp_size);
      strncpy(ftp_line->path, tmp_path, tmp_size - 1);
      ftp_line->path[tmp_size - 1] = '\0';
      
      if (pointers->first_ftp_line == NULL)
	{
	  pointers->first_ftp_line = current_ftp_ptr = ftp_line;
#ifdef DEBUG3
	  fprintf(stderr, "*** First ftp_line structure created (%p)\n",
		  ftp_line);
#endif
	}
      else
	{
	  current_ftp_ptr->next_ptr = ftp_line;
	  current_ftp_ptr = current_ftp_ptr->next_ptr;
#ifdef DEBUG3
	  fprintf(stderr, "*** New ftp_line structure added (%p)\n", ftp_line);
#endif
	}
#ifdef DEBUG3
      fprintf(stderr, "%s %d %s %lu %s %s\n", ftp_line->date,
	      ftp_line->seconds, ftp_line->host, ftp_line->data,
	      ftp_line->path, user_name);
#endif
    }

  if (!pointers->config->use_stdin)
    munmap(map_curr, stat_buf.st_size);

#ifdef DEBUGS
  fprintf(stderr, "parse_apache_logfile finished. (%ld)\n", time(NULL));
#endif
} /* parse_apache_logfile */
