#ifndef _XFERSTATS_H
#define _XFERSTATS_H

/*
 * Copyright 1997, 1998 Phil Schwan <pschwan@cmu.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include "config.h"

/* Fairly self-explanitory.  If you're using threads you can limit the number
 * of active threads at any one time with MAX_ACTIVE_THREADS */
#ifdef PTHREADS
#  include <pthread.h>
#  define MAX_ACTIVE_THREADS 2
#endif

#define VERSION "2.07"

/* Also self-explanitory.  The default place to look for the xferstats config
 * information */
#define DEFAULT_CONFIG "/usr/local/etc/xferstats.cfg"

/* The default xferlog filename */
#define FILENAME "/var/log/xferlog"

/* defining a MY_MALLOC wrapper as a macro saves me from having to check
 * the return value every time I malloc */
#ifdef DEBUG_MALLOC
#  define MY_MALLOC(ptr, size) if ((ptr = malloc(size)) == NULL)	     \
    { perror("malloc"); exit(1);					     \
    fprintf(stderr, "malloc at line: %d size: %d ptr: %p\n", __LINE__, size, \
	    ptr); }
#else
#  define MY_MALLOC(ptr, size) if ((ptr = malloc(size)) == NULL)	\
    { perror("malloc"); exit(1); }
#endif


/* 64 and 1024 are the wu-ftpd default values and unless you have some
 * reason to change them, I'd leave them alone.  If you do decide to change
 * them, MAXPATHLEN -must- be larger than MAXHOSTNAMELEN. */
#define MAXHOSTNAMELEN 64
#ifdef PATH_MAX
#  define MAXPATHLEN PATH_MAX
#else /* PATH_MAX */
#  define MAXPATHLEN 1024
#endif /* PATH_MAX */

/* my types */
enum
{
  T_FILE,
  T_DIR,
  T_HOST,
  T_TLD,
  T_DOMAIN,
  T_SIZE
};

/* individual days */
typedef struct _xfdays_t
{
  char name[16];
  unsigned LONGLONG data;
  unsigned int file_count;
  unsigned LONGLONG seconds;
  struct _xfdays_t *next_ptr;
  struct _xfdays_t *prev_ptr;
} xfdays_t;

/* used for hours of the day, days of the week, days of the month */
typedef struct _xfmult_t
{
  unsigned LONGLONG data;
  unsigned LONGLONG seconds;
  unsigned int file_count;
} xfmult_t;

/* struct used for file, directory, host, domain, and top-level domain data */
typedef struct _xfmisc_t
{
  char *name;
  unsigned LONGLONG data;
  unsigned int file_count;
  unsigned LONGLONG seconds;
  struct _xfmisc_t *next_ptr;
} xfmisc_t;

typedef struct _only_dir_t
{
  char * dir;
  unsigned short len; /* to avoid having to strlen it a zillion times */
} only_dir_t;

typedef struct _list_t
{
  void * item;
  struct _list_t * next;
} list_t;

typedef struct _ftp_entry_t
{
  /* the date the file was ftped */
  char date[25];

  /* the time it took to transfer the file (in seconds) */
  long seconds;

  /* pointer to the hostname of the ftper */
  char *host;

  /* the size of the file (in bytes, of course) */
  unsigned long data;

  /* pointer to the path and filename */
  char *path;
  
  /* pointer to the next file */
  struct _ftp_entry_t * next_ptr;
} ftp_entry_t;

typedef struct _config_t
{
  /* log file name */
  char * file_name;

  /* config file name */
  char * config_file;

  /* report on files ftped by real users.  0 == no */
  char real_traffic;

  /* report on files ftped by anonymous users.  0 == no */
  char anon_traffic;
  
  /* report on files ftped by guest users.  0 == no */
  char guest_traffic;
  
  /* report on inbound files (products of a PUT) */
  char inbound;

  /* report on outbound files (products of a GET) */
  char outbound;

  /* generate hourly reports.  0 == no */
  char hourly_traffic;

  /* generate day of the week reports.  0 == no */
  char dow_traffic;
  
  /* generate day of the month reports.  0 == no */
  char dom_traffic;
  
  /* generate domain reports.  0 == no */
  char domain_traffic;
  
  /* generate top-level domain reports.  0 == no */
  char tld_traffic;
  
  /* generate hosts reports.  0 == no */
  char host_traffic;
  
  /* generate total traffic by directory reports.  0 == no */
  char dir_traffic;
  
  /* generate file traffic report.  0 == no */
  char file_traffic;
  
  /* generate monthly traffic report.  0 == no */
  char monthly_traffic;
  
  /* generate HTML output.  0 == no */
  char html_output;
  
  /* graph path */
  char * graph_path;

  /* get the log from the standard input.  0 == no */
  char use_stdin;
  
  /* the log type being processed (0 == wu-ftpd, 1 == ncftpd) */
  unsigned char log_type;
  
  /* how deep in the directory tree should the total report by section
     compare? */
  char depth;
  
  /* what section should be displayed.  NULL == all sections */
  list_t * only_dir;

  /* what domain should be displayed.  NULL == all domains */
  /* char *only_tld; */
  /* to avoid having to strlen it a zillion times */
  /* int only_tld_length; */
  /* This will have to wait for another version. */
  
  /* how many files should be reported? */
  int number_file_stats;
  
  /* how many days should be listed in the daily stats? */
  int number_daily_stats;
  
  /* how many sections should be listed in the total report by section? */
  int number_dir_stats;
  
  /* how many top level domains should be listed in the TLD statistics? */
  int number_tld_stats;

  /* how many domains should be listed in the domain statistics? */
  int number_domain_stats;

  /* how many hosts should be listed in the domain statistics? */
  int number_host_stats;

  /* what's the max # of records in any one report? (does not apply to
   * hourly, day-of-the-week, day-of-the-month, or yearly reports)  Must
   * be a number larger than 9 (or 0 to disable the limit) */
  int max_report_size;

  /* if nonzero, write all reports to stdout instead of their respective
   * individual files */
  char stdout_output;

  /* if nonzero, don't put <HTML><BODY> ... </BODY></HTML> bits in HTML
   * output */
  char no_html_headers;

  /* if nonzero, output everything as one HTML page, without links.
   * single_page implies no_html_headers automatically.*/
  char single_page;

  /* if nonzero, xferstats will process all data and dump it to a database to
   * avoid having to reprocess it at the next run */
  char additive_db;
} config_t;

/* memory arena size */
#ifdef ENABLE_ARENAS
#define ARENA_SIZE 16384

typedef struct _arena_t
{
  __ptr_t memory;
  __ptr_t free;
#ifndef SMFS
  short int allocs;
#endif /* SMFS */
  struct _arena_t * prev_ptr;
  struct _arena_t * next_ptr;
} arena_t;
#else /* ENABLE_ARENAS */
typedef struct _arena_t
{
} arena_t;
#endif /* ENABLE_ARENAS */

/* used to get around pthread_create's argument-passing limitation */
typedef struct
{
  /* the memory arenas for many functions */
  arena_t * ftp_line_arena;
  arena_t * day_arena;
  arena_t * file_arena;
  arena_t * dir_arena;
  arena_t * host_arena;
  arena_t * tld_arena;
  arena_t * domain_arena;

  /* the logfile ptr */
  FILE * file_ptr;

  /* configuration data */
  config_t * config;

  /* the start of all of the data */
  ftp_entry_t * first_ftp_line;
  xfdays_t * first_day_ptr;
  xfdays_t * last_day_ptr;
  xfmisc_t * first_tld_ptr;
  xfmisc_t * first_domain_ptr;
  xfmisc_t * first_host_ptr;
  xfmisc_t * first_file_ptr;
  xfmisc_t * first_dir_ptr;
  xfmisc_t * first_size_ptr;
  xfmult_t hour_data[24];
  xfmult_t weekday_data[7];
  xfmult_t dom_data[31];
  xfmult_t month_data[12];

  /* counters for global totals */
  unsigned long file_count;
  unsigned LONGLONG data;

  /* the first and last day that we have data for */
  char low_date[25];
  char high_date[25];

#ifdef PTHREADS
  /* various mutexs, condition variables, and a counter */
  unsigned char active_threads;
  pthread_mutex_t * active_threads_mutex;
  pthread_cond_t * active_threshold_cond;
  pthread_mutex_t * file_data_running_mutex;
  pthread_mutex_t * host_data_running_mutex;
  pthread_mutex_t * dir_data_running_mutex;
  pthread_mutex_t * tld_data_running_mutex;
  pthread_mutex_t * domain_data_running_mutex;
  pthread_mutex_t * monthly_data_running_mutex;
  pthread_mutex_t * hourly_data_running_mutex;
  pthread_mutex_t * dom_data_running_mutex;
  pthread_mutex_t * dow_data_running_mutex;
  pthread_mutex_t * active_display_thread_mutex;
  pthread_cond_t * data_gen_finished_cond;
  pthread_mutex_t * data_gen_finished_cond_mutex;
#endif
} pointers_t;

/* prototypes */
/* arena.c */
__ptr_t amalloc(arena_t ** arena, size_t size);
void afree(arena_t * arena, __ptr_t memory);
void aclean(arena_t ** arena);

/* config.c */
int init_config(pointers_t * pointers);

/* display.c */
void display_daily_totals(pointers_t * pointers);
void display_misc_totals(pointers_t * pointers, int type);
void display_file_totals(pointers_t * pointers);
void display_dir_totals(pointers_t * pointers);
void display_hourly_totals(pointers_t * pointers);
void display_monthly_totals(pointers_t * pointers);
void display_dow_totals(pointers_t * pointers);
void display_dom_totals(pointers_t * pointers);
void display_tld_totals(pointers_t * pointers);
void display_domain_totals(pointers_t * pointers);
void display_host_totals(pointers_t * pointers);

/* parselog.c */
void parse_wuftpd_logfile(pointers_t * pointers);
void parse_ncftpd_logfile(pointers_t * pointers);
void parse_apache_logfile(pointers_t * pointers);

/* list.c */
list_t * list_new(void);
void list_add_item(list_t ** list, void * item);
void list_add_item_head(list_t ** list, void * item);
void list_add_item_tail(list_t ** list, void * item);
int list_remove_item(list_t ** list, void * item);
void list_destroy(list_t ** list);

#endif /* _XFERSTATS_H */
