/* Copyright (c) 1993, 1994  Washington University in Saint Louis
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution. 3. All advertising
 * materials mentioning features or use of this software must display the
 * following acknowledgement: This product includes software developed by the
 * Washington University in Saint Louis and its contributors. 4. Neither the
 * name of the University nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific
 * prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY WASHINGTON UNIVERSITY AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL WASHINGTON
 * UNIVERSITY OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef lint
static char rcsid[] = "@(#)$Id: extensions.c,v 1.25 1999/02/26 16:08:50 sob RELEASE sob $";
#endif /* not lint */

#include "config.h"

#include <stdio.h>
#include <errno.h>
#include <string.h>
#ifdef SYSSYSLOG
#include <sys/syslog.h>
#else
#include <syslog.h>
#endif
#include <time.h>
#include <pwd.h>
#include <setjmp.h>
#include <grp.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/param.h>

#if defined(HAVE_STATVFS)
#include <sys/statvfs.h>
#elif defined(HAVE_SYS_VFS)
#include <sys/vfs.h>
#elif defined(HAVE_SYS_MOUNT)
#include <sys/mount.h>
#endif

#include <arpa/ftp.h>

#include "pathnames.h"
#include "extensions.h"

#if defined(HAVE_FTW)
#include <ftw.h>
#else
#include "support/ftw.h"
#endif

#ifdef QUOTA
struct dqblk quota;
char
#ifdef __STDC__
*time_quota(long curstate, long softlimit, long timelimit, char* timeleft);
#else
*time_quota();
#endif
#endif

#ifdef HAVE_REGEX_H
#include <regex.h>
#endif

#if defined(REGEX) && defined(SVR4) && ! (defined(NO_LIBGEN))
#include <libgen.h>
#endif

extern int fnmatch(),
  type,
  transflag,
  ftwflag,
  authenticated,
  autospout_free,
  data,
  pdata,
  anonymous,
  guest;

#ifdef LOG_FAILED
  extern char the_user[];
#endif

#ifdef __STDC__
extern char **ftpglob(register char *v),
#else
extern char **ftpglob(),
#endif
 *globerr,
  remotehost[],
#ifdef THROUGHPUT
  remoteaddr[],
#endif
  hostname[],
  authuser[],
  chroot_path[],
 *autospout,
  Shutdown[];

char shuttime[30],
  denytime[30],
  disctime[30];

#ifdef __STDC__
extern char *wu_realpath(const char *pathname, char *result, char* chroot_path);
extern char *fb_realpath(const char *pathname, char *result);
#else
extern char *wu_realpath();
extern char *fb_realpath();
#endif

#ifndef REGEX
char *re_comp();
#elif defined(M_UNIX)
extern char *regcmp(), *regex();
#endif

#ifdef __STDC__
extern FILE *dataconn(char *name, off_t size, char *mode);
#else
extern FILE *dataconn();
#endif
FILE *dout;

time_t newer_time;

int show_fullinfo;

int
#ifdef __STDC__
check_newer(char *path, struct stat *st, int flag)
#else
check_newer(path,st,flag)
char *path;
struct stat *st;
int flag;
#endif
{

    if (st->st_mtime > newer_time) {
        if (show_fullinfo != 0) {
/* This always was a bug, because neither st_size nor time_t were required to
be compatible with int, but needs fixing properly for C9X. */

/* Some systems use one format, some another.  This takes care of the garbage */
#if (defined(BSD) && (BSD >= 199103)) && !defined(LONGOFF_T)
#define L_FORMAT "qd"
#define T_FORMAT "d"
#else
#ifdef _AIX42
#define L_FORMAT "lld"
#define T_FORMAT "d"
#else
#define L_FORMAT "d"
#define T_FORMAT "d"
#endif
#endif

            if (flag == FTW_F || flag == FTW_D) {
                fprintf(dout, "%s %" L_FORMAT " %" T_FORMAT " %s\n", flag == FTW_F ? "F" : "D",
                        st->st_size, st->st_mtime, path);
            }
        } else if (flag == FTW_F)
            fprintf(dout, "%s\n", path);
    }
    /* When an ABOR has been received (which sets ftwflag > 1) return a
     * non-zero value which causes ftw to stop tree traversal and return. */
    return (ftwflag > 1 ? 1 : 0);
}

#if defined(HAVE_STATVFS)
int getSize(s)
char *s;
{
    int c;
    struct statvfs buf;

    if (( c = statvfs(s, &buf)) != 0)
        return(0);

    return(buf.f_bavail * buf.f_frsize / 1024);
}
#elif defined(HAVE_SYS_VFS) || defined (HAVE_SYS_MOUNT)
int getSize(s)
char *s;
{
    int c;
    struct statfs buf;

    if (( c = statfs(s, &buf)) != 0)
        return(0);

    return(buf.f_bavail * buf.f_bsize / 1024);
}
#endif

/*************************************************************************/
/* FUNCTION  : msg_massage                                               */
/* PURPOSE   : Scan a message line for magic cookies, replacing them as  */
/*             needed.                                                   */
/* ARGUMENTS : pointer input and output buffers                          */
/*************************************************************************/

void
#ifdef __STDC__
msg_massage(char *inbuf, char *outbuf)
#else
msg_massage(inbuf,outbuf)
char *inbuf;
char *outbuf;
#endif
{
    char *inptr = inbuf;
    char *outptr = outbuf;
#ifdef QUOTA
    char timeleft[80];
#endif
    char buffer[MAXPATHLEN];
    time_t curtime;
    int limit;
    extern struct passwd *pw;
    struct aclmember *entry;
#ifdef VIRTUAL
    extern int virtual_mode;
    extern char virtual_email[];
#endif

    (void) acl_getclass(buffer);
    limit = acl_getlimit(buffer, NULL);

    while (*inptr) {
        if (*inptr != '%')
            *outptr++ = *inptr;
        else {
            entry = NULL;
            switch (*++inptr) {
            case 'E':
#ifdef VIRTUAL
		if (virtual_mode && virtual_email[0] != '\0')
		    sprintf(outptr, "%s", virtual_email);
		else
#endif
                if ( (getaclentry("email", &entry)) && ARG0 )
                    sprintf(outptr, "%s", ARG0); 
                else
                    *outptr = '\0';
                break;
            case 'N': 
                sprintf(outptr, "%d", acl_countusers(buffer)); 
                break; 
            case 'M':
                if (limit == -1)
                    strcpy(outptr,"unlimited");
                else
                    sprintf(outptr, "%d", limit);
                break;
            case 'T':
 	        (void) time(&curtime);
                strncpy(outptr, ctime(&curtime), 24);
                *(outptr + 24) = '\0';
                break;

            case 'F':
#if defined(HAVE_STATVFS) || defined(HAVE_SYS_VFS) || defined(HAVE_SYS_MOUNT)
                sprintf(outptr, "%lu", getSize("."));
#endif
                break;

            case 'C':
#ifdef HAVE_GETCWD
                (void) getcwd(outptr, MAXPATHLEN);
#else
#error	wu-ftpd on this platform has security deficiencies!!!
                (void) getwd(outptr);
#endif
                break;

            case 'R':
                strcpy(outptr, remotehost);
                break;

            case 'L':
                strcpy(outptr, hostname);
                break;

            case 'U':
#ifdef LOG_FAILED
                strcpy (outptr, the_user);
#else /* LOG_FAILED */
                strcpy(outptr,
                    (pw == NULL) ? "[unknown]" : pw->pw_name);
#endif /* LOG_FAILED */

                break;

            case 's':
                strncpy(outptr, shuttime, 24);
                *(outptr + 24) = '\0';
                break;

            case 'd':
                strncpy(outptr, disctime, 24);
                *(outptr + 24) = '\0';
                break;

            case 'r':
                strncpy(outptr, denytime, 24);
                *(outptr + 24) = '\0';
                break;

/* KH : cookie %u for RFC931 name */
            case 'u':
                if (authenticated) strncpy(outptr, authuser, 24);
                else strcpy(outptr,"[unknown]");
                *(outptr + 24) = '\0'; 
                break;

#ifdef QUOTA
            case 'B':
#ifdef QUOTA_BLOCKS  /* 1024-blocks instead of 512-blocks */
                sprintf(outptr, "%d", quota.dqb_bhardlimit % 2 ?
                 quota.dqb_bhardlimit / 2 + 1 : quota.dqb_bhardlimit / 2);
#else
                sprintf(outptr, "%d", quota.dqb_bhardlimit);
#endif
                break;

            case 'b':
#ifdef QUOTA_BLOCKS  /* 1024-blocks instead of 512-blocks */
                sprintf(outptr, "%d", quota.dqb_bsoftlimit % 2 ?
                 quota.dqb_bsoftlimit / 2 + 1 : quota.dqb_bsoftlimit / 2);
#else
                sprintf(outptr, "%d", quota.dqb_bsoftlimit);
#endif
                break;

            case 'Q':
#ifdef QUOTA_BLOCKS  /* 1024-blocks instead of 512-blocks */
                sprintf(outptr, "%d", quota.dqb_curblocks % 2 ?
                 quota.dqb_curblocks / 2 + 1 : quota.dqb_curblocks / 2);
#else
                sprintf(outptr, "%d", quota.dqb_curblocks);
#endif
                break;

            case 'I':
#if defined(AIX) || defined (LINUX) || defined (BSDI) || defined (DIGITAL)
                sprintf(outptr, "%d", quota.dqb_ihardlimit);
#else
                sprintf(outptr, "%d", quota.dqb_fhardlimit);
#endif
                break;

            case 'i':
#if defined(AIX) || defined (LINUX) || defined (BSDI) || defined (DIGITAL)
                sprintf(outptr, "%d", quota.dqb_isoftlimit);
#else
                sprintf(outptr, "%d", quota.dqb_fsoftlimit);
#endif
                break;

            case 'q':
#if defined(AIX) || defined (LINUX) || defined (BSDI) || defined (DIGITAL)
                sprintf(outptr, "%d", quota.dqb_curinodes);
#else
                sprintf(outptr, "%d", quota.dqb_curfiles);
#endif
                break;

            case 'H':
                time_quota(quota.dqb_curblocks,quota.dqb_bsoftlimit,
#if defined(AIX) || defined (LINUX) || defined (BSDI) || defined (DIGITAL)
                               quota.dqb_btime, timeleft);
#else
                               quota.dqb_btimelimit, timeleft);
#endif
                strcpy(outptr, timeleft);
                break;

            case 'h':
#if defined(AIX) || defined (LINUX) || defined (BSDI) || defined (DIGITAL)
                time_quota(quota.dqb_curinodes,quota.dqb_isoftlimit,
                               quota.dqb_itime, timeleft);
#else
                time_quota(quota.dqb_curfiles,quota.dqb_fsoftlimit,
                               quota.dqb_ftimelimit, timeleft);
#endif
                strcpy(outptr, timeleft);
                break;
#endif

            case '%':
                *outptr++ = '%';
                *outptr = '\0';
                break;

            default:
                *outptr++ = '%';
                *outptr++ = '?';
                *outptr = '\0';
                break;
            }
            while (*outptr)
                outptr++;
        }
        inptr++;
    }
    *outptr = '\0';
}

/*************************************************************************/
/* FUNCTION  : cwd_beenhere                                              */
/* PURPOSE   : Return 1 if the user has already visited this directory   */
/*             via C_WD.                                                 */
/* ARGUMENTS : a power-of-two directory function code (README, MESSAGE)  */
/*************************************************************************/

int
#ifdef __STDC__
cwd_beenhere(int dircode)
#else
cwd_beenhere(dircode)
int dircode;
#endif
{
    struct dirlist {
        struct dirlist *next;
        int dircode;
        char dirname[1];
    };

    static struct dirlist *head = NULL;
    struct dirlist *curptr;
    char cwd[MAXPATHLEN];

    (void) fb_realpath(".", cwd);

    for (curptr = head; curptr != NULL; curptr = curptr->next)
        if (strcmp(curptr->dirname, cwd) == 0) {
            if (!(curptr->dircode & dircode)) {
                curptr->dircode |= dircode;
                return (0);
            }
            return (1);
        }
    curptr = (struct dirlist *) malloc(strlen(cwd) + 1 + sizeof(struct dirlist));

    if (curptr != NULL) {
        curptr->next = head;
        head = curptr;
        curptr->dircode = dircode;
        strcpy(curptr->dirname, cwd);
    }
    return (0);
}

/*************************************************************************/
/* FUNCTION  : show_banner                                               */
/* PURPOSE   : Display a banner on the user's terminal before login      */
/* ARGUMENTS : reply code to use                                         */
/*************************************************************************/
 
void
#ifdef __STDC__
show_banner(int msgcode)
#else
show_banner(msgcode)
int msgcode;
#endif
{
    char *crptr,
      linebuf[1024],
      outbuf[1024];
    struct aclmember *entry = NULL;
    FILE *infile;

#ifdef VIRTUAL
    extern int virtual_mode;
    extern char virtual_banner[];

    if (virtual_mode) {
        infile = fopen(virtual_banner, "r");
	if (infile) {
 	    while (fgets(linebuf, 255, infile) != NULL) {
	           if ((crptr = strchr(linebuf, '\n')) != NULL)
		        *crptr = '\0';
		   msg_massage(linebuf, outbuf);
		   lreply(msgcode, "%s", outbuf);
		 }
	    fclose(infile);
#ifndef NO_SUCKING_NEWLINES
	    lreply(msgcode, "");
#endif
	  }
      }
    else {
#endif
      /* banner <path> */
      while (getaclentry("banner", &entry)) {
	    infile = fopen(ARG0, "r");
	    if (infile) {
	        while (fgets(linebuf, 255, infile) != NULL) {
		  if ((crptr = strchr(linebuf, '\n')) != NULL)
		    *crptr = '\0';
		  msg_massage(linebuf, outbuf);
		  lreply(msgcode, "%s", outbuf);
		}
		fclose(infile);
#ifndef NO_SUCKING_NEWLINES
		lreply(msgcode, "");
#endif
	    }
	 }
#ifdef VIRTUAL
    }
#endif
  }
/*************************************************************************/
/* FUNCTION  : show_message                                              */
/* PURPOSE   : Display a message on the user's terminal if the current   */
/*             conditions are right                                      */
/* ARGUMENTS : reply code to use, LOG_IN|CMD                             */
/*************************************************************************/

void
#ifdef __STDC__
show_message(int msgcode, int mode)
#else
show_message(msgcode,mode)
int msgcode;
int mode;
#endif
{
    char *crptr,
      linebuf[1024],
      outbuf[1024],
      class[MAXPATHLEN],
      cwd[MAXPATHLEN];
    int show,
      which;
    struct aclmember *entry = NULL;
    FILE *infile;

    if (mode == C_WD && cwd_beenhere(1) != 0)
        return;

#ifdef HAVE_GETCWD
    (void) getcwd(cwd,MAXPATHLEN-1);
#else
    (void) getwd(cwd);
#endif
    (void) acl_getclass(class);

    /* message <path> [<when> [<class>]] */
    while (getaclentry("message", &entry)) {
        if (!ARG0)
            continue;
        show = 0;

        if (mode == LOG_IN && (!ARG1 || !strcasecmp(ARG1, "login")))
            if (!ARG2)
                show++;
            else {
                for (which = 2; (which < MAXARGS) && ARG[which]; which++)
                    if (strcasecmp(class, ARG[which]) == 0)
                        show++;
            }
        if (mode == C_WD && ARG1 && !strncasecmp(ARG1, "cwd=", 4) &&
            (!strcmp((ARG1) + 4, cwd) || *(ARG1 + 4) == '*' ||
            !fnmatch((ARG1) + 4, cwd, FNM_PATHNAME)))
            if (!ARG2)
                show++;
            else {
                for (which = 2; (which < MAXARGS) && ARG[which]; which++)
                    if (strcasecmp(class, ARG[which]) == 0)
                        show++;
            }
        if (show && (int)strlen(ARG0) > 0) {
            infile = fopen(ARG0, "r");
            if (infile) {
                while (fgets(linebuf, 255, infile) != NULL) {
                    if ((crptr = strchr(linebuf, '\n')) != NULL)
                        *crptr = '\0';
                    msg_massage(linebuf, outbuf);
                    lreply(msgcode, "%s", outbuf);
                }
                fclose(infile);
#ifndef NO_SUCKING_NEWLINES
                lreply(msgcode, "");
#endif
            }
        }
    }
}

/*************************************************************************/
/* FUNCTION  : show_readme                                               */
/* PURPOSE   : Display a message about a README file to the user if the  */
/*             current conditions are right                              */
/* ARGUMENTS : pointer to ACL buffer, reply code, LOG_IN|C_WD            */
/*************************************************************************/

void
#ifdef __STDC__
show_readme(int code, int mode)
#else
show_readme(code,mode)
int code;
int mode;
#endif
{
    char **filelist,
      **sfilelist,
      class[MAXPATHLEN],
      cwd[MAXPATHLEN];
    int show,
      which,
      days;
    time_t clock;

    struct stat buf;
    struct tm *tp;
    struct aclmember *entry = NULL;

    if (cwd_beenhere(2) != 0)
        return;

#ifdef HAVE_GETCWD
    (void) getcwd(cwd,MAXPATHLEN-1);
#else
    (void) getwd(cwd);
#endif
    (void) acl_getclass(class);

    /* readme  <path> {<when>} */
    while (getaclentry("readme", &entry)) {
        if (!ARG0)
            continue;
        show = 0;

        if (mode == LOG_IN && (!ARG1 || !strcasecmp(ARG1, "login")))
            if (!ARG2)
                show++;
            else {
                for (which = 2; (which < MAXARGS) && ARG[which]; which++)
                    if (strcasecmp(class, ARG[which]) == 0)
                        show++;
            }
        if (mode == C_WD && ARG1 && !strncasecmp(ARG1, "cwd=", 4)
            && (!strcmp((ARG1) + 4, cwd) || *(ARG1 + 4) == '*' ||
                !fnmatch((ARG1) + 4, cwd, FNM_PATHNAME)))
            if (!ARG2)
                show++;
            else {
                for (which = 2; (which < MAXARGS) && ARG[which]; which++)
                    if (strcasecmp(class, ARG[which]) == 0)
                        show++;
            }
        if (show) {
            globerr = NULL;
            filelist = ftpglob(ARG0);
            sfilelist = filelist;  /* save to free later */
            if (!globerr) {
                while (filelist && *filelist) {
                   errno = 0;
                   if (!stat(*filelist, &buf) &&
                       (buf.st_mode & S_IFMT) == S_IFREG) {
                       lreply(code, "Please read the file %s", *filelist);
                       (void) time(&clock);
                       tp = localtime(&clock);
                       days = 365 * tp->tm_year + tp->tm_yday;
                       tp = localtime((time_t *)&buf.st_mtime);
                       days -= 365 * tp->tm_year + tp->tm_yday;
/*
                       if (days == 0) {
                         lreply(code, "  it was last modified on %.24s - Today",
                           ctime((time_t *)&buf.st_mtime));
                       } else {
*/
                         lreply(code, 
                           "  it was last modified on %.24s - %d day%s ago",
                           ctime((time_t *)&buf.st_mtime), days, days == 1 ? "" : "s");
/*
                       }
*/
                   }
                   filelist++;
                }
            }
            if (sfilelist) {
                blkfree(sfilelist);
                free((char *) sfilelist);
            }
        }
    }
}

/*************************************************************************/
/* FUNCTION  : deny_badxfertype                                          */
/* PURPOSE   : If user is in ASCII transfer mode and tries to retrieve a */
/*             binary file, abort transfer and display appropriate error */
/* ARGUMENTS : message code to use for denial, path of file to check for */
/*             binary contents or NULL to assume binary file             */
/*************************************************************************/

int
#ifdef __STDC__
deny_badasciixfer(int msgcode, char *filepath)
#else
deny_badasciixfer(msgcode,filepath)
int msgcode;
char *filepath;
#endif
{

    if (type == TYPE_A && !*filepath) {
        reply(msgcode, "This is a BINARY file, using ASCII mode to transfer will corrupt it.");
        return (1);
    }
    /* The hooks are here to prevent transfers of actual binary files, not
     * just TAR or COMPRESS mode files... */
    return (0);
}

/*************************************************************************/
/* FUNCTION  : is_shutdown                                               */
/* PURPOSE   :                                                           */
/* ARGUMENTS :                                                           */
/*************************************************************************/

int
#ifdef __STDC__
is_shutdown(int quiet, int new)
#else
is_shutdown(quiet, new)
int quiet;
int new;
#endif
{
    static struct tm tmbuf;
    static struct stat s_last;
    static time_t last = 0,
      shut,
      deny,
      disc;
    static int valid;
    static char text[2048];

    struct stat s_cur;

    FILE *fp;

    int deny_off,
      disc_off;

    time_t curtime = time(NULL);

    char buf[1024],
      linebuf[1024];

    if (Shutdown[0] == '\0' || stat(Shutdown, &s_cur))
        return (0);

    if (s_last.st_mtime != s_cur.st_mtime) {
        s_last = s_cur;
        valid = 0;

        fp = fopen(Shutdown, "r");
        if (fp == NULL)
            return (0);
        fgets(buf, sizeof(buf), fp);
        if (sscanf(buf, "%d %d %d %d %d %d %d", &tmbuf.tm_year, &tmbuf.tm_mon,
        &tmbuf.tm_mday, &tmbuf.tm_hour, &tmbuf.tm_min, &deny, &disc) != 7) {
            (void) fclose(fp);
            return (0);
        }
        valid = 1;
        deny_off = 3600 * (deny / 100) + 60 * (deny % 100);
        disc_off = 3600 * (disc / 100) + 60 * (disc % 100);

        tmbuf.tm_year -= 1900;
        tmbuf.tm_isdst = -1;
        shut = mktime(&tmbuf);
        strcpy(shuttime, ctime(&shut));

        disc = shut - disc_off;
        strcpy(disctime, ctime(&disc));

        deny = shut - deny_off;
        strcpy(denytime, ctime(&deny));

        text[0] = '\0';

        while (fgets(buf, sizeof(buf), fp) != NULL) {
            msg_massage(buf, linebuf);
            if ((strlen(text) + strlen(linebuf)) < sizeof(text))
                strcat(text, linebuf);
        }

        (void) fclose(fp);
    }
    if (!valid)
        return (0);

    /* if last == 0, then is_shutdown() only called with quiet == 1 so far */
    if (last == 0 && !quiet) {
        autospout = text;       /* warn them for the first time */
        autospout_free = 0;
        last = curtime;
    }
    /* if a new connection and past deny time, tell caller to drop 'em */
    if (new && curtime > deny)
        return (1);

    /* if past disconnect time, tell caller to drop 'em */
    if (curtime > disc)
        return (1);

    /* if less than 60 seconds to disconnection, warn 'em continuously */
    if (curtime > (disc - 60) && !quiet) {
        autospout = text;
        autospout_free = 0;
        last = curtime;
    }
    /* if less than 15 minutes to disconnection, warn 'em every 5 mins */
    if (curtime > (disc - 60 * 15)) {
        if ((curtime - last) > (60 * 5) && !quiet) {
            autospout = text;
            autospout_free = 0;
            last = curtime;
        }
    }
    /* if less than 24 hours to disconnection, warn 'em every 30 mins */
    if (curtime < (disc - 24 * 60 * 60) && !quiet) {
        if ((curtime - last) > (60 * 30)) {
            autospout = text;
            autospout_free = 0;
            last = curtime;
        }
    }
    /* if more than 24 hours to disconnection, warn 'em every 60 mins */
    if (curtime > (disc - 24 * 60 * 60) && !quiet) {
        if ((curtime - last) >= (24 * 60 * 60)) {
            autospout = text;
            autospout_free = 0;
            last = curtime;
        }
    }
    return (0);
}

void
#ifdef __STDC__
newer(char *date, char *path, int showlots)
#else
newer(date,path,showlots)
char *date;
char *path;
int showlots;
#endif
{
    struct tm tm;

    if (sscanf(date, "%04d%02d%02d%02d%02d%02d",
               &tm.tm_year, &tm.tm_mon, &tm.tm_mday,
               &tm.tm_hour, &tm.tm_min, &tm.tm_sec) == 6) {

        tm.tm_year -= 1900;
        tm.tm_mon--;
        tm.tm_isdst = -1;
        newer_time = mktime(&tm);
        dout = dataconn("file list", (off_t) - 1, "w");

        if (dout != NULL) {
            /* As ftw allocates storage it needs a chance to cleanup, setting
             * ftwflag prevents myoob from calling longjmp, incrementing
             * ftwflag instead which causes check_newer to return non-zero
             * which makes ftw return. */
            ftwflag = 1;
            transflag++;
            show_fullinfo = showlots;
#if defined(HAVE_FTW)
            ftw(path, check_newer, -1);
#else
            treewalk(path, check_newer, -1, NULL);
#endif

            /* don't send a reply if myoob has already replied */
            if (ftwflag == 1) {
                if (ferror(dout) != 0)
                    perror_reply(550, "Data connection");
                else
                    reply(226, "Transfer complete.");
            }

            (void) fclose(dout);
            data = -1;
            pdata = -1;
            transflag = 0;
            ftwflag = 0;
        }
    } else
        reply(501, "Bad DATE format");
}

int
#ifdef __STDC__
type_match(char *typelist)
#else
type_match(typelist)
char *typelist;
#endif
{
    if (anonymous && strcasestr(typelist, "anonymous"))
        return (1);
    if (guest && strcasestr(typelist, "guest"))
        return (1);
    if (!guest && !anonymous && strcasestr(typelist, "real"))
        return (1);

    if (strncasecmp (typelist, "class=", 6) == 0) {
      char class [1024];

      (void) acl_getclass (class);
      if (strcasecmp (typelist+6, class) == 0)
        return (1);
      }

    return (0);
}

int
#ifdef __STDC__
path_compare(char *p1, char *p2)
#else
path_compare(p1,p2)
char *p1;
char *p2;
#endif
{
    if ( fnmatch(p1, p2, NULL) == 0 ) /* 0 means they matched */
        return(strlen(p1));
    else
        return(-2);
}

void
#ifdef __STDC__
expand_id(void)
#else
expand_id()
#endif
{
  char class [1024];
  struct aclmember *entry = NULL;
  (void) acl_getclass (class);
  while (getaclentry ("upload", &entry)) {
    char *q;
    int i = 0;
    int options = 1;
    int classfound = 0;
    int classmatched = 0;
    while (options
    &&     (i < MAXARGS)
    &&     ((q = entry->arg[i]) != (char *)NULL)
    &&     (q[0] != '\0')) {
      if (strcasecmp (q, "absolute") == 0)
        i++;
      else if (strcasecmp (q, "relative") == 0)
        i++;
      else if (strncasecmp (q, "class=", 6) == 0) {
        i++;
        classfound = 1;
        if (strcasecmp (q+6, class) == 0)
          classmatched = 1;
      } else if (strcmp (q, "-") == 0) {
        i++;
        options = 0;
      } else
        options = 0;
    }
    if (!classfound || classmatched) {
      char buf [BUFSIZ];
      /*
      *  UID
      */
      if (((i + 3) < MAXARGS)
      &&  ((q = entry->arg[i+3]) != (char *)NULL)
      &&  (q[0] != '\0')
      &&  (strcmp (q, "*") != 0)) {
        if (q[0] == '%')
          sprintf (buf, "%s", q+1);
        else {
          struct passwd *pwent = getpwnam (q);
          if (pwent)
            sprintf (buf, "%d", pwent->pw_uid);
          else
            sprintf (buf, "%d", 0);
        }
        entry->arg[i+3] = (char *) malloc (strlen (buf) + 1);
if (entry->arg[i+3] == NULL) {
  syslog (LOG_ERR, "calloc error in expand_id");
  exit (0);
}
        strcpy (entry->arg[i+3], buf);
      }
      /*
      *  GID
      */
      if (((i + 4) < MAXARGS)
      &&  ((q = entry->arg[i+4]) != (char *)NULL)
      &&  (q[0] != '\0')
      &&  (strcmp (q, "*") != 0)) {
        if (q[0] == '%')
          sprintf (buf, "%s", q+1);
        else {
          struct group *grent = getgrnam (q);
          if (grent)
            sprintf (buf, "%d", grent->gr_gid);
          else
            sprintf (buf, "%d", 0);
          endgrent ();
        }
        entry->arg[i+4] = (char *) malloc (strlen (buf) + 1);
if (entry->arg[i+4] == NULL) {
  syslog (LOG_ERR, "calloc error in expand_id");
  exit (0);
}
        strcpy (entry->arg[i+4], buf);
      }
    }
  }
}

int
#ifdef __STDC__
fn_check(char *name)
#else
fn_check(name)
char *name;
#endif
{
  /* check to see if this is a valid file name... path-filter <type>
   * <message_file> <allowed_charset> <disallowed> */

  struct aclmember *entry = NULL;
  int   j;
  char *sp;
  char *path;
#ifdef M_UNIX
# ifdef REGEX
  char *regp;
# endif
#endif

#ifdef REGEXEC
  regex_t regexbuf;
  regmatch_t regmatchbuf;
#endif

#ifdef LINUX
  re_syntax_options = RE_SYNTAX_POSIX_EXTENDED;
#endif

  while (getaclentry("path-filter", &entry) && ARG0 != NULL) {
      if (type_match(ARG0) && ARG1 && ARG2) {

		  /*
		   * check *only* the basename
		   */

		  if (path = strrchr(name, '/'))  ++path;
		  else	path = name;

          /* is it in the allowed character set? */
#if defined(REGEXEC)
          if (regcomp(&regexbuf, ARG2, REG_EXTENDED) != 0) {
              reply(550, "REGEX error");
#elif defined(REGEX)
          if ((sp = regcmp(ARG2, (char *) 0)) == NULL) {
              reply(550, "REGEX error");
#else
          if ((sp = re_comp(ARG2)) != 0) {
              perror_reply(550, sp);
#endif
              return(0);
          }
#if defined(REGEXEC)
          if (regexec(&regexbuf, path, 1, &regmatchbuf, 0) != 0) {
#elif defined(REGEX)
# ifdef M_UNIX
          regp = regex(sp, path);
          free(sp);
          if (regp == NULL) {
# else
          if ((regex(sp, path)) == NULL) {
# endif
#else
          if ((re_exec(path)) != 1) {
#endif
              pr_mesg(550, ARG1);
              reply(550, "%s: Permission denied on server. (Filename (accept))", name);
              return(0);
          }
          /* is it in any of the disallowed regexps */

          for (j = 3; j < MAXARGS; ++j) {
              /* ARGj == entry->arg[j] */
              if (entry->arg[j]) {
#if defined(REGEXEC)
                  if (regcomp(&regexbuf, entry->arg[j], REG_EXTENDED) != 0) {
                      reply(550, "REGEX error");
#elif defined(REGEX)
                  if ((sp = regcmp(entry->arg[j], (char *) 0)) == NULL) {
                      reply(550, "REGEX error");
#else
                  if ((sp = re_comp(entry->arg[j])) != 0) {
                      perror_reply(550, sp);
#endif
                      return(0);
                  }
#if defined(REGEXEC)
                  if (regexec(&regexbuf, path, 1, &regmatchbuf, 0) == 0) {
#elif defined(REGEX)
# ifdef M_UNIX
                  regp = regex(sp, path);
                  free(sp);
                  if (regp != NULL) {
# else
                  if ((regex(sp, path)) != NULL) {
# endif
#else
                  if ((re_exec(path)) == 1) {
#endif
                      pr_mesg(550, ARG1);
                      reply(550, "%s: Permission denied on server. (Filename (deny))", name);
                      return(0);
                  }
              }
          }
      }
  }
  return(1);
}

int
#ifdef __STDC__
dir_check(char *name, uid_t *uid, gid_t *gid, int *d_mode, int *valid)
#else
dir_check(name,uid,gid,d_mode,valid)
char *name;
uid_t *uid;
gid_t *gid;
int *d_mode;
int *valid;
#endif
{
  struct aclmember *entry = NULL;
  int match_value = -1;
  char *ap2 = NULL;
  char *ap3 = NULL;
  char *ap4 = NULL;
  char *ap5 = NULL;
  char *ap6 = NULL;
  char *ap7 = NULL;
  char cwdir [MAXPATHLEN];
  char *pwdir;
  char abspwdir [MAXPATHLEN];
  char relpwdir [MAXPATHLEN];
  char path [MAXPATHLEN];
  char *sp;
  struct stat stbuf;
  int stat_result;
  char class [1024];
  extern char *home;

  (void) acl_getclass (class);

  *valid = 0;
  /* what's our current directory? */

  /* XXX We could use dynamic RAM to store this path, but I'd rather just bail
     out with an error. The rest of wu is so crufy that a long path might
     just blow up later */

  if ((strlen(name) + 1) > sizeof(path)) {
    perror_reply (550, "Path too long");
    return (-1);
  }

  strcpy(path, name);
  sp = strrchr(path, '/');
  if (sp)
    *sp = '\0';
  else
    strcpy (path, ".");

  if ((fb_realpath (path, cwdir)) == NULL) {
    perror_reply (550, "Could not determine cwdir");
    return (-1);
  }

  if ((fb_realpath (home, relpwdir)) == NULL) {
    perror_reply (550, "Could not determine pwdir");
    return (-1);
  }

  if ((wu_realpath (home, abspwdir, chroot_path)) == NULL) {
    perror_reply (550, "Could not determine pwdir");
    return (-1);
  }

  while (getaclentry ("upload", &entry)) {
    char *q;
    int i = 0;
    int options = 1;
    int classfound = 0;
    int classmatched = 0;
    pwdir = abspwdir;
    while (options
    &&     (i < MAXARGS)
    &&     ((q = entry->arg[i]) != (char *)NULL)
    &&     (q[0] != '\0')) {
      if (strcasecmp (q, "absolute") == 0) {
        i++;
        pwdir = abspwdir;
      } else if (strcasecmp (q, "relative") == 0) {
        i++;
        pwdir = relpwdir;
      } else if (strncasecmp (q, "class=", 6) == 0) {
        i++;
          classfound = 1;
        if (strcasecmp (q+6, class) == 0)
          classmatched = 1;
      } else if (strcmp (q, "-") == 0) {
        i++;
        options = 0;
      } else
        options = 0;
    }
    if (!classfound || classmatched) {
      int j;
      if (((i + 1) < MAXARGS)
      &&  ((q = entry->arg[i]) != (char *) NULL)
      &&  (q[0] != '\0')
      &&  (   (0 == strcmp       (q, "*"  ))
           || (0 == strcmp       (q, pwdir))
           || (0 <  path_compare (q, pwdir)))
      && ((j = path_compare (entry->arg[i+1], cwdir)) >= match_value)) {
        match_value = j;

        ap2 = NULL;
        if (((i + 2) < MAXARGS)
        &&  ((q = entry->arg[i+2]) != (char *) NULL)
        &&  (q[0] != '\0'))
          ap2 = q;

        ap3 = NULL;
        if (((i + 3) < MAXARGS)
        &&  ((q = entry->arg[i+3]) != (char *) NULL)
        &&  (q[0] != '\0'))
          ap3 = q;

        ap4 = NULL;
        if (((i + 4) < MAXARGS)
        &&  ((q = entry->arg[i+4]) != (char *) NULL)
        &&  (q[0] != '\0'))
          ap4 = q;

        ap5 = NULL;
        if (((i + 5) < MAXARGS)
        &&  ((q = entry->arg[i+5]) != (char *) NULL)
        &&  (q[0] != '\0'))
          ap5 = q;

        ap6 = NULL;
        if (((i + 6) < MAXARGS)
        &&  ((q = entry->arg[i+6]) != (char *) NULL)
        &&  (q[0] != '\0'))
          ap6 = q;

        ap7 = NULL;
        if (((i + 7) < MAXARGS)
        &&  ((q = entry->arg[i+7]) != (char *) NULL)
        &&  (q[0] != '\0'))
          ap7 = q;
      }
    }
  }

  if (anonymous
  &&  (match_value < 0)) {
    reply (550, "%s: Permission denied on server. (Upload dirs)", name);
    return (0);
  }
  if ((ap2 && !strcasecmp(ap2, "no"    ))
  ||  (ap3 && !strcasecmp(ap3, "nodirs"))
  ||  (ap6 && !strcasecmp(ap6, "nodirs"))) {
    reply (550, "%s: Permission denied on server. (Upload dirs)", name);
    return (0);
  }
  if ((ap3 && *ap3 == '*')
  ||  (ap4 && *ap4 == '*'))
    stat_result = stat (path, &stbuf);
  if (ap3)
    if ((ap3[0] != '*')
    ||  (ap3[1] != '\0'))
      *uid = atoi (ap3);    /* the uid  */
    else
      if (stat_result == 0)
        *uid = stbuf.st_uid;
  if (ap4)
    if ((ap4[0] != '*')
    ||  (ap4[1] != '\0'))
      *gid = atoi(ap4);    /* the gid */
    else
      if (stat_result == 0)
        *gid = stbuf.st_gid;
  if (ap7) {
    sscanf (ap7, "%o", d_mode);
    *valid = 1;
  } else if (ap5) {
    sscanf (ap5, "%o", d_mode);
    if (*d_mode & 0600)
      *d_mode |= 0100;
    if (*d_mode & 0060)
      *d_mode |= 0010;
    if (*d_mode & 0006)
      *d_mode |= 0001;
    *valid = 1;
  }
  return (1);
}

int
#ifdef __STDC__
upl_check(char *name, uid_t *uid, gid_t *gid, int *f_mode, int *valid)
#else
upl_check(name,uid,gid,f_mode,valid)
char *name;
uid_t *uid;
gid_t *gid;
int *f_mode;
int *valid;
#endif
{
  int  match_value = -1;
  char cwdir [MAXPATHLEN];
  char *pwdir;
  char abspwdir [MAXPATHLEN];
  char relpwdir [MAXPATHLEN];
  char path [MAXPATHLEN];
  char *sp;
  int  i;
  struct stat stbuf;
  int stat_result;
  char *ap1 = NULL;
  char *ap2 = NULL;
  char *ap3 = NULL;
  char *ap4 = NULL;
  char *ap5 = NULL;
  struct aclmember *entry = NULL;
  char class [1024];
  extern struct passwd *pw;
  extern char *home;

  *valid = 0;
  (void) acl_getclass (class);

  /* what's our current directory? */

  /* XXX We could use dynamic RAM to store this path, but I'd rather just bail
     out with an error. The rest of wu is so crufy that a long path might
     just blow up later */

  if ((strlen(name) + 1) > sizeof(path)) {
    perror_reply (553, "Path too long");
    return (-1);
  }

  strcpy (path, name);
  sp = strrchr (path, '/');
  if (sp)
    *sp = '\0';
  else
    strcpy (path, ".");

  if ((fb_realpath (path, cwdir)) == NULL) {
    perror_reply (553, "Could not determine cwdir");
    return (-1);
  }

  if ((wu_realpath (home, abspwdir, chroot_path)) == NULL) {
    perror_reply (553, "Could not determine pwdir");
    return (-1);
  }

  if ((fb_realpath (home, relpwdir)) == NULL) {
    perror_reply (553, "Could not determine pwdir");
    return (-1);
  }

  /*
  *  we are doing a "best match"... ..so we keep track of what "match
  *  value" we have received so far...
  */
  while (getaclentry ("upload", &entry)) {
    char *q;
    int i = 0;
    int options = 1;
    int classfound = 0;
    int classmatched = 0;
    pwdir = abspwdir;
    while (options
    &&     (i < MAXARGS)
    &&     ((q = entry->arg[i]) != (char *)NULL)
    &&     (q[0] != '\0')) {
      if (strcasecmp (q, "absolute") == 0) {
        i++;
        pwdir = abspwdir;
      } else if (strcasecmp (q, "relative") == 0) {
        i++;
        pwdir = relpwdir;
      } else if (strncasecmp (q, "class=", 6) == 0) {
        i++;
          classfound = 1;
        if (strcasecmp (q+6, class) == 0)
          classmatched = 1;
      } else if (strcmp (q, "-") == 0) {
        i++;
        options = 0;
      } else
        options = 0;
    }
    if (!classfound || classmatched) {
      int j;
      if (((i + 1) < MAXARGS)
      &&  ((q = entry->arg[i]) != (char *) NULL)
      &&  (q[0] != '\0')
      &&  (   (0 == strcmp       (q, "*"  ))
           || (0 == strcmp       (q, pwdir))
           || (0 <  path_compare (q, pwdir)))
      && ((j = path_compare (entry->arg[i+1], cwdir)) >= match_value)) {
        match_value = j;

        ap2 = NULL;
        if (((i + 2) < MAXARGS)
        &&  ((q = entry->arg[i+2]) != (char *) NULL)
        &&  (q[0] != '\0'))
          ap2 = q;

        ap3 = NULL;
        if (((i + 3) < MAXARGS)
        &&  ((q = entry->arg[i+3]) != (char *) NULL)
        &&  (q[0] != '\0'))
          ap3 = q;

        ap4 = NULL;
        if (((i + 4) < MAXARGS)
        &&  ((q = entry->arg[i+4]) != (char *) NULL)
        &&  (q[0] != '\0'))
          ap4 = q;

        ap5 = NULL;
        if (((i + 5) < MAXARGS)
        &&  ((q = entry->arg[i+5]) != (char *) NULL)
        &&  (q[0] != '\0'))
          ap5 = q;
      }
    }
  }

  if (ap3
  &&  (   (!strcasecmp ("dirs",   ap3))
       || (!strcasecmp ("nodirs", ap3))))
    ap3 = NULL;

  /*
  *  if we did get matches ... else don't do any of this stuff
  */
  if (match_value >= 0) {
    if (!strcasecmp (ap2, "yes")) {
      if ((ap3 && *ap3 == '*')
      ||  (ap4 && *ap4 == '*'))
        stat_result = stat (path, &stbuf);
      if (ap3)
        if ((ap3[0] != '*')
        ||  (ap3[1] != '\0'))
          *uid = atoi (ap3);    /* the uid  */
        else
          if (stat_result == 0)
            *uid = stbuf.st_uid;
      if (ap4) {
        if ((ap4[0] != '*')
        ||  (ap4[1] != '\0'))
          *gid = atoi (ap4);    /* the gid  */
        else
          if (stat_result == 0)
            *gid = stbuf.st_gid;
        *valid = 1;
      }
      if (ap5)
        sscanf (ap5, "%o", f_mode); /* the mode */
    } else {
      reply (553, "%s: Permission denied on server. (Upload)", name);
      return (-1);
    }
  } else {
    /*
    *  upload defaults to "permitted"
    */
    /* Not if anonymous */
    if (anonymous) {
      reply (553, "%s: Permission denied on server. (Upload)", name);
      return (-1);
    }
    return (1);
  }

  return (match_value);
}

int
#ifdef __STDC_
del_check(char *name)
#else
del_check(name)
char *name;
#endif
{
  int pdelete = (anonymous? 0 : 1);
  struct aclmember *entry = NULL;

  while (getaclentry("delete", &entry) && ARG0 && ARG1 != NULL) {
      if (type_match(ARG1))
          if (anonymous) {
              if (*ARG0 == 'y')
                  pdelete = 1;
          } else
          if (*ARG0 == 'n')
              pdelete = 0;
  }
  
/* H* fix: no deletion, period. You put a file here, I get to look at it. */
#ifdef PARANOID
  pdelete = 0;
#endif

  if (!pdelete) {
      reply(553, "%s: Permission denied on server. (Delete)", name);
      return(0);
  } else {
      return(1);
  }
}

/* The following is from the Debian add-ons. */

#define lbasename(x) (strrchr(x,'/')?1+strrchr(x,'/'):x)

int
#ifdef __STDC__
regexmatch(char *name, char *rgexp)
#else
regexmatch(name, rgexp)
char *name, *rgexp;
#endif
{
  
#ifdef M_UNIX
# ifdef REGEX
  char *regp;
# endif
#endif
  
#ifdef REGEXEC
  regex_t regexbuf;
  regmatch_t regmatchbuf;
#else
  char *sp;
#endif

#if defined(REGEXEC)
      if (regcomp(&regexbuf, rgexp, REG_EXTENDED) != 0) {
          reply(553, "REGEX error");
#elif defined(REGEX)
      if ((sp = regcmp(rgexp, (char *) 0)) == NULL) {
          reply(553, "REGEX error");
#else
      if ((sp = re_comp(rgexp)) != 0) {
          perror_reply(553, sp);
#endif
          return(0);
      }
  
#if defined(REGEXEC)
      if (regexec(&regexbuf, name, 1, &regmatchbuf, 0) != 0) {
#elif defined(REGEX)
# ifdef M_UNIX
      regp = regex(sp, name);
      free(sp);
      if (regp == NULL) {
# else
      if ((regex(sp, name)) == NULL) {
# endif
#else
      if ((re_exec(name)) != 1) {
#endif
              return(0);
      }
      return(1);
}

static int
#ifdef __STDC__
allow_retrieve (char *name)
#else
allow_retrieve (name)
char *name;
#endif
{
  char realname [MAXPATHLEN + 1];
  char localname [MAXPATHLEN + 1];
  char *whichname;
  int i;
  size_t len;
  struct aclmember *entry = NULL;
  char *p, *q;
  int options;
  int classfound;
  int classmatched;
  char class [1024];

  (void) acl_getclass (class);
  if ((name == (char *)NULL)
  ||  (*name == '\0'))
    return 0;
  fb_realpath (name, localname);
  wu_realpath (name, realname, chroot_path);
  while (getaclentry ("allow-retrieve", &entry)) {
    whichname = realname;
    i = 0;
    options = 1;
    classfound = 0;
    classmatched = 0;
    while (options
    &&     (i < MAXARGS)
    &&     ((q = entry->arg[i]) != (char *)NULL)
    &&     (q[0] != '\0')) {
      if (strcasecmp (q, "absolute") == 0) {
        i++;
        whichname = realname;
      } else if (strcasecmp (q, "relative") == 0) {
        i++;
        whichname = localname;
      } else if (strncasecmp (q, "class=", 6) == 0) {
        i++;
        classfound = 0;
        if (strcasecmp (q+6, class) == 0)
          classmatched = 1;
      } else if (strcmp (q, "-") == 0) {
        i++;
        options = 0;
      } else
        options = 0;
      }
    if (!classfound || classmatched) {
      for ( ; (i < MAXARGS) && ((q = entry->arg[i]) != (char *)NULL) && (q[0] != '\0'); i++) {
        len = strlen (q);
        p = (q[0] == '/') ? whichname : lbasename (whichname);
        if ((   (q[0] == '/')
             && (q[len - 1] == '/')
             && (strncmp (q, p, len) == 0))
        ||   (strcmp (p, q) == 0)
        ||   !fnmatch (p,q,NULL)) {
          return 1;
        }
      }
    }
  }
  return 0;
}

int
#ifdef __STDC__
checknoretrieve (char *name)
#else
checknoretrieve (name)
char *name;
#endif
{
  char realname [MAXPATHLEN + 1];
  char localname [MAXPATHLEN + 1];
  char *whichname;
  int i;
  size_t len;
  struct aclmember *entry = NULL;
  char *p, *q;
  int options;
  int classfound;
  int classmatched;
  char class [1024];

  (void) acl_getclass (class);
  if ((name == (char *)NULL)
  ||  (*name == '\0'))
    return 0;
  fb_realpath (name, localname);
  wu_realpath (name, realname, chroot_path);
  while (getaclentry ("noretrieve", &entry)) {
    whichname = realname;
    i = 0;
    options = 1;
    classfound = 0;
    classmatched = 0;
    while (options
    &&     (i < MAXARGS)
    &&     ((q = entry->arg[i]) != (char *)NULL)
    &&     (q[0] != '\0')) {
      if (strcasecmp (q, "absolute") == 0) {
        i++;
        whichname = realname;
      } else if (strcasecmp (q, "relative") == 0) {
        i++;
        whichname = localname;
      } else if (strncasecmp (q, "class=", 6) == 0) {
        i++;
        classfound = 1;
        if (strcasecmp (q+6, class) == 0)
          classmatched = 1;
      } else if (strcmp (q, "-") == 0) {
        i++;
        options = 0;
      } else
        options = 0;
      }
    if (!classfound || classmatched) {
      for ( ; (i < MAXARGS) && ((q = entry->arg[i]) != (char *)NULL) && (q[0] != '\0'); i++) {
        len = strlen (q);
        p = (q[0] == '/') ? whichname : lbasename (whichname);
        if ((   (q[0] == '/')
             && (q[len - 1] == '/')
             && (strncmp (q, p, len) == 0))
        ||   (strcmp (p, q) == 0)
        ||   !fnmatch (p,q,NULL)) {
          if (!allow_retrieve (name)) {
            reply (550, "%s is marked unretrievable", localname);
            return 1;
          }
        }
      }
    }
  }
  return 0;
}

#ifdef QUOTA

#ifndef MNTMAXSTR
#define MNTMAXSTR 2048 /* And hope it's enough */
#endif

#ifdef QUOTA_DEVICE 

int
#ifdef __STDC__
path_to_device (char *pathname,char *result)
#else
path_to_device (pathname, result)
     char *pathname, *result;
#endif
 
{
  FILE *fp;
#ifdef HAS_OLDSTYLE_GETMNTENT
  struct mnttab static_mp;
  struct mnttab *mp = &static_mp;
#else
  struct mntent *mp;
#endif
  struct mount_ent
    {
      char mnt_fsname[MNTMAXSTR], mnt_dir[MNTMAXSTR];
      struct mount_ent *next;
    }
  mountent;
  struct mount_ent *current, *start, *new;
  char path[1024], mnt_dir[1024], *pos;
  int flag = 1;

#ifdef HAS_OLDSTYLE_GETMNTENT
#endif

  start = current = NULL;
#ifdef HAS_OLDSTYLE_GETMNTENT
  fp = fopen (MNTTAB, "r");
#else
  fp = setmntent (MNTTAB, "r");
#endif
  if (fp == NULL)
    return 0;
#ifdef HAS_OLDSTYLE_GETMNTENT
  while (getmntent(fp, &static_mp)==0)
#else
  while (mp = getmntent (fp))
#endif
    {
      if (!(new = (struct mount_ent *) malloc (sizeof (mountent))))
       {
         perror ("malloc");
         flag = 0;
break;
       }

      if (!start)
       start = current = new;
      else
       current = current->next = new;

#ifdef HAS_OLDSTYLE_GETMNTENT
      strncpy (current->mnt_fsname, mp->mnt_special, strlen (mp->mnt_special) + 1);
      strncpy (current->mnt_dir, mp->mnt_mountp, strlen (mp->mnt_mountp) + 1);
#else
      strncpy (current->mnt_fsname, mp->mnt_fsname, strlen (mp->mnt_fsname) + 1);
      strncpy (current->mnt_dir, mp->mnt_dir, strlen (mp->mnt_dir) + 1);
#endif
    }
#ifdef HAS_OLDSTYLE_GETMNTENT
  fclose (fp);
#else
  endmntent (fp);
#endif
  current->next = NULL;

  wu_realpath (pathname, path, chroot_path);

  while (*path && flag)
    {
      current = start;
      while (current && flag)
       {
         if (strcmp (current->mnt_dir, "swap"))
           {
             wu_realpath (current->mnt_dir, mnt_dir, chroot_path);
             if (!strcmp (mnt_dir, path))
               {
                 flag = 0;
                 /* no support for remote quota yet */
                 if (!index (current->mnt_fsname, ':'))
                   strcpy (result, current->mnt_fsname);
               }
           }
         current = current->next;
       }
      if (!((pos = strrchr (path, '/')) - path) && strlen (path) > 1)
       strcpy (path, "/");
      else
       path[pos - path] = '\0';
    }
    while (current)
    {
      new = current->next;
      free (current);
      current = new;
    }
  return 1;
}
#endif

void
#ifdef __STDC__
get_quota(char *fs,int uid)
#else
get_quota(fs, uid)
char *fs;
int uid;
#endif
{
       char mnt_fsname[MNTMAXSTR];
#ifdef HAS_NO_QUOTACTL
    int dirfd;
    struct quotctl qp;

    if (path_to_device (fs, mnt_fsname)) {
	dirfd = open(fs, O_RDONLY);
	qp.op = Q_GETQUOTA;
	qp.uid = uid;
	qp.addr = (char *) &quota;
	ioctl(dirfd, Q_QUOTACTL, &qp);
	close(dirfd);
    }
#else
#ifdef QUOTA_DEVICE

       if (path_to_device (fs, mnt_fsname))
#ifdef LINUX
         quotactl (QCMD(Q_GETQUOTA,USRQUOTA), mnt_fsname, uid, (char *) &quota);
#else
         quotactl (Q_GETQUOTA, mnt_fsname, uid, (char *) &quota);
#endif
#else
  quotactl(fs,QCMD(Q_GETQUOTA,USRQUOTA),uid,(char*)&quota);
#endif
#endif /* HAS_NO_QUOTACTL */
}
 
char
#ifdef __STDC__
*time_quota(long curstate, long softlimit, long timelimit, char *timeleft)
#else
*time_quota(curstate, softlimit, timelimit, timeleft)
long curstate, softlimit, timelimit;
char *timeleft;
#endif
{
       struct timeval tv;

       gettimeofday(&tv, NULL);
       if (softlimit && curstate >= softlimit) {
               if (timelimit == 0) {
                       strcpy(timeleft, "NOT STARTED");
               } else if (timelimit > tv.tv_sec) {
                       fmttime(timeleft, timelimit - tv.tv_sec);
               } else {
                       strcpy(timeleft, "EXPIRED");
               }
       } else {
               timeleft[0] = '\0';
       }
       return(timeleft);
}

int
#ifdef __STDC__
fmttime(char *buf, register long time)
#else
fmttime(buf, time)
       char *buf;
       register long time;
#endif
{
       int i;
       static struct {
               int c_secs;             /* conversion units in secs */
               char * c_str;           /* unit string */
       } cunits [] = {
               {60*60*24*28, "months"},
               {60*60*24*7, "weeks"},
               {60*60*24, "days"},
               {60*60, "hours"},
               {60, "mins"},
               {1, "secs"}
       };

       if (time <= 0) {
               strcpy(buf, "EXPIRED");
               return;
       }
       for (i = 0; i < sizeof(cunits)/sizeof(cunits[0]); i++) {
               if (time >= cunits[i].c_secs)
                       break;
       }
       sprintf(buf, "%.1f %s", (double)time/cunits[i].c_secs, cunits[i].c_str);
}
#endif

#ifdef QUOTA
#ifdef LINUX
/* I have no idea why I can't find 'quotactl()' in my libs, here's the source - GAL */

/*
 * QUOTA    An implementation of the diskquota system for the LINUX
 *          operating system. QUOTA is implemented using the BSD systemcall
 *          interface as the means of communication with the user level.
 *          Should work for all filesystems because of integration into the
 *          VFS layer of the operating system.
 *          This is based on the Melbourne quota system wich uses both user and
 *          group quota files.
 *
 *          System call interface.
 *
 * Version: $Id: quotactl.c,v 2.3 1995/07/23 09:58:06 mvw Exp mvw $
 *
 * Author:  Marco van Wieringen <mvw@planets.ow.nl> <mvw@tnix.net>
 *
 *          This program is free software; you can redistribute it and/or
 *          modify it under the terms of the GNU General Public License
 *          as published by the Free Software Foundation; either version
 *          2 of the License, or (at your option) any later version.
 */
#if defined(__alpha__)
#include <errno.h>
#include <sys/types.h>
#include <syscall.h>
#include <asm/unistd.h>

int quotactl(int cmd, const char * special, int id, caddr_t addr)
{
	return syscall(__NR_quotactl, cmd, special, id, addr);
}
#else
#include <sys/types.h>
#define __LIBRARY__
#include <linux/unistd.h>

_syscall4(int, quotactl, int, cmd, const char *, special, int, id, caddr_t, addr);
#endif
#endif
#endif

#ifdef THROUGHPUT

int
#ifdef __STDC__
file_compare(char *patterns, char *file)
#else
file_compare(patterns, file)
    char *patterns;
    char *file;
#endif
{
    char buf[MAXPATHLEN];
    char *cp;
    char *cp2;
    int i;
    int matches = 0;

    strncpy(buf, patterns, sizeof(buf));
    buf[sizeof(buf)-1] = '\0';
    i = strlen(buf);
    buf[i++] = ',';
    buf[i++] = '\0';

    cp = buf;
    while ((cp2 = strchr(cp, ',')) != NULL) {
        *cp2++ = '\0';
        if (fnmatch(cp, file, 0) == 0) {
            matches = 1;
            break;
        }
    }
    return matches;
}

int
#ifdef __STDC__
remote_compare(char *patterns)
#else
remote_compare(patterns)
    char *patterns;
#endif
{
    char buf[MAXPATHLEN];
    char *cp;
    char *cp2;
    int i;
    int matches = 0;

    strncpy(buf, patterns, sizeof(buf));
    buf[sizeof(buf)-1] = '\0';
    i = strlen(buf);
    buf[i++] = ',';
    buf[i++] = '\0';

    cp = buf;
    while ((cp2 = strchr(cp, ',')) != NULL) {
        *cp2++ = '\0';
        if (hostmatch (cp, remoteaddr, remotehost)) {
            matches = 1;
            break;
        }
    }
    return matches;
}

void
#ifdef __STDC__
throughput_calc(char *name, int *bps, double *bpsmult)
#else
throughput_calc(name,bps,bpsmult)
    char *name;
    int *bps;
    double *bpsmult;
#endif
{
    int match_value = -1;
    char cwdir[MAXPATHLEN];
    char pwdir[MAXPATHLEN];
    char path[MAXPATHLEN];
    char file[MAXPATHLEN];
    char x[MAXPATHLEN];
    char *ap1 = NULL, *ap2 = NULL, *ap3 = NULL, *ap4 = NULL;
    struct aclmember *entry = NULL;
    extern struct passwd *pw;
    extern char *home;
    char *sp;
    int i;

    /* default is maximum throughput */
    *bps = -1;
    *bpsmult = 1.0;

  /* XXX We could use dynamic RAM to store this path, but I'd rather just bail
     out with an error. The rest of wu is so crufy that a long path might
     just blow up later */

  if ((strlen(name) + 1) > sizeof(path)) {
    return;
  }

    /* what's our current directory? */
    strcpy(path, name);
    if (sp = strrchr(path, '/'))
        *sp = '\0';
    else
        strcpy(path, ".");
    if (sp = strrchr(name, '/'))
        strcpy(file, sp + 1);
    else
        strcpy(file, name);
    if ((fb_realpath(path, cwdir)) == NULL) {
        return;
    }

    wu_realpath(home, pwdir, chroot_path);

    /* find best matching entry */
    while (getaclentry("throughput", &entry) && ARG0 && ARG1 && ARG2 && ARG3 && ARG4 && ARG5 != NULL) {
      if (   (   (0 == strcmp       (ARG0, "*"  ))
              || (0 == strcmp       (ARG0, pwdir))
              || (0 <  path_compare (ARG0, pwdir)))
          && ( ( i = path_compare (ARG1, cwdir) ) >= match_value )
         ) {
            if (file_compare(ARG2, file)) {
                if (remote_compare(ARG5)) {
                    match_value = i;
                    ap3 = ARG3;
                    ap4 = ARG4;
                }
            }
        }
    }

    /* if we did get matches */
    if (match_value >= 0) {
        if (strcasecmp(ap3, "oo") == 0)
            *bps = -1;
        else
            *bps = atoi(ap3);
        if (strcmp(ap4, "-") == 0)
            *bpsmult = 1.0;
        else
            *bpsmult = atof(ap4);
    }
    return;
}

void
#ifdef __STDC__
throughput_adjust(char *name)
#else
throughput_adjust(name)
    char *name;
#endif
{
    int match_value = -1;
    char pwdir[MAXPATHLEN];
    char cwdir[MAXPATHLEN];
    char path[MAXPATHLEN];
    char file[MAXPATHLEN];
    char buf[MAXPATHLEN];
    char *ap1 = NULL, *ap2 = NULL, *ap3 = NULL, *ap4 = NULL, *ap5 = NULL;
    char **pap3;
    struct aclmember *entry = NULL;
    extern struct passwd *pw;
    extern char *home;
    char *sp;
    int i;

  /* XXX We could use dynamic RAM to store this path, but I'd rather just bail
     out with an error. The rest of wu is so crufy that a long path might
     just blow up later */

  if ((strlen(name) + 1) > sizeof(path)) {
    return;
  }

    /* what's our current directory? */
    strcpy(path, name);
    if (sp = strrchr(path, '/'))
        *sp = '\0';
    else
        strcpy(path, ".");
    if (sp = strrchr(name, '/'))
        strcpy(file, sp + 1);
    else
        strcpy(file, name);
    if ((fb_realpath(path, cwdir)) == NULL) {
        return;
    }

    wu_realpath(home, pwdir, chroot_path);

    /* find best matching entry */
    while (getaclentry("throughput", &entry) && ARG0 && ARG1 && ARG2 && ARG3 && ARG4 && ARG5 != NULL) {
      if (   (   (0 == strcmp       (ARG0, "*"  ))
              || (0 == strcmp       (ARG0, pwdir))
              || (0 <  path_compare (ARG0, pwdir)))
          && ( ( i = path_compare (ARG1, cwdir) ) >= match_value )
         ) {
            if (file_compare(ARG2, file)) {
                if (remote_compare(ARG5)) {
                    match_value = i;
                    ap3 = ARG3;
                    pap3 = &ARG3;
                    ap4 = ARG4;
                }
            }
        }
    }

    /* if we did get matches */
    if (match_value >= 0) {
        if (strcasecmp(ap3, "oo") != 0) {
            if (strcmp(ap4, "-") != 0) {
                sprintf(buf, "%.0f", atoi(ap3) * atof(ap4));
                *pap3 = (char *) malloc(strlen(buf) + 1);
if (*pap3 == NULL) {
  syslog (LOG_ERR, "malloc error in throughput_adjust");
  exit (0);
}
                strcpy(*pap3, buf);
            }
        }
    }
    return;
}

#endif

static int CheckMethod = 0;

void
#ifdef __STDC__
SetCheckMethod (const char *method)
#else
SetCheckMethod (method)
    char *method;
#endif
{
    if ((strcasecmp (method, "md5"    ) == 0)
    ||  (strcasecmp (method, "rfc1321") == 0))
        CheckMethod = 0;
    else if ((strcasecmp (method, "crc"  ) == 0)
    ||       (strcasecmp (method, "posix") == 0))
        CheckMethod = 1;
    else {
        reply (500, "Unrecognized checksum method");
        return;
    }
    switch (CheckMethod) {
    default: reply (200, "Checksum method is now: MD5 (RFC1321)"); break;
    case 1:  reply (200, "Checksum method is now: CRC (POSIX)");   break;
    }
}

void
#ifdef __STDC__
ShowCheckMethod (void)
#else
ShowCheckMethod ()
#endif
{
    switch (CheckMethod) {
    default: reply (200, "Current checksum method: MD5 (RFC1321)"); break;
    case 1:  reply (200, "Current checksum method: CRC (POSIX)");   break;
    }
}

void
#ifdef __STDC__
CheckSum (const char *pathname)
#else
CheckSum (pathname)
    char *pathname;
#endif
{
    char *cmd;
    char buf [MAXPATHLEN];
    FILE* cmdf;
    FILE* ftpd_popen();
    struct stat st;

    if (stat (pathname, &st) == 0) {
        if ((st.st_mode & S_IFMT) != S_IFREG) {
            reply (500, "%s: not a plain file.", pathname);
            return;
        }
    } else {
        perror_reply (550, pathname);
        return;
    }

    switch (CheckMethod) {
    default: cmd = "/bin/md5sum"; break;
    case 1:  cmd = "/bin/cksum";  break;
    }

    if (strlen(cmd) + 1 + strlen (pathname) + 1 > sizeof(buf)) {
        reply (500, "Pathname too long");
        return;
    }
    sprintf(buf, "%s %s", cmd, pathname);

    cmdf = ftpd_popen(buf, "r", 0);
    if (!cmdf) {
        perror_reply(550, cmd);
    } else {
        if (fgets(buf, sizeof buf, cmdf)) {
            char *crptr = strchr (buf, '\n');
            if (crptr != NULL)
                *crptr = '\0';
            reply(200, "%s", buf);
        }
        ftpd_pclose(cmdf);
    }
}

void
#ifdef __STDC__
CheckSumLastFile (void)
#else
CheckSumLastFile ()
#endif
{
    extern char LastFileTransferred [];

    if (LastFileTransferred [0] == '\0')
        reply (500, "Nothing transferred yet");
    else
        CheckSum (LastFileTransferred);
}
