/* 
   Unix SMB/Netbios implementation.
   Version 3.0
   NBT netbios routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Module name: nameelect.c

   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   04 jul 96: lkcl@pires.co.uk
   added system to become a master browser by stages.

   30 July 96: David.Chappell@mail.trincoll.edu
   Expanded multiple workgroup domain master browser support.

*/

#include "includes.h"          

extern int ServerNMB;
extern int ClientDGRAM;

extern int DEBUGLEVEL;

extern struct in_addr ipgrp;

/* here are my election parameters */

extern time_t StartupTime;

extern struct subnet_record *subnetlist;

extern char *source_description[];


/*******************************************************************
  what to do if a master browser DOESN't exist
  ******************************************************************/
static void browser_gone(struct subnet_record *d,
						char *work_name, char *work_scope)
{
	struct work_record *work=find_workgroupstruct(d,work_name,work_scope,False);

	/* i don't know about this workgroup, therefore i don't care */
	if (!work || !d) return;

	/* don't do election stuff on the WINS subnet */
	if (ip_equal(d->bcast_ip,ipgrp)) return;

	if (brse_local_master(work->token))
	{
		DEBUG(2,("Forcing election on %s %s\n",
		           work->work_group,inet_ntoa(d->bcast_ip)));

		/* we can attempt to become master browser */
		work->needelection = True;
	}
	else
	{
		/* force an election. is it our job to do this? windows is
		   known to force elections if it detects problems...
         */
		send_election(d, work, 0, 0);

		/* only removes workgroup completely on a local interface */
		remove_workgroup(d, work,True);
	}
}

static void response_name_mst_chk(time_t timestamp, struct packet_struct *p,
                                   struct response_record *n);

/*******************************************************************
  occasionally check to see if the master browser is around

  this function results in browser_gone() being called,
  through response_name_mst_chk() if the query is negative.

  ******************************************************************/
void check_master_browser(time_t time_now)
{
  static time_t lastrun=0;
  struct subnet_record *d;

  if (!lastrun) lastrun = time_now;
  if (time_now < lastrun + CHECK_TIME_MST_BROWSE * 60)
    return;
  lastrun = time_now;

  for (d = subnetlist; d; d = d->next)
  {
    struct work_record *work;
    BOOL wins = ip_equal(d->bcast_ip,ipgrp);

    for (work = d->workgrouplist; work; work = work->next)
    {
	  struct nmb_name n;
	  make_nmb_name(&n,work->work_group, 0x1d, work->scope);

      /* if we are not the browse master of a workgroup, and we can't
         find a browser on the subnet, do something about it. */

      if (!AM_MASTER(work) && !ip_equal(d->bcast_ip,ipgrp))
      {
		  struct nmb_query_status *nmb_data;

		  nmb_data = (struct nmb_query_status *)(malloc(sizeof(*nmb_data)));

		  if (nmb_data)
		  {
			nmb_data->d = d;

            netbios_name_query(time_now, ServerNMB, response_name_mst_chk,
				  (void*)nmb_data, &n,
                   !wins,False,d->bcast_ip);
          }
      }
    }
  }
}


/****************************************************************************
  response from a name query to master browser check
  ****************************************************************************/
static void response_name_mst_chk(time_t timestamp, struct packet_struct *p,
                                   struct response_record *n)
{
	BOOL success = True;
	struct nmb_query_status *nmb_data = (struct nmb_query_status *)n->nmb_data;

	if (p)
	{
		struct nmb_packet *nmb = &p->packet.nmb;
		int rcode = nmb->header.rcode;
		char *rdata = nmb->answers->rdata;

		/* no action required here. it's when NO responses are received
		   that we need to do something. */

		if (rcode == 0 && rdata)
		{
		  DEBUG(4, ("Master browser exists for %s at %s (just checking!)\n",
					namestr(&n->name), inet_ntoa(n->send_ip)));
		}
		else
		{
			DEBUG(4, ("Negative reply for Master browse query. eek!\n"));
			success = False;
		}
	}
	else
    {
		/* if no response received, the master browser must have gone
		   down on that subnet, without telling anyone. */

		if (n->num_msgs == 0)
		{
			success = False;
		}
    }

	if (!success)
	{
		browser_gone(nmb_data->d, n->name.name, n->name.scope);
	}
}


/****************************************************************************
  send an election packet
  **************************************************************************/
void send_election(struct subnet_record *d, struct work_record *work,
				uint32 criterion, int timeup)
{
  pstring outbuf;
  char *p;
  struct nmb_name from, to;
  char *my_name;

  if (!d || !work) return;
  
  my_name = brse_server_alias(work->token);

  make_nmb_name(&from, my_name         , 0x0 , work->scope);
  make_nmb_name(&to  , work->work_group, 0x1e, work->scope);

  DEBUG(2,("Sending election to %s as %s ",
				inet_ntoa(d->bcast_ip),namestr(&from)));     
  DEBUG(2,("for workgroup %s\n", namestr(&to)));     

  bzero(outbuf,sizeof(outbuf));
  p = outbuf;
  CVAL(p,0) = ANN_Election; /* election */
  p++;

  CVAL(p,0) = (criterion == 0 && timeup == 0) ? 0 : ELECTION_VERSION;
  SIVAL(p,1,criterion);
  SIVAL(p,5,timeup*1000); /* ms - despite the spec */
  p += 13;
  strcpy(p,my_name);
  strupper(p);
  p = skip_string(p,1);
  
  send_mailslot_reply(BROWSE_MAILSLOT,ClientDGRAM,outbuf,PTR_DIFF(p,outbuf),
              &from, *iface_ip(d->bcast_ip),
              &to  , d->bcast_ip);
}


/****************************************************************************
  un-register a SELF name that got rejected.

  if this name happens to be rejected when samba is in the process
  of becoming a master browser (registering __MSBROWSE__, WORKGROUP(1d)
  or WORKGROUP(1b)) then we must stop being a master browser. sad.

  **************************************************************************/
void name_unregister_work(time_t time_now, struct subnet_record *d, int token,
			struct nmb_name *name, struct in_addr ip)
{
    struct work_record *work;

    char *work_name  = brse_domain_name (token);
    char *work_scope = brse_domain_scope(token);

    if (!(work = find_workgroupstruct(d,work_name,work_scope,False))) return;

    remove_netbios_name(time_now,&d->namelist,&d->names_last_modified,
	                    name,ip,False);

    if (!AM_MASTER(work) && !AM_DMBRSE(work)) return;

    if (ms_browser_name(name->name, name->name_type) ||
        (brse_workgroup_member(work->token) &&
         (name->name_type == 0x1d || name->name_type == 0x1b)))
    {
      int remove_type = 0;

      if (ms_browser_name(name->name, name->name_type))
        remove_type = SV_TYPE_MASTER_BROWSER|SV_TYPE_DOMAIN_MASTER;
      if (name->name_type == 0x1d)
        remove_type = SV_TYPE_MASTER_BROWSER;
      if (name->name_type == 0x1b)
        remove_type = SV_TYPE_DOMAIN_MASTER;
            
      become_nonmaster(time_now, d, work, remove_type);
    }
}


/****************************************************************************
  registers a name.

  if the name being added is a SELF name, we must additionally check
  whether to proceed to the next stage in samba becoming a master browser.

  **************************************************************************/
void name_register_work(time_t time_now, struct subnet_record *d, int token,
                struct nmb_name *name, struct nmb_ip *data,
				BOOL bcast)
{
  if (d)
  {
    DEBUG(4,("name_register_work: %s %d %s %s\n",
			inet_ntoa(d->bcast_ip), token, namestr(name),
			source_description[data->source]));
  }

  if (data->source == SELF)
  {
    char *work_name  = brse_domain_name (token);
    char *work_scope = brse_domain_scope(token);
    struct work_record *work=find_workgroupstruct(d,work_name,work_scope,False);

    add_netbios_entry(time_now, &d->namelist,&d->names_last_modified,
	                  name,data,!bcast,False);

    if (work)
    {
      if (work->state != MST_NONE && work->state != MST_BROWSER &&
          work->state != MST_DOMAIN)
      {
        /* samba is in the process of working towards master browser-ness.
           initiate the next stage.
         */
        become_master(time_now, d, work);
        return;
      }
    }
  }
}


/*******************************************************************
  become the master browser.

  this is done in stages. note that this could take a while, 
  particularly on a broadcast subnet, as we have to wait for
  the implicit registration of each name to be accepted.

  as each name is successfully registered, become_master() is
  called again, in order to initiate the next stage. 

  stage 1: was MST_NONE - go to MST_NONE and register ^1^2__MSBROWSE__^2^1.
  stage 2: was MST_WON  - go to MST_MSB  and register WORKGROUP(0x1d)
  stage 3: was MST_MSB  - go to MST_BROWSER and register WORKGROUP(0x1b)
  stage 4: was MST_BROWSER - go to MST_DOMAIN (do not pass GO, do not...)

  XXXX note: this code still does not cope with the distinction
  between different types of nodes, particularly between M and P
  nodes. that comes later.

  ******************************************************************/
void become_master(time_t time_now,
				struct subnet_record *d, struct work_record *work)
{
  pstring comment;

  /* domain type must be limited to domain enum + server type. it must
     not have SV_TYPE_SERVER or anything else with SERVER in it, else
     clients get confused and start thinking this entry is a server
     not a workgroup
   */
  uint32 domain_type = SV_TYPE_DOMAIN_ENUM|SV_TYPE_NT;
  struct nmb_ip nb;
  int nb_type;

  char *my_name   ;
  char *my_comment;

  struct nmb_name name;

  if (!work || !d) return;
  
  /* XXXX what to do about WINS pseudo-subnet? */
  nb.ip = *iface_ip(d->bcast_ip);
  nb.source = SELF;
  nb.death_time = GET_TTL(0);

  nb_type = ip_equal(ipgrp, d->bcast_ip) ? NB_HFLAG : NB_BFLAG;

  my_name    = brse_server_alias(work->token);
  my_comment = brse_server_comment(work->token);

  StrnCpy(comment, my_comment, 43);
  
  if (!brse_local_master(work->token))
  {
    DEBUG(1,("Should not become master for %s %s\n",
                    work->work_group,inet_ntoa(d->bcast_ip)));
    work->state = MST_NONE;
    return;
  }

  DEBUG(2,("Becoming master for %s %s (currently at stage %d)\n",
                    work->work_group,inet_ntoa(d->bcast_ip),work->state));
  
  switch (work->state)
  {
    case MST_NONE: /* while we were nothing but a server... */
    {
      DEBUG(3,("go to first stage: register ^1^2__MSBROWSE__^2^1\n"));
      work->state = MST_WON; /* ... an election win was successful */

      work->ElectionCriterion |= 0x5;

      /* update our server status */
      work->ServerType &= ~SV_TYPE_POTENTIAL_BROWSER;
      add_server_entry(time_now,d,work,my_name,work->ServerType,0,comment,True);

      make_nmb_name(&name, MSBROWSE,0x01,work->scope);
      /* add special browser name */
      nb.nb_flags = nb_type|NB_ACTIVE|NB_GROUP;
      add_my_name_entry(time_now, d,work->token,&name,&nb);

      /* DON'T do anything else after calling add_my_name_entry() */
      return;
    }
    case MST_WON: /* while nothing had happened except we won an election... */
    {
      DEBUG(3,("go to second stage: register as master browser\n"));
      work->state = MST_MSB; /* ... registering MSBROWSE was successful */

      /* add server entry on successful registration of MSBROWSE */
      add_server_entry(time_now,d,work,work->work_group,
                       domain_type,0,brse_server_alias(work->token),True);

      make_nmb_name(&name, work->work_group,0x1d,work->scope);
      /* add master name */
      nb.nb_flags = nb_type|NB_ACTIVE;
      add_my_name_entry(time_now, d,work->token,&name,&nb);
  
      /* DON'T do anything else after calling add_my_name_entry() */
      return;
    }
    case MST_MSB: /* while we were still only registered MSBROWSE state... */
    {
      DEBUG(3,("2nd stage complete: registered as master browser\n"));
      work->state = MST_BROWSER; /* ... registering WORKGROUP(1d) succeeded */

      /* update our server status */
      work->ServerType |= SV_TYPE_MASTER_BROWSER;
      add_server_entry(time_now,d,work,brse_server_alias(work->token),
                       work->ServerType,0,comment,True);

      if (work->serverlist == NULL) /* no servers! */
      {
        /* ask all servers on our local net to announce to us */
        announce_request(work, d->bcast_ip);
      }
      break;
   }

   case MST_BROWSER:
   {
      /* don't have to do anything: just report success */
      DEBUG(3,("3rd stage: become master browser!\n"));

      break;
   }

   case MST_DOMAIN_NONE:
   {
      if (brse_domain_master(work->token))
      {
        work->state = MST_DOMAIN_MEM; /* ... become domain member */
        DEBUG(3,("domain first stage: register as domain member\n"));

        make_nmb_name(&name, work->work_group,0x1e,work->scope);
        /* add domain member name */
        nb.nb_flags = nb_type|NB_ACTIVE|NB_GROUP;
        add_my_name_entry(time_now, d,work->token,&name,&nb);

        /* DON'T do anything else after calling add_my_name_entry() */
        return;
      }
      else
      {
        DEBUG(4,("samba not configured as a domain master.\n"));
      }
  
      break;
   }

   case MST_DOMAIN_MEM:
   {
      if (brse_domain_master(work->token))
      {
        work->state = MST_DOMAIN_TST; /* ... possibly become domain master */
        DEBUG(3,("domain second stage: register as domain master\n"));

        if (brse_domain_logons(work->token))
        {
          work->ServerType |= SV_TYPE_DOMAIN_MEMBER;
          add_server_entry(time_now,d,work,
                           my_name,work->ServerType,0,comment,True);
        }

        make_nmb_name(&name, work->work_group,0x1b,work->scope);
        /* add domain master name */
        nb.nb_flags = nb_type|NB_ACTIVE;
        add_my_name_entry(time_now, d,work->token,&name,&nb);

        /* DON'T do anything else after calling add_my_name_entry() */
        return;
      }
      else
      {
        DEBUG(4,("samba not configured as a domain master.\n"));
      }
  
      break;
    }

    case MST_DOMAIN_TST: /* while we were still a master browser... */
    {
      /* update our server status */
      if (brse_domain_master(work->token))
      {
        struct subnet_record *d1;
        uint32 update_type = 0;

        DEBUG(3,("domain third stage: samba is now a domain master.\n"));
        work->state = MST_DOMAIN; /* ... registering WORKGROUP(1b) succeeded */

        update_type |= DFLT_SERVER_TYPE | SV_TYPE_DOMAIN_MASTER | 
	  SV_TYPE_POTENTIAL_BROWSER;

        work->ServerType |= update_type;
        add_server_entry(time_now,d,work,
                 my_name,work->ServerType,0,comment,True);

        for (d1 = subnetlist; d1; d1 = d1->next)
        {
            struct work_record *w;
            if (ip_equal(d1->bcast_ip, d->bcast_ip)) continue;

            for (w = d1->workgrouplist; w; w = w->next)
            {
                struct server_record *s = find_server(w, my_name);
                if (strequal(w->work_group, work->work_group))
                {
                    w->ServerType |= update_type;
                }
                if (s)
                {
                    s->serv.type |= update_type;
                    DEBUG(4,("found server %s on %s: update to %8x\n",
                                    s->serv.name, inet_ntoa(d1->bcast_ip),
                                    s->serv.type));
                }
            }
        }
      }
  
      break;
    }

    case MST_DOMAIN:
    {
      /* don't have to do anything: just report success */
      DEBUG(3,("fifth stage: there isn't one yet!\n"));
      break;
    }
  }
}


/*******************************************************************
  unbecome the master browser. initates removal of necessary netbios 
  names, and tells the world that we are no longer a master browser.
  ******************************************************************/
void become_nonmaster(time_t time_now,
				struct subnet_record *d, struct work_record *work,
                int remove_type)
{
  int new_server_type = work->ServerType;

  pstring comment;

  char *my_name   ;
  char *my_comment;
  BOOL wins = ip_equal(d->bcast_ip, ipgrp);
  enum master_state state;
  struct in_addr ip;
  struct nmb_name name;

  if (!work || !d) return;
  
  /* XXXX what to do about WINS pseudo-subnet? */
  ip = *iface_ip(d->bcast_ip);

  my_name    = brse_server_alias  (work->token);
  my_comment = brse_server_comment(work->token);

  StrnCpy(comment, my_comment, 43);
  
  DEBUG(2,("Becoming non-master for %s alias %s\n",work->work_group, my_name));
  
  /* can only remove master or domain types with this function */
  remove_type &= SV_TYPE_MASTER_BROWSER|SV_TYPE_DOMAIN_MASTER;

  new_server_type &= ~remove_type;
  state = work->state;

  if (!(new_server_type & (SV_TYPE_MASTER_BROWSER|SV_TYPE_DOMAIN_MASTER)))
  {
    /* no longer a master browser of any sort */

    work->ServerType |= SV_TYPE_POTENTIAL_BROWSER;
    work->ElectionCriterion &= ~0x4;
    work->state = MST_NONE;

    make_nmb_name(&name, MSBROWSE,0x01,work->scope);
    remove_name_entry(time_now, d,work->token,&name,ip);
  }
  
  work->ServerType = new_server_type;

  if (!(work->ServerType & SV_TYPE_DOMAIN_MASTER))
  {
    if (state == MST_DOMAIN)
    {
      if (wins)
        work->state = MST_NONE;
      else
        work->state = MST_BROWSER;
    }
    make_nmb_name(&name, work->work_group,0x1b,work->scope);
    remove_name_entry(time_now, d,work->token,&name,ip);
  }

  if (!(work->ServerType & SV_TYPE_MASTER_BROWSER))
  {
    if (state >= MST_BROWSER)
      work->state = MST_NONE;
    make_nmb_name(&name, work->work_group,0x1d,work->scope);
    remove_name_entry(time_now, d,work->token,&name,ip);
  }

  /* update our internal records with our new server state */
  add_server_entry(time_now,d, work,
                   my_name, work->ServerType, 0, my_comment, True);

  /* announce change in status on local interface */
  if (!wins)
  {
    /* XXXX do we need also to do an announce with a time of zero
       if we are no longer a local master browser?
     */
    work->needannounce = True;
  }
}


/*******************************************************************
  run the election
  ******************************************************************/
void run_elections(time_t time_now)
{
  static time_t lastime = 0;
  
  struct subnet_record *d;
  
  /* send election packets once a second */
  if (lastime && time_now-lastime <= 0) return;
  
  lastime = time_now;
  
  for (d = subnetlist; d; d = d->next)
  {
    struct work_record *work;
    for (work = d->workgrouplist; work; work = work->next)
    {
      if (work->RunningElection)
      {
        send_election(d,work,work->ElectionCriterion,time_now-StartupTime);
          
        if (work->ElectionCount++ >= 4)
        {
          /* I won! now what :-) */
          DEBUG(2,(">>> Won election for %s on %s <<<\n",
               work->work_group,inet_ntoa(d->bcast_ip)));
          
          work->RunningElection = False;
          work->state = MST_NONE;

          become_master(time_now, d, work);
        }
      }
    }
  }
}


/*******************************************************************
  work out if I win an election
  ******************************************************************/
static BOOL win_election(time_t time_now,
				struct work_record *work,int version,uint32 criterion,
             int timeup,char *name)
{  
  int mytimeup = time_now - StartupTime;
  uint32 mycriterion = work->ElectionCriterion;
  char *my_name = brse_server_alias(work->token);

  DEBUG(4,("election comparison: %x:%x %x:%x %d:%d %s:%s\n",
	   version,ELECTION_VERSION,
	   criterion,mycriterion,
	   timeup,mytimeup,
	   name,my_name));

  if (version > ELECTION_VERSION) return(False);
  if (version < ELECTION_VERSION) return(True);
  
  if (criterion > mycriterion) return(False);
  if (criterion < mycriterion) return(True);

  if (timeup > mytimeup) return(False);
  if (timeup < mytimeup) return(True);

  if (strcasecmp(my_name,name) > 0) return(False);
  
  return(True);
}


/*******************************************************************
  process a election packet

  An election dynamically decides who will be the master. 
  ******************************************************************/
void process_election(struct packet_struct *p,char *buf)
{
  struct dgram_packet *dgram = &p->packet.dgram;
  struct in_addr ip = dgram->header.source_ip;
  struct subnet_record *d = find_subnet(ip);
  int version = CVAL(buf,0);
  uint32 criterion = IVAL(buf,1);
  int timeup = IVAL(buf,5)/1000;
  char *name = buf+13;
  struct work_record *work;

  if (!d) return;

  if (ip_equal(d->bcast_ip,ipgrp)) {
    DEBUG(3,("Unexpected election request from %s %s on WINS net\n",
	     name, inet_ntoa(p->ip)));
    return;
  }
  
  name[15] = 0;  

  DEBUG(3,("Election request from %s %s vers=%d criterion=%08x timeup=%d\n",
	   name,inet_ntoa(p->ip),version,criterion,timeup));
  
  
  for (work = d->workgrouplist; work; work = work->next)
  {
    if (!same_scope(dgram, brse_domain_scope(work->token))) continue;
    if (!brse_local_master(work->token)) continue;

    if (win_election(p->timestamp, work, version,criterion,timeup,name))
    {
        if (!work->RunningElection)
        {
          work->needelection = True;
          work->ElectionCount=0;
          work->state = MST_NONE;
        }
    }
    else
    {
        work->needelection = False;
          
        if (work->RunningElection || AM_MASTER(work))
        {
          work->RunningElection = False;
          DEBUG(3,(">>> Lost election on %s %s <<<\n",
               work->work_group,inet_ntoa(d->bcast_ip)));
          
          /* if we are the master then remove our masterly names */
          if (AM_MASTER(work))
          {
              become_nonmaster(p->timestamp, d, work,
                    SV_TYPE_MASTER_BROWSER|
                    SV_TYPE_DOMAIN_MASTER);
          }
        }
    }
  }
}


/****************************************************************************
  checks whether a browser election is to be run on any workgroup

  this function really ought to return the time between election
  packets (which depends on whether samba intends to be a domain
  master or a master browser) in milliseconds.

  ***************************************************************************/
BOOL check_elections(void)
{
  struct subnet_record *d;
  BOOL run_any_election = False;

  for (d = subnetlist; d; d = d->next)
    {
      struct work_record *work;
      for (work = d->workgrouplist; work; work = work->next)
    {
      run_any_election |= work->RunningElection;
      
      if (work->needelection && !work->RunningElection)
        {
          DEBUG(3,(">>> Starting election on %s %s <<<\n",
               work->work_group,inet_ntoa(d->bcast_ip)));
          work->ElectionCount = 0;
          work->RunningElection = True;
          work->needelection = False;
        }
    }
    }
  return run_any_election;
}

