/* 
   Unix SMB/Netbios implementation.
   Version 3.0
   NBT netbios routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
*/

#define WINS_LIST "wins.dat"

#define MAX_LOCAL_INTERFACES 16

#define GET_TTL(ttl) ((ttl)?MIN(ttl,lp_max_ttl()):lp_max_ttl())

/* NTAS uses 2, NT uses 1, WfWg uses 0 */
#define MAINTAIN_LIST    2
#define ELECTION_VERSION 1

/* maximum number of people winsd will support for proxy port 137 passing */
#define MAX_DGRAM_SIZE (576) /* tcp/ip datagram limit is 576 bytes */
#define MIN_DGRAM_SIZE 12

#define NMB_QUERY  0x20
#define NMB_STATUS 0x21

#define NMB_REG_MULTI   0x0f /* as defined by microsoft ;-) */
#define NMB_REG         0x05 /* see rfc1002.txt 4.2.2,3,5,6,7,8 */
#define NMB_REG_REFRESH 0x09 /* see rfc1002.txt 4.2.4 */
#define NMB_REL         0x06 /* see rfc1002.txt 4.2.9,10,11 */
#define NMB_WAIT_ACK    0x07 /* see rfc1002.txt 4.2.16 */
/* XXXX what about all the other types?? 0x1, 0x2, 0x3, 0x4, 0x8? */

#define FIND_SELF  0x01
#define FIND_WINS  0x02
#define FIND_LOCAL 0x04

/* NetBIOS flags */
#define NB_GROUP  0x80
#define NB_PERM   0x02
#define NB_ACTIVE 0x04
#define NB_CONFL  0x08
#define NB_DEREG  0x10
#define NB_BFLAG  0x00 /* broadcast node type */
#define NB_PFLAG  0x20 /* point-to-point node type */
#define NB_MFLAG  0x40 /* mixed bcast & p-p node type */
#define NB_HFLAG  0x60 /* microsoft 'hybrid' node type */
#define NB_FLGMSK 0x60


#define WG_BROWSE_SIGNATURE   0x017e
#define WG_BROWSE_VERSION     0xff08

#define HOST_BROWSE_SIGNATURE 0xaa55
#define HOST_BROWSE_VERSION   0x010f

#define WG_MAJOR_VERSION  0x03
#define WG_MINOR_VERSION  0x0a

#define HOST_MAJOR_VERSION  0x03 /* nt/as host version */
#define HOST_MINOR_VERSION  0x05 /* one better than current nt/as version */

#define REFRESH_TIME (15*60)
#define NAME_POLL_REFRESH_TIME (5*60)
#define NAME_POLL_INTERVAL 15

/* NetBIOS flag identifier */
#define NAME_PERMANENT(p) ((p) & NB_PERM)
#define NAME_ACTIVE(p)    ((p) & NB_ACTIVE)
#define NAME_CONFLICT(p)  ((p) & NB_CONFL)
#define NAME_DEREG(p)     ((p) & NB_DEREG)
#define NAME_GROUP(p)     ((p) & NB_GROUP)

#define NAME_BFLAG(p)     (((p) & NB_FLGMSK) == NB_BFLAG)
#define NAME_PFLAG(p)     (((p) & NB_FLGMSK) == NB_PFLAG)
#define NAME_MFLAG(p)     (((p) & NB_FLGMSK) == NB_MFLAG)
#define NAME_HFLAG(p)     (((p) & NB_FLGMSK) == NB_HFLAG)

/* server type identifiers */
#define AM_MASTER(work) (work->ServerType & SV_TYPE_MASTER_BROWSER)
#define AM_BACKUP(work) (work->ServerType & SV_TYPE_BACKUP_BROWSER)
#define AM_DMBRSE(work) (work->ServerType & SV_TYPE_DOMAIN_MASTER)

/* microsoft browser NetBIOS name */
#define MSBROWSE "\001\002__MSBROWSE__\002"

/* mail slots */
#define BROWSE_MAILSLOT    "\\MAILSLOT\\BROWSE"
#define NET_LOGON_MAILSLOT "\\MAILSLOT\\NET\\NETLOGON"

/* see rfc1002.txt 4.2.1.3 */
#define RR_TYPE_IP     0x01
#define RR_TYPE_NSRR   0x02
#define RR_TYPE_NULL   0x0a
#define RR_TYPE_NBIP   0x20
#define RR_TYPE_STATUS 0x21

#define RR_CLASS_IP    0x01

/* see rfc1002.txt 4.2.14 for name query RCODEs */
#define RCODE_QUERY_OK      0
#define RCODE_QUERY_NAM_ERR 3
#define RCODE_QUERY_CFT_ERR 7


/* see rfc1002.txt 4.2.6 for name registration RCODEs */
#define RCODE_REG_OK      0
#define RCODE_REG_ACT_ERR 6
#define RCODE_REG_CFT_ERR 7 /* not yet supported */


/* see rfc1002.txt 4.2.11 for name release RCODEs */
#define RCODE_REL_OK      0
#define RCODE_REL_ACT_ERR 6


enum name_source {LMHOSTS, REGISTER, SELF, DNS, DNSFAIL, EXPIRED};
enum node_type {B_NODE=0, P_NODE=1, M_NODE=2, NBDD_NODE=3};
enum packet_type {NMB_PACKET, DGRAM_PACKET};
enum master_state
{
   MST_NONE,
   MST_WON,
   MST_MSB,
   MST_BROWSER,
   MST_DOMAIN_NONE,
   MST_DOMAIN_MEM,
   MST_DOMAIN_TST,
   MST_DOMAIN
};

/* a netbios name structure */
struct nmb_name {
  char name[17];
  char scope[64];
  int name_type;
};

/* A server name and comment. */
struct server_identity
{
    char *name;
    char *comment;
    struct server_identity *next;
};

/* a netbios flags + ip address structure */
/* this is used for multi-homed systems and for internet group names */
struct nmb_ip
{
  struct in_addr ip; /* ip address of host that owns this name */
  uint16 nb_flags;      /* netbios flags */

  time_t death_time; /* time record must be removed (do not remove if 0) */
  time_t refresh_time; /* time record should be refreshed */

  enum name_source source; /* where the name came from */
};

/* this is the structure used for the local netbios name list */
struct name_record
{
  struct name_record *next;
  struct name_record *prev;

  struct nmb_name name;    /* the netbios name */
  struct nmb_ip *ip_flgs;  /* the ip + flags */
  int num_ips;             /* number of ip+flags entries */
};

/* browse and backup server cache for synchronising browse list */
struct browse_cache_record
{
    struct browse_cache_record *next;
    struct browse_cache_record *prev;

    char name [17];
    char scope[64];
    int type;
    char group[17];
    struct in_addr ip;
    time_t sync_time;
    BOOL synced;
    BOOL local;
};

/* this is used to hold the list of servers in my domain, and is */
/* contained within lists of domains */
struct server_record
{
  struct server_record *next;
  struct server_record *prev;

  struct server_info_struct serv;
  time_t death_time;  
};

/* a workgroup structure. it contains a list of servers */
struct work_record
{
  struct work_record *next;
  struct work_record *prev;

  struct server_record *serverlist;

  /* stage of development from non-master to master browser / domain master */
  enum master_state state;

  /* work group info */
  char    work_group[17];
  char    scope[64];
  int     token;        /* used when communicating with backup browsers */
  int     ServerType;

  /* announce info */
  time_t lastannounce_time;
  int announce_interval;
  BOOL    needannounce;


  /* election info */
  BOOL    RunningElection;
  BOOL    needelection;
  int     ElectionCount;
  uint32  ElectionCriterion;
};

/* a subnet structure. it contains a list of workgroups and netbios names*/

/* note that a subnet of 255.255.255.255 contains all the WINS netbios names.
   all communication from such nodes are on a non-broadcast basis: they
   are point-to-point (P nodes) or mixed point-to-point and broadcast
   (M nodes). M nodes use point-to-point as a preference, and will use
   broadcasting for certain activities, or will resort to broadcasting as a
   last resort, if the WINS server fails (users of wfwg will notice that their
   machine often freezes for 30 seconds at a time intermittently, if the WINS
   server is down).

   B nodes will have their own, totally separate subnet record, with their
   own netbios name set. these do NOT interact with other subnet records'
   netbios names, INCLUDING the WINS one (with an ip "address", so called,
   of 255.255.255.255)
*/

struct subnet_record
{
  struct subnet_record *next;
  struct subnet_record *prev;

  struct work_record *workgrouplist; /* list of workgroups */
  struct name_record *namelist;      /* list of netbios names */
  time_t names_last_modified;

  struct in_addr bcast_ip;
  struct in_addr mask_ip;
  struct in_addr myip;

};

/* a resource record */
struct res_rec {
  struct nmb_name rr_name;
  int rr_type;
  int rr_class;
  int ttl;
  int rdlength;
  char rdata[MAX_DGRAM_SIZE];
};

/* define a nmb packet. */
struct nmb_packet
{
  struct {
    int name_trn_id;
    int opcode;
    BOOL response;
    struct {
      BOOL bcast;
      BOOL recursion_available;
      BOOL recursion_desired;
      BOOL trunc;
      BOOL authoritative;
    } nm_flags;
    int rcode;
    int qdcount;
    int ancount;
    int nscount;
    int arcount;
  } header;

  struct {
    struct nmb_name question_name;
    int question_type;
    int question_class;
  } question;

  struct res_rec *answers;
  struct res_rec *nsrecs;
  struct res_rec *additional;
};


/* a datagram - this normally contains SMB data in the data[] array */
struct dgram_packet {
  struct {
    int msg_type;
    struct {
      enum node_type node_type;
      BOOL first;
      BOOL more;
    } flags;
    int dgm_id;
    struct in_addr source_ip;
    int source_port;
    int dgm_length;
    int packet_offset;
  } header;
  struct nmb_name source_name;
  struct nmb_name dest_name;
  int datasize;
  char data[MAX_DGRAM_SIZE];
};

/* define a structure used to queue packets. this will be a linked
 list of nmb packets */
struct packet_struct
{
  struct packet_struct *next;
  struct packet_struct *prev;
  struct in_addr ip;
  int port;
  int fd;
  time_t timestamp;
  enum packet_type packet_type;
  union {
    struct nmb_packet nmb;
    struct dgram_packet dgram;
  } packet;
};

/* initiated name queries recorded in this list to track any responses... */
/* sadly, we need to group everything together. i suppose that if this
   gets unwieldy, then a union ought to be considered. oh for c++... */
struct response_record
{
  struct response_record *next;
  struct response_record *prev;

  uint16 response_id;
  void (*fn)(time_t, struct packet_struct *, struct response_record *);

  struct nmb_name name;

  int quest_type; /* type of packet to be sent */
  int fd;
  int reply_port;

  void *nmb_data; /* pointer to dynamically allocated memory */

  struct nmb_ip nb;
  int num_ips;

  struct in_addr send_ip;

  BOOL bcast;
  BOOL recurse;

  int num_msgs;
  struct in_addr from_ip;

  time_t repeat_time;
  time_t repeat_interval;
  int    repeat_count;
};

/* a name query domain master browser server check structure */
/* a name query local master browser server check structure */
/* a name query sync server structure */
/* a name query find master structure */
/* a name status server check structure */
struct nmb_query_status
{
	struct subnet_record *d;
};

/* ids for netbios packet types */
#define ANN_HostAnnouncement         1
#define ANN_AnnouncementRequest      2
#define ANN_Election                 8
#define ANN_GetBackupListReq         9
#define ANN_GetBackupListResp       10
#define ANN_BecomeBackup            11
#define ANN_DomainAnnouncement      12
#define ANN_MasterAnnouncement      13
#define ANN_ResetBrowserState       14
#define ANN_LocalMasterAnnouncement 15


/* broadcast packet announcement intervals, in minutes */

/* search for master browsers of workgroups samba knows about, 
   except default */
#define CHECK_TIME_MST_BROWSE       5 

/* request backup browser announcements from other servers */
#define CHECK_TIME_ANNOUNCE_BACKUP 12

/* request host announcements from other servers: min and max of interval */
#define CHECK_TIME_MIN_HOST_ANNCE   3
#define CHECK_TIME_MAX_HOST_ANNCE  12

/* announce as Local Master to Domain Master */
#define CHECK_TIME_MST_ANNOUNCE    12

/* do all remote announcements this often */
#define REMOTE_ANNOUNCE_INTERVAL 180

#define DFLT_SERVER_TYPE (SV_TYPE_WORKSTATION | SV_TYPE_SERVER | \
			  SV_TYPE_TIME_SOURCE | SV_TYPE_SERVER_UNIX | \
			  SV_TYPE_PRINTQ_SERVER | SV_TYPE_SERVER_NT | \
			  SV_TYPE_NT)

