/*
   Unix SMB/Netbios implementation.
   Version 0.1
   WINS server routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   30 July 96: David.Chappell@mail.trincoll.edu
   Expanded multiple workgroup domain master browser support.

*/

#include "includes.h"

extern int DEBUGLEVEL;

extern pstring debugf;
pstring servicesf = CONFIGFILE;

int ServerNMB[MAX_LOCAL_INTERFACES]; /* wins server listening ports */
int ClientNMB[MAX_LOCAL_INTERFACES]; /* forwarding ports to nmbd */

extern int num_response_packets;

extern pstring myhostname;
static pstring host_file;

/* are we running as a daemon ? */
static BOOL is_daemon = False;

/* what server type are we currently */

time_t StartupTime =0;

struct name_record *namelist;
time_t names_last_modified=0;


 /****************************************************************************
  catch a sigterm
  ****************************************************************************/
static int sig_term()
{
  BlockSignals(True,SIGTERM);
  
  DEBUG(0,("Got SIGTERM: going down...\n"));
  
  /* write out wins.dat file if samba is a WINS server */
  write_netbios_names(time(NULL),WINS_LIST, namelist,&names_last_modified,True);
  
  exit(0);
}


/****************************************************************************
catch a sighup
****************************************************************************/
static int sig_hup(void)
{
  BlockSignals(True,SIGHUP);

  DEBUG(0,("Got SIGHUP\n"));
  write_netbios_names(time(NULL),WINS_LIST,namelist,&names_last_modified,True);
  reload_services(True);

  BlockSignals(False,SIGHUP);
#ifndef DONT_REINSTALL_SIG
  signal(SIGHUP,SIGNAL_CAST sig_hup);
#endif
  return(0);
}

/****************************************************************************
catch a sigpipe
****************************************************************************/
static int sig_pipe(void)
{
  BlockSignals(True,SIGPIPE);

  DEBUG(0,("Got SIGPIPE\n"));
  if (!is_daemon)
    exit(1);
  BlockSignals(False,SIGPIPE);
  return(0);
}

#if DUMP_CORE
/*******************************************************************
prepare to dump a core file - carefully!
********************************************************************/
static BOOL dump_core(void)
{
  char *p;
  pstring dname;
  strcpy(dname,debugf);
  if ((p=strrchr(dname,'/'))) *p=0;
  strcat(dname,"/corefiles");
  mkdir(dname,0700);
  sys_chown(dname,getuid(),getgid());
  chmod(dname,0700);
  if (chdir(dname)) return(False);
  umask(~(0700));

#ifndef NO_GETRLIMIT
#ifdef RLIMIT_CORE
  {
    struct rlimit rlp;
    getrlimit(RLIMIT_CORE, &rlp);
    rlp.rlim_cur = MAX(4*1024*1024,rlp.rlim_cur);
    setrlimit(RLIMIT_CORE, &rlp);
    getrlimit(RLIMIT_CORE, &rlp);
    DEBUG(3,("Core limits now %d %d\n",rlp.rlim_cur,rlp.rlim_max));
  }
#endif
#endif


  DEBUG(0,("Dumping core in %s\n",dname));
  return(True);
}
#endif


/****************************************************************************
possibly continue after a fault
****************************************************************************/
static void fault_continue(void)
{
#if DUMP_CORE
  dump_core();
#endif
}

/*******************************************************************
  expire old names from the namelist and server list
  ******************************************************************/
static void expire_names(time_t time_now)
{
  static time_t lastrun = 0;
  
  if (!lastrun) lastrun = time_now;
  if (time_now < lastrun + 15) return; /* give samba time to check its names */
  lastrun = time_now;

  /* this checks samba's NetBIOS names */
  check_expire_names(time_now); 
}

/*****************************************************************************
  reload the services file
  **************************************************************************/
BOOL reload_services(BOOL test)
{
  BOOL ret;
  extern fstring remote_machine;

  strcpy(remote_machine,"winsd");

  if (lp_loaded())
    {
      pstring fname;
      strcpy(fname,lp_configfile());
      if (file_exist(fname,NULL) && !strcsequal(fname,servicesf))
    {
      strcpy(servicesf,fname);
      test = False;
    }
    }

  if (test && !lp_file_list_changed())
    return(True);

  ret = lp_load(servicesf,True);

  /* perhaps the config filename is now set */
  if (!test) {
    DEBUG(3,("services not loaded\n"));
    reload_services(True);
  }

  load_interfaces();

  return(ret);
}



/****************************************************************************
load a netbios hosts file
****************************************************************************/
static BOOL load_hosts_file(time_t time_now, char *fname)
{
  FILE *f = fopen(fname,"r");
  pstring line;
  if (!f)
  {
    DEBUG(2,("Can't open lmhosts file %s\n",fname));
    return False;
  }

  while (!feof(f))
  {
      BOOL type_entered = False;
      pstring ip,name,flags,extra;
      struct nmb_name nb_name;
      char *ptr, *p;
      int count = 0;
      int type = 0x0;
      int nb_flgs = NB_PERM|NB_ACTIVE; /* lmhosts - permanent entries */
      struct in_addr ipaddr;
      enum name_source src = LMHOSTS;
      struct nmb_ip nb;

      if (!fgets_slash(line,sizeof(pstring),f)) continue;

      if (*line == '#') continue;

      strcpy(ip,"");
      strcpy(name,"");
      strcpy(flags,"");
      
      ptr = line;
      
      if (next_token(&ptr,ip   ,NULL)) ++count;
      if (next_token(&ptr,name ,NULL)) ++count;
      if (next_token(&ptr,flags,NULL)) ++count;
      if (next_token(&ptr,extra,NULL)) ++count;
      
      if (count <= 0) continue;
      
      if (count > 0 && count < 2) {
        DEBUG(0,("Ill formed hosts line [%s]\n",line));     
        continue;
      }
      
      if (count >= 4)
      {
        DEBUG(0,("too many columns in %s (obsolete syntax)\n",fname));
        continue;
      }
      
      DEBUG(4, ("lmhost entry: %s %s %s\n", ip, name, flags));
      
      /* XXXX need G (group) entry to be able to add static 0x1c names */
      if (strchr(flags,'G'))
      {
        nb_flgs |= NB_GROUP;
      }
      
      ipaddr = *interpret_addr2(ip);

      if (zero_ip(ipaddr) || count >= 4)
      {
        DEBUG(0,("zero ip obsolete in multi-interface winsd\n"));
        fclose(f);
        return True;
      }
     
      if ((p = strchr(name, '#')) != NULL)
      {
        type_entered = True;
        *p = 0;
        sscanf(p+1,"%x", &type);
      }

      /* XXXX need to read in scopes from hosts file */

      nb.ip = ipaddr;
      nb.source = src;
      nb.nb_flags = nb_flgs;
      nb.death_time = 0;

      make_nmb_name(&nb_name,name,type,NULL);
      add_netbios_entry(time_now, &namelist,&names_last_modified,
                        &nb_name,&nb,True,True);

      if (!type_entered && type == 0x0)
      {
        /* for 0x0 names, add 0x20 for back-compatibility with LANMANAGER */
        make_nmb_name(&nb_name,name,0x20,NULL);
        add_netbios_entry(time_now, &namelist,&names_last_modified,
                          &nb_name,&nb,True,True);
      }
    }
  
  fclose(f);
  return False;
}


/****************************************************************************
  listens for NMB packets, and queues them
  ***************************************************************************/
struct packet_struct *listen_for_nmb_packets(void)
{
	fd_set fds;
	int selrtn;
	int i;
	struct timeval timeout;

	FD_ZERO(&fds);

	for (i = 0; i < MAX_LOCAL_INTERFACES; i++)
    {
		if (ServerNMB[i] != -1) FD_SET(ServerNMB[i],&fds);
		if (ClientNMB[i] != -1) FD_SET(ClientNMB[i],&fds);
	}

	/* during elections and when expecting a netbios response packet we
	need to send election packets at tighter intervals 

	ideally it needs to be the interval (in ms) between time now and
	the time we are expecting the next netbios packet */

	timeout.tv_sec = num_response_packets ? 1 : WINSD_SELECT_LOOP;
	timeout.tv_usec = 0;

	selrtn = sys_select(&fds,&timeout);

	for (i = 0; i < MAX_LOCAL_INTERFACES; i++)
    {
		if (FD_ISSET(ClientNMB[i],&fds))
		{
			return read_packet(ClientNMB[i], NMB_PACKET);
		}
		if (FD_ISSET(ServerNMB[i],&fds))
		{
			return read_packet(ServerNMB[i], NMB_PACKET);
		}
	}

	return NULL;
}


/****************************************************************************
  The main select loop.
  ***************************************************************************/
static void process(void)
{
	while (True)
	{
		/* don't need a packet queue, like in nmbd */
		struct packet_struct *p = listen_for_nmb_packets();
		time_t t = time(NULL);

		if (p)
		{
			if (!process_nmb(p))
			{
				proxy_nmb_packet(t, p);
			}
			free_packet(p);
		}

		/* WINS server double-checking its name database */
		query_refresh_names(t);

		/* initiate expiring NetBIOS names */
		expire_names(t);

		/* remove and deal with NetBIOS responses not having been received */
		expire_netbios_response_entries(t);

		/* write names out to ascii file, if there have been any changes */
  		write_netbios_names(t,WINS_LIST, namelist, &names_last_modified, False);
	}
}


/****************************************************************************
  open the socket communication
****************************************************************************/
static BOOL open_sockets(BOOL isdaemon, int port)
{
	int i;
	int num_ifaces = iface_count();

	for (i = 0; i < MAX_LOCAL_INTERFACES; i++)
	{
		if (i >= num_ifaces)
		{
			ServerNMB[i] = -1;
		}
		else if (isdaemon)
		{
			int wins_port = port;
			uint32 addr  = interpret_addr(inet_ntoa(*iface_n_ip(i)));

			ServerNMB[i] = open_socket_in(SOCK_DGRAM,&wins_port,0,addr);
			if (ServerNMB[i] == -1) return(False);

			set_socket_options(ServerNMB[i],"SO_BROADCAST");
		}
		else
		{
			ServerNMB[i] = 0;
		}

	}

	for (i = 0; i < MAX_LOCAL_INTERFACES; i++)
	{
		if (i >= num_ifaces)
		{
			ClientNMB[i] = -1;
		}
		else if (isdaemon)
		{
			int nmb_port = 0;
			uint32 addr0 = interpret_addr(lp_socket_address());

			ClientNMB[i] = open_socket_in(SOCK_DGRAM,&nmb_port ,0,addr0);
			if (ClientNMB[i] == -1) return(False);

			set_socket_options(ClientNMB[i],"SO_BROADCAST");
		}
		else
		{
			ClientNMB[i] = 0;
		}

	}

	signal(SIGPIPE, SIGNAL_CAST sig_pipe);

	DEBUG(3,("Sockets opened.\n"));
	return True;
}


/****************************************************************************
  initialise connect, service and file structs
****************************************************************************/
static BOOL init_structs()
{
  extern struct nmb_name local_machine;

  /* XXXX must get correct scope */
  make_nmb_name(&local_machine, lp_server_alias(), 0x0, NULL);
  return True;
}

/****************************************************************************
usage on the program
****************************************************************************/
static void usage(char *pname)
{
  DEBUG(0,("Incorrect program usage - is the command line correct?\n"));

  printf("Usage: %s [-n name] [-D] [-p port] [-d debuglevel] [-l log basename]\n",pname);
  printf("Version %s\n",VERSION);
  printf("\t-D                    become a daemon\n");
  printf("\t-p port               listen on the specified port\n");
  printf("\t-d debuglevel         set the debuglevel\n");
  printf("\t-l log basename.      Basename for log/debug files\n");
  printf("\t-n netbiosname.       the netbios name to advertise for this host\n");
  printf("\t-H hosts file        load a netbios hosts file\n");
  printf("\n");
}


/****************************************************************************
  main program
  **************************************************************************/
 int main(int argc,char *argv[])
{
  int port = NMB_PORT;
  int opt;
  int i;

  extern FILE *dbf;
  extern char *optarg;

  time_t time_now;

  *host_file = 0;

  time_now = StartupTime = time(NULL);
  names_last_modified = StartupTime;

  TimeInit();

  strcpy(debugf,WINSLOGFILE);

  setup_logging(argv[0],False);

  charset_initialise();

#ifdef LMHOSTSFILE
  strcpy(host_file,LMHOSTSFILE);
#endif

  /* this is for people who can't start the program correctly */
  while (argc > 1 && (*argv[1] != '-')) {
    argv++;
    argc--;
  }

  fault_setup(fault_continue);

  signal(SIGHUP ,SIGNAL_CAST sig_hup);
  signal(SIGTERM,SIGNAL_CAST sig_term);

  while ((opt = getopt(argc, argv, "s:T:I:C:bAi:B:N:l:d:Dp:hSH:G:")) != EOF)
    {
      switch (opt)
    {
    case 's':
      strcpy(servicesf,optarg);
      break;      
    case 'N':
    case 'B':
    case 'I':
    case 'C':
    case 'G':
    case 'i':
      DEBUG(0,("Obsolete option '%c' used\n",opt));
      break;
    case 'H':
      strcpy(host_file,optarg);
      break;
    case 'l':
      sprintf(debugf,"%s.nmb",optarg);
      break;
    case 'D':
      is_daemon = True;
      break;
    case 'd':
      DEBUGLEVEL = atoi(optarg);
      break;
    case 'p':
      port = atoi(optarg);
      break;
    case 'h':
      usage(argv[0]);
      exit(0);
      break;
    default:
      if (!is_a_socket(0)) {
        usage(argv[0]);
      }
      break;
    }
    }

  DEBUG(1,("%s netbios nameserver version %s started\n",timestring(),VERSION));
  DEBUG(1,("Copyright Andrew Tridgell 1994-1996 Luke Leighton 1996\n"));

  get_myname(myhostname,NULL);

  init_structs();

  if (!reload_services(False))
    return(-1); 
  
  reload_services(True);

  if (!is_daemon && !is_a_socket(0))
  {
    DEBUG(0,("standard input is not a socket, assuming -D option\n"));
    is_daemon = True;
  }
  
  if (is_daemon)
  {
    DEBUG(2,("%s becoming a daemon\n",timestring()));
    become_daemon();
  }

  sleep(10);

  time_now = time(NULL);

  DEBUG(3,("Opening sockets %d\n", port));

  if (!open_sockets(is_daemon,port)) return 1;

  if (*host_file)
  {
    if (load_hosts_file(time_now, host_file)) return 1;
    DEBUG(3,("Loaded hosts file\n"));
  }

  load_netbios_names(time_now, &namelist, &names_last_modified);
  DEBUG(3,("Loaded names\n"));

  write_netbios_names(time_now,WINS_LIST,namelist,&names_last_modified,True);
  DEBUG(3,("Written names\n"));

  process();

  for (i = 0; i < MAX_LOCAL_INTERFACES; i++)
  {
    if (ServerNMB[i] != -1) close(ServerNMB[i]);
    if (ClientNMB[i] != -1) close(ClientNMB[i]);
  }

  if (dbf)
    fclose(dbf);
  return(0);
}
