/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains routines for loading and downloading
 *	fonts from PXL files.
 */

#include "types.h"
#include "font.h"
#include "pxl.h"
#include "tracedef.h"

#include "arith.p"
#include "fileio.p"
#include "fio.p"
#include "font.p"
#include "raster.p"

/*
 *	The following routine loads a font from a PXL file into memory.
 *	The pixel specification is compressed into as few longwords
 *	as necessary to store the pixel data.
 */

int Load_PXL (Font_Dir, Resolution, Font_Ptr)
char *Font_Dir;
struct Ratio *Resolution;
struct Font_Definition *Font_Ptr;
{
	auto   struct Char_Definition *Char_Ptr;
	auto   unsigned long *Raster_Base, *Raster_Ptr;
	auto   unsigned char *Pixel_Ptr;
	auto   pointer PXL_File;
	auto   unsigned long Size, Pixel_Size, PXL_Id_1, PXL_Id_2, Addr;
	auto   unsigned long File_Size, Scale;
	auto   int Index;
	static struct {
		unsigned short P_Wid;
		unsigned short P_Hgt;
		         short X0_Loc;
		         short Y0_Loc;
		unsigned long  Raster_Addr;
		fix_word TFM_Wid;
	} Dir_Entry;
	extern pointer Open_PXL_File();
	extern char *Mem_Alloc();
	msgcode DVIOUT_BADPXLFILE, DVIOUT_NOPXLFILE, DVIOUT_UNKPXLFORMAT;
/*
 *	Construct default file name; open the PXL file:
 */
	if ((PXL_File = Open_PXL_File (Font_Dir, Font_Ptr->Name_Ptr, Resolution, Font_Ptr->Magnification)) == 0) {
		Message (DVIOUT_NOPXLFILE, Font_Ptr->Name_Ptr, 1, Font_Ptr->Magnification, 0);
		return (0);
	}
/*
 *	Read in the Pixel Id numbers and compare them:
 */
	PXL_Id_1 = Read_Unsigned_Int_M (4, PXL_File);
	Set_EOF_M (PXL_File);
	PXL_Id_2 = 0;
	while ((File_Size = File_Position_M (PXL_File)) >= 20 &&
	       (PXL_Id_2 = Read_Unsigned_Int_Reverse_M (4, PXL_File)) == 0)
		;
	if (PXL_Id_2 != PXL_Id_1)
		goto Bad_File;
	File_Size -= 20;
	Font_Ptr->Pixel_Id = (ID_PXL << ID_V_TYPE) | (PXL_Id_1 & ID_M_FORMAT);
	if (trace_pixel)
		printf ("  Pixel File Id is %lu\n", PXL_Id_1);
	if (PXL_Id_1 != PXL_FORMAT_A) {
		Message (DVIOUT_UNKPXLFORMAT, Font_Ptr->Name_Ptr, 1, Font_Ptr->Magnification, 0);
		goto Bad_File1;
	}
/*
 *	Set the coding scheme, family name and face type to null:
 */
	Font_Ptr->Font_Coding[0] = '\0';
	Font_Ptr->Font_Family[0] = '\0';
	Font_Ptr->Font_Face = 0;
/*
 *	Locate the character directory.
 */
	Addr = Read_Unsigned_Int_Reverse_M (4, PXL_File) * 4;
	if (Addr != File_Size - 2048)
		goto Bad_File;
	if (trace_pixel)
		printf ("  Character Directory Address is %lu\n", Addr);
/*
 *	Read in the miscellaneous stuff at the end of the file
 *	(replaces some things already known):
 */
	Font_Ptr->Font_Design_Size = Read_Unsigned_Int_Reverse_M (4, PXL_File);
	Read_Unsigned_Int_Reverse_M (4, PXL_File);
	Font_Ptr->Checksum = Read_Unsigned_Int_Reverse_M (4, PXL_File);
	if (trace_pixel) {
		printf ("  Font Design Size is %lu\n", Font_Ptr->Font_Design_Size);
		printf ("  Font Checksum is %lu\n", Font_Ptr->Checksum);
	}
/*
 *	Load the directory information for each character. The
 *	'Font_Directory' array in the font definition was used, during pass
 *	1, to record the frequency of characters used in the file. If the
 *	character was not used, it isn't loaded.
 */
	Pixel_Size = 0;
	Set_File_Position_M (Addr, PXL_File);
	for (Index = 0; Index < 128; Index++) {
		if (trace_pixel)
			printf ("  Character Number %d:\n", Index);
/*
 *	Locate and read in the directory entry:
 */
		Dir_Entry.P_Wid = Read_Unsigned_Int_M (2, PXL_File);
		Dir_Entry.P_Hgt = Read_Unsigned_Int_M (2, PXL_File);
		Dir_Entry.X0_Loc = Read_Signed_Int_M (2, PXL_File);
		Dir_Entry.Y0_Loc = Read_Signed_Int_M (2, PXL_File);
		Dir_Entry.Raster_Addr = Read_Unsigned_Int_M (4, PXL_File) * 4;
		Dir_Entry.TFM_Wid = Read_Unsigned_Int_M (4, PXL_File);
		if (Dir_Entry.Raster_Addr >= Addr)
			goto Bad_File;
/*
 *	Now allocate enough space for the character descriptor. Zero values
 *	for both raster address and TFM width indicates that this character
 *	is not defined within this font:
 */
		if (Font_Ptr->Font_Directory[Index] == 0) {
			Char_Ptr = 0;
			if (trace_pixel)
				printf ("    (Not referenced for this font)\n");
		} else if (Dir_Entry.Raster_Addr == 0 && Dir_Entry.TFM_Wid == 0) {
			Char_Ptr = 0;
			Undef_Char_Count++;
			if (trace_pixel)
				printf ("    Not defined for this font\n");
		} else {
			Char_Ptr = (struct Char_Definition *) Mem_Alloc (sizeof (struct Char_Definition));
			Char_Ptr->Character_Code = Index;
			Char_Ptr->DVI_Width = Dir_Entry.TFM_Wid;
			Char_Ptr->Pixel_Width = Dir_Entry.P_Wid;
			Char_Ptr->Pixel_Height = Dir_Entry.P_Hgt;
			Char_Ptr->X_Origin = Dir_Entry.X0_Loc;
			Char_Ptr->Y_Origin = Dir_Entry.Y0_Loc;
			if ((Char_Addr[Index] = Dir_Entry.Raster_Addr) != 0)
				Pixel_Size += ((Char_Ptr->Pixel_Width + 7) >> 3) * Char_Ptr->Pixel_Height;
			if (trace_pixel) {
				printf ("    Width in fraction of design size is %ld\n", Char_Ptr->DVI_Width);
				printf ("    Width and Height in pixels is %u by %u\n", Char_Ptr->Pixel_Width,
					Char_Ptr->Pixel_Height);
				printf ("    Origin is at (%d,%d)\n", Char_Ptr->X_Origin, Char_Ptr->Y_Origin);
				printf ("    Raster Data Address is %lu\n", Char_Addr[Index]);
			}
/*
 *	Initialize dynamic parameters. The Driver Id value is set up first
 *	to store the character frequency. It is set again when the font
 *	is downloaded.
 */
			Char_Ptr->Driver_Id = (unsigned long) Font_Ptr->Font_Directory[Index];
		}
		Font_Ptr->Font_Directory[Index] = Char_Ptr;
	}
/*
 *	Now go back and read in the Raster data; it was not done
 *	previously to prevent wild thrashing in trying to access the
 *	pixel file and to prevent allocating a whole bunch of tiny
 *	memory segments, which takes time:
 */
	Pixel_Ptr = Mem_Alloc (Pixel_Size);
	for (Index = 0; Index < 128; Index++)
	if ((Char_Ptr = Font_Ptr->Font_Directory[Index]) != 0 && (Addr = Char_Addr[Index]) != 0) {
		Set_File_Position_M (Addr, PXL_File);
		Size = ((Char_Ptr->Pixel_Width + 31) >> 5) * Char_Ptr->Pixel_Height;
		Raster_Ptr = Raster_Base = (unsigned long *) Mem_Alloc (Size * 4);
		if (trace_pixel)
			printf ("  Raster data for character %u read in at %lu, %lu words\n",
				Index, Addr, Size);
		for (; Size > 0; Size--)
			*Raster_Ptr++ = Read_Unsigned_Int_M (4, PXL_File);
/*
 *	Convert the raster data to compressed form:
 */
		Raster_Word_to_Byte_M (Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height,
				       Raster_Base, Pixel_Ptr);
		Mem_Free (Raster_Base);
		Char_Ptr->Pixel_Array = Pixel_Ptr;
		Pixel_Ptr += ((Char_Ptr->Pixel_Width + 7) >> 3) * Char_Ptr->Pixel_Height;
	}
	Close_File_M (PXL_File);
/*
 *	Check for any additional undefined characters:
 */
	Undef_Char_Count += Check_Undefined_Char_M (Font_Ptr, 128, Font_Ptr->Char_Dir_Count);
	return (1);

Bad_File:
	Message (DVIOUT_BADPXLFILE, Font_Ptr->Name_Ptr, 1, Font_Ptr->Magnification, 0);
Bad_File1:
	Close_File_M (PXL_File);
	return (0);
}

/*
 *	Just a note here on the file naming conventions. Under TeX V1.3,
 *	a pixel file name was, for example, "TEX$FONTS:AMR10.1500PXL",
 *	which works just fine. Under TeX V2.0, the standard place for
 *	the same font is "SYS$TEX:[TEX.IM_FONTS.1500]AMR10.PXL", which
 *	requires some convolutions to construct the file name, and
 *	excludes the possibility of using a logical name for the
 *	directory - a decided disadvantage on a VAX (this level of the
 *	code does not know about logical names). So I have decided
 *	to stick with the former technique, as it is much cleaner and
 *	consistent with TeX, where a file's location can be completely
 *	determined by a) the file type, b) the file area, and c) the
 *	file name. Additionally, it is consistent with PK and GF file
 *	naming conventions.
 *
 *	If the latter style MUST be used, change the template below to:
 *
 *	    char *Template = "sys$tex:[tex.im_fonts.%04d]%s.pxl"
 *
 *	along with a corresponding change in the sprintf call arguments.
 *	The 'Font_Dir' argument will go unused. Alternatively, the
 *	logical name defining the pixel directory could be defined
 *	externally as something like:
 *
 *	    $ DEFINE tex_im_pixel dua0:[tex.im_fonts.]
 *
 *	and the template would then be "%s:[%04d]%s.pxl". Unfortunately,
 *	this still requires knowledge of VAX directory syntax (square
 *	brackets).
 */

pointer Open_PXL_File (Font_Dir, Font_Name, Resolution, Magnification)
char *Font_Dir, *Font_Name;
struct Ratio *Resolution;
unsigned long Magnification;
{
	static char Default_File[80], *Template = "%s.%04dpxl";

	sprintf (Default_File, Template, Font_Name,
		 Get_Font_Mag_M (5, Resolution, XN_Div_D_R_M (Magnification, Resolution->Numerator,
							      200 * Resolution->Denominator)));
	if (trace_pixel)
		printf ("Name of the PXL file is \"%s\"\n", Default_File);
	return (Open_File_M (Default_File, Font_Dir, "r", 0));
}
