/*
 * SOURCE:  texerr.c
 * PROJECT: EasyTeX
 *
 * PURPOSE: error scan routines for \TeX
 *
 * UPDATES: 11/18/1992 - extracted from applic.c due to overlays
 *
 * (c)M.Schollmeyer
 */
#include "main.h"

extern struct Editor ed;
extern struct MemoryHeader mh;

/*
 *    Name: viewerrors
 *  Return: void
 * Purpose: TeX error handling
 *
 *
 *
 * (c)M.Schollmeyer
 */
static struct {
    unsigned int line;
    BOOL noline;
    unsigned int logline;
    char text[SCREEN_WIDTH+1];
    unsigned int badness;
} texerr;

static void *fsptr;
static char bracelevel = 0; /* used to scan any brace level */

static void _getlogfilename( char * );
static BOOL _gettexerror( char _far *, unsigned int );
static void _setuperr( char *, char *, unsigned int );
static BOOL _boxwarning( char * );

#define WIN_WIDTH 67
#define WIN_HEIGHT 20
void viewerrors( int flag ) {

    static int fptr;
    static char _far *buf = NULL,      // buffer for log file
                _far *curr,            // current position in log file
                _far *end;             // end of log file buffer

    long size;
    char logfile[BUFSIZE];
    unsigned int errorlevel;
    int i;

    if( flag == VIEW_CLEAR ) {
        if( buf ) {
            free( buf );
            buf = NULL;
        }
        return;
    }

    if( flag == VIEW_TEXT ) {
        /* view message for error display */
        if( FP_SEG( buf ) )
            i = texerr.logline;
        else
            i = 0;

        e_savekoords();
        _getlogfilename( logfile );

        if( m_save( SAVE_NORMAL ) ) {
            e_savekoords();
            if( e_load( logfile ) ) {
                e_goto( 0, 0, 0, i );
                SETBIT( ed.Flags, TEMPFILE|PSEUDOFILE );
                e_refresh();
                e_drawflags();
            }
        }
        return;
    }

    if( FP_SEG( buf ) == 0 ) {
        // no logfile loaded, load new one...
        texerr.line = 0;
        texerr.logline = 0;
        _getlogfilename( logfile );
        fptr = Open( logfile, ACCESS_READ );
        if( fptr == -1 ) {
            DoErrorBox( HLP_VIEWERR, "Could not open log file\n'%s'", logfile );
            return;
        }
        size = Seek( fptr, 0L, SEEK_END );
        Seek( fptr, 0L, SEEK_START );

        buf = AllocHeap( (unsigned)(size)+1 );
        if( buf == NULL ) {
            Close( fptr );
            return;
        }
        _Read( fptr, buf, (unsigned int)size );
        curr = buf;
        end = buf + size;
        *end = EOF;
        Close( fptr );

        /* (re)initialize stack */
        FreeStack( fsptr );
        fsptr = NULL;
        bracelevel = 0;
    }

    errorlevel = GetProfileInt( PR_ERRORLEVEL );

    while( curr < end ) {
        if( _gettexerror( curr, errorlevel ) ) {

            if( errorlevel < texerr.badness ) {
                /* look for file name if there is anything on stack
                   and if we should */
                if( fsptr && (ed.Flags & TRACKFILES) ) {
                    /* There is a file to load */
                    if( ed.DiskFileName && strcmp( ed.DiskFileName, fsptr ) ) {
                        /* search for file name */
                        for( i = 0; i < MAXFILES; ++i ) {
                            if( !strcmp( fsptr, GetProfileString( PR_FILENAME(i), &profnull ) ) )
                                break;
                        }
                        if( i == MAXFILES ) {
                            /* visit the file */
                            if( m_save( SAVE_NORMAL ) ) {
                                e_savekoords();
                                if( e_load( fsptr ) ) {
                                    e_goto( 0, 0, 0, 0 );
                                    e_refresh();
                                    e_status( 1 );
                                    ed.Flags |= PSEUDOFILE;
                                    e_drawflags();
                                }
                            }
                        } else {
                            if( m_save( SAVE_NORMAL ) ) {
                                e_savekoords();
                                if( !e_setfile( i ) )
                                    ClearProfileData( PR_FILENAME(i) );
                            }
                        }
                    }
                }
                if( !texerr.noline &&
                    texerr.line < mh.numlines+1 && texerr.line > 0 ) {
                    e_goto( ed.WindowX, ed.WindowY, 0, texerr.line );
                    SetCursorPos( ed.CursorX - ed.WindowX , ed.CursorY - ed.WindowY + EDITW_TOP );
                }
                e_refresh();
                m_putmsg( MSG_TEXERR, texerr.text );
                ++curr;
                return;
            }

        } else if( (curr[0] == 0x0d) && (curr[1] == 0x0a) ) {
            /* increment line number */
            ++texerr.logline;
        }
        ++curr;
    }
    /* end of buffer reached, display message */
    m_putmsg( MSG_TEXERR, "<no more errors>" );
    free( buf );
    buf = NULL;
    FreeStack( fsptr );
    fsptr = NULL;
}
#undef WIN_WIDTH
#undef WIN_HEIGHT

static void _getlogfilename( char *buffer ) {

    smakec( buffer, "%p%b.log",
        GetProfileString(PR_TEXLASTFILE,"texput"));
}


static BOOL _gettexerror( char _far *line, unsigned int errorlevel ) {

    int i;
    char *cp;
    char buffer[_MAX_PATH], file[_MAX_PATH];
    unsigned int badness;

    if( *line == 0x0a ) {
        /* end of line */
        ++line;
        if( *line == '!' ) {
            /* TeX error */
            cp = line;
            if( ed.Flags & TRACKLINES ) {
                /* scan for next line */
                while( *line != 0x0d && *line != EOF ) ++line;
                if( *line == 0x0d ) {
                    // there are further lines, search for line number
                    ++line;
                    while( *line != EOF && !(line[0]==0x0a && line[1]=='!') ) {
                        if( line[0] == 0x0a && line[1] == 'l' && line[2] == '.' ) {
                            // line number found
                            line += 3;
                            _setuperr( cp, line, 20000 );
                            return TRUE;
                        }
                        ++line;
                    }
                }
            }
            _setuperr( cp, NULL, 20000 );
            return TRUE;
        } else if( _boxwarning(line) ) {
            cp = line;
            badness = 10000;

            if( ed.Flags & TRACKLINES ) {
                while( *line != 0x0d ) {
                    if( strncmp( line, "(badness ", 9 ) == 0 ) {
                        badness = atoi( line+9 );
                        if( badness == 0 ) badness = 10000;
                    } else if( strncmp( line, "lines ", 6 ) == 0 ) {
                        _setuperr( cp, line+6, 10000 );
                        return TRUE;
                    } else if( strncmp( line, "line ", 5 ) == 0 ) {
                        _setuperr( cp, line+5, 10000 );
                        return TRUE;
                    }
                    ++line;
                }
            }
            _setuperr( cp, NULL, 10000 );
            return TRUE;
        }
    } else if( ed.Flags & TRACKFILES ) {

        if( *line == '(' ) {
            /* could be a file name */
            ++line;
            /* make the file's name */
            strncpy( buffer, line, _MAX_PATH );
            for( cp = buffer; (*cp != ')') && (*cp != ' ') && (*cp != 0x0d); ++cp );
            *cp = '\0';
            file[0] = '\0';
            /* if 'buffer' contains no path, it should be in PR_TEXCWD */
            if( !strchr( buffer, '\\' ) && !strchr( buffer, ':' ) ) {
                strcpy( file, GetProfileString( PR_TEXCWD, &profnull ) );
                if( strlen(file) && file[strlen(file)-1] != '\\' )
                    strcat( file, "\\" );
            }
            strcat( file, buffer );
            if( _Access( file, 0 ) ) {
                /* it's a file name, add it to stack */
                strlow( file );
                fsptr = PushStack( fsptr, (APTR)file, strlen(file)+1 );
                bracelevel = 0;
            } else
                ++bracelevel;
        } else if( *line == ')' ) {
            if( bracelevel )
                --bracelevel;
            else {
                /* pop filename from stack */
                fsptr = PopStack( fsptr );
            }
        }
    }
    return FALSE;
}

static BOOL _boxwarning( char *cp ) {

    int i;

    static char *warn[] = {
        "Underfull \\hbox",
        "Overfull \\hbox",
        "Underfull \\vbox",
        "Overfull \\vbox",
    };

    for( i = sizeof(warn)/sizeof(char *)-1; i >= 0; --i ) {
        if( strncmp( cp, warn[i], strlen(warn[i]) ) == 0 )
            return TRUE;
    }
    return FALSE;
}

#define DISPFILELEN 20
static void _setuperr( char *msg, char *linenum, unsigned int badness ) {

    int i = 0;
    char *cp;
    char buffer[DISPFILELEN+1];

    texerr.badness = badness;

    cp = texerr.text;

    if( (ed.Flags & TRACKFILES) && fsptr ) {
        MakeShortName( buffer, fsptr, DISPFILELEN );
        i += sprintf( cp, "%s", buffer );
        cp = texerr.text + i;
    }

    if( linenum ) {
        texerr.line = atoi(linenum)-1;
        texerr.noline = FALSE;
        i += sprintf( cp, "(%d)", texerr.line );
        cp = texerr.text + i;
    } else {
        texerr.noline = TRUE;
    }

    if( i ) {
        /* any preceding info */
        *cp++ = ':';
        ++i;
    }

    for( ; (*msg != 0x0d) && (*msg != EOF) && (i < SCREEN_WIDTH-1); ++i )
           *cp++ = *msg++;
    *cp = '\0';
    return;
}



/* end of file texerr.c */
