/* keymap.c -- load a user defined keyboard map.
   
   Copyright (C) 1994 Ralph Schleicher  */

/* This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */


#include "config.h"

#include <kpathsea/c-ctype.h>
#include <kpathsea/pathsearch.h>

#define DEFUN_VOID(x)	x (void)
#define DEFUN(x, y, z)	x (z)
#define EXFUN(x, y)	x y
#define AND		,

#ifdef INI
#undef NONASCII
#endif

#ifdef NONASCII

#define COMMENT		'%'
#define CHAR		'`'
#define OCT		'\''
#define HEX		'\"'

static int EXFUN (get_code, (char *string, char **end));
static char *EXFUN (skip, (char *string));
static char *EXFUN (fgetl, (FILE *stream));

static char input[256];
static char output[256];

extern char xord[256];
extern char xchr[256];


void
DEFUN_VOID (loadkeymap)
{
  char *map, *name, *line, *pos;
  int num, ch, in, out;
  FILE *file;

  map = getenv ("TEXMAP");
  if (!map)
    return;

  name = kpse_path_search (".", map, true);
  if (!name)
    return;

  file = fopen (name, FOPEN_R_MODE);
  if (!file)
    goto done;

  for (ch = 0; ch < 256; ++ch)
    input[ch] = xord[ch];
  for (ch = 0; ch < 256; ++ch)
    output[ch] = xchr[ch];

  for (num = 1; ; ++num)
    {
      line = fgetl (file);
      if (!line)
	break;

      pos = skip (line);
      if (*pos)				/* Have something to do. */
	{
	  ch = get_code (pos, &pos);	/* Character code. */
	  if (ch < 0 || !*pos)
	    break;

	  in = get_code (pos, &pos);	/* Internal TeX code. */
	  if (in < 0)
	    break;

	  if (!*pos)
	    out = ch;			/* Transform back on default. */
	  else
	    {
	      out = get_code (pos, &pos);
	      if (out < 0 || *pos)	/* Line is done. */
		break;
	    }

	  input[ch] = in;
	  output[ch] = out;
	}

      free (line);
    }

  if (line)
    {
      fprintf (stderr, "%s:%d: Parse error\n", name, num);

      uexit (1);
    }
  else
    {
      for (ch = 0; ch < 256; ++ch)
	xord[ch] = input[ch];
      for (ch = 0; ch < 256; ++ch)
	xchr[ch] = output[ch];
    }

 done:

  if (name != map)
    free (name);
}


static int
DEFUN (get_code, (string, end),
char *string AND
char **end)
{
  char *s = string;
  int c = -1;

  errno = 0;
  switch (*string)
    {
    case CHAR:
      if (!*++string)
	break;
      c = *string;
      s = string + 1;
      break;
    case OCT:
      if (!*++string)
	break;
      c = strtol (string, &s, 8);
      break;
    case HEX:
      if (!*++string)
	break;
      c = strtol (string, &s, 16);
      break;
    case 0:
      break;
    default:
      c = strtol (string, &s, 10);
      break;
    }

  if (errno || c < 0 || c > 255)
    return -1;
  if (*s && !isspace (*s) && !(*s == COMMENT))
    return -1;
  if (end)
    *end = skip (s);

  return c;
}


static char *
DEFUN (skip, (string),
char *string)
{
  while (isspace (*string))
    ++string;
  if (*string == COMMENT)
    string = strchr (string, 0);

  return string;
}


#define LINE	128

static char *
DEFUN (fgetl, (stream),
FILE *stream)
{
  int alloc, len, end;
  char *line, buf[LINE];
  
  alloc = len = 0;
  line = 0;

  while (fgets (buf, LINE, stream))
    {
      end = len;
      len += strlen (buf);
      if (len + 1 > alloc)
        {
          alloc += LINE;
          line = xrealloc (line, alloc);
        }
      strcpy (&line[end], buf);
      if (line[len - 1] == '\n')
        {
	  --len;
	  line[len] = 0;
	  break;
        }
    }

  return line;
}

#else /* non NONASCII */

void
DEFUN_VOID (loadkeymap)
{
}

#endif /* not NONASCII */


#ifdef TEST

int
DEFUN (main, (argc, argv),
int argc AND
char **argv)
{
  int ch;

  loadkeymap ();

  for (ch = 0; ch < 256; ++ch)
    fprintf (stdout, "%2x\t%2x\t%2x\n", ch, input[ch], output[ch]);

  return 0;
}

#endif /* TEST */


/*
 * local variables:
 * compile-command: "gcc -DTEST -DNONASCII -o keymap keymap.c -lkpathsea
 * end:
 */
