// $Id: gweMysqlMetaResult.java,v 1.3 1997/11/08 16:08:59 xzhu Exp $
//////////////////////////////////////////////////////////////////
// mysql JDBC Driver
//
// Version : 0.9.2
//
// JDBC : 1.22
//
// LastModified: 12.03.1997
//
// Copyright (c) 1997 GWE Technologies Ltd. All rights reserved
// See http://gwe.co.uk/mysql/jdbc/copyright.html for full details

package gwe.sql;

import java.sql.*;
import gwe.sql.gweMysql;
import gwe.sql.gweMysqlColumn;

/**
 * This class provides the mysql implementation of the JDBC ResultSetMetaData
 * interface.
 */
public class gweMysqlMetaResult implements java.sql.ResultSetMetaData {
  /**
   * The result this meta-data exists for.
   */
  private gweMysql mysql;

  /**
   * Constructs a new instance of the gweMysqlMetaResult class
   * @param result the gweMysql object
   */
  public gweMysqlMetaResult(gweMysql result) {
    mysql = result;
  }
  
  /**
   * Returns the number of columns in this result set.
   * @see java.sqlResultSetMetaData#getColumnCount
   * @return number of columns
   */
  public int getColumnCount() throws SQLException {
    return mysql.getColumnCount();
  }

  /**
   * Should indicate if field is set AUTO INCREMENT, but no way of
   * getting the information from mysql at the moment.
   * @see java.sql.ResultSetMetaData#isAutoIncrement
   * @return false
   */
  public boolean isAutoIncrement(int column) throws SQLException { 
    return false;
  }

  /**
   * Column's Case matters to mysql
   * @see java.sql.ResultSetMetaData#isCaseSensitive
   * @return true
   */
  public boolean isCaseSensitive(int column) throws SQLException {
    return true;
  }

  /**
   * Can the column be used in a where clause?
   
   * @see java.sql.ResultSetMetaData#isSearchable
   * @return true unless column is a from of blob
   */
  public boolean isSearchable(int column) throws SQLException {
	try {
		gweMysqlColumn col = mysql.getColumn(column);
		return (!col.isBlob());
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * Is the column a cash value?
   * @see java.sql.ResultSetMetaData#isCurrency
   * @return false
   */
  public boolean isCurrency(int column) throws SQLException {
    return false;
  }

  /**
   * Determines if the column in question is nullable.
   * @see java.sql.ResultSetMetaData#isNullable
   * @exception SQLException thrown if the column is out of range
   * @return true if the column can accept null values, otherwise false
   */
  public int isNullable(int column) throws SQLException{
	try {
		gweMysqlColumn col = mysql.getColumn(column);
		return (col.isNotNull() ? columnNoNulls : columnNullable);
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * Is the column a signed number?
   * @see java.sql.ResultSetMetaData#isSigned
   * @return true
   */
  public boolean isSigned(int column) throws SQLException {
	try {
		gweMysqlColumn col = mysql.getColumn(column);
		return (!col.isUnsigned());
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * Gives the display size for this column.
   * @see java.sql.ResultSetMetaData#getColumnDisplaySize
   */
  public int getColumnDisplaySize(int column) throws SQLException {
	try {
		gweMysqlColumn col = mysql.getColumn(column);
		return (col.getLength());
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * This returns table_name.column_name.
   * @see java.sql.ResultSetMetaData#getColumnLabel
   * @param column the column whose label is wanted
   * @return the display value for the field
   */
  public String getColumnLabel(int column)
       throws SQLException {
	try {
		gweMysqlColumn col = mysql.getColumn(column);
		return (col.getFullName());
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * The name of the requested column.
   * @see java.sql.ResultSetMetaData#getColumnName
   * @exception thrown when the column desired is bad
   * @param column the column whose name is wanted
   * @return the name of the requested column
   */
  public String getColumnName(int column)
       throws SQLException {
	try {
		gweMysqlColumn col = mysql.getColumn(column);
		return (col.getName());
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * This does not apply to mysql.
   */
  public String getSchemaName(int column)
       throws SQLException {
    return "";
  }

  /**
   * What's the column's number of decimal digits?
   */
  public int getPrecision(int column) throws SQLException {
	try {
		gweMysqlColumn col = mysql.getColumn(column);
		return (col.getDecimals());
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * What's the column's number of digits ro right of decimal point?
   */
  public int getScale(int column) throws SQLException {
	try {
		gweMysqlColumn col = mysql.getColumn(column);
		return (col.getDecimals());
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * Gives the name of the table to which this column belongs.
   * @see java.sql.ResultSetMetaData#getTableName
   * @param column the column of the field this information is needed for
   * @return the table name
   */
  public String getTableName(int column)
       throws SQLException {
	try {
		gweMysqlColumn col = mysql.getColumn(column);
		return (col.getTable());
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * mysql does not support this
   */
  public String getCatalogName(int column)
       throws SQLException {
	return new String("");
  }

  /**
   * Gives the column type using the types in java.sql.Types.
   * @see java.sqlTypes
   * @see java.sql.ResultSetMetaData#getColumnType
   * @exception SQLException thrown for any number of reasons
   * @param column the column type information is needed on
   * @return the type as listed in java.sql.Types
   */
  public int getColumnType(int column) throws SQLException {
	try {
		gweMysqlColumn col = mysql.getColumn(column);
		switch (col.getType()) {
			case 0 : return Types.DECIMAL;
			case 1 : return Types.CHAR;
			case 2 : return Types.SMALLINT;
			case 3 : return Types.INTEGER;
			case 4 : return Types.FLOAT;
			case 5 : return Types.DOUBLE;
			case 6 : return Types.NULL;
			case 7 : return Types.TIMESTAMP;
			case 8 : return Types.BIGINT;
			case 9 : return Types.INTEGER;
			case 10 : return Types.DATE;
			case 11 : return Types.TIME;
			case 12 : return Types.TIMESTAMP;
			case 249 : return Types.VARBINARY;
			case 250 : return Types.LONGVARBINARY;
			case 251 : return Types.LONGVARBINARY;
			case 252 : return Types.LONGVARBINARY;
			case 253 : return Types.VARCHAR;
			case 254 : return Types.VARCHAR;
			default : throw new SQLException("Unknown data type.");
		}
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * Gives the column type as a string.
   * @see java.sql.ResultSetMetaData#getColumnTypeName
   * @exception SQLException thrown randomly for fun
   * @param column the column for which the type name is wanted
   * @return the name of the column type
   */
  public String getColumnTypeName(int column) throws SQLException {
    switch(getColumnType(column)) {
		case Types.DECIMAL : return "DECIMAL";
		case Types.CHAR : return "CHAR";
		case Types.SMALLINT : return "SMALLINT";
		case Types.FLOAT : return "FLOAT";
		case Types.DOUBLE : return "DOUBLE";
		case Types.NULL : return "NULL";
		case Types.TIME : return "TIME";
		case Types.BIGINT : return "BIGINT";
		case Types.INTEGER : return "INTEGER";
		case Types.VARBINARY : return "VARBINARY";
		case Types.LONGVARBINARY : return "LONGVARBINARY";
		case Types.VARCHAR : return "VARCHAR";
	    default : return "NULL";
    }
  }

  
  
  public boolean isReadOnly(int column) throws SQLException {
    return false;
  }

  public boolean isWritable(int column) throws SQLException {
    return true;
  }

  public boolean isDefinitelyWritable(int column) throws SQLException {
    return true;
  }
}
    


   
