/*****************************************************************************/
/*                                                                           */
/*                            MYPASSWD.C/MYGROUP.C                           */
/*                                                                           */
/*             set password/group membership for mysql databases             */
/*                                                                           */
/*****************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <locale.h>
#include <pwd.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <time.h>

#include <mysql.h>

#include "basename.h"
#include "mypasswd.h"


#define VERSION 	"2.00"

#define DATABASE_MAX	256
#define TABLE_MAX	256
#define FIELDNAME_MAX	256
#define USERNAME_MAX	256
#define PASSWORD_MAX	256	/* should be >= _PASSWORD_LEN */
#define GROUP_MAX   	256
#define HOSTNAME_MAX	256
#if PASSWORD_MAX > GROUP_MAX
# define OTHER_MAX      PASSWORD_MAX
#else
# define OTHER_MAX      GROUP_MAX
#endif

/* define behaviour */
#define MYPASSWD        0
#define MYGROUP         1

void interrupted(int sig);
void error(int status, int errnum, char *format, ...);
void usage(int status);


/* actual program name */
char *progname = NULL;
/* print error information ? */
Flag silent = FALSE;
int verbose = FALSE;
int behaviour = MYPASSWD;

char shortopts[] = "+HUPscdvqhV";
struct option longopts[] =
{
  { "host",          required_argument, NULL, 'H' },
  { "auth-user",     required_argument, NULL, 'U' },
  { "auth-password", required_argument, NULL, 'P' },
  { "scramble",      no_argument,       NULL, 's' },
  { "crypt",         no_argument,       NULL, 'c' },
  { "delete",        no_argument,       NULL, 'd' },
  { "verbose",       no_argument,       NULL, 'v' },
  { "quiet",         no_argument,       NULL, 'q' },
  { "help",          no_argument,       NULL, 'h' },
  { "version",       no_argument,       NULL, 'V' },
  { NULL, 0, NULL, 0 }
};



int main(int argc, char *argv[])
{
int optc;
char db[DATABASE_MAX+1] = "";
char table[TABLE_MAX+1] = "";
char userfield[FIELDNAME_MAX+1] = "";
char otherfield[FIELDNAME_MAX+1] = "";
char username[USERNAME_MAX+1] = "";
char othername[OTHER_MAX+1] = "";
char host[HOSTNAME_MAX+1] = "localhost";
char auth_user[USERNAME_MAX+1] = "";
char auth_password[PASSWORD_MAX+1] = "";
Flag enctype = CRYPTED;
Flag delete = FALSE;

  setlocale(LC_ALL, "");
  progname = getprogname(argv[0]);

  if (strcasecmp(progname, "mypasswd") == 0)
    behaviour = MYPASSWD;
  else if (strcasecmp(progname, "mygroup") == 0)
    behaviour = MYGROUP;
  else
    error(1, 0, "please call me \'mypasswd\' or \'mygroup\'");

  /* initialize variables for getopt() package and get options */
  opterr = 0;
  while ((optc = getopt_long(argc, argv, shortopts, longopts, NULL)) != EOF)
  {
    switch (optc)
    {
      case 'H' :  if (optarg == NULL)
                    usage(1);
                  strcpy(host, optarg);
                  break;
      case 'U' :  if (optarg == NULL)
                    usage(1);
                  strcpy(auth_user, optarg);
                  break;
      case 'P' :  if (optarg == NULL)
                    usage(1);
                  strcpy(auth_password, optarg);
                  break;
      case 's' :  if (behaviour != MYPASSWD)
                    usage(1);
                  enctype = SCRAMBLED;
                  break;
      case 'c' :  if (behaviour != MYPASSWD)
                    usage(1);
                  enctype = CRYPTED;
                  break;
      case 'd' :  delete = TRUE;
                  break;
      case 'v' :  verbose = TRUE;
                  break;
      case 'q' :  silent = TRUE;
                  break;
      case 'h' :  usage(0);
                  break;
      case 'V' :  fprintf(stderr, "%s version %s\n", progname, VERSION);
                  exit(0);
                  break;
      default  :  usage(1);
    }
  }

  switch (argc - optind)
  {
    case  6 :  strcpy(db, argv[optind]);
               strcpy(table, argv[optind+1]);
               strcpy(userfield, argv[optind+2]);
               strcpy(otherfield, argv[optind+3]);
               strcpy(username, argv[optind+4]);
               strcpy(othername, argv[optind+5]);
               break;
    case  5 :  strcpy(db, argv[optind]);
               strcpy(table, argv[optind+1]);
               strcpy(userfield, argv[optind+2]);
               strcpy(otherfield, argv[optind+3]);
               strcpy(username, argv[optind+4]);
               if (delete)
                 *othername = '\0';
               else
               {
                 if (behaviour == MYGROUP)
                   usage(1);
                 strcpy(othername, getpass("Password: "));
               }
               break;
    default :  usage(1);
  }

  atexit((void(*)(void)) closeDatabase);
  signal(SIGINT, interrupted);

  if (openDatabase(host, db, auth_user, auth_password) != TRUE)
    error(1, 0, DatabaseError());

  if (delete)
  {
    if (DeleteEntry(table, userfield, otherfield, username, othername) != TRUE)
      error(1, 0, DatabaseError());
  }
  else
  {
    if (behaviour == MYPASSWD)
    {
      if (checkPassword(othername, enctype) == FALSE)
        error(1, 0, "illegal password");
      if (encryptPassword(othername, othername, enctype) == NULL)
        error(1, 0, "illegal password");
    }
    if (SetEntry(table, userfield, otherfield, username, othername) != TRUE)
      error(1, 0, DatabaseError());
  }

  closeDatabase();
  return(0);
}


/* signal handler fr SIGINT */
void interrupted(int sig)
{
  closeDatabase();
  error(1, 0, "interrupted by signal %d", sig);
}


/* display error message (with reason) and eventually exit */
void error(int status, int errnum, char *format, ...)
{
  if (!silent)
  {
    fprintf(stderr, "%s: ", progname);

    if (format != NULL)
    {
    va_list args;

      va_start(args, format);
      vfprintf(stderr, format, args);
      va_end(args);
      if (errnum != 0)
        fprintf(stderr, ": %s\n", strerror(errnum));
      else
        fprintf(stderr, "\n");
    }
    else
    {
      if (errnum != 0)
        fprintf(stderr, "%s\n", strerror(errnum));
      else
        fprintf(stderr, "unknown error\n");
    }
  }

  if (status)
    exit(status);
}


/* display usage message and exit */
void usage(int status)
{
  if (behaviour == MYGROUP)
    fprintf(stderr, "Usage:  %s [<options>] <database> <group table>\n"
                    "%*s <user field> <group field> <user> [<group>]\n",
                    progname, 8+strlen(progname), "");
  else
    fprintf(stderr, "Usage:  %s [<options>] <database> <password table>\n"
                    "%*s <user field> <password field> <user> [<password>]\n",
                    progname, 8+strlen(progname), "");

  if (status == 0)
  {
    fprintf(stderr,
    "  where <options> are:\n"
    "    -H  --host <host>          connect to host <host> (default: localhost)\n"
    "    -U  --auth-userhost <user> connect to the database as <user>\n"
    "    -P  --auth-password <pw>   connect to the database with password <pw>\n");
    if (behaviour == MYGROUP)
      fprintf(stderr,
      "    -d  --delete               delete user [from group]\n");
    else
      fprintf(stderr,
      "    -s  --scramble             use mySQL password() function for passwords\n"
      "    -c  --crypt                use crypt() function for passwords (default)\n"
      "    -d  --delete               delete user\n");
    fprintf(stderr,
    "    -v  --verbose              display what happens / show SQL queries\n"
    "    -q  --quiet                silent operation (no error messages)\n"
    "    -h  --help                 show this help page\n"
    "    -V  --version              display version number\n");
  }
  else
    fprintf(stderr, "For help, type: %s -h\n", progname);

  exit(status);
}

