/* This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307 USA
 */
/*
 * File: sql.c
 * Usage: SQL interface routines
 *
 * Copyright (C) 1996-97 all source by Hal Roberts
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <guile/gh.h>

#include "sql_imp.h"

#define GH_SQL_BUF_LEN 256

/* dynamic array of sql db pointers, referenced by index number in the
   sql prims */
void **gh_sql_dbs;

/* current allocated size of mutex table */
int gh_sql_num_dbs;

int gh_sql_is_valid_db(int id) {
  
  return ((id >= 0) &&
	  (id < gh_sql_num_dbs) &&
	  (gh_sql_dbs[id]));
}

/* begin db-create section */

int gh_sql_realloc_dbs(int old_size, int new_size) {
  
  int i;
  char buf[GH_SQL_BUF_LEN];
  
  if (new_size < 0) {
    gh_sql_error("gh_sql_realloc_dbs: New size for mutex table is "
		   "negative (int overflow ?)");
    return 0;
  }
  
  if (old_size < 1) {
    if (!(gh_sql_dbs = malloc(new_size * (sizeof *gh_sql_dbs)))) {
      snprintf(buf, GH_SQL_BUF_LEN, "gh_sql_realloc_dbs: Unable to "
	       "allocate space for db table of size %d", new_size);
      gh_sql_error(buf);
      return 0;
    } 
  } else {
    if (!(gh_sql_dbs = realloc(gh_sql_dbs,new_size * (sizeof *gh_sql_dbs)))) {
      snprintf(buf, GH_SQL_BUF_LEN,"gh_sql_realloc_dbs: Unable to "
	       "reallocate space for db table of size %d", new_size);
      gh_sql_error(buf);
      return 0;
    } 
  }
  
  for (i = old_size; i < new_size; i++)
    gh_sql_dbs[i] = NULL;
  
  return 1;
}

int gh_sql_init_db(void	*db, char *db_name, char *host_name, 
		     char *user_name, char *pass) {
  
  char buf[GH_SQL_BUF_LEN];
  
  if (!(gh_sql_imp_connect(db, host_name, user_name, pass))) {
    snprintf(buf, GH_SQL_BUF_LEN, "gh_sql_init: Unable to connect to "
	     "server: %s \n", gh_sql_imp_error(db));
    gh_sql_error(buf);
    return 0;
  }
  
  if (!gh_sql_imp_select_db(db, db_name)) {
    snprintf(buf, GH_SQL_BUF_LEN, "Unable to find '%s' database: %s\n", 
	     db_name, gh_sql_imp_error(db));
    gh_sql_error(buf);
    gh_sql_imp_close(db);
    return 0;
  }
  
  return 1;
}

#define GH_DB_START 4
#define GH_DB_GROW 2

SCM gh_sql_create(char *db_name, char *host_name, char *user_name, 
		    char *pass) {
  
  int i;
  
  /* initialize mutex table on first call */
  if (gh_sql_num_dbs < 1) {
    if (gh_sql_realloc_dbs(0, GH_DB_START))
      gh_sql_num_dbs = GH_DB_START;
    else
      return SCM_BOOL_F;
  }    
  
  /* look for a free (NULL) spot in the gh_sql_dbs array */
  for (i = 0; (i < gh_sql_num_dbs) && gh_sql_dbs[i]; i++)
    ;
  
  /* if all current spots are full, increase the size of the array according
     to the constant set in threads.h */
  if (i == gh_sql_num_dbs) {
    if (gh_sql_realloc_dbs(gh_sql_num_dbs, 
			     GH_DB_GROW * gh_sql_num_dbs))
      gh_sql_num_dbs = GH_DB_GROW * gh_sql_num_dbs;      
    else
      return SCM_BOOL_F;
  }
  
  /* now 'i' should be set to the index of a void * in the table
     that needs to be allocated */
  
  /* allocate space for the mutex.  return an error if malloc fails */
  if (!(gh_sql_dbs[i] = gh_sql_alloc_db())) {
    gh_sql_error("gh_sql_create: Unable to allocate space for new db");
    return SCM_BOOL_F;
  }
  
  /* initialize the mutex */
  if (!gh_sql_init_db(gh_sql_dbs[i], db_name, 
		      host_name, user_name, pass)) {
    free(gh_sql_dbs[i]);
    gh_sql_dbs[i] = NULL;
    return SCM_BOOL_F;
  }
  
  /* return the index to the mutex */
  return gh_int2scm(i);
}

char *gh_sql_create_param(SCM arg) {
  
  if (gh_string_p(arg))
    return gh_scm2newstr(arg, NULL);
  else if (gh_null_p(arg)) 
    return NULL;
  else {
    gh_sql_error("gh_sql_create_param: Argument to db-create is neither a "
		   "string nor a null");
    return NULL;
  }
}

/*  gh_sql_create_prim returns the index of a newly allocated and
    initalized mutex from the mutex table. */ 
SCM gh_sql_create_prim (SCM scm_db_name, SCM scm_host_name, 
			  SCM scm_user_name, SCM scm_pass) {
  
  char *db_name, *host_name = NULL, *user_name = NULL, *pass = NULL;
  SCM ret;
  
  if (scm_host_name != SCM_UNDEFINED) {
    host_name = gh_sql_create_param(scm_host_name);
    if (scm_user_name != SCM_UNDEFINED) {
      user_name = gh_sql_create_param(scm_user_name);
      if (scm_pass != SCM_UNDEFINED) {
	pass = gh_sql_create_param(scm_pass);
      }
    }
  }
  
  if (gh_string_p(scm_db_name)) {
    db_name = gh_scm2newstr(scm_db_name, NULL);
    ret = gh_sql_create(db_name, host_name, user_name, pass);
    free(db_name);
  } else {
    gh_sql_error("gh_sql_create_prim: Argument to db-create is not a "
		   "string");
    ret = SCM_BOOL_F;
  }
  
  if (host_name) 
    free(host_name);
  if (user_name) 
    free(user_name);
  if (pass) 
    free(pass);
  
  return ret;
}

/* begin db-destroy section */

/* destroy mutex, free it, and set relevant pointer in mutex table
   to NULL */
SCM gh_sql_destroy(int id) {
  
  char buf[GH_SQL_BUF_LEN];
  
  /* if id is less than 0, the mutex is null */
  if (id < 0)
    return SCM_BOOL_F;
  
  /* if id is greater than size of mup table, id is bogus */
  if (id >= gh_sql_num_dbs) {
    snprintf(buf, GH_SQL_BUF_LEN, "gh_sql_destroy: Mutex id (%d) is "
	     "greater than current number of mutexes (%d)", 
	     id, gh_sql_num_dbs);
    gh_sql_error(buf);
    return SCM_BOOL_F;
  }
  
  /* if the indexed ptr is NULL, id is bogus */
  if (!gh_sql_dbs[id]) {
    snprintf(buf, GH_SQL_BUF_LEN, "gh_sql_destroy: Indexed mutex (%d) is "
	     "NULL", id);
    gh_sql_error(buf);
    return SCM_BOOL_F;
  }
  
  /* looks like the mutex is valid.  now we need to destroy it, free it,
     and set it to NULL */
  gh_sql_imp_close(gh_sql_dbs[id]);
  free(gh_sql_dbs[id]);
  
  gh_sql_dbs[id] = NULL;
  
  return SCM_BOOL_T;
}

/* mutex-destroy prim.  destroy and free the given mutex */
SCM gh_sql_destroy_prim (SCM arg) {
  
  if (!gh_number_p(arg)) {
    gh_sql_error("gh_sql_destroy_prim: Argument to db-destroy is not a "
		   "number");
    return SCM_BOOL_F;
  }
  
  return gh_sql_destroy(gh_scm2int(arg));
}

/* begin sql-query section */

/* send a query to the sql engine and return a scm list in the
 * following format: [[field1 field2 ...]  [data1 data2 ...]  ...  ]
 *
 * If there is an error, return NULL.  If the resultset is empty, the
 * return value will look like this: [[field1 field2 ...]]  */
SCM gh_sql_query(void *db, char *query) {
  
  void *res;
  char **row;
  
  char *field_name;

  int r, f, num_fields, num_rows;
  
  char buf[GH_SQL_BUF_LEN];
  
  SCM scm_fields, scm_rows, scm_data;
  
  if (!gh_sql_imp_query(db, query)) {
    snprintf(buf, GH_SQL_BUF_LEN, "gh_sql_query: Error performing sql "
	     "query: %s\n", gh_sql_imp_error(db));
    gh_sql_error(buf);
    return SCM_BOOL_F;
  }
  
  /* if the query returns data, stick it in the return record */
  if ((res = gh_sql_imp_store_result(db))) {
    
    /* find out how many fields and rows were returned */
    num_fields = gh_sql_imp_num_fields(res);
    num_rows = gh_sql_imp_num_rows(res);
    
    /* loop through the fields, putting each field name in the list of field 
     * names and each field type in the array of field types */
    scm_fields = gh_make_vector(gh_int2scm(num_fields), gh_int2scm(0)); 
    for (f = 0; f < num_fields; f++) {
      field_name = gh_sql_imp_fetch_field_name(res);
      gh_vector_set_x(scm_fields, gh_int2scm(f), gh_str02scm(field_name));
    }
    
    /* now we're going to make an array of arrays, with the first row
       being the fields and the rest of the rows being the result of the
       query */
    scm_rows = gh_make_vector(gh_int2scm(num_rows + 1), gh_int2scm(0));
    gh_vector_set_x(scm_rows, gh_int2scm(0), scm_fields);
    for (r = 0; r < num_rows; r++) {
      row = gh_sql_imp_fetch_row(res);
      scm_data = gh_make_vector(gh_int2scm(num_fields), gh_int2scm(0));
      for (f = 0; f < num_fields; f++) {
	gh_vector_set_x(scm_data, gh_int2scm(f), 
			gh_sql_imp_sql2scm(res, row, f));
      }
      gh_vector_set_x(scm_rows, gh_int2scm(r + 1), scm_data);
    }
    
    
    gh_sql_imp_free_result(res);
  } else /* if the query returns no data, return the number of rows affected */
    scm_rows = gh_int2scm(gh_sql_imp_affected_rows(db));
  
  
  return scm_rows;
}  


/* send a query to sql and return the results */
SCM gh_sql_query_prim(SCM scm_db, SCM scm_query) {
  
  SCM ret;
  
  int db_id;
  char *query;
  
  if (!gh_number_p(scm_db)) {
    gh_sql_error("gh_sql_query_prim: First sql-query argument must be a "
		   "number");
    return SCM_BOOL_F;
  }
  
  db_id = gh_scm2int(scm_db);
  if (!gh_sql_is_valid_db(db_id)) {
    gh_sql_error("gh_sql_query_prim: Invalid db identifier");
    return SCM_BOOL_F;
  }
  
  if (!gh_string_p(scm_query)) {
    gh_sql_error("gh_sql_query_prim: Second sql-query argument must be a "
		   "string");
    return SCM_BOOL_F;
  }

  query = gh_scm2newstr(scm_query, NULL);

  ret = gh_sql_query(gh_sql_dbs[db_id], query);

  free(query);

  return ret;
}

