/**
*** Program jdbcMysqlRSMdata.java
***    in product twz1jdbcForMysql, 
***    Copyright 1997, 1998 by Terrence W. Zellers.
***   
***  All rights explicitly reserved.
***
***  See file "LICENSE" in this package for conditions of use.
**/

package twz1.jdbc.mysql;
import twz1.jdbc.mysql.jdbcMysqlField;
import java.util.Hashtable;
import java.util.Vector; 
import java.sql.*;

public final class RSMd implements ResultSetMetaData 
{
/** The parent statement. */
jdbcMysqlStmt stmt; 

/** vector of column descriptors */
Vector cols;

/** Column count */
int cc;

/** Hashtable of column names */
Hashtable cnames;


/*-------------------------------------------------------------------+
|                       Error messages                               |
+-------------------------------------------------------------------*/

static final String[] errs = {
    "E1100 Parent statement is closed.",
    "E1101 Unsupported API",
    "E1102 API error. ",
    "E1103 Column is outside Result bounds.",
    "E1104 Invalid column type for this request -- ",
    "E1105 Column name not found -- ",
    "E1106 Error adding field."
                             };


/*===================================================================+
||        Enum of API methods and their names for wrappers          ||
+===================================================================*/

static final int GETCOLUMNCOUNT         = 0;
static final int ISAUTOINCREMENT        = 1;
static final int ISCASESENSITIVE        = 2;
static final int ISSEARCHABLE           = 3;
static final int ISCURRENCY             = 4;
static final int ISNULLABLE             = 5;
static final int ISSIGNED               = 6;
static final int GETCOLUMNDISPLAYSIZE   = 7;
static final int GETCOLUMNLABEL         = 8;
static final int GETCOLUMNNAME          = 9;
static final int GETSCHEMANAME          = 10;
static final int GETPRECISION           = 11;
static final int GETSCALE               = 12;
static final int GETTABLENAME           = 13;
static final int GETCATALOGNAME         = 14;
static final int GETCOLUMNTYPE          = 15;
static final int GETCOLUMNTYPENAME      = 16;
static final int ISREADONLY             = 17;
static final int ISWRITABLE             = 18;
static final int ISDEFINITELYWRITABLE   = 19;

static final String[] pmethods = {
                             ".",
                             "isAutoIncrememnt()",
                             "isCaseSensitive()", 
                             "isSearchable()",
                             "isCurrency()",
                             "isNullable()",
                             "isSigned()", 
                             "getColumnDisplaySize()",
                             "getColumnLabel()",
                             "getColumnName()",
                             "getSchemaName()",
                             "getPrecision()",
                             "getScale()",
                             "getTableName()", 
                             "getCatalogName()",
                             "getColumnType()",
                             "getColumnTypeName()",
                             "isReadOnly()",
                             "isWritable()",
                             "isDefinitelyWritable()",
                                 };

/*===================================================================+
||            Very simple constructor for ResultSetMetaData         ||
+===================================================================*/

RSMd(jdbcMysqlStmt st)
    {
    this.stmt = st;
    this.cols = new Vector();
    this.cnames = new Hashtable();
    this.cc = 0;
    }

RSMd() { stmt = null; cols = null; cnames = null; cc = 0; }

/** for our purposes the vector and hashtable can be shared */
RSMd copy()
    {
    RSMd nd = new RSMd();
    nd.cols = this.cols;
    nd.cnames = this.cnames;
    nd.cc = this.cc;
    return nd;
    }

/*-------------------------------------------------------------------+
|                close access to the result set metadata             |
+-------------------------------------------------------------------*/

/** An internal close kills it hard on the theory that a soft kill
*** might not show bugs as soon as this will.
**/

  /** Removed close to make it reusable and remove need for locks */

/*-------------------------------------------------------------------+
|                add another column to the description               |
+-------------------------------------------------------------------*/

void addField(jdbcMysqlField f) throws SQLException
    {
    try {
        String nomen = f.getColumnName();
        jdbcMysqlField oldf;
        cols.addElement(f);
        cc++;
        oldf = (jdbcMysqlField) cnames.put(nomen, f);
        if(oldf != null)
            {
            f.setNameUnique(false);
            oldf.setNameUnique(false);
            }
        }
    catch(Exception e) { errHandlerE(6, e); }
    }


/*===================================================================+
||             Various methods to retrieve column data              ||
+===================================================================*/


jdbcMysqlField getField(int column) throws SQLException
    {
    jdbcMysqlField f = null;
    if(column < 1 || column > cc)
        throw new SQLException(errs[3]);
    f = (jdbcMysqlField) cols.elementAt(column - 1);
    return f;
    }

jdbcMysqlField getField(String cname) throws SQLException
    {
    jdbcMysqlField f = null;
    f = (jdbcMysqlField) cnames.get(cname);
    if(f == null) throw new SQLException(errs[5] + cname);
    return f;
    }

jdbcMysqlField getField(String tname, String cname) throws SQLException
    {
    jdbcMysqlField f;
    for(int i = 0; i < cc; i++)
        {
        f = (jdbcMysqlField) cols.elementAt(i);
        if(f.getColumnTable().equals(tname) 
          && f.getColumnName().equals(cname)) return f;
        }
    throw new SQLException(errs[5] + tname + "." + cname);
    }


/*===================================================================+
||                       API methods                                ||
+===================================================================*/
/** See API docs */
public int getColumnCount() throws SQLException
    {  return iReq(GETCOLUMNCOUNT, 0); }

/** This should be testable, but the flags do not return this
*** information from a standard query. Currently returns false.
*** Future implementations may conduct a more extensive investigation
*** to recover this information via a call to describe.
**/
public boolean isAutoIncrement(int column) throws SQLException
    { return (iReq(ISAUTOINCREMENT, column) == 1); }


/** See API and mysql docs. Char, text and varchar fields are text
*** insensitive in where clauses according to MySQL docs. Otherwise
*** true.
**/
public boolean isCaseSensitive(int column) throws SQLException
    { return (iReq(ISCASESENSITIVE, column) == 1); }

/** See API -- all MySQL fields are searchable. */
public boolean isSearchable(int column) throws SQLException
    { return (iReq(ISSEARCHABLE, column) == 1); }

/** Nothing in MySQL describes currency, per se */
public boolean isCurrency(int column) throws SQLException
    { return (iReq(ISCURRENCY, column) == 1); }

/** See API */
public int isNullable(int column) throws SQLException
    { return iReq(ISNULLABLE, column) ; }
     
/** See API, but true only for numeric types with unsigned not set */
public boolean isSigned(int column) throws SQLException
    { return (iReq(ISSIGNED, column) == 1); }

/** See API.  this result could be very large for blobs. */
public int getColumnDisplaySize(int column) throws SQLException
    { return iReq(GETCOLUMNDISPLAYSIZE, column); }

/** See the API.  Returns the column name if it is unique and
*** the catenated table name and column otherwise.
**/
public String getColumnLabel(int column) throws SQLException
    { return sReq(GETCOLUMNLABEL, column); }

/** See API */
public String getColumnName(int column) throws SQLException
    { return sReq(GETCOLUMNNAME, column); }   

/** See API, returns "" */
public String getSchemaName(int column) throws SQLException
    { return sReq(GETSCHEMANAME, column); } 

/** See API.  For Mysql this is column size for numerics;
*** non-numerics are exceptional. Future releases may take
*** cognizance that signs and decimals lessen the mathematical
*** precision of returned values.
**/
public int getPrecision(int column) throws SQLException
    { return iReq(GETPRECISION, column); }

/** See API */
public int getScale(int column) throws SQLException
    { return iReq(GETSCALE, column); }

/** See API */
public String getTableName(int column) throws SQLException
    { return sReq(GETTABLENAME, column); }

/** See API.  For MySQL this is the name of the "database" to
*** which the connection is currently maintained.
**/
public String getCatalogName(int column) throws SQLException
    { return sReq(GETCATALOGNAME, column); }

/** See API. This is somewhat problematic as MySQL types do not 
*** map precisely to jdbc types.
**/
public int getColumnType(int column) throws SQLException
    { return iReq(GETCOLUMNTYPE, column); }

/** See API. Possible problems mapping MySQL to jdbc types. */
public String getColumnTypeName(int column) throws SQLException
    { return sReq(GETCOLUMNTYPENAME, column); }

/**  All cols are writable. */
public boolean isReadOnly(int column) throws SQLException
    { return (iReq(ISREADONLY, column) == 1); }

/** All cols are writable. */
public boolean isWritable(int column) throws SQLException
    { return (iReq(ISWRITABLE, column) == 1); }

public boolean isDefinitelyWritable(int column) throws SQLException
    { return (iReq(ISDEFINITELYWRITABLE, column) == 1); }

/*===================================================================+
||          Wrap boolean and int API requests in a lock             ||
+===================================================================*/

private int iReq(int req, int c) throws SQLException
    {
    String rName = pmethods[req];
    if(stmt == null || !stmt.open) 
           throw new SQLException(errs[0] + rName);  
    int rc = -1;
    int i, j, k;
    jdbcMysqlField f = null;
    try {
        if(req != GETCOLUMNCOUNT) f = getField(c);
        switch(req)
            {
            case GETCOLUMNCOUNT: rc = cols.size(); break;  
            case ISAUTOINCREMENT: // TODO - This needs fixed!
                rc = 0; break;
            case ISCASESENSITIVE: 
                rc = 1;
                i = f.getColumnType();
                j = f.getColumnFlags();
                if(  i == f.FIELD_TYPE_STRING
                  || i == f.FIELD_TYPE_CHAR
                  || i == f.FIELD_TYPE_VAR_STRING
                  || ( i == f.FIELD_TYPE_BLOB 
                     && ( j & f.BINARY_FLAG ) == 0
                     )
                  ) rc = 1;
                break;   
            case ISSEARCHABLE:  getField(c); rc = 1; break;
            case ISCURRENCY:    getField(c); rc = 0; break; 
            case ISNULLABLE:
                i = f.getColumnFlags();
                rc = ( i & f.NOT_NULL_FLAG ) > 0  
                     ? columnNoNulls
                     : columnNullable;
                break;
            case ISSIGNED:
                i = f.getColumnType();
                j = f.getColumnFlags();
                if(isTypeNumeric(i))
                    { rc = ( j & f.UNSIGNED_FLAG ) > 0 ? 0 : 1 ; }
                else rc =typeXcept(req, 0);
                break;
            case GETCOLUMNDISPLAYSIZE:
            /** According to some books (says correspondent) this is
            *** supposed to the max widht of the field.  (How does it
            *** differ from getColumnDisplaySize()????
            **/
            case GETPRECISION:
                i = f.getColumnSize();
                j = stmt.iGetMaxFieldSize();
                rc = j < i ? j : i;
                break;
            case GETSCALE:
                i = f.getColumnType();
                if(isTypeNumeric(i)) rc = f.getColumnDecimals();
                else rc = typeXcept(req, 0);
                break;
            case GETCOLUMNTYPE:
                i = f.getColumnType();
                rc = mapMySQLtypes(i, f);
                break;
            case ISREADONLY: rc = 0; break;
            case ISWRITABLE: rc = 1; break;
            case ISDEFINITELYWRITABLE: rc = 1; break;
            default: throw new SQLException(errs[1], " - " + rName);
            }
        return rc;
        }
    catch(Exception e)
        { throw new SQLException(errs[2] + "\n" + e.getMessage()); }
    }


/** Throw a bad type exception if rsmdXcept, else return r value.
*** @param n method name number.
*** @param r value to return if we are not taking an exception.
**/

int typeXcept(int n, int r) throws SQLException
    {
    if(!stmt.rsmdXcept) return r;
    throw new SQLException(errs[4] + pmethods[n]);
    }

/*===================================================================+
||               Wrap String API requests in a lock                 ||
+===================================================================*/

private String sReq(int req, int c) throws SQLException
    {
    String rName = pmethods[req];
    if(stmt == null || !stmt.open) 
           throw new SQLException(errs[0] + rName);  
    String rc = null;
    int i, j, k;
    String x, y;
    jdbcMysqlField f;
    try {
        f = getField(c);
        switch(req)
            {
            case GETCOLUMNNAME:
                rc = f.getColumnName();
                break;
            case GETCOLUMNLABEL:
                if(f.isNameUnique()) rc = f.getColumnName();
                else rc = new String(f.getColumnTable() + "."
                                    + f.getColumnName());
                break;
            case GETSCHEMANAME: rc = ""; break;
            case GETTABLENAME: rc = f.getColumnTable(); break;
            case GETCATALOGNAME:
                rc = f.getColumnCatalog(); break; 
            case GETCOLUMNTYPENAME:
                i = f.getColumnType();
                j = mapMySQLtypes(i, f);
                rc = sqlTypeNames(j);
                break;
            default: throw new SQLException(errs[1], " - " + rName);
            }
        return rc;
        }
    catch(Exception e)
        { throw new SQLException(errs[2] + "\n" + e.getMessage()); }
    }



static int mapMySQLtypes(int i, jdbcMysqlField f)
    {
    switch(i)
        {
        case jdbcMysqlField.FIELD_TYPE_DECIMAL:    return Types.DECIMAL;
        case jdbcMysqlField.FIELD_TYPE_CHAR :      return Types.CHAR;
        case jdbcMysqlField.FIELD_TYPE_SHORT :     return Types.SMALLINT;
        case jdbcMysqlField.FIELD_TYPE_LONG :      return Types.INTEGER;
        case jdbcMysqlField.FIELD_TYPE_FLOAT :     return Types.FLOAT;
        case jdbcMysqlField.FIELD_TYPE_DOUBLE :    return Types.DOUBLE; 
        case jdbcMysqlField.FIELD_TYPE_NULL :      return Types.NULL;
        case jdbcMysqlField.FIELD_TYPE_TIMESTAMP:  return Types.TIMESTAMP;
        case jdbcMysqlField.FIELD_TYPE_LONGLONG :  return Types.BIGINT;
        case jdbcMysqlField.FIELD_TYPE_INT24 :     return Types.BIGINT;
        case jdbcMysqlField.FIELD_TYPE_DATE :      return Types.DATE;   
        case jdbcMysqlField.FIELD_TYPE_TIME :      return Types.TIME;
        case jdbcMysqlField.FIELD_TYPE_DATETIME :  return Types.TIMESTAMP;
        case jdbcMysqlField.FIELD_TYPE_TINY_BLOB  :  
        case jdbcMysqlField.FIELD_TYPE_VAR_STRING : 
        case jdbcMysqlField.FIELD_TYPE_STRING     :
            if((f.cflags & f.BINARY_FLAG) > 0) return Types.VARBINARY;
            else return Types.VARCHAR;
        case jdbcMysqlField.FIELD_TYPE_BLOB       : 
        case jdbcMysqlField.FIELD_TYPE_MEDIUM_BLOB:
        case jdbcMysqlField.FIELD_TYPE_LONG_BLOB  :  
            if((f.cflags & f.BINARY_FLAG) > 0) return Types.LONGVARBINARY;
            else return Types.LONGVARCHAR;
        default: return Types.OTHER;
        }
    }

static String sqlTypeNames(int type)
    {
    switch(type)
        {
        case Types.DECIMAL       : return "DECIMAL";
        case Types.CHAR          : return "CHAR";
        case Types.SMALLINT      : return "SMALLINT";
        case Types.INTEGER       : return "INTEGER";
        case Types.FLOAT         : return "FLOAT";
        case Types.DOUBLE        : return "DOUBLE"; 
        case Types.NULL          : return "NULL";
        case Types.BIGINT        : return "BIGINT";
        case Types.DATE          : return "DATE";   
        case Types.TIME          : return "TIME";
        case Types.TIMESTAMP     : return "TIMESTAMP";
        case Types.VARBINARY     : return "VARBINARY";
        case Types.LONGVARBINARY : return "LONGVARBINARY";
        case Types.LONGVARCHAR   : return "LONGVARCHAR";
        case Types.VARCHAR       : return "VARCHAR";
        default: return "OTHER";
        }
    }

private boolean isTypeNumeric(int i)
    {
    return (  i == jdbcMysqlField.FIELD_TYPE_DECIMAL
           || i == jdbcMysqlField.FIELD_TYPE_TINY
           || i == jdbcMysqlField.FIELD_TYPE_SHORT
           || i == jdbcMysqlField.FIELD_TYPE_LONG
           || i == jdbcMysqlField.FIELD_TYPE_FLOAT
           || i == jdbcMysqlField.FIELD_TYPE_DOUBLE
           || i == jdbcMysqlField.FIELD_TYPE_LONGLONG
           || i == jdbcMysqlField.FIELD_TYPE_INT24
           );
    }

/*-------------------------------------------------------------------+
|                      Error handler                                 |
+-------------------------------------------------------------------*/

void errHandlerE(int n, Exception e) throws SQLException
    {
    String o = "\n" + errs[n] + jdbcMysqlBase.eMessage(e);
    jdbcMysqlBase.errMessage(o);
    }

}

