#include "udm_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <regex.h>
#include <string.h>
#include <errno.h>

#include "udm_common.h"
#include "udm_spell.h"
#include "udm_db.h"
#include "udm_charset.h"
#include "udm_proto.h"
#include "udm_parseurl.h"
#include "udm_parser.h"
#include "udm_conf.h"
#include "udm_indexer.h"
#include "udm_log.h"
#include "udm_hrefs.h"
#include "udm_robots.h"
#include "udm_utils.h"
#include "udm_xmalloc.h"

int UdmInit(){
	UdmInitCharset();
	UdmInitDB();
	UdmInitTZ();
	return(0);
}

char *DBHost       = NULL;
char *DBName       = NULL;
char *DBUser       = NULL;
char *DBPass       = NULL;
int  DBType        = UDM_DB_UNK;
int  DBMode	   = UDM_DBMODE_SINGLE;
int  DBPort        = 0;
int  DBUseLock     = 1;
int  Force1251     = 0;
int  TrackMode     = 0;

typedef struct filter_struct {
	regex_t filter;
	int     filter_type;
	int	reverse;
	char	*regstr;
} UDM_FILTER;

static char	_extra_headers[UDMSTRSIZ]="";
static char	_user_agent[UDMSTRSIZ]="";
static int	_max_doc_size=UDM_MAXDOCSIZE;

char * UdmExtraHeaders(){
	return(_extra_headers);
}
char * UdmUserAgent(){
	return(_user_agent);
}
int UdmMaxDocSize(){
	return(_max_doc_size);
}


/* One server structure to be available on all config levels*/
/* to pass parameters */

static UDM_SERVER	*	csrv=NULL;
static UDM_SERVER	*	Server=NULL;
static UDM_FILTER	*	Filter=NULL;
static UDM_ALIAS	*	Alias=NULL;

static int nservers=0,mservers=0;
static int nfilters=0,mfilters=0;
static int naliases=0,maliases=0;

static char conf_err_str[UDMSTRSIZ]="";

/************* Base 64 ****************************************/

#define BASE64_LEN(len) (4 * (((len) + 2) / 3) +2)
/* BASE64 encoding: 3x8 bits -> 4x6 bits */
static char base64[64] =
"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

static void base64_encode (const char *s, char *store, int length){
int i;
unsigned char *p = (unsigned char *)store;

	for (i = 0; i < length; i += 3){
		*p++ = base64[s[0] >> 2];
		*p++ = base64[((s[0] & 3) << 4) + (s[1] >> 4)];
		*p++ = base64[((s[1] & 0xf) << 2) + (s[2] >> 6)];
		*p++ = base64[s[2] & 0x3f];
		s += 3;
	}
	/*Pad the result*/
	if (i == length + 1) *(p - 1) = '=';
	else if (i == length + 2) *(p - 1) = *(p - 2) = '=';
	*p = '\0';
}


/************ Allow/Disallow/CheckOnly  stuff *****************/

static int AddFilter(char * filter,int filter_type,int reverse){
#define ERRSTRSIZE 100
char regerrstr[UDMSTRSIZ]="";
int err;
	if(nfilters>=mfilters){
		if(mfilters){
			mfilters+=16;
			Filter=(UDM_FILTER *)realloc(Filter,mfilters*sizeof(UDM_FILTER));
		}else{
			mfilters=16;
			Filter=(UDM_FILTER *)malloc(mfilters*sizeof(UDM_FILTER));
		}
	}
	err=regcomp(&(Filter[nfilters].filter),filter,REG_EXTENDED|REG_ICASE);
	if(err){
		regerror(err, &(Filter[nfilters].filter), regerrstr, ERRSTRSIZE);
		sprintf(conf_err_str,"Wrong regex in config file: %s: %s", filter,regerrstr);
		regfree(&(Filter[nfilters].filter));
		return(1);
	}
	Filter[nfilters].filter_type=filter_type;
	Filter[nfilters].reverse=reverse;
	Filter[nfilters].regstr=strdup(filter);
	nfilters++;
	return(0);
}
static void FreeFilters(){
int i;
	for(i=0; i < nfilters; i++){
		regfree(&(Filter[i].filter));
		free(Filter[i].regstr);
	}
	UDM_FREE(Filter);
	nfilters=0;
	mfilters=0;
}

int UdmFilterType(char * param,char * reason){
int i;
#define NS 10
regmatch_t subs[NS];
int err;
	reason[0]=0;
	for(i=0;i<nfilters;i++){
		err=regexec(&(Filter[i].filter),param,NS,subs,0);
		if((Filter[i].reverse)&&(err))
			break;
		if((!Filter[i].reverse)&&(!err))
			break;
	}
	if(i<nfilters){
		switch(Filter[i].filter_type){
			case UDM_ALLOW:		strcpy(reason,"Allow");break;
			case UDM_DISALLOW:	strcpy(reason,"Disallow");break;
			case UDM_HEAD:		strcpy(reason,"CheckOnly");break;
			case UDM_HREFONLY:	strcpy(reason,"HrefOnly");break;
			default:		strcpy(reason,"Unknown");break;
		}
		strcpy(UDM_STREND(reason),Filter[i].reverse?"NoMatch":"");
		strcpy(UDM_STREND(reason),Filter[i].regstr?Filter[i].regstr:"");
		return(Filter[i].filter_type);
	}
	strcpy(reason,"Allow by default");
	return(UDM_ALLOW);
}



/******************** ALIASES ***************************************/

static int AddAlias(char * find,char *replace){
	if(naliases>=maliases){
		if(maliases){
			maliases+=16;
			Alias=(UDM_ALIAS *)realloc(Alias,maliases*sizeof(UDM_ALIAS));
		}else{
			maliases=16;
			Alias=(UDM_ALIAS *)malloc(maliases*sizeof(UDM_ALIAS));
		}
	}
	Alias[naliases].find=strdup(find);
	Alias[naliases].replace=strdup(replace);
	naliases++;
	return(0);
}
static void FreeAliases(){
int i;
	for(i=0;i<naliases;i++){
		free(Alias[i].find);
		free(Alias[i].replace);
	}
	naliases=0;
	maliases=0;
	UDM_FREE(Alias);
}

UDM_ALIAS * UdmFindAlias(char * str){
int j;
	for(j=0;j<naliases;j++){
		if(!UDM_STRNCMP(str,Alias[j].find)){
			return(&Alias[j]);
		}
	}
	return(NULL);
}


/*************************** Servers *************************************/

#define DEFAULT_PROXY_PORT	3128

static int AddServer(UDM_SERVER * srv){

	if(nservers>=mservers){
		if(mservers){
			mservers+=16;
			Server=(UDM_SERVER *)realloc(Server,mservers*sizeof(UDM_SERVER));
		}else{
			mservers=16;
			Server=(UDM_SERVER *)malloc(mservers*sizeof(UDM_SERVER));
		}
	}
	Server[nservers].url=strdup(srv->url);
	Server[nservers].charset=srv->charset?strdup(srv->charset):NULL;
	Server[nservers].basic_auth=srv->basic_auth?strdup(srv->basic_auth):NULL;
	Server[nservers].htdb_list=srv->htdb_list?strdup(srv->htdb_list):NULL;
	Server[nservers].htdb_doc=srv->htdb_doc?strdup(srv->htdb_doc):NULL;
        Server[nservers].user=srv->user?strdup(srv->user):NULL;
        Server[nservers].passwd=srv->passwd?strdup(srv->passwd):NULL;
        Server[nservers].category=srv->category?strdup(srv->category):NULL;
        Server[nservers].tag=srv->tag?strdup(srv->tag):NULL;
	if(srv->proxy){
		char *s;
		Server[nservers].proxy=strdup(srv->proxy);
		if((s=strchr(Server[nservers].proxy,':'))){
			*s=0;
			Server[nservers].proxy_port=atoi(s+1)?atoi(s+1):DEFAULT_PROXY_PORT;
		}else{
			Server[nservers].proxy_port=DEFAULT_PROXY_PORT;
		}
	}else{
		Server[nservers].proxy=NULL;
		Server[nservers].proxy_port=0;
	}
	Server[nservers].period=srv->period;
	Server[nservers].net_errors=0;
	Server[nservers].outside=srv->outside;
	Server[nservers].maxhops=srv->maxhops;
	Server[nservers].gindex=srv->gindex;
	Server[nservers].gfollow=srv->gfollow;
	Server[nservers].deletebad=srv->deletebad;
	Server[nservers].userobots=srv->userobots;
	Server[nservers].bodyweight=srv->bodyweight;
	Server[nservers].titleweight=srv->titleweight;
	Server[nservers].urlweight=srv->urlweight;
	Server[nservers].urlhostweight=srv->urlhostweight;
	Server[nservers].urlpathweight=srv->urlpathweight;
	Server[nservers].urlfileweight=srv->urlfileweight;
	Server[nservers].descweight=srv->descweight;
	Server[nservers].keywordweight=srv->keywordweight;
	Server[nservers].max_net_errors=srv->max_net_errors;
	Server[nservers].read_timeout=srv->read_timeout;
	Server[nservers].delete_no_server=srv->delete_no_server;
	Server[nservers].correct_factor=srv->correct_factor;
	Server[nservers].incorrect_factor=srv->incorrect_factor;
	Server[nservers].number_factor=srv->number_factor;
	Server[nservers].alnum_factor=srv->alnum_factor;
	Server[nservers].min_word_length=srv->min_word_length;
	Server[nservers].max_word_length=srv->max_word_length;
	Server[nservers].use_mirror=srv->use_mirror;
	Server[nservers].use_clones=srv->use_clones;
	nservers++;
	return(0);
}
static void FreeServers(){
int i;
	for(i=0;i<nservers;i++){
		UDM_FREE(Server[i].url);
		UDM_FREE(Server[i].charset);
		UDM_FREE(Server[i].basic_auth);
		UDM_FREE(Server[i].htdb_list);
		UDM_FREE(Server[i].htdb_doc);
		UDM_FREE(Server[i].proxy);
		UDM_FREE(Server[i].category);
		UDM_FREE(Server[i].tag);
	}
	nservers=mservers=0;
	UDM_FREE(Server);
}
UDM_SERVER * UdmFindServer(char *url){
int i;
char *r;

	if(!Server)return(NULL);
	url=UdmRemove2Dot(url);
	r=strstr(url,"/robots.txt");
	/* If it's a robot.txt, cut to hostinfo and find result */
	if((r)&&(!strcmp(r,"/robots.txt"))){
		r++;*r=0;
		for(i=0;i<nservers;i++){
			if(!UDM_STRNCMP(url,Server[i].url)){
				*r='r';
				return(&Server[i]);
			}
		}
		*r='r';
	}else{
		for(i=0;i<nservers;i++)
			if(!UDM_STRNCMP(url,Server[i].url))
				return(&Server[i]);
	}
	return(0);
}
static int cmpserver(const void *s1,const void *s2){
	return(strlen(((UDM_SERVER*)s2)->url)-strlen(((UDM_SERVER*)s1)->url));
}
static int InitServer(UDM_SERVER * srv){
	srv->url=NULL;
	srv->proxy=NULL;
	srv->basic_auth=NULL;
	srv->charset=NULL;
	srv->htdb_list=NULL;
	srv->htdb_doc=NULL;
	srv->category=NULL;
	srv->proxy_port=0;
	srv->period=UDM_DEFAULT_REINDEX_TIME;
	srv->tag=NULL;
	srv->net_errors=0;
	srv->max_net_errors=UDM_MAXNETERRORS;
	srv->read_timeout=UDM_READ_TIMEOUT;
	srv->outside=0;
	srv->delete_no_server=1;
	srv->maxhops=UDM_DEFAULT_MAX_HOPS;
	srv->gindex=1;
	srv->gfollow=1;
	srv->deletebad=0;
	srv->userobots=1;
	srv->bodyweight=1;
	srv->titleweight=2;
	srv->keywordweight=2;
	srv->descweight=2;
	srv->urlweight=0;
	srv->urlhostweight=0;
	srv->urlpathweight=0;
	srv->urlfileweight=0;
	
	srv->correct_factor=1;
	srv->incorrect_factor=1;
	
	srv->number_factor=1;
	srv->alnum_factor=1;
	srv->min_word_length=1;
	srv->max_word_length=32;
	srv->use_mirror=UDM_MIRROR_NO;
	srv->use_clones=1;
	return(0);
}

__INDLIB__ char * UdmConfErrMsg(){
	return(conf_err_str);
}

/**************************** DBAddr ***********************************/
static int UdmStr2DBMode(char * str1){
int m;
	m = UDM_DBMODE_SINGLE;
	if(!UDM_STRNCASECMP(str1,"multi-crc"))m=UDM_DBMODE_MULTI_CRC;
	else
	if(!UDM_STRNCASECMP(str1,"crc-multi"))m=UDM_DBMODE_MULTI_CRC;
	else
	if(!UDM_STRNCASECMP(str1,"single"))m=UDM_DBMODE_SINGLE;
	else
	if(!UDM_STRNCASECMP(str1,"crc"))m=UDM_DBMODE_SINGLE_CRC;
	else
	if(!UDM_STRNCASECMP(str1,"multi"))m=UDM_DBMODE_MULTI;
	else
	if(!UDM_STRNCASECMP(str1,"word2url"))m=UDM_DBMODE_WORD2URL;
	return(m);
}
__INDLIB__ int UdmSetTrackMode(int mode){
	TrackMode=mode;
	return(TrackMode);
}
__INDLIB__ int UdmSetDBMode(char *str){
	DBMode=UdmStr2DBMode(str);
	return(DBMode);
}
static int UdmStr2DBType(char *str1) {
int t=UDM_DB_UNK;
	if(!UDM_STRNCASECMP(str1,"msql"))t=UDM_DB_MSQL;
	else
	if(!UDM_STRNCASECMP(str1,"solid"))t=UDM_DB_SOLID;
	else
	if(!UDM_STRNCASECMP(str1,"oracle"))t=UDM_DB_ORACLE;
	else
	if(!UDM_STRNCASECMP(str1,"mssql"))t=UDM_DB_MSSQL;
	else
	if(!UDM_STRNCASECMP(str1,"mysql"))t=UDM_DB_MYSQL;
	else
	if(!UDM_STRNCASECMP(str1,"pgsql"))t=UDM_DB_PGSQL;
	else
	if(!UDM_STRNCASECMP(str1,"ibase"))t=UDM_DB_IBASE;
	return(t);
}
__INDLIB__ int UdmSetDBType(char *str){
	DBType=UdmStr2DBType(str);
	return(DBType);
}
__INDLIB__ int UdmSetDBAddr(char * str){
UDM_URL url;
char *s;
int res;

	if((res=UdmParseURL(&url,str)))
		return(res);

	DBHost=strdup(url.hostname);
	DBType=UdmStr2DBType(url.schema);

	if(DBType==UDM_DB_IBASE){
		/* Ibase is a special case        */
		/* It's database name consists of */
		/* full path and file name        */ 
		DBName=malloc(strlen(url.path)+1+strlen(url.filename));
		sprintf(DBName,"%s%s",url.path,url.filename);
	}else{
		DBName=strdup(url.path);
		sscanf(url.path,"/%[^/]s",DBName);
	}

	DBPort=url.port;
	if((s=strchr(url.auth,':'))){
		*s=0;
		DBUser=strdup(url.auth);
		DBPass=strdup(s+1);
		*s=':';
	}else{
		DBUser=strdup(url.auth);
	}
	
	return(0);
}

/****************************  Load Configuration **********************/


#define IF_NUM_PARAM(src,name,dst,def) \
if(!UDM_STRNCASECMP(src,name)){if(sscanf(src+strlen(name),"%d",&(dst))!=1)dst=def;}

#define IF_BOOL_PARAM(src,name,dst,def) \
if(!UDM_STRNCASECMP(src,name)){ \
char xxx[1024]; \
if(sscanf(src+strlen(name),"%s",xxx)==1){ \
	if(!UDM_STRNCASECMP(xxx,"yes"))dst=1; \
	else	dst=0; \
}else \
	dst=def; \
}


__INDLIB__ int UdmLoadConfig(char *conf_name,int config_level,int load_flags){
int i=0; /* For line number */
FILE *config;
char str[UDMSTRSIZ]="";
char str1[UDMSTRSIZ]="";
char str2[UDMSTRSIZ]="";
char *ch;
int has_dict=0;


	if(config_level==0){ /* Do some initialization */
		csrv=(UDM_SERVER*)UdmXmalloc(sizeof(UDM_SERVER));
		InitServer(csrv);
		UdmSetMaxURLNumber(-1);
		sprintf(_user_agent,"%s/%s",UDM_USER_AGENT,VERSION);
		_extra_headers[0]=0;
		_max_doc_size=UDM_MAXDOCSIZE;
		UdmSetLogFacility("");
		Force1251=0;
		DBPort=0;
		UDM_FREE(DBHost);
		UDM_FREE(DBName);
		UDM_FREE(DBUser);
		UDM_FREE(DBPass);
		UdmSetDefaultCharset(UDM_CHARSET_NO);
		FreeFilters();
		FreeAliases();
		UDM_FREE(MirrorRoot);
		UDM_FREE(MirrorHeadersRoot);
		UdmAddURLFile(NULL);

		/* I have removed it to allow adding of URLs  */
		/* Passed in -i with -u command line args     */
		/* But it seems that UnloadConf() is required */
		/* REMOVED      UdmFreeHrefs(); */

		UdmFreeTypes();
		FreeServers();
	}

	/* Open config file */
	if(!(config=fopen(conf_name,"r"))){
		sprintf(conf_err_str,"Error: can't open config file '%s': %s",conf_name, strerror(errno));
		return(1);
	}

	/*  Read lines and parse */
	while(fgets(str1,sizeof(str1),config)){
		char *end;
		i++;
		if(!str1[0])continue;
		end=str1+strlen(str1)-1;
		while((end>=str1)&&(*end=='\r'||*end=='\n')){
			*end=0;if(end>str1)end--;
		}
		if(!str1[0])continue;
		if(str1[0]=='#')continue;

		if(*end=='\\'){
			*end=0;strcat(str,str1);
			continue;
		}
		strcat(str,str1);
		strcpy(str1,"");

		if(!UDM_STRNCASECMP(str,"DBAddr")){
			if(UdmSetDBAddr(UdmTrim(str+6," \r\n\t"))){
				sprintf(conf_err_str,"Bad DBAddr in config file '%s' line %d:%s",conf_name,i,str);
				return(1);
			}
		}else
		if(!UDM_STRNCASECMP(str,"DBHost")){
			DBHost=strdup(UdmTrim(str+6," \r\n\t"));
		}else
		if(!UDM_STRNCASECMP(str,"DBName")){
			DBName=strdup(UdmTrim(str+6," \r\n\t"));
		}else
		if(!UDM_STRNCASECMP(str,"DBUser")){
			DBUser=strdup(UdmTrim(str+6," \r\n\t"));
		}else
		if(!UDM_STRNCASECMP(str,"DBPass")){
			DBPass=strdup(UdmTrim(str+6," \r\n\t"));
		}else
		IF_NUM_PARAM(str,"DBPort",DBPort,0)
		else
		if(!UDM_STRNCASECMP(str,"LocalCharset")){
			UdmSetDefaultCharset(UdmGetCharset(UdmTrim(str+13," \r\n\t")));
		}else
		if(!UDM_STRNCASECMP(str,"DBType")){
			UdmSetDBType(UdmTrim(str+6," \r\n\t"));
		}else
		if(!UDM_STRNCASECMP(str,"DBMode")){
			DBMode=UdmStr2DBMode(UdmTrim(str+6," \r\n\t"));
		}else
		if(!UDM_STRNCASECMP(str,"DisallowNoMatch")){
			char *s,*lt;
			s=UdmGetToken(str," \t\r\n",&lt);
			while((s=UdmGetToken(NULL," \t\r\n",&lt))){
				if(AddFilter(s,UDM_DISALLOW,1))
					return(1);
			}
		}else
		if(!UDM_STRNCASECMP(str,"Disallow")){
			char *s,*lt;
			s=UdmGetToken(str," \t\r\n",&lt);
			while((s=UdmGetToken(NULL," \t\r\n",&lt))){
				if(AddFilter(s,UDM_DISALLOW,0))
					return(1);
			}
		}else
		if(!UDM_STRNCASECMP(str,"AllowNoMatch")){
			char *s,*lt;
			s=UdmGetToken(str," \t\r\n",&lt);
			while((s=UdmGetToken(NULL," \t\r\n",&lt))){
				if(AddFilter(s,UDM_ALLOW,1))
					return(1);
			}
		}else
		if(!UDM_STRNCASECMP(str,"Allow")){
			char *s,*lt;
			s=UdmGetToken(str," \t\r\n",&lt);
			while((s=UdmGetToken(NULL," \t\r\n",&lt))){
				if(AddFilter(s,UDM_ALLOW,0))
					return(1);
			}
		}else
		if(!UDM_STRNCASECMP(str,"CheckOnlyNoMatch")){
			char *s,*lt;
			s=UdmGetToken(str," \t\r\n",&lt);
			while((s=UdmGetToken(NULL," \t\r\n",&lt))){
				if(AddFilter(s,UDM_HEAD,1))
					return(1);
			}
		}else
		if(!UDM_STRNCASECMP(str,"CheckOnly")){
			char *s,*lt;
			s=UdmGetToken(str," \t\r\n",&lt);
			while((s=UdmGetToken(NULL," \t\r\n",&lt))){
				if(AddFilter(s,UDM_HEAD,0))
					return(1);
			}
		}else
		if(!UDM_STRNCASECMP(str,"HrefOnly")){
			char *s,*lt;
			s=UdmGetToken(str," \t\r\n",&lt);
			while((s=UdmGetToken(NULL," \t\r\n",&lt))){
				if(AddFilter(s,UDM_HREFONLY,0))
					return(1);
			}
		}else
		if(!UDM_STRNCASECMP(str,"HrefOnlyNoMatch")){
			char *s,*lt;
			s=UdmGetToken(str," \t\r\n",&lt);
			while((s=UdmGetToken(NULL," \t\r\n",&lt))){
				if(AddFilter(s,UDM_HREFONLY,1))
					return(1);
			}
		}else
		if(!UDM_STRNCASECMP(str,"AddType")){
			char *s,*s1,*lt;
			s=UdmGetToken(str," \t\r\n",&lt);
			if((s1=UdmGetToken(NULL," \t\r\n",&lt)))
				while((s=UdmGetToken(NULL," \t\r\n",&lt)))
					UdmAddType(s1,s,conf_err_str);
		}else
		if(!UDM_STRNCASECMP(str,"CharSet")){
			UDM_FREE(csrv->charset);
			if(sscanf(str+7,"%s",str1))
				csrv->charset=strdup(str1);
		}else
		if(!UDM_STRNCASECMP(str,"Proxy")){
			UDM_FREE(csrv->proxy);
			if(sscanf(str+5,"%s",str1)&&strcmp(str1,""))
				csrv->proxy=strdup(str1);
		}else
		if(!UDM_STRNCASECMP(str,"Category")){
			UDM_FREE(csrv->category);
			if(sscanf(str+9,"%s",str1))
			csrv->category=strdup(str1);
		}else
		if(!UDM_STRNCASECMP(str,"Tag")){
			UDM_FREE(csrv->tag);
			if(sscanf(str+3,"%s",str1))
			csrv->tag=strdup(str1);
		}else
		if(!UDM_STRNCASECMP(str,"HTTPHeader")){
			if(sscanf(str+11,"%[^\n\r]s",str1)&&strcmp(str1,"")){
				if(!UDM_STRNCMP(str1,"User-Agent: ")){
					strcpy(_user_agent,str1+12);
				}else{
					strcat(_extra_headers,str1);
					strcat(_extra_headers,"\r\n");
				}
			}
		}else
#ifdef HAVE_SYSLOG_H
		if(!UDM_STRNCASECMP(str,"SyslogFacility")){
			if(sscanf(str+14,"%s",str1)&&strcmp(str1,""))
				UdmSetLogFacility(str1);
		}else
#endif
		if(!UDM_STRNCASECMP(str,"AuthBasic")){
			UDM_FREE(csrv->basic_auth);
			UDM_FREE(csrv->user);
			UDM_FREE(csrv->passwd);
			if(sscanf(str+9,"%s",str1)){
				csrv->basic_auth=(char*)malloc(BASE64_LEN(strlen(str1)));
				base64_encode(str1,csrv->basic_auth,strlen(str1));
                                if ((ch=strchr(str1, ':'))){
                            		csrv->user=(char*)UdmXmalloc(ch-str1+1);
                            		udm_snprintf(csrv->user, ch-str1+1, "%s", str1);
                            		csrv->passwd=(char*)UdmXmalloc(strlen(ch)+1);
                            		sprintf(csrv->passwd, "%s", ch+1);
                                }

			}
		}else
		if(!UDM_STRNCASECMP(str,"HTDBList")){
			UDM_FREE(csrv->htdb_list);
			csrv->htdb_list=strdup(UdmTrim(str+8," \t\r\n"));
		}else
		if(!UDM_STRNCASECMP(str,"HTDBDoc")){
			UDM_FREE(csrv->htdb_doc);
			csrv->htdb_doc=strdup(UdmTrim(str+8," \t\r\n"));
		}else
		if(!UDM_STRNCASECMP(str,"Mime")){
			UdmAddParser(str+4);
		}else
		if(!UDM_STRNCASECMP(str,"Alias")){
			if(sscanf(str+5,"%s%s",str1,str2)==2)
				AddAlias(str1,str2);
			else{
				sprintf(conf_err_str,"Error in config file '%s' line %d:%s",conf_name,i,str);
				return(1);
			}
		}else
		if(!UDM_STRNCASECMP(str,"Server")){
			UDM_URL from;
			int res;
			if(sscanf(str+6,"%s",str1)!=1)
				continue;
			if((res=UdmParseURL(&from,str1))){
				switch(res){
				case UDM_PARSEURL_LONG:
					sprintf(conf_err_str,"File '%s' line %d: URL too long: %s",conf_name,i,str1);
					break;
				case UDM_PARSEURL_BAD:
				default:
					sprintf(conf_err_str,"File '%s' line %d: Bad formed URL: %s",conf_name,i,str1);
				}
				return(1);
			}
			csrv->url=strdup(str1);
			AddServer(csrv);
			if((!strcmp(from.schema,"http"))&&(load_flags&UDM_FLAG_ADD_SERV)&&(csrv->userobots)){
				sprintf(str,"%s://%s/%s",from.schema,from.hostinfo,"robots.txt");
				UdmAddHref(NULL,str,0,0,0);
			}
			if(load_flags&UDM_FLAG_ADD_SERV)
				UdmAddHref(NULL,str1,0,0,0);
		}
		else	IF_BOOL_PARAM(str,"FollowOutside",csrv->outside,0)
		else	IF_BOOL_PARAM(str,"Index",csrv->gindex,1)
		else	IF_BOOL_PARAM(str,"Follow",csrv->gfollow,1)
		else	IF_BOOL_PARAM(str,"Robots",csrv->userobots,1)
		else	IF_BOOL_PARAM(str,"DeleteBad",csrv->deletebad,0)
		else	IF_BOOL_PARAM(str,"DeleteNoServer",csrv->delete_no_server,1)
		else	IF_BOOL_PARAM(str,"ForceIISCharset1251",Force1251,0)
		else	IF_BOOL_PARAM(str,"Clones",csrv->use_clones,1)
		
		else	IF_NUM_PARAM(str,"MaxNetErrors",csrv->max_net_errors,UDM_MAXNETERRORS)
		else	IF_NUM_PARAM(str,"ReadTimeOut",csrv->read_timeout,UDM_READ_TIMEOUT)
		else	IF_NUM_PARAM(str,"Period",csrv->period,UDM_DEFAULT_REINDEX_TIME)
		else	IF_NUM_PARAM(str,"MaxHops",csrv->maxhops,0)
		else	IF_NUM_PARAM(str,"BodyWeight",csrv->bodyweight,1)
		else	IF_NUM_PARAM(str,"TitleWeight",csrv->titleweight,2)
		else	IF_NUM_PARAM(str,"DescWeight",csrv->descweight,2)
		else	IF_NUM_PARAM(str,"UrlWeight",csrv->urlweight,0)
		else    IF_NUM_PARAM(str,"UrlHostWeight",csrv->urlhostweight,0)
		else    IF_NUM_PARAM(str,"UrlPathWeight",csrv->urlpathweight,0)
		else    IF_NUM_PARAM(str,"UrlFileWeight",csrv->urlfileweight,0)
		else	IF_NUM_PARAM(str,"IspellCorrectFactor",csrv->correct_factor,1)
		else	IF_NUM_PARAM(str,"IspellIncorrectFactor",csrv->incorrect_factor,1)
		else	IF_NUM_PARAM(str,"NumberFactor",csrv->number_factor,1)
		else	IF_NUM_PARAM(str,"AlnumFactor",csrv->alnum_factor,1)
		else	IF_NUM_PARAM(str,"MinWordLength",csrv->min_word_length,1)
		else	IF_NUM_PARAM(str,"MaxWordLength",csrv->max_word_length,32)
		else	IF_NUM_PARAM(str,"KeywordWeight",csrv->keywordweight,2)
		else	IF_NUM_PARAM(str,"MaxDocSize",_max_doc_size,UDM_MAXDOCSIZE)
		else	if(!UDM_STRNCASECMP(str,"NoIndex"))csrv->gindex=0;
		else	if(!UDM_STRNCASECMP(str,"NoFollow"))csrv->gfollow=0;
		else
		if(!UDM_STRNCASECMP(str,"Affix")){
			char lang[20];
			if(load_flags&&UDM_FLAG_SPELL){
				if(2==sscanf(str+5,"%s%s",lang,str1)){
					if(*str1=='/')strcpy(str,str1);
					else	sprintf(str,"%s/%s",UDM_CONF_DIR,str1);
					if(UdmImportAffixes(lang,str,NULL,0)){
						sprintf(conf_err_str,"Can't load affix :%s",str);
						return(1);
					}
				}
				else {
					sprintf(conf_err_str,"Error in config file '%s' line %d:%s",conf_name,i,str);
					return(1);
				}
			}
		}else
		if(!UDM_STRNCASECMP(str,"Spell")){
			char lang[20];
			if(load_flags&&UDM_FLAG_SPELL){
				if(2==sscanf(str+5,"%s%s",lang,str1)){
					if(*str1=='/')strcpy(str,str1);
					else	sprintf(str,"%s/%s",UDM_CONF_DIR,str1);
					/* -----  0 or 1 ?  */
					if(UdmImportDictionary(lang,str,0,"")){
						sprintf(conf_err_str,"Can't load dictionary :%s",str);
						return(1);
					}
					has_dict=1;
				}else{
					sprintf(conf_err_str,"Error in config file '%s' line %d:%s",conf_name,i,str);
					return(1);
				}
			}
		}else
		if(!UDM_STRNCASECMP(str,"Include")){
			if((sscanf(str+7,"%s",str1))&&(config_level<5)){
				if(*str1=='/')strcpy(str,str1);
				else	sprintf(str,"%s/%s",UDM_CONF_DIR,str1);
				UdmLoadConfig(str,config_level+1,load_flags);
			}
		}else
		if(!UDM_STRNCASECMP(str,"MirrorRoot")){
			if((sscanf(str+10,"%s",str1))){
				if(*str1=='/')
					strcpy(str,str1);
				else
					sprintf(str,"%s/%s",UDM_CONF_DIR,str1);
			} else {
				sprintf(str,"%s/mirrors",UDM_CONF_DIR);
			}
			MirrorRoot = strdup(str);
			csrv->use_mirror=(csrv->use_mirror>0)?csrv->use_mirror:0;
		}else
		if(!UDM_STRNCASECMP(str,"MirrorHeadersRoot")){
			if((sscanf(str+17,"%s",str1))){
				if(*str1=='/')
					strcpy(str,str1);
				else
					sprintf(str,"%s/%s",UDM_CONF_DIR,str1);
			} else {
				sprintf(str,"%s/headers",UDM_CONF_DIR);
			}
			MirrorHeadersRoot = strdup(str);
			csrv->use_mirror=(csrv->use_mirror>0)?csrv->use_mirror:0;
		}else
		IF_NUM_PARAM(str,"MirrorPeriod",csrv->use_mirror,UDM_MIRROR_NO)
		else{
			sprintf(conf_err_str,"Error in config file line %d: %s",i,str);
			return(1);
		}
		str[0]=0;
	}
	fclose(config);

	/* On level0 : Free some variables, prepare others, etc */
	if(config_level==0){
		/* Add one virtual server if we want FollowOutside */
		/* or DeleteNoServer no	*/
		
		if((csrv->outside)||(!csrv->delete_no_server)){
			UDM_FREE(csrv->url);
			csrv->url="";
			AddServer(csrv);
		}else{
			UDM_FREE(csrv->url);
		}
		UDM_FREE(csrv->charset);
		UDM_FREE(csrv->proxy);
		UDM_FREE(csrv->basic_auth);
		UDM_FREE(csrv->htdb_list);
		UDM_FREE(csrv->htdb_doc);
		/*  Long name should be found first    */
		/*  to allow different options         */
		/*  for server and it's subdirectories */
		qsort((void*)Server,nservers,sizeof(UDM_SERVER),cmpserver);
		/* Sort ispell dictionay if it has been loaded */
		if(has_dict)UdmSortDictionary();
	}
	return(0);
}
