#!/usr/local/bin/perl
#
# ObjHier 27/07/2002
#
# cafeterra : data flow and data replication management
# Copyright (C) 2001  Abdellaziz TALEB
#
#This program is free software; you can redistribute it and/or
#modify it under the terms of the GNU General Public License
#as published by the Free Software Foundation; either version 2
#of the License, or (at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
#

#Class standard attributs
#
#SEE cafREP.pl for => database handle
#
#

use 5.005;
use strict;

package ObjHier;
 
# CLASS DESCRIPTION
# ATTRIBUTE :
#	1 - OBJS - ALL OBJECTS
#	2 - ATTRS
#	3 - VARS
#	4 - OBJOBJS - DEPENDANT OBJECTS
#	5 - ALLOBJS - ARRAY OF ALL OBJECT IDS
#	6 - UPTODATE - HASH OF ALL UPTODATE OBJECTS (when called by REPOSITORY)
#	7 - RELOAD   - OBJECTS THAT NEED TO BE RELOADED (when called by QUEUE)
#	8 - OBJBYID  - (QUEUE)
#	9 - OBJBYNAME - (QUEUE)
#	10 - FLOW -> SUBFLOWS, SCRIPTS, QUERIES - (QUEUE)
#	11 - SUBFLOWS -> CONNECTOR, CONTAINER, FIELDS, SCRIPTS, QUERIES - (QUEUE)
#  12 - ATTRIBUTES
#  13 - VARS
#  14 - EVENTS
#
#	15 - CLOSURES
#
# Deploy objects subs 

sub new {
	my $class = shift;
	my $dbh = shift;
	my $running = shift; #REPOSITORY - QUEUE
	my $contextid = shift;
	my $context = shift;

	my $objstatus = ($running eq "QUEUE") ? "online" : "%";
	bless 
		my $self = {
			__SPECIAL => {
				DBH => $dbh,
				RUNNING => $running,
				CONTEXTID => $contextid,
				CONTEXT => $context,
				STATUS => $objstatus,
			},
			CONTEXT => $context
			};
	bless $self, $class;
}

sub lastupdate {
	my $self = shift;

	if (@_) {
		$self->{__SPECIAL}{LASTUPDATE} = shift;
	}
	$self->{__SPECIAL}{LASTUPDATE};
}

sub dbh {
	my $self = shift;

	if (@_) {
		$self->{__SPECIAL}{DBH} = shift;
	}
	$self->{__SPECIAL}{DBH};
}

sub contextid {
	my $self = shift;

	if (@_) {
		$self->{__SPECIAL}{CONTEXTID} = shift;
	}
	$self->{__SPECIAL}{CONTEXTID};
}

sub context {
	my $self = shift;

	if (@_) {
		$self->{CONTEXT} = shift;
		$self->{__SPECIAL}{CONTEXT} = $self->{CONTEXT};
	}
	$self->{__SPECIAL}{CONTEXT};
}

sub running {
	my $self = shift;

	if (@_) {
		$self->{__SPECIAL}{RUNNING} = shift;
	}
	$self->{__SPECIAL}{RUNNING};
}

sub outdbh {
	my $self = shift;

	if (@_) {
		$self->{__SPECIAL}{OUTDBH} = shift;
	}
	$self->{__SPECIAL}{OUTDBH};
}

sub objstatus {
	my $self = shift;

	if (@_) {
		$self->{__SPECIAL}{STATUS} = shift;
	}
	$self->{__SPECIAL}{STATUS};
}

sub getscripthierarchy {
	my $self = shift;
	my $objid = shift;

	my $query;

	unless ($self->getobject($objid)) {
		my $object = $self->getdbhobject ($objid);
		unless ($self->getdata($objid)) {
			my $script = $self->getdbhdata($objid, 'scripts');

			my $dbh = $self->dbh();
			$query = $dbh->newquery( { tablename => "pscripts" });
			$query->stablerow([ [ "object_id", "=", $objid ] ]);
			my $parsetext = $dbh->hexecfetchrownop($query, 1);

			if ($parsetext->{parsetext} =~ /parsedtext\s*=/) {
				my $parsedtext; 
				eval $parsetext->{parsetext};
				$script->{parsetext} = $parsetext;
				$script->{parsedtext} = $parsedtext;
				$self->updateobj($objid, undef, $script);
			}
		}
	}

}#getscripthierarchy

sub getuserhierarchy {
	my $self = shift;
	my $objid = shift;

	my $query;

	my $object = $self->getdbhobject ($objid);
	my $user = $self->getdbhdata($objid, 'users');

}#getuserhierarchy

sub getserverhierarchy {
	my $self = shift;
	my $objid = shift;

	my $query;

	my $object = $self->getdbhobject ($objid);
	my $server = $self->getdbhdata($objid, 'server');

}#getserverhierarchy

sub getfieldhierarchy {
	my $self = shift;
	my $objid = shift;

	my $query;

	my $object = $self->getdbhobject ($objid);
	my $field = $self->getdbhdata($objid, 'field');

}#getfieldhierarchy

sub getproxyhierarchy {
	my $self = shift;
	my $objid = shift;

	my $object = $self->getdbhobject ($objid);
	if ($object and ($object->{id} == $objid)) {
		my $proxy = $self->getdbhdata($objid, 'cafproxy');

		my $attrdef;
		$attrdef->{object_id} = $objid;
		$attrdef->{objtype} = "proxy";
#		$attrdef->{protocolid} = $proxy->{protocolid};
#		$attrdef->{driverid} = $proxy->{driverid};

		$self->getdbhattrs($objid, $attrdef);
	}
}#getfieldhierarchy

sub getconnectorhierarchy {
	my $self = shift;
	my $objid = shift;
	my $nocontext = shift;
	my $contextid = $nocontext ? undef : $self->contextid();

	my $query;

	my $object = $self->getdbhobject ($objid);
	my $connector = $self->getdbhdata($objid, 'connector');

# Check if the connector has an alias for the actual context
	if ($contextid and ($connector->{ismaster} eq 'master')) {
		my $connector2;
		my $object2;
		my $dbh = $self->dbh();
		$query = $dbh->newquery( { tablename => "connector" });
		$query->stablerow([ [ "master_id", "in", [$objid] ], [ "contextid", "in", ["'$contextid'"] ], [ "ismaster", "=", "slave" ] ]);
		$connector2 = $dbh->hexecfetchrownop($query, 1);
		if ($connector2) {
			$query = $dbh->newquery({ id => $connector2->{object_id}, status => $self->objstatus() });
			$query->sobjectlist();
			$object2 = $dbh->hexecfetchrownop($query);

			$connector2->{object_id} = $objid;
			$connector2->{id} = $objid;
			$connector2->{master_id} = undef;
			$connector2->{ismaster} = 'master';
			$object2->{id} = $objid;
			$object2->{object_id} = $objid;

			$connector = $connector2;
			$object = $object2;

			$self->updateobj($objid, $object, $connector);
		}
	}

	# my $serverid = 
		
	my $server = $self->getdbhobject ($object->{parent_id});
	
	if ($server->{type} eq "server") {
		unless ($self->getdata($server->{id})) {
			$self->getserverhierarchy($server->{id});
		}
	}
	$self->myobject ($objid, $server->{object_id});

#	my $userid =

	my $user = $self->getdbhobject ($connector->{userid});
	
	if ($user->{type} eq "users") {
		unless ($self->getdata($user->{id})) {
			$self->getuserhierarchy($user->{id});
		}
	}
	$self->myobject ($objid, $user->{object_id});

	my $attrdef;
	$attrdef->{object_id} = $objid;
	$attrdef->{objtype} = "connector";
	$attrdef->{protocolid} = $connector->{protocolid};
	$attrdef->{driverid} = $connector->{driverid};

	if ($connector->{proxy_id} and ($connector->{proxy_id} > 0)) {
		unless ($self->getdata($connector->{proxy_id})) {
			$self->getproxyhierarchy($connector->{proxy_id});
		}
	}

	$self->getdbhattrs($objid, $attrdef);

	$self->getdbhvars($objid);

	$self->getdbhotherobjects($objid);

}#getconnectorhierarchy

sub getcontainerhierarchy {
	my $self = shift;
	my $objid = shift;

	my $query;
	my $object;
	my $container;

	my $object = $self->getdbhobject ($objid);
	my $container = $self->getdbhdata($objid, 'container');

	my $connector = $self->getdbhobject ($object->{parent_id});
	$self->myobject ($objid, $object->{parent_id});
	
	if ($connector->{type} eq "connector") {
		unless ($self->getdata($connector->{object_id})) {
			$self->getconnectorhierarchy($connector->{object_id});
		}

		$connector = $self->getobject($connector->{object_id});
	}

	my $fields = $self->getdbhobjectchilds($objid);

	foreach my $field (@$fields) {
		if ($field->{type} eq "field") {
			$self->fieldbyid($objid, $field->{id}, $field->{name});
			$self->myobject ($objid, $field->{object_id});
			$self->addobj($field->{object_id}, $field);
			unless ($self->getdata($field->{id})) {
				$self->getfieldhierarchy($field->{id});
			}
		}
	}

	my $attrdef;
	$attrdef->{object_id} = $objid;
	$attrdef->{objtype} = "container";
	$attrdef->{protocolid} = $connector->{protocolid};
	$attrdef->{driverid} = $connector->{driverid};

	$self->getdbhattrs($objid, $attrdef);
	$self->myobjectsfromchild ($objid, $object->{parent_id});

}#getcontainerhierarchy

sub getsubflowhierarchy {
	my $self = shift;
	my $objid = shift;

	my $query;
	my @objects;
	my $object = $self->getdbhobject ($objid);
	my $subflow = $self->getdbhdata($objid, 'subflow');

	my $container = $self->getdbhobject ($subflow->{container_id});

	if ($container->{type} eq "container") {
		unless ($self->getdata($container->{id})) {
			$self->getcontainerhierarchy($container->{id});
		}
	}
	$self->myobject ($objid, $subflow->{container_id});
	$self->mysubflows ($object->{parent_id}, $subflow->{flowdirection}, $objid);
	$self->myobjectsfromchild ($objid, $subflow->{container_id});

	$self->getdbhmapandscripts ($objid);

	my $attrdef;
	$attrdef->{object_id} = $objid;
	$attrdef->{objtype} = "subflow";
	$attrdef->{protocolid} = undef;
	$attrdef->{driverid} = undef;

	$self->getdbhattrs($objid, $attrdef);

	$self->getdbhvars($objid);

	$self->getdbhevents($objid);

	$self->getdbhotherobjects($objid);
}


sub getflowhierarchy {
	my $self = shift;
	my $objid = shift;

	my $query;
	my $object = $self->getdbhobject ($objid);
	my $data = $self->getdbhdata($objid, 'flow');

	my $subflows = $self->getdbhobjectchilds($objid);

	foreach my $subflow (@$subflows) {
		if ($subflow->{type} eq "subflow") {
			$self->addobj($objid, $subflow);
			unless ($self->getdata($subflow->{id})) {
				$self->getsubflowhierarchy($subflow->{id});
			}
		}
		$self->myobject ($objid, $subflow->{object_id});
		$self->myobjectsfromchild ($objid, $subflow->{object_id});
	}

	$self->getdbhscripts($objid);

	my $attrdef;
	$attrdef->{object_id} = $objid;
	$attrdef->{objtype} = "flow";
	$attrdef->{attrobjtype} = $data->{flowtype} if ($data->{flowtype} ne "simple");
	$attrdef->{protocolid} = undef;
	$attrdef->{driverid} = undef;

	$self->getdbhattrs($objid, $attrdef);

	$self->getdbhvars($objid);

	$self->getdbhevents($objid);

	$self->getdbhotherobjects($objid);
}

sub gethierarchy {
	my $self = shift;
	my $objid = shift;

	my $dbh = $self->dbh();

	$self->{____DURATION____}{STARTED} = $dbh->now();
	$self->{____DURATION____}{ZSTARTED} = cafUtils->datetime1();

	my $query = $dbh->newquery({id => $objid, status => $self->objstatus()});
	$query->sobjectlist();
	my $objects = $dbh->hexecfetchall($query);
	

	foreach my $obj (@$objects) {

		$self->addobj($obj->{id}, $obj);

		if ($obj->{type} eq "flow") { $self->getflowhierarchy($obj->{id}); }
		elsif ($obj->{type} eq "subflow") { $self->getsubflowhierarchy($obj->{id}); }
		elsif ($obj->{type} eq "connector") { $self->getconnectorhierarchy($obj->{id}, 'NOCONTEXT'); }
		elsif ($obj->{type} =~ /^script|^perl|^sql/) { $self->getscripthierarchy($obj->{id}); }
		elsif ($obj->{type} eq "user") { $self->getuserhierarchy($obj->{id}); }
		elsif ($obj->{type} eq "server") { $self->getserverhierarchy($obj->{id}); }
		elsif ($obj->{type} eq "container") { $self->getcontainerhierarchy($obj->{id}); }
	}
	my @objids = $self->allobjects();

	if ($self->running() eq "QUEUE") {
		if ($self->lastupdate()) {
			$query = $dbh->newquery({object_id => \@objids, since => $self->lastupdate(), contextid => $self->contextid(),});
			$query->sobjectstoupdate();
			$self->objectstoupdate($dbh->hexecfetchall($query));
		}
		$self->lastupdate($dbh->now());
	}
	elsif ($self->running() eq "REPOSITORY") {
		$query = $dbh->newquery({object_id => \@objids, contextid => $self->contextid() });
		$query->sdeployedtoupdate();
#		print $query->query(), "<P>";
		$self->objectstoupdate($dbh->hexecfetchall($query));
#		print $dbh->errstr() . " ", join(' - ', @objids), " <BR>";
	}
	$self->{____DURATION____}{ENDED} = $dbh->now();
	$self->{____DURATION____}{ZENDED} = cafUtils->datetime1();

	$self;
}
# END Deploy objects subs

sub loadobjecthierarchy {
	my $self = shift;
	my $objtype = shift;
	my $idorname = shift;

	my $objid;
	my $dbh = $self->dbh();

	if ($idorname =~ /^[[:digit:]]+$/) { $objid = $idorname; }
	else {
		my $q = $dbh->newquery({ name => $idorname, type => $objtype });
#		print "OBJ HIER get object $objtype $idorname\n\n";
		$q->sobjectlist();
		my $r = $dbh->execfetchrow($q, "finish");
		return undef unless ($r and $r->[0]);
		$objid = $r->[0];
	}

	return $objid if ($self->getobject($objid) and $self->getdata($objid));

	if ($objtype eq "flow") { $self->getflowhierarchy($objid); }
	elsif ($objtype eq "subflow") { $self->getsubflowhierarchy($objid); }
	elsif ($objtype eq "connector") { $self->getconnectorhierarchy($objid, 'NOCONTEXT'); }
	elsif ($objtype =~ /^script|^perl|^sql/) { $self->getscripthierarchy($objid); }
	elsif ($objtype eq "user") { $self->getuserhierarchy($objid); }
	elsif ($objtype eq "server") { $self->getserverhierarchy($objid); }
	elsif ($objtype eq "container") { $self->getcontainerhierarchy($objid); }
	else { return undef; }

	return $objid if ($self->getobject($objid) and $self->getdata($objid));
	return undef;
}

sub getdbhotherobjects {
	my $self = shift;
	my $objid = shift;

	my $dbh = $self->dbh();
	my $query = $dbh->newquery({parent_id => $objid});
	$query->sotherobject();
	my $others = $dbh->hexecfetchall($query);

	$self->otherobjects($objid, $others) if ($others and ($#$others >= 0));
	foreach my $other (@$others) {
		if ($other->{type} eq "connector") {
			$self->getconnectorhierarchy($other->{child_id});
			$self->myobject ($objid, $other->{child_id});
			$self->myobjectsfromchild ($objid, $other->{child_id});
		}
		if ($other->{type} eq "container") {
			$self->getcontainerhierarchy($other->{child_id});
			$self->myobject ($objid, $other->{child_id});
			$self->myobjectsfromchild ($objid, $other->{child_id});
		}
	}
}
	
sub getdbhscripts {
	my $self = shift;
	my $objid = shift;
	my $otherscripts = shift;

	my $dbh = $self->dbh();
	my $query = $dbh->newquery({object_id => $objid});
	$query->sscriptobject();
	my $scripts = $dbh->hexecfetchall($query) || [];

	$self->objectscripts($objid, $scripts) if ($scripts and ($#$scripts >= 0));

	if ($otherscripts) { push @$scripts, @$otherscripts; }

	foreach my $script (@$scripts) {
		unless ($self->getobject($script->{script_id}) and $self->getdata($script->{script_id})) {
			$self->getscripthierarchy($script->{script_id});
		}
	}
}

sub getdbhmapandscripts {
	my $self = shift;
	my $objid = shift;

	my $dbh = $self->dbh();
	my $query = $dbh->newquery({ subflow_id => $objid });
	$query->smappinglist();
	my $mapping = $dbh->hexecfetchall($query) || [];

#	print "<BR>getdbhmapandscripts $objid / $#$mapping\n";
	$self->mappinglist($objid,$mapping) if ($mapping and ($#$mapping >= 0));

	my @scripts;
	foreach my $map (@$mapping) {
		if ($map->{script_id}) {
			my %script = %$map;
			$script{step} = "ftransform";
			$script{scripttype} = "ftransform";
			$script{object_id} = $script{subflow_id};
			push @scripts, \%script;
		}
	}

	$self->getdbhscripts($objid, \@scripts);
}

sub getdbhvars {
	my $self = shift;
	my $objid = shift;

	my $dbh = $self->dbh();
	my $query = $dbh->newquery( { object_id => $objid });
	$query->sgetmyenvvars();
	my $varlist = $dbh->hexecfetchall($query);
	$self->varslist ($objid, $varlist) if ($varlist);
	$varlist;
}

sub getdbhevents {
	my $self = shift;
	my $objid = shift;

	my $dbh = $self->dbh();
	my $query = $dbh->newquery({ object_id => $objid });
	$query->seventslist();
	my $eventslist = $dbh->hexecfetchall($query);
	$self->eventslist($objid, $eventslist) if ($eventslist);
	$eventslist;
}

sub getdbhattrs {
	my $self = shift;
	my $objid = shift;
	my $attrdef = shift;

	my $dbh = $self->dbh();
	my $attrlist = $dbh->getmyattributes($attrdef);
	$self->attributelist($objid, $attrlist) if ($attrlist);
}

sub getdbhdata {
	my $self = shift;
	my $objid = shift;
	my $tablename = shift;

	my $data = $self->getdata($objid);
	unless ($data) {
		my $dbh = $self->dbh();
		my $query = $dbh->newquery( { tablename => $tablename });
		$query->stablerow([ [ "object_id", "=", $objid ] ]);
		$data = $dbh->hexecfetchrownop($query, 1);
		$self->addobj($objid, undef, $data);
	}
	$data;
}

sub getdbhobjectchilds {
	my $self = shift;
	my $objid = shift;

		my $objects;
		my $dbh = $self->dbh();
		my $query = $dbh->newquery({parent_id => $objid, status => $self->objstatus()});
		$query->sobjectlist();
		$objects = $dbh->hexecfetchall($query);
}

sub getdbhobject {
	my $self = shift;
	my $objid = shift;

	my $object = $self->getobject($objid);
#	unless ($objid) { print "<BR> $objid OUIIIII C'EST UN SCANDAAAAAAALE ", join ("#", caller()) } #, " ", join(" - ", %$object); }
	unless ($object) {
		my $dbh = $self->dbh();
		my $query = $dbh->newquery({id => $objid, status => $self->objstatus()});
		$query->sobjectlist();
		$object = $dbh->hexecfetchrownop($query, 1);
		$self->addobj($objid, $object);
	}
	$object;
}

sub updateobj {
	my $self = shift;
	my $objid = shift;
	my $object = shift;

	if (ref($object)) { 
		$self->{OBJS}{OBJECT}{$objid}{__OBJECT__} = 1;
		foreach my $k (keys %$object) { $self->{OBJS}{OBJECT}{$objid}{$k} = $object->{$k}; }
	}

	if (@_) {
		my $data = shift;
		$self->{OBJS}{OBJECT}{$objid}{__OBJECT__} = 1;
		foreach my $k (keys %$data) { $self->{OBJS}{OBJECT}{$objid}{$k} = $data->{$k}; }
	}
}

sub addobj {
	my $self = shift;
	my $objid = shift;
	my $object = shift;

#	unless ($objid) { print "<BR> $objid SCANDAAAAAAALE ", join ("#", caller()), " ", join(" - ", %$object); }
		
	$self->{OBJS}{OBJECT}{$objid} = {} unless ($self->{OBJS}{OBJECT}{$objid});
	unless ($self->{OBJS}{OBJECT}{$objid}{__OBJECT__}) {
		$self->{OBJS}{OBJECT}{$objid}{__OBJECT__} = 1;
		foreach my $k (keys %$object) { $self->{OBJS}{OBJECT}{$objid}{$k} = $object->{$k}; }
		$self->objectbyidx($object->{type}, $objid, $object->{name});
	}

	unless ($self->{OBJS}{OBJECT}{$objid}{__DATA__}) {
		if (@_) {
			my $data = shift;
			$self->{OBJS}{OBJECT}{$objid}{__DATA__} = 1;
			foreach my $k (keys %$data) { $self->{OBJS}{OBJECT}{$objid}{$k} = $data->{$k}; }
		}
	}
}


sub getobject {
	my $self = shift;
	my $objid = shift;

	return $self->{OBJS}{OBJECT}{$objid} if ($self->{OBJS}{OBJECT}{$objid}{__OBJECT__});
	return undef;
}

sub getdata {
	my $self = shift;
	my $objid = shift;

	return $self->{OBJS}{OBJECT}{$objid} if ($self->{OBJS}{OBJECT}{$objid}{__DATA__});
	return undef;
}

sub allobjects {
	my $self = shift;

	my @ret;
	foreach my $id (keys %{$self->{OBJS}{OBJECT}}) { if ($id) { push @ret, $id; } }

	wantarray ? return @ret : return \@ret;
}

sub subflows {
	my $self = shift;
	my $objid = shift;
	my $dir = uc(shift); # INCOM / OUTGO

	if (@_ or (! ($self->{$dir}{$objid}))) {
		my $extra = shift;
		unless (ref($extra)) {
			$extra = [];
			foreach my $id ($self->allobjects()) {
				my $obj = $self->getobject($id);
				my $flow = $self->getdata($id);
				if (($obj->{type} eq "subflow") and ($obj->{parent_id} eq $objid) and ($flow->{flowdirection} eq lc($dir))) {
					push @$extra, $obj->{id};
				}
			}
		}
		$self->{$dir}{$objid} = $extra;
	}
	return $self->{$dir}{$objid};
}

sub fieldbyname {
	my $self = shift;
	my $objid = shift;
	my $name = shift;

	if (@_) {
		my $id = shift;
		$self->{FLDIDS}{$objid}{$name} = $id;
		$self->{FLDNAMES}{$objid}{$id} = $name;
	}
	$self->{FLDIDS}{$objid}{$name};
}


sub fieldbyid {
	my $self = shift;
	my $objid = shift;
	my $id = shift;

	if (@_) {
		my $name = shift;
		$self->{FLDNAMES}{$objid}{$id} = $name;
		$self->{FLDIDS}{$objid}{$name} = $id;
	}
	$self->{FLDNAMES}{$objid}{$id};
}

sub myfieldids {
	my $self = shift;
	my $objid = shift;

	my @f = keys %{$self->{FLDIDS}{$objid}};
	\@f;
}

sub myfields {
	my $self = shift;
	my $objid = shift;

	my @f = map { $self->getobject($_); } values %{$self->{FLDIDS}{$objid}};
	@f = sort { $a->{fieldorder} <=> $b->{fieldorder} } @f;
	\@f;
}

sub objectbyidx {
	my $self = shift;
	my $category = uc(shift);
	my $idx = shift;

	if (@_) {
		my $idx2 = shift;
		$self->{$category}{$idx} = $idx2;
		$self->{$category}{$idx2} = $idx;
	}
	$self->{$category}{$idx};
}

sub containerbyname {
	shift->objectbyidx('CONTAINERS', @_);
}

sub containerbyid {
	shift->objectbyidx('CONTAINERS', @_);
}

sub scriptbyname {
	shift->objectbyidx('SCRIPTS', @_);
}

sub scriptbyid {
	shift->objectbyidx('SCRIPTS', @_);
}

sub connectorbyname {
	shift->objectbyidx('CONNECTORS', @_);
}

sub connectorbyid {
	shift->objectbyidx('CONNECTORS', @_);
}

sub attributelist {
	my $self = shift;
	my $objid = shift;

	if (@_) { $self->{ATTRIBUTES}->{$objid} = shift; }
	$self->{ATTRIBUTES}->{$objid};
}

sub eventslist {
	my $self = shift;
	my $objid = shift;

	if (@_) { $self->{EVENTS}->{$objid} = shift; }
	$self->{EVENTS}->{$objid};
}

sub varslist {
	my $self = shift;
	my $objid = shift;

	if ($objid) {
		if (@_) { $self->{VARS}->{$objid} = shift; }
		$self->{VARS}->{$objid};
	}
	else {
		$self->{VARS};
	}
}

sub objectscripts {
	my $self = shift;
	my $objid = shift;
	my $scripts;

	if (@_) {
		$scripts = shift;
		$self->{SCRIPTLIST}{$objid} = $scripts;
		foreach my $script (@$scripts) {
			my $step = $script->{step};
#			my $step = $script->{scripttype};
			my $usedfor = $script->{usedfor};
			my $script_id = $script->{script_id};
			if ($script_id) {
				$self->{SCRIPTSTEP}{$step}{$objid} = $script_id if ($step);
				$self->{SCRIPTUSEDFOR}{$usedfor}{$objid} = $script_id if ($usedfor);
			}
		}
	}
	$self->{SCRIPTLIST}{$objid};
}

sub scriptbystep {
	my $self = shift;
	my $objid = shift;
	my $step = shift;

	$self->{SCRIPTSTEP}{$step}{$objid};
}

sub scriptbyusedfor {
	my $self = shift;
	my $objid = shift;
	my $usedfor = shift;

	$self->{SCRIPTUSEDFOR}{$usedfor}{$objid};
}

sub mappinglist {
	my $self = shift;
	my $objid = shift;

	if (@_) {
		my $mapping = shift;
#	print "<BR> mappinglist for $objid = ", join(" - ", @$mapping), "\n";
		$self->{MAPPING}->{$objid} = [];
		foreach my $map (@$mapping) {
#	print "<BR> HEHE FOR $objid : MAPPING $map->{outgofield_id} to ", join(" - ", %$map);
			$self->{MAPPFIELDS}{$objid}{$map->{outgofield_id}} = $map;
			push @{$self->{MAPPING}->{$objid}}, $map->{outgofield_id};
		}
	}
	$self->{MAPPING}->{$objid};

}

sub mappfieldlist {
	my $self = shift;
	my $flowid = shift;

#	if ($self->{MAPPING}{$flowid}) { print "<BR> mappfieldlist = ", join(" - ", @{$self->{MAPPING}{$flowid}}), "\n"; }
	my @mapplist = values %{$self->{MAPPFIELDS}{$flowid}};

	wantarray ? @mapplist : \@mapplist;
}

sub mappfield {
	my $self = shift;
	my $flowid = shift;
	my $fldid = shift;

	return $self->{MAPPFIELDS}{$flowid}{fldid} if ($fldid);
	my @mapplist = values %{$self->{MAPPFIELDS}{$flowid}};

	\@mapplist;
}

sub otherobjects {
	my $self = shift;
	my $objid = shift;

	if (@_) {
		$self->{OTHEROBJECTS}{$objid} = shift;
	}
	$self->{OTHEROBJECTS}{$objid};
}

sub floworder {
	my $self = shift;
	my $objid = shift;

	my $object = $self->getobject($objid);
	return $object->{floworder};
}

sub flowdir {
	my $self = shift;
	my $objid = shift;

	my $object = $self->getobject($objid);
	return $object->{flowdirection};
}

sub mysubflows {
	my $self = shift;
	my $flowid = shift;
	my $dir = uc(shift);
	
	if (@_) {
		my $subflowid = shift;
		$self->{$dir}{$flowid} = [] unless $self->{$dir}{$flowid};
		push @{$self->{$dir}{$flowid}}, $subflowid;
		my @flowArr;
		@flowArr = sort { $self->floworder($a) <=> $self->floworder($b) } @{$self->{$dir}{$flowid}};
		$self->{$dir}{$flowid} = \@flowArr;
	
	}
	$self->{$dir}{$flowid};
}

sub myincomflows {
	shift->subflows(shift, 'INCOM');
}

sub myoutgoflows {
	shift->subflows(shift, 'OUTGO');
}

	
sub myobjects {
	my $self = shift;
	my $objid = shift;

	my @objects = keys %{$self->{OBJOBJS}{$objid}};
	push @objects, $objid unless ($self->{OBJOBJS}{$objid}{$objid});

	wantarray ? @objects : \@objects;
}

sub myobject {
	my $self = shift;
	my $objid = shift;
	my $childid = shift;

	$self->{OBJOBJS}{$objid}{$childid} = 1 if ($objid and $childid);
}

sub myobjectsfromchild {
	my $self = shift;
	my $objid = shift;
	my $childid = shift;

	foreach my $childobjid ($self->myobjects($childid)) {
		$self->myobject ($objid, $childobjid);
	}
}

sub objectstoupdate {
	my $self = shift;

	if (@_) {
		$self->{TOUPDATE} = shift;
	}
	$self->{TOUPDATE};
}

# Output to context

#sub outputtocontext {

sub runsqlobjectupdate {
	my $self =shift;
	my $objid = shift;
	my $INaction = shift;
	my $objecttype = shift;

	my $action = "u";
	return 1 unless ($objid > 0);
	my $qobject = $action . "object";
	my $qdata = "$action$objecttype";

	my $dbh = $self->outdbh();

	#print "<BR><B>-Updateing object ", join(' - ', %{$self->getobject($objid)}), "</B>";
	my %object = %{$self->getobject($objid)};
	my $object = \%object;
	my $query = $dbh->newquery($object);
	$query->$qobject();
	my $ret = $dbh->executefinish($query);
	if (($ret <= 0) and ($action eq "u")) {
#		$query = $dbh->newquery($self->getobject($objid));
		my %object = %{$self->getobject($objid)};
		$object = \%object;
		$query = $query->clear($object);
		$qobject = "iobject";
		$query->$qobject();
		$ret = $dbh->executefinish($query);
	}

	if ($ret > 0) {
#		$query = $dbh->newquery($self->getdata($objid));
		my %object = %{$self->getobject($objid)};
		$object = \%object;
		$query = $query->clear($object);
		$qobject = "iobject";
		$query->$qdata();
		$ret = $dbh->executefinish($query);
		if (($ret <= 0) and ($action eq "u")) {
#			$query = $dbh->newquery($self->getdata($objid));
			my %object = %{$self->getobject($objid)};
			$object = \%object;
			$query = $query->clear($object);
			$qobject = "iobject";
			$qdata = "i$objecttype";
			$query->$qdata();
			$ret = $dbh->executefinish($query);
		}
	}
	return $ret;
}

sub runsqlattributesupdate {
	my $self = shift;
	my $objid = shift;

	my $dbh = $self->outdbh();
	my $ret = 1;

	my %attrvalues;
	my $attrlist = $self->attributelist($objid) || [];
	foreach my $attr (@$attrlist) {
		$attrvalues{$attr->{attrdefid}} = $attr->{attrvalue} if ($attr->{attrdefid} and $attr->{attrvalue});
	}

#	print "<br> ATTRIBUTES for $objid = ", join(" % ", %attrvalues) if (%attrvalues);
	if (%attrvalues) { $ret = $dbh->updateattributes($objid, \%attrvalues); }
	$ret;
}

sub runsqleventsupdate {
	my $self = shift;
	my $objid = shift;

	my $dbh = $self->outdbh();
	my $ret = 1;

	my $eventlist = $self->eventslist($objid);

	$ret = $dbh->updatemyevents($objid, $eventlist) if ($eventlist and ($#$eventlist >= 0));
	$ret;
}

sub runsqlcontextupdate {
	my $self = shift;
	my $context = shift;

	my $dbh = $self->outdbh();
	my $ret = 1;

	my $query = $dbh->newquery($context);
	$query->ucontext();
	$ret = $dbh->executefinish($query);
	if ($ret == 0) {
		$query->icontext();
		$ret = $dbh->executefinish($query);
	}
	$ret;
}

sub runsqlenvvarsupdate {
	my $self = shift;
	my $objid = shift;

	my $dbh = $self->outdbh();

	my $varvalues = $self->varslist($objid);
	if ($varvalues and $#$varvalues >= 0) {
		$dbh->updateenvvars($objid, $varvalues);
	}
	else { 1; }
}

sub runsqlmappingupdate {
	my $self = shift;
	my $objid = shift;

	my $dbh = $self->outdbh();

	my $mapping = $self->mappfieldlist($objid);
#	print "<BR> UPDATING MAPPING LIST for $objid / $#$mapping\n";
	return 1 unless ($mapping and ($#$mapping >= 0));

	$dbh->updatemymapping($objid, $mapping);
}

sub runsqlotherobjectsupdate {
	my $self = shift;
	my $objid = shift;

	my $dbh = $self->outdbh();

	my $otherobjects = $self->otherobjects($objid);

	return 1 unless ($otherobjects and ($#$otherobjects >= 0));

	$dbh->updatemyobjects($objid, $otherobjects);
}

sub runsqlobjectscriptupdate {
	my $self = shift;
	my $objid = shift;
	my $type = shift;

	my $dbh = $self->outdbh();
	my $ret = 1;

	my $scripts = [];
	
	if ($type eq 'scripts') { $scripts->[0] = $self->getobject($objid); }
	else { $scripts = $self->objectscripts($objid); }

	if ($scripts and ($#$scripts >= 0)) {
		if ($type ne 'scripts') { $ret = $dbh->updatemyscripts($objid, $scripts); }
		else { $ret = 1; }
		if ($ret >= 0) {
			foreach my $script (@$scripts) {
				my ($pscript, $scrid);
				if ($type ne 'scripts') {
					$scrid = $script->{script_id};
					$pscript = $self->getobject($scrid);
				}
				else {
					$scrid = $objid;
					$pscript = $script;
				}
				if ($pscript->{parsetext}) {
					my $query = $dbh->newquery({ object_id => $scrid, parsetext => $pscript->{parsetext}{parsetext} });
					$query->dpscripts();
					$ret = $dbh->executefinish($query);
					last unless ($ret);
					$query->ipscripts();
					$ret = $dbh->executefinish($query);
				}
			}
		}
	}
	$ret;
}

sub runsqlobjectobjects {
	my $self = shift;
	my $objid = shift;

	my $dbh = $self->outdbh();

	my $myobjects = $self->myobjects($objid);

	my $query = $dbh->newquery({ owner_id => $objid });
	$query->dwobjectobjects();
	my $ret = $dbh->executefinish($query);

	return undef unless ($ret);
	$ret = 1;
#	print "<br> runsqlobjectobjects $ret";
	foreach my $id (@$myobjects) {
#		print "<BR> RUNSQLOBJECTOBJECTS $objid, $id";
		$query = $dbh->newquery({ owner_id => $objid, owned_id => $id });
		$query->iwobjectobjects();
		$dbh->executefinish($query);
		last unless ($ret > 0);
	}
	$ret;
}

sub runsqldeployedobject {
	my $self = shift;
	my $objid = shift;
	my $INaction = shift;

	my $action = "u";
	my $dbh = $self->dbh();
	my $outdbh = $self->outdbh();

	my $query = $dbh->newquery({ contextid => $self->contextid(), object_id => $objid, committed => "yes" });
	$query->udeployedobjects();
	my $ret = $dbh->executefinish($query);


	if ($ret <= 0) {
		$query = $dbh->newquery({ contextid => $self->contextid(), object_id => $objid, committed => "yes" });
		$query->ideployedobjects();
		$ret = $dbh->executefinish($query);
	}

	my $query = $outdbh->newquery({ contextid => $self->contextid(), object_id => $objid, committed => "yes" });
	$query->udeployedobjects();
	my $ret = $outdbh->executefinish($query);


	if ($ret <= 0) {
		$query = $outdbh->newquery({ contextid => $self->contextid(), object_id => $objid, committed => "yes" });
		$query->ideployedobjects();
		$ret = $outdbh->executefinish($query);
	}
#	$ret = $outdbh->executefinish($query) if ($ret > 0);

	return ($ret);
	
}

sub gentablename {
	my $self = shift;
	my $objid = shift;
 
	my $contextid = substr($self->contextid(), 0, 4);
	
	sprintf("z%s_%08.8d", $contextid, $objid);
}

sub runsqlcreatecontexttables {
	my $self = shift;
	my $flowdir = shift;
	my $objid = shift;

	my $outdbh = $self->outdbh();
	my $tableprefix = $self->gentablename($objid);
	my $query = $outdbh->newquery({ tableprefix => $tableprefix });
	if ($flowdir eq "incom") { $query->sincomobjects (); }
	else { $query->soutgoobjects (); }
#			print "<BR><B> YOYOYOYOUY ", $query->query();
	my $objects = $outdbh->hexecfetchall($query);

	my @suffixes;
	my %objects;
	my $ret = 1;
	if ($flowdir eq "incom") { @suffixes = ("seq", "i", "id", "ipk", "idpk"); }
	else { @suffixes = ("seq", "o", "od", "or", "opk", "odpk", "orpk"); }
	foreach my $object (@$objects) {
		 $objects{$object->{object_name}} = $object;
	}

	foreach my $suff (@suffixes) {
		my $objname = $tableprefix . "_" . $suff;
		my $queryname = "cobject_$suff";
		unless ($objects{$objname}) {
			$query = $outdbh->newquery({ tableprefix => $tableprefix });
			$query->$queryname();
			$ret = $outdbh->executefinish($query);
			$ret = 1 unless $outdbh->errstr();
#			print "<BR><B> $ret = YOYOYOYOUY ==>", $outdbh->err(), " <==<BR> ", $query->query();
			last unless ($ret > 0);
		}
	}
	$ret;
}

sub runsqlgeneratetables {
	my $self = shift;

	my $ret = 1;
	foreach my $objid (@{$self->allobjects}) {
		my $obj = $self->getobject($objid);
		if ($obj->{type} eq 'flow') {
			$ret = $self->runsqlcreatecontexttables("incom", $objid);
			return $ret unless ($ret > 0);
		}
		elsif (($obj->{type} eq 'subflow') and ($obj->{flowdirection} eq 'outgo')) {
			$ret = $self->runsqlcreatecontexttables("outgo", $objid);
			return $ret unless ($ret > 0);
		}
	}
	$ret;
}


sub outputupdatable {
	my $self = shift;

	my $context = $self->context();
	my $connector = $self->getdata($context->{connector_id});
	my $connectorobj = $self->getobject($context->{connector_id});

	my $user = $self->getdata($connector->{userid});
	my $server = $self->getdata($connectorobj->{parent_id});

	my $dbiattrs = { PrintError => 0, RaiseError => 1, AutoCommit => 0 };
	my $outdbh;
	cafDbg->pushstackdump(1);
	eval { $outdbh = refDBI->Connect({ connector => $connector, server => $server, user => $user, _ATTRS => $dbiattrs }); };

	unless ($outdbh and ref($outdbh)) {
		cafDbg->pusherror("10000; Unable to connect ". refDBI->errstr);
		return undef;
	}
	cafDbg->popstackdump();

	$self->outdbh($outdbh);

	my $updatable = $self->objectstoupdate();
#	print "objectstoupdate : ", join(' - ', @$updatable), "\n";
	my $action;

	my $ret;

	$ret = $self->runsqlcontextupdate($context);
	 if ($ret > 0) {
		foreach my $object (@$updatable) {

			next unless ($object->{object_id} > 0);
			my $type = $object->{type};
			$type = "scripts" if ($type eq "sql" or $type eq "perl");

			if ($object->{object_exist}) { $action = "u"; } else { $action = "i"; }
#			print "<BR><B>Updating Object $object->{object_id} $type $action => ", join(' - ', %$object), "</B>";

			$ret = $self->runsqlobjectupdate($object->{object_id}, $action, $type);
#			print "<BR><B>Finished Object $object->{object_id} $type $action</B>";

			$ret = $self->runsqlattributesupdate($object->{object_id}) if ($ret > 0);

			$ret = $self->runsqleventsupdate($object->{object_id}) if ($ret > 0);

			$ret = $self->runsqlenvvarsupdate($object->{object_id}) if ($ret > 0);

			$ret = $self->runsqlmappingupdate($object->{object_id}) if ($ret > 0);

			$ret = $self->runsqlotherobjectsupdate($object->{object_id}) if ($ret > 0);

			$ret = $self->runsqlobjectscriptupdate($object->{object_id}, $type) if ($ret > 0);

			$ret = $self->runsqlobjectobjects($object->{object_id}, $action) if ($ret > 0);

			$ret = $self->runsqldeployedobject($object->{object_id}, $action) if ($ret > 0);

			last unless ($ret > 0);
		}
	}
	$ret;
}

#DEBUG USAGE
sub dumpall {
        my $self = shift;
 
        my $__SPECIAL = $self->{__SPECIAL};
	delete $self->{__SPECIAL};

	require Data::Dumper;
#		$self->{TOUPDATE};
 
	my $dump = Data::Dumper->new([$self], ["Objects Hierarchy"]);
	$dump->Indent(1);
	my $ptext = $dump->Dump;
	$self->{__SPECIAL} = $__SPECIAL;
	$ptext;
}

sub dump {
        my $self = shift;
 
        my $__SPECIAL = $self->{__SPECIAL};
	delete $self->{__SPECIAL};

	require Data::Dumper;
#		$self->{TOUPDATE};
 
	my $dump = Data::Dumper->new([$self->{TOUPDATE}], ["updatedobjects"]);
	$dump->Indent(1);
	my $ptext = $dump->Dump;
	$self->{__SPECIAL} = $__SPECIAL;
	$ptext;
}

1;
