-- $Id: schema.sql,v 1.28 2004/02/04 17:44:09 mig Exp $
-- ####################################
-- Comas - Conference Management System
-- ####################################
-- Copyright 2003 CONSOL
-- Congreso Nacional de Software Libre (http://www.consol.org.mx/)
--   Gunnar Wolf <gwolf@gwolf.cx>
--   Manuel Rabade <mig@mig-29.net>

-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.

-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.

-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-- ####################################

-- ######################################
-- File: schema.sql
-- Database tables and structure


--
-- For further documentation on the tables and attributes' meaning, please
-- check the data dictionary at:
-- http://wiki.consol.org.mx/comas_wiki/DataDictionary
-- http://wiki.consol.org.mx/comas_wiki/RelationDescription
--

--
-- Delete the tables and their dependendent objects in order to avoid 
-- any conflicts

DROP TABLE config CASCADE;
DROP TABLE expense CASCADE;
DROP TABLE exp_status CASCADE;
DROP TABLE payment_type CASCADE;
DROP TABLE payment CASCADE;
DROP TABLE task_admin CASCADE;
DROP TABLE task CASCADE;
DROP TABLE admin CASCADE;
DROP TABLE price CASCADE;
DROP TABLE authors CASCADE;
DROP TABLE room CASCADE;
DROP TABLE timeslot_prop_type CASCADE;
DROP TABLE timeslot CASCADE;
DROP TABLE prop_type CASCADE;
DROP TABLE prop_status CASCADE;
DROP TABLE track CASCADE;
DROP TABLE proposal CASCADE;
DROP TABLE pers_status CASCADE;
DROP TABLE country CASCADE;
DROP TABLE state CASCADE;
DROP TABLE person_type CASCADE;
DROP TABLE studies CASCADE;
DROP TABLE person CASCADE;
DROP TABLE room_prop_type_max_people;
DROP TABLE sessions CASCADE;

--
-- Now start creating the tables...
--
CREATE TABLE studies (
	id serial PRIMARY KEY,
	descr text NOT NULL
);
REVOKE ALL ON studies FROM public;
GRANT SELECT ON studies TO comas;
GRANT ALL ON studies TO comas_adm;
GRANT UPDATE, SELECT ON studies_id_seq TO comas_adm;

CREATE TABLE person_type (
	id serial PRIMARY KEY,
	descr text NOT NULL
);
REVOKE ALL ON person_type FROM public;
GRANT SELECT ON person_type TO comas;
GRANT ALL ON person_type TO comas_adm;
GRANT UPDATE, SELECT ON person_type_id_seq TO comas_adm;


CREATE TABLE state (
	id serial PRIMARY KEY,
	descr text NOT NULL
);
REVOKE ALL ON state FROM public;
GRANT SELECT ON state TO comas;
GRANT ALL ON state TO comas_adm;
GRANT UPDATE, SELECT ON state_id_seq TO comas_adm;


CREATE TABLE country (
	id serial PRIMARY KEY,
	descr text NOT NULL
);
REVOKE ALL ON country FROM public;
GRANT SELECT ON country TO comas;
GRANT ALL ON country TO comas_adm;
GRANT UPDATE, SELECT ON country_id_seq TO comas_adm;


CREATE TABLE pers_status (
	id serial PRIMARY KEY,
	descr text NOT NULL
);
REVOKE ALL ON pers_status FROM public;
GRANT SELECT ON pers_status TO comas;
GRANT ALL ON pers_status TO comas_adm;
GRANT UPDATE, SELECT ON pers_status_id_seq TO comas_adm;


CREATE TABLE track (
	id serial PRIMARY KEY,
	descr text NOT NULL
);
REVOKE ALL ON track FROM public;
GRANT SELECT ON track TO comas;
GRANT ALL ON track TO comas_adm;
GRANT UPDATE, SELECT ON track_id_seq TO comas_adm;


CREATE TABLE prop_status (
	id serial PRIMARY KEY,
	descr text NOT NULL
);
REVOKE ALL ON prop_status FROM public;
GRANT SELECT ON prop_status TO comas;
GRANT ALL ON prop_status TO comas_adm;
GRANT UPDATE, SELECT ON prop_status_id_seq TO comas_adm;


CREATE TABLE prop_type (
	id serial PRIMARY KEY,
	descr text NOT NULL,
	duration interval NOT NULL
);
REVOKE ALL ON prop_type FROM public;
GRANT SELECT ON prop_type TO comas;
GRANT ALL ON prop_type TO comas_adm;
GRANT UPDATE, SELECT ON prop_type_id_seq TO comas_adm;


CREATE TABLE room (
	id serial PRIMARY KEY,
	descr text NOT NULL
);
REVOKE ALL ON room FROM public;
GRANT SELECT ON room TO comas;
GRANT ALL ON room TO comas_adm;
GRANT UPDATE, SELECT ON room_id_seq TO comas_adm;

CREATE TABLE price (
	person_type_id integer REFERENCES person_type(id) NOT NULL,
	prop_type_id integer REFERENCES prop_type(id) NOT NULL,
	until date NOT NULL,
	amount integer NOT NULL,
	PRIMARY KEY(person_type_id, prop_type_id, until)
);
REVOKE ALL ON price FROM public;
GRANT SELECT ON price TO comas;
GRANT ALL ON price TO comas_adm;

CREATE TABLE admin (
	id serial PRIMARY KEY,
	name text NOT NULL,
	login text NOT NULL,
	passwd text
);
REVOKE ALL ON admin FROM public;
GRANT SELECT ON admin TO comas;
GRANT ALL ON admin TO comas_adm;
GRANT SELECT, UPDATE ON admin_id_seq TO comas_adm;

CREATE TABLE task (
	id serial PRIMARY KEY,
	descr text NOT NULL
);
REVOKE ALL ON task FROM public;
GRANT SELECT ON task TO comas;
GRANT ALL ON task TO comas_adm;
GRANT SELECT, UPDATE ON task_id_seq TO comas_adm;

CREATE TABLE task_admin (
	admin_id integer REFERENCES admin(id) NOT NULL,
	task_id integer REFERENCES task(id) NOT NULL
);
REVOKE ALL ON task_admin FROM public;
GRANT SELECT ON task_admin TO comas;
GRANT ALL ON task_admin TO comas_adm;

CREATE TABLE payment_type (
	id serial PRIMARY KEY,
	descr text NOT NULL
);
REVOKE ALL ON payment_type FROM public;
GRANT SELECT ON payment_type TO comas;
GRANT ALL ON payment_type TO comas_adm;
GRANT SELECT, UPDATE ON payment_type_id_seq TO comas_adm;

CREATE TABLE exp_status (
	id serial PRIMARY KEY,
	descr text NOT NULL
);
REVOKE ALL ON exp_status FROM public;
GRANT SELECT ON exp_status TO comas;
GRANT ALL ON exp_status TO comas_adm;
GRANT SELECT, UPDATE ON exp_status_id_seq TO comas_adm;

CREATE TABLE config (
	name text PRIMARY KEY,
	descr text NOT NULL,
	value text NOT NULL
);
REVOKE ALL ON config FROM public;
GRANT SELECT, UPDATE ON config TO comas;
GRANT ALL ON config TO comas_adm;

CREATE TABLE person (
	id serial PRIMARY KEY,
	firstname text NOT NULL,
	famname text NOT NULL,
	login text NOT NULL,
	nickname text,
	passwd text,
	org text,
	dept text,
	title text,
	studies_id integer REFERENCES studies(id),
	email text,
	person_type_id integer REFERENCES person_type(id),
	postal_addr text,
	zip text,
	city text,
	state_id integer REFERENCES state(id),
	country_id integer REFERENCES country(id),
	phone text,
	fax text,
	birth date,
	resume text,
	reg_time timestamp NOT NULL DEFAULT now(),
	pers_status_id integer REFERENCES pers_status(id)
);
REVOKE ALL ON person FROM public;
GRANT SELECT, INSERT, UPDATE ON person TO comas;
GRANT ALL ON person TO comas_adm;
GRANT UPDATE, SELECT ON person_id_seq TO comas;

CREATE TABLE timeslot (
	id serial PRIMARY KEY,
	start_hr time NOT NULL,
	day date NOT NULL,
	room_id integer REFERENCES room(id) NOT NULL
);
REVOKE ALL ON timeslot FROM public;
GRANT SELECT ON timeslot TO comas;
GRANT ALL ON timeslot TO comas_adm;
GRANT SELECT, UPDATE ON timeslot_id_seq TO comas_adm;

CREATE TABLE proposal (
	id serial PRIMARY KEY,
	title text NOT NULL,
	abstract text NOT NULL,
	track_id integer REFERENCES track(id),
	prop_status_id integer REFERENCES prop_status(id) NOT NULL DEFAULT 1,
	prop_type_id integer REFERENCES prop_type(id) NOT NULL,
	comments text,
	filename text,
	timeslot_id integer REFERENCES timeslot(id)
);
REVOKE ALL ON proposal FROM public;
GRANT SELECT, INSERT, UPDATE, DELETE ON proposal TO comas;
GRANT ALL ON proposal TO comas_adm;
GRANT SELECT, UPDATE ON proposal_id_seq TO comas;

CREATE TABLE timeslot_prop_type (
	type integer REFERENCES prop_type(id) NOT NULL,
	timeslot integer REFERENCES timeslot(id) NOT NULL
);
REVOKE ALL ON timeslot_prop_type FROM public;
GRANT SELECT ON timeslot_prop_type TO comas;
GRANT ALL ON timeslot_prop_type TO comas_adm;

CREATE TABLE authors (
	person_id integer REFERENCES person(id) NOT NULL,
	proposal_id integer REFERENCES proposal(id) NOT NULL,
	mention_order integer NOT NULL,
	PRIMARY KEY(person_id, proposal_id)
);
REVOKE ALL ON authors FROM public;
GRANT ALL ON authors TO comas;
GRANT ALL ON authors TO comas_adm;

CREATE TABLE payment (
	id serial PRIMARY KEY,
	person_id integer REFERENCES person(id) NOT NULL,
	proposal_id integer REFERENCES proposal(id) NOT NULL,
	amount integer NOT NULL,
	admin_id integer REFERENCES admin(id),
	payment_type_id integer REFERENCES payment_type(id) NOT NULL,
	timestamp timestamp NOT NULL DEFAULT now()
);
REVOKE ALL ON payment FROM public;
GRANT SELECT, INSERT, UPDATE ON payment TO comas;
GRANT ALL ON payment TO comas_adm;
GRANT UPDATE ON payment_id_seq TO comas;
GRANT SELECT, UPDATE ON payment_id_seq TO comas_adm;

CREATE TABLE expense (
	id serial PRIMARY KEY,
	person_id integer REFERENCES person(id),
	exp_status_id integer REFERENCES person(id) NOT NULL,
	descr text NOT NULL,
	start timestamp,
	duration interval,
	amount integer NOT NULL
);
REVOKE ALL ON expense FROM public;
GRANT SELECT, INSERT, UPDATE ON expense TO comas;
GRANT ALL ON expense TO comas_adm;
GRANT UPDATE ON expense_id_seq TO comas;
GRANT SELECT, UPDATE ON expense_id_seq TO comas_adm;

CREATE TABLE sessions (
        id char(32) not null primary key,
        a_session text
);
REVOKE ALL ON sessions FROM public;
GRANT ALL ON sessions TO comas;
GRANT ALL ON sessions TO comas_adm;

CREATE TABLE room_prop_type_max_people (
	room_id integer REFERENCES room(id) NOT NULL,
	prop_type_id integer REFERENCES prop_type(id) NOT NULL,
	max_people integer NOT NULL DEFAULT 0
);
REVOKE ALL ON room_prop_type_max_people FROM public;
GRANT SELECT ON room_prop_type_max_people TO comas;
GRANT ALL ON room_prop_type_max_people TO comas_adm;

--
-- And now that everything is set, create the needed views
--

-- auth_prop: The list of all registered authors and their proposals
CREATE OR REPLACE VIEW auth_prop AS
SELECT pe.id AS person_id, pe.firstname, pe.famname, 
pr.id AS proposal_id, pr.title, pr.abstract 
FROM person pe, proposal pr, authors a
WHERE pe.id=a.person_id AND pr.id=a.proposal_id 
ORDER BY pr.id, a.mention_order;
GRANT SELECT ON auth_prop TO comas;
GRANT SELECT ON auth_prop TO comas_adm;

-- scheduled_proposals: A chronological overview of the present scheduled 
-- timeslots/proposals
CREATE OR REPLACE VIEW scheduled_proposals AS
SELECT pr.id, pr.title, r.id AS room_id, r.descr AS room_descr, t.id AS
timeslot_id, t.day, t.start_hr, pt.duration 
FROM proposal pr, room r, timeslot t, timeslot_prop_type tpt, prop_type pt
WHERE pt.id = tpt.type AND t.id = tpt.timeslot AND r.id = t.room_id AND 
pt.id = pr.prop_type_id AND pr.timeslot_id = t.id
ORDER BY t.day, t.start_hr, r.id;
GRANT SELECT ON scheduled_proposals TO comas;
GRANT SELECT ON scheduled_proposals TO comas_adm;

-- unscheduled_proposals: The list of proposals which have not yet been
-- scheduled
CREATE OR REPLACE VIEW unscheduled_proposals AS
SELECT id, title, prop_status_id, prop_type_id, comments FROM proposal
WHERE timeslot_id IS NULL ORDER BY id;

-- unscheduled_timeslots: Helps us locate all the holes in our schedule,
-- returns the timeslots which do not have something scheduled
CREATE OR REPLACE VIEW unscheduled_timeslots AS
SELECT r.descr AS room, t.id AS timeslot, t.day, t.start_hr, pt.duration,
pt.id AS prop_type
FROM room r, timeslot t, prop_type pt, timeslot_prop_type tpt
WHERE r.id = t.room_id AND t.id = tpt.timeslot AND pt.id = tpt.type AND 
t.id NOT IN (SELECT t2.id 
    FROM proposal pr, prop_type pt, timeslot t1, timeslot t2
    WHERE pt.id = pr.prop_type_id AND t1.id = pr.timeslot_id AND
    t2.start_hr >= t1.start_hr AND t2.start_hr < t1.start_hr + pt.duration
   )
ORDER BY t.day, t.start_hr, r.id;
GRANT SELECT ON unscheduled_timeslots TO comas;
GRANT SELECT ON unscheduled_timeslots TO comas_adm;

-- timeslots_not_available: Shows which timeslots (which should be free,
-- as enforced by the 'no_simultaneous_proposal' trigger) are not currently
-- available for hosting proposals, as they would be in the same room and time
-- as another, currently scheduled proposal. Of course, in the set of results
-- we include the prop_type_id, as a timeslot may be available for hosting a
-- specific prop_type but not another (different durations).
--
-- Timeslots which do not have yet been assigned a prop_type are not considered
-- to be complete, and thus will not be returned here (i.e. - They are not
-- available, but no change to the schedule can make them available)
CREATE OR REPLACE VIEW timeslots_not_available AS
SELECT DISTINCT t1.id AS timeslot_id, t1.day+start_hr AS start_hr, 
t1.day+t1.start_hr+p.duration AS end_hr, p.id AS prop_type_id 
FROM timeslot t1, prop_type p, timeslot_prop_type tp, (SELECT id AS t2_id,
    day+start_hr AS t2_start_hr, day+start_hr+duration AS t2_end_hr, room_id
    AS t2_room_id FROM scheduled_proposals) t2
WHERE tp.timeslot=t1.id AND tp.type=p.id AND t1.room_id=t2_room_id AND NOT 
(t2_end_hr<=t1.day+t1.start_hr OR t2_start_hr>=t1.day+t1.start_hr+p.duration)
ORDER BY timeslot_id, end_hr;
GRANT SELECT ON timeslots_not_available to comas;
GRANT SELECT ON timeslots_not_available to comas_adm;

-- timeslots_available: The opposite of timeslots_not_available, returns the
-- list of timeslots that can currently accept a proposal and the prop_type(s)
-- they can currently accept. Of course, timeslots that don't have a
-- timeslot_prop_type entry cannot be considered as available.
CREATE OR REPLACE VIEW timeslots_available AS
SELECT DISTINCT t.id AS timeslot_id, t.day+t.start_hr as start_hr,
t.day+t.start_hr+p.duration AS end_hr, p.id AS prop_type_id
FROM timeslot t, timeslot_prop_type pt, prop_type p
WHERE t.id=pt.timeslot AND p.id=pt.type  AND t.id||'--'||p.id
NOT IN (SELECT tna.timeslot_id||'--'||tna.prop_type_id 
        FROM timeslots_not_available tna) 
ORDER BY t.id, p.id;
GRANT SELECT ON timeslots_available to comas;
GRANT SELECT ON timeslots_available to comas_adm;

-- $Log: schema.sql,v $
-- Revision 1.28  2004/02/04 17:44:09  mig
-- - Aado la tabla room_prop_type_max_people
--
-- Revision 1.27  2003/12/20 04:19:54  mig
-- - Agrego tags Id y Log que expanda el CVS
--
