/**
 * ==========
 * pgExplorer
 * ==========
 * This source file is subject to the license specified in the
 * LICENSE file that is included in this package.
 *
 * @copyright 2000, 2001 Keith Wong
 * @author Keith Wong
 * @email keith@e-magine.com.au
 */
 
#include "dbfunctionmanager.h"
#include "../types/dbtype.h"
#include "../types/dbtypemanager.h"
#include "../../utils/debugger.h"
#include "../../utils/stringutils.h"
#include "../../utils/converter.h"

	/**
 	 * Constructor
	 */
  DBFunctionManager::DBFunctionManager()
  	: DBBaseManager()
  {
  } // end constructor
	
	/**
 	 * Constructor
 	 * It is assumed that the database connection object will remain alive during
 	 * the life of this object. Be very careful to ensure that no methods are called
 	 * on this object if the connection object no longer exists. If the connection
 	 * object has already been destroyed then unpredictable results will be returned. 	
   */		
  DBFunctionManager::DBFunctionManager(DBConnection *poDBConn)
  	: DBBaseManager(poDBConn)
  {
		m_oDBTypeMgr.setDBConnection(poDBConn);
  } // end constructor
	
	/**
 	 * Destructor
   */		
	DBFunctionManager::~DBFunctionManager()
	{
		// do nothing
	} // end destructor

	/**
	 * Used to set the database connection object
	 * @param poDBConn
	 */
	void DBFunctionManager::setDBConnection(DBConnection * poDBConn)
	{
		DBBaseManager::setDBConnection(poDBConn);
		m_oDBTypeMgr.setDBConnection(poDBConn);
	} // end setDBConnection
	
	/**
	 * Used to retrieve the list of functions for this connection.
	 * @return	a DBFunctionSet object that contains the function details
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if not connected
	 */
	void DBFunctionManager::retrieveListOfFunctions(DBFunctionSet & roDBFunctionSet) throw (SQLException, DBConnectionException)
	{
		string strMethodName = "DBFunctionManager::retrieveListOfFunctions";
		
		Debugger::entered(strMethodName);
		
		// sql statement used to retrieve all the function details
		string strSQL = "SELECT p.proname, u.usename, "
										"p.prorettype, p.proargtypes, l.lanname, "
										"d.description as comment "
										"FROM pg_proc p, pg_user u, pg_description d, pg_language l "
										"WHERE p.proowner = u.usesysid "
										"AND p.oid = d.objoid "
										"AND l.oid = p.prolang "
										"AND u.usesuper = 'f' "										
										"UNION ALL "
										"SELECT p.proname, u.usename, "
										"p.prorettype, p.proargtypes, l.lanname, "
										"NULL as comment "
										"FROM pg_proc p, pg_user u, pg_language l "
										"WHERE p.proowner = u.usesysid "
										"AND l.oid = p.prolang "										
										"AND p.oid NOT IN (select objoid from pg_description) "
										"AND u.usesuper = 'f' "	
										"ORDER BY proname";
										
		// execute query										
		m_poDBConn->executeQuery(strSQL, roDBFunctionSet.m_oFunctionList);										

 		// lets find out the names of the types
 		// need to use the type manager to find out the type names
		while (roDBFunctionSet.next())
		{ 		
  		vector<DBType> voDBTypes;
  		bool bReturnTypeExist = false;
  		bool bArgumentsExist = false;
  		Debugger::logTrace(strMethodName, "Return type = " + roDBFunctionSet.m_oFunctionList.getFieldValue("prorettype"));
  		if (StringUtils::trim(roDBFunctionSet.m_oFunctionList.getFieldValue("prorettype")).size() != 0)
  		{
  			DBType oReturnType;
  			bReturnTypeExist = true;
  			oReturnType.setTypeOid(Converter::stringToLong(StringUtils::trim(roDBFunctionSet.m_oFunctionList.getFieldValue("prorettype"))));
  			voDBTypes.push_back(oReturnType);	
  		} // end if return type exists			
  		if (StringUtils::trim(roDBFunctionSet.m_oFunctionList.getFieldValue("proargtypes")).size() != 0)			
  		{
  			bArgumentsExist = true;
  			// lets convert the type oids list into individual parts
  			vector<string> voTypeOids;
  			StringUtils::tokenize(StringUtils::trim(roDBFunctionSet.m_oFunctionList.getFieldValue("proargtypes")), " ", voTypeOids);
  			for (int nIdx = 0; nIdx < voTypeOids.size(); nIdx++)
  			{
  				Debugger::logTrace(strMethodName, "Adding type oid: " + voTypeOids[nIdx]);
  				DBType oType;
  				oType.setTypeOid(Converter::stringToLong(voTypeOids[nIdx]));
  				voDBTypes.push_back(oType);						
  			} // end for more return types
  		} // end if arguments exist
 			
  		m_oDBTypeMgr.retrieveListOfTypes(voDBTypes);
 			
  		int nTypeIdx = 0;
  		// lets assign the converted types
  		if (bReturnTypeExist == true)
  		{
  			roDBFunctionSet.m_vstrReturnTypes.push_back(voDBTypes[nTypeIdx++].getTypeName());
  		} // end if return type exist
  		else  		
  		{
  			roDBFunctionSet.m_vstrReturnTypes.push_back("");  		
  		} // end else return type does not exist
  		
  		// lets do the arg types
 			vector<string> vstrArgTypes;
 			for (; nTypeIdx < voDBTypes.size(); nTypeIdx++)
 			{
 				vstrArgTypes.push_back(voDBTypes[nTypeIdx].getTypeName());
 			} // end for more types
 			roDBFunctionSet.m_vvstrArgumentTypes.push_back(vstrArgTypes);
  		
		} // end while more records
		
		// lets reset this
		roDBFunctionSet.reset();
								
		Debugger::exited(strMethodName);
		
	} // end retrieveListOfFunctions
	
	/**
	 * Used to modify the function comment.
	 * @param			rstrFunctionName		the function name
	 * @param			rvstrArgumentTypes	the list of argument types
	 * @param			rstrComment					the comment to set for the function
	 * @exception SQLException if it cannot retrieve results
	 * @exception DBConnectionException if cannot connect
	 */
	void DBFunctionManager::modifyFunctionComment(const string & rstrFunctionName,
				const vector<string> & rvstrArgumentTypes, const string & rstrComment)
				throw (SQLException, DBConnectionException)
	{
		string strSQLComment;
		// build up argument list
		string strArgList = "(";
		for (int nIdx = 0; nIdx < rvstrArgumentTypes.size(); nIdx++)
		{
			if (nIdx == 0)
			{
				strArgList += rvstrArgumentTypes[nIdx];
			} // end if first arg
			else
			{
				strArgList += ", " + rvstrArgumentTypes[nIdx];
			} // end else not first arg
		} // end for more arg types
		strArgList += ")";
		
 		if (rstrComment == "")
 		{
 			strSQLComment = "COMMENT ON FUNCTION " + rstrFunctionName + strArgList + " IS NULL";
 		} // end if need to drop comment
 		else
 		{
 			strSQLComment = "COMMENT ON FUNCTION " + rstrFunctionName + strArgList +
 												" IS '" + StringUtils::databasestr(rstrComment) + "'";			
 		} // end else need to set new comment
		// set comment
		m_poDBConn->execute(strSQLComment);												
	
	} // end modifyFunctionComment
