/*
	enabledb.sql

	Creates the db schema for a pgjobs-enabled database.
	Invoke through: pgjobs enabledb ...

	Project: pgjobs
	Author:  Zlatko Michailov
	Created: 31-Oct-2003
	Updated:  7-Nov-2003
	Updated: 19-Dec-2003
	Updated: 29-Dec-2003
	Updated: 17-Jan-2004
	Updated: 18-Jan-2004
	Updated: 25-Jan-2004
	Updated: 13-May-2004
	Updated: 21-May-2004
	Updated: 24-May-2004
	Updated:  1-Jun-2004

	This file is provided as is, with no warranty. Use at your own risk.

	Copyright (c) 2003-2004, Zlatko Michailov

*/



--------------------------------------------------------------------------------



------------------------------------------------
-- Clean up
------------------------------------------------

drop view pgj_jobs;
drop view pgj_schedules;

drop table pgj_job_schedule_;
drop table pgj_job_;
drop table pgj_schedule_;

drop sequence pgj_schedule__schid_seq;
drop sequence pgj_job__jobid_seq;



------------------------------------------------
-- create schema in a transaction
------------------------------------------------

begin transaction;



------------------------------------------------
-- Schedules
------------------------------------------------

------------------------------------------------
-- Internal table for storing schedules.
-- Should not be used directly.
-- Consider view "pgj_schedules".
------------------------------------------------
create table pgj_schedule_
(
	schid			serial			primary key,
	schname			text			not null unique,
	schtype			char			not null,
	schyear			smallint		null,
	schmonth		smallint		null,
	schday			smallint		null,
	schhour			smallint		null,
	schminute		smallint		null
);



------------------------------------------------
-- Produces a friendly text representation of a
-- schedule definition from the internal values.
-- Invoked from view "pgj_schedules".
------------------------------------------------
create or replace function pgj_make_schedule
(
	char,		-- $1 type
	smallint,	-- $2 year
	smallint,	-- $3 month
	smallint,	-- $4 day
	smallint,	-- $5 hour
	smallint	-- $6 minute
)
returns text
as '
	declare
		schType_	alias for $1;
		schYear		alias for $2;
		schMonth	alias for $3;
		schDay		alias for $4;
		schHour		alias for $5;
		schMinute	alias for $6;

		schedule	text;
		ok			boolean;
		schType		char;

	begin
		schType		:= upper( schType_ );
		schedule	:= schType || '' '';
		ok			:= true;


		if schType = ''D'' then
			-- year
			if schYear is not null then
				schedule	:= schedule || lpad( schYear, 4, ''0'' ) || ''.'';
				ok			:= schMonth is not null;
			end if;

			-- month
			if ok and schMonth is not null then
				schedule	:= schedule || lpad( schMonth, 2, ''0'' ) || ''.'';
				ok			:= schDay is not null;
			end if;
		end if;


		if schType in ( ''D'', ''W'', ''I'' ) then
			-- day
			if ok and schDay is not null then
				schedule	:= schedule || lpad( schDay, 2, ''0'' ) || '' '';
				ok			:= schHour is not null;
			end if;

			-- hour
			if ok and schHour is not null then
				schedule	:= schedule || lpad( schHour, 2, ''0'' ) || '':'';
			end if;

			-- minute
			ok := ok and schMinute is not null;
			if ok then
				schedule	:= schedule || lpad( schMinute, 2, ''0'' );
			end if;
		else
			ok := false;
		end if;


		if not ok then
			schedule := null;
		end if;

		return schedule;
	end;
' language 'plpgsql';



------------------------------------------------
-- Creates a new- or updates an existing schedule.
-- Should not be called directly.
-- Consider functions "pgj_create_schedule" and
-- "pgj_update_schedule".
------------------------------------------------
create or replace function pgj_create_update_schedule
(
	text,	-- $1 name
	text,	-- $2 definition
	boolean	-- $3 isNew
)
returns boolean
as '
	declare
		schName_		alias for $1;
		schFriendly_	alias for $2;
		schIsNew		alias for $3;

		schName			text;
		schFriendly		text;

		schType			char;
		schYear			smallint;
		schMonth		smallint;
		schDay			smallint;
		schHour			smallint;
		schMinute		smallint;

		elmStart		smallint;
		elmLength		smallint;
		values			text;
		ok				boolean;
		temp			text;
	begin
		ok := schName_ is not null and
		      schFriendly_ is not null and
			  char_length( schName_ ) > 0 and
			  char_length( schFriendly_ ) > 0;

		-- trim
		if ok then
			schName		:= trim( schName_ );
			schFriendly	:= trim( schFriendly_ );

			ok := char_length( schName_ ) > 0 and
				  char_length( schFriendly_ ) > 0;
		end if;


		-- type
		if ok then
			schType	:= upper( substr( schFriendly, 1, 1 ) );
			ok		:= schType in ( ''D'', ''W'', ''I'' );
		end if;


		-- minute
		if ok then
			elmStart	:= char_length( schFriendly );
			elmLength	:= 0;

			while elmStart > 2 and substr( schFriendly, elmStart, 1 ) != '':'' loop
				elmStart 	:= elmStart - 1;
				elmLength	:= elmLength + 1;
			end loop;

			if elmLength > 0 then
				schMinute	:= substr( schFriendly, elmStart + 1, elmLength );
				ok			:= 0 <= schMinute and schMinute <= 59;
			end if;
		end if;


		-- hour
		if ok and elmLength > 0 then
			elmStart	:= elmStart - 1;
			elmLength	:= 0;

			while elmStart > 2 and substr( schFriendly, elmStart, 1 ) != '' '' loop
				elmStart 	:= elmStart - 1;
				elmLength	:= elmLength + 1;
			end loop;

			if elmLength > 0 then
				schHour		:= substr( schFriendly, elmStart + 1, elmLength );
				ok			:= 0 <= schHour and schHour <= 23;
			end if;
		end if;


		-- day
		if ok and elmLength > 0 then
			elmStart	:= elmStart - 1;
			elmLength	:= 0;

			while elmStart > 2 and substr( schFriendly, elmStart, 1 ) != ''.'' loop
				elmStart 	:= elmStart - 1;
				elmLength	:= elmLength + 1;
			end loop;

			if elmLength > 0 then
				schDay		:= substr( schFriendly, elmStart + 1, elmLength );
				ok			:= ( schType = ''D'' and -28 <= schDay and schDay <= 31 ) or
							   ( schType = ''W'' and   0 <= schDay and schDay <=  7 ) or
							   ( schType = ''I'' );
			end if;
		end if;


		-- month
		if ok and elmLength > 0 and schType = ''D'' then
			elmStart	:= elmStart - 1;
			elmLength	:= 0;

			while elmStart > 2 and substr( schFriendly, elmStart, 1 ) != ''.'' loop
				elmStart 	:= elmStart - 1;
				elmLength	:= elmLength + 1;
			end loop;

			if elmLength > 0 then
				schMonth	:= substr( schFriendly, elmStart + 1, elmLength );
				ok			:= 1 <= schMonth and schMonth <= 12;
			end if;
		end if;


		-- Year
		if ok and elmLength > 0 and schType = ''D'' then
			elmLength	:= elmStart - 3;

			if elmLength > 0 then
				schYear		:= substr( schFriendly, 3, elmLength );
				ok			:= 1900 <= schYear;
			end if;
		end if;


		-- Affect a row
		if ok then
			if schIsNew then
				-- Build INSERT statement
				temp := ''
					insert into pgj_schedule_( schname, schtype, schyear, schmonth,
												schday, schhour, schminute )
					values ( '' || coalesce( quote_literal( schName ), ''null'' )  || '', '' ||
								coalesce( quote_literal( schType ), ''null'' )  || '', '' ||
								coalesce( quote_literal( schYear ), ''null'' )  || '', '' ||
								coalesce( quote_literal( schMonth ), ''null'' ) || '', '' ||
								coalesce( quote_literal( schDay ), ''null'' )   || '', '' ||
								coalesce( quote_literal( schHour ), ''null'' )  || '', '' ||
								coalesce( quote_literal( schMinute ), ''null'' )|| '' );
					'';
			else
				-- Build UPDATE statement
				temp := ''
					update pgj_schedule_
					set	schtype   = '' || coalesce( quote_literal( schType ), ''null'' )  || '',
						schyear   =	'' || coalesce( quote_literal( schYear ), ''null'' )  || '',
						schmonth  =	'' || coalesce( quote_literal( schMonth ), ''null'' ) || '',
						schday    =	'' || coalesce( quote_literal( schDay ), ''null'' )   || '',
						schhour   =	'' || coalesce( quote_literal( schHour ), ''null'' )  || '',
						schminute =	'' || coalesce( quote_literal( schMinute ), ''null'' )|| ''
					where schname = '' || coalesce( quote_literal( schName ), ''null'' )  || '';
					'';
			end if;

			-- Execute SQL statement
			execute temp;
		end if;

		return ok;
	end;
' language 'plpgsql';



------------------------------------------------
-- Creates a new schedule.
------------------------------------------------
create or replace function pgj_create_schedule
(
	text,	-- $1 name
	text	-- $2 definition
)
returns boolean
as '
	begin
		return pgj_create_update_schedule( $1, $2, TRUE );
	end;
' language 'plpgsql';



------------------------------------------------
-- Updates the definition of an existing schedule.
------------------------------------------------
create or replace function pgj_update_schedule
(
	text,	-- $1 name
	text	-- $2 definition
)
returns boolean
as '
	begin
		return pgj_create_update_schedule( $1, $2, FALSE );
	end;
' language 'plpgsql';



------------------------------------------------
-- Deletes an existing schedule.
-- If the schedule is used by a job, "force"
-- is needed. Otherwise the function fails.
-- Anyway using force is not recommended.
------------------------------------------------
create or replace function pgj_drop_schedule
(
	text,	-- $1 name
	boolean	-- $2 force
)
returns boolean
as '
	declare
		schName			alias for $1;
		useForce		alias for $2;

		schID			int;
		useCount		int;
		ok				boolean;
		temp			text;

	begin
		useCount := 0;

		-- Obtain the schedule ID to avoid joints
		select into	schID S.schid
		from		pgj_schedule_ S
		where		S.schname = schName;

		if schID is not null then
			-- Check if the schedule is used for startup
			select into	useCount count( * )
			from		pgj_job_
			where		jobstartschid = schID;

			if useCount = 0 then
				-- Check if the schedule is used for recurrence
				select into	useCount count( * )
				from		pgj_job_schedule_
				where 		jsschid = schID;
			end if;
		end if;

		if useCount > 0 and useForce then
			temp := ''
					update	pgj_job_
					set		jobstartschid = null
					where	jobstartschid = '' || quote_literal( schID ) || '';
					'';
			execute temp;

			temp := ''
					delete
					from	pgj_job_schedule_
					where	jsschid = '' || quote_literal( schID ) || '';
					'';
			execute temp;

			useCount := 0;
		end if;

		-- At this point useCount must be 0
		-- if we may delete the schedule row
		ok := ( useCount = 0 and schID is not null );

		if ok then
			temp := ''
					delete
					from	pgj_schedule_
					where	schid = '' || quote_literal( schID ) || '';
					'';
			execute temp;
		end if;

		return ok;
	end;
' language 'plpgsql';



------------------------------------------------
-- Deletes an existing schedule without using force.
-- If the schedule is used by a job, the function fails.
------------------------------------------------
create or replace function pgj_drop_schedule
(
	text	-- $1 name
)
returns boolean
as '
	begin
		return pgj_drop_schedule( $1, FALSE );
	end;
' language 'plpgsql';



------------------------------------------------
-- Deletes all schedules that are not used by any job.
-- The task is performed by attempting to delete all
-- schedules without using force. For used schedules
-- such an attempt fails.
------------------------------------------------
create or replace function pgj_drop_unused_schedules
(
)
returns boolean
as '
	declare
		schName		text;
		crsrSchName	cursor for select S.schname from pgj_schedule_ S;
		dummy		boolean;

	begin
		open crsrSchName;
		fetch crsrSchName into schName;
		while found loop
			dummy := pgj_drop_schedule( schName );
			fetch crsrSchName into schName;
		end loop;
		close crsrSchName;

		return TRUE;
	end;
' language 'plpgsql';



------------------------------------------------
-- Produces an interval value from an I-schedule
-- broken into components.
------------------------------------------------
create or replace function pgj_schedule_to_interval
(
	smallint,	-- $1 day
	smallint,	-- $2 hour
	smallint	-- $3 minute
)
returns interval
as '
	declare
		schDay		alias for $1;
		schHour		alias for $2;
		schMinute	alias for $3;

		tiStr		text;

	begin
		tiStr := '''';

		if schDay is not null then
			tiStr := tiStr || schDay || '' day '';
		end if;

		if schHour is not null then
			tiStr := tiStr || schHour || '' hour '';
		end if;

		tiStr := tiStr || schMinute || '' minute '';

		return cast( tiStr as interval );
	end;
' language 'plpgsql';



------------------------------------------------
-- Returns the number of seconds in an interval.
------------------------------------------------
create or replace function pgj_interval_to_seconds
(
	interval	-- $1 ti
)
returns int
as '
	declare
		ti		alias for $1;

	begin
		return	extract( ''day''	from ti ) * 24 * 60 * 60 +
				extract( ''hour''	from ti ) * 60 * 60 +
				extract( ''minute''	from ti ) * 60 +
				extract( ''second''	from ti );
	end;
' language 'plpgsql';



------------------------------------------------
-- Returns the number of minutes in an interval.
------------------------------------------------
create or replace function pgj_interval_to_minutes
(
	interval	-- $1 ti
)
returns int
as '
	declare
		ti		alias for $1;

	begin
		return	extract( ''day''	from ti ) * 24 * 60 +
				extract( ''hour''	from ti ) * 60 +
				extract( ''minute''	from ti );
	end;
' language 'plpgsql';



------------------------------------------------
-- Returns the next trigger time of an I-schedule.
------------------------------------------------
create or replace function pgj_schedule_trigger_time_I
(
	timestamp,	-- $1 startupTime
	timestamp,	-- $2 currentTime
	smallint,	-- $3 schDay
	smallint,	-- $4 schHour
	smallint	-- $5 schMinute
)
returns timestamp
as '
	declare
		startupTime		alias for $1;
		currentTime		alias for $2;
		schDay			alias for $3;
		schHour			alias for $4;
		schMinute		alias for $5;

		schInterval		interval;
		upInterval		interval;
		schMinutes		int;
		upMinutes		int;
		nextIteration	int;

	begin
		schInterval	:= pgj_schedule_to_interval( schDay, schHour, schMinute );
		upInterval	:= currentTime - startupTime;

		schMinutes	:= pgj_interval_to_minutes( schInterval );
		upMinutes	:= pgj_interval_to_minutes( upInterval );
		
		nextIteration := ceil( cast( upMinutes as real ) / cast( schMinutes as real ) );
		
		return startupTime + ( schInterval * nextIteration );
	end;
' language 'plpgsql';



------------------------------------------------
-- Returns the next trigger time of a D- or
-- W-schedule.
------------------------------------------------
create or replace function pgj_schedule_trigger_time_DW
(
	timestamp,	-- $1 currentTime
	char,		-- $2 schType
	smallint,	-- $3 schYear
	smallint,	-- $4 schMonth
	smallint,	-- $5 schDay
	smallint,	-- $6 schHour
	smallint	-- $7 schMinute
)
returns timestamp
as '
	declare
		currentTime		alias for $1;
		schType			alias for $2;
		schYear			alias for $3;
		schMonth		alias for $4;
		schDay			alias for $5;
		schHour			alias for $6;
		schMinute		alias for $7;

		triggerTime		timestamp;
		retryInterval	interval;
		tsStr			text;
		yyyy			smallint;
		mm				smallint;
		dd				smallint;
		hh				smallint;
		nn				smallint;
		dow				smallint;

	begin
		retryInterval := interval ''00:00'';

		if schYear is null then
			retryInterval := interval ''1 year'';
			yyyy := extract( ''year'' from currentTime );
		else
			yyyy := schYear;
		end if;

		if schMonth is null then
			retryInterval := interval ''1 month'';
			mm := extract( ''month'' from currentTime );
		else
			mm := schMonth;
		end if;

		if schDay is null then
			retryInterval := interval ''1 day'';
			dd := extract( ''day'' from currentTime );
		else
			if schType = ''D'' then
				dd := schDay;
			else
				dow := extract( ''dow'' from currentTime );
				dd  := extract( ''day'' from currentTime ) + ( schDay + 7 - dow ) % 7;
				retryInterval := ''1 week'';
			end if;
		end if;

		if schHour is null then
			retryInterval := interval ''1 hour'';
			hh := extract( ''hour'' from currentTime );
		else
			hh := schHour;
		end if;

		nn := schMinute;

		tsStr := yyyy || ''-'' || mm || ''-'' || dd || '' '' || hh || '':'' || nn;
		triggerTime := cast( tsStr as timestamp );
		if triggerTime < currentTime then
			triggerTime := triggerTime + retryInterval;
		end if;

		if triggerTime < currentTime then
			triggerTime := null;
		end if;

		return triggerTime;
	end;
' language 'plpgsql';



------------------------------------------------
-- Returns the next trigger time of a schedule.
-- Redirects the call to the function specific
-- to the type.
------------------------------------------------
create or replace function pgj_schedule_trigger_time
(
	timestamp,	-- $1 startupTime
	timestamp,	-- $2 currentTime
	char,		-- $3 schType
	smallint,	-- $4 schYear
	smallint,	-- $5 schMonth
	smallint,	-- $6 schDay
	smallint,	-- $7 schHour
	smallint	-- $8 schMinute
)
returns timestamp
as '
	declare
		startupTime		alias for $1;
		currentTime		alias for $2;
		schType			alias for $3;
		schYear			alias for $4;
		schMonth		alias for $5;
		schDay			alias for $6;
		schHour			alias for $7;
		schMinute		alias for $8;

		triggerTime		timestamp;
	begin
		if schType in ( ''D'', ''W'' ) then
			triggerTime := pgj_schedule_trigger_time_DW( currentTime, schType,
									schYear, schMonth, schDay, schHour, schMinute );
		elsif schType = ''I'' then
			triggerTime := pgj_schedule_trigger_time_I( startupTime, currentTime,
													schDay, schHour, schMinute );
		else
			triggerTime := null;
		end if;

		return triggerTime;
	end;
' language 'plpgsql';



create view pgj_schedules
as
	select	*,
			pgj_make_schedule( schtype, schyear, schmonth, schday, schhour, schminute ) as schfriendly
	from	pgj_schedule_;


grant select
on pgj_schedules
to public;



------------------------------------------------
-- Jobs
------------------------------------------------

------------------------------------------------
-- Internal table for storing job info.
-- Should not be used directly.
-- Consider view "pgj_jobs".
------------------------------------------------
create table pgj_job_
(
	jobid			serial		primary key,
	jobname			text		not null unique,
	jobfunction		text		not null,
	jobisstartup	boolean		not null,
	jobstartschid	integer		null references pgj_schedule_ ( schid ),
	jobisstartschup	boolean		null
);



------------------------------------------------
-- Internal table for mapping jobs to recurrence
-- schedules. Should not be used directly.
-- Consider view "pgj_jobs".
------------------------------------------------
create table pgj_job_schedule_
(
	jsjobid			integer		not null references pgj_job_ ( jobid ),
	jsschid			integer		not null references pgj_schedule_ ( schid )
);



------------------------------------------------
-- Creates a new job.
------------------------------------------------
create or replace function pgj_create_job
(
	text,		-- $1 name
	text,		-- $2 func
	text,		-- $3 start schedule name
	text		-- $4 recur schedule name
)
returns boolean
as '
	declare
		jobName_		alias for $1;
		jobFunc_		alias for $2;
		schStart_		alias for $3;
		schRecur_		alias for $4;

		jobName			text;
		jobFunc			text;
		schStart		text;
		schRecur		text;
		ok				boolean;
		temp			text;
		schStartID		int;
		schRecurID		int;
		jobID			int;

	begin
		ok := jobName_ is not null and
			  jobFunc_ is not null and
			  char_length( jobName_ ) > 0 and
			  char_length( jobFunc_ ) > 0;

		-- Trim
		if ok then
			jobName		:= trim( jobName_ );
			jobFunc		:= trim( jobFunc_ );

			ok := char_length( jobName ) > 0 and
				  char_length( jobFunc ) > 0;
		end if;

		-- Find start schedule ID, if a name is provided
		if ok and schStart_ is not null then
			schStart := trim( schStart_ );
			if char_length( schStart ) > 0 then
				select into	schStartID S.schid
				from		pgj_schedule_ S
				where		S.schname = schStart;
			end if;
		end if;

		if ok then
			-- INSERT a job row
			temp := ''
				insert into pgj_job_( jobname, jobfunction, jobisstartup, jobstartschid )
				values ( '' ||  quote_literal( jobName ) || '', '' ||
								quote_literal( jobFunc ) || '', '' ||
								''FALSE, '' ||
								coalesce( quote_literal( schStartID ), ''null'' ) || '');
				'';
			execute temp;

			-- Find recurrence schedule ID, if a name is provided.
			-- Also find the job ID.
			if schRecur_ is not null then
				schRecur := trim( schRecur_ );
				if char_length( schRecur ) > 0 then
					select into	schRecurID S.schid
					from		pgj_schedule_ S
					where		S.schname = schRecur;

					select into	jobID J.jobid
					from		pgj_job_ J
					where		J.jobname = jobName;
				end if;
			end if;

			-- INSERT a job-schedule mapping row
			if schRecurID is not null and
			   jobID is not null then
				temp := ''
					insert into pgj_job_schedule_( jsjobid, jsschid )
					values ( '' ||  quote_literal( jobID ) || '', '' ||
									quote_literal( schRecurID ) || '');
					'';
				execute temp;
			end if;
		end if;

		return ok;
	end;
' language 'plpgsql';



------------------------------------------------
-- Updates an existing job.
-- To remove a start schedule, pass '' for $4.
------------------------------------------------
create or replace function pgj_update_job
(
	text,		-- $1 name
	text,		-- $2 rename
	text,		-- $3 func
	text		-- $4 start schedule name
)
returns boolean
as '
	declare
		jobName_		alias for $1;
		jobRename_		alias for $2;
		jobFunc_		alias for $3;
		schStart_		alias for $4;

		jobName			text;
		jobRename		text;
		jobFunc			text;
		schStart		text;
		ok				boolean;
		temp			text;
		schStartID		int;
		jobID			int;

	begin
		ok := jobName_ is not null and
			  char_length( jobName_ ) > 0;

		-- Trim
		if ok then
			jobName		:= trim( jobName_ );
			ok := char_length( jobName ) > 0;
		end if;

		if ok then
			if jobRename_ is not null then
				jobRename := trim( jobRename_ );
			end if;

			if jobFunc_ is not null then
				jobFunc := trim( jobFunc_ );
			end if;

			-- Find start schedule ID, if a name is provided
			if schStart_ is not null then
				schStart := trim( schStart_ );
				if char_length( schStart ) > 0 then
					select into	schStartID S.schid
					from		pgj_schedule_ S
					where		S.schname = schStart;
				end if;
			end if;

			-- Generate UPDATE statement.
			-- At least 1 atribute must be changed.
			ok := FALSE;
			temp := ''	update pgj_job_
						set '';

			if jobRename is not null and char_length( jobRename ) > 0 then
				ok := TRUE;
				temp := temp || '' jobname = '' || quote_literal( jobRename ) || '' '';
			end if;

			if jobFunc is not null and char_length( jobFunc ) > 0 then
				if ok then
					temp := temp || '', '';
				end if;

				ok := TRUE;
				temp := temp || '' jobfunction = '' || quote_literal( jobFunc ) || '' '';
			end if;

			if schStartID is not null then
				if ok then
					temp := temp || '', '';
				end if;

				ok := TRUE;
				temp := temp || '' jobstartschid = '' || quote_literal( schStartID ) || '' '';
			end if;

			if schStartID is null and char_length( schStart ) = 0 then
				if ok then
					temp := temp || '', '';
				end if;

				ok := TRUE;
				temp := temp || '' jobstartschid = null '';
			end if;

			temp := temp || ''where jobname = '' || quote_literal( jobName ) || '';'';
		end if;

		if ok then
			execute temp;
		end if;

		return ok;
	end;
' language 'plpgsql';



------------------------------------------------
-- Adds/removes a recurrence schedule to/from a job.
-- Both the schedule and the job must be existing.
-- Should not be called directly.
-- Consider functions "pgj_add_job_schedule" and
-- "pgj_drop_job_schedule" instead.
------------------------------------------------
create or replace function pgj_add_drop_job_schedule
(
	text,		-- $1 jobName
	text,		-- $2 schName
	boolean		-- $3 mustAdd
)
returns boolean
as '
	declare
		jobName_		alias for $1;
		schName_		alias for $2;
		mustAdd			alias for $3;

		jobName			text;
		schName			text;
		jobID			int;
		schID			int;
		ok				boolean;
		temp			text;

	begin
		ok := jobName_ is not null and
			  schName_ is not null and
			  char_length( jobName_ ) > 0 and
			  char_length( schName_ ) > 0;

		-- Trim
		if ok then
			jobName		:= trim( jobName_ );
			schName		:= trim( schName_ );

			ok := char_length( jobName ) > 0 and
				  char_length( schName ) > 0;
		end if;

		-- Find the job ID and schedule ID
		if ok then
			select into	jobID J.jobid
			from		pgj_job_ J
			where		J.jobname = jobName;

			select into	schID S.schid
			from		pgj_schedule_ S
			where		S.schname = schName;
			
			ok := ( jobID is not null and schID is not null );
		end if;

		-- INSERT/DELETE a job-schedule mapping row
		if ok then
			if mustAdd then
				-- INSERT
				temp := ''
					insert into pgj_job_schedule_( jsjobid, jsschid )
					values ( '' ||  quote_literal( jobID ) || '', '' ||
									quote_literal( schID ) || '');
					'';
			else
				temp := ''
					delete from pgj_job_schedule_
					where jsjobid = '' || quote_literal( jobID ) || '' and
						  jsschid = '' || quote_literal( schID ) || '';
					'';
			end if;

			execute temp;
		end if;

		return ok;
	end;
' language 'plpgsql';



------------------------------------------------
-- Adds a recurrence schedule to a job.
-- Both the schedule and the job must be existing.
------------------------------------------------
create or replace function pgj_add_job_schedule
(
	text,		-- $1 jobName
	text		-- $2 schName
)
returns boolean
as '
	begin
		return pgj_add_drop_job_schedule( $1, $2, TRUE );
	end;
' language 'plpgsql';



------------------------------------------------
-- Removes a recurrence schedule from a job.
-- Both the schedule and the job must be existing.
------------------------------------------------
create or replace function pgj_drop_job_schedule
(
	text,		-- $1 jobName
	text		-- $2 schName
)
returns boolean
as '
	begin
		return pgj_add_drop_job_schedule( $1, $2, FALSE );
	end;
' language 'plpgsql';



------------------------------------------------
-- Removes an existing job.
------------------------------------------------
create or replace function pgj_drop_job
(
	text		-- $1 jobName
)
returns boolean
as '
	declare
		jobName_		alias for $1;

		jobName			text;
		jobID			int;
		ok				boolean;
		temp			text;

	begin
		ok := jobName_ is not null and
			  char_length( jobName_ ) > 0;

		-- Trim
		if ok then
			jobName		:= trim( jobName_ );
			ok := char_length( jobName ) > 0;
		end if;

		-- Find the job ID
		if ok then
			select into	jobID J.jobid
			from		pgj_job_ J
			where		J.jobname = jobName;

			ok := ( jobID is not null );
		end if;

		if ok then
			-- DELETE job-schedule mapping rows
			temp := ''
				delete from pgj_job_schedule_
				where jsjobid = '' || quote_literal( jobID ) || '';
				'';
			execute temp;
			
			-- DELETE the job row
			temp := ''
				delete from pgj_job_ J
				where J.jobid = '' || quote_literal( jobID ) || '';
				'';
			execute temp;
		end if;

		return ok;
	end;
' language 'plpgsql';



------------------------------------------------
-- Creates a new startup job.
------------------------------------------------
create or replace function pgj_create_startup_job
(
	text,		-- $1 name
	text		-- $2 func
)
returns boolean
as '
	declare
		jobName_		alias for $1;
		jobFunc_		alias for $2;

		jobName			text;
		jobFunc			text;
		ok				boolean;
		temp			text;

	begin
		ok := jobName_ is not null and
			  jobFunc_ is not null and
			  char_length( jobName_ ) > 0 and
			  char_length( jobFunc_ ) > 0;

		-- Trim
		if ok then
			jobName		:= trim( jobName_ );
			jobFunc		:= trim( jobFunc_ );

			ok := char_length( jobName ) > 0 and
				  char_length( jobFunc ) > 0;
		end if;

		if ok then
			-- INSERT a job row
			temp := ''
				insert into pgj_job_( jobname, jobfunction, jobisstartup, jobstartschid )
				values ( '' ||  quote_literal( jobName ) || '', '' ||
								quote_literal( jobFunc ) || '',
								TRUE, null );
				'';
			execute temp;
		end if;

		return ok;
	end;
' language 'plpgsql';



------------------------------------------------
-- Public view for retrieving job info and mapped
-- schedules.
------------------------------------------------
create view pgj_jobs
as
	select	J.*,
			SS.schname as schnamestart,
			SS.schfriendly as schfriendlystart,
			RS.schname as schnamerecur,
			RS.schfriendly as schfriendlyrecur
	from	pgj_job_ as J
			left join pgj_schedules as SS
				on J.jobstartschid = SS.schid
			left join pgj_job_schedule_ as JS
				on J.jobid = JS.jsjobid
			left join pgj_schedules as RS
				on JS.jsschid = RS.schid;



------------------------------------------------
-- Triggerring
------------------------------------------------


------------------------------------------------
-- Internal table for storing temporary trigger
-- information. Should not be used directly.
------------------------------------------------
create table pgj_trigger_
(
	schid			integer		not null references pgj_schedule_ ( schid ),
	schtriggertime	timestamp	null,
	isup			boolean		null
);



------------------------------------------------
-- Internal table for storing system/session
-- Settings. Should not be used directly.
------------------------------------------------
create table pgj_settings_
(
	startuptime		timestamp	null
);

truncate table pgj_settings_;

insert into pgj_settings_( startuptime )
values( date_trunc( 'minute', current_timestamp() ) );



------------------------------------------------
-- Recomputes the trigger times for all schedules.
-- The output is stored in table pgj_trigger_.
------------------------------------------------
create or replace function pgj_recompute_schedules()
returns boolean
as '
	declare
		startupTime		timestamp;
		currentTime		timestamp;
		temp			text;

	begin
		select into	startupTime S.startuptime
		from		pgj_settings_ S;

		currentTime := date_trunc( ''minute'', current_timestamp() );

		temp := ''
			truncate table pgj_trigger_;

			insert into	pgj_trigger_( schid, schtriggertime )
			select		schid, pgj_schedule_trigger_time( '' || quote_literal( startupTime ) || '',
														  '' || quote_literal( currentTime ) || '',
														  schtype,
														  schyear,
														  schmonth,
														  schday,
														  schhour,
														  schminute )
			from		pgj_schedule_;

			update	pgj_trigger_
			set		isup = ( schtriggertime = '' || quote_literal( currentTime ) || '' );

			update	pgj_job_
			set		jobisstartschup = TRUE
			from	pgj_job_ J
					join pgj_trigger_ T
						on J.jobstartschid = T.schid
			where	T.isup;
			'';
		execute temp;

		return TRUE;
	end;
' language 'plpgsql';



------------------------------------------------
-- Returns the next trigger time.
------------------------------------------------
create or replace function pgj_trigger_time()
returns timestamp
as '
	declare
		triggerTime		timestamp;
		currentTime		timestamp;

	begin
		currentTime := date_trunc( ''minute'', current_timestamp() );

		select into	triggerTime min( schtriggertime )
		from		pgj_trigger_
		where		schtriggertime >= currentTime;

		if triggerTime = currentTime then
			triggerTime = currentTime + interval ''1 minute'';
		end if;

		return triggerTime;
	end;
' language 'plpgsql';



------------------------------------------------
-- Returns the number of seconds until the next
-- trigger time.
------------------------------------------------
create or replace function pgj_trigger_seconds()
returns int
as '
	declare
		triggerTime		timestamp;
		currentTime		timestamp;

	begin
		currentTime := date_trunc( ''second'', current_timestamp() );
		triggerTime := pgj_trigger_time();

		return	pgj_interval_to_seconds( triggerTime - currentTime );
	end;
' language 'plpgsql';



------------------------------------------------
-- Initializes a pgjobs session:
--		o records startuptime in pgj_settings
--		o records that no startup schedule is up yet
------------------------------------------------
create or replace function pgj_startup()
returns boolean
as '
	declare
		currentTime			timestamp;
		settingsRowCount	int;
		temp				text;

	begin
		currentTime := date_trunc( ''minute'', current_timestamp() );

		select	into settingsRowCount count( * )
		from	pgj_settings_;

		if settingsRowCount = 0 then
			temp := ''
				insert into pgj_settings_( startuptime )
				values( '' || quote_literal( currentTime ) || '' );
				'';
		else
			temp := ''
				update	pgj_settings_
				set		startuptime = '' || quote_literal( currentTime ) || '';
				'';
		end if;
		execute temp;

		temp := ''
			update	pgj_job_
			set		jobisstartschup = null;
			'';
		execute temp;

		return TRUE;
	end;
' language 'plpgsql';



------------------------------------------------
-- Returns a rowset of the jobs that are up for
-- execution.
------------------------------------------------
create view pgj_jobs_up
as
	select	distinct jobid, jobname, jobfunction
	from	pgj_job_ J
			join pgj_job_schedule_ JS
				on J.jobid = JS.jsjobid
			join pgj_trigger_ T
				on JS.jsschid = T.schid
	where	T.isup and
			( J.jobstartschid is null or
			  J.jobisstartschup );



------------------------------------------------
-- Returns a rowset of the startup jobs.
------------------------------------------------
create view pgj_jobs_startup
as
	select	jobid, jobname, jobfunction
	from	pgj_job_
	where	jobisstartup;



------------------------------------------------
-- Done
------------------------------------------------

commit transaction;



