#include "connection.h"
#include "statement.h"
#include "protocol.h"

/*-----------------------------------------------------------------------------
 * SQLBrowseConnect
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLBrowseConnect_(SQLHDBC ConnectionHandle, SQLTCHAR* ConnStrIn, SQLSMALLINT ConnStrInLength,
                  SQLTCHAR* ConnStrOut, SQLSMALLINT ConnStrOutMax,
                  SQLSMALLINT* pcbConnStrOut
								 )
{
	SQLRETURN nRet; /* SQL_NEED_DATA */
	Connection* pConnection = (Connection*) ConnectionHandle;

	ENTER_CONN(ConnectionHandle, _T("SQLBrowseConnect"));

	/* parse connection string */
	nRet = ParseConnectionString(pConnection, ConnStrIn, ConnStrInLength);
	if (SQL_NEED_DATA == nRet)
	{
		/* we need to compile string-request */
		PrepareConnectionStringRequest(ConnStrOut, ConnStrOutMax, pcbConnStrOut, pConnection->parameters);
	}
	else
	{
		/* try to connect */
		PrepareConnectionString(ConnStrOut, ConnStrOutMax, pcbConnStrOut, pConnection->parameters, pConnection->dsn);
		nRet = Connect(pConnection);
	}

	LEAVE_CONN(ConnectionHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * SQLDescribeParam
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLDescribeParam(SQLHSTMT StatementHandle, SQLUSMALLINT ParameterNumber, SQLSMALLINT* DataTypePtr,
								 SQLUINTEGER* ParameterSizePtr, SQLSMALLINT* DecimalDigitsPtr, SQLSMALLINT* NullablePtr)
{
	SQLRETURN nRet = SQL_SUCCESS;
	ENTER_STMT(StatementHandle, _T("SQLDescribeParam"));

	if (0 < ParameterNumber)
	{
		Descriptor* pAPD;
		Descriptor* pIPD;

		pAPD = GET_DESCRIPTOR(((Statement*)StatementHandle)->apd);
		pIPD = GET_DESCRIPTOR(((Statement*)StatementHandle)->ipd);

		if (ParameterNumber <= pAPD->header.count ||
		    ParameterNumber <= pIPD->header.count
		   )
		{
			/* concise type */
			if (NULL != DataTypePtr)
				*DataTypePtr = pIPD->id_records[ParameterNumber].common.concise_type;
			
			/* nullability */
			if (NULL != NullablePtr)
				*NullablePtr = pIPD->id_records[ParameterNumber].nullable;
			
			/* display size */
			/* decimal digits */
			nRet = SQLTypeDescriptor(pIPD->id_records[ParameterNumber].common.concise_type, /* IN: concise_type */
			                         pIPD->id_records[ParameterNumber].is_unsigned,         /* IN: unsigned value */
												      &pIPD->id_records[ParameterNumber].common.length,       /* IN: server specyfic size */
												       DecimalDigitsPtr,
												       NULL, NULL,
												       ParameterSizePtr,
												       NULL, NULL);
		}
		else
		{
			TCHAR paramNumber[SQLSMALLINT_LENGTH];

			/* no parameter binded with this number */
			_itot(++ParameterNumber, paramNumber, 10);
			SetError(SQL_HANDLE_STMT, StatementHandle, ERR_STMT_PARAM_NUMBER_NOT_BINDED, paramNumber, NULL);
			nRet = SQL_ERROR;
		}

		RET_DESCRIPTOR(pIPD);
		RET_DESCRIPTOR(pAPD);
	}
	else
	{
		/* bind bookmark's column */
		if (SQL_UB_OFF != ((Statement*)StatementHandle)->attributes.use_bookmarks)
		{
	
		}
		else
		{
			/* no bookmarks used */
			nRet = SQL_ERROR;
		}

	}

	LEAVE_STMT(StatementHandle, nRet);
}

/*-----------------------------------------------------------------------------
 * SQLSetPos
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLSetPos(SQLHSTMT StatementHandle,  SQLUSMALLINT irow, SQLUSMALLINT fOption, SQLUSMALLINT fLock)
{
	ENTER_STMT(StatementHandle, _T("SQLSetPos"));
	/* 78,79,80,83*/
	LEAVE_STMT(StatementHandle, SQL_ERROR);
}

/*-----------------------------------------------------------------------------
 * SQLBulkOperations
 *-----------------------------------------------------------------------------
 */
SQLRETURN SQL_API
SQLBulkOperations(SQLHSTMT StatementHandle, SQLSMALLINT Operation)
{
	SQLRETURN nRet = SQL_SUCCESS;
	Statement* pStatement = (Statement*)StatementHandle;
	ENTER_STMT(StatementHandle, _T("SQLBulkOperations"));

	switch(Operation)
	{
		case SQL_ADD:
		{/* -- insert data --
			* SQL_ATTR_ROW_ARRAY_SIZE containes the number of rows that application wants to insert,
			* SQL_ATTR_ROW_STATUS_PTR can be checked for results
			* data bound by the SQLBindCol
			*/
			uint32      table_oid;
			SQLSMALLINT columns;
			SQLSMALLINT i;
			Descriptor* pIRD = GET_DESCRIPTOR(pStatement->ird);

			/* check insert possibility */
			if (0 >= (columns = pIRD->header.count))
			{
				SetError(SQL_HANDLE_STMT, pStatement, ERR_BULK_NO_COLUMNS_TO, _T("add"), NULL);
				nRet = SQL_SUCCESS_WITH_INFO;
			}
			else
			{
				if (0 >= (table_oid = pIRD->id_records[0].table_oid))
					nRet = SQL_ERROR;
				else for(i=columns-1;i>=0;i--)
					if (table_oid != pIRD->id_records[i].table_oid)
					{
						nRet = SQL_ERROR;
						break;
					}
				if (SQL_ERROR == nRet)
					SetError(SQL_HANDLE_STMT, pStatement, ERR_BULK_DIFFERENT_TABLES, NULL);
				else
				{
					Descriptor* pARD = GET_DESCRIPTOR(pStatement->ard);
					/* -- check - 'all columns bound to the driver by the application?' -- */
					if (pARD->header.count >= pIRD->header.count)
						for (i=columns-1;i>=0;i--)
						{
							if (SQL_FALSE == pARD->ad_records[i].bound)
								break;
						}

					if (0 <= i)
					{/* not all columns were bound to the driver by the application */
						SetError(SQL_HANDLE_STMT, pStatement, ERR_STMT_NOT_ENOUGH_PARAMETERS, NULL);
						nRet = SQL_ERROR;
					}
					else
					{
						/* check parameters for SQL_DATA_AT_EXEC */
						for (i=0;i<columns;i++)
						{
							if (SQL_DATA_AT_EXEC == *pARD->ad_records[i].indicator_ptr)
							{
								pStatement->need_data = _T('c');
								nRet = SQL_NEED_DATA;
								break;
							}
						}

						if (SQL_NEED_DATA != nRet)
						{
							TCHAR         schema[SCHEMA_MAX_LENGTH+1];
							TCHAR         table[TABLE_MAX_LENGTH+1];
							SQLUSMALLINT* status_ptr;
							SQLINTEGER    rows;
							Statement*    pAddStatement;

							GetStmtAttr(pStatement, SQL_ATTR_ROW_ARRAY_SIZE, &rows);       /* number of rows to insert */
							GetStmtAttr(pStatement, SQL_ATTR_ROW_STATUS_PTR, &status_ptr); /* info array to return */

							GetDescField(pIRD, 1, SQL_DESC_SCHEMA_NAME, schema, sizeof(schema), NULL, NULL); /* schema name */
							GetDescField(pIRD, 1, SQL_DESC_TABLE_NAME,  table, sizeof(table), NULL, NULL);   /* table name */

							/* alloc new statement */
							if (pAddStatement = AllocStatement(pStatement->connection))
							{/*  prepare insert query */
								SQLSMALLINT col_length;
								TCHAR  questions[(COLUMN_MAX_LENGTH+CHR_SIZEOF(",?,"))*MAX_COLUMNS_IN_TABLE+1];
								TCHAR* insertQuery;
								TCHAR* col_name = questions;
				
								for (i=columns-1;i>=0;i--)
								{/* prepare questions */
									*(col_name++) = _T('?');
									*(col_name++) = _T(',');
								}
								*(--col_name) = _T('\0');

								for (i=1;i<=columns;i++)
								{/* prepare column-name's list */
									GetDescField(pIRD, i, SQL_DESC_BASE_COLUMN_NAME, ++col_name, COLUMN_MAX_LENGTH+1, &col_length, NULL);
									col_name += col_length/sizeof(TCHAR);
									*col_name = _T(',');
								}
								*col_name = _T('\0');

								if (insertQuery = GetText(_T("INSERT INTO \"\?\".\"\?\"(\?)VALUES(\?)"), schema, table, questions+2*pIRD->header.count/* column names */, questions, NULL))
								{
									Descriptor* pIPD = GET_DESCRIPTOR(pAddStatement->ipd);
									PrepareStatement(pAddStatement, insertQuery, SQL_NTS);
									free(insertQuery);

									if (SQL_ERROR != ReallocDescriptorRecords(pIPD, columns))
									{
										SQLSMALLINT j;
										/* transaction */
										BeginTransaction(pAddStatement, TO_DRIVER);

										/* bind parameters and execute prepared query */
										for(j=0;j<rows;j++)
										{
											/* translate parameters */
											for (i=0;i<columns;i++)
											{/* convert parameters from SQL_C_TYPE into backend's specific format(text) */
												CD_REC* common = &pARD->ad_records[i].common;
			
												if (NULL == (pIPD->id_records[i].common.data_ptr = (SQLPOINTER)PrepareParameter(pStatement, common->data_ptr, common->octet_length, common->concise_type, pARD->ad_records[i].indicator_ptr)))
												{/* no more memory */
													nRet = SQL_ERROR;
													break;
												}
											}

											nRet = ((SQL_SUCCESS == (nRet = DeclarePortal(pAddStatement))) &&
															(SQL_SUCCESS == Stmt_SendMessageToBackend(pAddStatement->connection, MSG_Execute, pAddStatement)) &&
															(SQL_SUCCESS == Stmt_SendMessageToBackend(pAddStatement->connection, MSG_Sync, pAddStatement)) &&
															(SQL_SUCCESS == WaitForBackendReply(pAddStatement->connection,  MSG_ReadyForQuery, pAddStatement))
													 ) ? SQL_ROW_ADDED : SQL_ROW_ERROR;
												
											/* update status array */
											if (status_ptr)
												status_ptr[j] = nRet;
										}
										EndTransaction(SQL_HANDLE_STMT, pStatement, SQL_COMMIT /* in any case */, TO_DRIVER);
										nRet = SQL_SUCCESS;
									}
									else
									{
										SetError(SQL_HANDLE_STMT, pStatement, ERR_NOT_ENOUGH_MEMORY, NULL);
									}
									RET_DESCRIPTOR(pIPD);
								}
								FreeStatement(pAddStatement, SQL_DROP);
							}
						}
					}
					RET_DESCRIPTOR(pARD);
				}
			}
			RET_DESCRIPTOR(pIRD);
			break;
		}
		case SQL_UPDATE_BY_BOOKMARK:
		case SQL_DELETE_BY_BOOKMARK:
		case SQL_FETCH_BY_BOOKMARK:
		default:
			/* Driver Manager is not supposed to let us come here, even more - it's impossible! */
			SetError(SQL_HANDLE_STMT, StatementHandle, ERR_INVALID_BULK_OPERATION, NULL);
			nRet = SQL_ERROR;
	}

	LEAVE_STMT(StatementHandle, nRet);
}
