/*
 * Written by Tomas Dosoudil
 * This file is distributed under BSD-style license.
 *
 */

#include	<postgres.h>
#include	<storage/bufpage.h>
#include	<access/htup.h>
#include	<access/itup.h>
#include	<access/transam.h>
#include	<access/nbtree.h>
#include	"output.h"
#include	"pgdview.h"

/* 
 * print file number 
 */
void
PrintFileName(char *fileName, int numberOfFile) 
{
	if (numberOfFile == 0)
		printf("Filename: %s\n", fileName);
	else	
		printf("Filename: %s.%d\n", fileName, numberOfFile);
}

/*
 * page info
 * it prints number of page, tuples and free space
 * tuple format is: total(used/delete/unused)
 */
void 
PrintPageInfo(int pageNumber, uint32 itemCount, uint32 itemUnused, uint32 itemNormal,
			  uint32 itemRedirect, uint32 itemDead, Size freeSpace)
{
	printf("Page: %4d\tTuples: %3d(U:%3d N:%3d R:%3d D:%3d)\tFree Space: %4d B\n",
			pageNumber, itemCount, itemUnused, itemNormal, itemRedirect, itemDead, freeSpace);
}

/* 
 * print pageheader 
 */
void
PrintPageHeader(PageHeader pageHeader)
{
    printf("\tpd_lsn: %08X%08X\n", (pageHeader->pd_lsn).xlogid, (pageHeader->pd_lsn).xrecoff);
	printf("\t  xlogid: %08X\n", (pageHeader->pd_lsn).xlogid);
	printf("\t  xrecoff: %08X\n", (pageHeader->pd_lsn).xrecoff);
    printf("\tpd_tli: %04X\n", pageHeader->pd_tli);
    printf("\tpd_flags: %04X\n", pageHeader->pd_flags);
    
	/* show values in pd_flags if are set */
	if (pageHeader->pd_flags) {
		if (PageHeaderHasFreeLines(pageHeader)) 
			printf("\t  PD_HAS_FREE_LINES\n");   

		if (PageHeaderHasPageFull(pageHeader)) 
			printf("\t  PD_PAGE_FULL\n"); 

		if (PageHeaderHasValidFlagBits(pageHeader)) 
			printf("\t  PD_VALID_FLAG_BITS\n"); 
	} 
    
    printf("\tpd_lower: %04X\n", pageHeader->pd_lower);
    printf("\tpd_upper: %04X\n", pageHeader->pd_upper);
    printf("\tpd_special: %04X\n", pageHeader->pd_special);
    printf("\tpd_pagesize_version: %04X\n", pageHeader->pd_pagesize_version);
	printf("\t  Page size: %02X\n", (pageHeader->pd_pagesize_version & 0xFF00));
	printf("\t  Layout version: %02X\n", (pageHeader->pd_pagesize_version & 0x00FF));
    printf("\tpd_prune_xid: %08X\n", pageHeader->pd_prune_xid);		
}

/* 
 * print itempointer on position
 */
void
PrintPageItemPointer(int index, ItemId item)
{
	printf("\t\tTuple: %3X\tOffset: %4X\tLength: %4X\tFlags: %02X ", index, 
			ItemIdGetOffset(item), ItemIdGetLength(item), ItemIdGetFlags(item));

	if (ItemIdGetFlags(item) == LP_UNUSED)
		printf("(Unused)\n");
	else if (ItemIdGetFlags(item) == LP_NORMAL)
		printf("(Normal)\n");
	else if (ItemIdGetFlags(item) == LP_REDIRECT)
		printf("(Redirect)\n");			
	else
		printf("(Dead)\n");

}

/* 
 * print redirected itempointer
 */
void
PrintPageRedirectItemPointer(int index)
{	
	printf("\t\t\tRedirected to tuple: %d\n", index);
}

/* 
 * print header of tuple 
 */
void
PrintHeapTupleHeader(HeapTupleHeader hTupHeader)
{
	printf("\t\t\tt_xmin: %08X\n", HeapTupleHeaderGetXmin(hTupHeader));
	printf("\t\t\tt_xmax: %08X\n", HeapTupleHeaderGetXmax(hTupHeader));
	printf("\t\t\tt_cid:  %08X\n", HeapTupleHeaderGetRawCommandId(hTupHeader));
	printf("\t\t\tt_xvac: %08X\n", HeapTupleHeaderGetXvac(hTupHeader));
	/* 
 	 * order of structures:
     * HeapTupleHeader -> ItemPointerData -> { BlockIdData -> { bi_hi, bi_lo }, ip_posid } 
     */
	printf("\t\t\tt_ctid: %04X%04X%04X\n", hTupHeader->t_ctid.ip_blkid.bi_hi, 
			hTupHeader->t_ctid.ip_blkid.bi_lo, hTupHeader->t_ctid.ip_posid);
	printf("\t\t\tt_infomask: %04X\n", hTupHeader->t_infomask);
	
	/* show values in t_infomask */
	if (hTupHeader->t_infomask) {
		if (_HeapTupleHasNulls(hTupHeader)) 
			printf("\t\t\t  HEAP_HASNULL\n");
	
		if (_HeapTupleHasVarWidth(hTupHeader)) 
			printf("\t\t\t  HEAP_HASVARWIDTH\n");
	
		if (_HeapTupleHasExternal(hTupHeader)) 
			printf("\t\t\t  HEAP_HASEXTERNAL\n");

		if (HeapTupleHasOid(hTupHeader)) {
			printf("\t\t\t  HEAP_HASOID\n");
			printf("\t\t\t   Oid: %X\n", _HeapTupleGetOid(hTupHeader));
		}
			
		if (HeapTupleHasComboCid(hTupHeader)) 
			printf("\t\t\t  HEAP_COMBOCID\n");
			
		if (HeapTupleHasXmaxExclLock(hTupHeader)) 
			printf("\t\t\t  HEAP_XMAX_EXCL_LOCK\n");
			
		if (HeapTupleHasXmaxSharedLock(hTupHeader)) 
			printf("\t\t\t  HEAP_XMAX_SHARED_LOCK\n");
			
		if (HeapTupleHasXminCommitted(hTupHeader)) 
			printf("\t\t\t  HEAP_XMIN_COMMITTED\n");

		if (HeapTupleHasXminInvalid(hTupHeader)) 
			printf("\t\t\t  HEAP_XMIN_INVALID\n");

		if (HeapTupleHasXmaxCommitted(hTupHeader)) 
			printf("\t\t\t  HEAP_XMAX_COMMITTED\n");

		if (HeapTupleHasXmaxInvalid(hTupHeader)) 
			printf("\t\t\t  HEAP_XMAX_INVALID\n");
			
		if (HeapTupleHasXmaxIsMulti(hTupHeader)) 
			printf("\t\t\t  HEAP_XMAX_IS_MULTI\n");

		if (HeapTupleHasUpdated(hTupHeader)) 
			printf("\t\t\t  HEAP_UPDATED\n");

		if (HeapTupleHasMovedOff(hTupHeader)) 
			printf("\t\t\t  HEAP_MOVED_OFF\n");
	
		if (HeapTupleHasMovedIn(hTupHeader)) 
			printf("\t\t\t  HEAP_MOVED_IN\n");

		if (HeapTupleHasXactMask(hTupHeader)) 
			printf("\t\t\t  HEAP_XACT_MASK\n");
	}
	
	printf("\t\t\tt_infomask2: %04X\n", hTupHeader->t_infomask2);
	/* show values in t_infomask2 */
	if (hTupHeader->t_infomask2) {
		if (HeapTupleHasNattsMask(hTupHeader)) {
			printf("\t\t\t  Attributes: %X\n", 
				    HeapTupleGetNumAttributes(hTupHeader));
			printf("\t\t\t  HEAP_NATTS_MASK\n");
		}
		
		if (HeapTupleHasHotUpdated(hTupHeader)) 
			printf("\t\t\t  HEAP_HOT_UPDATED\n");
			
		if (HeapTupleHasOnlyTuple(hTupHeader)) 
			printf("\t\t\t  HEAP_ONLY_TUPLE\n");
			
		if (HeapTupleHasXastMask2(hTupHeader)) 
			printf("\t\t\t  HEAP2_XACT_MASK\n");			
	}
	
	printf("\t\t\tt_hoff: %02X\n", hTupHeader->t_hoff);
}

void
PrintIndexTupleHeader(IndexTuple iTupHeader) 
{
	printf("\t\t\tt_tid: %04X%04X%04X\n", iTupHeader->t_tid.ip_blkid.bi_hi,
			iTupHeader->t_tid.ip_blkid.bi_lo, iTupHeader->t_tid.ip_posid);
	printf("\t\t\tt_info: %04X\n",iTupHeader->t_info);	
	printf("\t\t\t Size: %X\n", iTupHeader->t_info & INDEX_SIZE_MASK);
}

void
PrintSpecialSpace(BTPageOpaque metaOpaque) 
{
	printf("\tbtpo_prev: %04X\n", metaOpaque->btpo_prev);
	printf("\tbtpo_next: %04X\n", metaOpaque->btpo_next);
	printf("\tbtpo: %X\n", metaOpaque->btpo.level);
	printf("\tbtpo_flags: %02X\n", metaOpaque->btpo_flags);
	printf("\tbtpo_cycleid: %02X\n", metaOpaque->btpo_cycleid);
}

/*
 * print tuple data 
 * format is like in hexdump: offset data(2x8B)
 */
void
PrintHeapTupleData(HeapTupleHeader hTupHeader, uint32 tupleOffset, uint32 tupleLength, bool optReadable)
{
	uint8	*userData;		/* point to data */
	uint32	userDataLen;	/* data length */
	uint32	dataOffset;		/* data offset in page */
	uint32	i = 0,
			j = 0;			/* counters */
	uint32	mod = 0;
	char	buf[16];

	/* point to tuple data */
	userData = (uint8*) hTupHeader;
	userData += hTupHeader->t_hoff;

	userDataLen = tupleLength - hTupHeader->t_hoff;
	dataOffset = tupleOffset + hTupHeader->t_hoff;

	printf("\t\t\t\t%04X | ", dataOffset);	/* offset */
	for (j = 0; j < userDataLen; j++) {

		if (((j % 8) == 0) && (j != 0))
			printf(" ");

		if (((j % 16) == 0) && (j != 0)) {
			/* readable */
			if (optReadable == TRUE) {
				printf("|");
				for (i = 0; i < 16; i++) {
					if (isprint(buf[i]) && !isspace(buf[i]))
						printf("%c", buf[i]);
					else
						printf(".");
				}
				i = 0;
			}
			
			
			printf("\n\t\t\t\t%04X | ", dataOffset + j);	/* new line, print offset */
		}

		printf("%02X ", userData[j]);
		if (optReadable == TRUE) {
			buf[i] = userData[j];
			i++;
		}
	}
	
	/* last readable line */
	if (optReadable == TRUE) {
		i = 0;
		mod = j % 16;
		
		if (mod) {
			/* align to previous readable lines */
			while (j % 16) {
				if (((j % 8) == 0))
					printf(" ");
				j++;
				printf("   ");
			}
			
			printf(" |");
			while (i < mod) {
				if (isprint(buf[i]) && !isspace(buf[i]))
					printf("%c", buf[i]);
				else
					printf(".");
				
				i++;
			}
		}
	}
	
	printf("\n");
}

void
Usage(void) 
{
	printf("pgdview [-p [num] -i [num] [-cmHIsdrfh] { -D db_path table_oid | file_name }\n");
	printf("All pages:\n");
	printf(" -p page\n");
	printf(" -c page header\n");
	printf(" -i item list\n");

	printf("\nHeap page:\n");
	printf(" -H heap tuple header\n");
	printf(" -d tuple data\n");
	printf(" -r readable data output\n");

	printf("\nIndex page:\n");
	printf(" -I index tuple header\n");
	printf(" -s special space\n");
	printf(" -m meta data page\n");

	printf("\n");
	printf(" -f follow ctid\n");
	printf(" -D database path\n");
	printf(" -h help\n");
}
