/**
 * $Id: backend.c,v 1.2 2004/12/01 06:29:47 mr-russ Exp $
 *
 * database backend functions
 *
 * Copyright (c) 2001 by Joerg Wendland, Bret Mogilefsky
 * see included file COPYING for details
 *
 */

#include "nss-pgsql.h"
#include <libpq-fe.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>

#define MIN(a, b)  (((a) < (b)) ? (a) : (b))

static PGconn *_conn = NULL;
static int _isopen = 0;
static int shadow = 0;

int backend_isopen(char type)
{
	if (type == 's' && shadow == 1) {
		return (_isopen > 0);
	} else if (type == 'x') {
		// Return value for close query
		return (_isopen > 0);
	} else if (type != 's' && shadow == 0) {
		return (_isopen > 0);
	} else {
		return 0;
	}
}

/*
 * read configuration and connect to database
 */
int backend_open(char type)
{
	if (type == 's' && shadow == 0 && _conn != NULL) {
		// Need to reconnect to db as different user
		cleanup();
	} else if (type != 's' && shadow == 1 && _conn != NULL) {
		// Need to reconnect to db as different user
		cleanup();
	}
	
	if (type == 's') {
		shadow = 1;
	} else {
		shadow = 0;
	}
		
	if(!_isopen) {
		if ((shadow && readconfig(CFGROOTFILE)) ||
			(!shadow && readconfig(CFGFILE))) {
			if (_conn == NULL) {
				_conn = PQconnectdb(getcfg("connectionstring"));
			}

			if(PQstatus(_conn) == CONNECTION_OK) {
				++_isopen;
			} else {
				print_msg("\nCould not connect to database\n");
			}
		}
	}

	return (_isopen > 0);
}

/*
 * close connection to database and clean up configuration
 */
void backend_close(const char *what)
{
	--_isopen;
	if(!_isopen) {
		if(what != NULL) {
			// We started a transaction block for prepare, we need to close it
			PQexec(_conn, "COMMIT");
		}
		PQfinish(_conn);
		_conn = NULL;
	}
	if(_isopen < 0) {
		_isopen = 0;
	}
}

/*
 *  prepare a cursor in database
 */
inline void backend_prepare(const char *what)
{
	char *stmt;
	PQexec(_conn, "BEGIN TRANSACTION");
	asprintf(&stmt, "DECLARE nss_pgsql_internal_%s_curs CURSOR FOR "
	                "%s FOR READ ONLY", what, getcfg(what));
	PQexec(_conn, stmt);
	free(stmt);
}


/*
  With apologies to nss_ldap...
  Assign a single value to *valptr from the specified row in the result
*/
enum nss_status
copy_attrval_n(PGresult *res,
               int attrib_number,
               char **valptr, char **buffer, size_t *buflen, int row)
{

	const char *sptr;
	size_t slen;

	sptr = PQgetvalue(res, row, attrib_number);
	slen = strlen(sptr);
	if(*buflen < slen+1) {
		return NSS_STATUS_TRYAGAIN;
	}
	strncpy(*buffer, sptr, slen);
	(*buffer)[slen] = '\0';
		
	*valptr = *buffer;

	*buffer += slen + 1;
	*buflen -= slen + 1;

	return NSS_STATUS_SUCCESS;
}


/*
  With apologies to nss_ldap...
  Assign a single value to *valptr.
*/
enum nss_status
copy_attrval (PGresult *res,
              int attrib_number,
              char **valptr, char **buffer, size_t *buflen)
{
	return copy_attrval_n(res, attrib_number, valptr, buffer, buflen, 0);
}


/*
 * return array of strings containing usernames that are member of group with gid 'gid'
 */
enum nss_status getgroupmem(gid_t gid,
                            struct group *result,
                            char *buffer,
                            size_t buflen)
{
	char *stmt;
	PGresult *res;
	int n, t = 0;
	enum nss_status status = NSS_STATUS_NOTFOUND;
	size_t ptrsize;

	asprintf(&stmt, getcfg("getgroupmembersbygid"), gid);
	res = PQexec(_conn, stmt);

	n = PQntuples(res);

	if(!n) {
		goto BAIL_OUT;
	}

	// Make sure there's enough room for the array of pointers to group member names
	ptrsize = (n+1) * sizeof(const char *);
	if(buflen < ptrsize) {
		status = NSS_STATUS_TRYAGAIN;
		goto BAIL_OUT;
	}

	result->gr_mem = (char**)buffer;

	buffer += ptrsize;
	buflen -= ptrsize;

	for(t = 0; t < n; t++) {
		// Should return only 1 column, use the first one.
		// FIXME: LOG error if issues
		status = copy_attrval_n(res, 0, &(result->gr_mem[t]), &buffer, &buflen, t);
		if(status != NSS_STATUS_SUCCESS)
			goto BAIL_OUT;
	}
	result->gr_mem[n] = NULL;
	
 BAIL_OUT:

	PQclear(res);
	free(stmt);

	return status;
}

/*
 * 'convert' a PGresult to struct group
 */
enum nss_status res2grp(PGresult *res,
                        struct group *result,
                        char *buffer,
                        size_t buflen)
{
	enum nss_status status = NSS_STATUS_NOTFOUND;

#ifdef COMMENTED_OUT
	char **i;
#endif

	if(!PQntuples(res)) {
		goto BAIL_OUT;
	}

	// Carefully copy attributes into buffer.  Return NSS_STATUS_TRYAGAIN if not enough room.
	status = copy_attrval (res, 0, &result->gr_name    , &buffer, &buflen);
	if(status != NSS_STATUS_SUCCESS) goto BAIL_OUT;

	status = copy_attrval (res, 1, &result->gr_passwd  , &buffer, &buflen);
	if(status != NSS_STATUS_SUCCESS) goto BAIL_OUT;

	result->gr_gid = (gid_t)strtoul(PQgetvalue(res, 0, 2), (char**)NULL, 10);

	status = getgroupmem(result->gr_gid, result, buffer, buflen);
 
#ifdef COMMENTED_OUT
	print_msg("Converted a res to a grp:\n");
	print_msg("GID: %d\n", result->gr_gid);
	print_msg("Name: %s\n", result->gr_name);
	print_msg("Password: %s\n", result->gr_passwd);
	i = result->gr_mem;
	while(*i) {
		print_msg("Member: %s\n", *i++);
	}
	print_msg("\n");
#endif

 BAIL_OUT:
	return status;
}

/*
 * 'convert' a PGresult to struct passwd
 */
enum nss_status res2pwd(PGresult *res, struct passwd *result,
                        char *buffer,
                        size_t buflen)
{
	enum nss_status status = NSS_STATUS_NOTFOUND;

	if(!PQntuples(res)) {
		goto BAIL_OUT;
	}

	// Carefully copy attributes into buffer.  Return NSS_STATUS_TRYAGAIN if not enough room.
	// Must return passwd_name, passwd_passwd, passwd_gecos, passwd_dir, passwd_shell, passwd_uid, passwd_gid
	status = copy_attrval (res, 0, &result->pw_name  , &buffer, &buflen);
	if(status != NSS_STATUS_SUCCESS) goto BAIL_OUT;

	status = copy_attrval (res, 1, &result->pw_passwd, &buffer, &buflen);
	if(status != NSS_STATUS_SUCCESS) goto BAIL_OUT;

	status = copy_attrval (res, 2, &result->pw_gecos , &buffer, &buflen);
	if(status != NSS_STATUS_SUCCESS) goto BAIL_OUT;

	status = copy_attrval (res, 3, &result->pw_dir   , &buffer, &buflen);
	if(status != NSS_STATUS_SUCCESS) goto BAIL_OUT;

	status = copy_attrval (res, 4, &result->pw_shell , &buffer, &buflen);
	if(status != NSS_STATUS_SUCCESS) goto BAIL_OUT;

	// Can be less careful with uid/gid
	result->pw_uid = (uid_t) strtoul(PQgetvalue(res, 0, 5), (char**)NULL, 10);
	result->pw_gid = (gid_t) strtoul(PQgetvalue(res, 0, 6), (char**)NULL, 10);

#ifdef COMMENTED_OUT
	print_msg("Converted a res to a pwd:\n");
	print_msg("UID: %d\n", result->pw_uid);
	print_msg("GID: %d\n", result->pw_gid);
	print_msg("Name: %s\n", result->pw_name);
	print_msg("Password: %s\n", result->pw_passwd);
	print_msg("Gecos: %s\n", result->pw_gecos);
	print_msg("Dir: %s\n", result->pw_dir);
	print_msg("Shell: %s\n", result->pw_shell);
#endif

BAIL_OUT:
	return status;
}

/*
 * fetch a row from cursor
 */
PGresult *fetch(char *what)
{
	char *stmt;
	PGresult *res;

	asprintf(&stmt, "FETCH FROM nss_pgsql_internal_%s_curs", what);
	if(_conn == NULL) {
		D("Did a fetch with the database closed!");
	}
	if(PQstatus(_conn) != CONNECTION_OK) {
		D("oops! die connection is futsch");
		return NULL;
	}
	res = PQexec(_conn, stmt);
	free(stmt);

	return res;
}

/*
 * get a group entry from cursor
 */
enum nss_status backend_getgrent(struct group *result,
                                 char *buffer,
                                 size_t buflen,
                                 int *errnop)
{
	PGresult *res;
	enum nss_status status = NSS_STATUS_NOTFOUND;

	res = fetch("allgroups");
	if(res) {
		status = res2grp(res, result, buffer, buflen);
		PQclear(res);
	}
	return status;
}    

/*
 * get a passwd entry from cursor
 */
enum nss_status backend_getpwent(struct passwd *result,
                                 char *buffer,
                                 size_t buflen,
                                 int *errnop)
{
	PGresult *res;
	enum nss_status status = NSS_STATUS_NOTFOUND;

	res = fetch("allusers");
	if(res) {
		status = res2pwd(res, result, buffer, buflen);
		PQclear(res);
	}
	return status;
}    

/*
 * backend for getpwnam()
 */
enum nss_status backend_getpwnam(const char *name, struct passwd *result,
                                 char *buffer, size_t buflen, int *errnop)
{
	char *stmt, *ename;
	PGresult *res;
	size_t len;
	enum nss_status status = NSS_STATUS_NOTFOUND;

	len = strlen(name);
	ename = malloc(2*len+1);
	PQescapeString(ename, name, len);
	asprintf(&stmt, getcfg("getpwnam"), ename);

	res = PQexec(_conn, stmt);
	if(res) {
		// Fill result structure with data from the database
		status = res2pwd(res, result, buffer, buflen);
		 PQclear(res);
	}
	free(stmt);
	free(ename);
    
	return status;
}

/*
 * backend for getpwuid()
 */
enum nss_status backend_getpwuid(uid_t uid, struct passwd *result,
	char *buffer, size_t buflen, int *errnop)
{
	char *stmt;
	PGresult *res;
	enum nss_status status = NSS_STATUS_NOTFOUND;
    
	asprintf(&stmt, getcfg("getpwuid"), uid);
	res = PQexec(_conn, stmt);
	if(res) {
		// Fill result structure with data from the database
		status = res2pwd(res, result, buffer, buflen);
		PQclear(res);
	}
	free(stmt);
    
	return status;
}

/*
 * backend for getgrnam()
 */
enum nss_status backend_getgrnam(const char *name,
                                 struct group *result,
                                 char *buffer,
                                 size_t buflen,
                                 int *errnop)
{
	char *stmt, *ename;
	PGresult *res;
	size_t len;
	enum nss_status status = NSS_STATUS_NOTFOUND;

	len = strlen(name);
	ename = malloc(2*len+1);
	PQescapeString(ename, name, len);

	asprintf(&stmt, getcfg("getgrnam"), ename);
	res = PQexec(_conn, stmt);
	if(res) {
		status = res2grp(res, result, buffer, buflen);
		PQclear(res);
	}

	free(stmt);
	free(ename);
    
	return status;
}


/*
 * backend for getgrgid()
 */
enum nss_status backend_getgrgid(gid_t gid,
                                 struct group *result,
                                 char *buffer,
                                 size_t buflen,
                                 int *errnop)
{
	char *stmt;
	PGresult *res;
	enum nss_status status = NSS_STATUS_NOTFOUND;
    
	asprintf(&stmt, getcfg("getgrgid"), gid);
	res = PQexec(_conn, stmt);
	if(res) {
		status = res2grp(res, result, buffer, buflen);
		PQclear(res);
	}
	free(stmt);
    
	return status;
}


size_t backend_initgroups_dyn(const char *user,
                              gid_t group,
                              long int *start,
                              long int *size,
                              gid_t **groupsp,
                              long int limit,
                              int *errnop)
{
	char *stmt, *euser;
	PGresult *res;
	size_t len;
	gid_t *groups = *groupsp;
	int rows;

	len = strlen(user);
	euser = malloc(2*len+1);
	PQescapeString(euser, user, len);

	asprintf(&stmt, getcfg("groups_dyn"), euser, group);
	res = PQexec(_conn, stmt);

	rows = PQntuples(res);

	if(rows+(*start) > *size) {
		// Have to make the result buffer bigger
		long int newsize = rows + (*start);
		newsize = (limit > 0) ? MIN(limit, newsize) : newsize;
		*groupsp = groups = realloc(groups, newsize * sizeof(*groups));
		*size = newsize;
	}

	rows = (limit > 0) ? MIN(rows, limit - *start) : rows;

	while(rows--) {
		groups[*start] = atoi(PQgetvalue(res, rows, 0));
		*start += 1;
	}

	PQclear(res);
	free(stmt);
	free(euser);
    
	return *start;
}

/*
 * 'convert' a PGresult to struct shadow
 */
enum nss_status res2shadow(PGresult *res, struct spwd *result,
                           char *buffer, size_t buflen)
{
	enum nss_status status = NSS_STATUS_NOTFOUND;

	if(!PQntuples(res)) {
		goto BAIL_OUT;
	}

	// Carefully copy attributes into buffer.  Return NSS_STATUS_TRYAGAIN if not enough room.
	status = copy_attrval (res, 0, &result->sp_namp, &buffer, &buflen);
	if(status != NSS_STATUS_SUCCESS) goto BAIL_OUT;

	status = copy_attrval (res, 1, &result->sp_pwdp, &buffer, &buflen);
	if(status != NSS_STATUS_SUCCESS) goto BAIL_OUT;

	// Can be less careful with int variables
	result->sp_lstchg = (long int) atol(PQgetvalue(res, 0, 2));
	result->sp_min = (long int) atol(PQgetvalue(res, 0, 3));
	result->sp_max = (long int) atol(PQgetvalue(res, 0, 4));
	result->sp_warn = (long int) atol(PQgetvalue(res, 0, 5));
	result->sp_inact = (long int) atol(PQgetvalue(res, 0, 6));
	result->sp_expire = (long int) atol(PQgetvalue(res, 0, 7));
	result->sp_flag = (unsigned long int) atol(PQgetvalue(res, 0, 8));

#ifdef COMMENTED_OUT
	print_msg("Converted a res to a pwd:\n");
	print_msg("Name: %s\n", result->sp_namp);
	print_msg("Password: %s\n", result->sp_pwdp);
	print_msg("lastchange: %d\n", result->sp_lstchg);
	print_msg("min: %d\n", result->sp_min);
	print_msg("max: %d\n", result->sp_max);
	print_msg("warn: %d\n", result->sp_warn);
	print_msg("inact: %d\n", result->sp_inact);
	print_msg("expire: %d\n", result->sp_expire);
	print_msg("flag: %d\n", result->sp_flag);
#endif

BAIL_OUT:
	return status;
}

/*
 * get a passwd entry from cursor
 */
enum nss_status backend_getspent(struct spwd *result, char *buffer,
                                 size_t buflen, int *errnop)
{
	PGresult *res;
	enum nss_status status = NSS_STATUS_NOTFOUND;

	res = fetch("shadow");
	if(res) {
		status = res2shadow(res, result, buffer, buflen);
		PQclear(res);
	}
	return status;
}    

/*
 * backend for getspnam()
 */
enum nss_status backend_getspnam(const char *name, struct spwd *result,
                                 char *buffer, size_t buflen, int *errnop)
{
	char *stmt, *ename;
	PGresult *res;
	size_t len;
	enum nss_status status = NSS_STATUS_NOTFOUND;

	len = strlen(name);
	ename = malloc(2*len+1);
	PQescapeString(ename, name, len);
	asprintf(&stmt, getcfg("shadowbyname"), ename);

	res = PQexec(_conn, stmt);
	if(res) {
		status = res2shadow(res, result, buffer, buflen);
		PQclear(res);
	}
	free(stmt);
	free(ename);
    
	return status;
}

