/**
 * $Id: config.c,v 1.2 2004/12/01 06:29:47 mr-russ Exp $
 *
 * configfile parser
 *
 * Copyright (c) 2001 by Joerg Wendland, Bret Mogilefsky
 * see included file COPYING for details
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "nss-pgsql.h"

#define HASHMAX 73
#define CFGLINEMAX 512

static char *_options[HASHMAX];
static unsigned int _confisopen = 0;

unsigned int texthash(const char *str);


/*
 * create a simple hash from a string
 */
unsigned int texthash(const char *str)
{
	int i, s;

	for(i = s = 0; str[i]; i++) {
		s += str[i];
	}

	return s % HASHMAX;
}

/*
 * read configfile and save values in hashtable
 */
int readconfig(char* configfile)
{
	FILE *cf;
	char line[CFGLINEMAX], key[CFGLINEMAX], val[CFGLINEMAX], *c;
	unsigned int h;
	unsigned int lineno = 0;

	if(_confisopen)
		return 1;
	
	if(!(cf = fopen(configfile, "r"))) {
		print_msg("could not open config file " CFGFILE "\n");
		return 0;
	}
	h = 0;
	while(h < HASHMAX) {
		_options[h] = NULL;
		++h;
	}

	while(fgets(line, CFGLINEMAX, cf)) {
		lineno++;

		/* remove comments */
		c = strstr(line, "#");
		if(c) {
			line[c-line] = 0;
		}

		if (*line == 0 || *line == '\n')
			continue;

		/* read options */
		if(sscanf(line, " %s = %[^\n]", key, val) < 2) {
			print_err("line %d in " CFGFILE " is unparseable: \"%s\"\n", lineno, line);
		} else {
			h = texthash(key);
			if (_options[h] != NULL ) {
				print_err("line %d in " CFGFILE " is a duplicate hash: \"%s\"\n", lineno, key);
			} else {
				_options[h] = malloc(strlen(val)+1);
				strcpy(_options[h], val);
			}
		}
	}
	fclose(cf);

	_confisopen = 1;
	atexit(cleanup);

	return 1;
}

/*
 * free the hashmap, close connection to db if open
 */
void cleanup(void)
{
	int i;

	if(_confisopen) {
		for(i = 0; i < HASHMAX; i++) {
			free(_options[i]);
		}
	}
	_confisopen = 0;

	while(backend_isopen('x')) {
		backend_close(NULL);
	}
}


/*
 * get value for 'key' from the hashmap
 */
inline char *getcfg(const char *key)
{
	return _options[texthash(key)] ? _options[texthash(key)] : "";
}
