/******************************** -*- C -*- ****************************
 *
 *	Byte code interpreter primitives include file
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988,89,90,91,92,94,95,99,2000,2001,2002,2003
 * Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later
 * version.
 *
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 ***********************************************************************/

#ifdef OPTIMIZE
#define RECEIVER_IS_INT(x)		(true)
#define RECEIVER_IS_CLASS(x, y)		(true)
#define RECEIVER_IS_A_KIND_OF(x, y)		(true)
#define PRIM_CHECKS_RECEIVER		PRIM_SUCCEED
#else
#define RECEIVER_IS_INT(x)		IS_INT((x))
#define RECEIVER_IS_CLASS(x, y)		IS_CLASS((x), (y))
#define RECEIVER_IS_A_KIND_OF(x, y)		is_a_kind_of((x), (y))
#define PRIM_CHECKS_RECEIVER		(PRIM_SUCCEED | PRIM_FAIL)
#endif

#ifdef HAVE_GMP
#define PRIM_USES_GMP                   (PRIM_SUCCEED | PRIM_FAIL)
#else
#define PRIM_USES_GMP                   PRIM_FAIL
#endif

#if SIZEOF_OFF_T == 4
#define FROM_OFF_T(integer)    from_c_int_32(integer)
#define TO_OFF_T(integer)      to_c_int_32(integer)
#define IS_OFF_T(oop)          is_c_int_32(oop)
#else
#define FROM_OFF_T(integer)    from_c_int_64(integer)
#define TO_OFF_T(integer)      to_c_int_64(integer)
#define IS_OFF_T(oop)          is_c_int_64(oop)
#endif

#ifdef USE_JIT_TRANSLATION
#define PRIM_FAILED			return ((long) -1)
#define PRIM_SUCCEEDED			return ((long) 0)
#define PRIM_SUCCEEDED_RELOAD_IP	return ((long) native_ip)
#else
#define PRIM_FAILED			return (true)
#define PRIM_SUCCEEDED			return (false)
#define PRIM_SUCCEEDED_RELOAD_IP	return (false)
#endif

#define INT_BIN_OP(op, noOverflow) {            \
    OOP	oop1;					\
    OOP	oop2;					\
    oop2 = POP_OOP();				\
    oop1 = POP_OOP();				\
    if (RECEIVER_IS_INT(oop1) && IS_INT(oop2)) {\
      long iarg1, iarg2;			\
      iarg1 = TO_INT(oop1);			\
      iarg2 = TO_INT(oop2);			\
						\
      op;                                       \
      if(noOverflow || !INT_OVERFLOW(iarg1)) {	\
        PUSH_INT(iarg1);			\
        PRIM_SUCCEEDED;				\
      }                                         \
    }						\
    UNPOP(2);					\
    PRIM_FAILED;                                \
  }

#define BOOL_BIN_OP(operator) {					\
    OOP	oop1;							\
    OOP	oop2;							\
    oop2 = POP_OOP();						\
    oop1 = POP_OOP();						\
    if (RECEIVER_IS_INT(oop1) && IS_INT(oop2)) {		\
      PUSH_BOOLEAN( ((long)oop1) operator ((long)oop2) );	\
      PRIM_SUCCEEDED;						\
    }								\
    UNPOP(2);							\
    PRIM_FAILED;                                                \
  }


/* SmallInteger + arg  */
static long
VMpr_SmallInteger_plus(int id,
                       volatile int numArgs,
		       OOP methodOOP)
{
  _gst_primitives_executed++;
  INT_BIN_OP (iarg1 += iarg2, false);
}

/* SmallInteger - arg */
static long
VMpr_SmallInteger_minus(int id,
			volatile int numArgs,
			OOP methodOOP)
{
  _gst_primitives_executed++;
  INT_BIN_OP (iarg1 -= iarg2, false);
}

/* SmallInteger < arg */
static long
VMpr_SmallInteger_lt(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  _gst_primitives_executed++;
  BOOL_BIN_OP (<);
}

/* SmallInteger > arg */
static long
VMpr_SmallInteger_gt(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  _gst_primitives_executed++;
  BOOL_BIN_OP (>);
}

/* SmallInteger <= arg */
static long
VMpr_SmallInteger_le(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  _gst_primitives_executed++;
  BOOL_BIN_OP (<=);
}

/* SmallInteger >= arg */
static long
VMpr_SmallInteger_ge(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  _gst_primitives_executed++;
  BOOL_BIN_OP (>=);
}

/* SmallInteger =, == arg */
static long
VMpr_SmallInteger_eq(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  _gst_primitives_executed++;
  BOOL_BIN_OP ( ==);
}

/* SmallInteger ~=, ~~ arg */
static long
VMpr_SmallInteger_ne(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  _gst_primitives_executed++;
  BOOL_BIN_OP (!=);
}

/* SmallInteger * arg */
static long
VMpr_SmallInteger_times(int id,
			volatile int numArgs,
			OOP methodOOP)
{
  _gst_primitives_executed++;
  INT_BIN_OP (iarg1 = mul_with_check (iarg1, iarg2), false);
}

/* SmallInteger / arg */
/* quotient as exact as possible */
static long
VMpr_SmallInteger_divide(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (RECEIVER_IS_INT (oop1) && IS_INT (oop2) && oop2 != FROM_INT (0))
    {
      long iarg1, iarg2, result;
      iarg1 = TO_INT (oop1);
      iarg2 = TO_INT (oop2);
      result = iarg1 / iarg2;
      iarg2 *= result;
      if (iarg1 == iarg2)
	{
	  PUSH_INT (result);
	  PRIM_SUCCEEDED;
	}
    }
  UNPOP (2);
  PRIM_FAILED;
}

/* SmallInteger \\ arg */
/* remainder truncated towards -infinity */
static long
VMpr_SmallInteger_modulo(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (RECEIVER_IS_INT (oop1) && IS_INT (oop2) && oop2 != FROM_INT (0))
    {
      long iarg1, iarg2, result;
      iarg1 = TO_INT (oop1);
      iarg2 = TO_INT (oop2);
      result = iarg1 % iarg2;
      PUSH_INT (result
		&& ((result ^ iarg2) < 0) ? result + iarg2 : result);
      PRIM_SUCCEEDED;
    }
  UNPOP (2);
  PRIM_FAILED;
}

/* SmallInteger // arg */
/* quotient truncated towards -infinity */
static long
VMpr_SmallInteger_intDiv(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (RECEIVER_IS_INT (oop1) && IS_INT (oop2) && oop2 != FROM_INT (0))
    {
      long iarg1, iarg2;
      iarg1 = TO_INT (oop1);
      iarg2 = TO_INT (oop2);
      if (iarg2 < 0)
	{			/* make the divisor positive */
	  iarg1 = -iarg1;
	  iarg2 = -iarg2;
	}
      if (iarg1 < 0)		/* differing signs => use black magic */
	PUSH_INT (-((iarg2 - 1 - iarg1) / iarg2));
      else
	PUSH_INT (iarg1 / iarg2);

      PRIM_SUCCEEDED;
    }
  UNPOP (2);
  PRIM_FAILED;
}

/* SmallInteger quo: arg */
/* quotient truncated towards 0 */
static long
VMpr_SmallInteger_quo(int id,
		      volatile int numArgs,
		      OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (RECEIVER_IS_INT (oop1) && IS_INT (oop2))
    {
      long iarg1, iarg2;
      iarg1 = TO_INT (oop1);
      iarg2 = TO_INT (oop2);
      if (iarg2 != 0)
	{
	  PUSH_INT (iarg1 / iarg2);
	  PRIM_SUCCEEDED;
	}
    }
  UNPOP (2);
  PRIM_FAILED;
}

/* SmallInteger bitAnd: arg */
static long
VMpr_SmallInteger_bitAnd(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  _gst_primitives_executed++;
  INT_BIN_OP (iarg1 &= iarg2, true);
}

/* SmallInteger bitOr: arg */
static long
VMpr_SmallInteger_bitOr(int id,
			volatile int numArgs,
			OOP methodOOP)
{
  _gst_primitives_executed++;
  INT_BIN_OP (iarg1 |= iarg2, true);
}

/* SmallInteger bitXor: arg */
static long
VMpr_SmallInteger_bitXor(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  _gst_primitives_executed++;
  INT_BIN_OP (iarg1 ^= iarg2, true);
}

/* SmallInteger bitShift: arg */
static long
VMpr_SmallInteger_bitShift(int id,
			   volatile int numArgs,
			   OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (RECEIVER_IS_INT (oop1) && IS_INT (oop2))
    {
      long iarg1;
      long iarg2;
      iarg1 = TO_INT (oop1);
      iarg2 = TO_INT (oop2);
      if (iarg2 < 0)
	{
	  PUSH_INT (iarg1 >> -iarg2);
	  PRIM_SUCCEEDED;
	}
      if (iarg2 < ST_INT_SIZE)
	{
	  long result = iarg1 << iarg2;
	  if ((result >> iarg2) == iarg1 && !INT_OVERFLOW(result))
	    {
	      PUSH_INT (result);
	      PRIM_SUCCEEDED;
	    }
	}
    }
  UNPOP (2);
  PRIM_FAILED;
}

/* SmallInteger scramble */

static long
VMpr_SmallInteger_scramble(int id,
			   volatile int numArgs,
			   OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_INT (oop1))
    {
      PUSH_INT (scramble (TO_INT (oop1)));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}


/* SmallInteger asFloatD */

static long
VMpr_SmallInteger_asFloatD(int id,
			  volatile int numArgs,
			  OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_INT (oop1))
    {
      PUSH_OOP (floatd_new ((double) TO_INT (oop1)));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* SmallInteger asFloatE */

static long
VMpr_SmallInteger_asFloatE(int id,
			  volatile int numArgs,
			  OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_INT (oop1))
    {
      PUSH_OOP (floate_new ((float) TO_INT (oop1)));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* SmallInteger asFloatQ */

static long
VMpr_SmallInteger_asFloatQ(int id,
			  volatile int numArgs,
			  OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_INT (oop1))
    {
      PUSH_OOP (floatq_new ((long double) TO_INT (oop1)));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

static long
VMpr_LargeInteger_eq(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  OOP oop1, oop2;
  int result;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  result = _gst_mpz_cmp (&a, &b);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&b);
  PUSH_BOOLEAN (result == 0);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}

static long
VMpr_LargeInteger_ne(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  OOP oop1, oop2;
  int result;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  result = _gst_mpz_cmp (&a, &b);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&b);
  PUSH_BOOLEAN (result != 0);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_lt(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  OOP oop1, oop2;
  int result;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  result = _gst_mpz_cmp (&a, &b);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&b);
  PUSH_BOOLEAN (result < 0);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_le(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  OOP oop1, oop2;
  int result;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  result = _gst_mpz_cmp (&a, &b);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&b);
  PUSH_BOOLEAN (result <= 0);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_gt(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  OOP oop1, oop2;
  int result;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  result = _gst_mpz_cmp (&a, &b);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&b);
  PUSH_BOOLEAN (result > 0);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_ge(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  OOP oop1, oop2;
  int result;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  result = _gst_mpz_cmp (&a, &b);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&b);
  PUSH_BOOLEAN (result >= 0);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_times(int id,
			volatile int numArgs,
			OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  gst_mpz c = { 0, 0, NULL };
  OOP oop1, oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  _gst_mpz_mul (&c, &a, &b);
  oop3 = _gst_oop_from_mpz (&c);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&b);
  _gst_mpz_clear (&c);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}



static long
VMpr_LargeInteger_intDiv(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  gst_mpz q = { 0, 0, NULL };
  gst_mpz r = { 0, 0, NULL };
  OOP oop1, oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2) && oop2 != FROM_INT (0))
    _gst_mpz_fdiv_qr_si (&q, &a, TO_INT(oop2));
  else if (IS_OOP (oop2)
	   && SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class)
    {
      _gst_mpz_from_oop (&b, oop2);
      _gst_mpz_fdiv_qr (&q, &r, &a, &b);
      _gst_mpz_clear (&r);
      _gst_mpz_clear (&b);
    }
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  oop3 = _gst_oop_from_mpz (&q);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&q);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_modulo(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  gst_mpz q = { 0, 0, NULL };
  gst_mpz r = { 0, 0, NULL };
  OOP oop1, oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2) && oop2 != FROM_INT (0))
    {
      long rem = _gst_mpz_fdiv_qr_si (&q, &a, TO_INT(oop2));
      oop3 = FROM_INT (rem);
    }
  else if (IS_OOP (oop2)
	   && SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class)
    {
      _gst_mpz_from_oop (&b, oop2);
      _gst_mpz_fdiv_qr (&q, &r, &a, &b);
      oop3 = _gst_oop_from_mpz (&r);
      _gst_mpz_clear (&r);
      _gst_mpz_clear (&b);
    }
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  _gst_mpz_clear (&a);
  _gst_mpz_clear (&q);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_quo(int id,
		      volatile int numArgs,
		      OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  gst_mpz q = { 0, 0, NULL };
  gst_mpz r = { 0, 0, NULL };
  OOP oop1, oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2) && oop2 != FROM_INT (0))
    _gst_mpz_tdiv_qr_si (&q, &a, TO_INT(oop2));
  else if (IS_OOP (oop2)
	   && SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class)
    {
      _gst_mpz_from_oop (&b, oop2);
      _gst_mpz_tdiv_qr (&q, &r, &a, &b);
      _gst_mpz_clear (&r);
      _gst_mpz_clear (&b);
    }
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  oop3 = _gst_oop_from_mpz (&q);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&q);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_rem(int id,
		      volatile int numArgs,
		      OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  gst_mpz q = { 0, 0, NULL };
  gst_mpz r = { 0, 0, NULL };
  OOP oop1, oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2) && oop2 != FROM_INT (0))
    {
      long rem = _gst_mpz_tdiv_qr_si (&q, &a, TO_INT(oop2));
      oop3 = FROM_INT (rem);
    }
  else if (IS_OOP (oop2)
	   && SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class)
    {
      _gst_mpz_from_oop (&b, oop2);
      _gst_mpz_tdiv_qr (&q, &r, &a, &b);
      oop3 = _gst_oop_from_mpz (&r);
      _gst_mpz_clear (&r);
      _gst_mpz_clear (&b);
    }
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  _gst_mpz_clear (&a);
  _gst_mpz_clear (&q);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_negated(int id,
			  volatile int numArgs,
			  OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  gst_mpz c = { 0, 0, NULL };
  OOP oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  _gst_mpz_sub (&c, &a, &b);
  oop3 = _gst_oop_from_mpz (&c);
  _gst_mpz_clear (&b);
  _gst_mpz_clear (&c);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_bitAnd(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  gst_mpz c = { 0, 0, NULL };
  OOP oop1, oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  _gst_mpz_and (&c, &a, &b);
  oop3 = _gst_oop_from_mpz (&c);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&b);
  _gst_mpz_clear (&c);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_bitOr(int id,
			volatile int numArgs,
			OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  gst_mpz c = { 0, 0, NULL };
  OOP oop1, oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  _gst_mpz_ior (&c, &a, &b);
  oop3 = _gst_oop_from_mpz (&c);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&b);
  _gst_mpz_clear (&c);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_bitXor(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  gst_mpz c = { 0, 0, NULL };
  OOP oop1, oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  _gst_mpz_xor (&c, &a, &b);
  oop3 = _gst_oop_from_mpz (&c);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&b);
  _gst_mpz_clear (&c);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_bitInvert(int id,
			    volatile int numArgs,
			    OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz c = { 0, 0, NULL };
  OOP oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop2);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  _gst_mpz_com (&c, &a);
  oop3 = _gst_oop_from_mpz (&c);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&c);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_bitShift(int id,
			   volatile int numArgs,
			   OOP methodOOP)
{
#ifdef HAVE_GMP
  int n;
  gst_mpz a = { 0, 0, NULL };
  gst_mpz c = { 0, 0, NULL };
  OOP oop1, oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop2))
    n = TO_INT (oop2);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (n > 0)
    _gst_mpz_mul_2exp (&c, &a, n);
  else
    _gst_mpz_div_2exp (&c, &a, -n);

  oop3 = _gst_oop_from_mpz (&c);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&c);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#endif
  PRIM_FAILED;
}


static long
VMpr_LargeInteger_plus(int id,
		       volatile int numArgs,
		       OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  gst_mpz c = { 0, 0, NULL };
  OOP oop1, oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  _gst_mpz_add (&c, &a, &b);
  oop3 = _gst_oop_from_mpz (&c);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&b);
  _gst_mpz_clear (&c);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_minus(int id,
			volatile int numArgs,
			OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  gst_mpz c = { 0, 0, NULL };
  OOP oop1, oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  _gst_mpz_sub (&c, &a, &b);
  oop3 = _gst_oop_from_mpz (&c);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&b);
  _gst_mpz_clear (&c);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_gcd(int id,
		      volatile int numArgs,
		      OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  gst_mpz b = { 0, 0, NULL };
  gst_mpz c = { 0, 0, NULL };
  OOP oop1, oop2, oop3;

  _gst_primitives_executed++;
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (IS_INT (oop2)
      || SUPERCLASS (OOP_CLASS (oop2)) == _gst_large_integer_class
      || OOP_CLASS (oop2) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&b, oop2);
  else
    {
      _gst_mpz_clear (&a);
      UNPOP (2);
      PRIM_FAILED;
    }

  if (oop1 == FROM_INT(0) || oop2 == FROM_INT(0)
      || (IS_OOP(oop1) && OOP_CLASS (oop1) == _gst_large_zero_integer_class)
      || (IS_OOP(oop2) && OOP_CLASS (oop2) == _gst_large_zero_integer_class))
    /* Return the non-zero number between a and b */
    _gst_mpz_add (&c, &a, &b);
  else
    _gst_mpz_gcd (&c, &a, &b);

  oop3 = _gst_oop_from_mpz (&c);
  _gst_mpz_clear (&a);
  _gst_mpz_clear (&b);
  _gst_mpz_clear (&c);
  PUSH_OOP (oop3);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_asFloatD(int id,
			  volatile int numArgs,
			  OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  OOP oop1, oop2;
  double d;

  _gst_primitives_executed++;
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  d = _gst_mpz_get_d (&a);
  oop2 = floatd_new (d);
  _gst_mpz_clear (&a);
  PUSH_OOP (oop2);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_asFloatE(int id,
			  volatile int numArgs,
			  OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  OOP oop1, oop2;
  double d;

  _gst_primitives_executed++;
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  d = _gst_mpz_get_d (&a);
  oop2 = floate_new (d);
  _gst_mpz_clear (&a);
  PUSH_OOP (oop2);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_LargeInteger_asFloatQ(int id,
			  volatile int numArgs,
			  OOP methodOOP)
{
#ifdef HAVE_GMP
  gst_mpz a = { 0, 0, NULL };
  OOP oop1, oop2;
  double d;

  _gst_primitives_executed++;
  oop1 = POP_OOP ();
  if (IS_INT (oop1)
      || SUPERCLASS (OOP_CLASS (oop1)) == _gst_large_integer_class
      || OOP_CLASS (oop1) == _gst_large_zero_integer_class)
    _gst_mpz_from_oop (&a, oop1);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  d = _gst_mpz_get_ld (&a);
  oop2 = floatq_new (d);
  _gst_mpz_clear (&a);
  PUSH_OOP (oop2);
  PRIM_SUCCEEDED;
#else
  PRIM_FAILED;
#endif
}


static long
VMpr_FloatD_arith(int id,
		 volatile int numArgs,
		 OOP methodOOP)
{
  double farg2;
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_CLASS (oop2, _gst_floatd_class))
    farg2 = FLOATD_OOP_VALUE (oop2);
  else if (IS_INT (oop2))
    farg2 = (double) TO_INT (oop2);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (RECEIVER_IS_CLASS (oop1, _gst_floatd_class))
    {
      double farg1;
      farg1 = FLOATD_OOP_VALUE (oop1);
      switch (id)
	{
	case 41:
	  PUSH_OOP (floatd_new (farg1 + farg2));
	  break;
	case 42:
	  PUSH_OOP (floatd_new (farg1 - farg2));
	  break;
	case 43:
	  PUSH_BOOLEAN (farg1 < farg2);
	  break;
	case 44:
	  PUSH_BOOLEAN (farg1 > farg2);
	  break;
	case 45:
	  PUSH_BOOLEAN (farg1 <= farg2);
	  break;
	case 46:
	  PUSH_BOOLEAN (farg1 >= farg2);
	  break;
	case 47:
	  PUSH_BOOLEAN (farg1 == farg2);
	  break;
	case 48:
	  PUSH_BOOLEAN (farg1 != farg2);
	  break;
	case 49:
	  PUSH_OOP (floatd_new (farg1 * farg2));
	  break;
	case 50:
	  PUSH_OOP (floatd_new (farg1 / farg2));
	  break;
	}
      PRIM_SUCCEEDED;
    }

  UNPOP (2);
  PRIM_FAILED;
}

/* FloatD truncated */
static long
VMpr_FloatD_truncated(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floatd_class))
    {
      double oopValue = FLOATD_OOP_VALUE (oop1);
      if COMMON (oopValue >= 0.0 && oopValue <= MAX_ST_INT)
	{
	  PUSH_INT ((long) (oopValue + 0.000000000000005));
	  PRIM_SUCCEEDED;
	}
      else if COMMON (oopValue < 0.0 && oopValue >= MIN_ST_INT)
	{
	  PUSH_INT ((long) (oopValue - 0.000000000000005));
	  PRIM_SUCCEEDED;
	}
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* FloatD fractionPart */
static long
VMpr_FloatD_fractionPart(int id,
			volatile int numArgs,
			OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floatd_class))
    {
      double farg1;
      farg1 = FLOATD_OOP_VALUE (oop1);
      farg1 -= (farg1 < 0.0) ? ceil (farg1) : floor (farg1);

      PUSH_OOP (floatd_new (farg1));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* FloatD exponent */
static long
VMpr_FloatD_exponent(int id,
		    volatile int numArgs,
		    OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floatd_class))
    {
      double farg1;
      int intArg1;

      farg1 = FLOATD_OOP_VALUE (oop1);
      if (farg1 == 0.0)
	intArg1 = 1;
      else
	{
	  frexp (FLOATD_OOP_VALUE (oop1), &intArg1);
	  intArg1--;
	}
      PUSH_INT (intArg1);
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* FloatD timesTwoPower: */
static long
VMpr_FloatD_timesTwoPower(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floatd_class) && IS_INT (oop2))
    {
      double farg1;
      long iarg2;
      farg1 = FLOATD_OOP_VALUE (oop1);
      iarg2 = TO_INT (oop2);
      PUSH_OOP (floatd_new (ldexp (farg1, iarg2)));
      PRIM_SUCCEEDED;
    }
  UNPOP (2);
  PRIM_FAILED;
}

/* FloatD asFloatE */

static long
VMpr_FloatD_asFloatE(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floatd_class))
    {
      PUSH_OOP (floate_new (FLOATD_OOP_VALUE (oop1)));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* FloatD asFloatQ */

static long
VMpr_FloatD_asFloatQ(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floatd_class))
    {
      PUSH_OOP (floatq_new (FLOATD_OOP_VALUE (oop1)));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

static long
VMpr_FloatE_arith(int id,
		 volatile int numArgs,
		 OOP methodOOP)
{
  double farg2;
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_CLASS (oop2, _gst_floate_class))
    farg2 = FLOATE_OOP_VALUE (oop2);
  else if (IS_INT (oop2))
    farg2 = (double) TO_INT (oop2);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (RECEIVER_IS_CLASS (oop1, _gst_floate_class))
    {
      double farg1;
      farg1 = FLOATE_OOP_VALUE (oop1);
      switch (id)
	{
	case 41:
	  PUSH_OOP (floate_new (farg1 + farg2));
	  break;
	case 42:
	  PUSH_OOP (floate_new (farg1 - farg2));
	  break;
	case 43:
	  PUSH_BOOLEAN (farg1 < farg2);
	  break;
	case 44:
	  PUSH_BOOLEAN (farg1 > farg2);
	  break;
	case 45:
	  PUSH_BOOLEAN (farg1 <= farg2);
	  break;
	case 46:
	  PUSH_BOOLEAN (farg1 >= farg2);
	  break;
	case 47:
	  PUSH_BOOLEAN (farg1 == farg2);
	  break;
	case 48:
	  PUSH_BOOLEAN (farg1 != farg2);
	  break;
	case 49:
	  PUSH_OOP (floate_new (farg1 * farg2));
	  break;
	case 50:
	  PUSH_OOP (floate_new (farg1 / farg2));
	  break;
	}
      PRIM_SUCCEEDED;
    }

  UNPOP (2);
  PRIM_FAILED;
}

/* FloatD truncated */
static long
VMpr_FloatE_truncated(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floate_class))
    {
      double oopValue = FLOATE_OOP_VALUE (oop1);
      if COMMON (oopValue >= 0.0 && oopValue <= MAX_ST_INT)
	{
	  PUSH_INT ((long) (oopValue + 0.000000000000005));
	  PRIM_SUCCEEDED;
	}
      else if COMMON (oopValue < 0.0 && oopValue >= MIN_ST_INT)
	{
	  PUSH_INT ((long) (oopValue - 0.000000000000005));
	  PRIM_SUCCEEDED;
	}
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* FloatD fractionPart */
static long
VMpr_FloatE_fractionPart(int id,
			volatile int numArgs,
			OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floate_class))
    {
      double farg1;
      farg1 = FLOATE_OOP_VALUE (oop1);
      farg1 -= (farg1 < 0.0) ? ceil (farg1) : floor (farg1);

      PUSH_OOP (floate_new (farg1));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* FloatD exponent */
static long
VMpr_FloatE_exponent(int id,
		    volatile int numArgs,
		    OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floate_class))
    {
      double farg1;
      int intArg1;

      farg1 = FLOATE_OOP_VALUE (oop1);
      if (farg1 == 0.0)
	intArg1 = 1;
      else
	{
	  frexp (FLOATE_OOP_VALUE (oop1), &intArg1);
	  intArg1--;
	}
      PUSH_INT (intArg1);
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* FloatD timesTwoPower: */
static long
VMpr_FloatE_timesTwoPower(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floate_class) && IS_INT (oop2))
    {
      double farg1;
      long iarg2;
      farg1 = FLOATE_OOP_VALUE (oop1);
      iarg2 = TO_INT (oop2);
      PUSH_OOP (floate_new (ldexp (farg1, iarg2)));
      PRIM_SUCCEEDED;
    }
  UNPOP (2);
  PRIM_FAILED;
}

/* FloatE asFloatD */

static long
VMpr_FloatE_asFloatD(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floate_class))
    {
      PUSH_OOP (floatd_new (FLOATE_OOP_VALUE (oop1)));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* FloatD asFloatQ */

static long
VMpr_FloatE_asFloatQ(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floate_class))
    {
      PUSH_OOP (floatq_new (FLOATE_OOP_VALUE (oop1)));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

static long
VMpr_FloatQ_arith(int id,
		 volatile int numArgs,
		 OOP methodOOP)
{
  long double farg2;
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_CLASS (oop2, _gst_floatq_class))
    farg2 = FLOATQ_OOP_VALUE (oop2);
  else if (IS_INT (oop2))
    farg2 = (long double) TO_INT (oop2);
  else
    {
      UNPOP (2);
      PRIM_FAILED;
    }

  if (RECEIVER_IS_CLASS (oop1, _gst_floatq_class))
    {
      long double farg1;
      farg1 = FLOATQ_OOP_VALUE (oop1);
      switch (id)
	{
	case 41:
	  PUSH_OOP (floatq_new (farg1 + farg2));
	  break;
	case 42:
	  PUSH_OOP (floatq_new (farg1 - farg2));
	  break;
	case 43:
	  PUSH_BOOLEAN (farg1 < farg2);
	  break;
	case 44:
	  PUSH_BOOLEAN (farg1 > farg2);
	  break;
	case 45:
	  PUSH_BOOLEAN (farg1 <= farg2);
	  break;
	case 46:
	  PUSH_BOOLEAN (farg1 >= farg2);
	  break;
	case 47:
	  PUSH_BOOLEAN (farg1 == farg2);
	  break;
	case 48:
	  PUSH_BOOLEAN (farg1 != farg2);
	  break;
	case 49:
	  PUSH_OOP (floatq_new (farg1 * farg2));
	  break;
	case 50:
	  PUSH_OOP (floatq_new (farg1 / farg2));
	  break;
	}
      PRIM_SUCCEEDED;
    }

  UNPOP (2);
  PRIM_FAILED;
}

/* FloatD truncated */
static long
VMpr_FloatQ_truncated(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floatq_class))
    {
      long double oopValue = FLOATQ_OOP_VALUE (oop1);
      if COMMON (oopValue >= 0.0 && oopValue <= MAX_ST_INT)
	{
	  PUSH_INT ((long) (oopValue + 0.000000000000005));
	  PRIM_SUCCEEDED;
	}
      else if COMMON (oopValue < 0.0 && oopValue >= MIN_ST_INT)
	{
	  PUSH_INT ((long) (oopValue - 0.000000000000005));
	  PRIM_SUCCEEDED;
	}
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* FloatD fractionPart */
static long
VMpr_FloatQ_fractionPart(int id,
			volatile int numArgs,
			OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floatq_class))
    {
      long double farg1;
      farg1 = FLOATQ_OOP_VALUE (oop1);
      farg1 -= (farg1 < 0.0) ? ceill (farg1) : floorl (farg1);

      PUSH_OOP (floatq_new (farg1));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* FloatD exponent */
static long
VMpr_FloatQ_exponent(int id,
		    volatile int numArgs,
		    OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floatq_class))
    {
      long double farg1;
      int intArg1;

      farg1 = FLOATQ_OOP_VALUE (oop1);
      if (farg1 == 0.0)
	intArg1 = 1;
      else
	{
	  frexpl (FLOATQ_OOP_VALUE (oop1), &intArg1);
	  intArg1--;
	}
      PUSH_INT (intArg1);
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* FloatD timesTwoPower: */
static long
VMpr_FloatQ_timesTwoPower(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floatq_class) && IS_INT (oop2))
    {
      long double farg1;
      long iarg2;
      farg1 = FLOATQ_OOP_VALUE (oop1);
      iarg2 = TO_INT (oop2);
      PUSH_OOP (floatq_new (ldexpl (farg1, iarg2)));
      PRIM_SUCCEEDED;
    }
  UNPOP (2);
  PRIM_FAILED;
}

/* FloatQ asFloatD */

static long
VMpr_FloatQ_asFloatD(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floatq_class))
    {
      PUSH_OOP (floatd_new (FLOATQ_OOP_VALUE (oop1)));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* FloatD asFloatQ */

static long
VMpr_FloatQ_asFloatE(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (RECEIVER_IS_CLASS (oop1, _gst_floatq_class))
    {
      PUSH_OOP (floate_new (FLOATQ_OOP_VALUE (oop1)));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}



/* Object at:, Object basicAt: */

static long
VMpr_Object_basicAt(int id,
		    volatile int numArgs,
		    OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_INT (oop2))
    {
      long arg2;
      arg2 = TO_INT (oop2);
      oop1 = index_oop (oop1, arg2);
      if (oop1)
	{
	  SET_STACKTOP (oop1);
	  PRIM_SUCCEEDED;
	}
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* Object at:put:, Object basicAt:put: */
static long
VMpr_Object_basicAtPut(int id,
		       volatile int numArgs,
		       OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  OOP oop3;
  _gst_primitives_executed++;

  oop3 = POP_OOP ();
  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_INT (oop2) && !IS_OOP_READONLY (oop1))
    {
      long arg2;
      arg2 = TO_INT (oop2);
      if (index_oop_put (oop1, arg2, oop3))
	{
	  SET_STACKTOP (oop3);
	  PRIM_SUCCEEDED;
	}
    }

  UNPOP (2);
  PRIM_FAILED;
}

/* Object basicSize; Object size; String size */
static long
VMpr_Object_basicSize(int id,
		      volatile int numArgs,
		      OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  PUSH_INT (NUM_INDEXABLE_FIELDS (oop1));
  PRIM_SUCCEEDED;
}

/* String at:; gst_string basicAt: */
static long
VMpr_String_basicAt(int id,
		    volatile int numArgs,
		    OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_INT (oop2))
    {
      long arg2;
      arg2 = TO_INT (oop2);
      oop1 = index_string_oop (oop1, arg2);
      if (oop1)
	{
	  SET_STACKTOP (oop1);
	  PRIM_SUCCEEDED;
	}
    }

  UNPOP (1);
  PRIM_FAILED;
}

/* String basicAt:put:; gst_string at:put: */
static long
VMpr_String_basicAtPut(int id,
		       volatile int numArgs,
		       OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  OOP oop3;
  _gst_primitives_executed++;

  oop3 = POP_OOP ();
  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_INT (oop2) && IS_CLASS (oop3, _gst_char_class)
      && !IS_OOP_READONLY (oop1))
    {
      long arg2;
      arg2 = TO_INT (oop2);
      if (index_string_oop_put (oop1, arg2, oop3))
	{
	  SET_STACKTOP (oop3);
	  PRIM_SUCCEEDED;
	}
    }

  UNPOP (2);
  PRIM_FAILED;
}

/* CompiledBlock numArgs:numTemps:bytecodes:depth:literals: */

static long
VMpr_CompiledBlock_create(int id,
			  volatile int numArgs,
			  OOP methodOOP)
{
  OOP *_gst_literals = OOP_TO_OBJ (POP_OOP ())->data;
  int depth = TO_INT (POP_OOP ());
  OOP bytecodesOOP = POP_OOP ();
  int blockTemps = TO_INT (POP_OOP ());
  int blockArgs = TO_INT (POP_OOP ());
  bytecodes bytecodes = _gst_extract_bytecodes (bytecodesOOP);

  OOP block =
    _gst_block_new (blockArgs, blockTemps, bytecodes, depth, _gst_literals);

  _gst_primitives_executed++;
  SET_STACKTOP (block);
  PRIM_SUCCEEDED;
}

/* CompiledMethod literals:numArgs:numTemps:primitive:bytecodes:depth: */
static long
VMpr_CompiledMethod_create(int id,
			   volatile int numArgs,
			   OOP methodOOP)
{
  int depth = TO_INT (POP_OOP ());
  OOP bytecodesOOP = POP_OOP ();
  int primitive = TO_INT (POP_OOP ());
  int methodTemps = TO_INT (POP_OOP ());
  int methodArgs = TO_INT (POP_OOP ());
  OOP literals = POP_OOP ();
  bytecodes bytecodes = _gst_extract_bytecodes (bytecodesOOP);

  OOP method =
    _gst_make_new_method (primitive, methodArgs, methodTemps, depth,
			  literals,
			  bytecodes, _gst_nil_oop, _gst_nil_oop);

  _gst_primitives_executed++;
  SET_STACKTOP (method);
  PRIM_SUCCEEDED;
}

/* Object shallowCopy */
static long
VMpr_Object_shallowCopy(int id,
		        volatile int numArgs,
		        OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  oop1 = _gst_object_copy (oop1);
  SET_STACKTOP (oop1);
  PRIM_SUCCEEDED;
}

/* Behavior basicNew; Behavior new; */
static long
VMpr_Behavior_basicNew(int id,
		       volatile int numArgs,
		       OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_OOP (oop1))
    {
      if (!CLASS_IS_INDEXABLE (oop1))
	{
	  /* Note: you cannot pass &STACKTOP() because if the stack
	     moves it ain't valid anymore by the time it is set!!! */
	  OOP result;
	  instantiate (oop1, &result);
	  SET_STACKTOP (result);
	  PRIM_SUCCEEDED;
	}
    }
  PRIM_FAILED;
}

/* Behavior new:; Behavior basicNew: */
static long
VMpr_Behavior_basicNewColon(int id,
			    volatile int numArgs,
			    OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_OOP (oop1) && IS_INT (oop2))
    {
      if (CLASS_IS_INDEXABLE (oop1))
	{
	  long arg2;
	  arg2 = TO_INT (oop2);
	  if (arg2 >= 0)
	    {
	      OOP result;
	      instantiate_with (oop1, arg2, &result);
	      SET_STACKTOP (result);
	      PRIM_SUCCEEDED;
	    }
	}
    }

  UNPOP (1);
  PRIM_FAILED;
}

/* Object become: */
static long
VMpr_Object_become(int id,
		   volatile int numArgs,
		   OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_OOP (oop1) && !IS_OOP_READONLY (oop1))
    {
      if (IS_OOP (oop2) && !IS_OOP_READONLY (oop2))
	{
	  _gst_swap_objects (oop1, oop2);
	  /* ??? maybe we should do `_gst_self = oop2' here? */
	  PRIM_SUCCEEDED;
	}
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* Object instVarAt: */
static long
VMpr_Object_instVarAt(int id,
		      volatile int numArgs,
		      OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_INT (oop2))
    {
      long arg2;
      arg2 = TO_INT (oop2);
      if (CHECK_BOUNDS_OF (oop1, arg2))
	{
	  SET_STACKTOP (inst_var_at (oop1, arg2));
	  PRIM_SUCCEEDED;
	}
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* Object inst_var_at:put: */
static long
VMpr_Object_instVarAtPut(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  OOP oop3;
  _gst_primitives_executed++;

  oop3 = POP_OOP ();
  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_INT (oop2))
    {
      long arg2;
      arg2 = TO_INT (oop2);
      if (CHECK_BOUNDS_OF (oop1, arg2))
	{
	  if (inst_var_at_put (oop1, arg2, oop3))
	    PRIM_SUCCEEDED;
	}
    }
  UNPOP (2);
  PRIM_FAILED;
}

/* Object asOop; Object hash; Symbol hash */
static long
VMpr_Object_hash(int id,
		 volatile int numArgs,
		 OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (IS_OOP (oop1))
    {
      PUSH_INT (OOP_INDEX (oop1));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* SmallInteger asObject; SmallInteger asObjectNoFail */
static long
VMpr_SmallInteger_asObject(int id,
			   volatile int numArgs,
			   OOP methodOOP)
{
  OOP oop1;
  long arg1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  arg1 = TO_INT (oop1);
  if (_gst_oop_index_valid (arg1))
    {
      SET_STACKTOP (OOP_AT (arg1));
      PRIM_SUCCEEDED;
    }

  PRIM_FAILED;
}

/* Behavior someInstance */
static long
VMpr_Behavior_someInstance(int id,
			   volatile int numArgs,
			   OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  for (oop2 = _gst_all_oops_table;
       oop2 < &_gst_oop_table[_gst_oop_table_size]; oop2++)
    {
      if (OOP_VALID (oop2) && oop1 == OOP_CLASS (oop2))
	{
	  SET_STACKTOP (oop2);
	  PRIM_SUCCEEDED;
	}
    }
  PRIM_FAILED;
}

/* Object nextInstance */
static long
VMpr_Object_nextInstance(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (OOP_CLASS (oop1) == _gst_char_class)
    {
      /* Characters are one after another - at the end there is
         _gst_nil_oop */
      oop1++;
      if (_gst_char_class == OOP_CLASS (oop1))
	{
	  SET_STACKTOP (oop1);
	  PRIM_SUCCEEDED;
	}
    }
  else if (IS_OOP (oop1) && oop1 >= _gst_oop_table)
    {
      /* There is just one copy of all other builtin objects, so fail
         for a builtin */
      OOP class_oop = OOP_CLASS (oop1);
      for (++oop1; oop1 <= _gst_last_used_oop; oop1++)
	{
	  if (OOP_VALID (oop1) && class_oop == OOP_CLASS (oop1))
	    {
	      SET_STACKTOP (oop1);
	      PRIM_SUCCEEDED;
	    }
	}
    }
  PRIM_FAILED;
}

static long
VMpr_ContextPart_continue(int id,
			  volatile int numArgs,
			  OOP methodOOP)
{
  OOP oop2;
  OOP oop1;

  oop2 = POP_OOP();
  oop1 = STACKTOP();
  if (RECEIVER_IS_A_KIND_OF (OOP_CLASS (oop1), _gst_context_part_class))
    {
      unwind_to (oop1);
      SET_STACKTOP (oop2);
      PRIM_SUCCEEDED_RELOAD_IP;
    }
  else
    {
      UNPOP (1);
      PRIM_FAILED;
    }
}

/* BlockClosure blockCopy: */

static long
VMpr_BlockClosure_blockCopy(int id,
			    volatile int numArgs,
			    OOP methodOOP)
{
  gst_block_closure closure, original;
  OOP closureOOP;
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  closure = (gst_block_closure) new_instance (_gst_block_closure_class,
					      &closureOOP);

  original = (gst_block_closure) OOP_TO_OBJ (oop1);
  closure->outerContext = oop2;
  closure->block = original->block;
  closure->receiver = _gst_self;
  SET_STACKTOP (closureOOP);
  PRIM_SUCCEEDED;
}

/* BlockClosure value
   BlockClosure value:
   BlockClosure value:value:
   BlockClosure value:value:value: */
static long
VMpr_BlockClosure_value(int id,
			volatile int numArgs,
			OOP methodOOP)
{
  _gst_primitives_executed++;
  if (send_block_value (numArgs))
    PRIM_FAILED;
  else
    PRIM_SUCCEEDED_RELOAD_IP;
}

static long
VMpr_BlockClosure_valueAndResumeOnUnwind(int id,
		  			 volatile int numArgs,
		  			 OOP methodOOP)
{
  gst_method_context context;

  _gst_primitives_executed++;

  context = (gst_method_context) OOP_TO_OBJ (_gst_this_context_oop);
  context->flags |= MCF_IS_UNWIND_CONTEXT;
  if (send_block_value (numArgs))
    PRIM_FAILED;
  else
    PRIM_SUCCEEDED_RELOAD_IP;

}

/* BlockClosure valueWithArguments: */
static long
VMpr_BlockClosure_valueWithArguments(int id,
				     volatile int numArgs,
				     OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_CLASS (oop2, _gst_array_class))
    {
      int i;
      numArgs = NUM_INDEXABLE_FIELDS (oop2);
      for (i = 1; i <= numArgs; i++)
	PUSH_OOP (ARRAY_AT (oop2, i));

      if (send_block_value (numArgs))
	{
	  POP_N_OOPS (numArgs);
	  PUSH_OOP (oop2);
	  PRIM_FAILED;
	}
      else
	PRIM_SUCCEEDED_RELOAD_IP;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* Object perform:
   Object perform:with:
   Object perform:with:with:
   Object perform:with:with:with: */
static long
VMpr_Object_perform(int id,
		    volatile int numArgs,
		    OOP methodOOP)
{
  OOP oop1;
  OOP *oopVec = alloca (numArgs * sizeof (OOP));
  int i;
  _gst_primitives_executed++;

  /* pop off the arguments (if any) */
  numArgs--;
  for (i = 0; i < numArgs; i++)
    oopVec[i] = POP_OOP ();

  oop1 = POP_OOP ();		/* the selector */
  if (IS_CLASS (oop1, _gst_symbol_class)
      && check_send_correctness (STACKTOP (), oop1, numArgs))
    {

      /* push the args back onto the stack */
      while (--i >= 0)
	PUSH_OOP (oopVec[i]);

      SEND_MESSAGE (oop1, numArgs, false);
      PRIM_SUCCEEDED_RELOAD_IP;
    }

  if (IS_CLASS (oop1, _gst_compiled_method_class))
    {
      gst_compiled_method method;
      method_header header;

      method = (gst_compiled_method) OOP_TO_OBJ (oop1);
      header = method->header;
      if (header.numArgs == numArgs)
	{
	  /* push the args back onto the stack */
	  while (--i >= 0)
	    PUSH_OOP (oopVec[i]);

	  _gst_send_method (oop1);
	  PRIM_SUCCEEDED_RELOAD_IP;
	}
    }

  UNPOP (numArgs + 1);
  PRIM_FAILED;
}

/* Object perform:withArguments: */
static long
VMpr_Object_performWithArguments(int id,
				 volatile int numArgs,
				 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  numArgs = NUM_INDEXABLE_FIELDS (oop2);
  if (IS_CLASS (oop2, _gst_array_class)
      && IS_CLASS (oop1, _gst_symbol_class)
      && check_send_correctness (STACKTOP (), oop1, numArgs))
    {

      int i;
      for (i = 1; i <= numArgs; i++)
	PUSH_OOP (ARRAY_AT (oop2, i));

      SEND_MESSAGE (oop1, numArgs, false);
      PRIM_SUCCEEDED_RELOAD_IP;
    }

  if (IS_CLASS (oop1, _gst_compiled_method_class))
    {
      gst_compiled_method method;
      method_header header;

      method = (gst_compiled_method) OOP_TO_OBJ (oop1);
      header = method->header;
      if (header.numArgs == numArgs)
	{
	  int i;
	  for (i = 1; i <= numArgs; i++)
	    PUSH_OOP (ARRAY_AT (oop2, i));

	  _gst_send_method (oop1);
	  PRIM_SUCCEEDED_RELOAD_IP;
	}
    }

  UNPOP (2);
  PRIM_FAILED;
}

/* Semaphore signal */

static long
VMpr_Semaphore_signal(int id,
		      volatile int numArgs,
		      OOP methodOOP)
{
  OOP oop1;
  int_state oldSigMask;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  oldSigMask = _gst_disable_interrupts ();	/* block out
						   everything! */
  sync_signal (oop1);
  _gst_enable_interrupts (oldSigMask);
  PRIM_SUCCEEDED;
}

/* Semaphore wait */
static long
VMpr_Semaphore_wait(int id,
		    volatile int numArgs,
		    OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  _gst_sync_wait (oop1);
  PRIM_SUCCEEDED;
}

/* Process resume */
static long
VMpr_Process_resume(int id,
		    volatile int numArgs,
		    OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (resume_process (oop1))
    PRIM_SUCCEEDED;
  else
    PRIM_FAILED;
}

/* Process yield */
static long
VMpr_Process_yield(int id,
		   volatile int numArgs,
		   OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  usleep (1000);		/* exhibit good behavior */
  oop1 = STACKTOP ();
  if (oop1 == get_active_process ())
    {
      SET_STACKTOP (_gst_nil_oop);	/* this is our return value */
      if (is_process_ready (oop1))
	sleep_process (oop1);	/* move to the end of the list */

      ACTIVE_PROCESS_YIELD ();
    }
  PRIM_SUCCEEDED;
}

/* Behavior flushCache */

static long
VMpr_Behavior_flushCache(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  _gst_primitives_executed++;
  _gst_invalidate_method_cache ();
  PRIM_SUCCEEDED;
}

/* CompiledCode discardTranslation */
static long
VMpr_CompiledCode_discardTranslation(int id,
				     volatile int numArgs,
				     OOP methodOOP)
{
  _gst_primitives_executed++;
#ifdef USE_JIT_TRANSLATION
  discard_native_code (STACKTOP ());
#endif
  PRIM_SUCCEEDED;
}

/* Object changeClassTo: */
static long
VMpr_Object_changeClassTo (int id,
			   volatile int numArgs,
			   OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (is_a_kind_of (OOP_CLASS (oop1), _gst_behavior_class))
    {
      mst_Object obj;
      obj = OOP_TO_OBJ (STACKTOP ());
      obj->objClass = oop1;
      PRIM_SUCCEEDED;
    }
  UNPOP (1);			/* trying to do Bad Things */
  PRIM_FAILED;
}

/* Time class timezoneBias */
static long
VMpr_Time_timezoneBias(int id,
		       volatile int numArgs,
		       OOP methodOOP)
{
  _gst_primitives_executed++;
  SET_STACKTOP_INT (_gst_current_time_zone_bias ());
  PRIM_SUCCEEDED;
}

/* Time class timezone */
static long
VMpr_Time_timezone(int id,
		   volatile int numArgs,
		   OOP methodOOP)
{
  OOP oop1;
  char *result;
  _gst_primitives_executed++;

  result = _gst_current_time_zone_name ();
  oop1 = _gst_string_new (result);
  SET_STACKTOP (oop1);
  xfree (result);
  PRIM_SUCCEEDED;
}

/* Time class secondClock
    -- note: this primitive has different
       semantics from those defined in the
       book.  This primitive returns the
       seconds since/to Jan 1, 2000 00:00:00
       instead of Jan 1,1901. */
static long
VMpr_Time_secondClock(int id,
		      volatile int numArgs,
		      OOP methodOOP)
{
  _gst_primitives_executed++;
  /* 10957 = days between 1970 and 2000 */
  SET_STACKTOP_INT (_gst_get_time () - 86400 * 10957);
  PRIM_SUCCEEDED;
}

/* Time class millisecondClock -- Note: the semantics of this
   primitive are different than those described in the book.  This
   primitive returns the number of milliseconds since midnight
   today. */
static long
VMpr_Time_millisecondClock(int id,
			   volatile int numArgs,
			   OOP methodOOP)
{
  _gst_primitives_executed++;
  SET_STACKTOP_INT (_gst_get_milli_time () % (24 * 60 * 60 * 1000));
  PRIM_SUCCEEDED;
}

/* Processor signal: semaphore
             atMilliseconds: deltaMilliseconds */
static long
VMpr_Processor_signalAtMilliseconds(int id,
				    volatile int numArgs,
				    OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop2))
    {
      long arg2;
      arg2 = TO_INT (oop2);
      if (arg2 <= 0)
	{
	  int_state oldSigMask;
	  oldSigMask = _gst_disable_interrupts ();	/* block out
							   everything! */
	  sync_signal (oop1);
	  _gst_enable_interrupts (oldSigMask);
	}
      else
	{
	  _gst_async_timed_wait (oop1, arg2);
	  PRIM_SUCCEEDED;
	}
    }

  UNPOP (2);
  PRIM_FAILED;
}

/* Processor isTimeoutProgrammed */
static long
VMpr_Processor_isTimeoutProgrammed(int id,
				   volatile int numArgs,
				   OOP methodOOP)
{
  _gst_primitives_executed++;
  SET_STACKTOP_BOOLEAN (_gst_is_timeout_programmed ());
  PRIM_SUCCEEDED;
}

/* String hash */

static long
VMpr_String_hash(int id,
		 volatile int numArgs,
		 OOP methodOOP)
{
  unsigned long hash;
  long spec;
  gst_uchar *base;
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  spec = OOP_INSTANCE_SPEC (oop1);
#ifndef OPTIMIZE
  if (!
      (spec &
       ((~0 << ISP_NUMFIXEDFIELDS) | ISP_ISPOINTERS | ISP_ISWORDS)))
    {
#endif
      base = STRING_OOP_CHARS (oop1);
      hash = _gst_hash_string (base, NUM_INDEXABLE_FIELDS (oop1));
      SET_STACKTOP_INT (hash);
      PRIM_SUCCEEDED;
#ifndef OPTIMIZE
    }
#endif
  PRIM_FAILED;
}

/* ByteArray primReplaceFrom:to:with:startingAt
   ByteArray replaceFrom:to:withString:startingAt:
   String replaceFrom:to:withByteArray:startingAt:
   String primReplaceFrom:to:with:startingAt:*/
static long
VMpr_ByteArray_replaceFromToWithStartingAt(int id,
					   volatile int numArgs,
					   OOP methodOOP)
{
  OOP srcIndexOOP, srcOOP, dstEndIndexOOP, dstStartIndexOOP, dstOOP;
  int dstEndIndex, dstStartIndex, srcIndex, dstLen, srcLen, dstRangeLen;
  long spec;
  gst_uchar *dstBase, *srcBase;
  _gst_primitives_executed++;

  srcIndexOOP = POP_OOP ();
  srcOOP = POP_OOP ();
  dstEndIndexOOP = POP_OOP ();
  dstStartIndexOOP = POP_OOP ();
  if (IS_INT (srcIndexOOP) && IS_INT (dstStartIndexOOP)
      && IS_INT (dstEndIndexOOP) && !IS_INT (srcOOP))
    {
      spec = OOP_INSTANCE_SPEC (srcOOP);
      if (!(spec & (ISP_ISWORDS | ISP_ISPOINTERS)))
	{
	  /* dstEnd is inclusive: (1 to: 1) has length 1 */
	  dstEndIndex = TO_INT (dstEndIndexOOP);
	  dstStartIndex = TO_INT (dstStartIndexOOP);
	  srcIndex = TO_INT (srcIndexOOP);
	  dstOOP = STACKTOP ();
	  dstLen = NUM_INDEXABLE_FIELDS (dstOOP);
	  srcLen = NUM_INDEXABLE_FIELDS (srcOOP);
	  dstRangeLen = dstEndIndex - dstStartIndex + 1;
	  if ((dstRangeLen >= 0 && dstEndIndex <= dstLen
	       && dstStartIndex > 0))
	    {
	      if (dstRangeLen > 0)
		{		/* don't do it unless somethings to
				   copy */
		  if ((srcIndex <= srcLen) && (srcIndex > 0)
		      && (srcIndex + dstRangeLen - 1 <= srcLen))
		    {
		      /* do the copy */
		      dstBase = STRING_OOP_CHARS (dstOOP);
		      srcBase = STRING_OOP_CHARS (srcOOP);
		      memmove (&dstBase[dstStartIndex - 1],
			      &srcBase[srcIndex - 1], dstRangeLen);
		    }
		}
	      PRIM_SUCCEEDED;
	    }
	}
    }

  UNPOP (4);
  PRIM_FAILED;
}

/* Object ==, Character = */

static long
VMpr_Object_identity(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  PUSH_BOOLEAN (oop1 == oop2);
  PRIM_SUCCEEDED;
}

/* Object class */
static long
VMpr_Object_class(int id,
		  volatile int numArgs,
		  OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (IS_INT (oop1))
    PUSH_OOP (_gst_small_integer_class);
  else
    PUSH_OOP (OOP_CLASS (oop1));

  PRIM_SUCCEEDED;
}


/* quit: status
   Always fail because if it succeeds we don't return */
static long
VMpr_ObjectMemory_quit(int id,
		       volatile int numArgs,
		       OOP methodOOP)
{
  OOP oop1;
  long arg1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_INT (oop1))
    {
      _gst_msg_sendf (NULL, "%v %o changed: %S",
		      _gst_object_memory_class, "aboutToQuit");
      arg1 = TO_INT (oop1);
      exit (arg1);
    }
  PRIM_FAILED;
}


/* ------- GNU Smalltalk specific primitives begin here -------------------- */
/* Dictionary at: */
static long
VMpr_Dictionary_at(int id,
		   volatile int numArgs,
		   OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();

  SET_STACKTOP (dictionary_at (oop1, oop2));
  PRIM_SUCCEEDED;
}

/* Dictionary at: put: */
static long
VMpr_Dictionary_atPut(int id,
		      volatile int numArgs,
		      OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  OOP oop3;
  _gst_primitives_executed++;

  oop3 = POP_OOP ();
  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  DICTIONARY_AT_PUT (oop1, oop2, oop3);
  SET_STACKTOP (oop3);
  PRIM_SUCCEEDED;
}

/* This is not defined in terms of #error: in a .st file because some 
   of the required functionality may not be present when it gets
   first invoked, say during the loading of the first kernel files.
   We'll redefine it later. */

/* Object doesNotUnderstand:
 * Object error: */
static long
VMpr_Object_bootstrapException(int id,
			       volatile int numArgs,
			       OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  _gst_debug ();
  if (id == 130)
    printf ("%O did not understand selector %O\n\n",
      	    oop1, MESSAGE_SELECTOR (oop2));
  else
    printf ("%O error: %#O\n\n", oop1, oop2);

  _gst_show_backtrace ();
  _gst_show_stack_contents ();

  /* cannot halt - not a problem, since this is only temporary. The
     worst thing that can happen is that the image is not rebuilt
     correctly, but they'll understand it because it will result in
     loads of error messages.  Remember that these primitives are here
     for emergency conditions. */

  PRIM_SUCCEEDED;
}


/* Character class value: */

static long
VMpr_Character_create(int id,
		      volatile int numArgs,
		      OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_INT (oop2))
    {
      long arg2;
      arg2 = TO_INT (oop2);
      if (arg2 >= 0 && arg2 <= 255)
	{
	  SET_STACKTOP (CHAR_OOP_AT (arg2));
	  PRIM_SUCCEEDED;
	}
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* Character value */
static long
VMpr_Character_value(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  SET_STACKTOP_INT (CHAR_OOP_VALUE (oop1));
  PRIM_SUCCEEDED;
}

/* Symbol class intern: aString */
static long
VMpr_Symbol_intern(int id,
		   volatile int numArgs,
		   OOP methodOOP)
{
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = STACKTOP ();		/* keeps this guy referenced while
				   being interned */
  if (IS_CLASS (oop2, _gst_string_class))
    {
      OOP internedString;
      internedString = _gst_intern_string_oop (oop2);
      (void) POP_OOP ();

      SET_STACKTOP (internedString);
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* Dictionary new */
static long
VMpr_Dictionary_new(int id,
		    volatile int numArgs,
		    OOP methodOOP)
{
  _gst_primitives_executed++;
  SET_STACKTOP (_gst_dictionary_new (37));
  PRIM_SUCCEEDED;
}


/* Memory addressOfOOP: oop */
static long
VMpr_Memory_addressOfOOP(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_OOP (oop2))
    {
      PUSH_OOP (FROM_C_ULONG ((long) oop2));
      PRIM_SUCCEEDED;
    }
  UNPOP (2);
  PRIM_FAILED;
}

/* Memory addressOf: oop */
static long
VMpr_Memory_addressOf(int id,
		      volatile int numArgs,
		      OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_OOP (oop2))
    {
      PUSH_OOP (FROM_C_ULONG ((long) OOP_TO_OBJ (oop2)));
      PRIM_SUCCEEDED;
    }
  UNPOP (2);
  PRIM_FAILED;
}


/* SystemDictionary backtrace */

static long
VMpr_SystemDictionary_backtrace(int id,
				volatile int numArgs,
				OOP methodOOP)
{
  _gst_primitives_executed++;
  _gst_show_backtrace ();
  PRIM_SUCCEEDED;
}

/* SystemDictionary getTraceFlag: anIndex */
static long
VMpr_SystemDictionary_getTraceFlag(int id,
				   volatile int numArgs,
				   OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop2))
    {
      long arg2;
      mst_Boolean *boolAddr;
      arg2 = TO_INT (oop2);
      boolAddr = bool_addr_index (arg2);
      if (boolAddr != NULL)
	{
	  oop1 = *boolAddr ? _gst_true_oop : _gst_false_oop;
	  PUSH_OOP (oop1);
	  PRIM_SUCCEEDED;
	}
    }

  UNPOP (2);
  PRIM_FAILED;
}

/* SystemDictionary setTraceFlag: anIndex to: aBoolean */
static long
VMpr_SystemDictionary_setTraceFlag(int id,
				   volatile int numArgs,
				   OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop1))
    {
      mst_Boolean *boolAddr;
      long arg1 = TO_INT (oop1);
      boolAddr = bool_addr_index (arg1);
      if (boolAddr != NULL)
	{
	  *boolAddr = (oop2 == _gst_true_oop);
	  SET_EXCEPT_FLAG (true);
	  PRIM_SUCCEEDED;
	}
    }

  UNPOP (2);
  PRIM_FAILED;
}

/* CObject class alloc: nBytes */

static long
VMpr_CObject_alloc(int id,
		   volatile int numArgs,
		   OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_INT (oop2))
    {
      long arg2;
      arg2 = TO_INT (oop2);
      SET_STACKTOP (_gst_alloc_cobject (oop1, arg2));
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* Memory type: aType at: anAddress */
static long
VMpr_Memory_at(int id,
	       volatile int numArgs,
	       OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  OOP oop3;
  _gst_primitives_executed++;

  oop3 = POP_OOP ();
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_C_LONG (oop3) && IS_INT (oop2))
    {
      long arg1, arg2;
      arg1 = TO_INT (oop2);
      arg2 = TO_C_LONG (oop3);
      switch (arg1)
	{
	case 0:		/* char */
	  /* may want to use Character instead? */
	  PUSH_OOP (CHAR_OOP_AT (*(char *) arg2));
	  PRIM_SUCCEEDED;
	case 1:		/* unsigned char */
	  PUSH_OOP (CHAR_OOP_AT (*(unsigned char *) arg2));
	  PRIM_SUCCEEDED;
	case 2:		/* short */
	  PUSH_INT (*(short *) arg2);
	  PRIM_SUCCEEDED;
	case 3:		/* unsigned short */
	  PUSH_INT (*(unsigned short *) arg2);
	  PRIM_SUCCEEDED;
	case 4:		/* long */
	  PUSH_OOP (FROM_C_LONG (*(long *) arg2));
	  PRIM_SUCCEEDED;
	case 5:		/* unsigned long */
	  PUSH_OOP (FROM_C_ULONG (*(unsigned long *) arg2));
	  PRIM_SUCCEEDED;
	case 6:		/* float */
	  PUSH_OOP (floate_new (*(float *) arg2));
	  PRIM_SUCCEEDED;
	case 7:		/* double */
	  PUSH_OOP (floatd_new (*(double *) arg2));
	  PRIM_SUCCEEDED;
	case 8:		/* string */
	  if (*(char **) arg2)
	    PUSH_OOP (_gst_string_new (*(char **) arg2));
	  else
	    PUSH_OOP (_gst_nil_oop);

	  PRIM_SUCCEEDED;
	case 9:		/* OOP */
	  PUSH_OOP (*(OOP *) arg2);
	  PRIM_SUCCEEDED;
	case 10:		/* int */
	  PUSH_OOP (FROM_C_INT (*(int *) arg2));
	  PRIM_SUCCEEDED;
	case 11:		/* unsigned int */
	  PUSH_OOP (FROM_C_UINT (*(unsigned int *) arg2));
	  PRIM_SUCCEEDED;
	}
    }

  UNPOP (3);
  PRIM_FAILED;
}

/* Memory type: aType at: anAddress put: aValue */
static long
VMpr_Memory_atPut(int id,
		  volatile int numArgs,
		  OOP methodOOP)
{
  OOP oop4;
  OOP oop3;
  OOP oop2;
  _gst_primitives_executed++;

  oop4 = POP_OOP ();
  oop3 = POP_OOP ();
  oop2 = POP_OOP ();
  /* don't pop the receiver */
  if (IS_C_LONG (oop3) && IS_INT (oop2))
    {
      long arg1, arg2;
      arg1 = TO_INT (oop2);
      arg2 = TO_C_LONG (oop3);
      switch (arg1)
	{
	case 0:		/* char */
	case 1:		/* unsigned char */
	  /* may want to use Character instead? */
	  if (IS_CLASS (oop4, _gst_char_class))
	    {
	      *(char *) arg2 = CHAR_OOP_VALUE (oop4);
	      PRIM_SUCCEEDED;
	    }
	  else if (IS_INT (oop4))
	    {
	      *(char *) arg2 = (char) TO_INT (oop4);
	      PRIM_SUCCEEDED;
	    }
	  break;
	case 2:		/* short */
	case 3:		/* unsigned short */
	  if (IS_INT (oop4))
	    {
	      *(short *) arg2 = (short) TO_INT (oop4);
	      PRIM_SUCCEEDED;
	    }
	  break;
	case 4:		/* long */
	case 5:		/* unsigned long */
	  if (IS_C_LONG (oop4))
	    {
	      *(long *) arg2 = TO_C_LONG (oop4);
	      PRIM_SUCCEEDED;
	    }
	  break;
	case 6:		/* float */
	  if (IS_CLASS (oop4, _gst_floatd_class))
	    {
	      *(float *) arg2 = (float) FLOATD_OOP_VALUE (oop4);
	      PRIM_SUCCEEDED;
	    }
	  else if (IS_CLASS (oop4, _gst_floate_class))
	    {
	      *(float *) arg2 = FLOATE_OOP_VALUE (oop4);
	      PRIM_SUCCEEDED;
	    }
	  else if (IS_CLASS (oop4, _gst_floatq_class))
	    {
	      *(float *) arg2 = (float) FLOATQ_OOP_VALUE (oop4);
	      PRIM_SUCCEEDED;
	    }
	  break;
	case 7:		/* double */
	  if (IS_CLASS (oop4, _gst_floatd_class))
	    {
	      *(double *) arg2 = FLOATD_OOP_VALUE (oop4);
	      PRIM_SUCCEEDED;
	    }
	  else if (IS_CLASS (oop4, _gst_floate_class))
	    {
	      *(double *) arg2 = (double) FLOATE_OOP_VALUE (oop4);
	      PRIM_SUCCEEDED;
	    }
	  else if (IS_CLASS (oop4, _gst_floatq_class))
	    {
	      *(double *) arg2 = (double) FLOATQ_OOP_VALUE (oop4);
	      PRIM_SUCCEEDED;
	    }
	  break;
	case 8:		/* string */
	  if (IS_CLASS (oop4, _gst_string_class)
	      || IS_CLASS (oop4, _gst_symbol_class))
	    {
	      /* Char* cast on the right side needed because
	         _gst_to_cstring returns gst_uchar * */
	      *(char **) arg2 = (char *) _gst_to_cstring (oop4);
	      PRIM_SUCCEEDED;
	    }
	  break;
	case 9:		/* OOP */
	  *(OOP *) arg2 = oop4;
	  PRIM_SUCCEEDED;
	case 10:		/* int */
	case 11:		/* unsigned int */
	  if (IS_C_INT (oop4))
	    {
	      *(int *) arg2 = TO_C_INT (oop4);
	      PRIM_SUCCEEDED;
	    }
	  break;
	}
    }

  UNPOP (3);
  PRIM_FAILED;
}


/* methodsFor: category */
static long
VMpr_Behavior_methodsFor(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  _gst_primitives_executed++;
  _gst_set_compilation_category (POP_OOP ());
  _gst_set_compilation_class (STACKTOP ());
  _gst_display_compilation_trace ("Compiling", true);
  _gst_compile_code = true;
  PRIM_SUCCEEDED;
}

/* methodsFor: category ifTrue: condition */
static long
VMpr_Behavior_methodsForIfTrue(int id,
			       volatile int numArgs,
			       OOP methodOOP)
{
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  _gst_set_compilation_category (POP_OOP ());
  _gst_set_compilation_class (STACKTOP ());
  if (oop2 == _gst_true_oop)
    _gst_display_compilation_trace ("Conditionally compiling", true);
  else
    {
      _gst_skip_compilation = true;
      _gst_display_compilation_trace ("Conditionally skipping", true);
    }
  _gst_compile_code = true;
  PRIM_SUCCEEDED;
}


/* ProcessorScheduler signal: aSemaphore onInterrupt: anInteger */
static long
VMpr_Processor_signalOnInterrupt(int id,
				 volatile int numArgs,
				 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop2))
    {
      long arg2;
      arg2 = TO_INT (oop2);
      _gst_async_interrupt_wait (oop1, arg2);
      PRIM_SUCCEEDED;
    }

  UNPOP (2);
  PRIM_FAILED;
}

/* ObjectMemory _gst_space_grow_rate */

static long
VMpr_ObjectMemory_getSpaceGrowRate(int id,
				   volatile int numArgs,
				   OOP methodOOP)
{
  _gst_primitives_executed++;
  SET_STACKTOP (floatd_new ((double) _gst_space_grow_rate));
  PRIM_SUCCEEDED;
}

/* ObjectMemory _gst_space_grow_rate: */
static long
VMpr_ObjectMemory_setSpaceGrowRate(int id,
				   volatile int numArgs,
				   OOP methodOOP)
{
  long arg1;
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (IS_CLASS (oop1, _gst_floatd_class))
    arg1 = (int) FLOATD_OOP_VALUE (oop1);
  else if (IS_CLASS (oop1, _gst_floate_class))
    arg1 = (int) FLOATE_OOP_VALUE (oop1);
  else if (IS_CLASS (oop1, _gst_floatq_class))
    arg1 = (int) FLOATQ_OOP_VALUE (oop1);
  else if (IS_INT (oop1))
    arg1 = TO_INT (oop1);
  else
    {
      UNPOP (1);
      PRIM_FAILED;
    }

  if (arg1 > 0 && arg1 <= 500)
    {
      _gst_space_grow_rate = arg1;
      PRIM_SUCCEEDED;
    }

  UNPOP (1);
  PRIM_FAILED;
}

/* ObjectMemory _gst_grow_threshold_percent */
static long
VMpr_ObjectMemory_getGrowThresholdPercent (int id,
					   volatile int numArgs,
					   OOP methodOOP)
{
  _gst_primitives_executed++;
  SET_STACKTOP (floatd_new ((double) _gst_grow_threshold_percent));
  PRIM_SUCCEEDED;
}

/* ObjectMemory _gst_grow_threshold_percent: */
static long
VMpr_ObjectMemory_setGrowThresholdPercent(int id,
					  volatile int numArgs,
					  OOP methodOOP)
{
  OOP oop1;
  long arg1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (IS_CLASS (oop1, _gst_floatd_class))
    arg1 = (int) FLOATD_OOP_VALUE (oop1);
  else if (IS_CLASS (oop1, _gst_floate_class))
    arg1 = (int) FLOATE_OOP_VALUE (oop1);
  else if (IS_CLASS (oop1, _gst_floatq_class))
    arg1 = (int) FLOATQ_OOP_VALUE (oop1);
  else if (IS_INT (oop1))
    arg1 = TO_INT (oop1);
  else
    {
      UNPOP (1);
      PRIM_FAILED;
    }
  if (arg1 > 0 && arg1 < 100)
    {
      _gst_grow_threshold_percent = arg1;
      PRIM_SUCCEEDED;
    }

  UNPOP (1);
  PRIM_FAILED;
}

/* ObjectMemory _gst_big_object_threshold */
static long
VMpr_ObjectMemory_getBigObjectThreshold (int id,
					 volatile int numArgs,
					 OOP methodOOP)
{
  _gst_primitives_executed++;
  SET_STACKTOP_INT (_gst_big_object_threshold);
  PRIM_SUCCEEDED;
}

/* ObjectMemory _gst_big_object_threshold: */
static long
VMpr_ObjectMemory_setBigObjectThreshold(int id,
					volatile int numArgs,
					OOP methodOOP)
{
  OOP oop1;
  long arg1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (IS_NIL (oop1))
    arg1 = INT_MAX < MAX_ST_INT ? INT_MAX : MAX_ST_INT;
  else if (IS_INT (oop1))
    arg1 = TO_INT (oop1);
  else
    {
      UNPOP (1);
      PRIM_FAILED;
    }

  if (arg1 >= 0)
    {
      _gst_big_object_threshold = arg1;
      PRIM_SUCCEEDED;
    }

  UNPOP (1);
  PRIM_FAILED;
}

/* ObjectMemory _gst_grow_to: numBytes */
static long
VMpr_ObjectMemory_growTo(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  OOP oop1;
  long arg1;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  if (IS_INT (oop1))
    {
      arg1 = TO_INT (oop1);
      if (_gst_grow_to (arg1))
	PRIM_SUCCEEDED;
    }

  UNPOP (1);
  PRIM_FAILED;
}

/* CObject class alloc: nbytes type: aType */

static long
VMpr_CObject_allocType (int id,
			volatile int numArgs,
			OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop1 = POP_OOP ();
  oop2 = POP_OOP ();
  if (IS_INT (oop2))
    {
      long arg2;
      PTR ptr;
      arg2 = TO_INT (oop2);
      ptr = xmalloc (arg2);

      SET_STACKTOP (_gst_c_object_new_typed (ptr, oop1));
      PRIM_SUCCEEDED;
    }
  UNPOP (2);
  PRIM_FAILED;
}

/* sin */
static long
VMpr_Float_sin(int id,
	       volatile int numArgs,
	       OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_CLASS (oop1, _gst_floatd_class))
    {
      double farg1 = FLOATD_OOP_VALUE (oop1);
      SET_STACKTOP (floatd_new (sin (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floate_class))
    {
      double farg1 = (double) FLOATE_OOP_VALUE (oop1);
      SET_STACKTOP (floate_new (sin (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floatq_class))
    {
      long double farg1 = FLOATQ_OOP_VALUE (oop1);
      SET_STACKTOP (floatq_new (sinl (farg1)));
      PRIM_SUCCEEDED;
    }
  PRIM_FAILED;
}

/* cos */
static long
VMpr_Float_cos(int id,
	       volatile int numArgs,
	       OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_CLASS (oop1, _gst_floatd_class))
    {
      double farg1 = FLOATD_OOP_VALUE (oop1);
      SET_STACKTOP (floatd_new (cos (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floate_class))
    {
      double farg1 = (double) FLOATE_OOP_VALUE (oop1);
      SET_STACKTOP (floate_new (cos (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floatq_class))
    {
      long double farg1 = FLOATQ_OOP_VALUE (oop1);
      SET_STACKTOP (floatq_new (cosl (farg1)));
      PRIM_SUCCEEDED;
    }
  PRIM_FAILED;
}

/* tan */
static long
VMpr_Float_tan(int id,
	       volatile int numArgs,
	       OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_CLASS (oop1, _gst_floatd_class))
    {
      double farg1 = FLOATD_OOP_VALUE (oop1);
      SET_STACKTOP (floatd_new (tan (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floate_class))
    {
      double farg1 = (double) FLOATE_OOP_VALUE (oop1);
      SET_STACKTOP (floate_new (tan (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floatq_class))
    {
      long double farg1 = FLOATQ_OOP_VALUE (oop1);
      SET_STACKTOP (floatq_new (tanl (farg1)));
      PRIM_SUCCEEDED;
    }
  PRIM_FAILED;
}

/* arcSin */
static long
VMpr_Float_arcSin(int id,
	       volatile int numArgs,
	       OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_CLASS (oop1, _gst_floatd_class))
    {
      double farg1 = FLOATD_OOP_VALUE (oop1);
      SET_STACKTOP (floatd_new (asin (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floate_class))
    {
      double farg1 = (double) FLOATE_OOP_VALUE (oop1);
      SET_STACKTOP (floate_new (asin (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floatq_class))
    {
      long double farg1 = FLOATQ_OOP_VALUE (oop1);
      SET_STACKTOP (floatq_new (asinl (farg1)));
      PRIM_SUCCEEDED;
    }
  PRIM_FAILED;
}

/* arcCos */
static long
VMpr_Float_arcCos(int id,
	       volatile int numArgs,
	       OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_CLASS (oop1, _gst_floatd_class))
    {
      double farg1 = FLOATD_OOP_VALUE (oop1);
      SET_STACKTOP (floatd_new (acos (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floate_class))
    {
      double farg1 = (double) FLOATE_OOP_VALUE (oop1);
      SET_STACKTOP (floate_new (acos (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floatq_class))
    {
      long double farg1 = FLOATQ_OOP_VALUE (oop1);
      SET_STACKTOP (floatq_new (acosl (farg1)));
      PRIM_SUCCEEDED;
    }
  PRIM_FAILED;
}

/* arcTan */
static long
VMpr_Float_arcTan(int id,
	       volatile int numArgs,
	       OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_CLASS (oop1, _gst_floatd_class))
    {
      double farg1 = FLOATD_OOP_VALUE (oop1);
      SET_STACKTOP (floatd_new (atan (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floate_class))
    {
      double farg1 = (double) FLOATE_OOP_VALUE (oop1);
      SET_STACKTOP (floate_new (atan (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floatq_class))
    {
      long double farg1 = FLOATQ_OOP_VALUE (oop1);
      SET_STACKTOP (floatq_new (atanl (farg1)));
      PRIM_SUCCEEDED;
    }
  PRIM_FAILED;
}

/* exp */
static long
VMpr_Float_exp(int id,
	       volatile int numArgs,
	       OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_CLASS (oop1, _gst_floatd_class))
    {
      double farg1 = FLOATD_OOP_VALUE (oop1);
      SET_STACKTOP (floatd_new (exp (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floate_class))
    {
      double farg1 = (double) FLOATE_OOP_VALUE (oop1);
      SET_STACKTOP (floate_new (exp (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floatq_class))
    {
      long double farg1 = FLOATQ_OOP_VALUE (oop1);
      SET_STACKTOP (floatq_new (expl (farg1)));
      PRIM_SUCCEEDED;
    }
  PRIM_FAILED;
}

/* ln */
static long
VMpr_Float_ln(int id,
	      volatile int numArgs,
	      OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_CLASS (oop1, _gst_floatd_class))
    {
      double farg1 = FLOATD_OOP_VALUE (oop1);
      SET_STACKTOP (floatd_new (log (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floate_class))
    {
      double farg1 = (double) FLOATE_OOP_VALUE (oop1);
      SET_STACKTOP (floate_new (log (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floatq_class))
    {
      long double farg1 = FLOATQ_OOP_VALUE (oop1);
      SET_STACKTOP (floatq_new (logl (farg1)));
      PRIM_SUCCEEDED;
    }
  PRIM_FAILED;
}

/* SystemDictionary resetStatistics */
static long
VMpr_SystemDictionary_resetStatistics(int id,
				      volatile int numArgs,
				      OOP methodOOP)
{
  _gst_primitives_executed++;
#ifdef PROFBLOCK
  ps.numThisContexts = 0;
  ps.numMethodAllocs = 0;
  ps.numMethodFrees = 0;
  ps.numMessageSends = 0;
  ps.numValues = 0;
  ps.numCacheCollisions = 0;
  ps.stackSizeSum = 0;
  ps.stack_depth = 0;
  ps.max_stack_depth = 0;
  ps.numMinorGCs = 0;
  ps.numMajorGCs = 0;
  init_bytecode_counter ();
#endif
  PRIM_SUCCEEDED;
}

/* SystemDictionary printStatistics */
static long
VMpr_SystemDictionary_printStatistics(int id,
				      volatile int numArgs,
				      OOP methodOOP)
{
  _gst_primitives_executed++;
#ifdef PROFBLOCK
  printf ("%d thisContexts, of %d contexts used %d were lifo\n",
	  ps.numThisContexts, ps.numMethodAllocs, ps.numMethodFrees);
  printf ("%d message sends, %d were value: methods\n",
	  ps.numMessageSends, ps.numValues);
  printf ("%d method cache collisions\n", ps.numCacheCollisions);
  printf ("%d maxDepth\n", ps.max_stack_depth);
  printf ("%d gc's (%d minor, %d major)\n",
	  ps.numMinorGCs + ps.numMajorGCs, ps.numMinorGCs,
	  ps.numMajorGCs);

  print_bytecode_counts ();
#else
  printf ("VM profiling not active\n");
#endif
  PRIM_SUCCEEDED;
}

/* raisedTo: aNumber -- receiver ** aNumber */
static long
VMpr_Float_pow(int id,
	       volatile int numArgs,
	       OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  double farg1, farg2;
  long double lfarg1, lfarg2;
  mst_Boolean long_double = false;

  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_CLASS (oop1, _gst_floatd_class))
    lfarg1 = farg1 = FLOATD_OOP_VALUE (oop1);
  else if (IS_CLASS (oop1, _gst_floate_class))
    lfarg1 = farg1 = FLOATE_OOP_VALUE (oop1);
  else if (IS_CLASS (oop1, _gst_floatq_class))
    {
      long_double = true;
      lfarg1 = farg1 = FLOATQ_OOP_VALUE (oop1);
    }
  else
    {
      UNPOP (1);
      PRIM_FAILED;
    }

  if (IS_CLASS (oop2, _gst_floatd_class))
    lfarg2 = farg2 = FLOATD_OOP_VALUE (oop2);
  else if (IS_CLASS (oop2, _gst_floate_class))
    lfarg2 = farg2 = FLOATE_OOP_VALUE (oop2);
  else if (IS_CLASS (oop2, _gst_floatq_class))
    {
      long_double = true;
      lfarg2 = farg2 = FLOATQ_OOP_VALUE (oop2);
    }
  else
    {
      UNPOP (1);
      PRIM_FAILED;
    }

  if ((lfarg1 == 0.0 && lfarg2 <= 0.0) || lfarg1 < 0.0)
    {
      UNPOP (1);
      PRIM_FAILED;
    }

  if (long_double)
    {
      if (IS_NAN (lfarg1) || IS_NAN (lfarg2))
        /* The C99 standard mandates that pow(1, NaN) = 1.0 and pow
           (NaN, 0.0) = 1.0, which is plain wrong.  We take the
           liberty to make these results be NaN. */
        SET_STACKTOP (floatq_new (lfarg1 + lfarg2));
      else
        SET_STACKTOP (floatq_new (expl (lfarg2 * logl (lfarg1))));
    }

  else
    {
      if (IS_NAN (farg1) || IS_NAN (farg2))
        /* The C99 standard mandates that pow(1, NaN) = 1.0 and pow
           (NaN, 0.0) = 1.0, which is plain wrong.  We take the
           liberty to make these results be NaN. */
        SET_STACKTOP (floatd_new (farg1 + farg2));
      else
        SET_STACKTOP (floatd_new (exp (lfarg2 * log (lfarg1))));
    }

  PRIM_SUCCEEDED;
}

/* CObject free */
static long
VMpr_CObject_free(int id,
		  volatile int numArgs,
		  OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (RECEIVER_IS_A_KIND_OF (OOP_CLASS (oop1), _gst_c_object_class))
    {
      _gst_free_cobject (oop1);	/* free allocated space */
      SET_STACKTOP (_gst_nil_oop);
      PRIM_SUCCEEDED;
    }
  PRIM_FAILED;
}

/* sqrt */
static long
VMpr_Float_sqrt(int id,
		volatile int numArgs,
		OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_CLASS (oop1, _gst_floatd_class))
    {
      double farg1 = FLOATD_OOP_VALUE (oop1);
      SET_STACKTOP (floatd_new (sqrt (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floate_class))
    {
      double farg1 = FLOATE_OOP_VALUE (oop1);
      SET_STACKTOP (floate_new (sqrt (farg1)));
      PRIM_SUCCEEDED;
    }
  else if (IS_CLASS (oop1, _gst_floatq_class))
    {
      long double farg1 = FLOATQ_OOP_VALUE (oop1);
      SET_STACKTOP (floatq_new (sqrtl (farg1)));
      PRIM_SUCCEEDED;
    }
  PRIM_FAILED;
}

/* ceiling, floor */
static long
VMpr_Float_ceiling_floor(int id,
		volatile int numArgs,
		OOP methodOOP)
{
  OOP oop1;
  double farg1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_CLASS (oop1, _gst_floatd_class))
    farg1 = FLOATD_OOP_VALUE (oop1);
  else if (IS_CLASS (oop1, _gst_floate_class))
    farg1 = (double) FLOATE_OOP_VALUE (oop1);
  else if (IS_CLASS (oop1, _gst_floatq_class))
    farg1 = (double) FLOATQ_OOP_VALUE (oop1);
  else
    PRIM_FAILED;

  if COMMON ((farg1 > MIN_ST_INT) && farg1 < MAX_ST_INT)
    {
      switch (id)
        {
        case 168:
          SET_STACKTOP_INT ((long) ceil (farg1));
          PRIM_SUCCEEDED;
        case 169:
          SET_STACKTOP_INT ((long) floor (farg1));
          PRIM_SUCCEEDED;
        }
    }

  PRIM_FAILED;
}

/* Behavior basicNewInFixedSpace */
static long
VMpr_Behavior_basicNewFixed(int id,
			    volatile int numArgs,
			    OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_OOP (oop1))
    {
      if (!CLASS_IS_INDEXABLE (oop1))
	{
	  OOP result;
	  instantiate (oop1, &result);
	  _gst_make_fixed_oop (result);
	  SET_STACKTOP (result);
	  PRIM_SUCCEEDED;
	}
    }
  PRIM_FAILED;
}

/* Behavior basicNewInFixedSpace: */
static long
VMpr_Behavior_basicNewFixedColon(int id,
				 volatile int numArgs,
				 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_OOP (oop1) && IS_INT (oop2))
    {
      if (CLASS_IS_INDEXABLE (oop1))
	{
	  long arg2;
	  arg2 = TO_INT (oop2);
	  if (arg2 >= 0)
	    {
	      OOP result;
	      instantiate_with (oop1, arg2, &result);
	      _gst_make_fixed_oop (result);
	      SET_STACKTOP (result);
	      PRIM_SUCCEEDED;
	    }
	}
    }

  UNPOP (1);
  PRIM_FAILED;
}

static long
VMpr_Behavior_makeFixed(int id,
			volatile int numArgs,
			OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  if (IS_OOP (oop1) && !IS_OOP_READONLY (oop1))
    {
      if (_gst_make_fixed_oop (oop1))
	PRIM_SUCCEEDED;

    }
  PRIM_FAILED;
}


/* CObject at: byteoffset type: aType */

static long
VMpr_CObject_at(int id,
		volatile int numArgs,
		OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  OOP oop3;
  _gst_primitives_executed++;

  oop3 = POP_OOP ();
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_INT (oop2))
    {
      char *addr;
      long arg2;
      arg2 = TO_INT (oop2);
      if (id == 182)
	{
	  addr = *(char **) COBJECT_VALUE (oop1);
	  if (addr == 0)
	    {
	      PUSH_OOP (_gst_nil_oop);
	      PRIM_SUCCEEDED;
	    }
	}
      else
	addr = COBJECT_VALUE (oop1);

      addr += arg2;		/* compute effective address */
      if (IS_INT (oop3))
	{			/* int type spec means a scalar type */
	  long arg3;
	  arg3 = TO_INT (oop3);

	  switch (arg3)
	    {
	    case 0:
	      PUSH_OOP (CHAR_OOP_AT (*(gst_uchar *) addr));
	      PRIM_SUCCEEDED;

	    case 1:
	      PUSH_OOP (CHAR_OOP_AT (*(gst_uchar *) addr));
	      PRIM_SUCCEEDED;

	    case 2:
	      PUSH_INT (*(short *) addr);
	      PRIM_SUCCEEDED;

	    case 3:
	      PUSH_INT (*(unsigned short *) addr);
	      PRIM_SUCCEEDED;

	    case 4:
	      PUSH_OOP (FROM_C_LONG (*(long *) addr));
	      PRIM_SUCCEEDED;

	    case 5:
	      PUSH_OOP (FROM_C_ULONG (*(unsigned long *) addr));
	      PRIM_SUCCEEDED;

	    case 6:
	      PUSH_OOP (floate_new (*(float *) addr));
	      PRIM_SUCCEEDED;

	    case 7:
	      PUSH_OOP (floatd_new (*(double *) addr));
	      PRIM_SUCCEEDED;

	    case 8:
	      {
		char **strAddr;
		strAddr = (char **) addr;
		if (*strAddr)
		  {
		    PUSH_OOP (_gst_string_new (*strAddr));
		    PRIM_SUCCEEDED;
		  }
		else
		  {
		    PUSH_OOP (_gst_nil_oop);
		    PRIM_SUCCEEDED;
		  }
	      }
	    case 9:
	      PUSH_OOP (*(OOP *) addr);
	      PRIM_SUCCEEDED;

	    case 10:
	      PUSH_OOP (FROM_C_INT (*(int *) addr));
	      PRIM_SUCCEEDED;

	    case 11:
	      PUSH_OOP (FROM_C_UINT (*(unsigned int *) addr));
	      PRIM_SUCCEEDED;
	    }

	}
      else
	{			/* non int type means use the type as
				   the type of the effective address */
	  /* It's an oddball case, but it does seem possible that oop3
	     could get GC'ed out of existence before it gets used,
	     since oop3 is not on the stack, and if
	     _gst_c_object_new_typed could cause a GC */
	  inc_ptr incPtr;

	  incPtr = INC_SAVE_POINTER ();
	  INC_ADD_OOP (oop3);
	  PUSH_OOP (_gst_c_object_new_typed (addr, oop3));
	  INC_RESTORE_POINTER (incPtr);
	  PRIM_SUCCEEDED;
	}
    }
  UNPOP (3);
  PRIM_FAILED;
}

/* set the value; 184 derefs first *
 * CObject at: byteOffset put: aValue type: aType */
static long
VMpr_CObject_atPut(int id,
		   volatile int numArgs,
		   OOP methodOOP)
{
  /* I don't think that this deals properly with setting the pointer
     value as opposed to setting the pointed-to value. */
  OOP oop1;
  OOP oop2;
  OOP oop3;
  OOP oop4;
  _gst_primitives_executed++;

  oop4 = POP_OOP ();
  oop3 = POP_OOP ();
  oop2 = POP_OOP ();
  oop1 = STACKTOP ();

  if (IS_INT (oop2))
    {
      char *addr;
      long arg2;
      arg2 = TO_INT (oop2);
      if (id == 184)
	addr = *(char **) COBJECT_VALUE (oop1);
      else
	addr = COBJECT_VALUE (oop1);

      addr += arg2;		/* compute effective address */
      if (IS_INT (oop4))
	{			/* int type spec means a scalar type */
	  long arg4;
	  arg4 = TO_INT (oop4);
	  switch (arg4)
	    {
	    case 0:		/* char */
	    case 1:		/* uchar */
	      if (IS_CLASS (oop3, _gst_char_class))
		{
		  *addr = CHAR_OOP_VALUE (oop3);
		  PRIM_SUCCEEDED;
		}
	      else if (IS_INT (oop3))
		{
		  *(char *) addr = (char) TO_INT (oop3);
		  PRIM_SUCCEEDED;
		}
	      break;

	    case 2:		/* short */
	    case 3:		/* ushort */
	      if (IS_INT (oop3))
		{
		  *(short *) addr = (short) TO_INT (oop3);
		  PRIM_SUCCEEDED;
		}
	      break;

	    case 4:		/* long */
	    case 5:		/* ulong */
	      if (IS_C_LONG (oop3))
		{
		  *(long *) addr = (long) TO_C_LONG (oop3);
		  PRIM_SUCCEEDED;
		}
	      break;

	    case 6:
	      {
		float *floatAddr;
		floatAddr = (float *) addr;
		if (IS_INT (oop3))
		  {
		    *floatAddr = (float) TO_INT (oop3);
		    PRIM_SUCCEEDED;
		  }
		else if (IS_CLASS (oop3, _gst_floatd_class))
		  {
		    *floatAddr = (float) FLOATD_OOP_VALUE (oop3);
		    PRIM_SUCCEEDED;
		  }
		else if (IS_CLASS (oop3, _gst_floate_class))
		  {
		    *floatAddr = (float) FLOATE_OOP_VALUE (oop3);
		    PRIM_SUCCEEDED;
		  }
		else if (IS_CLASS (oop3, _gst_floatq_class))
		  {
		    *floatAddr = (float) FLOATQ_OOP_VALUE (oop3);
		    PRIM_SUCCEEDED;
		  }
	      }
	      break;

	    case 7:		/* double */
	      {
		double *doubleAddr;
		doubleAddr = (double *) addr;
		if (IS_INT (oop3))
		  {
		    *doubleAddr = TO_INT (oop3);
		    PRIM_SUCCEEDED;
		  }
		else if (IS_CLASS (oop3, _gst_floatd_class))
		  {
		    *doubleAddr = FLOATD_OOP_VALUE (oop3);
		    PRIM_SUCCEEDED;
		  }
		else if (IS_CLASS (oop3, _gst_floate_class))
		  {
		    *doubleAddr = FLOATE_OOP_VALUE (oop3);
		    PRIM_SUCCEEDED;
		  }
		else if (IS_CLASS (oop3, _gst_floatq_class))
		  {
		    *doubleAddr = FLOATQ_OOP_VALUE (oop3);
		    PRIM_SUCCEEDED;
		  }
	      }
	      break;

	    case 8:		/* string */
	      {			/* note that this does not allow for
				   replacemnt in place */
		/* to replace in place, use replaceFrom: */
		char **strAddr;
		strAddr = (char **) addr;
		if (oop3 == _gst_nil_oop)
		  {
		    *strAddr = (char *) 0;
		    PRIM_SUCCEEDED;
		  }
		else
		  {
		    if (is_a_kind_of
			(OOP_CLASS (oop3), _gst_string_class))
		      {
			*strAddr = (char *) _gst_to_cstring (oop3);
			PRIM_SUCCEEDED;
		      }
		  }
		break;
	      }

	    case 9:
	      *(OOP *) addr = oop3;
	      PRIM_SUCCEEDED;

	    case 10:		/* int */
	    case 11:		/* uint */
	      if (IS_C_INT (oop3))
		{
		  *(int *) addr = (int) TO_C_INT (oop3);
		  PRIM_SUCCEEDED;
		}
	      break;

	    }
	}
      else
	{			/* non int type means use the type as
				   the type of the effective address */
	  *(PTR *) addr = COBJECT_VALUE (oop3);	/* IS THIS RIGHT?!? */
	  PRIM_SUCCEEDED;
	}
    }

  UNPOP (3);
  PRIM_FAILED;
}

/* CString replaceWith: aString */
static long
VMpr_CString_replaceWith(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();

  /* assumes the receiver is already pointing at an area of memory that 
     is the correct size; does not (re)allocate receiver's string at
     all. */

  if (IS_CLASS (oop2, _gst_string_class)
      || IS_CLASS (oop2, _gst_byte_array_class))
    {
      unsigned long srcLen;
      gst_uchar *dstBase, *srcBase;
      srcBase = STRING_OOP_CHARS (oop2);
      srcLen = NUM_INDEXABLE_FIELDS (oop2);

      dstBase = *(gst_uchar **) COBJECT_VALUE (oop1);
      memcpy (dstBase, srcBase, srcLen);
      dstBase[srcLen] = '\0';	/* since it's a CString type, we NUL
				   term it */
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* ByteArray class fromCdata: aCObject size: anInteger */
static long
VMpr_ByteArray_fromCData(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  OOP oop3;
  _gst_primitives_executed++;

  oop3 = POP_OOP ();
  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_INT (oop3))
    {
      long arg3 = TO_INT (oop3);
      OOP byteArrayOOP =
	_gst_byte_array_new (COBJECT_VALUE (oop2), arg3);
      SET_STACKTOP (byteArrayOOP);
      PRIM_SUCCEEDED;
    }
  UNPOP (2);
  PRIM_FAILED;
}

/* String class fromCdata: aCObject size: anInteger */
static long
VMpr_String_fromCData(int id,
		      volatile int numArgs,
		      OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  OOP oop3;
  _gst_primitives_executed++;

  oop3 = POP_OOP ();
  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (IS_INT (oop3))
    {
      long arg3 = TO_INT (oop3);
      OOP stringOOP =
	_gst_counted_string_new (COBJECT_VALUE (oop2), arg3);
      SET_STACKTOP (stringOOP);
      PRIM_SUCCEEDED;
    }
  UNPOP (2);
  PRIM_FAILED;
}

/* String asCdata: aCType
 * ByteArray asCdata: aCType */
static long
VMpr_ByteArray_asCData(int id,
		       volatile int numArgs,
		       OOP methodOOP)
{
  PTR data;
  int size;
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
#ifndef OPTIMIZE
  if ((IS_CLASS (oop1, _gst_string_class) && id == 203)
      || (IS_CLASS (oop1, _gst_byte_array_class) && id == 204))
    {
#endif
      if (is_a_kind_of (OOP_CLASS (oop2), _gst_c_type_class))
	{
	  size = NUM_INDEXABLE_FIELDS (oop1);
	  data = xmalloc (size);
	  if (data)
	    {
	      memcpy (data, OOP_TO_OBJ (oop1)->data, size);
	      SET_STACKTOP (_gst_c_object_new_typed (data, oop2));
	      PRIM_SUCCEEDED;
	    }
	}
#ifndef OPTIMIZE
    }
#endif
  UNPOP (1);
  PRIM_FAILED;
}


/* SystemDictionary byteCodeCounter */
static long
VMpr_SystemDictionary_byteCodeCounter(int id,
				      volatile int numArgs,
				      OOP methodOOP)
{
  _gst_primitives_executed++;
  SET_STACKTOP_INT (_gst_bytecode_counter);
  PRIM_SUCCEEDED;
}

/* SystemDictionary debug */
static long
VMpr_SystemDictionary_debug(int id,
			    volatile int numArgs,
			    OOP methodOOP)
{
  _gst_primitives_executed++;
  _gst_debug ();		/* used to allow gdb to stop based on * 
				   Smalltalk execution paths. */
  PRIM_SUCCEEDED;
}

/* Object isReadOnly */

static long
VMpr_Object_isReadOnly(int id,
		       volatile int numArgs,
		       OOP methodOOP)
{
  OOP oop1;
  _gst_primitives_executed++;

  oop1 = STACKTOP ();
  SET_STACKTOP_BOOLEAN (IS_OOP_READONLY (oop1));
  PRIM_SUCCEEDED;
}

/* Object makeReadOnly: */
static long
VMpr_Object_makeReadOnly(int id,
			 volatile int numArgs,
			 OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = STACKTOP ();
  if (oop2 == _gst_true_oop)
    {
      MAKE_OOP_READONLY (oop1, true);
      PRIM_SUCCEEDED;
    }
  else if (oop2 == _gst_false_oop)
    {
      MAKE_OOP_READONLY (oop1, false);
      PRIM_SUCCEEDED;
    }
  else
    {
      UNPOP (1);
      PRIM_FAILED;
    }
}

/* Behavior compileString: aString */

static long
VMpr_Behavior_compileString(int id,
			    volatile int numArgs,
			    OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  _gst_compile_code = true;	/* tell the lexer we do internal
				   compiles */
  if (IS_CLASS (oop2, _gst_string_class))
    _gst_push_smalltalk_string (oop2);
  else
    _gst_push_stream_oop (oop2);

  _gst_set_compilation_class (oop1);
  _gst_parse_stream ();
  _gst_pop_stream (true);
  PUSH_OOP (_gst_latest_compiled_method);
  PRIM_SUCCEEDED;
}

/* Behavior compileString: aString ifError: aBlock */
static long
VMpr_Behavior_compileStringIfError(int id,
				   volatile int numArgs,
				   OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  OOP oop3;
  _gst_primitives_executed++;

  oop3 = POP_OOP ();
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();
  if (IS_CLASS (oop3, _gst_block_closure_class))
    {
      mst_Boolean oldReportErrors = _gst_report_errors;

      if (oldReportErrors)
	{
	  /* only clear out these guys on first transition */
	  _gst_first_error_str = _gst_first_error_file = NULL;
	}
      _gst_report_errors = false;
      _gst_compile_code = true;	/* tell the lexer we do internal
				   compiles */

      if (IS_CLASS (oop2, _gst_string_class))
	_gst_push_smalltalk_string (oop2);
      else
	_gst_push_stream_oop (oop2);

      _gst_set_compilation_class (oop1);
      _gst_parse_stream ();
      _gst_pop_stream (true);

      if (_gst_first_error_str != NULL)
	{
	  PUSH_OOP (oop3);	/* block context */
	  if (_gst_first_error_file != NULL)
	    {
	      PUSH_OOP (_gst_string_new (_gst_first_error_file));
	      xfree (_gst_first_error_file);
	    }
	  else
	    PUSH_OOP (_gst_nil_oop);

	  PUSH_INT (_gst_first_error_line);
	  PUSH_OOP (_gst_string_new (_gst_first_error_str));
	  xfree (_gst_first_error_str);
	  _gst_first_error_str = _gst_first_error_file = NULL;
	  _gst_report_errors = oldReportErrors;
	  if (send_block_value (3))
	    PRIM_FAILED;
	  else
	    PRIM_SUCCEEDED_RELOAD_IP;
	}
      else
	{
	  _gst_report_errors = oldReportErrors;
	  PUSH_OOP (_gst_latest_compiled_method);
	}
      PRIM_SUCCEEDED;
    }
  UNPOP (3);
  PRIM_FAILED;
}

/* CFunctionDescriptor for: funcNameString
            returning: returnTypeSymbol
	    withArgs: argsArray */
static long
VMpr_CFuncDescriptor_create(int id,
			    volatile int numArgs,
			    OOP methodOOP)
{
  OOP oop1;
  OOP oop2;
  OOP oop3;
  OOP oop4;
  _gst_primitives_executed++;

  oop4 = POP_OOP ();
  oop3 = POP_OOP ();
  oop2 = POP_OOP ();
  oop1 = POP_OOP ();

  if (IS_CLASS (oop2, _gst_string_class)
      && (IS_CLASS (oop3, _gst_symbol_class)
	  || is_a_kind_of (OOP_CLASS (oop3), _gst_c_type_class))
      && (IS_CLASS (oop4, _gst_array_class)
	  || IS_CLASS (oop4, _gst_undefined_object_class)))
    {
      PUSH_OOP (_gst_make_descriptor (oop2, oop3, oop4));
      PRIM_SUCCEEDED;
    }
  UNPOP (4);
  PRIM_FAILED;
}

/* Object snapshot: aString */
static long
VMpr_ObjectMemory_snapshot(int id,
			   volatile int numArgs,
			   OOP methodOOP)
{
  char *fileName;
  OOP oop2;
  _gst_primitives_executed++;

  oop2 = POP_OOP ();
  if (IS_CLASS (oop2, _gst_string_class))
    {
      fileName = _gst_to_cstring (oop2);
      _gst_save_to_file (fileName);
      xfree (fileName);
      PRIM_SUCCEEDED;
    }
  UNPOP (1);
  PRIM_FAILED;
}

/* Object basicPrint */
static long
VMpr_Object_basicPrint(int id,
		       volatile int numArgs,
		       OOP methodOOP)
{
  _gst_primitives_executed++;
  printf ("Object: %O", STACKTOP ());
  fflush (stdout);
  PRIM_SUCCEEDED;
}

/* Object makeWeak */
static long
VMpr_Object_makeWeak(int id,
		     volatile int numArgs,
		     OOP methodOOP)
{
  _gst_primitives_executed++;
  MAKE_OOP_WEAK (STACKTOP ());
  PRIM_SUCCEEDED;
}

/* FileDescriptor>>#fileOp..., variadic */

static long
VMpr_FileDescriptor_fileOp(int id,
			   volatile int numArgs,
			   OOP methodOOP)
{
  char *fileName, *fileName2, *fileMode;
  gst_file_stream fileStream;
  int fd;
  OOP oop1;
  OOP *oopVec = alloca (numArgs * sizeof (OOP));
  int i;
  long arg1;

  /* Ordering of file operations must match that used in
     FileStream.st */
  enum
  {
    PRIM_OPEN_FILE,		/* open:mode: */
    PRIM_CLOSE_FILE,		/* no args */
    PRIM_PUT_CHARS,		/* data:from:to: */
    PRIM_GET_CHARS,		/* data:from:to: */
    PRIM_FSEEK_SET,		/* position: */
    PRIM_FTELL,			/* no args */
    PRIM_FEOF,			/* no args */
    PRIM_OPEN_PIPE,		/* open:mode: */
    PRIM_FSEEK_CUR,		/* skip: */
    PRIM_FSIZE,			/* no args */
    PRIM_FTRUNCATE,		/* no args */
    PRIM_FILEIN,		/* no args */
    PRIM_FILEIN_AT,		/* line:file:charPos: */
    PRIM_SYNC_POLL,		/* read/write/exception */
    PRIM_ASYNC_POLL,		/* operation:semaphore: */
    PRIM_IS_PIPE,		/* no args */
    PRIM_MK_TEMP                /* base: */
  };

  _gst_primitives_executed++;

  for (i = numArgs; --i >= 0;)
    oopVec[i] = POP_OOP ();

  oop1 = STACKTOP ();
  if (!IS_INT (oopVec[0]))
    {
      UNPOP (numArgs);
      PRIM_FAILED;
    }

  arg1 = TO_INT (oopVec[0]);
  switch (arg1) {
  case PRIM_OPEN_FILE:
  case PRIM_OPEN_PIPE:
    {
      struct stat st;
      int is_pipe;

      /* open: fileName[1] mode: mode[2] or popen: command[1] dir:
         direction[2] */
      fileName = _gst_to_cstring (oopVec[1]);
      fileMode = _gst_to_cstring (oopVec[2]);
      if (arg1 == PRIM_OPEN_FILE)
        {
          fd = _gst_open_file ((char *) fileName, (char *) fileMode);
          fstat (fd, &st);
          is_pipe =
            S_ISFIFO(st.st_mode) ? true :
            S_ISREG(st.st_mode) && st.st_size > 0 ? false : -1;
        }
      else
        {
          fd = _gst_open_pipe (fileName, fileMode);
          is_pipe = true;
        }

      xfree (fileName);
      xfree (fileMode);
      if (fd < 0)
        {
          UNPOP (numArgs);
          PRIM_FAILED;
        }


      _gst_set_file_stream_file (oop1, fd, oopVec[1],
                                 is_pipe,
                                 fcntl (fd, F_GETFL), false);
    }
    
    PRIM_SUCCEEDED;

  case PRIM_MK_TEMP:
    /* open: fileName[1] mode: mode[2] or popen: command[1] dir:
       direction[2] */
    fileName = _gst_to_cstring (oopVec[1]);
    asprintf (&fileName2, "%sXXXXXX", fileName);
    fd = mkstemp ((char *) fileName2);

    xfree (fileName);
    if (fd < 0)
      {
	xfree (fileName2);
	UNPOP (numArgs);
	PRIM_FAILED;
      }
    
    _gst_set_file_stream_file (oop1, fd, _gst_string_new (fileName2),
			       arg1 == PRIM_OPEN_PIPE,
			       fcntl (fd, F_GETFL), false);

    xfree (fileName2);
    PRIM_SUCCEEDED;
  }

  fileStream = (gst_file_stream) OOP_TO_OBJ (oop1);
  if (IS_NIL (fileStream->file))
    {
      UNPOP (numArgs);
      PRIM_FAILED;
    }
  fd = TO_INT (fileStream->file);
  switch (arg1)
    {

    case PRIM_CLOSE_FILE:	/* FileStream close */
      SET_STACKTOP_INT (close (fd));
      PRIM_SUCCEEDED;

    case PRIM_FSEEK_SET:		/* like gst_file_stream position:
				   position */
      if (IS_OFF_T (oopVec[1]) &&
	  lseek (fd, TO_OFF_T (oopVec[1]), SEEK_SET) < 0)
	{
	  errno = 0;
	  break;
	}
      else
	PRIM_SUCCEEDED;

    case PRIM_FTELL:		/* FileStream position */
      {
	off_t off = lseek(fd, 0, SEEK_CUR);
        if (off < 0)
	  {
	    errno = 0;
	    break;
	  }

	SET_STACKTOP (FROM_OFF_T (off));
	PRIM_SUCCEEDED;
      }

    case PRIM_FEOF:
      {				/* FileStream atEnd */
	off_t oldPos;
	oldPos = lseek (fd, 0, SEEK_CUR);
	SET_STACKTOP_BOOLEAN (oldPos >= 0
			      && lseek (fd, 0, SEEK_END) == oldPos);
	lseek (fd, oldPos, SEEK_SET);
	errno = 0;
	PRIM_SUCCEEDED;
      }

    case PRIM_FSIZE:
      {
	struct stat statBuf;
	if (fstat (fd, &statBuf) < 0)
	  {
	    errno = 0;
	    break;
	  }
	SET_STACKTOP_INT (statBuf.st_size);
	PRIM_SUCCEEDED;
      }

    case PRIM_PUT_CHARS:
      if (is_a_kind_of (OOP_CLASS (oopVec[1]), _gst_string_class))
	{
	  char *data = STRING_OOP_CHARS (oopVec[1]);
	  long from = TO_INT (oopVec[2]);
	  long to = TO_INT (oopVec[3]);
	  int result;
	  if (to >= from - 1
	      && from > 0 && to <= NUM_INDEXABLE_FIELDS (oopVec[1]))
	    {
	      result =
		_gst_full_write (fd, data + from - 1, to - from + 1);
	      if (result >= 0)
		{
		  SET_STACKTOP_INT (result);
		  PRIM_SUCCEEDED;
		}
	    }
	}
      break;

    case PRIM_GET_CHARS:		/* only works for strings */
      if (is_a_kind_of (OOP_CLASS (oopVec[1]), _gst_string_class))
	{
	  char *data = STRING_OOP_CHARS (oopVec[1]);
	  long from = TO_INT (oopVec[2]);
	  long to = TO_INT (oopVec[3]);
	  int result;
	  if (to >= from - 1
	      && from > 0 && to <= NUM_INDEXABLE_FIELDS (oopVec[1]))
	    {
	      do
		{
		  errno = 0;
		  result = read (fd, data + from - 1, to - from + 1);
		} 
	      while ((result == -1) && (errno == EINTR));

	      if (result >= 0)
		{
		  SET_STACKTOP_INT (result);
		  PRIM_SUCCEEDED;
		}
	    }
	}
      break;

    case PRIM_FTRUNCATE:
      {
	off_t pos;
	pos = lseek (fd, 0, SEEK_CUR);
	if (pos < 0)
	  break;

	ftruncate (fd, pos);
	PRIM_SUCCEEDED;
      }

    case PRIM_FSEEK_CUR:		/* FileDescriptor skip: */
      if (IS_OFF_T (oopVec[1]) &&
	  lseek (fd, TO_OFF_T (oopVec[1]), SEEK_CUR) < 0)
	break;
      else
	PRIM_SUCCEEDED;

    case PRIM_SYNC_POLL:
      {
	int result;

	result = _gst_sync_file_polling (fd, TO_INT (oopVec[1]));
	if (result >= 0)
	  {
	    SET_STACKTOP_INT (result);
	    PRIM_SUCCEEDED;
	  }
      }
      break;

    case PRIM_ASYNC_POLL:
      {
	int result;

	result =
	  _gst_async_file_polling (fd, TO_INT (oopVec[1]), oopVec[2]);
	if (result >= 0)
	  PRIM_SUCCEEDED;
      }
      break;

    case PRIM_IS_PIPE:
      {
        off_t result;

        result = lseek (fd, 0, SEEK_END);
        if (result == -1)
          {
            if (errno == ESPIPE || errno == EINVAL)
              {
                SET_STACKTOP (_gst_true_oop);
                errno = 0;
              }
            else
              PRIM_FAILED;
          }
        else
          {
            lseek (fd, result, SEEK_SET);
            SET_STACKTOP (_gst_false_oop);
          }

        PRIM_SUCCEEDED;
      }
      break;

    case PRIM_FILEIN:
      {
	fileName = _gst_to_cstring (fileStream->name);
	if (_gst_file_is_readable (fileName))
	  {
	    _gst_push_unix_file (fd, fileName);
	    _gst_use_undeclared++;
	    _gst_parse_stream ();
	    _gst_use_undeclared--;
	    _gst_pop_stream (false);	/* we didn't open it, so we
					   don't close it */
	    PRIM_SUCCEEDED;
	  }
	xfree (fileName);
	break;
      }

      /* FileStream fileInLine: lineNum fileName: aString at:
         charPosInt */
    case PRIM_FILEIN_AT:
      {
	char *fileName, *realFileName;

	fileName = _gst_to_cstring (fileStream->name);
	realFileName = NULL;
	if (_gst_file_is_readable (fileName))
	  {
	    if (IS_INT (oopVec[1])
		&& (IS_NIL (oopVec[2]) ||
		    (IS_CLASS (oopVec[2], _gst_string_class)
		     && IS_INT (oopVec[3]))))
	      {
		long arg2;
		long arg4;
		arg2 = TO_INT (oopVec[1]);
		if (!IS_NIL (oopVec[2]))
		  {
		    arg4 = TO_INT (oopVec[3]);
		    realFileName = _gst_to_cstring (oopVec[2]);
		  }
		else
		  arg4 = 0;

		_gst_push_unix_file (fd, fileName);
		_gst_set_stream_info (arg2, realFileName, arg4);
		_gst_parse_stream ();
		_gst_pop_stream (false);	/* we didn't open it,
						   so we don't close it 
						 */
		PRIM_SUCCEEDED;
	      }
	  }
	xfree (fileName);
	if (realFileName)
	  xfree (realFileName);

	break;
      }

    }

  UNPOP (numArgs);
  PRIM_FAILED;
}

/* C callout primitive. */

static long
VMpr_callout(int id,
	     volatile int numArgs,
	     OOP methodOOP)
{
  mst_Boolean oldInCCode;
  volatile mst_Boolean result = false;	/* to please GCC */
#ifdef STACK_JMPBUFS
  interp_jmp_buf localJmpBuf;
#endif

  _gst_primitives_executed++;

#ifndef USE_JIT_TRANSLATION
  /* Fail if we were put in the caches used by #at:, #at:put:, or
     #size. */
  if (methodOOP == NULL)
    return (true);
#endif

#ifdef STACK_JMPBUFS
  localJmpBuf.old = c_callout_jmp_buf;
  c_callout_jmp_buf = &localJmpBuf;
#else
  c_callout_jmp_buf->old = c_callout_jmp_buf;
#endif

  in_interpreter = false;
  oldInCCode = in_ccode;
  in_ccode = true;
  if (setjmp (c_callout_jmp_buf->jmpBuf) == 0)
    result = _gst_invoke_croutine (numArgs, methodOOP);

  in_ccode = oldInCCode;
  in_interpreter = !in_ccode;
  c_callout_jmp_buf = c_callout_jmp_buf->old;
  if (result)
    PRIM_SUCCEEDED;
  else
    PRIM_FAILED;
}


static long
VMpr_Object_addToBeFinalized(int id,
			     volatile int numArgs,
			     OOP methodOOP)
{
  _gst_primitives_executed++;
  MARK_OOP_TO_FINALIZE (STACKTOP (), true);
  PRIM_SUCCEEDED;
}

static long
VMpr_Object_removeToBeFinalized(int id,
				volatile int numArgs,
				OOP methodOOP)
{
  _gst_primitives_executed++;
  MARK_OOP_TO_FINALIZE (STACKTOP (), false);
  PRIM_SUCCEEDED;
}

/* Namespace current: aNamespace */
static long
VMpr_Namespace_setCurrent(int id,
			  volatile int numArgs,
			  OOP methodOOP)
{
  _gst_primitives_executed++;
  _gst_current_namespace = STACKTOP ();

  /* Always fail */
  PRIM_FAILED;
}

/* brd Sun Oct 24 15:36:15 PDT 1993 */
/* added primitives 256-264 to support the browser */
static long
VMpr_ObjectMemory_compact(int id,
			  volatile int numArgs,
			  OOP methodOOP)
{
  _gst_primitives_executed++;
  _gst_scavenge ();
  PRIM_SUCCEEDED;
}

static long
VMpr_HOLE(int id,
	  volatile int numArgs,
	  OOP methodOOP)
{
  _gst_primitives_executed++;
  _gst_errorf ("Unhandled primitive operation %d", id);

  UNPOP (numArgs);
  PRIM_FAILED;
}

void
_gst_init_primitives()
{
  int i;
  for (i = 0; i < 1023; i++)
    {
      _gst_primitive_table[i].name = NULL;
      _gst_primitive_table[i].attributes = PRIM_FAIL;
      _gst_primitive_table[i].id = i;
      _gst_primitive_table[i].func = VMpr_HOLE;
    }

#define DEF_PRIM(primID, primAttr, primFunc, primName) \
  (_gst_primitive_table[i].name = (primName), \
   _gst_primitive_table[i].attributes = (primAttr), \
   _gst_primitive_table[i].id = (primID), \
   _gst_primitive_table[i++].func = (primFunc))

#define DEF_PRIM2(primID, primAttr, primFunc) \
  (_gst_primitive_table[i].name = #primFunc, \
   _gst_primitive_table[i].attributes = (primAttr), \
   _gst_primitive_table[i].id = (primID), \
   _gst_primitive_table[i++].func = (primFunc))

  i = 0;

  /* I'm still using the pre-2.0 primitive numbers as identifiers.
     It might yield a microscopic performance improvement to use
     0..n for functions that implement more than one primitive;
     note that the current implementation of inlined primitives
     in xlat.c uses the id to distinguish the code to emit. */

  /* SmallInteger primitives */
  DEF_PRIM2 (1, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_plus);
  DEF_PRIM2 (2, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_minus);
  DEF_PRIM2 (3, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_lt);
  DEF_PRIM2 (4, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_gt);
  DEF_PRIM2 (5, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_le);
  DEF_PRIM2 (6, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_ge);
  DEF_PRIM2 (7, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_eq);
  DEF_PRIM2 (8, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_ne);
  DEF_PRIM2 (9, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_times);
  DEF_PRIM2 (10, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_divide);
  DEF_PRIM2 (11, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_modulo);
  DEF_PRIM2 (12, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_intDiv);
  DEF_PRIM2 (13, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_quo);
  DEF_PRIM2 (14, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_bitAnd);
  DEF_PRIM2 (15, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_bitOr);
  DEF_PRIM2 (16, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_bitXor);
  DEF_PRIM2 (17, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_bitShift);
  DEF_PRIM2 (18, PRIM_CHECKS_RECEIVER, VMpr_SmallInteger_scramble);
  DEF_PRIM2 (40, PRIM_CHECKS_RECEIVER, VMpr_SmallInteger_asFloatD);
  DEF_PRIM2 (40, PRIM_CHECKS_RECEIVER, VMpr_SmallInteger_asFloatE);
  DEF_PRIM2 (40, PRIM_CHECKS_RECEIVER, VMpr_SmallInteger_asFloatQ);

  /* LargeInteger primitives */
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_eq);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_ne);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_lt);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_le);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_gt);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_ge);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_times);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_intDiv);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_rem);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_quo);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_modulo);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_negated);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_bitAnd);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_bitInvert);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_bitOr);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_bitXor);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_bitShift);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_plus);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_minus);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_gcd);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_negated);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_asFloatD);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_asFloatE);
  DEF_PRIM2 (0, PRIM_USES_GMP, VMpr_LargeInteger_asFloatQ);

  /* FloatD primitives */
  DEF_PRIM (41, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatD_arith, "VMpr_FloatD_plus");
  DEF_PRIM (42, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatD_arith, "VMpr_FloatD_minus");
  DEF_PRIM (43, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatD_arith, "VMpr_FloatD_lt");
  DEF_PRIM (44, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatD_arith, "VMpr_FloatD_gt");
  DEF_PRIM (45, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatD_arith, "VMpr_FloatD_le");
  DEF_PRIM (46, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatD_arith, "VMpr_FloatD_ge");
  DEF_PRIM (47, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatD_arith, "VMpr_FloatD_eq");
  DEF_PRIM (48, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatD_arith, "VMpr_FloatD_ne");
  DEF_PRIM (49, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatD_arith, "VMpr_FloatD_times");
  DEF_PRIM (50, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatD_arith, "VMpr_FloatD_divide");
  DEF_PRIM2 (51, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatD_truncated);
  DEF_PRIM2 (52, PRIM_CHECKS_RECEIVER, VMpr_FloatD_fractionPart);
  DEF_PRIM2 (53, PRIM_CHECKS_RECEIVER | PRIM_RETURN_SMALL_SMALLINTEGER, VMpr_FloatD_exponent);
  DEF_PRIM2 (54, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatD_timesTwoPower);
  DEF_PRIM2 (55, PRIM_CHECKS_RECEIVER, VMpr_FloatD_asFloatE);
  DEF_PRIM2 (56, PRIM_CHECKS_RECEIVER, VMpr_FloatD_asFloatQ);

  /* FloatE primitives */
  DEF_PRIM (41, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatE_arith, "VMpr_FloatE_plus");
  DEF_PRIM (42, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatE_arith, "VMpr_FloatE_minus");
  DEF_PRIM (43, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatE_arith, "VMpr_FloatE_lt");
  DEF_PRIM (44, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatE_arith, "VMpr_FloatE_gt");
  DEF_PRIM (45, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatE_arith, "VMpr_FloatE_le");
  DEF_PRIM (46, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatE_arith, "VMpr_FloatE_ge");
  DEF_PRIM (47, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatE_arith, "VMpr_FloatE_eq");
  DEF_PRIM (48, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatE_arith, "VMpr_FloatE_ne");
  DEF_PRIM (49, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatE_arith, "VMpr_FloatE_times");
  DEF_PRIM (50, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatE_arith, "VMpr_FloatE_divide");
  DEF_PRIM2 (51, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatE_truncated);
  DEF_PRIM2 (52, PRIM_CHECKS_RECEIVER, VMpr_FloatE_fractionPart);
  DEF_PRIM2 (53, PRIM_CHECKS_RECEIVER | PRIM_RETURN_SMALL_SMALLINTEGER, VMpr_FloatE_exponent);
  DEF_PRIM2 (54, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatE_timesTwoPower);
  DEF_PRIM2 (55, PRIM_CHECKS_RECEIVER, VMpr_FloatE_asFloatD);
  DEF_PRIM2 (56, PRIM_CHECKS_RECEIVER, VMpr_FloatE_asFloatQ);

  /* FloatQ primitives */
  DEF_PRIM (41, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatQ_arith, "VMpr_FloatQ_plus");
  DEF_PRIM (42, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatQ_arith, "VMpr_FloatQ_minus");
  DEF_PRIM (43, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatQ_arith, "VMpr_FloatQ_lt");
  DEF_PRIM (44, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatQ_arith, "VMpr_FloatQ_gt");
  DEF_PRIM (45, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatQ_arith, "VMpr_FloatQ_le");
  DEF_PRIM (46, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatQ_arith, "VMpr_FloatQ_ge");
  DEF_PRIM (47, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatQ_arith, "VMpr_FloatQ_eq");
  DEF_PRIM (48, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatQ_arith, "VMpr_FloatQ_ne");
  DEF_PRIM (49, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatQ_arith, "VMpr_FloatQ_times");
  DEF_PRIM (50, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatQ_arith, "VMpr_FloatQ_divide");
  DEF_PRIM2 (51, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatQ_truncated);
  DEF_PRIM2 (52, PRIM_CHECKS_RECEIVER, VMpr_FloatQ_fractionPart);
  DEF_PRIM2 (53, PRIM_CHECKS_RECEIVER | PRIM_RETURN_SMALL_SMALLINTEGER, VMpr_FloatQ_exponent);
  DEF_PRIM2 (54, PRIM_SUCCEED | PRIM_FAIL, VMpr_FloatQ_timesTwoPower);
  DEF_PRIM2 (55, PRIM_CHECKS_RECEIVER, VMpr_FloatQ_asFloatD);
  DEF_PRIM2 (56, PRIM_CHECKS_RECEIVER, VMpr_FloatQ_asFloatE);

  /* Float primitives */
  DEF_PRIM (168, PRIM_SUCCEED | PRIM_FAIL, VMpr_Float_ceiling_floor, "VMpr_Float_ceil");
  DEF_PRIM (169, PRIM_SUCCEED | PRIM_FAIL, VMpr_Float_ceiling_floor, "VMpr_Float_floor");
  DEF_PRIM2 (160, PRIM_SUCCEED | PRIM_FAIL, VMpr_Float_exp);
  DEF_PRIM2 (161, PRIM_SUCCEED | PRIM_FAIL, VMpr_Float_ln);
  DEF_PRIM2 (164, PRIM_SUCCEED | PRIM_FAIL, VMpr_Float_pow);
  DEF_PRIM2 (166, PRIM_SUCCEED | PRIM_FAIL, VMpr_Float_sqrt);
  DEF_PRIM2 (176, PRIM_SUCCEED | PRIM_FAIL, VMpr_Float_tan);
  DEF_PRIM2 (177, PRIM_SUCCEED | PRIM_FAIL, VMpr_Float_sin);
  DEF_PRIM2 (178, PRIM_SUCCEED | PRIM_FAIL, VMpr_Float_cos);
  DEF_PRIM2 (179, PRIM_SUCCEED | PRIM_FAIL, VMpr_Float_arcTan);
  DEF_PRIM2 (180, PRIM_SUCCEED | PRIM_FAIL, VMpr_Float_arcSin);
  DEF_PRIM2 (181, PRIM_SUCCEED | PRIM_FAIL, VMpr_Float_arcCos);

  /* Object accessing primitives */
  DEF_PRIM2 (60, PRIM_SUCCEED | PRIM_FAIL | PRIM_INLINED, VMpr_Object_basicAt);
  DEF_PRIM2 (61, PRIM_SUCCEED | PRIM_FAIL | PRIM_INLINED, VMpr_Object_basicAtPut);
  DEF_PRIM2 (62, PRIM_SUCCEED | PRIM_RETURN_SMALL_SMALLINTEGER | PRIM_INLINED, VMpr_Object_basicSize);
  DEF_PRIM2 (63, PRIM_SUCCEED | PRIM_FAIL | PRIM_INLINED, VMpr_String_basicAt);
  DEF_PRIM2 (64, PRIM_SUCCEED | PRIM_FAIL | PRIM_INLINED, VMpr_String_basicAtPut);
  DEF_PRIM2 (73, PRIM_SUCCEED | PRIM_FAIL | PRIM_INLINED, VMpr_Object_instVarAt);
  DEF_PRIM2 (74, PRIM_SUCCEED | PRIM_FAIL | PRIM_INLINED, VMpr_Object_instVarAtPut);
  DEF_PRIM2 (75, PRIM_SUCCEED | PRIM_FAIL, VMpr_Object_hash);
  DEF_PRIM2 (104, PRIM_CHECKS_RECEIVER, VMpr_String_hash);
  DEF_PRIM2 (105, PRIM_SUCCEED | PRIM_FAIL, VMpr_ByteArray_replaceFromToWithStartingAt);
  DEF_PRIM2 (134, PRIM_SUCCEED | PRIM_FAIL, VMpr_Symbol_intern);

  /* Class creation and meta-object protocol primitives */
  DEF_PRIM2 (68, PRIM_SUCCEED, VMpr_CompiledBlock_create);
  DEF_PRIM2 (69, PRIM_SUCCEED, VMpr_CompiledMethod_create);
  DEF_PRIM2 (70, PRIM_SUCCEED, VMpr_Object_shallowCopy);
  DEF_PRIM2 (70, PRIM_SUCCEED | PRIM_FAIL | PRIM_INLINED, VMpr_Behavior_basicNew);
  DEF_PRIM2 (71, PRIM_SUCCEED | PRIM_FAIL | PRIM_INLINED, VMpr_Behavior_basicNewColon);
  DEF_PRIM2 (77, PRIM_SUCCEED | PRIM_FAIL, VMpr_Behavior_someInstance);
  DEF_PRIM2 (78, PRIM_SUCCEED | PRIM_FAIL, VMpr_Object_nextInstance);
  DEF_PRIM2 (72, PRIM_SUCCEED | PRIM_FAIL, VMpr_Object_become);
  DEF_PRIM2 (91, PRIM_SUCCEED | PRIM_FAIL, VMpr_Object_changeClassTo);
  DEF_PRIM2 (76, PRIM_SUCCEED | PRIM_FAIL, VMpr_SmallInteger_asObject);
  DEF_PRIM2 (110, PRIM_SUCCEED | PRIM_INLINED, VMpr_Object_identity);
  DEF_PRIM2 (111, PRIM_SUCCEED, VMpr_Object_class);
  DEF_PRIM2 (170, PRIM_SUCCEED | PRIM_FAIL, VMpr_Behavior_basicNewFixed);
  DEF_PRIM2 (171, PRIM_SUCCEED | PRIM_FAIL, VMpr_Behavior_basicNewFixedColon);
  DEF_PRIM2 (172, PRIM_SUCCEED | PRIM_FAIL, VMpr_Behavior_makeFixed);
  DEF_PRIM2 (253, PRIM_SUCCEED, VMpr_Object_makeWeak);
  DEF_PRIM2 (233, PRIM_SUCCEED, VMpr_Object_isReadOnly);
  DEF_PRIM2 (234, PRIM_SUCCEED | PRIM_FAIL, VMpr_Object_makeReadOnly);
  DEF_PRIM2 (256, PRIM_SUCCEED, VMpr_Object_addToBeFinalized);
  DEF_PRIM2 (257, PRIM_SUCCEED, VMpr_Object_removeToBeFinalized);

  /* Virtual machine primitives */
  DEF_PRIM2 (89, PRIM_SUCCEED, VMpr_Behavior_flushCache);
  DEF_PRIM2 (90, PRIM_SUCCEED, VMpr_CompiledCode_discardTranslation);
  DEF_PRIM2 (0, PRIM_CHECKS_RECEIVER | PRIM_RELOAD_IP, VMpr_ContextPart_continue);
  DEF_PRIM2 (80, PRIM_SUCCEED, VMpr_BlockClosure_blockCopy);
  DEF_PRIM2 (81, PRIM_FAIL | PRIM_RELOAD_IP | PRIM_CACHE_NEW_IP, VMpr_BlockClosure_value);
  DEF_PRIM2 (81, PRIM_FAIL | PRIM_RELOAD_IP, VMpr_BlockClosure_valueAndResumeOnUnwind);
  DEF_PRIM2 (82, PRIM_FAIL | PRIM_RELOAD_IP, VMpr_BlockClosure_valueWithArguments);
  DEF_PRIM2 (83, PRIM_FAIL | PRIM_RELOAD_IP, VMpr_Object_perform);
  DEF_PRIM2 (84, PRIM_FAIL | PRIM_RELOAD_IP, VMpr_Object_performWithArguments);

  /* Process handling primitives */
  DEF_PRIM2 (85, PRIM_SUCCEED | PRIM_CHECK_INTERRUPT, VMpr_Semaphore_signal);
  DEF_PRIM2 (86, PRIM_SUCCEED | PRIM_CHECK_INTERRUPT, VMpr_Semaphore_wait);
  DEF_PRIM2 (87, PRIM_SUCCEED | PRIM_FAIL | PRIM_CHECK_INTERRUPT, VMpr_Process_resume);
  DEF_PRIM2 (88, PRIM_SUCCEED | PRIM_CHECK_INTERRUPT, VMpr_Process_yield);
  DEF_PRIM2 (100, PRIM_SUCCEED | PRIM_FAIL | PRIM_CHECK_INTERRUPT, VMpr_Processor_signalAtMilliseconds);
  DEF_PRIM2 (101, PRIM_SUCCEED, VMpr_Processor_isTimeoutProgrammed);

  /* Time primitives */
  DEF_PRIM2 (96, PRIM_SUCCEED, VMpr_Time_timezoneBias);
  DEF_PRIM2 (97, PRIM_SUCCEED, VMpr_Time_timezone);
  DEF_PRIM2 (98, PRIM_SUCCEED, VMpr_Time_secondClock);
  DEF_PRIM2 (99, PRIM_SUCCEED, VMpr_Time_millisecondClock);

  /* Bootstrapping primitives */
  DEF_PRIM2 (128, PRIM_SUCCEED, VMpr_Dictionary_at);
  DEF_PRIM2 (129, PRIM_SUCCEED, VMpr_Dictionary_atPut);
  DEF_PRIM2 (135, PRIM_SUCCEED, VMpr_Dictionary_new);
  DEF_PRIM (130, PRIM_SUCCEED, VMpr_Object_bootstrapException, "VMpr_Object_bootstrapDNU");
  DEF_PRIM (131, PRIM_SUCCEED, VMpr_Object_bootstrapException, "VMpr_Object_bootstrapError");

  /* Character primitives */
  DEF_PRIM2 (132, PRIM_SUCCEED | PRIM_FAIL, VMpr_Character_create);
  DEF_PRIM2 (133, PRIM_SUCCEED, VMpr_Character_value);

  /* C interface primitives */
  DEF_PRIM2 (249, PRIM_SUCCEED | PRIM_FAIL, VMpr_CFuncDescriptor_create);
  DEF_PRIM2 (255, PRIM_SUCCEED | PRIM_FAIL, VMpr_callout);
  DEF_PRIM2 (144, PRIM_SUCCEED | PRIM_FAIL, VMpr_CObject_alloc);
  DEF_PRIM2 (165, PRIM_CHECKS_RECEIVER, VMpr_CObject_free);
  DEF_PRIM2 (158, PRIM_SUCCEED | PRIM_FAIL, VMpr_CObject_allocType);
  DEF_PRIM (182, PRIM_SUCCEED | PRIM_FAIL, VMpr_CObject_at, "VMpr_CObject_derefAt");
  DEF_PRIM (183, PRIM_SUCCEED | PRIM_FAIL, VMpr_CObject_at, "VMpr_CObject_at");
  DEF_PRIM (184, PRIM_SUCCEED | PRIM_FAIL, VMpr_CObject_atPut, "VMpr_CObject_derefAtPut");
  DEF_PRIM (185, PRIM_SUCCEED | PRIM_FAIL, VMpr_CObject_atPut, "VMpr_CObject_atPut");
  DEF_PRIM2 (188, PRIM_SUCCEED | PRIM_FAIL, VMpr_CString_replaceWith);
  DEF_PRIM2 (200, PRIM_SUCCEED | PRIM_FAIL, VMpr_ByteArray_fromCData);
  DEF_PRIM2 (201, PRIM_SUCCEED | PRIM_FAIL, VMpr_String_fromCData);
  DEF_PRIM (203, PRIM_CHECKS_RECEIVER, VMpr_ByteArray_asCData, "VMpr_String_asCData");
  DEF_PRIM (204, PRIM_CHECKS_RECEIVER, VMpr_ByteArray_asCData, "VMpr_ByteArray_asCData");
  DEF_PRIM2 (138, PRIM_SUCCEED | PRIM_FAIL, VMpr_Memory_addressOfOOP);
  DEF_PRIM2 (139, PRIM_SUCCEED | PRIM_FAIL, VMpr_Memory_addressOf);
  DEF_PRIM2 (145, PRIM_SUCCEED | PRIM_FAIL, VMpr_Memory_at);
  DEF_PRIM2 (146, PRIM_SUCCEED | PRIM_FAIL, VMpr_Memory_atPut);

  /* Low-level SystemDictionary and ObjectMemory primitives */
  DEF_PRIM2 (152, PRIM_SUCCEED | PRIM_FAIL | PRIM_CHECK_INTERRUPT, VMpr_Processor_signalOnInterrupt);
  DEF_PRIM2 (140, PRIM_SUCCEED, VMpr_SystemDictionary_backtrace);
  DEF_PRIM2 (141, PRIM_SUCCEED | PRIM_FAIL, VMpr_SystemDictionary_getTraceFlag);
  DEF_PRIM2 (142, PRIM_SUCCEED | PRIM_FAIL, VMpr_SystemDictionary_setTraceFlag);
  DEF_PRIM2 (162, PRIM_SUCCEED, VMpr_SystemDictionary_resetStatistics);
  DEF_PRIM2 (163, PRIM_SUCCEED, VMpr_SystemDictionary_printStatistics);
  DEF_PRIM2 (153, PRIM_SUCCEED, VMpr_ObjectMemory_getSpaceGrowRate);
  DEF_PRIM2 (154, PRIM_SUCCEED | PRIM_FAIL, VMpr_ObjectMemory_setSpaceGrowRate);
  DEF_PRIM2 (155, PRIM_SUCCEED, VMpr_ObjectMemory_getGrowThresholdPercent);
  DEF_PRIM2 (156, PRIM_SUCCEED | PRIM_FAIL, VMpr_ObjectMemory_setGrowThresholdPercent);
  DEF_PRIM2 (153, PRIM_SUCCEED, VMpr_ObjectMemory_getBigObjectThreshold);
  DEF_PRIM2 (154, PRIM_SUCCEED | PRIM_FAIL, VMpr_ObjectMemory_setBigObjectThreshold);
  DEF_PRIM2 (157, PRIM_SUCCEED | PRIM_FAIL, VMpr_ObjectMemory_growTo);
  DEF_PRIM2 (231, PRIM_SUCCEED, VMpr_SystemDictionary_byteCodeCounter);
  DEF_PRIM2 (232, PRIM_SUCCEED, VMpr_SystemDictionary_debug);
  DEF_PRIM2 (251, PRIM_SUCCEED | PRIM_FAIL, VMpr_ObjectMemory_snapshot);
  DEF_PRIM2 (252, PRIM_SUCCEED, VMpr_Object_basicPrint);
  DEF_PRIM2 (117, PRIM_FAIL, VMpr_ObjectMemory_quit);
  DEF_PRIM2 (260, PRIM_SUCCEED, VMpr_ObjectMemory_compact);
  
  /* Compilation primitives */
  DEF_PRIM2 (150, PRIM_SUCCEED, VMpr_Behavior_methodsFor);
  DEF_PRIM2 (151, PRIM_SUCCEED, VMpr_Behavior_methodsForIfTrue);
  DEF_PRIM2 (235, PRIM_SUCCEED, VMpr_Behavior_compileString);
  DEF_PRIM2 (236, PRIM_FAIL | PRIM_SUCCEED | PRIM_RELOAD_IP, VMpr_Behavior_compileStringIfError);
  DEF_PRIM2 (254, PRIM_SUCCEED | PRIM_FAIL, VMpr_FileDescriptor_fileOp);
  DEF_PRIM2 (258, PRIM_FAIL, VMpr_Namespace_setCurrent);
}

long _gst_execute_primitive_operation (int primitive,
				       volatile int numArgs,
				       OOP methodOOP)
{
  prim_table_entry *pte = &_gst_primitive_table[primitive];
#ifdef PROFBLOCK
  primitives[primitive]++;
#endif

  long result = pte->func (pte->id, numArgs, methodOOP);
  last_primitive = primitive;
  return result;
}

prim_table_entry *
_gst_get_primitive_attributes (int primitive)
{
  return &_gst_primitive_table[primitive];
}

#undef INT_BIN_OP
#undef BOOL_BIN_OP
#undef DEF_PRIM
#undef DEF_PRIM2
