/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file exchangedb/pg_select_kyc_accounts.c
 * @brief Implementation of the select_kyc_accounts function for Postgres
 * @author Christian Grothoff
 */
#include "taler/platform.h"
#include "taler/taler_error_codes.h"
#include "taler/taler_dbevents.h"
#include "taler/taler_pq_lib.h"
#include "pg_select_kyc_accounts.h"
#include "pg_helper.h"


/**
 * Closure for #handle_aml_result.
 */
struct KycAccountResultContext
{
  /**
   * Function to call on each result.
   */
  TALER_EXCHANGEDB_AmlAccountListCallback cb;

  /**
   * Closure for @e cb.
   */
  void *cb_cls;

  /**
   * Plugin context.
   */
  struct PostgresClosure *pg;

  /**
   * Set to #GNUNET_SYSERR on serious errors.
   */
  enum GNUNET_GenericReturnValue status;
};


/**
 * Function to be called with the results of a SELECT statement
 * that has returned @a num_results results.  Helper function
 * for #TEH_PG_select_kyc_accounts().
 *
 * @param cls closure of type `struct KycAccountResultContext *`
 * @param result the postgres result
 * @param num_results the number of results in @a result
 */
static void
handle_kyc_account_result (void *cls,
                           PGresult *result,
                           unsigned int num_results)
{
  struct KycAccountResultContext *ctx = cls;

  for (unsigned int i = 0; i<num_results; i++)
  {
    uint64_t rowid;
    struct TALER_NormalizedPaytoHashP h_payto;
    char *comments = NULL;
    struct GNUNET_TIME_Timestamp open_time
      = GNUNET_TIME_UNIT_FOREVER_TS;
    struct GNUNET_TIME_Timestamp close_time
      = GNUNET_TIME_UNIT_FOREVER_TS;
    bool to_investigate;
    bool high_risk;
    struct TALER_FullPayto payto;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_uint64 ("kyc_target_serial_id",
                                    &rowid),
      GNUNET_PQ_result_spec_auto_from_type ("h_payto",
                                            &h_payto),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_string ("comments",
                                      &comments),
        NULL),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_timestamp ("open_time",
                                         &open_time),
        NULL),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_timestamp ("close_time",
                                         &close_time),
        NULL),
      GNUNET_PQ_result_spec_bool ("to_investigate",
                                  &to_investigate),
      GNUNET_PQ_result_spec_bool ("high_risk",
                                  &high_risk),
      GNUNET_PQ_result_spec_string ("payto_uri",
                                    &payto.full_payto),
      GNUNET_PQ_result_spec_end
    };

    if (GNUNET_OK !=
        GNUNET_PQ_extract_result (result,
                                  rs,
                                  i))
    {
      GNUNET_break (0);
      ctx->status = GNUNET_SYSERR;
      return;
    }
    ctx->cb (ctx->cb_cls,
             rowid,
             &h_payto,
             open_time,
             close_time,
             comments,
             high_risk,
             to_investigate,
             payto);
    GNUNET_PQ_cleanup_result (rs);
  }
}


enum GNUNET_DB_QueryStatus
TEH_PG_select_kyc_accounts (
  void *cls,
  enum TALER_EXCHANGE_YesNoAll investigation_only,
  enum TALER_EXCHANGE_YesNoAll open_only,
  enum TALER_EXCHANGE_YesNoAll high_risk_only,
  uint64_t offset,
  int64_t limit,
  TALER_EXCHANGEDB_AmlAccountListCallback cb,
  void *cb_cls)
{
  struct PostgresClosure *pg = cls;
  uint64_t ulimit = (limit > 0) ? limit : -limit;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_uint64 (&offset),
    GNUNET_PQ_query_param_uint64 (&ulimit),
    GNUNET_PQ_query_param_bool ((TALER_EXCHANGE_YNA_ALL ==
                                 investigation_only)),
    GNUNET_PQ_query_param_bool ((TALER_EXCHANGE_YNA_YES ==
                                 investigation_only)),
    GNUNET_PQ_query_param_bool ((TALER_EXCHANGE_YNA_ALL ==
                                 open_only)),
    GNUNET_PQ_query_param_bool ((TALER_EXCHANGE_YNA_YES ==
                                 open_only)),
    GNUNET_PQ_query_param_bool ((TALER_EXCHANGE_YNA_ALL ==
                                 high_risk_only)),
    GNUNET_PQ_query_param_bool ((TALER_EXCHANGE_YNA_YES ==
                                 high_risk_only)),
    GNUNET_PQ_query_param_end
  };
  struct KycAccountResultContext ctx = {
    .cb = cb,
    .cb_cls = cb_cls,
    .pg = pg,
    .status = GNUNET_OK
  };
  enum GNUNET_DB_QueryStatus qs;
  const char *stmt = (limit > 0)
    ? "select_kyc_accounts_inc"
    : "select_kyc_accounts_dec";

  PREPARE (pg,
           "select_kyc_accounts_inc",
           "SELECT"
           " kt.kyc_target_serial_id"
           ",kt.h_normalized_payto AS h_payto"
           ",kt.open_time"
           ",kt.close_time"
           ",lo.jproperties ->> 'FILE_NOTE' AS comments"
           ",lo.jproperties ->> 'open_date' AS open_time"
           ",COALESCE(lo.to_investigate,FALSE) AS to_investigate"
           ",COALESCE((lo.jproperties ->> 'HIGH_RISK_CUSTOMER')::bool,FALSE) AS high_risk"
           ",wt.payto_uri"
           " FROM kyc_targets kt"
           " JOIN wire_targets wt"
           "   ON (wt.h_normalized_payto = kt.h_normalized_payto)"
           " LEFT JOIN legitimization_outcomes lo"
           "   ON (lo.h_payto = kt.h_normalized_payto)"
           " WHERE (kyc_target_serial_id > $1)"
           // select most recent outcomes only
           "   AND COALESCE (lo.is_active, TRUE)"
           "   AND ($3 OR (COALESCE(lo.to_investigate,FALSE) = $4))"
           // Account is open if we had an AML outcome
           "   AND ($5 OR ((lo.outcome_serial_id IS NULL) = $6))"
           "   AND ($7 OR ((COALESCE((lo.jproperties ->>'high_risk')::bool,FALSE) = $8)))"
           " ORDER BY kt.kyc_target_serial_id ASC"
           " LIMIT $2");
  PREPARE (pg,
           "select_kyc_accounts_dec",
           "SELECT"
           " kt.kyc_target_serial_id"
           ",kt.h_normalized_payto AS h_payto"
           ",kt.open_time"
           ",kt.close_time"
           ",lo.jproperties ->> 'FILE_NOTE' AS comments"
           ",lo.jproperties ->> 'open_date' AS open_time"
           ",COALESCE(lo.to_investigate,FALSE) AS to_investigate"
           ",COALESCE((lo.jproperties ->> 'HIGH_RISK_CUSTOMER')::bool,FALSE) AS high_risk"
           ",wt.payto_uri"
           " FROM kyc_targets kt"
           " JOIN wire_targets wt"
           "   ON (wt.h_normalized_payto = kt.h_normalized_payto)"
           " LEFT JOIN legitimization_outcomes lo"
           "   ON (lo.h_payto = kt.h_normalized_payto)"
           " WHERE (kyc_target_serial_id < $1)"
           // select most recent outcomes only
           "   AND COALESCE (lo.is_active, TRUE)"
           "   AND ($3 OR (COALESCE(lo.to_investigate,FALSE) = $4))"
           // Account is open if we had an AML outcome
           "   AND ($5 OR ((lo.outcome_serial_id IS NULL) = $6))"
           "   AND ($7 OR ((COALESCE((lo.jproperties ->>'high_risk')::bool,FALSE) = $8)))"
           " ORDER BY kt.kyc_target_serial_id DESC"
           " LIMIT $2");
  qs = GNUNET_PQ_eval_prepared_multi_select (pg->conn,
                                             stmt,
                                             params,
                                             &handle_kyc_account_result,
                                             &ctx);
  if (GNUNET_OK != ctx.status)
    return GNUNET_DB_STATUS_HARD_ERROR;
  return qs;
}
