/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file testing_api_cmd_get_product_image.c
 * @brief command to test GET /products/$HASH/image
 * @author Bohdan Potuzhnyi
 */
#include "platform.h"
#include <taler/taler_exchange_service.h>
#include <taler/taler_testing_lib.h>
#include "taler_merchant_service.h"
#include "taler_merchant_testing_lib.h"


/**
 * State of a "GET product image" CMD.
 */
struct GetProductImageState
{
  /**
   * Handle for a "GET product image" request.
   */
  struct TALER_MERCHANT_ProductImageGetHandle *pigh;

  /**
   * The interpreter state.
   */
  struct TALER_TESTING_Interpreter *is;

  /**
   * Base URL of the merchant serving the request.
   */
  const char *merchant_url;

  /**
   * Reference for a POST or PATCH /products CMD (optional).
   */
  const char *product_reference;

  /**
   * Expected HTTP response code.
   */
  unsigned int http_status;

  /**
   * Expected image as a data URL.
   */
  char *expected_image;

  /**
   * Hash over the expected image, lowercase hex encoding.
   */
  const char *image_hash;
};


/**
 * Callback for a /products/$HASH/image operation.
 *
 * @param cls closure for this function
 * @param pir response details
 */
static void
get_product_image_cb (void *cls,
                      const struct TALER_MERCHANT_ProductImageGetResponse *pir)
{
  struct GetProductImageState *gis = cls;

  gis->pigh = NULL;
  if (gis->http_status != pir->hr.http_status)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u (%d) to command %s\n",
                pir->hr.http_status,
                (int) pir->hr.ec,
                TALER_TESTING_interpreter_get_current_label (gis->is));
    TALER_TESTING_interpreter_fail (gis->is);
    return;
  }
  switch (pir->hr.http_status)
  {
  case MHD_HTTP_OK:
    if (NULL != gis->expected_image)
    {
      if (0 != strcmp (pir->details.ok.image,
                       gis->expected_image))
      {
        GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                    "Product image did not match expected value\n");
        TALER_TESTING_interpreter_fail (gis->is);
        return;
      }
    }
    break;
  default:
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "Unhandled HTTP status.\n");
  }
  TALER_TESTING_interpreter_next (gis->is);
}


/**
 * Run the "GET product image" CMD.
 *
 * @param cls closure.
 * @param cmd command being run now.
 * @param is interpreter state.
 */
static void
get_product_image_run (void *cls,
                       const struct TALER_TESTING_Command *cmd,
                       struct TALER_TESTING_Interpreter *is)
{
  struct GetProductImageState *gis = cls;
  const struct TALER_TESTING_Command *product_cmd;

  gis->is = is;
  gis->expected_image = NULL;
  if (NULL != gis->product_reference)
  {
    const char *product_image;

    product_cmd = TALER_TESTING_interpreter_lookup_command (
      is,
      gis->product_reference);
    if (NULL == product_cmd)
    {
      GNUNET_break (0);
      TALER_TESTING_interpreter_fail (is);
      return;
    }
    if (GNUNET_OK !=
        TALER_TESTING_get_trait_product_image (product_cmd,
                                               &product_image))
    {
      TALER_TESTING_interpreter_fail (is);
      return;
    }
    gis->expected_image = GNUNET_strdup (product_image);
  }
  gis->pigh
    = TALER_MERCHANT_product_image_get (
        TALER_TESTING_interpreter_get_context (is),
        gis->merchant_url,
        gis->image_hash,
        &get_product_image_cb,
        gis);
  GNUNET_assert (NULL != gis->pigh);
}


/**
 * Free the state of a "GET product image" CMD, and possibly
 * cancel a pending operation thereof.
 *
 * @param cls closure.
 * @param cmd command being run.
 */
static void
get_product_image_cleanup (void *cls,
                           const struct TALER_TESTING_Command *cmd)
{
  struct GetProductImageState *gis = cls;

  if (NULL != gis->pigh)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "GET /products/$HASH/image operation did not complete\n");
    TALER_MERCHANT_product_image_get_cancel (gis->pigh);
  }
  GNUNET_free (gis->expected_image);
  GNUNET_free (gis);
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_get_product_image (const char *label,
                                     const char *merchant_url,
                                     const char *product_reference,
                                     const char *image_hash,
                                     unsigned int http_status)
{
  struct GetProductImageState *gis;

  gis = GNUNET_new (struct GetProductImageState);
  gis->merchant_url = merchant_url;
  gis->product_reference = product_reference;
  gis->http_status = http_status;
  gis->image_hash = image_hash;
  {
    struct TALER_TESTING_Command cmd = {
      .cls = gis,
      .label = label,
      .run = &get_product_image_run,
      .cleanup = &get_product_image_cleanup
    };

    return cmd;
  }
}
