/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AbsoluteTime,
  AmountString,
  Duration,
  NotificationType,
  TalerCorebankApiClient,
  TransactionMajorState,
  TransactionMinorState,
  j2s,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig } from "../harness/denomStructures.js";
import {
  createSimpleTestkudosEnvironmentV2,
  createWalletDaemonWithClient,
  makeTestPaymentV2,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

const coinCommon = {
  cipher: "RSA" as const,
  durationLegal: "3 years",
  durationSpend: "2 years",
  durationWithdraw: "7 days",
  feeDeposit: "TESTKUDOS:0",
  feeRefresh: "TESTKUDOS:0",
  feeRefund: "TESTKUDOS:0",
  feeWithdraw: "TESTKUDOS:0",
  rsaKeySize: 1024,
};

/**
 * Run test for a peer push payment with balance locked behind a pending refresh.
 */
export async function runWalletBlockedPayPeerPushTest(t: GlobalTestState) {
  // Set up test environment

  const coinConfigList: CoinConfig[] = [
    {
      ...coinCommon,
      name: "n1",
      value: "TESTKUDOS:1",
    },
    {
      ...coinCommon,
      name: "n5",
      value: "TESTKUDOS:5",
    },
  ];

  const { bank, exchange, merchant, merchantAdminAccessToken } = await createSimpleTestkudosEnvironmentV2(
    t,
    coinConfigList,
  );

  const bankClient = new TalerCorebankApiClient(bank.baseUrl);

  // Withdraw digital cash into the wallet.

  const { walletClient: w1 } = await createWalletDaemonWithClient(t, {
    name: "w1",
    persistent: true,
    config: {
      testing: {
        devModeActive: true,
      },
    },
  });

  await withdrawViaBankV3(t, {
    walletClient: w1,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await w1.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  // Prevent the wallet from doing refreshes by injecting a 5xx
  // status for all refresh requests.
  await w1.call(WalletApiOperation.ApplyDevExperiment, {
    devExperimentUri: "taler://dev-experiment/start-block-refresh",
  });

  // Do a payment that causes a refresh.
  await makeTestPaymentV2(t, {
    merchant,
    merchantAdminAccessToken,
    walletClient: w1,
    order: {
      summary: "test",
      amount: "TESTKUDOS:2",
    },
  });

  const checkResp = await w1.call(WalletApiOperation.CheckPeerPushDebit, {
    amount: "TESTKUDOS:18" as AmountString,
  });

  console.log(`check resp ${j2s(checkResp)}`);

  const readyCond = w1.waitForNotificationCond((n) => {
    return (
      n.type === NotificationType.TransactionStateTransition &&
      n.transactionId.startsWith("txn:peer-push-debit:") &&
      n.newTxState.major === TransactionMajorState.Pending &&
      n.newTxState.minor === TransactionMinorState.Ready
    );
  });

  const confirmResp = await w1.call(WalletApiOperation.InitiatePeerPushDebit, {
    partialContractTerms: {
      summary: "hi!",
      amount: "TESTKUDOS:18" as AmountString,
      purse_expiration: AbsoluteTime.toProtocolTimestamp(
        AbsoluteTime.addDuration(
          AbsoluteTime.now(),
          Duration.fromSpec({ hours: 1 }),
        ),
      ),
    },
  });

  console.log(`confirm resp ${j2s(confirmResp)}`);

  await w1.call(WalletApiOperation.ApplyDevExperiment, {
    devExperimentUri: "taler://dev-experiment/stop-block-refresh",
  });

  await readyCond;
}

runWalletBlockedPayPeerPushTest.suites = ["wallet"];
