/*
 * tclxslt.c --
 *
 *  Interface to Gnome libxslt.
 *
 * Copyright (c) 2001-2002 Zveno Pty Ltd
 * http://www.zveno.com/
 *
 * Zveno Pty Ltd makes this software and associated documentation
 * available free of charge for any purpose.  You may make copies
 * of the software but you must include all of this notice on any copy.
 *
 * Zveno Pty Ltd does not warrant that this software is error free
 * or fit for any purpose.  Zveno Pty Ltd disclaims any liability for
 * all claims, expenses, losses, damages and costs any user may incur
 * as a result of using, copying or modifying the software.
 *
 * $Id: tclxslt.c,v 1.8 2002/02/28 12:03:46 balls Exp $
 *
 */

#include "tclxslt.h"

#undef TCL_STORAGE_CLASS
#define TCL_STORAGE_CLASS DLLEXPORT

#ifdef __WIN32__
#     include "win/win32config.h"
#endif

/*
 * For Darwin (MacOS X) in particular, but also others
 */
 
#ifndef __WIN32__
#	define DLLIMPORT EXTERN
#endif

/*
 * XSLT stylesheet object type
 */

Tcl_ObjType TclXSLT_StylesheetObjType = {
  "xsltstylesheet",
  TclXSLTFreeStylesheet,
  TclXSLTDupStylesheet,
  TclXSLTUpdateStylesheet,
  TclXSLTSetFromAnyStylesheet
};

/*
 * Extension management
 */

typedef struct TclXSLT_Extension {
  Tcl_Interp *interp;
  Tcl_Obj *nsuri;
  Tcl_Obj *tclns;
} TclXSLT_Extension;

Tcl_HashTable extensions;

/*
 * Prototypes for procedures defined later in this file:
 */

/*
 * Declarations for externally visible functions.
 */

EXTERN int      Xslt_Init _ANSI_ARGS_((Tcl_Interp *interp));

/*
 * Forward declarations for private functions.
 */

static void TclXSLTGenericError _ANSI_ARGS_((void *ctx, const char *msg, ...));

static int TclXSLTTransformCommand _ANSI_ARGS_((ClientData dummy,
						Tcl_Interp *interp,
						int objc,
						Tcl_Obj *CONST objv[]));
static int TclXSLTConfigureCommand _ANSI_ARGS_((ClientData dummy,
						Tcl_Interp *interp,
						int objc,
						Tcl_Obj *CONST objv[]));
static int TclXSLTExtensionCommand _ANSI_ARGS_((ClientData dummy,
						Tcl_Interp *interp,
						int objc,
						Tcl_Obj *CONST objv[]));
static void TclXSLT_RegisterAll _ANSI_ARGS_((TclXSLT_Extension *extinfo,
						const xmlChar *nsuri));

/* static xsltExtInitFunction TclXSLTExtInit; */
static void *TclXSLTExtInit _ANSI_ARGS_((xsltTransformContextPtr ctxt,
					const xmlChar *URI));
/* static xsltExtShutdownFunction TclXSLTExtShutdown; */
static void TclXSLTExtShutdown _ANSI_ARGS_((xsltTransformContextPtr ctxt,
					    const xmlChar *URI,
					    void *userdata));
/* static xmlXPathEvalFunc TclXSLTExtFunction; */
static void TclXSLTExtFunction _ANSI_ARGS_((xmlXPathParserContextPtr xpathCtxt,
					    int nargs));

/*
 * Error context for passing error result back to caller.
 */

typedef struct GenericError_Info {
  Tcl_Interp *interp;
  int code;
  Tcl_Obj *msg;
} GenericError_Info;

static GenericError_Info errorInfo = {
  NULL,
  TCL_OK,
  NULL
};

/*
 * Switch tables
 */

/*
 * Since DOM parsing is separate to XSL transformation,
 * no need to pass tcldom switches.
 */

static char *configureCommandOptions[] = {
  "-baseuri",
  (char *) NULL
};
enum configureCommandOptions {
  TCLXSLT_CFG_BASEURI
};

static char *extensionCommandMethods[] = {
  "add",
  "remove",
  (char *) NULL
};
enum extensionCommandMethods {
  TCLXSLT_EXT_ADD,
  TCLXSLT_EXT_REMOVE
};

/*
 *----------------------------------------------------------------------------
 *
 * Xslt_Init --
 *
 *  Initialisation routine for loadable module
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Creates commands in the interpreter,
 *
 *----------------------------------------------------------------------------
 */

int
Xslt_Init (interp)
     Tcl_Interp *interp;	/* Interpreter to initialise */
{
  /*
   * Use minimum version for maximum usefulness.
   * Requested by Darrin.Wortlehock@i7.com.au
   */
  Tcl_InitStubs(interp, "8.1", 0);

  /* package require dom 2.1 */

  Tcl_RegisterObjType(&TclXSLT_StylesheetObjType);

  Tcl_CreateObjCommand(interp, "xslt::transform", TclXSLTTransformCommand, NULL, NULL);
  Tcl_CreateObjCommand(interp, "xslt::extension", TclXSLTExtensionCommand, NULL, NULL);

  /*
   * configure is now obsolete, since TclDOM can parse documents
   * with the base URI set.

  Tcl_CreateObjCommand(interp, "xslt::configure", TclXSLTConfigureCommand, NULL, NULL);
  */

  Tcl_InitHashTable(&extensions, TCL_STRING_KEYS);

  Tcl_PkgProvide(interp, "xslt", "1.1");

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXSLTTransformCommand --
 *
 *  Command for xslt::transform command.
 *
 * Results:
 *  Returns result document from transformation.
 *
 * Side effects:
 *  Stylesheet may be compiled.
 *
 *----------------------------------------------------------------------------
 */

static int
TclXSLTTransformCommand(dummy, interp, objc, objv)
     ClientData dummy;
     Tcl_Interp *interp;
     int objc;
     Tcl_Obj *CONST objv[];
{
  Tcl_Obj *ssheetObjPtr;
  xsltStylesheetPtr ssheetPtr;
  xmlDocPtr doc, result;
  char **params;
  int nbparams = 0, i, errcode;

  if (objc < 3) {
    Tcl_WrongNumArgs(interp, 1, objv, "stylesheet xml ?args...?");
    return TCL_ERROR;
  }

  if ((objc - 3) % 2) {
    Tcl_SetResult(interp, "no value for parameter\n", NULL);
    return TCL_ERROR;
  }

  if (TclDOM_GetDocFromObj(interp, objv[2], &doc) != TCL_OK) {
    return TCL_ERROR;
  }

  ssheetObjPtr = objv[1];
  if (TclXSLT_GetStylesheetFromObj(interp, ssheetObjPtr, &ssheetPtr) != TCL_OK) {
    return TCL_ERROR;
  }

  params = (char **) Tcl_Alloc(sizeof(char **) * (objc - 3 + 1));
  for (i = 3; i < objc; i++) {
    params[nbparams++] = Tcl_GetStringFromObj(objv[i++], NULL);
    params[nbparams++] = Tcl_GetStringFromObj(objv[i], NULL);
  }
  params[nbparams] = NULL;

  /*
   * Perform the transformation
   */

  errorInfo.interp = interp;
  errorInfo.code = TCL_OK;
  if (errorInfo.msg) {
    Tcl_DecrRefCount(errorInfo.msg);
  }
  errorInfo.msg = NULL;

  result = xsltApplyStylesheet(ssheetPtr, doc, (const char **)params);

  if (result == NULL) {
    Tcl_SetResult(interp, "no result document", NULL);
    return TCL_ERROR;
  }

  errcode = TCL_OK;
  if (errorInfo.code != TCL_OK) {
    if (errorInfo.msg) {
      Tcl_SetObjResult(interp, errorInfo.msg);
    }
    errcode = TCL_ERROR;
  }

  Tcl_SetObjResult(interp, TclDOM_CreateObjFromDoc(result));

  errorInfo.interp = NULL;
  errorInfo.code = TCL_OK;
  if (errorInfo.msg) {
    Tcl_DecrRefCount(errorInfo.msg);
  }
  errorInfo.msg = NULL;

  return errcode;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXSLTGenericError --
 *
 *  Handler for stylesheet errors.
 *
 *  NB. Cannot distinguish between errors and use of xsl:message element.
 *
 * Results:
 *  Stores error message.
 *
 * Side effects:
 *  Transform will return error condition.
 *
 *----------------------------------------------------------------------------
 */

static void
TclXSLTGenericError (void *ctx, const char *msg, ...)
{
  va_list args;
  char buf[2048];
  int len;

  errorInfo.code = TCL_ERROR;

  if (!errorInfo.msg) {
    errorInfo.msg = Tcl_NewObj();
    Tcl_IncrRefCount(errorInfo.msg);
  }

  va_start(args,msg);
  len = vsnprintf(buf, 2047, msg, args);
  va_end(args);

  Tcl_AppendToObj(errorInfo.msg, buf, len);

}

/*
 *----------------------------------------------------------------------------
 *
 * TclXSLT_GetStylesheetFromObj --
 *
 *  Retrieve the stylesheet from a Tcl_Obj.
 *
 * Results:
 *  Returns a stylesheet.
 *
 * Side effects:
 *  Tcl object may be converted to a stylesheet object.
 *
 *----------------------------------------------------------------------------
 */

int
TclXSLT_GetStylesheetFromObj(interp, objPtr, ssheetPtr)
     Tcl_Interp *interp;
     register Tcl_Obj *objPtr;
     xsltStylesheetPtr *ssheetPtr;
{
  if (objPtr->typePtr != &TclXSLT_StylesheetObjType) {
    int result = TclXSLTSetFromAnyStylesheet(interp, objPtr);
    if (result != TCL_OK) {
      return result;
    }
  }
  *ssheetPtr = (xsltStylesheetPtr) objPtr->internalRep.otherValuePtr;
  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 * Manage a stylesheet object
 *
 *----------------------------------------------------------------------------
 */

int
TclXSLTSetFromAnyStylesheet(interp, objPtr)
     Tcl_Interp *interp;
     Tcl_Obj *objPtr;
{
  xsltStylesheetPtr ssheetPtr;
  char *ssheetxml;
  xmlDocPtr ssheetdoc;
  int ssheetlen;

  if (objPtr->typePtr == &TclDOM_DocObjType) {
    ssheetdoc = (xmlDocPtr) objPtr->internalRep.otherValuePtr;
  } else if (objPtr->typePtr == &TclDOM_NodeObjType) {
    Tcl_SetResult(interp, "cannot compile stylesheet from a DOM node - use DOM document instead", NULL);
    return TCL_ERROR;
  } else {
    if (TclDOM_GetDocFromObj(interp, objPtr, &ssheetdoc) != TCL_OK) {
      return TCL_ERROR;
    }
  }

  /*
   * Prepare for compiling stylesheet
   */

  errorInfo.interp = interp;
  errorInfo.code = TCL_OK;
  if (errorInfo.msg) {
    Tcl_DecrRefCount(errorInfo.msg);
  }
  errorInfo.msg = NULL;

  /*
   * Compile stylesheet
   */

  if ((ssheetPtr = xsltParseStylesheetDoc(ssheetdoc)) == NULL) {
    Tcl_SetResult(interp, "error compiling stylesheet", NULL);
    return TCL_ERROR;
  }

  if (ssheetPtr->errors > 0) {
    Tcl_SetResult(interp, "error compiling XSLT stylesheet", NULL);
    return TCL_ERROR;
  }

  if (errorInfo.code != TCL_OK) {

    if (errorInfo.msg) {
      Tcl_SetObjResult(interp, errorInfo.msg);
      Tcl_DecrRefCount(errorInfo.msg);
    }

    errorInfo.interp = NULL;
    errorInfo.code = TCL_OK;
    errorInfo.msg = NULL;

    return TCL_ERROR;
  }

  objPtr->internalRep.otherValuePtr = (VOID *) ssheetPtr;
  objPtr->typePtr = &TclXSLT_StylesheetObjType;

  errorInfo.interp = NULL;
  errorInfo.code = TCL_OK;
  if (errorInfo.msg) {
    Tcl_DecrRefCount(errorInfo.msg);
  }
  errorInfo.msg = NULL;

  return TCL_OK;
}

/*
 * Serializing the stylesheet's document may not
 * return the original document exactly.
 */

void
TclXSLTUpdateStylesheet(objPtr)
     Tcl_Obj *objPtr;
{
  xsltStylesheetPtr ssheetPtr = (xsltStylesheetPtr) objPtr->internalRep.otherValuePtr;
  Tcl_Obj *docObj;
  char *buf;

  docObj = TclDOM_CreateObjFromDoc(ssheetPtr->doc);
  buf = Tcl_GetStringFromObj(docObj, &objPtr->length);

  memcpy(objPtr->bytes, buf, objPtr->length);

  Tcl_DecrRefCount(docObj);
}

void
TclXSLTDupStylesheet(srcPtr, dstPtr)
     Tcl_Obj *srcPtr;
     Tcl_Obj *dstPtr;
{
  dstPtr->internalRep.otherValuePtr = srcPtr->internalRep.otherValuePtr;
  dstPtr->typePtr = &TclXSLT_StylesheetObjType;
}

void
TclXSLTFreeStylesheet(objPtr)
     Tcl_Obj *objPtr;
{
  xsltFreeStylesheet((xsltStylesheetPtr) objPtr->internalRep.otherValuePtr);
  /* Does the xmlDocPtr structure also need to be freed? */
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXSLTConfigureCommand --
 *
 *  Command for xslt::configure command.
 *
 * Results:
 *  Depends on option.
 *
 * Side effects:
 *  Configuration value may be stored.
 *
 *----------------------------------------------------------------------------
 */

static int
TclXSLTConfigureCommand(dummy, interp, objc, objv)
     ClientData dummy;
     Tcl_Interp *interp;
     int objc;
     Tcl_Obj *CONST objv[];
{
  int option;

  Tcl_SetResult(interp, "obsolete command", NULL);
  return TCL_ERROR;

  if (objc < 3) {
    Tcl_WrongNumArgs(interp, 1, objv, "option value ?option value...?");
    return TCL_ERROR;
  }

  objc--;
  objv++;

  while (objc > 1) {
    if (objc == 1) {
      Tcl_SetResult(interp, "no value for option", NULL);
      return TCL_ERROR;
    }

    if (Tcl_GetIndexFromObj(interp, objv[0], configureCommandOptions, 
			    "option", 0, &option) != TCL_OK) {
      return TCL_ERROR;
    }

    switch ((enum configureCommandOptions) option) {
    case TCLXSLT_CFG_BASEURI:
      /*
      if (defaultStylesheetBaseURIPtr) {
	Tcl_DecrRefCount(defaultStylesheetBaseURIPtr);
      }
      defaultStylesheetBaseURIPtr = objv[1];
      Tcl_IncrRefCount(objv[1]);
      defaultStylesheetBaseURI = Tcl_GetStringFromObj(objv[1], NULL);
      */
      break;

    default:
      Tcl_AppendResult(interp, "unknown option \"", Tcl_GetStringFromObj(objv[1], NULL), "\"", NULL);
      return TCL_ERROR;
      break;
    }

    objv += 2;
    objc -= 2;
  }

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXSLTExtensionCommand --
 *
 *  Command for xslt::extension command.
 *
 * Results:
 *  Depends on method.
 *
 * Side effects:
 *  Depends on method
 *
 *----------------------------------------------------------------------------
 */

static int
TclXSLTExtensionCommand(dummy, interp, objc, objv)
     ClientData dummy;
     Tcl_Interp *interp;
     int objc;
     Tcl_Obj *CONST objv[];
{
  int method, new;
  TclXSLT_Extension *extinfo;
  Tcl_HashEntry *entry;

  if (objc < 2) {
    Tcl_WrongNumArgs(interp, 1, objv, "method ?args ...?");
    return TCL_ERROR;
  }

  if (Tcl_GetIndexFromObj(interp, objv[1], extensionCommandMethods,
			  "method", 0, &method) != TCL_OK) {
    return TCL_ERROR;
  }

  switch ((enum extensionCommandMethods) method) {

  case TCLXSLT_EXT_ADD:
    if (objc != 4) {
      Tcl_WrongNumArgs(interp, 2, objv, "nsuri tcl-namespace");
      return TCL_ERROR;
    }

    if (xsltRegisterExtModule(Tcl_GetStringFromObj(objv[2], NULL),
			      TclXSLTExtInit,
			      TclXSLTExtShutdown)) {
      Tcl_SetResult(interp, "cannot register extension module", NULL);
    }

    extinfo = (TclXSLT_Extension *) Tcl_Alloc(sizeof(TclXSLT_Extension));
    extinfo->interp = interp;
    extinfo->nsuri = objv[2];
    extinfo->tclns = objv[3];

    entry = Tcl_CreateHashEntry(&extensions, Tcl_GetStringFromObj(objv[2], NULL), &new);

    if (!new) {
      Tcl_SetResult(interp, "extension already exists", NULL);
      Tcl_Free((char *) extinfo);
      return TCL_ERROR;
    }

    Tcl_IncrRefCount(objv[2]);
    Tcl_IncrRefCount(objv[3]);

    Tcl_SetHashValue(entry, extinfo);

    /*
     * Register all of the functions now... as of libxslt1.0.3 registering the
     * module doesn't seem to be enough.
     */

    TclXSLT_RegisterAll(extinfo, Tcl_GetStringFromObj(objv[2], NULL));

    Tcl_ResetResult(interp);

    break;

  case TCLXSLT_EXT_REMOVE:
    if (objc != 3) {
      Tcl_WrongNumArgs(interp, 2, objv, "nsuri");
      return TCL_ERROR;
    }

    /*
     * TODO: Remove previously registered elements and functions.
    */

    entry = Tcl_FindHashEntry(&extensions, Tcl_GetStringFromObj(objv[2], NULL));
    if (entry == NULL) {
      Tcl_SetResult(interp, "unknown XML Namespace URI", NULL);
      return TCL_ERROR;
    }

    extinfo = (TclXSLT_Extension *) Tcl_GetHashValue(entry);
    Tcl_DecrRefCount(extinfo->nsuri);
    Tcl_DecrRefCount(extinfo->tclns);
    Tcl_Free((char *) extinfo);

    Tcl_DeleteHashEntry(entry);

    break;

  default:
    Tcl_SetResult(interp, "unknown method", NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXSLTExtInit --
 *
 *  Load extensions into a transformation context.
 *
 * Results:
 *  Returns pointer to extension data.
 *  Elements and functions are pre-registered.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------------
 */

static void *
TclXSLTExtInit(ctxt, URI)
     xsltTransformContextPtr ctxt;
     const xmlChar *URI;
{
  Tcl_HashEntry *entry;

  entry = Tcl_FindHashEntry(&extensions, URI);
  if (entry == NULL) {
    /* Extension module was removed */
    return NULL;
  }

  return Tcl_GetHashValue(entry);
}

void
TclXSLT_RegisterAll(extinfo, nsuri)
    TclXSLT_Extension *extinfo;
    const xmlChar *nsuri;
{
  Tcl_Obj *cmdPtr, *listPtr, *objPtr;
  int ret, i, len;

  /*
   * Q: How to distinguish between extension elements and functions?
   * A: Use the formal parameters.  If the command can accept
   * a variable argument list, then it is registered as a function.
   * Otherwise it will be registered as an extension (and expected
   * to accept certain arguments).
   *
   * At the moment, only functions are supported.
   * TODO: Now that we have TclDOM add support for elements.
   * We'll need to append the DOM node of the element as an
   * argument.
   */

  cmdPtr = Tcl_NewObj();
  Tcl_AppendStringsToObj(cmdPtr, "::xslt::getprocs ", NULL);
  Tcl_AppendObjToObj(cmdPtr, extinfo->tclns);
  ret = Tcl_EvalObjEx(extinfo->interp, cmdPtr, TCL_EVAL_GLOBAL|TCL_EVAL_DIRECT);
  listPtr = Tcl_GetObjResult(extinfo->interp);

  if (ret != TCL_OK || listPtr == NULL) {
    /*
     * No such Tcl namespace, therefore no functions to register.
     */
    return;
  }

  ret = Tcl_ListObjLength(extinfo->interp, listPtr, &len);
  if (ret != TCL_OK || len == 0) {
    return;
  }

  for (i = 0; i < len; i++) {

    if (Tcl_ListObjIndex(extinfo->interp, listPtr, i, &objPtr) != TCL_OK) {
      continue;
    }

    /* TODO: Decide between function or element here */

    xsltRegisterExtModuleFunction((const xmlChar *) Tcl_GetStringFromObj(objPtr, NULL),
    	nsuri,
    	TclXSLTExtFunction);
  }

  return;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXSLTExtFunction --
 *
 *  Handles evaluation of an extension function.
 *
 * Results:
 *  Returns string returned by Tcl command evaluation.
 *
 * Side effects:
 *  Depends on Tcl command evaluated.
 *
 *----------------------------------------------------------------------------
 */

static void 
TclXSLTExtFunction(xpathCtxt, nargs)
     xmlXPathParserContextPtr xpathCtxt;
     int nargs;
{
  xsltTransformContextPtr xformCtxt;
  TclXSLT_Extension *extinfo;
  Tcl_Obj *cmdPtr;
  xmlXPathObjectPtr obj;
  int ret;

  xformCtxt = xsltXPathGetTransformContext(xpathCtxt);

  /*
   * In order to find the instance data we need the
   * XML Namespace URI of this function.
   */

  extinfo = (TclXSLT_Extension *) xsltGetExtData(xformCtxt,
						 xpathCtxt->context->functionURI);

  /*
   * Start constructing the script by first defining the command.
   */

  cmdPtr = Tcl_DuplicateObj(extinfo->tclns);
  Tcl_AppendStringsToObj(cmdPtr, "::", xpathCtxt->context->function, NULL);

  /*
   * Each argument on the stack is converted to a string
   * and passed as an argument to the Tcl command.
   *
   * TODO: pass nodes as TclDOM nodes instead of converting to string.
   */

  while (nargs) {
    Tcl_Obj *objPtr;
    /*
     * All arguments are converted to string.
     * May as well do this before popping the value.
     */
    xmlXPathStringFunction(xpathCtxt, 1);
    obj = (xmlXPathObjectPtr) valuePop(xpathCtxt);
    objPtr = Tcl_NewStringObj(obj->stringval, strlen(obj->stringval));
    Tcl_ListObjReplace(extinfo->interp, cmdPtr, 1, 0, 1, &objPtr);

    xmlXPathFreeObject(obj);

    nargs--;
  }

  ret = Tcl_EvalObjEx(extinfo->interp, cmdPtr, TCL_EVAL_GLOBAL | TCL_EVAL_DIRECT);

  if (ret == TCL_OK) {
    valuePush(xpathCtxt,
	xmlXPathNewString(Tcl_GetStringFromObj(Tcl_GetObjResult(extinfo->interp), NULL)));
  } else {
    xmlGenericError(xmlGenericErrorContext,
		    Tcl_GetStringFromObj(Tcl_GetObjResult(extinfo->interp), NULL));
    /* Need to define a new error code - this is the closest in meaning */
    xpathCtxt->error = XPATH_UNKNOWN_FUNC_ERROR;
  }
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXSLTExtShutdown --
 *
 *  Clean up.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  None.
 *
 *----------------------------------------------------------------------------
 */

static void
TclXSLTExtShutdown(ctxt, URI, userdata)
     xsltTransformContextPtr ctxt;
     const xmlChar *URI;
     void *userdata;
{
  TclXSLT_Extension *extinfo = (TclXSLT_Extension *) userdata;

  /* Nothing to do */
}
