/* docObj.c --
 *
 *	This module manages libxml2 xmlDocPtr Tcl objects.
 *
 * Copyright (c) 2003 Zveno Pty Ltd
 * http://www.zveno.com/
 *
 * Zveno Pty Ltd makes this software and associated documentation
 * available free of charge for any purpose.  You may make copies
 * of the software but you must include all of this notice on any copy.
 *
 * Zveno Pty Ltd does not warrant that this software is error free
 * or fit for any purpose.  Zveno Pty Ltd disclaims any liability for
 * all claims, expenses, losses, damages and costs any user may incur
 * as a result of using, copying or modifying the software.
 *
 * $Id: docObj.c,v 1.1.2.5 2003/07/28 01:05:47 balls Exp $
 */

#include "docObj.h"
#include "tclInt.h"

#define TCL_DOES_STUBS \
    (TCL_MAJOR_VERSION > 8 || TCL_MAJOR_VERSION == 8 && (TCL_MINOR_VERSION > 1 || \
    (TCL_MINOR_VERSION == 1 && TCL_RELEASE_LEVEL == TCL_FINAL_RELEASE)))

#undef TCL_STORAGE_CLASS
#define TCL_STORAGE_CLASS DLLEXPORT

/*
 * Basic list for tracking Tcl_Obj's for a document.
 */

typedef struct ObjList {
  Tcl_Obj *objPtr;
  struct ObjList *next;
} ObjList;

/*
 * Prototypes for procedures defined later in this file:
 */

static void DestroyTclDoc _ANSI_ARGS_((TclXML_libxml2_Document *tDocPtr));

Tcl_FreeInternalRepProc	TclXMLlibxml2_DocFree;
Tcl_DupInternalRepProc	TclXMLlibxml2_DocDup;
Tcl_UpdateStringProc	TclXMLlibxml2_DocUpdate;
Tcl_SetFromAnyProc	TclXMLlibxml2_DocSetFromAny;

Tcl_ObjType TclXMLlibxml2_DocObjType = {
  "libxml2-doc",
  TclXMLlibxml2_DocFree,
  TclXMLlibxml2_DocDup,
  TclXMLlibxml2_DocUpdate,
  TclXMLlibxml2_DocSetFromAny
};

typedef struct ThreadSpecificData {

  /*
   * Hash table for mapping string rep to doc structure.
   */

  Tcl_HashTable *documents;
  int docCntr;

  /*
   * Hash table for tracking doc objects.
   */

  Tcl_HashTable *docByPtr;

  /*
   * For debugging
   */

  Tcl_Channel stderrChan;
  char dbgbuf[200];
} ThreadSpecificData;
static Tcl_ThreadDataKey dataKey;

/*
 * libxml2 is mostly thread-safe, but just-in-case use a mutex to control access.
 */

TCL_DECLARE_MUTEX(libxml2)

/*
 *----------------------------------------------------------------------------
 *
 * TclXML_libxml2_InitDocObj --
 *
 *  Initialise this module.
 *
 * Results:
 *  Returns success code
 *
 * Side effects:
 *  Memory may be allocated
 *
 *----------------------------------------------------------------------------
 */

int
TclXML_libxml2_InitDocObj(interp)
    Tcl_Interp *interp;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  int dbgMode;

  tsdPtr->documents = (Tcl_HashTable *) Tcl_Alloc(sizeof(Tcl_HashTable));
  Tcl_InitHashTable(tsdPtr->documents, TCL_STRING_KEYS);
  tsdPtr->docByPtr = (Tcl_HashTable *) Tcl_Alloc(sizeof(Tcl_HashTable));
  Tcl_InitHashTable(tsdPtr->docByPtr, TCL_ONE_WORD_KEYS);

  tsdPtr->docCntr = 0;

  tsdPtr->stderrChan = Tcl_GetChannel(interp, "stderr", &dbgMode);

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXML_libxml2_NewDocObj --
 *
 *  Creates a new xmlDocPtr and wraps it in a Tcl_Obj.
 *
 * Results:
 *  Returns a *TclObj
 *
 * Side effects:
 *  Objects allocated.
 *
 *----------------------------------------------------------------------------
 */

Tcl_Obj *
TclXML_libxml2_NewDocObj(interp)
     Tcl_Interp *interp;
{
  xmlDocPtr new;

  Tcl_MutexLock(&libxml2);
  new = xmlNewDoc("1.0");
  Tcl_MutexUnlock(&libxml2);
  if (!new) {
    Tcl_SetResult(interp, "unable to create document", NULL);
    return NULL;
  }

  return TclXML_libxml2_CreateObjFromDoc(new);
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXML_libxml2_CreateObjFromDoc --
 *
 *  Create a Tcl_Obj to wrap a xmlDocPtr.
 *
 * Results:
 *  Returns Tcl_Obj*.
 *
 * Side effects:
 *  Allocates object.
 *
 *----------------------------------------------------------------------------
 */

Tcl_Obj *
TclXML_libxml2_CreateObjFromDoc (docPtr)
  xmlDocPtr docPtr;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXML_libxml2_Document *tDocPtr;
  Tcl_HashEntry *entryPtr;
  Tcl_Obj *objPtr;
  ObjList *listPtr;

  /*
   * This xmlDocPtr may already have been wrapped by a Tcl object.
   * If so, return an already existing wrapper.
   * If not, create a new wrapper.
   */

  entryPtr = Tcl_FindHashEntry(tsdPtr->docByPtr, (ClientData) docPtr);
  if (entryPtr) {
    tDocPtr = (TclXML_libxml2_Document *) Tcl_GetHashValue(entryPtr);

    if (tDocPtr->objs) {
      /* The first object is sufficient */
      listPtr = (ObjList *) tDocPtr->objs;
      objPtr = listPtr->objPtr;

    } else {
      /* Create a new Tcl_Obj to refer to existing structure */
      objPtr = Tcl_NewObj();

      listPtr = (ObjList *) Tcl_Alloc(sizeof(ObjList));
      listPtr->objPtr = objPtr;
      listPtr->next = NULL;
      tDocPtr->objs = (void *) listPtr;

      objPtr->length = strlen(tDocPtr->token);
      objPtr->bytes = Tcl_Alloc(objPtr->length + 1);
      strcpy(objPtr->bytes, tDocPtr->token);
      objPtr->internalRep.otherValuePtr = (void *) tDocPtr;
      objPtr->typePtr = &TclXMLlibxml2_DocObjType;
    }

  } else {
    int new;

    objPtr = Tcl_NewObj();

    tDocPtr = (TclXML_libxml2_Document *) Tcl_Alloc(sizeof(TclXML_libxml2_Document));
    tDocPtr->docPtr = docPtr;
    tDocPtr->token = Tcl_Alloc(20);
    sprintf(tDocPtr->token, "doc%d", tsdPtr->docCntr++);
    tDocPtr->keep = TCLXML_LIBXML2_DOCUMENT_IMPLICIT;
    tDocPtr->dom = NULL;
    tDocPtr->domfree = NULL;
    tDocPtr->apphook = NULL;
    tDocPtr->appfree = NULL;

    listPtr = (ObjList *) Tcl_Alloc(sizeof(ObjList));
    listPtr->objPtr = objPtr;
    listPtr->next = NULL;
    tDocPtr->objs = (void *) listPtr;

    entryPtr = Tcl_CreateHashEntry(tsdPtr->documents, tDocPtr->token, &new);
    Tcl_SetHashValue(entryPtr, (ClientData) tDocPtr);
    entryPtr = Tcl_CreateHashEntry(tsdPtr->docByPtr, (ClientData) docPtr, &new);
    Tcl_SetHashValue(entryPtr, (ClientData) tDocPtr);

    objPtr->length = strlen(tDocPtr->token);
    objPtr->bytes = Tcl_Alloc(objPtr->length + 1);
    strcpy(objPtr->bytes, tDocPtr->token);
    objPtr->internalRep.otherValuePtr = (void *) tDocPtr;
    objPtr->typePtr = &TclXMLlibxml2_DocObjType;
  }

  return objPtr;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXML_libxml2_GetDocFromObj --
 *
 *  Retrieve the xmlDocPtr from a Tcl object.
 *
 * Results:
 *  Returns success code.
 *
 * Side effects:
 *  May set internal rep of object.
 *
 *----------------------------------------------------------------------------
 */

int
TclXML_libxml2_GetDocFromObj (interp, objPtr, docPtr)
     Tcl_Interp *interp;
     Tcl_Obj *objPtr;
     xmlDocPtr *docPtr;
{
  TclXML_libxml2_Document *tDocPtr;

  if (TclXML_libxml2_GetTclDocFromObj(interp, objPtr, &tDocPtr) != TCL_OK) {
    return TCL_ERROR;
  }

  *docPtr = tDocPtr->docPtr;

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXML_libxml2_GetTclDocFromNode --
 *
 *  Retrieve a pointer to the TclXML Doc structure from a xmlNodePtr.
 *
 * Results:
 *  Returns success code.
 *
 * Side effects:
 *  Sets pointer
 *
 *----------------------------------------------------------------------------
 */

int
TclXML_libxml2_GetTclDocFromNode (interp, nodePtr, tDocPtrPtr)
     Tcl_Interp *interp;
     xmlNodePtr nodePtr;
     TclXML_libxml2_Document **tDocPtrPtr;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->docByPtr, (ClientData) nodePtr->doc);
  if (!entryPtr) {
    *tDocPtrPtr = NULL;
    Tcl_SetResult(interp, "document not known", NULL);
    return TCL_ERROR;
  }

  *tDocPtrPtr = (TclXML_libxml2_Document *) Tcl_GetHashValue(entryPtr);

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXML_libxml2_GetTclDocFromObj --
 *
 *  Retrieve the TclXML_libxml2_Document from a Tcl object.
 *
 * Results:
 *  Returns success code.
 *
 * Side effects:
 *  May set internal rep of object.
 *
 *----------------------------------------------------------------------------
 */

int
TclXML_libxml2_GetTclDocFromObj (interp, objPtr, tDocPtr)
     Tcl_Interp *interp;
     Tcl_Obj *objPtr;
     TclXML_libxml2_Document **tDocPtr;
{
  if (objPtr->typePtr == &TclXMLlibxml2_DocObjType) {
    *tDocPtr = (TclXML_libxml2_Document *) objPtr->internalRep.otherValuePtr;
  } else if (TclXMLlibxml2_DocSetFromAny(interp, objPtr) == TCL_OK) {
    *tDocPtr = (TclXML_libxml2_Document *) objPtr->internalRep.otherValuePtr;
  } else {
    return TCL_ERROR;
  }

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXML_libxml2_DestroyDocument --
 *
 *  Manage destruction of a document.
 *  The trick here is to make sure that all Tcl_Obj's
 *  that reference this document have their internal rep
 *  invalidated.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Memory deallocated, object internal reps changed.
 *
 *----------------------------------------------------------------------------
 */

void
TclXML_libxml2_DestroyDocument (tDocPtr)
     TclXML_libxml2_Document *tDocPtr;
{
  ObjList *listPtr = (ObjList *) tDocPtr->objs;
  ObjList *next;

  /*
   * Invalidate the internal representation of all Tcl_Obj's
   * that refer to this document.
   */
  while (listPtr) {
    next = listPtr->next;
    TclXMLlibxml2_DocFree(listPtr->objPtr);
    listPtr = next;
  }

  if (tDocPtr->keep == TCLXML_LIBXML2_DOCUMENT_KEEP) {
    DestroyTclDoc(tDocPtr);
  }
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXML_libxml2_DocKeep --
 *
 *  Changes how the document's destruction is handled.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Changes document configuration.
 *
 *----------------------------------------------------------------------------
 */

void
TclXML_libxml2_DocKeep(objPtr, keep)
     Tcl_Obj *objPtr;
     TclXML_libxml2_DocumentHandling keep;
{
  TclXML_libxml2_Document *tDocPtr;

  if (TclXML_libxml2_GetTclDocFromObj(NULL, objPtr, &tDocPtr) != TCL_OK) {
    return;
  }

  tDocPtr->keep = keep;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2_DocSetFromAny --
 *
 *  Finds the xmlDocPtr wrapper for a Tcl object.
 *
 * Results:
 *  Returns success code.
 *
 * Side effects:
 *  Changes the Tcl_Obj's internal rep.
 *
 *----------------------------------------------------------------------------
 */

int
TclXMLlibxml2_DocSetFromAny(interp, objPtr)
     Tcl_Interp *interp;
     Tcl_Obj *objPtr;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  Tcl_HashEntry *entryPtr;
  TclXML_libxml2_Document *tDocPtr;
  ObjList *listPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->documents, Tcl_GetStringFromObj(objPtr, NULL));

  if (entryPtr) {

    if (objPtr->typePtr != NULL && objPtr->typePtr->freeIntRepProc != NULL) {
      objPtr->typePtr->freeIntRepProc(objPtr);
    }

    objPtr->internalRep.otherValuePtr = Tcl_GetHashValue(entryPtr);
    objPtr->typePtr = &TclXMLlibxml2_DocObjType;

    tDocPtr = (TclXML_libxml2_Document *) objPtr->internalRep.otherValuePtr;

    /*
     * Add this object to the Tcl_Obj list.
     * NB. There should be no duplicates.
     */
    listPtr = (ObjList *) tDocPtr->objs;
    if (listPtr == NULL) {
      listPtr = (ObjList *) Tcl_Alloc(sizeof(ObjList));
      listPtr->objPtr = objPtr;
      listPtr->next = NULL;
      tDocPtr->objs = listPtr;
    } else {
      ObjList *newPtr;

      newPtr = (ObjList *) Tcl_Alloc(sizeof(ObjList));
      newPtr->objPtr = objPtr;
      newPtr->next = listPtr;
      tDocPtr->objs = (void *) newPtr;
    }

  } else {

    if (interp) {
      Tcl_ResetResult(interp);
      Tcl_AppendResult(interp, "token \"", Tcl_GetStringFromObj(objPtr, NULL), "\" is not a libxml2 document", NULL);
    }

    return TCL_ERROR;
  }

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2_DocUpdate --
 *
 *  Finds the token for a xmlDocPtr wrapper.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Changes the Tcl_Obj's string rep.
 *
 *----------------------------------------------------------------------------
 */

void
TclXMLlibxml2_DocUpdate(objPtr)
     Tcl_Obj *objPtr;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->docByPtr, objPtr->internalRep.otherValuePtr);
  Tcl_InvalidateStringRep(objPtr);
  if (entryPtr == NULL) {
    objPtr->bytes = NULL;
    objPtr->length = 0;
  } else {
    TclXML_libxml2_Document *tDocPtr = (TclXML_libxml2_Document *) Tcl_GetHashValue(entryPtr);
    objPtr->length = strlen(tDocPtr->token);
    objPtr->bytes = Tcl_Alloc(objPtr->length + 1);
    strcpy(objPtr->bytes, tDocPtr->token);
  }
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2_DocDup --
 *
 *  Duplicates the Tcl wrapper.
 *  NB. This does *not* copy the document itself - it simply creates
 *  another reference to the same document.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Changes the target Tcl_Obj.
 *
 *----------------------------------------------------------------------------
 */

void
TclXMLlibxml2_DocDup(srcPtr, dstPtr)
     Tcl_Obj *srcPtr;
     Tcl_Obj *dstPtr;
{
  TclXML_libxml2_Document *tDocPtr;
  ObjList *listPtr;

  if (dstPtr->typePtr != NULL && dstPtr->typePtr->freeIntRepProc != NULL) {
    dstPtr->typePtr->freeIntRepProc(dstPtr);
  }

  tDocPtr = (TclXML_libxml2_Document *) srcPtr->internalRep.otherValuePtr;
  listPtr = (ObjList *) Tcl_Alloc(sizeof(ObjList));
  listPtr->objPtr = dstPtr;
  listPtr->next = ((ObjList *) tDocPtr->objs)->next;
  tDocPtr->objs = listPtr;

  Tcl_InvalidateStringRep(dstPtr);

  dstPtr->internalRep.otherValuePtr = srcPtr->internalRep.otherValuePtr;
  dstPtr->typePtr = srcPtr->typePtr;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2_DocFree --
 *
 *  Removes a Tcl wrapper to a libxml2 document.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  May free the document.
 *
 *----------------------------------------------------------------------------
 */

void
TclXMLlibxml2_DocFree(objPtr)
     Tcl_Obj *objPtr;
{
  TclXML_libxml2_Document *tDocPtr = (TclXML_libxml2_Document *) objPtr->internalRep.otherValuePtr;
  ObjList *listPtr = tDocPtr->objs;
  ObjList *prevPtr = NULL;

  while (listPtr) {
    if (listPtr->objPtr == objPtr) {
      break;
    }
    prevPtr = listPtr;
    listPtr = listPtr->next;
  }

  if (listPtr == NULL) {
    /* internal error */
  } else if (prevPtr == NULL) {
    tDocPtr->objs = listPtr->next;
  } else {
    prevPtr->next = listPtr->next;
  }
  Tcl_Free((char *) listPtr);

  if (tDocPtr->objs == NULL && tDocPtr->keep == TCLXML_LIBXML2_DOCUMENT_IMPLICIT) {
    DestroyTclDoc(tDocPtr);
  }

  objPtr->internalRep.otherValuePtr = NULL;
  objPtr->typePtr = NULL;
}

/*
 *----------------------------------------------------------------------------
 *
 * DestroyTclDoc --
 *
 *  Destroy the Tcl wrapper for a document.
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Free memory.
 *
 *----------------------------------------------------------------------------
 */

void
DestroyTclDoc(tDocPtr)
     TclXML_libxml2_Document *tDocPtr;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  Tcl_HashEntry *entryPtr;

  if (tDocPtr->domfree) {
    (tDocPtr->domfree)(tDocPtr->dom);
  }
  if (tDocPtr->appfree) {
    (tDocPtr->appfree)(tDocPtr->dom);
  }

  entryPtr = Tcl_FindHashEntry(tsdPtr->documents, tDocPtr->token);
  if (entryPtr) {
    Tcl_DeleteHashEntry(entryPtr);
  } else {
    /* Internal error */
  }

  entryPtr = Tcl_FindHashEntry(tsdPtr->docByPtr, (ClientData) tDocPtr->docPtr);
  if (entryPtr) {
    Tcl_DeleteHashEntry(entryPtr);
  } else {
    /* Internal error */
  }

  Tcl_MutexLock(&libxml2);
  xmlFreeDoc(tDocPtr->docPtr);
  Tcl_MutexUnlock(&libxml2);

  Tcl_Free(tDocPtr->token);
  Tcl_Free((char *) tDocPtr);
}
