/* tcllibxml2.c --
 *
 *	A Tcl wrapper for libxml2.
 *
 * Copyright (c) 2003 Zveno Pty Ltd
 * http://www.zveno.com/
 *
 * Zveno Pty Ltd makes this software and associated documentation
 * available free of charge for any purpose.  You may make copies
 * of the software but you must include all of this notice on any copy.
 *
 * Zveno Pty Ltd does not warrant that this software is error free
 * or fit for any purpose.  Zveno Pty Ltd disclaims any liability for
 * all claims, expenses, losses, damages and costs any user may incur
 * as a result of using, copying or modifying the software.
 *
 * $Id: tcllibxml2.c,v 1.1.2.7 2003/08/19 07:20:13 balls Exp $
 */

#include "tcllibxml2.h"
#include "tclInt.h"
#include <libxml/tree.h>
#include <libxml/parserInternals.h>

#define TCL_DOES_STUBS \
    (TCL_MAJOR_VERSION > 8 || TCL_MAJOR_VERSION == 8 && (TCL_MINOR_VERSION > 1 || \
    (TCL_MINOR_VERSION == 1 && TCL_RELEASE_LEVEL == TCL_FINAL_RELEASE)))

#undef TCL_STORAGE_CLASS
#define TCL_STORAGE_CLASS DLLEXPORT

/*
 * The structure below is used to refer to a libxml2 parser object.
 */

typedef struct TclXMLlibxml2Info {
  Tcl_Interp *interp;		/* Interpreter for this instance */

  xmlParserCtxtPtr ctxt;	/* libxml2 parser context */
  xmlSAXHandlerPtr defaultHandler; /* Default SAX handlers */
  Tcl_Obj *docObjPtr;		/* Result of parsing */
  TclXML_libxml2_DocumentHandling keep;	/* Document handling flag */

  TclXML_Info *xmlinfo;		/* Generic data structure */

  xmlExternalEntityLoader defaultLoader; /* default loader overridden while parsing */

  Tcl_HashTable *scope;		/* XML namespaces in scope */

} TclXMLlibxml2Info;

/*
 * Forward declarations for private functions.
 */

static void TclXMLlibxml2GenericError _ANSI_ARGS_((void *ctx, const char *msg, ...));

static ClientData TclXMLlibxml2Create _ANSI_ARGS_((Tcl_Interp *interp, 
                    TclXML_Info *xmlinfo));
/*
static ClientData TclXMLlibxml2CreateEntityParser _ANSI_ARGS_((
                    Tcl_Interp *interp, ClientData clientData));
*/
static int	TclXMLlibxml2Delete _ANSI_ARGS_((ClientData clientData));
static int	TclXMLlibxml2Parse _ANSI_ARGS_((ClientData clientData, 
                    char *data, int len, int final));
static int	TclXMLlibxml2Configure _ANSI_ARGS_((ClientData clientdata, 
						    Tcl_Obj *CONST optionPtr,
						    Tcl_Obj *CONST valuePtr));
static int	TclXMLlibxml2Get _ANSI_ARGS_((ClientData clientData, 
                    int objc, Tcl_Obj *CONST objv[]));

/*
 * Externally visible functions
 */

/*
 * libxml2 callbacks
 */

static void TclXMLlibxml2StartElement _ANSI_ARGS_((void *ctx,
						   CONST xmlChar *fullname,
						   CONST xmlChar **atts));
static void TclXMLlibxml2EndElement _ANSI_ARGS_((void *ctx,
						 CONST xmlChar *name));
static void TclXMLlibxml2Characters _ANSI_ARGS_((void *ctx,
						 CONST xmlChar *ch,
						 int len));
static void TclXMLlibxml2CDATABlock _ANSI_ARGS_((void *ctx,
						 CONST xmlChar *ch,
						 int len));
static void TclXMLlibxml2IgnorableWS _ANSI_ARGS_((void *ctx,
						 CONST xmlChar *ch,
						 int len));
static void TclXMLlibxml2PI _ANSI_ARGS_((void *ctx,
                                         CONST xmlChar *target,
                                         CONST xmlChar *data));
static void TclXMLlibxml2Comment _ANSI_ARGS_((void *ctx,
                                         CONST xmlChar *value));
static xmlParserInputPtr TclXMLlibxml2ExternalEntity _ANSI_ARGS_((void *ctx,
                                         CONST xmlChar *publicId,
                                         CONST xmlChar *systemId));
static void TclXMLlibxml2NotationDecl _ANSI_ARGS_((void *ctx,
                                         CONST xmlChar *name,
                                         CONST xmlChar *publicId,
                                         CONST xmlChar *systemId));
static void TclXMLlibxml2ElementDecl _ANSI_ARGS_((void *ctx,
                                         CONST xmlChar *name,
                                         int type,
                                         xmlElementContentPtr content));
static void TclXMLlibxml2AttributeDecl _ANSI_ARGS_((void *ctx,
                                         CONST xmlChar *elem,
                                         CONST xmlChar *fullname,
                                         int type,
                                         int def,
                                         CONST xmlChar *defaultValue,
                                         xmlEnumerationPtr tree));

static xmlParserInputPtr TclXMLlibxml2ExternalEntityLoader _ANSI_ARGS_((const char *URL,
								 const char *ID,
								 xmlParserCtxtPtr ctxt));

/*
 * libxml2 private functions
 */

EXTERN xmlParserCtxtPtr xmlCreateMemoryParserCtxt _ANSI_ARGS_((const char *buffer, int size));

typedef struct ThreadSpecificData {

  /*
   * Map parser context to data structure
   */

  Tcl_HashTable *contexts;

  /*
   * Interpose on default external entity loader
   */

  xmlExternalEntityLoader defaultLoader;

} ThreadSpecificData;
static Tcl_ThreadDataKey dataKey;

/*
 * libxml2 is mostly thread-safe, but there are issues with error callbacks
 */

TCL_DECLARE_MUTEX(libxml2)

/*
 * Switch tables
 */

#ifndef CONST84
#define CONST84 /* Before 8.4 no 'const' required */
#endif

/*
 * Error context for passing error result back to caller.
 */

typedef struct GenericError_Info {
  Tcl_Interp *interp;
  int code;
  Tcl_Obj *msg;
} GenericError_Info;

/*
 * Default values
 */

EXTERN int xmlLoadExtDtdDefaultValue;


/*
 *----------------------------------------------------------------------------
 *
 * Tclxml_libxml2_Init --
 *
 *  Initialisation routine for loadable module
 *
 * Results:
 *  None.
 *
 * Side effects:
 *  Creates commands in the interpreter,
 *
 *----------------------------------------------------------------------------
 */

int
Tclxml_libxml2_Init (interp)
     Tcl_Interp *interp;	/* Interpreter to initialise */
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXML_ParserClassInfo *classinfo;

#ifdef USE_TCL_STUBS
  if (Tcl_InitStubs(interp, "8.1", 0) == NULL) {
    return TCL_ERROR;
  }
#endif
#ifdef USE_TCLXML_STUBS
  if (TclXML_InitStubs(interp, TCLXML_VERSION, 1) == NULL) {
    return TCL_ERROR;
  }
#endif

  classinfo = (TclXML_ParserClassInfo *) ckalloc(sizeof(TclXML_ParserClassInfo));
  classinfo->name = Tcl_NewStringObj("libxml2", -1);
  classinfo->create = TclXMLlibxml2Create;
  classinfo->createCmd = NULL;
  classinfo->createEntity = NULL; /* TclXMLlibxml2CreateEntityParser; */
  classinfo->createEntityCmd = NULL;
  classinfo->parse = TclXMLlibxml2Parse;
  classinfo->parseCmd = NULL;
  classinfo->configure = TclXMLlibxml2Configure;
  classinfo->configureCmd = NULL;
  classinfo->get = TclXMLlibxml2Get;
  classinfo->getCmd = NULL;
  classinfo->destroy = TclXMLlibxml2Delete;
  classinfo->destroyCmd = NULL;
  classinfo->reset = NULL;
  classinfo->resetCmd = NULL;

  if (TclXML_RegisterXMLParser(interp, classinfo) != TCL_OK) {
    Tcl_SetResult(interp, "unable to register parser", NULL);
    return TCL_ERROR;
  }

  /* Configure the libxml2 parser */

  Tcl_MutexLock(&libxml2);

  xmlInitParser();
  xmlSubstituteEntitiesDefault(1);

  /*
   * 6 will load external entities.
   * 0 will not.
   * TODO: provide configuration option for setting this value.
   */
  xmlLoadExtDtdDefaultValue = 6;

  /* Setting the variable is insufficient - must create namespace too. */
  if (Tcl_VarEval(interp, "namespace eval ::xml::libxml2 {variable libxml2version ", xmlParserVersion, "}\n", NULL) != TCL_OK) {
    return TCL_ERROR;
  }

  Tcl_MutexUnlock(&libxml2);

  tsdPtr->contexts = (Tcl_HashTable *) Tcl_Alloc(sizeof(Tcl_HashTable));
  Tcl_InitHashTable(tsdPtr->contexts, TCL_ONE_WORD_KEYS);

  TclXML_libxml2_InitDocObj(interp);

  /*
   * TclExpat doesn't attempt to use the stubs API, so follow suit.
  #if TCL_DOES_STUBS
  */
  #if 0
    {
      extern Tcllibxml2Stubs tcllibxml2Stubs;
      if (Tcl_PkgProvideEx(interp, "xml::libxml2", TCLXML_LIBXML2_VERSION,
	(ClientData) &tcllibxml2Stubs) != TCL_OK) {
        return TCL_ERROR;
      }
    }
  #else
    if (Tcl_PkgProvide(interp, "xml::libxml2", TCLXML_LIBXML2_VERSION) != TCL_OK) {
      return TCL_ERROR;
    }
  #endif

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2Create --
 *
 *	Prepare for parsing.
 *	This doesn't actually do much; most of the
 *	action happens at the point of parsing.
 *
 * Results:
 *	Standard Tcl result.
 *
 * Side effects:
 *	This creates a libxml2 parser.
 *
 *----------------------------------------------------------------------------
 */

static ClientData
TclXMLlibxml2Create(interp, xmlinfo)
     Tcl_Interp *interp;
     TclXML_Info *xmlinfo;
{
  TclXMLlibxml2Info *info;

  /*
   * Create the data structures for this parser.
   */

  if (!(info = (TclXMLlibxml2Info *) Tcl_Alloc(sizeof(TclXMLlibxml2Info)))) {
    Tcl_Free((char *) info);
    Tcl_SetResult(interp, "unable to create parser", NULL);
    return NULL;
  }
  info->interp = interp;
  info->xmlinfo = xmlinfo;
  info->ctxt = NULL;
  info->docObjPtr = NULL;
  info->keep = TCLXML_LIBXML2_DOCUMENT_IMPLICIT;
  info->scope = (Tcl_HashTable *) Tcl_Alloc(sizeof(Tcl_HashTable));
  Tcl_InitHashTable(info->scope, TCL_STRING_KEYS);

  return (ClientData) info;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2Delete --
 *
 *	Destroy the libxml2 parser structure.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Frees any memory allocated for the XML parser.
 *
 *----------------------------------------------------------------------------
 */

static int
TclXMLlibxml2Delete(clientData)
     ClientData clientData;
{
  TclXMLlibxml2Info *info = (TclXMLlibxml2Info *) clientData;

  if (info->ctxt) {
    /*
     * NB. info->ctxt->myDoc is not freed.
     */
    Tcl_MutexLock(&libxml2);
    xmlFreeParserCtxt(info->ctxt);
    Tcl_MutexUnlock(&libxml2);
  }
  if (info->docObjPtr) {
    Tcl_DecrRefCount(info->docObjPtr);
  }
  Tcl_DeleteHashTable(info->scope);
  Tcl_Free((char *) info->scope);
  Tcl_Free((char *) info);

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2Parse --
 *
 *	Wrapper to invoke libxml2 parser and check return result.
 *
 *	NB. Most of the logic from xmlSAXUserParseMemory is used here.
 *
 * Results:
 *     TCL_OK if no errors, TCL_ERROR otherwise.
 *
 * Side effects:
 *     Sets interpreter result as appropriate.
 *
 *----------------------------------------------------------------------------
 */

static int
TclXMLlibxml2Parse(clientData, data, len, final)
     ClientData clientData;
     char *data;
     int len;
     int final;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXMLlibxml2Info *info = (TclXMLlibxml2Info *) clientData;
  xmlSAXHandlerPtr saxHandler, oldsax;
  xmlDocPtr docPtr;
  GenericError_Info *errorInfoPtr;
  xmlGenericErrorFunc old_xmlGenericError;
  void * oldErrorCtx;
  Tcl_HashEntry *entryPtr;
  int new;

  if (final == 0) {
    Tcl_SetResult(info->interp, "partial input not yet supported", NULL);
    return TCL_ERROR;
  }

  Tcl_MutexLock(&libxml2);

  info->ctxt = xmlCreateMemoryParserCtxt(data, len);

  if (!info->ctxt) {
    Tcl_SetResult(info->interp, "unable to create parser context", NULL);

    Tcl_MutexUnlock(&libxml2);
    return TCL_ERROR;
  }

  /*
   * Dynamically setup the SAX handler, since we only want to interpose
   * on those operations for which a TclXML callback has been defined.
   * We also setup a SAX handler to store the original (default) handlers.
   */

  saxHandler = (xmlSAXHandlerPtr) Tcl_Alloc(sizeof(struct _xmlSAXHandler));
  initxmlDefaultSAXHandler(saxHandler, xmlGetWarningsDefaultValue);
  info->defaultHandler = (xmlSAXHandlerPtr) Tcl_Alloc(sizeof(struct _xmlSAXHandler));
  initxmlDefaultSAXHandler(info->defaultHandler, xmlGetWarningsDefaultValue);
  /* TODO: implement all callbacks */
  saxHandler->startElement = TclXMLlibxml2StartElement;
  saxHandler->endElement = TclXMLlibxml2EndElement;
  saxHandler->characters = TclXMLlibxml2Characters;
  saxHandler->cdataBlock = TclXMLlibxml2CDATABlock;
  if (!info->xmlinfo->nowhitespace) {
    saxHandler->ignorableWhitespace = TclXMLlibxml2IgnorableWS;
  }
  saxHandler->processingInstruction = TclXMLlibxml2PI;
  saxHandler->comment = TclXMLlibxml2Comment;
  saxHandler->notationDecl = TclXMLlibxml2NotationDecl;
  saxHandler->elementDecl = TclXMLlibxml2ElementDecl;
  saxHandler->attributeDecl = TclXMLlibxml2AttributeDecl;
  saxHandler->resolveEntity = TclXMLlibxml2ExternalEntity;

  oldsax = info->ctxt->sax;
  info->ctxt->sax = saxHandler;
  entryPtr = Tcl_CreateHashEntry(tsdPtr->contexts, (ClientData) info->ctxt, &new);
  if (!entryPtr) {
    info->ctxt->sax = oldsax;
    xmlFreeParserCtxt(info->ctxt);
    info->ctxt = NULL;
    Tcl_Free((char *) saxHandler);
    Tcl_SetResult(info->interp, "unable to create hash entry (internal error)", NULL);

    Tcl_MutexUnlock(&libxml2);
    return TCL_ERROR;
  }
  Tcl_SetHashValue(entryPtr, (ClientData) info);

  if (info->xmlinfo->base) {
    info->ctxt->input->filename = Tcl_GetStringFromObj(info->xmlinfo->base, NULL);
  }

  tsdPtr->defaultLoader = xmlGetExternalEntityLoader();
  if (info->xmlinfo->entity || info->xmlinfo->entitycommand) {
    xmlSetExternalEntityLoader(TclXMLlibxml2ExternalEntityLoader);
  }

  /*
   * Create a generic error handler... just in case
   */

  errorInfoPtr = (GenericError_Info *) Tcl_Alloc(sizeof(GenericError_Info));
  errorInfoPtr->msg = NULL;
  errorInfoPtr->interp = info->interp;
  errorInfoPtr->code = TCL_OK;

  /*
   * Save the previous error context so that it can
   * be restored upon completion of parsing.
   */
  old_xmlGenericError = xmlGenericError;
  oldErrorCtx = xmlGenericErrorContext;

  xmlSetGenericErrorFunc((void *) errorInfoPtr, TclXMLlibxml2GenericError);

  xmlParseDocument(info->ctxt);
  if (info->ctxt->wellFormed) {
    docPtr = info->ctxt->myDoc;
    info->docObjPtr = TclXML_libxml2_CreateObjFromDoc(info->ctxt->myDoc);
    Tcl_IncrRefCount(info->docObjPtr);
  } else {
    docPtr = NULL;
    info->docObjPtr = NULL;
  }

  info->ctxt->sax = oldsax;
  xmlFreeParserCtxt(info->ctxt); /* does not free info->ctxt->myDoc */
  info->ctxt = NULL;

  xmlSetExternalEntityLoader(tsdPtr->defaultLoader);
  xmlCleanupParser();

  xmlSetGenericErrorFunc((void *) oldErrorCtx, old_xmlGenericError);

  if (docPtr == NULL) {
    Tcl_MutexUnlock(&libxml2);
    if (errorInfoPtr->msg) {
      Tcl_SetObjResult(info->interp, errorInfoPtr->msg);
    } else {
      Tcl_SetResult(info->interp, "document not well-formed", NULL);
    }
    return TCL_ERROR;
  }

  /*
   * Make sure base URI is stored in the document.
   * Setting the input filename is insufficient.
   */

  if (info->xmlinfo->base && docPtr->URL == NULL) {
    char * buf;
    int len;

    buf = Tcl_GetStringFromObj(info->xmlinfo->base, &len);
    docPtr->URL = Tcl_Alloc(len + 1);
    strcpy((char *) docPtr->URL, buf);
  }

  Tcl_MutexUnlock(&libxml2);

  TclXML_libxml2_DocKeep(info->docObjPtr, info->keep);

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 *	Callback handlers.
 *
 *	The mutex is locked during parsing, so no need to acquire it below here.
 *
 *----------------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2StartElement --
 *
 *	Start tag callback implementation
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Depends on application callback.
 *
 *----------------------------------------------------------------------------
 */

static void
TclXMLlibxml2StartElement(ctx, fullname, atts)
     void *ctx;
     CONST xmlChar *fullname;
     CONST xmlChar **atts;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXMLlibxml2Info *info;
  xmlParserCtxtPtr ctxt = (xmlParserCtxtPtr) ctx;
  xmlNsPtr ns;
  Tcl_Obj *attListObj, *nsDeclsObj, *nsuri;
  int count;
  Tcl_HashEntry *entryPtr;
  xmlChar *prefix;
  char *localpart;

  entryPtr = Tcl_FindHashEntry(tsdPtr->contexts, ctx);
  if (!entryPtr) {
    /* Background error */
    return;
  }
  info = (TclXMLlibxml2Info *) Tcl_GetHashValue(entryPtr);

  (info->defaultHandler->startElement)(ctx, fullname, atts);

  attListObj = Tcl_NewListObj(0, NULL);
  nsDeclsObj = Tcl_NewListObj(0, NULL);
  if (atts) {
    for (count = 0; atts[count]; count += 2) {
      if (Tcl_StringMatch(atts[count], "xmlns:*")) {
	Tcl_ListObjAppendElement(NULL, nsDeclsObj, Tcl_NewStringObj(atts[count + 1], -1));
	Tcl_ListObjAppendElement(NULL, nsDeclsObj, Tcl_NewStringObj(atts[count] + strlen("xmlns:"), -1));
      } else if (Tcl_StringMatch(atts[count], "xmlns")) {
	Tcl_ListObjAppendElement(NULL, nsDeclsObj, Tcl_NewStringObj(atts[count + 1], -1));
	Tcl_ListObjAppendElement(NULL, nsDeclsObj, Tcl_NewObj());
      } else {
	Tcl_ListObjAppendElement(NULL, attListObj, Tcl_NewStringObj(atts[count], -1));
	Tcl_ListObjAppendElement(NULL, attListObj, Tcl_NewStringObj(atts[count + 1], -1));
      }
    }
  }

  localpart = xmlSplitQName(ctxt, fullname, &prefix);
  if (prefix != NULL) {
    ns = xmlSearchNs(ctxt->myDoc, ctxt->node, prefix);
    if (ns != NULL) {
      nsuri = Tcl_NewStringObj(ns->href, -1);
    } else {
      /* This is an error */
      nsuri = NULL;
    }
  } else {
    /* check for default namespace */
    ns = xmlSearchNs(ctxt->myDoc, ctxt->node, NULL);
    if (ns != NULL) {
      nsuri = Tcl_NewStringObj(ns->href, -1);
    } else {
      nsuri = NULL;
    }
  }

  TclXML_ElementStartHandler(info->xmlinfo,
			     Tcl_NewStringObj(localpart, -1),
			     nsuri,
			     attListObj, nsDeclsObj);

}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2EndElement --
 *
 *	End tag callback implementation
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Depends on application callback.
 *
 *----------------------------------------------------------------------------
 */

static void
TclXMLlibxml2EndElement(ctx, name)
     void *ctx;
     CONST xmlChar *name;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXMLlibxml2Info *info;
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->contexts, ctx);
  if (!entryPtr) {
    /* Background error */
    return;
  }
  info = (TclXMLlibxml2Info *) Tcl_GetHashValue(entryPtr);

  (info->defaultHandler->endElement)(ctx, name);

  TclXML_ElementEndHandler(info->xmlinfo,
	Tcl_NewStringObj(name, -1));
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2Characters --
 *
 *      Characters callback implementation
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Depends on application callback.
 *
 *----------------------------------------------------------------------------
 */

static void
TclXMLlibxml2Characters(ctx, ch, len)
     void *ctx;
     CONST xmlChar *ch;
     int len;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXMLlibxml2Info *info = (TclXMLlibxml2Info *) ctx;
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->contexts, ctx);
  if (!entryPtr) {
    /* Background error */
    return;
  }
  info = (TclXMLlibxml2Info *) Tcl_GetHashValue(entryPtr);

  (info->defaultHandler->characters)(ctx, ch, len);

  TclXML_CharacterDataHandler(info->xmlinfo,
	Tcl_NewStringObj(ch, len));
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2CDATABlock --
 *
 *      Characters callback implementation
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Depends on application callback.
 *
 *----------------------------------------------------------------------------
 */

static void
TclXMLlibxml2CDATABlock(ctx, ch, len)
     void *ctx;
     CONST xmlChar *ch;
     int len;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXMLlibxml2Info *info = (TclXMLlibxml2Info *) ctx;
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->contexts, ctx);
  if (!entryPtr) {
    /* Background error */
    return;
  }
  info = (TclXMLlibxml2Info *) Tcl_GetHashValue(entryPtr);

  (info->defaultHandler->cdataBlock)(ctx, ch, len);

  TclXML_CharacterDataHandler(info->xmlinfo,
	Tcl_NewStringObj(ch, len));
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2IgnorableWS --
 *
 *      Characters callback implementation
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Depends on application callback.
 *
 *----------------------------------------------------------------------------
 */

static void
TclXMLlibxml2IgnorableWS(ctx, ch, len)
     void *ctx;
     CONST xmlChar *ch;
     int len;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXMLlibxml2Info *info = (TclXMLlibxml2Info *) ctx;
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->contexts, ctx);
  if (!entryPtr) {
    /* Background error */
    return;
  }
  info = (TclXMLlibxml2Info *) Tcl_GetHashValue(entryPtr);

  (info->defaultHandler->ignorableWhitespace)(ctx, ch, len);

  TclXML_CharacterDataHandler(info->xmlinfo,
	Tcl_NewStringObj(ch, len));
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2PI --
 *
 *      Processing instruction callback implementation
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Depends on application callback.
 *
 *----------------------------------------------------------------------------
 */

static void
TclXMLlibxml2PI(ctx, target, data)
     void *ctx;
     CONST xmlChar *target;
     CONST xmlChar *data;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXMLlibxml2Info *info = (TclXMLlibxml2Info *) ctx;
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->contexts, ctx);
  if (!entryPtr) {
    /* Background error */
    return;
  }
  info = (TclXMLlibxml2Info *) Tcl_GetHashValue(entryPtr);

  (info->defaultHandler->processingInstruction)(ctx, target, data);

  TclXML_ProcessingInstructionHandler(info->xmlinfo,
	Tcl_NewStringObj(target, -1), Tcl_NewStringObj(data, -1));
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2Comment --
 *
 *      Comment callback implementation
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Depends on application callback.
 *
 *----------------------------------------------------------------------------
 */

static void
TclXMLlibxml2Comment(ctx, value)
     void *ctx;
     CONST xmlChar *value;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXMLlibxml2Info *info = (TclXMLlibxml2Info *) ctx;
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->contexts, ctx);
  if (!entryPtr) {
    /* Background error */
    return;
  }
  info = (TclXMLlibxml2Info *) Tcl_GetHashValue(entryPtr);

  (info->defaultHandler->comment)(ctx, value);

  TclXML_CommentHandler(info->xmlinfo,
	Tcl_NewStringObj(value, -1));
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2ExternalEntity --
 *
 *      Resolve external entity reference
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Depends on application callback.
 *
 *----------------------------------------------------------------------------
 */

static xmlParserInputPtr
TclXMLlibxml2ExternalEntity(ctx, publicId, systemId)
     void *ctx;
     CONST xmlChar *publicId;
     CONST xmlChar *systemId;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXMLlibxml2Info *info = (TclXMLlibxml2Info *) ctx;
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->contexts, ctx);
  if (!entryPtr) {
    /* Background error */
    return NULL;
  }
  info = (TclXMLlibxml2Info *) Tcl_GetHashValue(entryPtr);

  TclXML_ExternalEntityRefHandler(info->xmlinfo,
				  NULL, NULL,
				  Tcl_NewStringObj(systemId, -1),
				  Tcl_NewStringObj(publicId, -1));

  return (info->defaultHandler->resolveEntity)(ctx, publicId, systemId);
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2NotationDecl --
 *
 *      Notation declaration callback implementation
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Depends on application callback.
 *
 *----------------------------------------------------------------------------
 */

static void
TclXMLlibxml2NotationDecl(ctx, name, publicId, systemId)
     void *ctx;
     CONST xmlChar *name;
     CONST xmlChar *publicId;
     CONST xmlChar *systemId;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXMLlibxml2Info *info = (TclXMLlibxml2Info *) ctx;
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->contexts, ctx);
  if (!entryPtr) {
    /* Background error */
    return;
  }
  info = (TclXMLlibxml2Info *) Tcl_GetHashValue(entryPtr);

  (info->defaultHandler->notationDecl)(ctx, name, publicId, systemId);

  TclXML_NotationDeclHandler(info->xmlinfo,
	Tcl_NewStringObj(name, -1),
        Tcl_NewObj(),
        Tcl_NewStringObj(systemId, -1),
        Tcl_NewStringObj(publicId, -1));
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2ElementDecl --
 *
 *      Element declaration callback implementation
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Depends on application callback.
 *
 *----------------------------------------------------------------------------
 */

static void
TclXMLlibxml2ElementDecl(ctx, name, type, content)
     void *ctx;
     CONST xmlChar *name;
     int type;
     xmlElementContentPtr content;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXMLlibxml2Info *info = (TclXMLlibxml2Info *) ctx;
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->contexts, ctx);
  if (!entryPtr) {
    /* Background error */
    return;
  }
  info = (TclXMLlibxml2Info *) Tcl_GetHashValue(entryPtr);

  (info->defaultHandler->elementDecl)(ctx, name, type, content);

  /*
   * TODO: construct content spec
   */

  TclXML_ElementDeclHandler(info->xmlinfo,
	Tcl_NewStringObj(name, -1),
        Tcl_NewObj());
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2AttributeDecl --
 *
 *      Attribute declaration callback implementation
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     Depends on application callback.
 *
 *----------------------------------------------------------------------------
 */

static void
TclXMLlibxml2AttributeDecl(ctx, elem, fullname, type, def, defaultValue, tree)
     void *ctx;
     CONST xmlChar *elem;
     CONST xmlChar *fullname;
     int type;
     int def;
     CONST xmlChar *defaultValue;
     xmlEnumerationPtr tree;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXMLlibxml2Info *info = (TclXMLlibxml2Info *) ctx;
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->contexts, ctx);
  if (!entryPtr) {
    /* Background error */
    return;
  }
  info = (TclXMLlibxml2Info *) Tcl_GetHashValue(entryPtr);

  (info->defaultHandler->attributeDecl)(ctx, elem, fullname, type, def, defaultValue, tree);

  /*
   * TODO: construct attribute definition
   */

  TclXML_AttlistDeclHandler(info->xmlinfo,
	Tcl_NewStringObj(elem, -1),
        Tcl_NewObj());
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2Configure --
 *
 *      Set options for the parser.
 *
 * Results:
 *     None.
 *
 * Side effects:
 *     None (there are no options to set).
 *
 *----------------------------------------------------------------------------
 */

static int
TclXMLlibxml2Configure(clientData, optionPtr, valuePtr)
    ClientData clientData;
    Tcl_Obj *CONST optionPtr;
    Tcl_Obj *CONST valuePtr;
{
  TclXMLlibxml2Info *info = (TclXMLlibxml2Info *) clientData;
  int option, len;
  char *value;
  CONST84 char *Options[] = {
    "-keep",
    NULL
  };
  enum Options {
    OPTION_KEEP
  };
  CONST84 char *KeepOptions[] = {
    "normal",
    "implicit",
    NULL
  };
  enum KeepOptions {
    OPTION_KEEP_NORMAL,
    OPTION_KEEP_IMPLICIT
  };

  if (Tcl_GetIndexFromObj(info->interp, optionPtr, Options,
			  "option", 0, &option) != TCL_OK) {
    /*
     * Just ignore options we don't understand
     */
    return TCL_OK;
  }

  switch ((enum Options) option) {
  case OPTION_KEEP:

    value = Tcl_GetStringFromObj(valuePtr, &len);
    if (len == 0) {
      info->keep = TCLXML_LIBXML2_DOCUMENT_KEEP;
      if (info->docObjPtr) {
	TclXML_libxml2_DocKeep(info->docObjPtr, TCLXML_LIBXML2_DOCUMENT_KEEP);
	return TCL_BREAK;
      }
    } else {
      if (Tcl_GetIndexFromObj(info->interp, valuePtr, KeepOptions,
			      "value", 0, &option) != TCL_OK) {
	return TCL_ERROR;
      }
      switch ((enum KeepOptions) option) {
      case OPTION_KEEP_NORMAL:
	info->keep = TCLXML_LIBXML2_DOCUMENT_KEEP;
	if (info->docObjPtr) {
	  TclXML_libxml2_DocKeep(info->docObjPtr, TCLXML_LIBXML2_DOCUMENT_KEEP);
	}
	return TCL_BREAK;

      case OPTION_KEEP_IMPLICIT:
	info->keep = TCLXML_LIBXML2_DOCUMENT_IMPLICIT;
	if (info->docObjPtr) {
	  TclXML_libxml2_DocKeep(info->docObjPtr, TCLXML_LIBXML2_DOCUMENT_IMPLICIT);
	}
	return TCL_BREAK;

      default:
	Tcl_SetResult(info->interp, "bad value", NULL);
	return TCL_ERROR;
      }
    }

    break;

  default:

    Tcl_SetResult(info->interp, "no such option", NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2Get --
 *
 *      Retrieve data from the parser.
 *
 * Results:
 *     Depends on argument.
 *
 * Side effects:
 *     May create Tcl object.
 *
 *----------------------------------------------------------------------------
 */

static int
TclXMLlibxml2Get(clientData, objc, objv)
    ClientData clientData;
    int objc;
    Tcl_Obj *CONST objv[];
{
  TclXMLlibxml2Info *info = (TclXMLlibxml2Info *) clientData;
  CONST84 char *methods[] = {
    "document",
    NULL
  };
  enum methods {
    TCLXML_LIBXML2_GET_DOCUMENT
  };
  int option;

  if (objc != 1) {
    Tcl_WrongNumArgs(info->interp, 0, objv, "method");
    return TCL_ERROR;
  }

  if (Tcl_GetIndexFromObj(info->interp, objv[0], methods,
                          "method", 0, &option) != TCL_OK) {
    return TCL_ERROR;
  }

  switch ((enum methods) option) {
  case TCLXML_LIBXML2_GET_DOCUMENT:
    if (info->docObjPtr) {
      Tcl_SetObjResult(info->interp, info->docObjPtr);
    }

    break;

  default:
    Tcl_SetResult(info->interp, "unknown method", NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2ExternalEntityLoader --
 *
 *      Retrieve an external entity, allowing interposing by the application.
 *
 * Results:
 *     External entity parsed.
 *
 * Side effects:
 *     Depends on application callback.
 *
 *----------------------------------------------------------------------------
 */

static xmlParserInputPtr
TclXMLlibxml2ExternalEntityLoader(URL, ID, ctxt)
    const char *URL;
    const char *ID;
    xmlParserCtxtPtr ctxt;
{
  ThreadSpecificData *tsdPtr = TCL_TSD_INIT(&dataKey);
  TclXMLlibxml2Info *info;
  Tcl_HashEntry *entryPtr;

  entryPtr = Tcl_FindHashEntry(tsdPtr->contexts, (ClientData) ctxt);
  if (!entryPtr) {
    /* Background error */
    return NULL;
  }
  info = (TclXMLlibxml2Info *) Tcl_GetHashValue(entryPtr);

  if (!info) {
    return NULL;
  }

  if (info->xmlinfo->entity) {
    (info->xmlinfo->entity)(info->interp, info->xmlinfo->clientData, NULL, NULL, Tcl_NewStringObj(URL, -1), Tcl_NewStringObj(ID, -1));
  } else if (info->xmlinfo->entitycommand) {
    Tcl_Obj *cmdPtr = Tcl_DuplicateObj(info->xmlinfo->entitycommand);
    
    if (URL) {
      Tcl_ListObjAppendElement(info->interp, cmdPtr, Tcl_NewStringObj(URL, -1));
    } else {
      Tcl_ListObjAppendElement(info->interp, cmdPtr, Tcl_NewListObj(0, NULL));
    }
    if (ID) {
      Tcl_ListObjAppendElement(info->interp, cmdPtr, Tcl_NewStringObj(ID, -1));
    } else {
      Tcl_ListObjAppendElement(info->interp, cmdPtr, Tcl_NewListObj(0, NULL));
    }
  } else {
    return (tsdPtr->defaultLoader)(URL, ID, ctxt);
  }
  return NULL; /* never reached */
}

/*
 *----------------------------------------------------------------------------
 *
 * TclXMLlibxml2GenericError --
 *
 *      Handler for parser errors
 *
 * Results:
 *     Stores error message.
 *
 * Side effects:
 *     Parser will (eventually) return error condition.
 *
 *----------------------------------------------------------------------------
 */

static void
TclXMLlibxml2GenericError(void *ctx, const char *msg, ...)
{
  va_list args;
  GenericError_Info *errorInfoPtr = (GenericError_Info *) ctx;
  char buf[2048];
  int len;

  errorInfoPtr->code = TCL_ERROR;
  
  if (!errorInfoPtr->msg) {
    errorInfoPtr->msg = Tcl_NewObj();
    Tcl_IncrRefCount(errorInfoPtr->msg);
  }
  
  va_start(args,msg);
  len = vsnprintf(buf, 2047, msg, args);
  va_end(args);
  
  Tcl_AppendToObj(errorInfoPtr->msg, buf, len);

}
