# xsltcache.tcl --
#
#	Handles performing XSLT transformations,
#	caching documents and results.
#
# Copyright (c) 2002 Zveno Pty Ltd
# http://www.zveno.com/
#
# $Id: xsltcache.tcl,v 1.5 2002/12/10 05:28:59 balls Exp $

package require xslt 2.5

package provide xslt::cache 2.5

namespace eval xslt::cache {
    namespace export transform transformdoc flush

    variable sources
    array set sources {}
    variable stylesheets
    array set stylesheets {}
    variable results
    array set results {}
}

# xslt::cache::transform --
#
#	Perform an XSLT transformation.
#
# Arguments:
#	src	Filename of source document
#	ssheet	Filename of stylesheet document
#	args	Configuration options, stylesheet parameters
#
# Results:
#	Result document token

proc xslt::cache::transform {src ssheet args} {
    variable sources
    variable stylesheets
    variable results

    # Separate parameters from options
    set parameters {}
    set options {}
    foreach {key value} $args {
	switch -glob -- $key {
	    -* {
		lappend options $key $value
	    }
	    default {
		lappend parameters $key $value
	    }
	}
    }

    # Normalize the parameter list
    array set paramArray $parameters
    set parameters {}
    foreach name [lsort [array names paramArray]] {
	lappend parameters $name $paramArray($name)
    }

    set hash $src.$ssheet.$parameters

    array set opts {
	-xmlinclude 1
    }
    array set opts $options

    set readSource [ReadXML $src -xmlinclude $opts(-xmlinclude)]

    set readStylesheet 1
    if {[info exists stylesheets($ssheet)]} {
	if {[file mtime $ssheet] < $stylesheets($ssheet,time)} {
	    set readStylesheet 0
	}
    }
    if {$readStylesheet} {
	catch {rename $stylesheets($ssheet) {}}
	ReadXML $ssheet -xmlinclude $opts(-xmlinclude)

	set stylesheets($ssheet) [xslt::compile $sources($ssheet)]
	set stylesheets($ssheet,time) [clock seconds]
    }

    if {$readSource || $readStylesheet || ![info exists results($hash)]} {

	set results($hash) [eval [list $stylesheets($ssheet)] transform [list $sources($src)] $parameters]
	set results($hash,time) [clock seconds]
    }

    return $results($hash)
}

# xslt::cache::ReadXML --
#
#	Internal proc to manage parsing a document.
#	Used for both source and stylesheet documents.
#
# Arguments:
#	src	Filename of source document
#	args	Configuration options
#
# Results:
#	Returns 1 if document was read.  Returns 0 if document is cached.

proc xslt::cache::ReadXML {src args} {
    variable sources
    array set opts {
	-xmlinclude 1
    }
    array set opts $args

    set readSource 1
    if {[info exists sources($src)]} {
	if {[file mtime $src] < $sources($src,time)} {
	    set readSource 0
	}
    }
    if {$readSource} {
	catch {dom::destroy $sources($src)}
	set ch [open $src]
	set sources($src) [dom::parse [read $ch] -baseuri file://$src]
	close $ch
	if {$opts(-xmlinclude)} {
	    dom::xinclude $sources($src)
	}
	set sources($src,time) [clock seconds]
    }

    return $readSource
}

# xslt::cache::transformdoc --
#
#	Perform an XSLT transformation on a DOM document.
#
# Arguments:
#	src	DOM token of source document
#	ssheet	Filename of stylesheet document
#	args	Configuration options, stylesheet parameters
#
# Results:
#	Result document token

proc xslt::cache::transformdoc {src ssheet args} {
    variable sources
    variable stylesheets

    # Separate parameters from options
    set parameters {}
    set options {}
    foreach {key value} $args {
	switch -glob -- $key {
	    -* {
		lappend options $key $value
	    }
	    default {
		lappend parameters $key $value
	    }
	}
    }

    # Normalize the parameter list
    array set paramArray $parameters
    set parameters {}
    foreach name [lsort [array names paramArray]] {
	lappend parameters $name $paramArray($name)
    }

    array set opts {
	-xmlinclude 1
    }
    array set opts $options

    set readStylesheet 1
    if {[info exists stylesheets($ssheet)]} {
	if {[file mtime $ssheet] < $stylesheets($ssheet,time)} {
	    set readStylesheet 0
	}
    }
    if {$readStylesheet} {
	catch {rename $stylesheets($ssheet) {}}
	ReadXML $ssheet -xmlinclude $opts(-xmlinclude)

	set stylesheets($ssheet) [xslt::compile $sources($ssheet)]
	set stylesheets($ssheet,time) [clock seconds]
    }

    set result [eval [list $stylesheets($ssheet)] transform [list $src] $parameters]

    return $result
}

# ::xslt::cache::flush --
#
#	Flush the cache
#
# Arguments:
#	src	source document filename
#	ssheet	stylesheet document filename
#	args	parameters
#
# Results:
#	Returns the empty string.
#	If all arguments are given then all entries corresponding
#	to that transformation are destroyed.
#	If the source and/or stylesheet are given then all
#	entries corresponding to those documents are destroyed.

proc xslt::cache::flush {src ssheet args} {
    variable sources
    variable stylesheets
    variable results

    # Normalize parameter list
    array set paramArray $args
    set parameters {}
    foreach name [lsort [array names paramArray]] {
	lappend parameters $name $paramArray($name)
    }

    set hash $src.$ssheet.$parameters

    switch -glob [string length $src],[string length $ssheet],[llength $args] {
	0,0,* {
	    # Special case: flush all
	    unset sources
	    array set sources {}
	    unset stylesheets
	    array set stylesheets {}
	    unset results
	    array set results {}
	}

	0,*,0 {
	    # Flush all entries for the given stylesheet
	    catch {rename $stylesheets($ssheet) {}}
	    catch {unset stylesheets($ssheet)}
	    catch {unset stylesheets($ssheet,time)}

	    foreach entry [array names results *.$ssheet.*] {
		catch {dom::destroy $results($entry)}
		catch {unset results($entry)}
		catch {unset results($entry,time)}
	    }
	}

	*,0,0 {
	    # Flush all entries for the given source document
	    catch {dom::destroy $sources($src)}
	    catch {unset sources($src)}
	    catch {unset sources($src,time)}
	    foreach entry [array names results $src.*] {
		catch {dom::destroy $results($entry)}
		catch {unset results($entry)}
		catch {unset results($entry,time)}
	    }
	}

	default {
	    # Flush specific entry
	    catch {dom::destroy $results($hash)}
	    catch {unset results($hash)}
	    catch {unset results($hash,time)}
	}
    }
}
