########################################################################
#
#  xmlgen -- generate XML by writing Tcl code
#
# (C) 2002 Harald Kirsch
#
# $Revision: 1.2 $, $Date: 2002/05/05 15:20:49 $
########################################################################
namespace eval ::xmlgen {
  if {![info exist VERSION]} {
    set VERSION 0.0
  }
  package provide xmlgen $VERSION

  namespace export declaretag buffer put esc
  
  ## will be elongated and trimmed back by (recursive) calls to doTag,
  ## i.e. by tag-procs. However, it is only used by makeTagAndBody.
  variable indent ""

  ## a regular expression used by makeTagAndBody to identify
  ## tag-arguments which are attribute-value pairs as well as to
  ## dissect them into these two parts. The attribute name must match
  ## the definition of 'Name' found in the XML spec:
  ##    http://www.w3c.org/TR/2000/REC-xml-20001006#NT-Name
  ## 'CombiningChar' and 'Extender' are not yet considered.
  variable attrre {^ *([A-Za-z_:][a-zA-Z0-9_.:-]*)=(.*)}

  ## A pattern used with [string match] to check if the first body
  ## argument of a markup proc is a control character which describes
  ## how to handle the body.
  set controlchars {[-!+.]}

  ## A pattern used with [string match] to check if the control
  ## character argument of a markup proc requests printing rather than
  ## returning the marked up text.
  set controlprint {[-!+]}

  ## Output normally goes just to stdout, but proc buffer may be used
  ## to generate a context in which output is appended to this
  ## variable. 
  ## NOTE: this is not thread-save if two threads operate in the same
  ## interpreter. 
  variable buffer ""

  ## We want to replace the original puts by our own implementations
  ## depending on context. However, we need of course the original
  ## somewhere, so we keep it as tclputs. Then, initially, we make the
  ## "normal" ::puts an alias for the saved proc.
  rename ::puts ::xmlgen::tclputs
  interp alias {} ::puts   {}  ::xmlgen::tclputs

  ## As a convenience, there is a [put] in addition to [puts] which
  ## can have an arbitary number of arguments that are just printed
  ## By default, put will output to stdout. In the context of
  ## ::xmlgen::buffer, it appends to variable buffer.
  interp alias {} ::xmlgen::put   {} ::xmlgen::putStream 

  ## A drop in for the standard puts which is activated within a
  ## ::xmlgen::buffer.
  proc puts {args} {
    set i 0
    if {"-nonewline"==[lindex $args $i]} {
      set nl ""
      incr i
    } else {
      set nl \n
    }
    ## If there are still two args, the first is supposed to be an
    ## explicit output channel and we leave it to the original puts to
    ## handle that.
    if {[llength $args]-$i!=1} {
      eval tclputs $args
      return
    }
    variable buffer
    append buffer [lindex $args $i] $nl
  }
      

  ## The version of [put] used when collecting output in a buffer.
  proc putBuf {args} {
    variable buffer
    append buffer [join $args]
  }
  ## The version of [put] used when immediately printing to stdout.
  proc putStream {args} {
    tclputs -nonewline [join $args]
  }

  ## Arrange for further output to be appended to variable bufname
  ## instead of being send automatically to stdout
  proc buffer {bufname body} {
    ## save the current buffer locally
    variable buffer
    set keptBuffer $buffer
    set buffer {}

    interp alias {} ::xmlgen::put  {} ::xmlgen::putBuf
    interp alias {} ::puts         {} ::xmlgen::puts
    set err [catch {uplevel 1 $body}]
    interp alias {} ::puts         {} ::xmlgen::tclputs
    interp alias {} ::xmlgen::put  {} ::xmlgen::putStream 

    ## copy the collected buffer to the requested var and restore the
    ## previous buffer
    upvar $bufname b
    set b $buffer
    set buffer $keptBuffer
    if {$err} {
      return -code error -errorinfo $::errorInfo
    }
  }

  ## See manual page for description of this function.
  proc makeTagAndBody {tagname l {specialAttributes {}} } {
    variable attrre
    variable indent
    variable controlchars
    variable controlprint

    #regsub -all "\n\[\t ]*" $l . bla
    #puts stdout "\n$tagname: $bla"

    ## If specialAttributes is set, we put those attributes into the
    ## array instead of assembling them into the tag.
    if {"$specialAttributes"==""} {
      array set sAttr {}
    } else {
      upvar $specialAttributes sAttr
    }

    ## Collect arguments as long as they look like attribute-value
    ## pairs, i.e. as long as they match $attrre.
    set opentag "<$tagname"
    set L [llength $l]
    for {set i 0} {$i<$L} {incr i} {
      set arg [lindex $l $i]
      if {![regexp $attrre $arg -> attr value]} break
      if {[info exists sAttr($attr)] || ""=="$tagname"} {
	set sAttr($attr) $value
      } else {
	append opentag " $attr=\"[esc $value]\""
      }
    }
    
    ## If there is at least one element left in $l, it is already in
    ## arg and might be the argument controlling how to handle the
    ## body. 
    set haveControl 0;			# see safety belt below
    set control .
    if {$i<$L} {
      if {[string match $controlchars $arg]} {
	set control $arg
	incr i
	set haveControl 1
      } elseif {[string length $arg]==1} {
	append emsg \
	    "single character starting body not allowed to " \
	    "guard against bugs"
	return -code error $emsg
      }
    }

    ## If there are elements left in $l they are joined into the
    ## body. Otherwise the body is empty and opentag and closetag need
    ## special handling.
    if {$i<$L} {
      set body [lrange $l $i end]
      if 0 {
	## If the body is a one-element list, we unpack one list
	## level. Otherwise we are most likely on a continued line like
	##    table ! tr ! td bla
	## where the body of e.g. table has already several elements
	if {[llength $body]==1} {set body [lindex $body 0]}
      } else {
	set body [join $body]
      }
      append opentag ">"
      set closetag "</$tagname>"
      if {[string match $controlprint $control]} {
	set opentag "\n$indent$opentag"
	set closetag "\n$indent$closetag"
      }
      #regsub -all "\n\[\t ]*" $body . bla
      #puts stdout "body now: $bla\n"

    } else {
      ## Leave a space in front of "/>" for being able to use XHTML
      ## with most HTML-browsers
      set body {}
      append opentag " />"
      set closetag ""
    }

    ## Put on the safety belt
    if {!$haveControl && [regexp "^\[\t \]*\n" $body]} {
      append msg \
	  "body starts with newline but no control " \
	  "character was given:"
      set b [split $body \n]
      if {[llength $b]>3} {
	append msg [join [lrange $b 0 3] \n] "  ..."
      } else {
	 append msg $body
      }
      return -code error $msg
    }

    return [list $opentag $control $body $closetag]
  }

  ## Evaluates, substitutes and prints or just returns the body
  ## enclosed in the given opentag and closetag.
  proc runbody {opentag control body closetag} {
    switch -exact -- $control {
      "!" {
	variable indent
	set ind $indent
	append indent { }
	uplevel 1 [list ::xmlgen::put $opentag]
	uplevel 1 $body
	uplevel 1 [list ::xmlgen::put $closetag]
	set indent $ind
      }
      "+" {
	set body [string trim $body "\n \t"]
	uplevel 1 [list ::xmlgen::put $opentag]
	#uplevel 1 "::xmlgen::put \[subst {$body}\] {\n}"
	uplevel 1 "::xmlgen::put \[subst {$body}\]"
	uplevel 1 [list ::xmlgen::put $closetag]
      }
      "-" {
	set body [string trim $body "\n \t"]
	uplevel 1 [list ::xmlgen::put $opentag]
	uplevel 1 [list ::xmlgen::put $body]
	uplevel 1 [list ::xmlgen::put $closetag]
      }
      "." {
	return "$opentag$body$closetag"
      }
      default {
	return -code error "unknown control string `$control'"
      }
    }
  }
  
  ## Generic function to handle a tag-proc and its arguments.
  proc doTag {tagname args} {
    variable tagAndBodyProc

    foreach {opentag control body closetag} \
	[makeTagAndBody $tagname $args] break

    return [uplevel 1 [list ::xmlgen::runbody $opentag \
			   $control $body $closetag]]
  }

  ## Makes a tagname into a tag-proc by making it into an alias for
  ## doTag.
  proc declaretag {funcname {tagname {}}} {
    if {"$tagname"==""} {
      set tagname $funcname
    }
    set ns [string trimright [uplevel 1 "namespace current"] :]
    interp alias {} [set ns]::$funcname   {} ::xmlgen::doTag $tagname
  }

  ## Convert text so that it is safe to use it as an attribute value
  ## surrouned by double quotes as well as character data. See the
  ## definition of AttValue and CharData:
  ## http://www.w3c.org/TR/2000/REC-xml-20001006#NT-AttValue
  ## http://www.w3c.org/TR/2000/REC-xml-20001006#NT-CharData
  proc esc {args} {
    regsub -all "&" [eval concat $args] "\\&amp;" args
    regsub -all "<" $args "\\&lt;" args
    regsub -all ">" $args "\\&gt;" args
    regsub -all "\"" $args "\\&\#34;" args
    regsub -all "]" $args "\\&\#93;" args
    return $args
  }

  
}

########################################################################

## If sourced by another script, its time to return
if {"[file tail $argv0]"!=[file tail [info script]]} return

##
## trivial example code ################################################
##
## the better examples are found in the package htmlgen.

namespace import ::xmlgen::*

foreach x {b tt it code body title html p table td tr th pre hr} {
  declaretag $x $x
}

body {
  puts dddddddddddddddddd
}
buffer Page {
  body {
    puts dddddddddddddddddd
  }
}
puts ||$Page||
