/* frame.c,v 1.35 2005/05/22 22:20:37 jenglish Exp
 * Copyright (c) 2004, Joe English
 *
 * Tile widget set: frame and labelframe widgets
 *
 * Note to maintainers: UNDER NO CIRCUMSTANCES is anyone to add
 * the -colormap, -visual, or -container resources to this widget.
 * Thank you.
 */

#include <tk.h>

#include "tkTheme.h"
#include "widget.h"
#include "compat.h"

/* ======================================================================
 * +++ Frame widget:
 */

typedef struct
{
    Tcl_Obj	*borderWidthObj;
    Tcl_Obj	*paddingObj;
    Tcl_Obj	*reliefObj;
    Tcl_Obj 	*widthObj;
    Tcl_Obj 	*heightObj;
} FramePart;

typedef struct 
{
    WidgetCore	core;
    FramePart	frame;
    FRAME_COMPAT_DECLS
} Frame;

static Tk_OptionSpec FrameOptionSpecs[] =
{
    {TK_OPTION_PIXELS, "-borderwidth", "borderWidth", "BorderWidth", "0",
	Tk_Offset(Frame,frame.borderWidthObj), -1,
	0,0,GEOMETRY_CHANGED },
    {TK_OPTION_STRING, "-padding", "padding", "Pad", NULL,
	Tk_Offset(Frame,frame.paddingObj), -1, 
	TK_OPTION_NULL_OK,0,GEOMETRY_CHANGED },
    {TK_OPTION_RELIEF, "-relief", "relief", "Relief", NULL,
	Tk_Offset(Frame,frame.reliefObj), -1, 
	TK_OPTION_NULL_OK,0,0 },
    {TK_OPTION_PIXELS, "-width", "width", "Width", "0",
	Tk_Offset(Frame,frame.widthObj), -1,
	0,0,GEOMETRY_CHANGED },
    {TK_OPTION_PIXELS, "-height", "height", "Height", "0",
	Tk_Offset(Frame,frame.heightObj), -1,
	0,0,GEOMETRY_CHANGED },

    WIDGET_INHERIT_OPTIONS(CoreOptionSpecs)
};

FRAME_COMPAT_OPTIONS

static WidgetCommandSpec FrameCommands[] =
{
    { "configure",	WidgetConfigureCommand },
    { "cget",		WidgetCgetCommand },
    { "instate",	WidgetInstateCommand },
    { "state",  	WidgetStateCommand },
    { NULL, NULL }
};

/*
 * FrameMargins --
 * 	Compute internal margins for a frame widget.
 * 	This includes the -borderWidth, plus any additional -padding.
 * 	The Labelframe widget adjusts margins further.
 */
static Ttk_Padding FrameMargins(Frame *framePtr)
{
    Ttk_Padding margins = Ttk_UniformPadding(0);
    int borderWidth = 0;
    Ttk_Padding border;

    /* Check -padding:
     */
    if (framePtr->frame.paddingObj) {
	Ttk_GetPaddingFromObj(NULL, 
	    framePtr->core.tkwin, framePtr->frame.paddingObj, &margins);
    }

    /* Add padding for border:
     */
    if (framePtr->frame.borderWidthObj) {
	Tk_GetPixelsFromObj(NULL, 
	    framePtr->core.tkwin, framePtr->frame.borderWidthObj, &borderWidth);
	border = Ttk_UniformPadding((short)borderWidth);
    }

    return Ttk_AddPadding(margins, border);
}

/* FrameSize procedure --
 * 	The frame doesn't request a size of its own by default,
 * 	but it does have an internal border.  See also <<NOTE-SIZE>>
 */
static int FrameSize(void *recordPtr, int *widthPtr, int *heightPtr)
{
    Frame *framePtr = recordPtr;
    Ttk_Padding margins = FrameMargins(framePtr);
    Ttk_SetMargins(framePtr->core.tkwin, margins);
    return 0;
}

/*
 * FrameConfigure -- configure hook.
 *	<<NOTE-SIZE>> Usually the size of a frame is controlled by 
 *	a geometry manager (pack, grid); the -width and -height
 *	options are only effective if geometry propagation is turned
 *	off or if the [place] GM is used for child widgets.
 *
 *	To avoid geometry blinking, we issue a geometry request
 *	in the Configure hook instead of the Size hook, and only
 *	if -width and/or -height is nonzero.
 */

static int FrameConfigure(Tcl_Interp *interp, void *recordPtr, int mask)
{
    Frame *framePtr = recordPtr;
    int width, height;

    /*
     * Make sure -padding resource, if present, is correct:
     */
    if (framePtr->frame.paddingObj) {
	Ttk_Padding unused;
	if (Ttk_GetPaddingFromObj(interp,
		    	framePtr->core.tkwin,
			framePtr->frame.paddingObj,
			&unused) != TCL_OK) {
	    return TCL_ERROR;
	}
    }

    /* See <<NOTE-SIZE>>
     */
    if (  TCL_OK != Tk_GetPixelsFromObj(
	    interp,framePtr->core.tkwin,framePtr->frame.widthObj,&width)
       || TCL_OK != Tk_GetPixelsFromObj(
	    interp,framePtr->core.tkwin,framePtr->frame.heightObj,&height)
       ) 
    {
	return TCL_ERROR;
    }

    if (width > 0 || height > 0) {
	Tk_GeometryRequest(framePtr->core.tkwin, width, height);
    }

    return CoreConfigure(interp, recordPtr, mask);
}

/* public */
WidgetSpec FrameWidgetSpec =
{
    "TFrame",			/* className */
    sizeof(Frame),		/* recordSize */
    FrameCompatOptionSpecs,	/* optionSpecs */
    FrameCommands,		/* subcommands */
    NullInitialize,		/* initializeProc */
    NullCleanup,		/* cleanupProc */
    FrameConfigure,		/* configureProc */
    NullPostConfigure,		/* postConfigureProc */
    WidgetGetLayout, 		/* getLayoutProc */
    FrameSize,			/* sizeProc */
    WidgetDoLayout,		/* layoutProc */
    WidgetDisplay,		/* displayProc */
    WIDGET_SPEC_END		/* sentinel */
};

/* ======================================================================
 * +++ Labelframe widget:
 */

#define LABELFRAME_SPACE 4

typedef enum {
    LABELANCHOR_E, LABELANCHOR_EN, LABELANCHOR_ES,
    LABELANCHOR_N, LABELANCHOR_NE, LABELANCHOR_NW,
    LABELANCHOR_S, LABELANCHOR_SE, LABELANCHOR_SW,
    LABELANCHOR_W, LABELANCHOR_WN, LABELANCHOR_WS
} LabelAnchor;

static const char *labelAnchorStrings[] = {
    "e", "en", "es", "n", "ne", "nw", "s", "se", "sw", "w", "wn", "ws", NULL
};

/* LabelAnchorSide, LabelAnchorSticky --
 * 	Returns the side (resp. sticky flags) corresponding
 * 	to a LabelAnchor value.
 */
static Ttk_Side LabelAnchorSide(LabelAnchor labelAnchor)
{
    switch (labelAnchor) {
	default:
	case LABELANCHOR_W: case LABELANCHOR_WN: case LABELANCHOR_WS:
	    return TTK_SIDE_LEFT;
	case LABELANCHOR_E: case LABELANCHOR_EN: case LABELANCHOR_ES:
	    return TTK_SIDE_RIGHT;
	case LABELANCHOR_N: case LABELANCHOR_NE: case LABELANCHOR_NW:
	    return TTK_SIDE_TOP;
	case LABELANCHOR_S: case LABELANCHOR_SE: case LABELANCHOR_SW:
	    return TTK_SIDE_BOTTOM;
    }
}

static Ttk_Sticky LabelAnchorSticky(LabelAnchor labelAnchor)
{
    switch (labelAnchor) {
	default:		return 0;
	case LABELANCHOR_EN:
	case LABELANCHOR_WN:	return TTK_STICK_N;
	case LABELANCHOR_ES:
	case LABELANCHOR_WS:	return TTK_STICK_S;
	case LABELANCHOR_NE:
	case LABELANCHOR_SE:	return TTK_STICK_E;
	case LABELANCHOR_NW:
	case LABELANCHOR_SW:	return TTK_STICK_W;
    }
}

/*
 * Labelframe widget record:
 */
typedef struct {
    LabelAnchor	labelAnchor;
    Tcl_Obj	*textObj;
    Tcl_Obj	*fontObj;
    Tcl_Obj 	*underlineObj;
    Tcl_Obj	*labelWidgetObj;

    Tk_Window	labelWidget;	/* Set in configureProc */
    Ttk_Box	slaveParcel;	/* Current position of slave widget */ 
} LabelframePart;

typedef struct
{
    WidgetCore  	core;
    FramePart   	frame;
    LabelframePart	label;
    LABELFRAME_COMPAT_DECLS
} Labelframe;

static Tk_OptionSpec LabelframeOptionSpecs[] =
{
    {TK_OPTION_STRING_TABLE, "-labelanchor", "labelAnchor", "LabelAnchor",
	"nw", -1, Tk_Offset(Labelframe, label.labelAnchor),
        0, (ClientData) labelAnchorStrings, GEOMETRY_CHANGED},
    {TK_OPTION_STRING, "-text", "text", "Text", "",
	Tk_Offset(Labelframe,label.textObj), -1, 
	0,0,GEOMETRY_CHANGED }, 
    {TK_OPTION_FONT, "-font", "font", "Font", NULL,
	Tk_Offset(Labelframe,label.fontObj), -1, 
	TK_OPTION_NULL_OK,0,GEOMETRY_CHANGED },
    {TK_OPTION_INT, "-underline", "underline", "Underline",
	"-1", Tk_Offset(Labelframe,label.underlineObj), -1, 
	0,0,0 },
    {TK_OPTION_WINDOW, "-labelwidget", "labelWidget", "LabelWidget", NULL,
	Tk_Offset(Labelframe,label.labelWidgetObj), -1,
	TK_OPTION_NULL_OK,0,GEOMETRY_CHANGED },

    /*
     * Override Frame defaults:
     */
    {TK_OPTION_RELIEF, "-relief", "relief", "Relief", "groove",
	Tk_Offset(Labelframe,frame.reliefObj), -1, 
	TK_OPTION_NULL_OK,0,0 },
    {TK_OPTION_PIXELS, "-borderwidth", "borderWidth", "BorderWidth", "2",
	Tk_Offset(Labelframe,frame.borderWidthObj), -1,
	TK_OPTION_NULL_OK,0,GEOMETRY_CHANGED },

    WIDGET_INHERIT_OPTIONS(FrameOptionSpecs)
};

LABELFRAME_COMPAT_OPTIONS

/*
 * Maintainable(interp, slave, master) --
 * 	Utility routine.  Verifies that 'master' may be used to maintain
 *	the geometry of 'slave' (via Tk_MaintainGeometry):
 * 
 * 	+ 'master' is either 'slave's parent -OR-
 * 	+ 'master is a descendant of 'slave's parent.
 * 	+ 'slave' is not a toplevel window
 * 	+ 'slave' belongs to the same toplevel as 'master'
 *
 * Side effects:
 * 	+ If 'slave' is not a direct child of 'master', raises 'slave'
 * 	  above the sibling window containing 'master' 
 */

static int Maintainable(Tcl_Interp *interp, Tk_Window slave, Tk_Window master)
{
    Tk_Window ancestor = master, parent = Tk_Parent(slave), sibling = NULL;

    if (Tk_IsTopLevel(slave) || slave == master) {
	goto badWindow;
    }

    while (ancestor != parent) {
	if (Tk_IsTopLevel(ancestor)) {
	    goto badWindow;
	}
	sibling = ancestor;
	ancestor = Tk_Parent(ancestor);
    }

    if (sibling != NULL) {
	Tk_RestackWindow(slave, Above, sibling);
    }

    return 1;

badWindow:
    Tcl_AppendResult(interp, 
	"can't use ", Tk_PathName(slave),
	" as slave of ", Tk_PathName(master),
	NULL);
    return 0;
}

/*
 * LabelframeSlaveEventHandler --
 * 	Tracks when slave is destroyed, notifies parent.
 */
static void LabelframeGeometryLostSlaveProc(ClientData, Tk_Window);/*forward*/

static void
LabelframeSlaveEventHandler(ClientData clientData, XEvent *eventPtr)
{
    Labelframe *lframe = clientData;
    if (eventPtr->type == DestroyNotify) {
	LabelframeGeometryLostSlaveProc(clientData,lframe->label.labelWidget);
    }
}

/*
 * Labelframe geometry manager procedures:
 * ClientData is the labelframe widget.
 */
static void
LabelframeGeometryRequestProc(ClientData clientData, Tk_Window slave)
{
    Labelframe *lframePtr = clientData;
    WidgetChanged(&lframePtr->core, RELAYOUT_REQUIRED);
}

static void
LabelframeGeometryLostSlaveProc(ClientData clientData, Tk_Window slave)
{
    Labelframe *lframePtr = clientData;

    Tcl_DecrRefCount(lframePtr->label.labelWidgetObj);
    lframePtr->label.labelWidgetObj = 0;
    lframePtr->label.labelWidget = 0;

    /* See also: LabelframeUnregisterSlave */
    Tk_DeleteEventHandler(
	slave, StructureNotifyMask, LabelframeSlaveEventHandler, clientData);
    Tk_UnmaintainGeometry(slave, lframePtr->core.tkwin);

    WidgetChanged(&lframePtr->core, RELAYOUT_REQUIRED);
}

static Tk_GeomMgr LabelframeGeometryManager = {
    "labelframe",
    LabelframeGeometryRequestProc,
    LabelframeGeometryLostSlaveProc
};

static void
LabelframeRegisterSlave(Labelframe *lframePtr, Tk_Window slave)
{
    Tk_CreateEventHandler(
	slave, StructureNotifyMask, LabelframeSlaveEventHandler, lframePtr);
    Tk_ManageGeometry(slave, &LabelframeGeometryManager, lframePtr);
    lframePtr->label.slaveParcel = Ttk_MakeBox(-1,-1,-1,-1);
    /* Tk_MaintainGeometry() called later */
}

static void
LabelframeUnregisterSlave(Labelframe *lframePtr, Tk_Window slave)
{
    Tk_ManageGeometry(slave, NULL, 0);

    /* See also: LabelframeGeometryLostSlaveProc */
    Tk_DeleteEventHandler(
	slave, StructureNotifyMask, LabelframeSlaveEventHandler, lframePtr);
    Tk_UnmaintainGeometry(slave, lframePtr->core.tkwin);

    Tk_UnmapWindow(slave);
}

/* LabelframePlaceSlave --
 * 	Sets the position and size of the labelwidget.  
 */
static void
LabelframePlaceSlave(Labelframe *lframePtr, Ttk_Box parcel)
{
    Ttk_Box current = lframePtr->label.slaveParcel;

    if (   current.x == parcel.x
    	&& current.y == parcel.y
    	&& current.width == parcel.width
    	&& current.height == parcel.height
    ) {
    	/* Already in the right place - do nothing */
	return;
    }

    Tk_MaintainGeometry(
	lframePtr->label.labelWidget, lframePtr->core.tkwin,
	parcel.x,parcel.y, parcel.width,parcel.height);

    lframePtr->label.slaveParcel = parcel;
}

/* LabelframeLabelSize --
 * 	Extract the requested width and height of the labelframe's label:
 * 	taken from the label widget if specified, otherwise the text label.
 */

static void 
LabelframeLabelSize(Labelframe *lframePtr, int *widthPtr, int *heightPtr)
{
    WidgetCore *corePtr = &lframePtr->core;
    Tk_Window labelWidget = lframePtr->label.labelWidget;

    if (labelWidget) {
	*widthPtr = Tk_ReqWidth(labelWidget);
	*heightPtr = Tk_ReqHeight(labelWidget);
    } else {
	Ttk_LayoutNode *textNode = Ttk_LayoutFindNode(corePtr->layout, "text");
	if (textNode) { 
	    Ttk_LayoutNodeReqSize(
		corePtr->layout, textNode, widthPtr, heightPtr);
	} else {
	    *widthPtr = *heightPtr = 0;
	}
    }
}

/*
 * LabelframeSize --
 * 	Like the frame, this doesn't request a size of its own
 * 	but it does have internal padding and a minimum size.
 */
static int LabelframeSize(void *recordPtr, int *widthPtr, int *heightPtr)
{
    Labelframe *lframePtr = recordPtr;
    WidgetCore *corePtr = &lframePtr->core;
    Ttk_Padding margins = FrameMargins((Frame*)recordPtr);
    Tcl_Obj *outsideObj = Ttk_QueryOption(corePtr->layout,"-labeloutside", 0);
    int labelOutside = 0;
    int labelWidth, labelHeight;
    int borderWidth = 2;

    /* Adjust margins based on label size and position:
     */
    LabelframeLabelSize(lframePtr, &labelWidth, &labelHeight);

    /* Adjust for -labeloutside / -labelspace:
    */
    if (outsideObj)
	Tcl_GetBooleanFromObj(NULL, outsideObj, &labelOutside);
    if (labelOutside) {
	Tcl_Obj *labelspaceObj=Ttk_QueryOption(corePtr->layout,"-labelspace",0);
	int labelspace = 1;
	if (labelspaceObj)
	    Tk_GetPixelsFromObj(NULL,corePtr->tkwin,labelspaceObj,&labelspace);
	labelHeight += labelspace;
    }

    switch (LabelAnchorSide(lframePtr->label.labelAnchor)) {
	case TTK_SIDE_LEFT:	margins.left   += labelWidth;	break;
	case TTK_SIDE_RIGHT:	margins.right  += labelWidth;	break;
	case TTK_SIDE_TOP:	margins.top    += labelHeight;	break;
	case TTK_SIDE_BOTTOM:	margins.bottom += labelHeight;	break;
    }

    Ttk_SetMargins(corePtr->tkwin,margins);

    /* Request minimum size based on border width and label size:
     */
    Tk_GetPixelsFromObj(NULL, 
	    corePtr->tkwin, lframePtr->frame.borderWidthObj, &borderWidth);
    Tk_SetMinimumRequestSize(corePtr->tkwin, 
	    labelWidth + 2*borderWidth + LABELFRAME_SPACE,
	    labelHeight + 2*borderWidth + LABELFRAME_SPACE);

    return 0;
}

static void LabelframeDoLayout(void *recordPtr)
{
    Labelframe *lframePtr = recordPtr;
    WidgetCore *corePtr = &lframePtr->core;
    Ttk_Box borderParcel = Ttk_WinBox(corePtr->tkwin);
    Ttk_Side side = LabelAnchorSide(lframePtr->label.labelAnchor);
    Ttk_Sticky sticky = LabelAnchorSticky(lframePtr->label.labelAnchor);
    Ttk_LayoutNode
	*textNode = Ttk_LayoutFindNode(corePtr->layout, "text"),
	*borderNode = Ttk_LayoutFindNode(corePtr->layout, "border");
    Tcl_Obj *outsideObj = Ttk_QueryOption(corePtr->layout,"-labeloutside", 0);
    Tcl_Obj *insetObj = Ttk_QueryOption(corePtr->layout,"-labelinset", 0);
    int labelOutside = 0; 	/* true=>place label outside border */
    int labelinset = 0;		/* distance between label and adjacent corner */
    int lw, lh;			/* Label width and height */
    Ttk_Box labelParcel;

    /*
     * Do base layout:
     */
    Ttk_PlaceLayout(corePtr->layout,corePtr->state,borderParcel);

    /*
     * Compute label parcel:
     */
    LabelframeLabelSize(lframePtr, &lw, &lh);
    labelParcel = Ttk_PlaceBox(&borderParcel, lw, lh, side, sticky);

    /*
     * Place label on top of border, unless -labeloutside is set:
     */
    if (outsideObj) {
	Tcl_GetBooleanFromObj(NULL, outsideObj, &labelOutside);
    }
    if (!labelOutside) {
	/* Move border edge so it's over label:
	*/
	switch (side) {
	    case TTK_SIDE_LEFT: 	borderParcel.x -= lw / 2;
	    case TTK_SIDE_RIGHT:	borderParcel.width += lw/2; 	break;
	    case TTK_SIDE_TOP:  	borderParcel.y -= lh / 2;
	    case TTK_SIDE_BOTTOM:	borderParcel.height += lh / 2;	break;
	}
    } else {
	/* Add extra space between label and border:
	 */
	Tcl_Obj *labelspaceObj=Ttk_QueryOption(corePtr->layout,"-labelspace",0);
	int labelspace = 1;
	if (labelspaceObj)
	    Tk_GetPixelsFromObj(NULL,corePtr->tkwin,labelspaceObj,&labelspace);

	borderParcel.y += labelspace;
	borderParcel.height -= labelspace;
    }

    /* Move label away from corner:
    */
    if (insetObj) {
	/* Inset specified by style:
	 */
	Tk_GetPixelsFromObj(NULL, corePtr->tkwin, insetObj, &labelinset);
    } else if (!labelOutside) {		
	/* Compute default based on -borderwidth:
	 */
	Tk_GetPixelsFromObj(NULL, 
	    corePtr->tkwin, lframePtr->frame.borderWidthObj, &labelinset);
	labelinset = labelinset * 2 + LABELFRAME_SPACE;
    } else {			
	/* No extra inset
	 */
	labelinset = 0;
    }

    switch (lframePtr->label.labelAnchor) {
	default:			break; /*no-op*/
	case LABELANCHOR_NW:
	case LABELANCHOR_SW:	labelParcel.x += labelinset; break;
	case LABELANCHOR_NE:
	case LABELANCHOR_SE:	labelParcel.x -= labelinset; break;
	case LABELANCHOR_EN:
	case LABELANCHOR_WN:	labelParcel.y += labelinset; break;
	case LABELANCHOR_ES:
	case LABELANCHOR_WS:	labelParcel.y -= labelinset; break;
    }

    /* 
     * Place border and label:
     */
    if (borderNode)
	Ttk_LayoutNodeSetParcel(borderNode, borderParcel);
    if (lframePtr->label.labelWidget)
	LabelframePlaceSlave(lframePtr, labelParcel);
    if (textNode)
	Ttk_LayoutNodeSetParcel(textNode, labelParcel);
}

static void LabelframeCleanup(void *recordPtr)
{
    Labelframe *lframePtr = recordPtr;

    if (lframePtr->label.labelWidget) {
	LabelframeUnregisterSlave(lframePtr,lframePtr->label.labelWidget);
	lframePtr->label.labelWidget = 0;
    }
}

static int LabelframeConfigure(Tcl_Interp *interp,void *recordPtr,int mask)
{
    Labelframe *lframePtr = recordPtr;
    Tk_Window labelWidget=NULL, oldLabelWidget=lframePtr->label.labelWidget;

    /* Validate -labelwidget option:
     */
    if (lframePtr->label.labelWidgetObj) {
	const char *pathName = Tcl_GetString(lframePtr->label.labelWidgetObj);
	if (pathName && *pathName) {
	    labelWidget = 
		Tk_NameToWindow(interp, pathName, lframePtr->core.tkwin);
	    if (!labelWidget) {
		return TCL_ERROR;
	    }
	    if (!Maintainable(interp, labelWidget, lframePtr->core.tkwin)) {
		return TCL_ERROR;
	    }
	}
    }

    /* Base class configuration:
     */
    if (FrameConfigure(interp, recordPtr, mask) != TCL_OK) {
	return TCL_ERROR;
    }

    /* Update -labelwidget changes, if any:
     */
    if (labelWidget != oldLabelWidget) {
	if (oldLabelWidget) {
	    LabelframeUnregisterSlave(lframePtr, oldLabelWidget);
	}
	if (labelWidget) {
	    LabelframeRegisterSlave(lframePtr, labelWidget);
	}
	lframePtr->label.labelWidget = labelWidget;
    }

    return TCL_OK;
}

/* public */
WidgetSpec LabelframeWidgetSpec =
{
    "TLabelframe",		/* className */
    sizeof(Labelframe),		/* recordSize */
    LabelframeCompatOptionSpecs,/* optionSpecs */
    FrameCommands,		/* subcommands */
    NullInitialize,		/* initializeProc */
    LabelframeCleanup,		/* cleanupProc */
    LabelframeConfigure,	/* configureProc */
    NullPostConfigure,  	/* postConfigureProc */
    WidgetGetLayout, 		/* getLayoutProc */
    LabelframeSize,		/* sizeProc */
    LabelframeDoLayout,		/* layoutProc */
    WidgetDisplay,		/* displayProc */
    WIDGET_SPEC_END		/* sentinel */
};

