/* manager.c,v 1.2 2005/03/18 02:03:47 jenglish Exp
 *
 * Copyright 2005, Joe English.  Freely redistributable.
 *
 * Tile widget set: support routines for geometry managers.
 */

#include <string.h>
#include <tk.h>
#include "manager.h"

/*------------------------------------------------------------------------
 * +++ The Geometry Propagation Dance.
 *
 * When a slave window requests a new size or some other parameter changes,
 * the manager recomputes the required size for the master window and calls
 * Tk_GeometryRequest().  This is scheduled as an idle handler so multiple
 * updates can be processed as a single batch.
 *
 * If all goes well, the master's manager will process the request
 * (and so on up the chain to the toplevel window), and the master
 * window will eventually receive a <Configure> event.  At this point
 * it recomputes the size and position of all slaves and places them.
 *
 * If all does not go well, however, the master's request may be ignored
 * (typically because the top-level window has a fixed, user-specified size).
 * Tk doesn't provide any notification when this happens; to account for this,
 * we also schedule an idle handler to call the layout procedure
 * after making a geometry request.
 *
 * +++ Slave removal <<NOTE-LOSTSLAVE>>.
 *
 * There are three conditions under which a slave is removed:
 *
 * (1) Another GM claims control
 * (2) Manager voluntarily relinquishes control
 * (3) Slave is destroyed
 *
 * In case (1), Tk calls the manager's lostSlaveProc.
 * Case (2) is performed by calling Tk_ManageGeometry(slave,NULL,0);
 * in this case Tk does _not_ call the LostSlaveProc (documented behavior).
 * Tk doesn't handle case (3) either; to account for that we
 * register an event handler on the slave widget to track <Destroy> events.
 *
 */

#define MGR_UPDATE_PENDING	0x1
#define MGR_RESIZE_REQUIRED	0x2
#define MGR_RELAYOUT_REQUIRED	0x4

static void ManagerIdleProc(void *);	/* forward */

/* ++ ScheduleUpdate --
 * 	Schedule a call to recompute the size and/or layout,
 *	depending on flags.
 */
static void ScheduleUpdate(Manager *mgr, unsigned flags)
{
    if (!(mgr->flags & MGR_UPDATE_PENDING)) {
	Tcl_DoWhenIdle(ManagerIdleProc, mgr);
	mgr->flags |= MGR_UPDATE_PENDING;
    }
    mgr->flags |= flags;
}

/* ++ RecomputeSize --
 * 	Recomputes the required size of the master window,
 * 	makes geometry request.
 */
static void RecomputeSize(Manager *mgr)
{
    int width = 1, height = 1;

    if (mgr->managerSpec->sizeProc(mgr->managerData, &width, &height)) {
	Tk_GeometryRequest(mgr->masterWindow, width, height);
	ScheduleUpdate(mgr, MGR_RELAYOUT_REQUIRED);
    }
    mgr->flags &= ~MGR_RESIZE_REQUIRED;
}

/* ++ RecomputeLayout --
 * 	Recompute geometry of all slaves.
 */
static void RecomputeLayout(Manager *mgr)
{
    mgr->managerSpec->layoutProc(mgr->managerData);
    mgr->flags &= ~MGR_RELAYOUT_REQUIRED;
}

/* ++ ManagerIdleProc --
 * 	DoWhenIdle procedure for deferred updates.
 */
static void ManagerIdleProc(ClientData clientData)
{
    Manager *mgr = clientData;
    mgr->flags &= ~MGR_UPDATE_PENDING;

    if (mgr->flags & MGR_RESIZE_REQUIRED) {
	RecomputeSize(mgr);
    } else if (mgr->flags & MGR_RELAYOUT_REQUIRED) {
	RecomputeLayout(mgr);
    }
}

/*------------------------------------------------------------------------
 * +++ Event handlers.
 */

/* ++ ManagerEventHandler --
 * 	Recompute slave layout when master widget is resized.
 */
static const int ManagerEventMask = StructureNotifyMask;
static void ManagerEventHandler(ClientData clientData, XEvent *eventPtr)
{
    Manager *mgr = clientData;
    if (eventPtr->type == ConfigureNotify) {
	RecomputeLayout(mgr);
    }
}

/* ++ SlaveEventHandler --
 * 	Notifies manager when a slave is destroyed
 * 	(see <<NOTE-LOSTSLAVE>>).
 */
static const unsigned SlaveEventMask = StructureNotifyMask;
static void SlaveEventHandler(ClientData clientData, XEvent *eventPtr)
{
    Slave *slave = clientData;
    if (eventPtr->type == DestroyNotify) {
	slave->manager->managerSpec->tkGeomMgr.lostSlaveProc(
	    clientData, slave->slaveWindow);
    }
}

/*------------------------------------------------------------------------
 * +++ Manager initialization and cleanup.
 */

Manager *CreateManager(
    ManagerSpec *managerSpec, void *managerData, Tk_Window masterWindow)
{
    Manager *mgr = (Manager*)ckalloc(sizeof(*mgr));

    mgr->managerSpec 	= managerSpec;
    mgr->managerData	= managerData;
    mgr->masterWindow	= masterWindow;
    mgr->slaveOptionTable= 0;
    mgr->nSlaves 	= 0;
    mgr->slaves 	= NULL;
    mgr->flags  	= 0;

    Tk_CreateEventHandler(
	mgr->masterWindow, ManagerEventMask, ManagerEventHandler, mgr);

    return mgr;
}

void DeleteManager(Manager *mgr)
{
    Tk_DeleteEventHandler(
	mgr->masterWindow, ManagerEventMask, ManagerEventHandler, mgr);

    Tk_CancelIdleCall(ManagerIdleProc, mgr);

    if (mgr->slaveOptionTable)
	Tk_DeleteOptionTable(mgr->slaveOptionTable);

    if (mgr->slaves) {
	/* @@@ More cleanup needed here */
	ckfree((ClientData)mgr->slaves);
    }

    ckfree((ClientData)mgr);
}

/*------------------------------------------------------------------------
 * +++ Slave management.
 */
static Slave *CreateSlave(
    Tcl_Interp *interp, Manager *mgr, Tk_Window slaveWindow)
{
    Slave *slave = (Slave*)ckalloc(sizeof(*slave));
    int status;

    slave->slaveWindow = slaveWindow;
    slave->manager = mgr;
    slave->slaveData = ckalloc(mgr->managerSpec->slaveSize);
    memset(slave->slaveData, 0, mgr->managerSpec->slaveSize);

    if (!mgr->slaveOptionTable) {
	mgr->slaveOptionTable =
	    Tk_CreateOptionTable(interp, mgr->managerSpec->slaveOptionSpecs);
    }

    status = Tk_InitOptions(
	interp, slave->slaveData, mgr->slaveOptionTable, slaveWindow);

    if (status != TCL_OK) {
	ckfree((ClientData)slave->slaveData);
	ckfree((ClientData)slave);
	return NULL;
    }

    return slave;
}

static void DeleteSlave(Slave *slave)
{
    Tk_FreeConfigOptions(
	slave->slaveData, slave->manager->slaveOptionTable, slave->slaveWindow);
    ckfree((ClientData)slave->slaveData);
    ckfree((ClientData)slave);
}

/* ++ InsertSlave --
 * 	Adds slave to the list of managed windows.
 */
static void InsertSlave(Manager *mgr, Slave *slave, int index)
{
    int endIndex = mgr->nSlaves++;
    mgr->slaves = (Slave**)ckrealloc(
	    (ClientData)mgr->slaves, mgr->nSlaves * sizeof(Slave *));

    while (endIndex > index) {
	mgr->slaves[endIndex] = mgr->slaves[endIndex - 1];
	--endIndex;
    }

    mgr->slaves[index] = slave;

    Tk_ManageGeometry(slave->slaveWindow,
	&mgr->managerSpec->tkGeomMgr, (ClientData)slave);

    Tk_CreateEventHandler(slave->slaveWindow,
	SlaveEventMask, SlaveEventHandler, (ClientData)slave);

    mgr->managerSpec->SlaveAdded(mgr,slave);

    ScheduleUpdate(mgr, MGR_RESIZE_REQUIRED);
}

/* RemoveSlave --
 * 	Unmanage and delete the slave.
 *
 * NOTES/ASSUMPTIONS:
 *
 * [1] It's safe to call Tk_UnmapWindow(tkwin) even if this routine
 *     is called from the DestroyNotify binding.
 */
static void RemoveSlave(Manager *mgr, int index)
{
    Slave *slave = mgr->slaves[index];
    int i;

    /* Notify manager:
     */
    mgr->managerSpec->SlaveRemoved(mgr,slave);

    /* Remove from array:
     */
    --mgr->nSlaves;
    for (i = index ; i < mgr->nSlaves; ++i) {
	mgr->slaves[i] = mgr->slaves[i+1];
    }

    /* Clean up:
     */
    Tk_DeleteEventHandler(
	slave->slaveWindow, SlaveEventMask, SlaveEventHandler, slave);
    Tk_UnmapWindow(slave->slaveWindow);			/* Note [1] */

    DeleteSlave(slave);

    ScheduleUpdate(mgr, MGR_RESIZE_REQUIRED);
}

/*------------------------------------------------------------------------
 * +++ Tk_GeomMgr hooks.
 */

void ManagerGeometryRequestProc(ClientData clientData, Tk_Window slaveWindow)
{
    Slave *slave = clientData;
    ScheduleUpdate(slave->manager, MGR_RESIZE_REQUIRED);
}

void ManagerLostSlaveProc(ClientData clientData, Tk_Window slaveWindow)
{
    Slave *slave = clientData;
    int index = SlaveIndex(slave->manager, slave->slaveWindow);

    if (index < 0) {
	/* @@@ Don't Panic */
	Tcl_Panic("Lost slave that wasn't ours?");
	return;
    }
    RemoveSlave(slave->manager, index);
}

/*------------------------------------------------------------------------
 * +++ Public API.
 */

/* ++ AddSlave --
 */
int AddSlave(
    Tcl_Interp *interp, Manager *mgr, Tk_Window slaveWindow,
    int index, int objc, Tcl_Obj *CONST objv[])
{
    Slave *slave = CreateSlave(interp, mgr, slaveWindow);
    if (ConfigureSlave(interp, mgr, slave, objc, objv) != TCL_OK) {
	DeleteSlave(slave);
	return TCL_ERROR;
    }
    InsertSlave(mgr, slave, index);
    return TCL_OK;
}

/* ++ ConfigureSlave --
 */
int ConfigureSlave(
    Tcl_Interp *interp,Manager *mgr,Slave *slave,int objc,Tcl_Obj *CONST objv[])
{
    Tk_SavedOptions savedOptions;
    unsigned mask = 0;

    /* ASSERT: mgr->slaveOptionTable != NULL */

    if (Tk_SetOptions(interp, slave->slaveData, mgr->slaveOptionTable,
	    objc, objv, slave->slaveWindow, &savedOptions, &mask) != TCL_OK)
    {
	return TCL_ERROR;
    }

    if (mgr->managerSpec->SlaveConfigured(interp,mgr,slave,mask) != TCL_OK) {
	Tk_RestoreSavedOptions(&savedOptions);
	return TCL_ERROR;
    }

    Tk_FreeSavedOptions(&savedOptions);
    ScheduleUpdate(mgr, MGR_RELAYOUT_REQUIRED);
    return TCL_OK;
}

/* ++ ForgetSlave --
 */
void ForgetSlave(Manager *mgr, int slaveIndex)
{
    Tk_Window slaveWindow = mgr->slaves[slaveIndex]->slaveWindow;
    RemoveSlave(mgr, slaveIndex);
    Tk_ManageGeometry(slaveWindow, NULL, 0);
}

/* LayoutChanged, SizeChanged --
 * 	Schedule a relayout, resp. resize request.
 */
void LayoutChanged(Manager *mgr)
{
    ScheduleUpdate(mgr, MGR_RELAYOUT_REQUIRED);
}

void SizeChanged(Manager *mgr)
{
    ScheduleUpdate(mgr, MGR_RESIZE_REQUIRED);
}

/*------------------------------------------------------------------------
 * +++ Utility routines.
 */

/* ++ SlaveIndex --
 * 	Returns the index of specified slave window, -1 if not found.
 */
int SlaveIndex(Manager *mgr, Tk_Window slaveWindow)
{
    int index;
    for (index = 0; index < mgr->nSlaves; ++index)
	if (mgr->slaves[index]->slaveWindow == slaveWindow)
	    return index;
    return -1;
}

/* ++ GetSlaveFromObj(interp, mgr, objPtr, indexPtr) --
 * 	Return the index of the slave specified by objPtr.
 * 	Slaves may be specified as an integer index or
 * 	as the name of the managed window.
 *
 * Returns:
 * 	Pointer to slave; stores slave index in *indexPtr.
 * 	On error, returns NULL and leaves an error message in interp.
 */

Slave *GetSlaveFromObj(
    Tcl_Interp *interp, Manager *mgr, Tcl_Obj *objPtr, int *indexPtr)
{
    const char *string = Tcl_GetString(objPtr);
    int slaveIndex = 0;
    Tk_Window tkwin;

    /* Try interpreting as an integer first:
     */
    if (Tcl_GetIntFromObj(NULL, objPtr, &slaveIndex) == TCL_OK) {
	if (slaveIndex < 0 || slaveIndex >= mgr->nSlaves) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp,
		"Slave index ", Tcl_GetString(objPtr), " out of bounds",
		NULL);
	    return NULL;
	}
	*indexPtr = slaveIndex;
	return mgr->slaves[slaveIndex];
    }

    /* Try interpreting as a slave window name;
     */
    if (   (*string == '.')
	&& (tkwin = Tk_NameToWindow(interp, string, mgr->masterWindow)))
    {
	slaveIndex = SlaveIndex(mgr, tkwin);
	if (slaveIndex < 0) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp,
		string, " is not managed by ", Tk_PathName(mgr->masterWindow),
		NULL);
	    return NULL;
	}
	*indexPtr = slaveIndex;
	return mgr->slaves[slaveIndex];
    }

    Tcl_ResetResult(interp);
    Tcl_AppendResult(interp, "Invalid slave specification ", string, NULL);
    return NULL;
}

/* ++ MoveSlave(mgr, fromIndex, toIndex) --
 * 	Rearrange slaves.
 */

void MoveSlave(Manager *mgr, int fromIndex, int toIndex)
{
    Slave *moved = mgr->slaves[fromIndex];

    /* Shuffle down: */
    while (fromIndex > toIndex) {
	mgr->slaves[fromIndex] = mgr->slaves[fromIndex - 1];
	--fromIndex;
    }
    /* Or, shuffle up: */
    while (fromIndex < toIndex) {
	mgr->slaves[fromIndex] = mgr->slaves[fromIndex + 1];
	++fromIndex;
    }
    /* ASSERT: fromIndex == toIndex */
    mgr->slaves[fromIndex] = moved;

    /* Schedule a relayout.  In general, rearranging slavs
     * may also change the size:
     */
    ScheduleUpdate(mgr, MGR_RESIZE_REQUIRED);
}

/* ++ Manageable --
 * 	Returns 1 if slave can be managed by manager;
 * 	otherwise returns 0 and leaves an error message in interp.
 *
 * Conditions:
 * 	+ slave must be a direct child of master;
 * 	+ slave must not be a top-level window.
 *
 * See also:
 * 	Maintainable().
 */
int Manageable(Tcl_Interp *interp, Tk_Window slave, Tk_Window master)
{
    if (Tk_Parent(slave) != master) {
	Tcl_AppendResult(interp,
	    Tk_PathName(slave), " is not a child of ",
	    Tk_PathName(master),
	    NULL);
	return 0;
    }

    if (Tk_IsTopLevel(slave)) {
	Tcl_AppendResult(interp, "can't add ", Tk_PathName(slave),
	    ": toplevel window",
	    NULL);
	return 0;
    }

    return 1;
}

